/* Certificate Management library
 * 
 * Copyright (C) 2005 Nokia. All rights reserved.
 * Author: Ed Bartosh <Eduard.Bartosh@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "test.h"
#include <string.h>

void test_search_key(PARAM * param);
void test_search_cert(PARAM * param);
void test_search_crl(PARAM * param);

void test_search(PARAM * param)
{
    TEST_TITLE("Search functions");

    test_search_key(param);
    test_search_cert(param);
    test_search_crl(param);
}

gboolean compare_key(EVP_PKEY * a, EVP_PKEY * b, const gboolean pub)
{
    gboolean result = FALSE;
    
    unsigned char *buf_a = NULL;
    int len_a = 0;
    
    unsigned char *buf_b = NULL;
    int len_b = 0;
    
    if (pub)
    {
        len_a = i2d_PublicKey(a, &buf_a);
        len_b = i2d_PublicKey(b, &buf_b);
    }
    else
    {
        len_a = i2d_PrivateKey(a, &buf_a);
        len_b = i2d_PrivateKey(b, &buf_b);
    }

    if ((len_a > 0) && (len_b > 0))
    {
        if (len_a == len_b)
        {
            if (0 == memcmp(buf_a, buf_b, len_a))
            {
                result = TRUE;
            }
        }
    }
    
    g_free(buf_a);
    g_free(buf_b);
    
    return result;
}

void test_search_key(PARAM * param)
{
    TEST_TITLE("Search key");
    unsigned int key_id;
    EVP_PKEY * key;
    
    GSList *pubs = CST_pub_key_search_by_name(param->st, param->name);
    TEST_DO_SIMPLE(pubs != NULL);
    if (pubs)
    {
        key_id = GPOINTER_TO_UINT(pubs->data);
        key = CST_get_pub_key(param->st, key_id);
        TEST_DO_SIMPLE(compare_key(key, param->public_key, TRUE));
        printf("keyID: %u\n", key_id);
        EVP_PKEY_free(key);
        g_slist_free(pubs);
    }
    
    GSList *privs = CST_priv_key_search_by_name(param->st, param->name);
    TEST_DO_SIMPLE(privs != NULL);
    if (privs)
    {
        key_id = GPOINTER_TO_UINT(privs->data);
        printf("keyID: %u\n", key_id);
        TEST_DO_SIMPLE(key_id > 0);
        key = CST_get_key(param->st, key_id, "bad password");
        TEST_DO_SIMPLE(NULL == key);
        TEST_DO_SIMPLE(CST_ERROR_PASSWORD_WRONG == CST_last_error());
        key = CST_get_key(param->st, key_id, DEF_PASS);
        if (key != NULL)
        {
            TEST_DO_SIMPLE(compare_key(key, param->private_key, FALSE));
            EVP_PKEY_free(key);
        }
        else
        {
            TEST_DO_SIMPLE(FALSE);
        }
        
        g_slist_free(privs);
    }

    TEST_TITLE("Error codes for CST_get_key()");
    TEST_DO_SIMPLE(NULL == CST_get_key(NULL, 0, ""));
    TEST_DO_SIMPLE(CST_ERROR_PARAM_INCORRECT == CST_last_error());
    
    TEST_DO_SIMPLE(NULL == CST_get_key(param->st, 9999, ""));
    printf("Error code: %i\n", CST_last_error());
    TEST_DO_SIMPLE(CST_ERROR_KEY_NOTFOUND == CST_last_error());
    

    /* TODO: check count founded keys */
    /* TODO: free memory */
}

void test_search_cert(PARAM * param)
{
    TEST_TITLE("Search cert");

    TEST_TITLE("Search by subject name");
    GSList * list = CST_search_by_subj_name(param->st,
                                X509_get_subject_name(param->ca));
    TEST_DO_SIMPLE(NULL != list);
    g_slist_free(list);

    TEST_TITLE("Search by domain name");
    list = CST_search_by_domain_name(param->st, "www.test.org");
    TEST_DO_SIMPLE(NULL != list);
    g_slist_free(list);

    TEST_TITLE("Search by email");
    /* TODO: need same test cert with exist email */
    list = CST_search_by_email(param->st, "email@test.org");
    TEST_DO_SIMPLE(NULL == list);
    g_slist_free(list);

    list = CST_search_by_email(param->st, "ivanov@mail.ru");
    TEST_DO_SIMPLE(NULL != list);
    g_slist_free(list);

    list = CST_search_by_email(param->st, "sotona999@gmail.com");
    TEST_DO_SIMPLE(NULL != list);
    g_slist_free(list);

    TEST_TITLE("Search by UID");
    guint cid =
        CST_search_by_UID(param->st, X509_get_issuer_name(param->cert),
                          X509_get_serialNumber(param->cert));
    TEST_DO_SIMPLE(cid > 0);
    /* TODO: check what all fields ok (compare) */

    /* TODO: check count of founded cerst */

    TEST_TITLE("Search issuer");
    guint issuer_id = CST_search_issuer(param->st, param->user_cert);
    TEST_DO_SIMPLE(param->ca_uid == issuer_id);

    TEST_TITLE("Search by puroses an folders");
    list = CST_search_by_folder_and_purpose(param->st, CST_FOLDER_CA, CST_PURPOSE_CA);
    TEST_DO_SIMPLE(NULL != list);
    TEST_DO_SIMPLE(5 < g_slist_length(list));
    g_slist_free(list);

    list = CST_search_by_folder_and_purpose(param->st, CST_FOLDER_UNKNOWN, CST_PURPOSE_CA);
    TEST_DO_SIMPLE(NULL == list);

    list = CST_search_by_folder_and_purpose(param->st, CST_FOLDER_PERSONAL, CST_PURPOSE_SSL_CLIENT);
    TEST_DO_SIMPLE(NULL != list);
    TEST_DO_SIMPLE(1 == g_slist_length(list));
    if (NULL != list)
    {
        TEST_DO_SIMPLE(GPOINTER_TO_UINT(list->data) == param->user_cert_uid);
    }
    g_slist_free(list);
}

void test_search_crl(PARAM * param)
{
    TEST_TITLE("Search crl");
}
