/* Certificate Management library
 * 
 * Copyright (C) 2005 Nokia. All rights reserved.
 * Author: Ed Bartosh <Eduard.Bartosh@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/**
   @file cst_stub.c

   Certificate storage API

   Stub of function
*/

#include "cst_t.h"
#include <stdio.h>
#include <string.h>
#include <openssl/pem.h>
#include <openssl/x509v3.h>

/* Filenames of stub cert and keys */
#define FN_CERT     "files/cert.pem"
#define FN_CA       "files/cacert.pem"
#define FN_PRIVKEY  "files/caprivkey.pem"


/* Stub implementation */

cst_t_seqnum CST_import_priv_key_adv(CST * st, X509_NAME * account,
                          FILE * file, unsigned char *inpass,
                          unsigned char *outpass,
                          GError **error)
{
    return cst_import_key(st, file, inpass, outpass, account, FALSE, error);
}

cst_t_seqnum CST_import_priv_key_adv_DER(CST * st, X509_NAME * account,
                          FILE * file, unsigned char *outpass, 
                          GError **error)   
{
    return cst_import_key_DER(st, file, outpass, account, FALSE, error);
}

cst_t_seqnum CST_import_pub_key_adv(CST * st, X509_NAME * account,
                                  FILE * file, GError **error)
{
    return cst_import_key(st, file, NULL, NULL, account, TRUE, error);
}

cst_t_seqnum CST_import_pub_key_adv_DER(CST * st, X509_NAME * account,
                                  FILE * file, GError **error)
{
    return cst_import_key_DER(st, file, NULL, account, TRUE, error);
}

int CST_import_CRL(CST * st, FILE * file)
{
    return crl_import(st, file, NULL);
}

int CST_import_CRL_DER(CST * st, FILE * file)
{
    return crl_import_DER(st, file);
}

int CST_import_priv_key(CST * st, X509_NAME * account, FILE * file,
                        unsigned char *inpass, unsigned char *outpass)
{
    GError *p_error = NULL;
    cst_import_key(st, file, inpass, outpass, account, FALSE, &p_error);
    int result = p_error ? p_error->code : CST_ERROR_OK;
    g_clear_error(&p_error);
    return result;
}

int CST_import_priv_key_DER(CST * st, X509_NAME * account, FILE * file,
                        unsigned char *outpass)
{
    GError *p_error = NULL;
    cst_import_key_DER(st, file, outpass, account, FALSE, &p_error);
    int result = p_error ? p_error->code : CST_ERROR_OK;
    g_clear_error(&p_error);
    return result;   
}

int CST_export_priv_key_DER(CST * st, EVP_PKEY * key, FILE * file,
                        unsigned char *password)
{
    return cst_export_key_fmt(st, file, key, password, FALSE, 
                              IMPORT_FORMAT_DER);
}

int CST_export_priv_key(CST * st, EVP_PKEY * key, FILE * file,
                        unsigned char *password)
{
    return cst_export_key_fmt(st, file, key, password, FALSE, 
                              IMPORT_FORMAT_PEM);
}

int CST_append_priv_key(CST * st, X509_NAME * account, EVP_PKEY * key, unsigned char *p)
{
    return CST_append_EVP_PKEY(st, key, account, FALSE, p);
}

int CST_import_pub_key(CST * st, X509_NAME * account, FILE * file)
{
    GError *p_error = NULL;
    cst_import_key(st, file, NULL, NULL, account, TRUE, &p_error);
    int result = p_error ? p_error->code : CST_ERROR_OK;
    g_clear_error(&p_error);
    return result;
}

int CST_import_pub_key_DER(CST * st, X509_NAME * account, FILE * file)
{
    GError *p_error = NULL;
    cst_import_key_DER(st, file, NULL, account, TRUE, &p_error);
    int result = p_error ? p_error->code : CST_ERROR_OK;
    g_clear_error(&p_error);
    return result;   
}

int CST_export_all_pub_key_DER(CST * st, X509_NAME * account, FILE * file)
{
    return cst_export_keys_fmt(st, file, account, NULL, TRUE, IMPORT_FORMAT_DER);
}

int CST_export_all_pub_key(CST * st, X509_NAME * account, FILE * file)
{
    return cst_export_keys_fmt(st, file, account, NULL, TRUE, IMPORT_FORMAT_PEM);
}

int CST_export_pub_key_DER(CST * st, EVP_PKEY * key, FILE * file)
{
    return cst_export_key_fmt(st, file, key, NULL, TRUE, 
                              IMPORT_FORMAT_DER);
}

int CST_export_pub_key(CST * st, EVP_PKEY * key, FILE * file)
{
    return cst_export_key_fmt(st, file, key, NULL, TRUE, 
                              IMPORT_FORMAT_PEM);
}

int CST_append_pub_key(CST * st, X509_NAME * account, EVP_PKEY * key)
{
    return CST_append_EVP_PKEY(st, key, account, TRUE, NULL);
}

int CST_delete_all_pub_key(CST * st, X509_NAME * account)
{
    return cst_delete_keys(st, account, TRUE);
}

int CST_delete_all_priv_key(CST * st, X509_NAME * account)
{
    return cst_delete_keys(st, account, FALSE);
}

int CST_delete_pub_key(CST * st, const cst_t_seqnum id)
{
    return cst_delete_key(st, id, TRUE);
}

int CST_delete_priv_key(CST * st, const cst_t_seqnum id)
{
    return cst_delete_key(st, id, FALSE);
}

GSList * CST_get_chain_id_by_id(CST * st, const cst_t_seqnum certID)
{
    GSList * result = NULL;
    if (st)
    {
        if (certID > 0)
        {
            X509 * x = CST_get_cert(st, certID);
            if (x)
            {
                result = CST_get_chain_id(st, x);
                X509_free(x);    
            }
        }
    }
    else
    {
        CST_error(CST_ERROR_PARAM_INCORRECT);
    }
    return result;
}
    
GSList * CST_get_chain_id(CST * st, X509 * x)
{
    TRACE("*");
    GSList * result = NULL;
    
    if (st && x)
    {
        STACK_OF(X509) * sk = CST_get_chain(st, x);

        if (sk)
        {
            int i;
            t_seqnum certID = 0;
            for (i = sk_X509_num(sk) - 1; i >= 0; i--)
            {
                certID = CST_search_by_X509(st, sk_X509_value(sk, i));
                if (certID == 0)
                {
                    g_slist_free(result);
                    result = NULL;
                    break;
                }
                result = g_slist_append(result, SEQNUM_TO_POINTER(certID));
            }
            cert_free_stack(sk);
        }
    }
    else
    {
        CST_error(CST_ERROR_PARAM_INCORRECT);
    }
    return result;
}

cst_t_seqnum CST_search_issuer(CST * st, X509 * cert)
{
    GSList * possible_issuers;
    t_seqnum result = 0;

    if (st && cert)
    {
        CST_LOCK_BEGIN(LOCK_SH);
            possible_issuers =
                cst_search_by_subj_name(st, X509_get_issuer_name(cert));
            
            TRACE_LIST(possible_issuers);
            if (possible_issuers)
            {
                GSList * i_pi = NULL;
                t_seqnum certID = 0;
                X509 * issuer_cert = NULL;
                
                for (i_pi = possible_issuers; i_pi != NULL; i_pi = i_pi->next)
                {
                    certID = SEQNUM_FROM_POINTER(i_pi->data);
                    issuer_cert = cert_get_X509(st, certID);
                    
                    if ((X509_check_issued(issuer_cert, cert) == X509_V_OK) 
                        && (cst_get_folder(st, certID) == CST_FOLDER_CA)
                        && (cst_is_purpose(st, certID, CST_PURPOSE_CA) > 0))
                    {
                        result = certID;
                    }
                    X509_free(issuer_cert);

                    if (result > 0) break;
                }
                g_slist_free(possible_issuers);
            }   
        CST_LOCK_END;
    }
    else
    {
        CST_error(CST_ERROR_PARAM_INCORRECT);
    }
    return result;
}

STACK_OF(X509) * CST_get_chain(CST * st, X509 * x)
{
    TRACE("*");
    if (st && x)
    {
        STACK_OF(X509) * result = sk_X509_new_null();
        GSList * possible_issuers;
        GSList * i_pi;
        t_seqnum certID;
        X509 *last = x;
        X509 *issuer_cert = last;

        while (issuer_cert)
        {
            if (x != issuer_cert)
            {
                sk_X509_push(result, issuer_cert);
            }
            
            if (cert_is_root_x(issuer_cert))
            {
                CST_error(CST_ERROR_OK);
                break;
            }

            possible_issuers =
                CST_search_by_subj_name(st, X509_get_issuer_name(issuer_cert));
            TRACE_LIST(possible_issuers);

            if (possible_issuers)
            {
                last = issuer_cert;
                issuer_cert = NULL;

                for (i_pi = possible_issuers; i_pi != NULL; i_pi = i_pi->next)
                {
                    certID = SEQNUM_FROM_POINTER(i_pi->data);
                    issuer_cert = CST_get_cert(st, certID);
                    
                    if ((X509_check_issued(issuer_cert, last) == X509_V_OK) 
                        && (CST_get_folder(st, certID) == CST_FOLDER_CA)
                        && (CST_is_purpose(st, certID, CST_PURPOSE_CA) > 0))
                    {
                        TRACE("  ok");
                        break;
                    }
                    
                    X509_free(issuer_cert);
                    issuer_cert = NULL;
                }
       
                g_slist_free(possible_issuers);
      
            }
            else
            {
                CST_error(CST_ERROR_CHAIN_INCOMPLETE);
                break;
            }

            CST_error(CST_ERROR_CHAIN_INCOMPLETE);
        }
        
        if(sk_X509_num(result) == 0)
        {
            sk_X509_free(result);
            result = NULL;
        }
        
        return result;
    }
    else
    {
        CST_error(CST_ERROR_PARAM_INCORRECT);
        return NULL;
    }
}


X509_NAME *CST_get_issued_by_dn(X509 * cert)
{
    if (cert) return X509_NAME_dup(X509_get_issuer_name(cert));
    else return NULL;
}

X509_NAME *CST_get_subject_dn(X509 * cert)
{
    if (cert) return X509_NAME_dup(X509_get_subject_name(cert));
    else return NULL;
}

static time_t asn1time_to_unix(ASN1_TIME * time)
{
    g_assert(time);

    struct tm r;
    memset(&r, 0, sizeof(r));

    int j;
    char *d = time->data;
    int size = time->length;

    if (V_ASN1_UTCTIME == time->type)
    {
        if (time->length < 10)
            return 0;
        //if (d[time->length - 1] == 'Z') gmt=1;
        for (j = 0; j < 10; j++)
        {
            if ((d[j] > '9') || (d[j] < '0'))
                return 0;
        }

        r.tm_year = (d[0] - '0') * 10 + (d[1] - '0');
        if (r.tm_year < 50)
        {
            r.tm_year += 100;
        }


    } else if (V_ASN1_GENERALIZEDTIME == time->type)
    {
        if (time->length < 12)
            return 0;
        //if (d[sz-1] == 'Z') gmt=1;

        for (j = 0; j < 12; j++)
        {
            if ((d[j] > '9') || (d[j] < '0'))
                return 0;
        }

        r.tm_year =
            (d[0] - '0') * 1000 + (d[1] - '0') * 100 + (d[2] - '0') * 10 +
            (d[3] - '0');

        d += 2;
        size -= 2;
    } else
    {
        return 0;
    }

    r.tm_mon = (d[2] - '0') * 10 + (d[3] - '0') - 1;
    if ((r.tm_mon > 11) || (r.tm_mon < 0))
        return 0;

    r.tm_mday = (d[4] - '0') * 10 + (d[5] - '0');
    if ((r.tm_mday < 1) || (r.tm_mday > 31))
        return 0;

    r.tm_hour = (d[6] - '0') * 10 + (d[7] - '0');

    r.tm_min = (d[8] - '0') * 10 + (d[9] - '0');

    r.tm_sec = 0;
    if (size > 11)
    {
        if ((d[10] >= '0') && (d[10] <= '9') && (d[11] >= '0')
            && (d[11] <= '9'))
        {
            r.tm_sec = (d[10] - '0') * 10 + (d[11] - '0');
        }
    }

    return mktime(&r);
}

time_t CST_get_valid_from(X509 * cert)
{
    if (cert) return asn1time_to_unix(X509_get_notBefore(cert));
    else return 0;
}

time_t CST_get_valid_to(X509 * cert)
{
    if (cert) return asn1time_to_unix(X509_get_notAfter(cert));
    else return 0;
}

int CST_is_expired(X509 * cert)
{
    if (cert)
        return (X509_cmp_current_time(X509_get_notBefore(cert)) > 0)
            || (X509_cmp_current_time(X509_get_notAfter(cert)) < 0);
    else return FALSE;
}

int CST_get_state(CST * st, X509 * cert)
{
    if (st && cert)
    {
        if (CST_is_expired(cert))
        {
            return CST_STATE_EXPIRED;
        }

        if (CST_is_revoked(st, cert))
        {
            return CST_STATE_REVOKED;
        }

        if (CST_is_valid(st, cert))
        {
            return CST_STATE_VALID;
        }
        else
        {
            return CST_STATE_NOTVALID;
        }
    }
    else
    {
        return -1;
    }
}

ASN1_INTEGER *CST_get_serial_number(X509 * cert)
{
    if (cert) return ASN1_INTEGER_dup(X509_get_serialNumber(cert));
    else return NULL;
}

char *CST_get_serial_number_t(X509 * cert)
{
    if (cert)
    {
        ASN1_INTEGER *serial = X509_get_serialNumber(cert);
        return buffer_to_hex(serial->data, serial->length);
    }
    else
    {
        return NULL;
    }
}

char *fingerprint(X509 * cert, const EVP_MD * digest)
{
    // SEE: apps/x509.c 865
    if (cert)
    {
        g_assert(digest);

        int md_size;
        unsigned char md_buffer[EVP_MAX_MD_SIZE];

        if (!X509_digest(cert, digest, md_buffer, &md_size))
        {
            return NULL;
        }

        return buffer_to_hex(md_buffer, md_size);
    }
    else
    {
        return NULL;
    }
}

char *CST_get_fingerprint_MD5(X509 * cert)
{
    return fingerprint(cert, EVP_md5());
}

char *CST_get_fingerprint_SHA1(X509 * cert)
{
    return fingerprint(cert, EVP_sha1());
}

char *CST_get_fingerprint(X509 * cert)
{
    return CST_get_fingerprint_MD5(cert);
}

char *CST_get_email(X509 * cert)
{
    if (cert)
    {
       return cert_get_email_x(cert);
    }
    return NULL;
}
    
char *CST_get_domain_name(X509 * cert)
{
    if (cert)
    {
        return cert_get_domain_name_x(cert);
    }
    return NULL;
}

char *CST_get_public_key_alg(X509 * cert)
{
    if (cert)
    {
        const int BUF_SIZE = 80;
        char buf[BUF_SIZE + 1];
        memset(buf, 0, BUF_SIZE + 1);
        if (0 == OBJ_obj2txt(buf, BUF_SIZE, cert->cert_info->key->algor->algorithm, 0));
        {
            return strdup(buf);
        }
    }
    return NULL;
}

int CST_is_root(X509 * cert)
{
    if (cert)
    {
        return cert_is_root_x(cert);
    }
    return FALSE;
}

int CST_is_CA(X509 * x)
{
    int i;
    X509_EXTENSION * e = NULL;
    X509V3_EXT_METHOD * m = NULL;
    STACK_OF(CONF_VALUE) * values = NULL;
    CONF_VALUE * item = NULL;
    unsigned char * p = NULL;
    void * es = NULL;
    int result = TRUE;
    
    for (i = 0; i < X509_get_ext_count(x); i++)
    {
        e = X509_get_ext(x, i);
        if (NID_basic_constraints == OBJ_obj2nid(X509_EXTENSION_get_object(e)))
        {
            result = FALSE;
            m = X509V3_EXT_get(e);
            
            if (m && m->i2v && e->value && e->value->data)
            {
                p = e->value->data;
                if (m->it)
                {
                    es = ASN1_item_d2i(NULL, &p, e->value->length, ASN1_ITEM_ptr(m->it));
                }
                else
                {
                    es = m->d2i(NULL, &p, e->value->length);
                }
                
                values = m->i2v(m, es, NULL);
                item = NULL;
                if (values)
                {
                    int j = 0;
                    for (j = 0; j < sk_CONF_VALUE_num(values); j++)
                    {
                        item = sk_CONF_VALUE_value(values, j);
                        if (strcmp(item->name, "CA") == 0)
                        {
                            /* printf("%s = %s", item->name, item->value); */
                            if (strcmp(item->value, "TRUE") == 0)
                            {
                                result = TRUE;
                            }
                            break;
                        }
                    }
                }
            }
            /* LOG_DEBUG("CA\n"); */
        }
    }

    return result;
}

int CST_is_root_id(CST * st, const cst_t_seqnum certID)
{
    if (st && (certID > 0))
    {
        X509 * x = CST_get_cert(st, certID);
        if (x)
        {
            int result = CST_is_root(x);
            X509_free(x);
            return result;
        }
    }
    return FALSE;
}

char *CST_EVP_PKEY_to_text(EVP_PKEY * pkey)
{
    char *result = NULL;
    if (pkey)
    {
        if (pkey->type == EVP_PKEY_RSA || pkey->type == EVP_PKEY_DSA)
        {
            BIO *mem = BIO_new(BIO_s_mem());
            
            if (pkey->type == EVP_PKEY_RSA)
            {
                RSA_print(mem, pkey->pkey.rsa,0);
            }
            else if (pkey->type == EVP_PKEY_DSA)
            {
                DSA_print(mem, pkey->pkey.dsa,0);
            }

            BUF_MEM *bptr;
            BIO_get_mem_ptr(mem, &bptr);
            result = g_malloc0(bptr->length + 1);
            memcpy(result, bptr->data, bptr->length);
            BIO_free(mem);
        }
    }
    return result;
}

GSList * CST_pub_key_search_by_name(CST * st, X509_NAME * name)
{
    return cst_key_search_by_name(st, name, TRUE);
}

GSList * CST_priv_key_search_by_name(CST * st, X509_NAME * name)
{
    return cst_key_search_by_name(st, name, FALSE);
}

static int cst_is_valid_callback(int is_ok, X509_STORE_CTX *ctx)
{
    return is_ok;
}

int CST_is_valid_for(CST * st, X509 * cert, const cst_t_cert_purpose purpose)
{
    TRACE("*");
    if (st && cert)
    {         
        t_seqnum certID = CST_search_by_UID(st, 
                X509_get_issuer_name(cert),
                X509_get_serialNumber(cert));

        if (certID > 0)
        {
            if (CST_is_purpose(st, certID, purpose) > 0)
            {
                return TRUE;
            }
        }
        else
        {
            // TODO: Do check of purposes from X509 certificate!!!
            return CST_is_valid(st, cert);
        }
    }
    else
    {
        CST_error(CST_ERROR_PARAM_INCORRECT);
    }
    return FALSE;
}

static gboolean cst_is_valid_internal(CST * st, X509 * cert, GSList * chain)
{
    TRACE("*");
    g_assert(st && cert);

    int result = FALSE;
    
    if (chain) 
    {
        TRACE_LIST(chain);
        // TODO: Check that if not trusted then must be CA
        //if ((cst_is_purpose(st, issuerID, CST_PURPOSE_CA) > 0)) 
        //{   
            X509_STORE * store = X509_STORE_new();

            if (store)
            {
                X509_STORE_set_verify_cb_func(store, cst_is_valid_callback);

                GSList * i = NULL;
                X509 * issuerX509 = NULL;
                for (i = chain; i != NULL; i = i->next)
                {
                    issuerX509 = cert_get_X509(st, SEQNUM_FROM_POINTER(i->data));
                    if (issuerX509)
                    {
                        X509_STORE_add_cert(store, issuerX509);
                        X509_free(issuerX509);
                    }
                }
                
                //if (issuerX509)
                //{
                    X509_STORE_CTX *ctx = X509_STORE_CTX_new();
                    
                    if (ctx)
                    {
                        if (X509_STORE_CTX_init(ctx, store, cert, NULL) 
                                && X509_verify_cert(ctx))
                        {
                            result = TRUE;
                        }
                       
#if 0
                        X509 * trace_x = ctx->current_cert;
                        const int TRACE_BUF_SZ = 60;
                        char *trace_buf[TRACE_BUF_SZ];
                        if (trace_x)
                        {
                            X509_NAME_oneline(X509_get_subject_name(trace_x), trace_buf, TRACE_BUF_SZ);
                            printf(">>>%s\n", trace_buf);
                        }
                        int err = X509_STORE_CTX_get_error(ctx);
                        TRACE_S(X509_verify_cert_error_string(err));
#endif
                        X509_STORE_CTX_free(ctx);
                    }
                    
                //    X509_free(issuerX509);
                //}

                X509_STORE_free(store);
            }
        //}
    }
    return result;
   
}

static GSList * cst_get_all_chain(CST * st, X509 * xcert)
{
    g_assert(st && xcert);
    GSList * result = NULL;

    if (!cert_is_root_x(xcert))
    {
        GSList * possible_issuers = cst_search_by_subj_name(st,
                                X509_get_issuer_name(xcert));
        GSList * i = NULL;
        X509 * x = NULL;
        GSList * r = NULL;
        GSList * j = NULL;
        GSList * r_item = NULL;
        t_seqnum certID = 0;
        for (i = possible_issuers; i != NULL; i = i->next)
        {
            certID = SEQNUM_FROM_POINTER(i->data);
            if (cst_is_purpose(st, certID, CST_PURPOSE_CA) > 0)
            {    
                x = cert_get_X509(st, certID);
                if (x)
                {
                    r = cst_get_all_chain(st, x);
                    X509_free(x);
                    if (NULL != r)
                    {
                        TRACE_LIST(r);
                        for (j = r; j != NULL; j = j->next)
                        {
                            r_item = (GSList *) j->data;
                            r_item = g_slist_prepend(r_item, SEQNUM_TO_POINTER(i->data));
                            result = g_slist_prepend(result, r_item);
                        }
                        g_slist_free(r);
                    }
                }
            }
        }
        g_slist_free(possible_issuers);
    }
    else
    {
        result = g_slist_prepend(result, NULL);
    }
    
    return result;
}

int CST_is_valid(CST * st, X509 * cert)
{
    TRACE("*");

    int result = FALSE;

    if (st && cert)
    {
        CST_LOCK_BEGIN(LOCK_SH);
            GSList * chains = cst_get_all_chain(st, cert);
            GSList * i;

            /* Note that we don't verify whether the cert has
             * expired or not, because this is done in
             * CST_get_state */
            if (NULL == chains)
                result = TRUE;
            else if (SEQNUM_FROM_POINTER(chains->data) == 0)
                    result = TRUE;

            for (i = chains; i != NULL; i = i->next)
            {
                if (!result && cst_is_valid_internal(st, cert, (GSList *)(i->data)))
                {
                    result = TRUE;
                }
                g_slist_free((GSList *)(i->data));
            }
            g_slist_free(chains);
        CST_LOCK_END;
    }
    else
    {
        CST_error(CST_ERROR_PARAM_INCORRECT);
    }
    return result;
}

static int cst_is_valid_f_locked(CST * st, FILE * file, const int fmt, GError **error)
{
    g_assert(st && file);
    int result = FALSE;

    X509 *x = NULL;
    
    switch (fmt)
    {
    case IMPORT_FORMAT_PEM:
        x = PEM_read_X509(file, NULL, NULL, NULL);
        break;
    case IMPORT_FORMAT_DER:
        x = d2i_X509_fp(file, NULL);
        break;
    }

    if (x)
    {
        result = CST_is_valid(st, x);
        X509_free(x);
    }
    
    
    return result;
}

int CST_is_valid_f(CST * st, FILE * file, GError **error)
{
    if (st && file)
    {
        return cst_is_valid_f_locked(st, file, IMPORT_FORMAT_PEM, error);
    }
    else
    {
        CST_error(CST_ERROR_PARAM_INCORRECT);
        return FALSE;
    }
}

int CST_is_valid_f_DER(CST * st, FILE * file, GError **error)
{
    if (st && file)
    {
        return cst_is_valid_f_locked(st, file, IMPORT_FORMAT_DER, error);
    }
    else
    {
        CST_error(CST_ERROR_PARAM_INCORRECT);
        return FALSE;
    }
}

cst_t_seqnum CST_get_assigned_key(CST * st, const cst_t_seqnum certID)
{
    t_seqnum result = 0;
    if (st && (certID > 0))
    {
        CST_LOCK_BEGIN_S(LOCK_SH);
            result = cst_get_keylink(st, certID);
        CST_LOCK_END;
    }
    else
    {
        CST_error(CST_ERROR_PARAM_INCORRECT);
    }
    return result;
}

CST_STACK_OF_X509 * CST_all_CA()
{
    g_type_init();
    CST * st = CST_open(0, NULL);

    if (st == NULL)
    {
        return NULL;
    }
    
    STACK_OF(X509) * sk = sk_X509_new_null();
    GSList * list = CST_search_by_folder(st, CST_FOLDER_CA);
    GSList * i;
    for (i = list; i != NULL; i = i->next)
    {
        X509 * x = CST_get_cert(st, GPOINTER_TO_UINT(i->data));
        if (x == NULL)
            continue;
        
        sk_X509_push(sk, x);
        //printf("certID = %u\n", GPOINTER_TO_UINT(i->data));
    }
    g_slist_free(list);
    
    CST_free(st);

    return sk;
}
