/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the EAL package.
 *
 * The Initial Developer of the Original Code is Nokia Corporation.
 * Portions created by the Initial Developer are Copyright (C) 2005
 * the Initial Developer. All Rights Reserved.
 *
 * Contact: Leonid Zolotarev <leonid.zolotarev@nokia.com>
 *
 * ***** END LICENSE BLOCK ***** */

#include <glib-object.h>
#include "gwebcommon.h"

#ifndef __GWEBWIDGET_H__
#define __GWEBWIDGET_H__

G_BEGIN_DECLS

#define G_WEBWIDGET_SIGNAL_URL_CHANGED "url_changed"

/** The browser changed the current url, for example slashdot.org -> http://www.slashdot.org
  * @param engine The engine
  * @param url (const gchar*) The new url
  */
void g_web_widget_signal_url_changed(GObject *browser_window,
                                     const gchar *url);

#define G_WEBWIDGET_SIGNAL_TITLE "title_changed"

/** The page title changed
  * @param engine_widget Engine widget
  * @param title The new title of the page
  */
void g_web_widget_signal_title(GObject *engine_widget,
                               const gchar *title,
                               GObject *browser_window);

#define G_WEBWIDGET_SIGNAL_START_LOADING "start_loading"

/** A page has started loading
  * @param engine_widget Engine widget
  * @param url (const gchar*) The url that started to load
  */
void g_web_widget_signal_start_loading(GObject *engine_widget,
                                        const gchar *url,
					GObject *browser_window);

#define G_WEBWIDGET_SIGNAL_FINISHED_LOADING "finished_loading"

/** The pages has finished loading
  * @param engine_widget Engine widget
  * @param has_frames TRUE if the page is a frameset. FALSE if it's a
  * normal document even if it contains iframe tags and even if those
  * iframes contain documents which themselves are framesets.
  * @param finish_status
  * @param page_size
  */
void g_web_widget_signal_finished_loading(GObject *engine_widget,
                                          gint finish_status,
					  gint page_size,
					  gboolean has_frames,
					  GObject *browser_window);

#define G_WEBWIDGET_SIGNAL_LOADING_PROGRESS "loading_progress"

/** Indicate page load progress
  * @param engine_widget Engine widget
  *
  * @param loaded_images (gint) Nnumber of images, external
  * scripts and other inline elements loaded (or skipped if the loading
  * is turned off) for this page. This gives a crude approximation of
  * how much work has been done and how much is left.
  *
  * @param total_images (gint) Total number of images,
  * external scripts and other inline elements on the page. This is the
  * "currently known" number and will change while a HTML document is
  * loaded and parsed.
  */
void g_web_widget_signal_loading_progress(GObject *browser_window,
                                          gint loaded_images,
					  gint total_images);

#define G_WEBWIDGET_SIGNAL_SECURITY_MODE_CHANGED "security_mode_changed"

/** The security display mode has changed
  * @param engine_widget Engine widget
  * @param(GtkWebEngineSecurityMode) the new security mode
  */
void g_web_widget_signal_security_mode_changed(GObject *engine_widget,
                                                gint mode,
						GObject *browser_window);

#define G_WEBWIDGET_SIGNAL_AUTHENTIFICATION_REQUIRED "authentication_required"

/** Http authentication is required
  * @param engine_widget Engine widget
  * @param authenticaion_type a GtkWebEngineAuthenticationType to indicate what kind of authentication was requested
  * @param server (const gchar*) The server that requested the authentication
  * @param message (const gchar*) A message from the server
  */
void g_web_widget_signal_authentication_required(GObject *engine_widget,
                                                 gint authentication_type,
                                                 const gchar *server,
                                                 const gchar *message,
                                                 GObject *browser_window);

#define G_WEBWIDGET_SIGNAL_ALERT "alert"

/** An alert dialog was requested
  * @param engine_widget Engine widget
  * @param message (const gchar*) The message
  */
void g_web_widget_signal_alert(GObject *engine_widget,
    				    const gchar *message,
				    GObject *browser_window);

#define G_WEBWIDGET_SIGNAL_CONFIRM "confirm"

/** A confirm (OK/Cancel) dialog was requested
  * @param engine_widget Engine widget
  * @param message (const gchar*) The message
  */
gboolean g_web_widget_signal_confirm(GObject *engine_widget,
                                 const gchar *message,
				 GObject *browser_window);

#define G_WEBWIDGET_SIGNAL_PROMPT "prompt"

/** A Ecmascript prompt dialog was requested
  * @param engine_widget Engine widget
  * @param message (const gchar*) A message
  * @param default_value (const gchar*) Default value for the text entry
  *
  * @return a g_malloced string that Engine will free, or NULL if cancel was pressed
  */
gchar *g_web_widget_signal_prompt(GObject * engine_widget,
                                  const gchar *message,
				  const gchar *default_value,
				  GObject *browser_window);

#define G_WEBWIDGET_SIGNAL_MAILTO "mailto"

/** A mailto link was clicked
  * @param engine_widget Engine widget
  * @param url (const gchar*) The mailto url
  */
void g_web_widget_signal_mailto(GObject *engine_widget,
                                const gchar *url,
				GObject *browser_window);

#define G_WEBWIDGET_SIGNAL_SELECT "select"

/** A Ecmascript select dialog was requested
+  * @param engine_widget Engine widget
+  * @param title (const gchar*) The dialog title
+  * @param message (const gchar*) A message
+  * @param info (GList) A list of options
+  * @param selected_item (gint*) The user choice
+  *
+  * @return TRUE if OK was pressed
+  */
gboolean g_web_widget_signal_select(GObject * engine_widget, const gchar *title,
                           const gchar *message, gpointer info,
                           gint *selected_item, GObject * self);

#define G_WEBWIDGET_SIGNAL_UNKNOWN_PROTOCOL "unknown_protocol"

/** A link with an unknown protocol was clicked
  * @param engine_widget Engine widget
  * @param url (const gchar*) The unknown url
  */
void g_web_widget_signal_unknown_protocol(GObject *engine_widget,
                                          const gchar *url,
					  GObject *browser_window);

#define G_WEBWIDGET_SIGNAL_CLOSE_WINDOW "close_window"

/** The widget wants to be closed
  * This method should destroy the widget and all ui that is attached to it (typically a toplevel window with buttons, urlentry etc....)
  * @param engine_widget Engine widget
  */
void g_web_widget_signal_close_window(GObject *engine_widget,
                                      GObject *browser_window);

#define G_WEBWIDGET_SIGNAL_WINDOW_REQUEST "window_request"

/** Callback function that Engine uses to request a new browser window
  * Signal name: "window-request"
  *
  * @param engine_widget Engine widget
  * @param context. If this is non-NULL, Engine  has already created a context for this window to use. Use this context when creating a new widget
  *                 If Engine has already created a context for the window to use, the other parameters may be overridden by Engine.
  *
  * @param url Url to load
  * @param in_background True if Engine wants to open this window in the background
  * @param unrequested True if opening of the window wasn't requested by the user.
  */
void g_web_widget_signal_window_request(GObject * engine_widget,
                                        gpointer context,
					const gchar *url,
					gboolean in_background, 
					GObject *browser_window);

#define G_WEBWIDGET_SIGNAL_DOWNLOAD_REQUEST "download_request"

/** The browser wants to download this file
  *
  * The UI should use this information to initiate a transfer using the gtkEngine_transfer api
  *
  * @param engine_widget Engine widget
  * @param url The url that was requested for download
  * @param suggested_filename The suggested filename
  * @param mimetype The mimetype
  * @param size The size
  * @param mode DownloadRequestMode
  */
void g_web_widget_signal_download_request(GObject *engine_widget,
                                          const gchar *url,
					  const gchar *suggested_filename,
					  const gchar *mimetype,
					  glong size,
					  gint mode,
					  GObject *browser_window);

#define G_WEBWIDGET_SIGNAL_RAISE_REQUEST "raise_request"

/** The browser requests to raise the current window
  * @param engine_widget Engine widget
  */
void g_web_widget_signal_raise_request(GObject *engine_widget,
                                       GObject *browser_window);

#define G_WEBWIDGET_SIGNAL_LOWER_REQUEST "lower_request"

/** The browser requests to lower the current window
  * @param engine_widget Engine widget
  * @param void No parameter to this signal
  */
void g_web_widget_signal_lower_request(GObject *engine_widget,
                                       GObject *browser_window);

#define G_WEBWIDGET_SIGNAL_UPLOAD_DIALOG "upload_dialog"

/** Called when an upload dialog is requested
  * Signal name: "upload-dialog"
  *
  * @param engine_widget Engine widget
  * @param path The default path
  * @param filter The default filter
  * @param result Pointer to where the file name should be saved. The implementor must allocate the space with g_malloc, it will be freed inside Engine.
  *
  * @return TRUE if OK was pressed
  */
gboolean g_web_widget_signal_upload_dialog(GObject *engine_widget,
                                	    const gchar *path,
					    const gchar *filter,
					    gchar **result,
					    GObject *browser_window);

#define G_WEBWIDGET_SIGNAL_FILE_UPLOAD_STARTED "file_upload_started"

/** Started uploading file(s).
  * E.g. The user submited a form containing a file input.
  * @param engine_widget Engine widget
  */
void g_web_widget_signal_file_upload_started(GObject *engine_widget,
                                             GObject *browser_window);

#define G_WEBWIDGET_SIGNAL_FILE_UPLOAD_PROGRESS "file_upload_progress"

/** File upload progress.
  * @param engine_widget Engine widget
  * @param bytes   The number of bytes sent so far.
  * @param message Status message.
  */
void g_web_widget_signal_file_upload_progress(GObject *engine_widget,
                                        	gint bytes,
						const gchar *message,
						GObject *browser_window);

#define G_WEBWIDGET_SIGNAL_FILE_UPLOAD_FINISHED "file_upload_finished"

/** The file uploading has finished
  * @param engine_widget Engine widget
  */
void g_web_widget_signal_file_upload_finished(GObject *engine_widget,
                                              GObject *browser_window);

#define G_WEBWIDGET_SIGNAL_WEBENGINE_POPUP_MENU "webengine_popup_menu"

/** A popup-menu was requested
  * Signal name: "webengine-popup-menu"
  * @param engine_widget Engine widget
  * @param menutype Which type of menu, GWebEnginePopupMenuType
  * @param x X point of popup-menu
  * @param y Y point of popup-menu
  * @param text The text that was clicked
  * @param url The URL that was clicked
  * @param image_url The URL of the image that was clicked
  * @param frame_url The URL of the frame in focus
  */
void g_web_widget_signal_webengine_popup_menu(GObject *engine_widget,
                                                guint menutype,
						gint x,
						gint y,
						const gchar *text,
						const gchar *url,
						const gchar *image_url,
						const gchar *frame_url,
						GObject *browser_window);

#define G_WEBWIDGET_SIGNAL_EXTERNAL_ACTION "external_action"

/** An external action (as defined in input.ini) was requested from the browser
  * The external action signal can be added in the [Browser Widget] context. When
  * the browser widget (or any of its children) is focused, this signal will be
  * emitted along with the following string, if key in the definition was pressed.
  *
  * Example (to be added to input.ini)
  *
  * [Gtk Widget]
  * Esc = External Action, "Escape was pressed"
  * Esc ctrl = External Action, "Escape and control was pressed"
  *
  * @param engine_widget Engine widget
  * @param action The description of the action as described in preferences
  */
void g_web_widget_signal_external_action(GObject *engine_widget,
                                         const gchar *action);

#define G_WEBWIDGET_SIGNAL_MODAL_DIALOG "modal_dialog"

/** Callback that Engine uses to request a modal dialog with a message
  * Signal name: "modal-dialog"
  * @param engine_widget Engine widget
  * @param caption The caption for the dialog
  * @param message The message in the dialog
  * @param type The type of dialog
  * @param button1 First button
  * @param button2 Second button (G_WEB_DIALOG_NO_BUTTON if no button)
  * @param button3 Third button (G_WEB_DIALOG_NO_BUTTON if no button)
  * @return The implementation if this function should return the button that was clicked as a reply
  */
GWebDialogButton g_web_widget_signal_modal_dialog(GObject *engine_widget,
                                                  const gchar *caption,
						  const gchar *message,
						  GWebDialogMessageType type,
						  GWebDialogButton button1,
						  GWebDialogButton button2,
						  GWebDialogButton button3,
						  GObject *browser_window);

#define G_WEBWIDGET_SIGNAL_ENCODING "encoding"

/** The page has decided on an encoding
  * @param engine_widget Engine widget
  */
void g_web_widget_signal_encoding(GObject *engine_widget,
                                  GWebEngineEncoding encoding,
				  GObject *browser_window);

#define G_WEBWIDGET_SIGNAL_FRAME_STACKING_CHANGED "frame_stacking_changed"

/** Frame stacking mode changed
  * Signal name: "frame-stacking-changed"
  * @param engine_widget Engine widget
  * @param on TRUE if frame stacking was turned on, false if it was turned off
  */
void g_web_widget_signal_frame_stacking_changed(GObject *engine_widget,
                                                gboolean on,
						GObject *browser_window);

#define G_WEBWIDGET_SIGNAL_MODAL_CERTIFICATE_DIALOG "modal-certificate-dialog"

/** Callback that Engine uses to request a modal certificate dialog with a message
  * Signal name: "modal-certificate-dialog"
  * @param engine_widget Engine widget
  * @param caption The caption for the dialog
  * @param message The message in the dialog
  * @param type The type of dialog
  * @param button1 First button
  * @param button2 Second button (G_WEB_DIALOG_NO_BUTTON if no button)
  * @param button3 Third button (G_WEB_DIALOG_NO_BUTTON if no button)
  * @return The implementation if this function should return the button that was clicked as a reply
  */
GWebDialogButton g_web_widget_signal_modal_certificate_dialog(GObject *engine_widget,
								   const gchar *caption,
								   GWebCertificateMessage message,
								   GWebDialogMessageType type,
								   GWebDialogButton button1,
								   GWebDialogButton button2,
								   GWebDialogButton button3,
								   GObject *browser_window);

#define  G_WEBWIDGET_SIGNAL_RSS_REQUEST "rss-request"

/** A RSS feed was selected.
  * @param engine_widget Engine widget
  * @param url (gchar*) The url to the rss feed
  * @param title (gchar*) The title of the rss feed
  */
void g_web_widget_signal_rss_request(GObject *engine_widget,
					gchar *url,
					gchar *title,
					GObject *browser_window);

G_END_DECLS

#endif /* __GWEBWIDGET_H__ */
