/**
  @file server.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2006 Nokia Corporation. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License, version 2, as
  published by the Free Software Foundation.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdint.h>
#include <errno.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/socket.h>

#include <glib.h>

#include <bluetooth/bluetooth.h>
#include <bluetooth/rfcomm.h>
#include <bluetooth/sdp.h>
#include <bluetooth/sdp_lib.h>

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include "log.h"
#include "sdp.h"
#include "client.h"
#include "server.h"

struct server_context {
    bdaddr_t            local_address;

    uint32_t            record_handle;

    GIOChannel         *io;
    struct sockaddr_rc  addr;

    GSList             *clients;

    int                 limit;

    char               *root;
};

static struct server_context *ctx = NULL;

static int server_socket(uint8_t *channel)
{
    int sock;

    sock = socket(AF_BLUETOOTH, SOCK_STREAM, BTPROTO_RFCOMM);
    if (sock < 0) {
        error("server socket: %s", g_strerror(errno));
        return -1;
    }

    ctx->addr.rc_family = AF_BLUETOOTH;
    ctx->addr.rc_channel = *channel;
    bacpy(&ctx->addr.rc_bdaddr, &ctx->local_address);
    if (bind(sock, (struct sockaddr*)&ctx->addr, sizeof(ctx->addr)) < 0) {
        error("server bind: %s", g_strerror(errno));
        close(sock);
        return -1;
    }

    if (listen(sock, 1) < 0) {
        error("server listen: %s", g_strerror(errno));
        close(sock);
        return -1;
    }

    if (*channel == 0) {
        struct sockaddr_rc sa = {0};
        unsigned int sa_len = sizeof(struct sockaddr_rc);
        getsockname(sock, (struct sockaddr*)&sa, &sa_len);
        *channel = sa.rc_channel;
    }

    return sock;
}

static void dc_callback(ClientConnection *client, struct server_context *server)
{
    server->clients = g_slist_remove(server->clients, client);
}

static gboolean connect_callback(GIOChannel *io, GIOCondition cond,
                                        struct server_context *server)
{
    int cli_fd, srv_fd;
    socklen_t size;
    struct sockaddr_rc cli_addr;
    ClientConnection *client;

    srv_fd = g_io_channel_unix_get_fd(io);

    if (!(cond & G_IO_IN)) {
        error("Error on server socket");
        raise(SIGTERM);
        return FALSE;
    }

    size = sizeof(struct sockaddr_rc);
    cli_fd = accept(srv_fd, (struct sockaddr*)&cli_addr, &size);
    if (cli_fd < 0) {
        error("accept: %s", g_strerror(errno));
        /* Ignore this error since we can just continue listening */
        return TRUE;
    }

    if (g_slist_length(server->clients) >= server->limit) {
        debug("Maximum client limit (%d) reached: refusing connect attempt",
                server->limit);
        close(cli_fd);
        return TRUE;
    }

    client = client_new(server, cli_fd, &cli_addr, server->root);
    if (!client)
        return TRUE;

    client_add_disconnect_callback(client, (client_dc_cb_t)dc_callback, server);

    server->clients = g_slist_append(server->clients, client);
    
    return TRUE;
}

gboolean server_start(bdaddr_t *local, uint8_t channel, const char *root, int limit)
{
    int sock;

    g_assert(ctx == NULL);

    ctx = g_new0(struct server_context, 1);

    bacpy(&ctx->local_address, local);

    ctx->limit = limit;

    if (root)
        ctx->root = g_strdup(root);

    sock = server_socket(&channel);
    if (sock < 0) {
        g_free(ctx);
        ctx = NULL;
        return FALSE;
    }

    if (!add_opp_record(&ctx->local_address, channel, &ctx->record_handle)) {
        error("Adding OPP record failed");
        close(sock);
        g_free(ctx);
        ctx = NULL;
        return FALSE;
    }

    ctx->io = g_io_channel_unix_new(sock);
    g_io_channel_set_close_on_unref(ctx->io, TRUE);

    g_io_add_watch(ctx->io, G_IO_IN | G_IO_ERR | G_IO_HUP,
                   (GIOFunc)connect_callback, ctx);

    debug("Server started on channel %u", channel);

    return TRUE;
}

void server_stop(void)
{
    g_assert(ctx != NULL);

    if (!remove_record(&ctx->local_address, ctx->record_handle))
        error("Removing service record failed");

    g_slist_foreach(ctx->clients, (GFunc)client_close, NULL);
    g_slist_free(ctx->clients);

    g_io_channel_shutdown(ctx->io, FALSE, NULL);
    g_io_channel_unref(ctx->io);

    g_free(ctx->root);

    g_free(ctx);
    ctx = NULL;
}

gboolean server_has_target(ServerContext *server, const unsigned char *target,
                                unsigned int length) 
{
    /* Return FALSE for now since there's no FTP support */
    return FALSE;
}

