/**
  @file sdp.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2006 Nokia Corporation. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License, version 2, as
  published by the Free Software Foundation.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <errno.h>

#include <glib.h>

#include <bluetooth/bluetooth.h>
#include <bluetooth/sdp.h>
#include <bluetooth/sdp_lib.h>

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include "log.h"
#include "sdp.h"

gboolean add_ftp_record(bdaddr_t *dev, uint8_t chan, uint32_t *handle)
{
    error("FTP support not yet implemented");
    return FALSE;
}

gboolean add_opp_record(bdaddr_t *dev, uint8_t chan, uint32_t *handle)
{
    gboolean ret = TRUE;
    sdp_list_t *svclass_id, *pfseq, *apseq, *root;
    uuid_t root_uuid, opush_uuid, l2cap_uuid, rfcomm_uuid, obex_uuid;
    sdp_profile_desc_t profile[1];
    sdp_list_t *aproto, *proto[3];
    sdp_data_t *channel;
    uint8_t formats[] = { 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0xFF };
    void *dtds[sizeof(formats)], *values[sizeof(formats)];
    int i;
    uint8_t dtd = SDP_UINT8;
    sdp_data_t *sflist;
    sdp_record_t rec;
    sdp_session_t *session = NULL;

    session = sdp_connect(dev, BDADDR_LOCAL, 0);
    if (!session) {
        error("sdp_connect: %s", g_strerror(errno));
        return FALSE;
    }

    memset(&rec, 0, sizeof(sdp_record_t));
    rec.handle = 0xffffffff;

    sdp_uuid16_create(&root_uuid, PUBLIC_BROWSE_GROUP);
    root = sdp_list_append(0, &root_uuid);
    sdp_set_browse_groups(&rec, root);

    sdp_uuid16_create(&opush_uuid, OBEX_OBJPUSH_SVCLASS_ID);
    svclass_id = sdp_list_append(0, &opush_uuid);
    sdp_set_service_classes(&rec, svclass_id);

    sdp_uuid16_create(&profile[0].uuid, OBEX_OBJPUSH_PROFILE_ID);
    profile[0].version = 0x0100;
    pfseq = sdp_list_append(0, profile);
    sdp_set_profile_descs(&rec, pfseq);

    sdp_uuid16_create(&l2cap_uuid, L2CAP_UUID);
    proto[0] = sdp_list_append(0, &l2cap_uuid);
    apseq = sdp_list_append(0, proto[0]);

    sdp_uuid16_create(&rfcomm_uuid, RFCOMM_UUID);
    proto[1] = sdp_list_append(0, &rfcomm_uuid);
    channel = sdp_data_alloc(SDP_UINT8, &chan);
    proto[1] = sdp_list_append(proto[1], channel);
    apseq = sdp_list_append(apseq, proto[1]);

    sdp_uuid16_create(&obex_uuid, OBEX_UUID);
    proto[2] = sdp_list_append(0, &obex_uuid);
    apseq = sdp_list_append(apseq, proto[2]);

    aproto = sdp_list_append(0, apseq);
    sdp_set_access_protos(&rec, aproto);

    for (i = 0; i < sizeof(formats); i++) {
        dtds[i] = &dtd;
        values[i] = &formats[i];
    }
    sflist = sdp_seq_alloc(dtds, values, sizeof(formats));
    sdp_attr_add(&rec, SDP_ATTR_SUPPORTED_FORMATS_LIST, sflist);

    sdp_set_info_attr(&rec, "OBEX Object Push", 0, 0);

    if (sdp_device_record_register(session, dev, &rec, SDP_RECORD_PERSIST) < 0) {
        error("OBEX Object Push Service Record registration failed");
        ret = FALSE;
        goto out;
    }

    *handle = rec.handle;

    debug("Registered OPP record handle %08X", *handle);

out:
    sdp_list_free(root, NULL);
    sdp_list_free(svclass_id, NULL);
    sdp_list_free(pfseq, NULL);
    sdp_list_free(apseq, NULL);
    sdp_list_free(aproto, NULL);
    sdp_list_free(proto[0], NULL);
    sdp_list_free(proto[1], NULL);
    sdp_list_free(proto[2], NULL);
    sdp_data_free(channel);
    sdp_list_free(rec.attrlist, (sdp_free_func_t)sdp_data_free);
    sdp_list_free(rec.pattern, free);

    sdp_close(session);

    return ret;
}

gboolean remove_record(bdaddr_t *dev, uint32_t handle)
{
    uint32_t range = 0x0000ffff;
    sdp_list_t *attr;
    sdp_session_t *sess;
    sdp_record_t *rec;

    sess = sdp_connect(BDADDR_ANY, BDADDR_LOCAL, SDP_RETRY_IF_BUSY);
    if (!sess) {
        error("sdp_connect(local): %s", g_strerror(errno));
        return FALSE;
    }

    attr = sdp_list_append(0, &range);
    rec = sdp_service_attr_req(sess, handle, SDP_ATTR_REQ_RANGE, attr);
    sdp_list_free(attr, 0);

    if (!rec) {
        error("Service Record not found");
        sdp_close(sess);
        return FALSE;
    }

    if (sdp_device_record_unregister(sess, dev, rec)) {
        error("Failed to unregister service record: %s", g_strerror(errno));
        sdp_close(sess);
        return FALSE;
    }

    sdp_close(sess);

    return TRUE;
}
