/**
  @file obex.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2006 Nokia Corporation. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License, version 2, as
  published by the Free Software Foundation.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdint.h>
#include <errno.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <fcntl.h>

#include <glib.h>

#include <bluetooth/bluetooth.h>

#include <openobex/obex.h>

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include "log.h"
#include "util.h"
#include "client.h"
#include "obex.h"

typedef struct obex_connect_hdr {
    uint8_t  version;
    uint8_t  flags;
    uint16_t mtu;
} __attribute__ ((packed)) obex_connect_hdr_t;


static void cmd_connect(obex_t *handle, obex_object_t *object)
{
    ClientConnection *client = OBEX_GetCustomData(handle);
    obex_headerdata_t hv;
    uint8_t hi;
    unsigned int hlen;
    uint8_t *ptr;

    if (OBEX_ObjectGetNonHdrData(object, &ptr)
            != sizeof(obex_connect_hdr_t))
        debug("Invalid packet content.");
    else {
        obex_connect_hdr_t *nonhdrdata = (obex_connect_hdr_t *)ptr;
        uint16_t mtu = g_ntohs(nonhdrdata->mtu);
        int new_size;
        debug("Version: 0x%02x. Flags: 0x%02x  OBEX packet length: %d",
                nonhdrdata->version, nonhdrdata->flags, mtu);
        /* Leave space for headers */
        new_size = mtu - 200;
        debug("Resizing stream chunks to %d", new_size);
        client_set_tx_max(client, new_size);
    }

    while (OBEX_ObjectGetNextHeader(handle, object, &hi, &hv, &hlen)) {
#ifdef DEBUG
        char *str;
#endif

        switch (hi) {
            case OBEX_HDR_TARGET:
#ifdef DEBUG
                str = bytestr(hv.bs, hlen);
                debug("CONNECT target: %s", str);
                g_free(str);
#endif
                if (!client_allow_target(client, hv.bs, hlen)) {
                    debug("Refusing target!");
                    OBEX_ObjectSetRsp(object, OBEX_RSP_FORBIDDEN, OBEX_RSP_FORBIDDEN);
                    OBEX_ObjectReParseHeaders(handle, object);
                    return;
                }

                break;
            default:
                debug("CONNECT skipped header 0x%02X", hi);
                break;
        }
    }

    OBEX_ObjectReParseHeaders(handle, object);
    
    OBEX_ObjectSetRsp(object, OBEX_RSP_CONTINUE, OBEX_RSP_SUCCESS);
}

static void cmd_get(obex_t *handle, obex_object_t *object)
{
    obex_headerdata_t hv;
    uint8_t hi;
    unsigned int hlen;
    gboolean have_name = FALSE, have_type = FALSE;

    debug("GET");

    while (OBEX_ObjectGetNextHeader(handle, object, &hi, &hv, &hlen)) {
        switch (hi) {
            case OBEX_HDR_NAME:
                have_name = TRUE;
                break;
            case OBEX_HDR_TYPE:
                have_type = TRUE;
                break;
            default:
                debug("Skipped GET header 0x%02X", hi);
                break;
        }
    }

    OBEX_ObjectReParseHeaders(handle, object);

    if (have_type && have_name)
        OBEX_ObjectSetRsp(object, OBEX_RSP_FORBIDDEN, OBEX_RSP_FORBIDDEN);
    else
        OBEX_ObjectSetRsp(object, OBEX_RSP_NOT_FOUND, OBEX_RSP_NOT_FOUND);
}

static void cmd_put(obex_t *handle, obex_object_t *object, ClientConnection *client)
{
    ObjectTransfer *xfer;

    xfer = client_get_transfer(client, OBEX_CMD_PUT);

    g_assert(xfer);

    if (xfer->is_delete) {
        debug("Refusing delete operation");
        xfer->success = FALSE;
        OBEX_ObjectSetRsp(object, OBEX_RSP_FORBIDDEN, OBEX_RSP_FORBIDDEN);
    }
}

static gboolean write_to_file(ClientConnection *client, unsigned char *buf, size_t len, int *err)
{
    ObjectTransfer *xfer;
    int written = 0;
   
    xfer = client_get_transfer(client, OBEX_CMD_PUT);

    if (xfer->fd < 0) {
        xfer->fd = open(xfer->filename, O_WRONLY | O_CREAT, 0600);
        if (xfer->fd < 0) {
            if (err)
                *err = errno;
            error("Unable to open %s for writing: %s", xfer->filename, g_strerror(errno));
            return FALSE;
        }
        debug("%s opened for writing", xfer->filename);

    }

    while (written < len) {
        int ret;
        
        ret = write(xfer->fd, buf + written, len - written);
        if (ret < 0) {
            if (err)
                *err = errno;
            error("write: %s", g_strerror(errno));
            return FALSE;
        }

        written += ret;
    }

    xfer->counter += len;

    xfer->data_len = 0;

    return TRUE;
}

static void obex_readstream(obex_t *handle, obex_object_t *object)
{
    int actual;
    ClientConnection *client = OBEX_GetCustomData(handle);
    ObjectTransfer *xfer;
    const uint8_t *buf;

    xfer = client_get_transfer(client, OBEX_CMD_PUT);
    if (!xfer) {
        /* This means that a packet with invalid opcode arrived */
        OBEX_ObjectSetRsp(object, OBEX_RSP_BAD_REQUEST, OBEX_RSP_BAD_REQUEST);
        return;
    }

    xfer->is_delete = FALSE;

    actual = OBEX_ObjectReadStream(handle, object, &buf);
    if (actual < 0) {
        debug("Error on OBEX stream");
        return;
    }

    if (!xfer->filename) {
        debug("Storing %d bytes to temporary buffer", actual);
        
        if (actual + xfer->data_len > xfer->buf_size) {
            debug("No space in buffer to store incoming data!");
            return;
        }

        memcpy(xfer->buf + xfer->data_len, buf, actual);
        xfer->data_len += actual;
    }
    else if (!write_to_file(client, (unsigned char *)buf, actual, NULL)) {
        error("Unable to write to %s", xfer->filename);
        client_transfer_set_success(client, FALSE);
        OBEX_ObjectSetRsp(object, OBEX_RSP_FORBIDDEN, OBEX_RSP_FORBIDDEN);
    }
}

static void check_put_headers(obex_t *handle, obex_object_t *object)
{
    ClientConnection *client = OBEX_GetCustomData(handle);
    ObjectTransfer *xfer;
    obex_headerdata_t hv;
    uint8_t hi;
    unsigned int hlen;

    xfer = client_get_transfer(client, OBEX_CMD_PUT);
    if (!xfer) {
        /* This means that a packet with invalid opcode arrived */
        OBEX_ObjectSetRsp(object, OBEX_RSP_BAD_REQUEST, OBEX_RSP_BAD_REQUEST);
        return;
    }

    /* Just return if already parsed */
    if (xfer->name || xfer->type)
        return;

    debug("PUT headers:");

    while (OBEX_ObjectGetNextHeader(handle, object, &hi, &hv, &hlen))   {
        switch (hi) {
            case OBEX_HDR_BODY:
                debug("\tBODY header, length=%u", hlen);
                if (hlen == 0) {
                    debug("Refusing delete operation");
                    OBEX_ObjectSetRsp(object, OBEX_RSP_FORBIDDEN, OBEX_RSP_FORBIDDEN);
                    return;
                }

                break;

            case OBEX_HDR_NAME:
                if (xfer->name) {
                    debug("Ignoring multiple name headers");
                    break;
                }

                xfer->name = utf16_to_utf8(hv.bs, hlen);

                if (xfer->name)
                    debug("  Name: %s", xfer->name);
                else
                    debug("Unable to decode name header");

                break;

            case OBEX_HDR_TYPE:
                if (xfer->type) {
                    debug("Ignoring multiple type headers");
                    break;
                }

                /* Ensure null termination */
                if (hv.bs[hlen - 1] != '\0')
                    break;

                if (!g_utf8_validate((const char *) hv.bs, -1, NULL)) {
                    debug("Invalid type header: %s", hv.bs);
                    break;
                }

                xfer->type = g_strdup((const char *)hv.bs);

                debug("  Type: %s", xfer->type);

                break;

            case OBEX_HDR_LENGTH:
                if (xfer->length != 0) {
                    debug("Ignoring multiple length headers");
                    break;
                }

                xfer->length = hv.bq4;

                debug("  Length: %d", xfer->length);

                break;

            case OBEX_HDR_TIME:
                if (xfer->time) {
                    debug("Ignoring multiple time headers");
                    break;
                }

                xfer->time = parse_iso8601((char*)hv.bs, hlen);

                debug("  Time: %s", chomp(ctime(&xfer->time)));
                
                break;
            case OBEX_HDR_DESCRIPTION:
                if (xfer->dscr) {
                    debug("Ignoring multiple description headers");
                    break;
                }

                xfer->dscr = utf16_to_utf8(hv.bs, hlen);

                if (xfer->dscr)
                    debug("  Description: %s", xfer->dscr);
                else
                    debug("Unable to decode description header");

                break;

            case OBEX_HDR_COUNT:
                xfer->count = hv.bq4;

                debug("  Count: %d", xfer->count);

                if (xfer->count > 1) {
                    debug("Rejecting PUT with more than one object");
                    OBEX_ObjectSetRsp(object, OBEX_RSP_FORBIDDEN, OBEX_RSP_FORBIDDEN);
                    return;
                }

                break;

            default:
                debug("Skipped header %02X\n", hi);
                break;
        }
    }

    OBEX_ObjectReParseHeaders(handle, object);

    xfer->filename = client_authorize(client);
    if (!xfer->filename) {
        OBEX_ObjectSetRsp(object, OBEX_RSP_FORBIDDEN, OBEX_RSP_FORBIDDEN);
        return;
    }

    debug("Transfer authorized with filename %s", xfer->filename);

    client_transfer_start(client);

    if (xfer->data_len) {
	    int err;

	    if (!write_to_file(client, xfer->buf, xfer->data_len, &err)) {
                client_transfer_set_success(client, FALSE);

                switch(err) {
                    case ENOSPC:
                        OBEX_ObjectSetRsp(object, OBEX_RSP_REQ_ENTITY_TOO_LARGE,
                                          OBEX_RSP_REQ_ENTITY_TOO_LARGE);
                        break;
                    default:
                        OBEX_ObjectSetRsp(object, OBEX_RSP_FORBIDDEN, OBEX_RSP_FORBIDDEN);
                        break;
                }
                return;
            }

	    xfer->data_len = 0;
    }

    OBEX_ObjectSetRsp(object, OBEX_RSP_CONTINUE, OBEX_RSP_SUCCESS);
}

void obex_event_handler(obex_t *handle, obex_object_t *object, int mode,
                        int event, int obex_cmd, int obex_rsp)
{
    ClientConnection *client = OBEX_GetCustomData(handle);

    debug("client_event_handler: mode=%d, event=%d, obex_cmd=%d, obex_rsp=%d",
            mode, event, obex_cmd, obex_rsp);

    switch (event) {
        case OBEX_EV_PROGRESS:
            debug("PROGRESS");
            if (!client_progress(client)) {
                debug("client_progress returned FALSE, refusing transfer");
                client_transfer_set_success(client, FALSE);
                OBEX_ObjectSetRsp(object, OBEX_RSP_FORBIDDEN, OBEX_RSP_FORBIDDEN);
            }
            break;

        case OBEX_EV_ABORT:
            debug("ABORT");
            client_transfer_set_success(client, FALSE);
            client_transfer_complete(client);
            OBEX_ObjectSetRsp(object, OBEX_RSP_SUCCESS, OBEX_RSP_SUCCESS);
            break;

        case OBEX_EV_REQDONE:
            debug("REQDONE");
            switch (obex_cmd) {
                case OBEX_CMD_DISCONNECT:
                    OBEX_TransportDisconnect(handle);
                    break;
                case OBEX_CMD_PUT:
                case OBEX_CMD_GET:
                    client_transfer_complete(client);
                    break;
                default:
                    break;
            }
            break;

        case OBEX_EV_REQHINT:
            debug("REQHINT");
            switch (obex_cmd) {
                case OBEX_CMD_PUT:
                    OBEX_ObjectReadStream(handle, object, NULL);
                    /* No break on purpose */
                case OBEX_CMD_GET:
                case OBEX_CMD_CONNECT:
                case OBEX_CMD_DISCONNECT:
                    OBEX_ObjectSetRsp(object, OBEX_RSP_CONTINUE, OBEX_RSP_SUCCESS);
                    break;
                default:
                    OBEX_ObjectSetRsp(object, OBEX_RSP_NOT_IMPLEMENTED,
                                      OBEX_RSP_NOT_IMPLEMENTED);
                    break;
            }
            break;

        case OBEX_EV_REQCHECK:
            debug("REQCHECK");
            switch (obex_cmd) {
                case OBEX_CMD_PUT:
                    check_put_headers(handle, object);
                default:
                    break;
            }
            break;

        case OBEX_EV_REQ:
            switch (obex_cmd) {
                case OBEX_CMD_DISCONNECT:
                    debug("DISCONNECT");
                    OBEX_ObjectSetRsp(object, OBEX_RSP_SUCCESS, OBEX_RSP_SUCCESS);
                    break;
                    
                case OBEX_CMD_CONNECT:
                    debug("CONNECT");
                    cmd_connect(handle, object);
                    break;
                    
                case OBEX_CMD_SETPATH:
                    debug("SETPATH");
                    OBEX_ObjectSetRsp(object, OBEX_RSP_FORBIDDEN, OBEX_RSP_FORBIDDEN);
                    break;

                case OBEX_CMD_GET:
                    cmd_get(handle, object);
                    break;

                case OBEX_CMD_PUT:
                    debug("PUT");
                    check_put_headers(handle, object);
                    cmd_put(handle, object, client);
                    break;

                default:
                    debug("Unknown request: 0x%X", obex_cmd);
                    OBEX_ObjectSetRsp(object, OBEX_RSP_NOT_IMPLEMENTED, OBEX_RSP_NOT_IMPLEMENTED);
                    break;
            }
            break;

        case OBEX_EV_STREAMAVAIL:
            debug("STREAMAVAIL");
            obex_readstream(handle, object);
            break;

        case OBEX_EV_LINKERR:
            debug("LINKERR");
            break;

        case OBEX_EV_PARSEERR:
            debug("PARSEERR");
            break;

        case OBEX_EV_UNEXPECTED:
            debug("UNEXPECTED");
            break;

        default:
            debug("Unknown event %d", event);
            break;
    }
}

