/**
  @file main.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2006 Nokia Corporation. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License, version 2, as
  published by the Free Software Foundation.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdint.h>
#include <string.h>
#include <getopt.h>
#include <signal.h>
#include <errno.h>
#include <ctype.h>
#include <pwd.h>
#include <sys/types.h>
#include <sys/socket.h>

#include <glib.h>
#include <glib-object.h>

#include <bluetooth/bluetooth.h>
#include <bluetooth/hci.h>
#include <bluetooth/hci_lib.h>

#include <dbus/dbus.h>

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include "log.h"
#include "dbus.h"
#include "daemon.h"
#include "server.h"

#define DEFAULT_CLIENT_LIMIT 5

#define PIDFILE "/tmp/obexsrv.pid"

static char *program_name;

static int signal_pipe[2];

/** Command line config options */
static struct {
    gboolean  daemon;    /**< Enter daemon mode (detach to background) */
    gboolean  syslog;    /**< Force logging to syslog */
    gchar    *root;      /**< Default directory to store files into */
    gchar    *user;      /**< User to run as */
    bdaddr_t  local_dev; /**< Which HCI device to use */
    uint8_t   channel;   /**< Which RFCOMM channel to bind to */
    gint      limit;     /**< Max number of client connections */
} cfg;

static GMainLoop *event_loop = NULL;

/* For getopt */
static struct option const long_options[] = {
    {"deamon",   no_argument,       0, 'd'},
    {"syslog",   no_argument,       0, 'l'},
    {"help",     no_argument,       0, 'h'},
    {"version",  no_argument,       0, 'V'},
    {"root",     required_argument, 0, 'r'},
    {"device",   required_argument, 0, 'i'},
    {"channel",  required_argument, 0, 'c'},
    {"limit",    required_argument, 0, 'L'},
    {"user",     required_argument, 0, 'u'},
    {NULL, 0, NULL, 0}
};

/** Print usage information
 * @param status exit with status
 */
static void usage(int status)
{
    printf("%s - OBEX Server %s\n", program_name, VERSION);

    printf("Compilation flags: ");
#ifdef DEBUG
    printf("+DEBUG ");
#else
    printf("-DEBUG ");
#endif

    printf(
     "\nUsage: %s [OPTION]...\n"
     "Options:\n"
     "-h, --help                 Display this help and exit\n"
     "-V, --version              Output version information and exit\n"
     "-d, --daemon               Send process to the background\n"
     "-r, --root                 Default directory to store files into\n"
     "-u, --user                 User to run as\n"
     "-i, --device               Which HCI device to use (default: hci0)\n"
     "-c, --channel              RFCOMM channel to listen on\n"
     "-L, --limit                Limit number of client connections\n"
     "\n", program_name);

    exit(status);
}

/** Process commandline options.
 * @param argc Parameter given to main()
 * @param argv Parameter given to main()
 * @returns Index of first non-option argument
 */
static gint decode_switches(int argc, char *argv[])
{
    gint c;

    memset(&cfg, 0, sizeof(cfg));

    bacpy(&cfg.local_dev, BDADDR_ANY);
    cfg.limit = DEFAULT_CLIENT_LIMIT;

    while ((c = getopt_long(argc, argv,
                    "h"   /* help      */
                    "V"   /* version   */
                    "d"   /* daemon    */
                    "l"   /* syslog    */
                    "i:"  /* device    */
                    "r:"  /* root      */
                    "c:"  /* channel   */
                    "L:"  /* limit     */
                    "u:"  /* user      */
                    ,long_options, NULL)) != EOF) {
        switch (c) {
            case 'V':
                printf("Bluetooth Connection Daemon %s\n", VERSION);
                exit(EXIT_SUCCESS);

            case 'h':
                usage(EXIT_SUCCESS);

            case 'd':
                cfg.daemon = TRUE;
                cfg.syslog = TRUE;
                break;

            case 'l':
                cfg.syslog = TRUE; 
                break;

            case 'i':
                if (!strncmp(optarg, "hci", 3)) {
                    if (hci_devba(atoi(optarg + 3), &cfg.local_dev)) {
                        error("%s is not a valid hci device name", optarg);
                        exit(EXIT_FAILURE);
                    }
                }
                else {
                    if (str2ba(optarg, &cfg.local_dev)) {
                        error("%s is not a valid address", optarg);
                        exit(EXIT_FAILURE);
                    }
                }
                break;  

            case 'r':
                cfg.root = g_strdup(optarg);
                break;

            case 'c':
                cfg.channel = (uint8_t) strtol(optarg, NULL, 0);
                break;

            case 'L':
                cfg.limit = strtol(optarg, NULL, 0);
                break;

            case 'u':
                cfg.user = strdup(optarg);
                break;

            default:
                usage(EXIT_FAILURE);
        }
    }

    return optind;
}

static void handle_signal(int sig)
{
    report("Signal %d received: %s.", sig, g_strsignal(sig));
    g_main_loop_quit(event_loop);
}

static void exit_cleanup(void)
{
    server_stop();
    (void) remove_pid(PIDFILE);
    report("Exiting.");
}

static void signal_handler(int sig)
{   
        send(signal_pipe[1], &sig, sizeof(sig), MSG_DONTWAIT);
}   

static gboolean signal_cb(GIOChannel *chan, GIOCondition cond, gpointer data)
{
    int fd;

    if (cond != G_IO_IN) {
        error("signal_cb: cond != G_IO_IN");
        handle_signal(SIGTERM);
        g_io_channel_unref(chan);
        return FALSE;
    }

    fd = g_io_channel_unix_get_fd(chan);
    g_assert(fd >= 0);

    for (;;) {
        int sig;

        if (recv(fd, &sig, sizeof(sig), MSG_DONTWAIT) != sizeof(sig))
            break;

        handle_signal(sig);
    }

    return TRUE;
}

static void bind_unix_signals(void)
{
    GIOChannel *gio;

    if (socketpair(AF_UNIX, SOCK_STREAM, 0, signal_pipe) < 0)
        die("socketpair: %s", g_strerror(errno));

    gio = g_io_channel_unix_new(signal_pipe[0]);
    g_io_add_watch(gio, G_IO_IN | G_IO_PRI | G_IO_ERR | G_IO_HUP,
                   signal_cb, NULL);

    if (signal(SIGINT, signal_handler) == SIG_ERR)
        die("signal(SIGINT) failed");
    if (signal(SIGTERM, signal_handler) == SIG_ERR)
        die("signal(SIGTERM) failed");
}

static gboolean change_user(const char *user)
{
    struct passwd *pw;

    if (geteuid() != 0) {
        error("Must be root to switch users");
        return FALSE;
    }

    pw = getpwnam(user);
    if (!pw) {
        error("getpwnam(%s): %s", user, g_strerror(errno));
        return FALSE;
    }

    if (setgid(pw->pw_gid) < 0 || setuid(pw->pw_uid) < 0) {
        error("setgid or setuid failed: %s", g_strerror(errno));
        return FALSE;
    }

    return TRUE;
}

int main(int argc, char *argv[])
{
    gint i, old_pid;

    program_name = argv[0];
    i = decode_switches(argc, argv);

    open_log("obexsrv", cfg.syslog);

    old_pid = check_pid(PIDFILE);
    if (old_pid)
        die("Unable to run: another instance running (PID %d)", old_pid);

    if (cfg.daemon && daemon(0, 0) < 0)
        die("daemonizing failed: %s", g_strerror(errno));

    g_type_init();

    if (cfg.user && !change_user(cfg.user))
        die("Unable to change user to %s", cfg.user);

    if (!server_start(&cfg.local_dev, cfg.channel, cfg.root, cfg.limit))
        die("Unable to start server");

    if (!init_dbus())
        die("D-BUS init failed");

    write_pid(PIDFILE);
    atexit(exit_cleanup);

    bind_unix_signals();

    event_loop = g_main_loop_new(NULL, FALSE);

    report("OBEX Server %s started.", VERSION);
    g_main_loop_run(event_loop);

    deinit_dbus();

    g_main_loop_unref(event_loop);

    exit(EXIT_SUCCESS);
}

