/**
  @file dbus.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2006 Nokia Corporation. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License, version 2, as
  published by the Free Software Foundation.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <string.h>
#include <errno.h>
#include <glib.h>

#include <dbus/dbus.h>
#include <dbus/dbus-glib-lowlevel.h>

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include "log.h"
#include "client.h"
#include "dbus-helper.h"
#include "dbus.h"

#define AUTH_TIMEOUT (1000 * 60)

#define OBEXSRV_SERVICE             "com.nokia.ObexServer"
#define OBEXSRV_PATH                "/com/nokia/ObexServer"
#define OBEXSRV_REQ_IF              "com.nokia.ObexServer"
#define OBEXSRV_XFER_IF             "com.nokia.ObexServer.Transfer"
                                    
#define OBEXSRV_ERR                 "com.nokia.ObexServer.Error"
#define OBEXSRV_ERR_NOT_REGISTERED  OBEXSRV_ERR "NotRegistered"
#define OBEXSRV_ERR_REGISTERED      OBEXSRV_ERR "Registered"
#define OBEXSRV_ERR_FORBIDDEN       OBEXSRV_ERR "Forbidden"
#define OBEXSRV_ERR_REFUSED         OBEXSRV_ERR "Refused"
#define OBEXSRV_ERR_NOT_FOUND       OBEXSRV_ERR "NotFound"
#define OBEXSRV_ERR_INVALID_ARGS    OBEXSRV_ERR "InvalidArguments"

/* Interface which registered handlers implement */
#define OBEXSRV_HANDLER_IF         "com.nokia.ObexServer.Handler"

typedef struct {
    char *object_path;
    char *bus_name;
} HandlerLocation;

typedef struct {
    char *object_path;
    ClientConnection *client;
    ObjectTransfer *xfer;
    gboolean canceled;
} TransferInfo;

typedef DBusHandlerResult (*handler_func)(DBusMessage *message,
                                          DBusConnection *connection);

typedef struct {
    const char *interface;
    const char *name;
    handler_func func;
} method_handler_t;

static HandlerLocation *handler = NULL;

/* List of TransferInfo objects */
static GSList *transfers = NULL;

static DBusConnection *bus = NULL;

static DBusHandlerResult reg_handler_request(DBusMessage    *message,
                                          DBusConnection *connection);
static DBusHandlerResult unreg_handler_request(DBusMessage    *message,
                                            DBusConnection *connection);

static void srv_client_disconnect(ClientConnection *client, TransferInfo *info);

static method_handler_t handlers[] = {
    { OBEXSRV_REQ_IF, "RegisterHandler", reg_handler_request },
    { OBEXSRV_REQ_IF, "UnregisterHandler", unreg_handler_request },
    { NULL }
};

static void remove_handler(void)
{
    debug("remove_handler: %s, %s", handler->bus_name, handler->object_path);
    g_free(handler->object_path);
    g_free(handler->bus_name);
    g_free(handler);
    handler = NULL;
}

static void handler_exit_callback(const char *name, HandlerLocation *handler)
{
    debug("Handler %s exited without unregistering", handler->bus_name);

    g_assert(handler == handler);

    remove_handler();
}

static DBusHandlerResult unreg_handler_request(DBusMessage *message,
                                            DBusConnection *connection)
{
    DBusError derr;
    DBusMessage *reply;
    const char *path;

    if (!handler)
        goto not_registered;

    dbus_error_init(&derr);
    dbus_message_get_args(message, &derr,
                          DBUS_TYPE_STRING, &path,
                          DBUS_TYPE_INVALID);
    if (dbus_error_is_set(&derr)) {
        reply = dbus_message_new_error(message, OBEXSRV_ERR_INVALID_ARGS,
                                       derr.message);
        dbus_error_free(&derr);
        if (!reply)
            return DBUS_HANDLER_RESULT_NEED_MEMORY;

        goto out;
    }

    if (g_str_equal(dbus_message_get_sender(message), handler->bus_name)
            && g_str_equal(path, handler->object_path)) {
        reply = dbus_message_new_method_return(message);
        if (!reply)
            return DBUS_HANDLER_RESULT_NEED_MEMORY;
        remove_name_listener(connection, handler->bus_name,
                             (name_cb)handler_exit_callback, handler);                             
        remove_handler();
        goto out;
    }
    
not_registered:
    reply = dbus_message_new_error(message, OBEXSRV_ERR_NOT_REGISTERED,
                                   "No such handler registered");
    if (!reply)
        return DBUS_HANDLER_RESULT_NEED_MEMORY;
out:
    dbus_connection_send(bus, reply, NULL);
    dbus_message_unref(reply);
    return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusHandlerResult reg_handler_request(DBusMessage *message,
                                          DBusConnection *connection)
{
    DBusError derr;
    DBusMessage *reply;
    const char *path;

    if (handler) {
        reply = dbus_message_new_error(message, OBEXSRV_ERR_REGISTERED,
                                       "A handler is already registered");
        if (!reply)
            return DBUS_HANDLER_RESULT_NEED_MEMORY;

        goto out;
    }

    dbus_error_init(&derr);
    dbus_message_get_args(message, &derr,
                          DBUS_TYPE_STRING, &path,
                          DBUS_TYPE_INVALID);
    if (dbus_error_is_set(&derr)) {
        reply = dbus_message_new_error(message, OBEXSRV_ERR_INVALID_ARGS,
                                       derr.message);

        dbus_error_free(&derr);

        if (!reply)
            return DBUS_HANDLER_RESULT_NEED_MEMORY;

        goto out;
    }

    reply = dbus_message_new_method_return(message);
    if (!reply)
        return DBUS_HANDLER_RESULT_NEED_MEMORY;

    handler = g_new0(HandlerLocation, 1);
    handler->object_path = g_strdup(path);
    handler->bus_name = g_strdup(dbus_message_get_sender(message));

    add_name_listener(connection, handler->bus_name,
                      (name_cb)handler_exit_callback, handler);

    debug("Handler registered at %s, %s",
            handler->bus_name, handler->object_path);

out:
    dbus_connection_send(bus, reply, NULL);
    dbus_message_unref(reply);
    return DBUS_HANDLER_RESULT_HANDLED;
}

/* Handler for D-Bus method calls */
static DBusHandlerResult obexsrv_method_handler(DBusConnection *connection,
                                                DBusMessage    *message,
                                                void           *user_data)
{
    method_handler_t *handler;

    for (handler = handlers; handler->interface != NULL; handler++) {
        if (dbus_message_is_method_call(message, handler->interface,
                                        handler->name)) {
            debug("Received %s", handler->name);
            return handler->func(message, connection);
        }
    }

    return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}


static DBusObjectPathVTable obexsrv_req_vtable = {
    .message_function    = obexsrv_method_handler,
    .unregister_function = NULL
};

static const char *name_error_string(int ret)
{
    switch (ret) {
        case DBUS_REQUEST_NAME_REPLY_IN_QUEUE:
            return "in queue";
        case DBUS_REQUEST_NAME_REPLY_EXISTS:
            return "name exists";
        case DBUS_REQUEST_NAME_REPLY_ALREADY_OWNER:
            return "already owner";
        default:
            return "";
    }
}

/* Create bindings for D-Bus handlers */
gboolean init_dbus(void)
{
    DBusError derr;
    int ret;

    g_assert(bus == NULL);

    dbus_error_init(&derr);
    bus = dbus_bus_get(DBUS_BUS_SYSTEM, &derr);
    if (bus == NULL) {
        error("System D-BUS connection failed: %s", derr.message);
        dbus_error_free(&derr);
        return FALSE;
    }

    dbus_connection_setup_with_g_main(bus, NULL);

    dbus_error_init(&derr);
    ret = dbus_bus_request_name(bus, OBEXSRV_SERVICE, 0, &derr);
    if (ret != DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER) {
        error("Could not aquire D-BUS name '%s': %s (ret: %d)",
                OBEXSRV_SERVICE, name_error_string(ret), ret);
        if (dbus_error_is_set(&derr)) {
            debug("%s", derr.message);
            dbus_error_free(&derr);
        }
        return FALSE;
    }

    if (!dbus_connection_register_object_path(bus, OBEXSRV_PATH,
                                              &obexsrv_req_vtable,
                                              NULL)) {
        error("dbus_connection_register_object_path failed");
        dbus_bus_release_name(bus, OBEXSRV_SERVICE, NULL);
        return FALSE;
    }

    return TRUE;
}

void deinit_dbus(void)
{
    g_assert(bus != NULL);
    dbus_connection_unregister_object_path(bus, OBEXSRV_PATH);
    dbus_bus_release_name(bus, OBEXSRV_SERVICE, NULL);
    dbus_connection_unref(bus);
    bus = NULL;
}

static DBusHandlerResult transfer_method_handler(DBusConnection *connection,
                                                 DBusMessage    *message,
                                                 void           *user_data)
{
    DBusMessage *reply;
    const char *path;
    TransferInfo *info = user_data;

    if (!dbus_message_is_method_call(message, OBEXSRV_XFER_IF, "Cancel"))
        return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;

    path = dbus_message_get_path(message);

    g_assert(info);

    if (!handler || strcmp(handler->bus_name, dbus_message_get_sender(message))) {
        reply = dbus_message_new_error(message, OBEXSRV_ERR_FORBIDDEN,
                                       "You are not the owner of this transfer");
        if (!reply) {
            error("Unable to allocate new message");
            return DBUS_HANDLER_RESULT_NEED_MEMORY;
        }

        goto out;
    }

    reply = dbus_message_new_method_return(message);
    if (!reply)
        return DBUS_HANDLER_RESULT_NEED_MEMORY;

    debug("Transfer %s was canceled", path);

    info->canceled = TRUE;

out:
    dbus_connection_send(connection, reply, NULL);
    dbus_message_unref(reply);
    return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusObjectPathVTable transfer_req_vtable = {
    .message_function    = transfer_method_handler,
    .unregister_function = NULL
};

static gboolean add_transfer(const char *path, ClientConnection *client,
                             ObjectTransfer *xfer)
{
    TransferInfo *info;

    info = g_new0(TransferInfo, 1);
    info->object_path = g_strdup(path);
    info->client = client;
    info->xfer = xfer;

    if (!dbus_connection_register_object_path(bus, path,
                                              &transfer_req_vtable,
                                              info)) {
        error("dbus_connection_register_object_path(%s) failed", path);
        g_free(info);
        return FALSE;
    }

    transfers = g_slist_append(transfers, info);

    client_add_disconnect_callback(client,
                                   (client_dc_cb_t)srv_client_disconnect,
                                   info);

    return TRUE;
}

static void remove_transfer(TransferInfo *info)
{
    g_assert(g_slist_find(transfers, info));

    client_remove_disconnect_callback(info->client,
                                      (client_dc_cb_t)srv_client_disconnect,
                                      info);

    transfers = g_slist_remove(transfers, info);

    dbus_connection_unregister_object_path(bus, info->object_path);

    g_free(info->object_path);
    g_free(info);
}

static void srv_client_disconnect(ClientConnection *client, TransferInfo *info)
{
    remove_transfer(info);
}

char *srv_dbus_auth(ClientConnection *client,
                    ObjectTransfer   *xfer,
                    gpointer          user_data)
{
    DBusError derr;
    char *xfer_path = NULL, *name, *type;
    const char *bt_addr, *filename = NULL;
    DBusMessage *msg = NULL, *reply = NULL;

    if (!handler)
        return NULL;

    msg = dbus_message_new_method_call(handler->bus_name,
                                       handler->object_path,
                                       OBEXSRV_HANDLER_IF,
                                       "Authorize");
    if (!msg) {
        error("Out of memory during dbus_message_new_method_call");
        return NULL;
    }

    xfer_path = g_strdup_printf("%s/Transfer/%p", OBEXSRV_PATH, xfer);
    bt_addr   = client_get_bt_address(client);
    name      = xfer->name ? xfer->name : "";
    type      = xfer->type ? xfer->type : "";

    append_dbus_args(msg,
                     DBUS_TYPE_STRING, &xfer_path,
                     DBUS_TYPE_STRING, &bt_addr,
                     DBUS_TYPE_STRING, &name,
                     DBUS_TYPE_STRING, &type,
                     DBUS_TYPE_INT32, &xfer->length,
                     DBUS_TYPE_INT32, &xfer->time,
                     DBUS_TYPE_INVALID);

    dbus_error_init(&derr);
    reply = dbus_connection_send_with_reply_and_block(bus, msg, AUTH_TIMEOUT, &derr);
    if (dbus_error_is_set(&derr))
        goto out;

    dbus_error_init(&derr);
    dbus_message_get_args(reply, &derr,
                          DBUS_TYPE_STRING, &filename,
                          DBUS_TYPE_INVALID);
    if (dbus_error_is_set(&derr))
        goto out;

    if (!add_transfer(xfer_path, client, xfer))
        filename = NULL;

out:
    g_free(xfer_path);
    if (msg)
        dbus_message_unref(msg);
    if (reply)
        dbus_message_unref(reply);
    if (dbus_error_is_set(&derr)) {
        debug("%s", derr.message);
        dbus_error_free(&derr);
    }

    if (filename)
        return g_strdup(filename);
    else
        return NULL;
}

gboolean srv_dbus_progr(ClientConnection *client,
                        ObjectTransfer   *xfer,
                        gpointer          user_data)
{
    GSList *l;
    DBusMessage *msg;
    TransferInfo *info = NULL;

    for (l = transfers; l != NULL; l = l->next) {
        TransferInfo *i = l->data;
        if (i->xfer == xfer) {
            info = i;
            break;
        }
    }

    g_assert(info);

    if (info->canceled)
        return FALSE;

    msg = dbus_message_new_signal(info->object_path, OBEXSRV_XFER_IF, "Progress");
    if (!msg) {
        error("Out of memory during dbus_message_new_signal");
        return TRUE;
    }

    append_dbus_args(msg,
                     DBUS_TYPE_INT32, &xfer->counter,
                     DBUS_TYPE_INT32, &xfer->length,
                     DBUS_TYPE_INVALID);

    dbus_connection_send(bus, msg, NULL);
    dbus_message_unref(msg);

    return TRUE;
}

void srv_dbus_xfer(ClientConnection *client,
                   ObjectTransfer   *xfer,
                   gboolean          started,
                   gboolean          success,
                   gpointer          user_data)
{
    GSList *l;
    DBusMessage *msg;
    TransferInfo *info = NULL;

    for (l = transfers; l != NULL; l = l->next) {
        TransferInfo *i = l->data;
        if (i->xfer == xfer) {
            info = i;
            break;
        }
    }

    g_assert(info);

    msg = dbus_message_new_signal(info->object_path, OBEXSRV_XFER_IF,
                                  started ? "Started" : "Completed");
    if (!msg) {
        error("Out of memory during dbus_message_new_signal");
        goto done;
    }

    append_dbus_args(msg,
                     DBUS_TYPE_BOOLEAN, &success,
                     DBUS_TYPE_INVALID);
    
    dbus_connection_send(bus, msg, NULL);
    dbus_message_unref(msg);

done:
    /* Remove transfer if it was completed */
    if (!started)
        remove_transfer(info);

}
