/**
  @file dbus-helper.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2006 Nokia Corporation. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License, version 2, as
  published by the Free Software Foundation.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <glib.h>

#include <dbus/dbus.h>

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include "log.h"
#include "dbus-helper.h"

static gboolean name_listener_initialized = FALSE;

static GSList *name_listeners = NULL;

typedef struct {
    name_cb  func;
    gpointer user_data;
} NameCallback;

typedef struct {
    char     *name;
    GSList   *callbacks;
} NameData;

static gint name_data_cmp(NameData *data, const char *name)
{
    return strcmp(data->name, name);
}

static NameData *name_data_find(const char *name)
{
    GSList *match;

    match = g_slist_find_custom(name_listeners, name, (GCompareFunc)name_data_cmp);
    if (!match)
        return NULL;

    return match->data;
}

static NameCallback *name_callback_find(GSList *callbacks, name_cb func, gpointer user_data)
{
    GSList *l;

    for (l = callbacks; l != NULL; l = l->next) {
        NameCallback *cb = l->data;

        if (cb->func == func && cb->user_data == user_data)
            return cb;
    }

    return NULL;
}

static gboolean name_data_add(const char *name, name_cb func, gpointer user_data)
{
    gboolean first = FALSE;
    NameData *data;
    NameCallback *cb;

    cb = g_new0(NameCallback, 1);
    cb->func = func;
    cb->user_data = user_data;

    data = name_data_find(name);
    if (!data) {
        data = g_new0(NameData, 1);
        data->name = g_strdup(name);
        name_listeners = g_slist_append(name_listeners, data);
        first = TRUE;
    }

    data->callbacks = g_slist_append(data->callbacks, cb);

    return first;
}

static void name_data_free(NameData *data)
{
    g_free(data->name);
    g_free(data);
}

static void name_data_remove(const char *name, name_cb func, gpointer user_data)
{
    NameData *data;
    NameCallback *cb;

    data = name_data_find(name);
    if (!data)
        return;

    cb = name_callback_find(data->callbacks, func, user_data);
    if (!cb)
        return;

    data->callbacks = g_slist_remove(data->callbacks, cb);
    g_free(cb);

    if (!data->callbacks) {
        name_listeners = g_slist_remove(name_listeners, data);
        name_data_free(data);
    }
}

static void name_call_listener(NameCallback *cb, const char *name)
{
    cb->func(name, cb->user_data);
}

static DBusHandlerResult name_exit_filter(DBusConnection *connection,
                                          DBusMessage *message,
                                          void *user_data)
{
    if (dbus_message_is_signal(message,
                               DBUS_INTERFACE_DBUS,
                               "NameOwnerChanged")) {
        NameData *data;
        char *name, *old, *new;

        if (!dbus_message_get_args(message, NULL,
                                   DBUS_TYPE_STRING, &name,
                                   DBUS_TYPE_STRING, &old,
                                   DBUS_TYPE_STRING, &new,
                                   DBUS_TYPE_INVALID)) {
            error("Invalid arguments for NameOwnerChanged signal");
            goto out;
        }

        /* We are not interested of service creations */
        if (*new != '\0')
            goto out;

        data = name_data_find(name);
        if (!data) {
            error("Got NameOwnerChanged signal for %s which has no listeners", name);
            goto out;
        }

        g_slist_foreach(data->callbacks, (GFunc)name_call_listener, name);
        name_listeners = g_slist_remove(name_listeners, data);
        name_data_free(data);
    }

out:
    return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

void append_dbus_args(DBusMessage *message, int first_arg_type, ...)
{
    dbus_bool_t ret;
    va_list ap;

    va_start(ap, first_arg_type);
    ret = dbus_message_append_args_valist(message, first_arg_type, ap);
    va_end(ap);

    if (ret == FALSE)
        die("dbus_message_append_args failed");
}

gboolean add_name_listener(DBusConnection *connection, const char *name,
                           name_cb func, gpointer user_data)
{
    DBusError err;
    char *match_string;

    debug("add_name_listener(%s)", name);

    if (!name_listener_initialized) {
        if (!dbus_connection_add_filter(connection, name_exit_filter, NULL, NULL)) {
            error("dbus_connection_add_filter() failed");
            return FALSE;
        }
        name_listener_initialized = TRUE;
    }

    /* name_data_add returns FALSE if there already was an entry for the name  */
    if (!name_data_add(name, func, user_data))
        return TRUE;

    match_string = g_strdup_printf("interface=%s,member=NameOwnerChanged,arg0=%s",
                                   DBUS_INTERFACE_DBUS, name);

    dbus_error_init(&err);
    dbus_bus_add_match(connection, match_string, &err);

    g_free(match_string);

    if (dbus_error_is_set(&err)) {
        error("Adding owner match rule for %s failed: %s", name, err.message);
        dbus_error_free(&err);
        name_data_remove(name, func, user_data);
        return FALSE;
    }
    
    return TRUE;
}

gboolean remove_name_listener(DBusConnection *connection, const char *name,
                              name_cb func, gpointer user_data)
{
    NameData *data;
    NameCallback *cb;
    DBusError err;
    char *match_string;

    debug("remove_name_listener(%s)", name);

    data = name_data_find(name);
    if (!data) {
        error("remove_name_listener: no listener for %s", name);
        return FALSE;
    }

    cb = name_callback_find(data->callbacks, func, user_data);
    if (!cb) {
        error("No matching callback found for %s", name);
        return FALSE;
    }

    data->callbacks = g_slist_remove(data->callbacks, cb);
    g_free(cb);

    if (data->callbacks)
        return TRUE;

    match_string = g_strdup_printf("interface=%s,member=NameOwnerChanged,arg0=%s",
                                   DBUS_INTERFACE_DBUS, name);

    dbus_error_init(&err);
    dbus_bus_remove_match(connection, match_string, &err);

    g_free(match_string);

    if (dbus_error_is_set(&err)) {
        error("Removing owner match rule for %s failed: %s", name, err.message);
        dbus_error_free(&err);
        return FALSE;
    }

    name_data_remove(name, func, user_data);
    
    return TRUE;
}

