/**
    @file startup_highscore.c

    Implementation of high scores.

    Copyright (c) 2004, 2005 Nokia Corporation.
	
    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the
    Free Software Foundation, Inc., 59 Temple Place - Suite 330,
    Boston, MA 02111-1307, USA.
*/

#include <string.h>
#include <stdlib.h>
#include "startup_highscore.h"
#include "startup_i18n.h"

GString *highscore_read_file(void)
{
  gchar *file = settings_get_string(SETTINGS_MAHJONG_HIGH_SCORES);

  /* FIXME: Init somewhere else? */
  if (! gnome_vfs_initialized())
    if (! gnome_vfs_init() ) {
      if (file != NULL)
        g_free(file);
      return NULL;
    }

  /* Test if file exists */
  if ( g_file_test(file, G_FILE_TEST_EXISTS | G_FILE_TEST_IS_REGULAR )) {
    GnomeVFSHandle *handle;
    
    /* Try to open the file */
    if ( gnome_vfs_open(&handle, file, GNOME_VFS_OPEN_READ) ==
         GNOME_VFS_OK) {

      static const GnomeVFSFileSize DATA_BUFFER_SIZE = 1024;
      GnomeVFSFileSize cnt;
      GnomeVFSResult res;
      gchar data[DATA_BUFFER_SIZE+1];
      GString *scores = g_string_new("");

      /* Read file contents */
      do {
        res = gnome_vfs_read(handle, (gpointer)data,
          DATA_BUFFER_SIZE, &cnt);

        if (cnt>0) {
          g_assert(cnt<=DATA_BUFFER_SIZE);

          data[cnt]=0;
          g_string_append(scores,data);
        }
      } while (res==GNOME_VFS_OK && cnt>0);

      gnome_vfs_close(handle);

      if (file != NULL)
        g_free(file);

      return scores;
    }
  }

  if (file != NULL)
    g_free(file);

  return NULL;
}

void highscore_load(struct HighScore *score)
{
  guint i=0;
  GString *data = highscore_read_file();
  gchar *tmp = NULL;
  gchar buf[50];

  for (i=0; i<HIGHSCORE_SIZE; i++) {
    score[i].level=NULL;
    score[i].time=NULL;
    score[i].date=NULL;
  }

  if (data == NULL) return;
  g_assert(data);
  tmp = data->str;

  for (i=0; i<HIGHSCORE_SIZE; i++) {
    gchar *tmp2 = NULL;
    int intmp;
    time_t time;

    /* Get first space separated entry */
    if (strlen(tmp)==0) break;
    tmp2 = g_strstr_len(tmp,strlen(tmp)," ");
    if (tmp2==NULL) break;
    *tmp2 = 0;

    /* Set it as time */
    intmp = atoi(tmp);
    g_snprintf(buf, 49, "%02d:%02d", intmp/60, intmp%60);
    score[i].time = g_strdup(buf);

    /* Get second space separated entry */
    if (tmp2>=data->str+data->len-1) break;
    tmp = tmp2+1;
    if (strlen(tmp)==0) break;
    tmp2 = g_strstr_len(tmp,strlen(tmp)," ");
    if (tmp2==NULL) break;
    *tmp2 = 0;

    /* Set it as date */
    time = atoi(tmp);
    static const char *gcc_bug = "%x";
    strftime(buf, 49, gcc_bug, localtime(&time));
    score[i].date = g_strdup(buf);

    /* Get everything till the end of line */
    if (tmp2>=data->str+data->len-1) break;
    tmp = tmp2+1;
    if (strlen(tmp)==0) break;
    tmp2 = g_strstr_len(tmp,strlen(tmp),"\n");
    if (tmp2!=NULL) {
      *tmp2=0;
    }

    /* Set it as level name */
    gchar *locale_id = g_strdup("game_va_mahjong_settings_board_x");
    locale_id[strlen(locale_id) - 1] = *tmp;
    score[i].level = g_strdup(gettext(locale_id));
    g_free(locale_id);

    /* Checks */
    if (tmp2==NULL) break;
    if (strlen(tmp)==0) break;
    if (tmp2>=data->str+data->len-1) break;
    tmp = tmp2+1;
  }

  g_string_free(data, TRUE);
}

void highscore_unload(struct HighScore *score)
{
  guint i=0;
  
  for (i=0; i<HIGHSCORE_SIZE; i++) {
    if (score[i].level) {
      g_free(score[i].level);
      score[i].level=NULL;
    }
    if (score[i].time) {
      g_free(score[i].time);
      score[i].time=NULL;
    }
    if (score[i].date) {
      g_free(score[i].date);
      score[i].date=NULL;
    }
  }
}
