/**
    @file dbus.c

    DBUS functionality.

    Copyright (c) 2004, 2005 Nokia Corporation.
	
    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the
    Free Software Foundation, Inc., 59 Temple Place - Suite 330,
    Boston, MA 02111-1307, USA.
*/

#include "dbus.h"
#include "state_save.h"

/* Define topping */

AppData *_app_data = NULL;



void
osso_top_callback(const gchar * arguments, gpointer data)
{
    AppData *app_data = (AppData *) data;
    osso_log(LOG_DEBUG, "%s\n", __FUNCTION__);

    g_assert(app_data);
    gtk_window_present(GTK_WINDOW(app_data->app_ui_data->main_view));
}

gint
dbus_req_handler(const gchar * interface,
                 const gchar * method,
                 GArray * arguments, gpointer data, osso_rpc_t * retval)
{
    (void) interface;
    return handle_game_message(method, arguments, data, retval);
}

/* Send d-bus message */
osso_return_t
send_dbus_message(const gchar * service,
                  const gchar * object_path,
                  const gchar * iface,
                  const gchar * method, GArray * args, osso_rpc_t * retval)
{
    osso_log(LOG_DEBUG, "Sending message %s to: %s\n", method, service);
    return osso_rpc_run(_app_data->app_osso_data->osso,
                        service, object_path, iface, method, retval, 0
                        // args
        );
}

/* Depending on the state of hw, do something */
void
hw_event_handler(osso_hw_state_t * state, gpointer data)
{
    (void) data;

    if (state->shutdown_ind)
    {
        /* Rebooting */
        game_clean_up_and_quit(NULL, ((AppData *) data)->app_ui_data);
        /* gtk_main_quit(); */
    }
    if (state->memory_low_ind)
    {
        /* Memory low */
        ui_show_save_nomem_dialog(((AppData *) data)->app_ui_data);
        if (((AppData *) data)->app_ui_data->low_memory == FALSE)
        {
            ((AppData *) data)->app_ui_data->low_memory = TRUE;
            // Why shouldn't we dimm the button, if there is no memory? This
            // is stupid.
            // dimm_everything(data);
        }
    }
    else
    {
        if (((AppData *) data)->app_ui_data->low_memory)
        {
            ((AppData *) data)->app_ui_data->low_memory = FALSE;
            ui_hide_save_nomem_dialog(((AppData *) data)->app_ui_data);
            // undimm_everything(data);
        }
    }
    if (state->save_unsaved_data_ind)
    {
        /* Unsaved data should be saved */
    }
    if (state->system_inactivity_ind)
    {
        /* Minimum activity */
    }
}

/* Do initialization for OSSO, create osso context, set topping callback,
 * dbus-message handling callbaks, and hw-event callbacks. TODO: System bus
 * still not seem working well, so HW-event callbacks no tested */

gboolean
init_osso(AppData * app_data)
{
    _app_data = app_data;
    AppUIData *app = app_data->app_ui_data;
    gchar *service = NULL, *path = NULL, *iface = NULL;

    osso_return_t ret;
    /* Init osso */
    osso_log(LOG_INFO, "Initializing osso");
    osso_log(LOG_DEBUG, "Initializing osso");

    gchar *appl_name = g_strdup_printf("osso_%s_startup",
                                       startup_config_read_entry(app->
                                                                 indt_startup_config,
                                                                 "Name",
                                                                 NULL));
    app_data->app_osso_data->osso =
        osso_initialize(appl_name,
                        startup_config_read_entry(app->
                                                  indt_startup_config,
                                                  "Version", NULL), TRUE,
                        NULL);
    g_free(appl_name);


    if (app_data->app_osso_data->osso == NULL)
    {
        osso_log(LOG_ERR, "Osso initialization failed");
        return FALSE;
    }
    g_assert(app_data->app_osso_data->osso);

    /* Set topping callback */
    ret = osso_application_set_top_cb(app_data->app_osso_data->osso,
                                      startup_app_top_event_cb, app);

    if (ret != OSSO_OK)
    {
        osso_log(LOG_ERR, "Could not set topping callback");
    }

    app->service =
        (gchar *) startup_config_read_entry(app->indt_startup_config,
                                            "ServiceName", NULL),
        app->path =
        (gchar *) startup_config_read_entry(app->indt_startup_config,
                                            "PathName", NULL),
        app->iface =
        (gchar *) startup_config_read_entry(app->indt_startup_config,
                                            "InterfaceName", NULL),
        service = g_strdup_printf("%s_startup", app->service);
    path = g_strdup_printf("%s_startup", app->path);
    iface = g_strdup_printf("%s_startup", app->iface);

    /* Set handling d-bus messages from session bus */
    ret = osso_rpc_set_cb_f(app_data->app_osso_data->osso,
                            service,
                            path, iface, startup_app_rpc_event_cb, app);

    g_free(service);
    g_free(path);
    g_free(iface);

    if (ret != OSSO_OK)
    {
        osso_log(LOG_ERR, "Could not set callback for receiving messages");
    }


    /* ... And these for the old game. Otherwise chess won't get the load 
     * message */
    service = g_strdup_printf("%s.startup", app->service);
    path = g_strdup_printf("%s/startup", app->path);
    iface = g_strdup_printf("%s.startup", app->iface);
    ret = osso_rpc_set_cb_f(app_data->app_osso_data->osso,
                            service,
                            path, iface, startup_app_rpc_event_cb, app);

    g_free(service);
    g_free(path);
    g_free(iface);


    if (ret != OSSO_OK)
    {
        osso_log(LOG_ERR, "Could not set callback for receiving messages");
    }

    /* Set handling changes in HW states. Note: not tested */
    ret = osso_hw_set_event_cb(app_data->app_osso_data->osso,
                               NULL, hw_event_handler, app_data);

    if (ret != OSSO_OK)
    {
        osso_log(LOG_ERR, "Could not set callback for HW monitoring");
    }

    return TRUE;
}

/* Deinitialize osso specific data TODO: Check of return values from osso */
gboolean
deinit_osso(AppData * app_data)
{
    osso_application_unset_top_cb(app_data->app_osso_data->osso,
                                  startup_app_top_event_cb,
                                  app_data->app_ui_data);
    gchar *service =
        g_strdup_printf("%s.startup", app_data->app_ui_data->service);
    gchar *path =
        g_strdup_printf("%s/startup", app_data->app_ui_data->path);
    gchar *iface =
        g_strdup_printf("%s.startup", app_data->app_ui_data->iface);

    /* Set handling d-bus messages from session bus */
    osso_rpc_unset_cb_f(app_data->app_osso_data->osso,
                        service,
                        path,
                        iface,
                        startup_app_rpc_event_cb, app_data->app_ui_data);

    g_free(service);
    g_free(path);
    g_free(iface);
    service =
        g_strdup_printf("%s_startup", app_data->app_ui_data->service);
    path = g_strdup_printf("%s_startup", app_data->app_ui_data->path);
    iface = g_strdup_printf("%s_startup", app_data->app_ui_data->iface);
    osso_rpc_unset_cb_f(app_data->app_osso_data->osso,
                        service,
                        path,
                        iface,
                        startup_app_rpc_event_cb, app_data->app_ui_data);

    g_free(service);
    g_free(path);
    g_free(iface);

    osso_hw_unset_event_cb(app_data->app_osso_data->osso, NULL);

    /* Deinit osso */
    g_assert(app_data->app_osso_data->osso);
    osso_deinitialize(app_data->app_osso_data->osso);

    return TRUE;
}

AppData *
get_app_data()
{
    return _app_data;
}



/* Send async d-bus message */
osso_return_t
send_dbus_message_async(const gchar * service,
                        const gchar * object_path,
                        const gchar * iface,
                        const gchar * method, GArray * args, gpointer data)
{
    osso_log(LOG_DEBUG, "Sending async message %s to: %s\n", method, service);
    AppData *ad = get_app_data();
    ad->app_ui_data->async_command_sending = TRUE;
    return osso_rpc_async_run(_app_data->app_osso_data->osso,
                              service,
                              object_path,
                              iface,
                              method, dbus_message_async_result, data, 0
                              // args
        );
}


void
dbus_message_async_result(const gchar * interface,
                          const gchar * method,
                          osso_rpc_t * retval, gpointer data)
{
    osso_log(LOG_DEBUG, "Async message %s return from %s\n", method,
             interface);
    AppData *ad = get_app_data();
    ad->app_ui_data->async_command_sending = FALSE;

    if (ad == NULL)
        return;

    if (strcmp(method, GAME_SAVE_METHOD) == 0)
    {
        if (retval->type == DBUS_TYPE_BOOLEAN)
        {
            if (retval->value.b == TRUE)
                ad->app_ui_data->chess_save_done = GAME_SAVE_OK;
            else
                ad->app_ui_data->chess_save_done = GAME_SAVE_ERR;
        }
        else
        {
            ad->app_ui_data->chess_save_done = GAME_SAVE_ERR;
        }
    }
}
