/*
 * Copyright © 2004 Keith Packard
 * Copyright © 2005 Eric Anholt
 * Copyright © 2006 Nokia Corporation
 *
 * Permission to use, copy, modify, distribute and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the names of the authors and/or copyright holders
 * not be used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  The authors and
 * copyright holders make no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without any express
 * or implied warranty.
 *
 * THE AUTHORS AND COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES WITH REGARD TO
 * THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS, IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER
 * RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
 * CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Author: Daniel Stone <daniel.stone@nokia.com>
 * Based on ati_video.c by Eric Anholt, which was based on mach64video.c by
 * Keith Packard.
 */

#ifdef HAVE_KDRIVE_CONFIG_H
#include <kdrive-config.h>
#endif

#include "omap.h"
#include "omapfb.h"
#include <linux/fb.h>
#include <sys/ioctl.h>

#include <X11/extensions/Xv.h>
#include "fourcc.h"

#define MAKE_ATOM(a) MakeAtom(a, sizeof(a) - 1, TRUE)

#ifndef max
#define max(x, y) (((x) >= (y)) ? (x) : (y))
#endif

static KdVideoEncodingRec DummyEncoding[] = {
    /* Max width and height are filled in later. */
    { 0, "XV_IMAGE", -1, -1, { 1, 1 } },
};

static KdImageRec Images[] = {
    XVIMAGE_YUY2, /* OMAPFB_COLOR_YUY422 */
    XVIMAGE_UYVY, /* OMAPFB_COLOR_YUV422 */
    XVIMAGE_I420, /* OMAPFB_COLOR_YUV420 */
    XVIMAGE_YV12, /* OMAPFB_COLOR_YUV420 */
};

#define NUM_IMAGES (sizeof(Images) / sizeof(Images[0]))

static KdVideoFormatRec Formats[] = {
    { 16, TrueColor },
};

#define NUM_FORMATS (sizeof(Formats) / sizeof(Formats[0]))

static KdAttributeRec Attributes[] = {
    { XvSettable | XvGettable, OMAP_VSYNC_NONE, OMAP_VSYNC_FORCE,
      "XV_OMAP_VSYNC" },
    { XvSettable | XvGettable, 0, 0xffff, "XV_COLORKEY" },
};

#define NUM_ATTRIBUTES (sizeof(Attributes) / sizeof(Attributes[0]))

static Atom xvColorKey, xvVSync;

static void omapVideoStop(KdScreenInfo *screen, pointer data, Bool exit);

/**
 * Check if plane attributes have changed.
 */
static _X_INLINE int
omapPlaneIsDirty(OmapPortPriv *port_info, int id, int src_w, int src_h,
                 int dst_x, int dst_y, int dst_w, int dst_h)
{
    const OmapPlaneInfo *plane = port_info->plane;

    if (plane->dirty || port_info->fourcc != id || port_info->src_w != src_w ||
        port_info->src_h != src_h || port_info->dst_x != dst_x ||
        port_info->dst_y != dst_y || port_info->dst_w != dst_w ||
        port_info->dst_h != dst_h || plane->ext_state & OMAP_EXT_PENDING)
        return 1;
    else
        return 0;
}

/**
 * Since we can have differing formats in the framebuffer, we want to
 * block updates for a couple of frames.
 */
static void omapUnblockUpdates(OmapPortPriv *port_info)
{
    if (port_info->timer) {
        TimerCancel(port_info->timer);
        port_info->timer = NULL;
    }

    port_info->omaps->block_updates &= ~(1 << port_info->plane->id);
}

static void omapEmptyClip(OmapPortPriv *port_info)
{
    if (REGION_NOTEMPTY(port_info->omaps->screen->pScreen, &port_info->clip)) {
        if (port_info->drawable)
            KXVPaintRegion(port_info->drawable, &port_info->clip,
                           port_info->omaps->screen->pScreen->blackPixel);
        port_info->drawable = NULL;
        REGION_SUBTRACT(port_info->omaps->screen->pScreen,
                        port_info->omaps->video_region,
                        port_info->omaps->video_region, &port_info->clip);
        REGION_EMPTY(port_info->omaps->screen->pScreen, &port_info->clip);
        omapSyncEngine(port_info->omaps->omapc);
    }
}

static void
omapDisplayFrame(KdScreenInfo *screen, OmapPortPriv *port_info)
{

    /* If updates have been blocked because of migration, we need to
     * ensure that the colourkey has been correctly painted, and
     * the migration has been completed, before we force an
     * update. */
    if (port_info->plane->ext_state & OMAP_EXT_PENDING) {
        DebugF("omapDisplayFrame: plane %d pending migration\n",
               port_info->plane->id);
        omapSyncEngine(port_info->omaps->omapc);
    }
    else if (port_info->omaps->block_updates & (1 << port_info->plane->id)) {
        omapSyncEngine(port_info->omaps->omapc);
        DebugF("omapDisplayFrame: unblocking updates for %d\n",
               port_info->plane->id);
        omapUnblockUpdates(port_info);

        if (!port_info->omaps->block_updates)
            omapForceFullScreenUpdate(port_info->omaps, FALSE);
    }
    else {
        omapFlushDamage(port_info->plane, NULL);
    }
}

/**
 * Stop the video overlay, optionally clearing both the memory, and the clip
 * region.  Don't set either to FALSE unless you know what you're doing.
 */
static void
_omapStopVideo(OmapPortPriv *port_info, Bool clearmem)
{
    int is_migrated;

    if ((OMAP_GET_EXT(port_info->plane) == OMAP_EXT_MIGRATED ||
        port_info->plane->ext_state == (OMAP_EXT_CANDIDATE | OMAP_EXT_PENDING)))
        is_migrated = 1;
    else
        is_migrated = 0;

    omapSyncEngine(port_info->omaps->omapc);

    if (is_migrated || port_info->visibility >= VisibilityFullyObscured)
        omapEmptyClip(port_info);

    if (clearmem)
        bzero(port_info->plane->fb, port_info->plane->fb_size);

    if (port_info->plane->state >= OMAP_STATE_ACTIVE)
        omapPlaneDisable(port_info->plane);

    port_info->omaps->individual_updates &= ~(1 << port_info->plane->id);
    port_info->plane->state = OMAP_STATE_GRABBED;

    if (is_migrated) {
        DebugF("_omapStopVideo: forcing full-screen update on %d (non-exit)\n",
               port_info->plane->id);
        omapForceFullScreenUpdate(port_info->omaps, TRUE);
    }

    DebugF("_omapStopVideo: stopped plane %d\n", port_info->plane->id);
}

static CARD32 omapBlockUpdatesTimer(OsTimerPtr timer, CARD32 now, pointer data)
{
    OmapPortPriv *port_info = (OmapPortPriv *) data;

    DebugF("omapBlockUpdatesTimer: timeout on plane %d, ext state %x\n",
           port_info->plane->id, port_info->plane->ext_state);
    port_info->timer = NULL;

    omapUnblockUpdates(port_info);

    switch (port_info->plane->ext_state) {
    case OMAP_EXT_NONE:
        break;

    case OMAP_EXT_MIGRATED:
        omapDisplayFrame(port_info->omaps->screen, port_info);
        omapForceFullScreenUpdate(port_info->omaps, FALSE);
        break;

    case (OMAP_EXT_CANDIDATE | OMAP_EXT_PENDING):
    case (OMAP_EXT_MIGRATED | OMAP_EXT_PENDING):
        omapEmptyClip(port_info);
        _omapStopVideo(port_info, TRUE);
    default:
        omapForceFullScreenUpdate(port_info->omaps, TRUE);
        break;
    }

    return 0;
}

static void omapBlockUpdates(OmapPortPriv *port_info)
{
    if (port_info->timer)
        TimerCancel(port_info->timer);
    port_info->omaps->block_updates |= (1 << port_info->plane->id);
    port_info->timer = TimerSet(NULL, 0, 2000, omapBlockUpdatesTimer,
                                (pointer) port_info);
}

static void omapCheckClip(OmapPortPriv *port_info)
{
    OmapPlaneInfo *tmp;

    if (port_info->visibility == VisibilityUnobscured) {
        if (OMAP_GET_EXT(port_info->plane) == OMAP_EXT_CANDIDATE) {
            /* There can only be one. */
            for (tmp = port_info->omaps->omapc->planes; tmp; tmp = tmp->next) {
                if (OMAP_GET_EXT(tmp) == OMAP_EXT_MIGRATED) {
                    DebugF("omapCheckClip: not migrating second candidate %d\n",
                           port_info->plane->id);
                    return;
                }
            }

            DebugF("omapCheckClip: migrating unclipped candidate %d\n",
                   port_info->plane->id);
            port_info->plane->ext_state = OMAP_EXT_MIGRATED | OMAP_EXT_PENDING;
            port_info->plane->dirty = TRUE;
            DebugF("omapCheckClip: blocking UI updates for %d\n",
                   port_info->plane->id);
            omapBlockUpdates(port_info);
        }
    }
    else {
        if (OMAP_GET_EXT(port_info->plane) == OMAP_EXT_MIGRATED) {
            DebugF("omapCheckClip: migrating clipped video %d\n",
                   port_info->plane->id);
            port_info->plane->ext_state = OMAP_EXT_CANDIDATE | OMAP_EXT_PENDING;
            if (port_info->visibility == VisibilityPartiallyObscured) {
                DebugF("omapCheckClip: blocking UI updates for %d\n",
                       port_info->plane->id);
                omapBlockUpdates(port_info);
            }
            else {
                DebugF("omapCheckClip: not blocking updates for fully "
                       "obscured video %d\n", port_info->plane->id);
                omapUnblockUpdates(port_info);
                
                omapEmptyClip(port_info);
            }
            port_info->plane->dirty = TRUE;
        }
        else if (port_info->visibility >= VisibilityFullyObscured) {
            omapEmptyClip(port_info);
        }
    }
}

/**
 * When the clip on a window changes, check it and stash it away, so we
 * don't end up with any clipped windows on the external controller.
 */
static void
omapClipNotify(KdScreenInfo *screen, void *data, WindowPtr window, int dx,
               int dy)
{
    OmapPortPriv *port_info = data;

    port_info->visibility = window->visibility;
    omapCheckClip(port_info);
}

/**
 * Xv attributes get/set support.
 */
static int
omapGetPortAttribute(KdScreenInfo *screen, Atom attribute, int *value,
                     pointer data)
{
    OmapPortPriv *port_info = data;

    ENTER();

    if (attribute == xvVSync) {
        *value = port_info->plane->vsync;
        LEAVE();
        return Success;
    }
    else if (attribute == xvColorKey) {
        *value = port_info->plane->colorkey;
        LEAVE();
        return Success;
    }

    LEAVE();
    return BadMatch;
}

static int
omapSetPortAttribute(KdScreenInfo *screen, Atom attribute, int value,
                     pointer data)
{
    OmapPortPriv *port_info = data;

    ENTER();

    if (attribute == xvVSync) {
        if (value < OMAP_VSYNC_NONE || value > OMAP_VSYNC_FORCE) {
            LEAVE();
            return BadValue;
        }

        if (!(port_info->plane->caps & OMAPFB_CAPS_TEARSYNC) && value) {
            ErrorF("omapSetPortAttribute: requested vsync on a non-sync "
                   "capable port\n");
            LEAVE();
            return BadValue;
        }

        port_info->plane->vsync = value;
        LEAVE();
        return Success;
    }
    else if (attribute == xvColorKey) {
        if (value < 0 || value > 0xffff) {
            LEAVE();
            return BadValue;
        }

        port_info->plane->colorkey = value;
        port_info->plane->dirty = TRUE;
        LEAVE();
        return Success;
    }

    LEAVE();
    return BadMatch;
}

/**
 * Clip the image size to the visible screen.
 */
static void
omapQueryBestSize(KdScreenInfo *screen, Bool motion, short vid_w,
                  short vid_h, short dst_w, short dst_h,
                  unsigned int *p_w, unsigned int *p_h, pointer data)
{
    if (dst_w < screen->width)
        *p_w = dst_w;
    else
        *p_w = screen->width;

    if (dst_h < screen->width)
        *p_h = dst_h;
    else
        *p_h = screen->width;
}


/**
 * Copy packed video data for downscaling, where 'scaling' in this case
 * means just removing lines.  Unfortunately, since we don't want to get
 * into the business of swapping the U and V channels, we remove a
 * macroblock (2x1) at a time.
 */
void
omapCopyPackedData(KdScreenInfo *screen, OmapPortPriv *port_info, CARD8 *src,
    CARD8 *dst, int randr, int srcPitch, int dstPitch, const int srcW,
    const int srcH, int top, int left, int h, int w, const int dstW,
    const int dstH)
{
    int i = 0, k = 0, ih = 0, jh = 0, jhn = (srcH - dstH);
    int kh = 0, khn = (srcW - dstW), lh = 0;

    if ((randr & RR_Rotate_All) != RR_Rotate_0) {
        ErrorF("omapCopyPackedData: rotation not supported\n");
        return;
    }

    if (top || left) {
        ErrorF("omapCopyPackedData: partial updates not supported\n");
        return;
    }

    if (srcW & 1 || dstW & 1) {
        ErrorF("omapCopyPackedData: widths must be multiples of two\n");
        return;
    }

    w >>= 1;

    for (i = 0; i < h; i++, src += srcPitch, ih += (srcH - dstH)) {
        if (port_info->vscale && (jh == ih || (jh < ih && jhn > ih))) {
            jh += srcH;
            jhn += srcH;
            continue;
        }

        /* memcpy the whole lot if we can: it's a lot quicker. */
        if (!port_info->hscale) {
            memcpy(dst, src, srcPitch);
        }
        else {
	    CARD32 *s = (CARD32 *) src;
            CARD32 *d = (CARD32 *) dst;

            kh = 0;
            khn = 2 * srcW;
            lh = 2 * (srcW - dstW);
            /* failing this, do 32-bit copies by hand for the pixels we
             * need. */
            for (k = 0; k < w; k++, s++, lh += 2 * (srcW - dstW)) {
                if (k != (w - 1) && (kh == lh || (kh < lh && khn > lh))) {
                    kh += (2 * srcW);
                    khn += (2 * srcW);
                }
                else {
                    *d++ = *s;
                }
            }

        }
        dst += dstPitch;
    }
}

/**
 * Copy I420 data to the custom 'YUV420' format, which is actually:
 * y11 u11,u12,u21,u22 u13,u14,u23,u24 y12 y14 y13
 * y21 v11,v12,v21,v22 v13,v14,v23,v24 y22 y24 y23
 *
 * The third and fourth luma components are swapped.  Yes, this is weird.
 *
 * So, while we have the same 2x2 macroblocks in terms of colour granularity,
 * we actually require 4x2.  We lop off the last 1-3 lines if width is not a
 * multiple of four, and let the hardware expand.
 *
 * FIXME: Target for arg reduction.
 */
static void
omapCopyPlanarDataYUV420(KdScreenInfo *screen, OmapPortPriv *port_info, CARD8 *srcb,
                         CARD8 *dstb, int randr, int srcPitch, int srcPitch2,
                         int dstPitch, int srcW, int srcH, int top, int left,
                         int h, int w, int id)
{
    CARD8 *srcy, *srcu, *srcv, *dst;
    CARD16 *d1;
    CARD32 *d2;
    int i, j;

    if ((randr & RR_Rotate_All) != RR_Rotate_0) {
        ErrorF("omapCopyPlanarData: rotation not supported\n");
        return;
    }

    if (top || left || h != srcH || w != srcW) {
        ErrorF("omapCopyPlanarData: offset updates not supported\n");
        return;
    }

    srcy = srcb;
    srcv = srcy + h * srcPitch;
    srcu = srcv + (h >> 1) * srcPitch2;
    dst = dstb;

    if (id == FOURCC_I420) {
        CARD8 *tmp = srcv;
        srcv = srcu;
        srcu = tmp;
    }

    w >>= 2;
    for (i = 0; i < h; i++) {
        CARD32 *sy = (CARD32 *) srcy;
        CARD16 *sc;

        sc = (CARD16 *) ((i & 1) ? srcv : srcu);
        d1 = (CARD16 *) dst;

        for (j = 0; j < w; j++) {
            if (((unsigned long) d1) & 3) {
                /* Luma 1, chroma 1. */
                *d1++ = (*sy & 0x000000ff) | ((*sc & 0x00ff) << 8);
                /* Chroma 2, luma 2. */
                *d1++ = ((*sc & 0xff00) >> 8) | (*sy & 0x0000ff00);
            }
            else {
                d2 = (CARD32 *) d1;
                /* Luma 1, chroma 1, chroma 2, luma 2. */
                *d2++ = (*sy & 0x000000ff) | (*sc << 8) |
                        ((*sy & 0x0000ff00) << 16);
                d1 = (CARD16 *) d2;
            }
            /* Luma 4, luma 3. */
            *d1++ = ((*sy & 0xff000000) >> 24) | ((*sy & 0x00ff0000) >> 8);
            sy++;
            sc++;
        }

        dst += dstPitch;
        srcy += srcPitch;
        if (i & 1) {
            srcu += srcPitch2;
            srcv += srcPitch2;
        }
    }
}

/**
 * Copy and expand planar (I420) -> packed (UYVY) video data, including
 * downscaling, by just removing two lines at a time.
 *
 * FIXME: Target for arg reduction.
 */
static void
omapExpandPlanarData(KdScreenInfo *screen, OmapPortPriv *port_info, CARD8 *src,
    CARD8 *dstb, int randr, int srcPitch, int srcPitch2, int dstPitch,
    int srcW, int srcH, int height, int top, int left, int h, int w, int id,
    int dstW, int dstH)
{
    CARD8 *src1, *src2, *src3, *dst1;
    int srcDown = srcPitch, srcDown2 = srcPitch2;
    int srcRight = 2, srcRight2 = 1, srcNext = 1;
    int i = 0, k = 0, ih = (srcH - dstH), jh = 0, jhn = srcH;
    int kh = 0, khn = 0, lh = 0;

    if ((randr & RR_Rotate_All) != RR_Rotate_0) {
        ErrorF("omapExpandPlanarData: rotation not supported\n");
        return;
    }
    if (top || left) {
        ErrorF("omapExpandPlanarData: partial updates not supported\n");
        return;
    }

    /* compute source data pointers */
    src1 = src;
    src2 = src1 + height * srcPitch;
    src3 = src2 + (height >> 1) * srcPitch2;

    if (id == FOURCC_I420) {
        CARD8 *tmp = src2;
        src2 = src3;
        src3 = tmp;
    }

    dst1 = dstb;

    w >>= 1;
    for (i = 0; i < h; i++, ih += (srcH - dstH)) {
        if (port_info->vscale && (jh == ih || (jh < ih && jhn > ih))) {
            jh += srcH;
            jhn += srcH;
        }

        else {
	    CARD32 *dst = (CARD32 *)dst1;
            CARD8 *s1l = src1;
            CARD8 *s1r = src1 + srcNext;
            CARD8 *s2 = src2;
            CARD8 *s3 = src3;
        
            kh = 0;
            khn = 2 * srcW;
            lh = 2 * (srcW - dstW);

            for (k = 0; k < w; k++, lh += 2 * (srcW - dstW)) {
                if (port_info->hscale && k != (w - 1) &&
                    (kh == lh || (kh < lh && khn > lh))) {
                    kh += (2 * srcW);
                    khn += (2 * srcW); 
                }
                else {
                    *dst++ = *s1l | (*s1r << 16) | (*s3 << 8) | (*s2 << 24);
                }

                s1l += srcRight;
                s1r += srcRight;
                s2 += srcRight2;
                s3 += srcRight2;
            }

            dst1 += dstPitch;
        }

        src1 += srcDown;
	if (i & 1) {
	    src2 += srcDown2;
	    src3 += srcDown2;
	}
    }
}

/**
 * Start the video overlay; relies on data in port_info being sensible for
 * the current frame.
 */
static Bool
_omapStartVideo(OmapPortPriv *port_info)
{
    if (port_info->plane->state >= OMAP_STATE_ACTIVE) {
        DebugF("omapStartVideo: plane %d still active!\n",
               port_info->plane->id);
        _omapStopVideo(port_info, TRUE);
        usleep(5000);
    }

    /* Hilariously, this will actually trigger upscaling for odd widths,
     * since we'll output n-1 pixels, and have the engine scale to n;
     * dispc hangs if you try to feed it sub-macroblocks.
     *
     * When feeding YUV420 to the external controller, we have 4x2
     * macroblocks in essence, so lop off up to the last three lines, and
     * let the hardware scale.
     */
    port_info->plane->src_area.x = 0;
    port_info->plane->dst_area.x = port_info->dst_x;
    port_info->plane->dst_area.width = port_info->dst_w;

    if (port_info->hscale)
        port_info->plane->src_area.width = port_info->dst_w & ~1;
    else
        port_info->plane->src_area.width = port_info->src_w;

    port_info->plane->src_area.y = 0;
    port_info->plane->dst_area.y = port_info->dst_y;
    port_info->plane->dst_area.height = port_info->dst_h;

    if (port_info->vscale)
        port_info->plane->src_area.height = port_info->dst_h & ~1;
    else
        port_info->plane->src_area.height = port_info->src_h;

    if (!omapPlaneEnable(port_info->plane)) {
        DebugF("_omapStartVideo: couldn't enable plane %d\n",
               port_info->plane->id);
        return FALSE;
    }

    port_info->omaps->individual_updates |= (1 << port_info->plane->id);
    port_info->plane->state = OMAP_STATE_ACTIVE;

    DebugF("_omapStartVideo: enabled plane %d\n", port_info->plane->id);

    return TRUE;
}

/**
 * Stop an overlay.  exit is whether or not the client's exiting.
 */
void
omapVideoStop(KdScreenInfo *screen, pointer data, Bool exit)
{
    OmapPortPriv *port_info = data;
    OmapScreenInfo *omaps = port_info->omaps;

    ENTER();

    _omapStopVideo(port_info, TRUE);

    if (port_info->timer) {
        TimerCancel(port_info->timer);
        port_info->timer = NULL;
    }

    port_info->plane->dirty = TRUE;

    if (exit) {
        omapEmptyClip(port_info);

        if (omaps->block_updates & (1 << port_info->plane->id)) {
            DebugF("omapVideoStop: unblocking updates for %d\n",
                   port_info->plane->id);
            omapUnblockUpdates(port_info);
        }

        if (port_info->plane->caps & OMAPFB_CAPS_TEARSYNC)
            port_info->plane->vsync = OMAP_VSYNC_TEAR;
        else
            port_info->plane->vsync = OMAP_VSYNC_NONE;

        port_info->visibility = VisibilityPartiallyObscured;
        port_info->plane->ext_state = OMAP_EXT_NONE;
        port_info->plane->state = OMAP_STATE_STOPPED;
        port_info->plane->colorkey = OMAP_DEFAULT_CKEY;
    }

    port_info->drawable = NULL;

    LEAVE();
}


/**
 * Set up port_info with the specified parameters, and start the overlay.
 */
static Bool
omapSetupOverlay(KdScreenInfo *screen, OmapPortPriv *port_info, int id,
                 int src_w, int src_h, int dst_x, int dst_y, int dst_w,
                 int dst_h, DrawablePtr drawable)
{
    WindowPtr pWin;
    ENTER();

    if (port_info->plane->state >= OMAP_STATE_ACTIVE) {
        DebugF("omapSetupOverlay: restarting overlay %d\n",
               port_info->plane->id);
        _omapStopVideo(port_info, TRUE);
    }

    if (dst_w >= src_w)
        port_info->hscale = FALSE;
    else
        port_info->hscale = TRUE;
    if (dst_h >= src_h)
        port_info->vscale = FALSE;
    else
        port_info->vscale = TRUE;

    port_info->src_w = src_w;
    port_info->src_h = src_h;
    port_info->dst_w = dst_w;
    port_info->dst_h = dst_h;
    port_info->dst_x = dst_x;
    port_info->dst_y = dst_y;
    port_info->fourcc = id;
    port_info->drawable = drawable;
    port_info->plane->ext_state = OMAP_EXT_NONE;

    switch (id) {
    case FOURCC_YV12:
    case FOURCC_I420:
        if ((port_info->plane->colors & OMAPFB_COLOR_YUV420) &&
            !port_info->hscale && !port_info->vscale &&
            !(dst_x & 3) && !(dst_w & 3) && !(dst_y & 1) && !(dst_h & 1)) {
            port_info->plane->ext_state = OMAP_EXT_CANDIDATE;
        }
    case FOURCC_YUY2:
        port_info->plane->format = OMAPFB_COLOR_YUY422;
        port_info->plane->bpp = 2;
        break;
    case FOURCC_UYVY:
        port_info->plane->format = OMAPFB_COLOR_YUV422;
        port_info->plane->bpp = 2;
        break;
    default:
        ErrorF("omapSetupOverlay: bad FourCC %d!\n", port_info->fourcc);
        LEAVE();
        return FALSE;
    }

    pWin = LookupIDByType(drawable->id, RT_WINDOW);
    if (pWin)
        port_info->visibility = pWin->visibility;
    else
        port_info->visibility = VisibilityPartiallyObscured;
    omapCheckClip(port_info);

    port_info->plane->dirty = TRUE;

    LEAVE();

    return _omapStartVideo(port_info);
}

/**
 * ReputImage hook.  We always fail here if we're mid-migration or
 * on Hailstorm; we want stopped video to actually be _stopped_, due
 * to Hailstorm limitations.
 */
static int
omapReputImage(KdScreenInfo *screen, DrawablePtr drawable, short drw_x,
               short drw_y, RegionPtr clipBoxes, pointer data)
{
    OmapPortPriv *port_info = data;
    int ret;

    ENTER();

    switch (port_info->plane->ext_state) {
    case OMAP_EXT_CANDIDATE:
    case OMAP_EXT_NONE:
        if (!REGION_EQUAL(screen->pScreen, clipBoxes, &port_info->clip)) {
            REGION_SUBTRACT(screen->pScreen, port_info->omaps->video_region,
                            port_info->omaps->video_region, &port_info->clip);
            REGION_COPY(screen->pScreen, &port_info->clip, clipBoxes);
            REGION_UNION(screen->pScreen, port_info->omaps->video_region,
                         port_info->omaps->video_region, &port_info->clip);
        }

        KXVPaintRegion(drawable, &port_info->clip, port_info->plane->colorkey);
        omapSyncEngine(port_info->omaps->omapc);
        omapDisplayFrame(screen, port_info);
        omapForceFullScreenUpdate(port_info->omaps, FALSE);
        ret = Success;
        break;

    default:
        omapForceFullScreenUpdate(port_info->omaps, FALSE);
        ret = BadValue;
    }

    return ret;
}

/**
 * XvPutImage hook.  This does not deal with rotation or partial updates.
 *
 * Calls out to omapCopyPlanarData (unobscured planar video),
 * omapExpandPlanarData (downscaled planar),
 * omapCopyPackedData (downscaled packed), KdXVCopyPlanarData (obscured planar),
 * or KdXVCopyPackedData (packed).
 */
static int
omapPutImage(KdScreenInfo *screen, DrawablePtr drawable,
             short src_x, short src_y,
             short dst_x, short dst_y,
             short src_w, short src_h,
             short dst_w, short dst_h,
             int id,
             unsigned char *buf,
             short width,
             short height,
             Bool sync,
             RegionPtr clip_boxes,
             pointer data)
{
    OmapPortPriv *port_info = (OmapPortPriv *) data;
    int updates_blocked = 0;
    int need_ckey = 0;

    if (dst_x + dst_w > screen->width || dst_y + dst_h > screen->height) {
        ErrorF("omapPutImage: specified dimensions (%d, %d) at (%d, %d) are "
               "larger than the screen (%d, %d)\n", dst_w, dst_h, dst_x,
               dst_y, screen->width, screen->height);
        return BadValue;
    }
    if (width != src_w || height != src_h) {
        ErrorF("omapPutImage: can't put partial images\n");
        return BadValue;
    }

    if (omapPlaneIsDirty(port_info, id, src_w, src_h, dst_x, dst_y, dst_w,
                         dst_h) || !port_info->plane->fb) {
        if (!(port_info->omaps->block_updates & (1 << port_info->plane->id))) {
            DebugF("omapPutImage: blocking updates for %d\n",
                   port_info->plane->id);
            omapBlockUpdates(port_info);
            updates_blocked = 1;
        }

        if (!omapSetupOverlay(screen, port_info, id, src_w, src_h, dst_x,
                              dst_y, dst_w, dst_h, drawable)) {
            ErrorF("omapPutImage: failed to set up overlay: from (%d, %d) "
                   "to (%d, %d) at (%d, %d) on plane %d\n", src_w, src_h,
                   dst_w, dst_h, dst_x, dst_y, port_info->plane->id);
            if (updates_blocked) {
                DebugF("omapPutImage: unblocking updates for %d\n",
                       port_info->plane->id);
                omapUnblockUpdates(port_info);
            }
            return BadAlloc;
        }
    }

    if (!REGION_EQUAL(screen->pScreen, &port_info->clip, clip_boxes)) {
        REGION_SUBTRACT(screen->pScreen, port_info->omaps->video_region,
                        port_info->omaps->video_region, &port_info->clip);
        REGION_COPY(screen->pScreen, &port_info->clip, clip_boxes);
        REGION_UNION(screen->pScreen, port_info->omaps->video_region,
                     port_info->omaps->video_region, &port_info->clip);
        need_ckey = 1;
    }

#if 0
    DebugF("omapPutImage: putting image from (%d, %d, %d, %d) to "
           "(%d, %d, %d, %d)\n", src_x, src_y, src_w, src_h, dst_x, dst_y,
           dst_w, dst_h);
#endif

    /* Sync the engine first, so we don't draw over something that's still
     * being scanned out. */
    if (port_info->plane->vsync != OMAP_VSYNC_NONE)
        omapSyncEngine(port_info->omaps->omapc);

    /* dispc locks up when we try downscaling, so we do it in software
     * instead.  The external controller can scale just fine, though. */
    switch (id) {
    case FOURCC_UYVY:
    case FOURCC_YUY2:
        if (port_info->hscale || port_info->vscale)
            omapCopyPackedData(screen, port_info, buf, port_info->plane->fb,
                               RR_Rotate_0, width << 1, port_info->plane->pitch,
                               src_w, src_h, src_x, src_y, height,
                               width, dst_w, dst_h);
        else
            KdXVCopyPackedData(screen, buf, port_info->plane->fb, RR_Rotate_0,
                               width << 1, port_info->plane->pitch, src_w, src_h, src_x,
                               src_y, height, width);
        break;

    case FOURCC_YV12:
    case FOURCC_I420:
        if (OMAP_GET_EXT(port_info->plane) == OMAP_EXT_MIGRATED) {
            omapCopyPlanarDataYUV420(screen, port_info, buf, port_info->plane->fb,
                                    RR_Rotate_0, width, width >> 1,
                                    port_info->plane->pitch, src_w, src_h, src_x,
                                    src_y, height, width, id);
        }
        else {
            if (port_info->hscale || port_info->vscale) {
                omapExpandPlanarData(screen, port_info, buf,
                                     port_info->plane->fb, RR_Rotate_0,
                                     width, (width >> 1),
                                     port_info->plane->pitch, src_w, src_h,
                                     height, src_x, src_y, height,
                                     width, id, dst_w, dst_h);
            }
            else {
                KdXVCopyPlanarData(screen, buf, port_info->plane->fb, RR_Rotate_0,
                                   width, (width >> 1), port_info->plane->pitch,
                                   src_w, src_h, height, src_x, src_y, height,
                                   width, id);
            }
        }
        break;
    }

    omapDisplayFrame(screen, port_info);
    port_info->plane->ext_state &= ~(OMAP_EXT_PENDING);

    if (need_ckey)
        KXVPaintRegion(drawable, &port_info->clip, port_info->plane->colorkey);

    return Success;    
}

/**
 * Give image size and pitches.
 */
static int
omapQueryImageAttributes(KdScreenInfo *screen, int id, unsigned short *w,
                         unsigned short *h, int *pitches, int *offsets)
{
    int size = 0, tmp = 0;

    if (*w > screen->width)
        *w = screen->width;
    if (*h > screen->width)
        *h = screen->width;

    *w = (*w + 1) & ~1;
    if (offsets)
        offsets[0] = 0;

    switch (id)
    {
    case FOURCC_I420:
    case FOURCC_YV12:
        size = *w;
        if (pitches)
            pitches[0] = size;
        size *= *h;
        if (offsets)
            offsets[1] = size;
        tmp = (*w >> 1);
        if (pitches)
            pitches[1] = pitches[2] = tmp;
        tmp *= (*h >> 1);
        size += tmp;
        if (offsets)
            offsets[2] = size;
        size += tmp;
        break;
    case FOURCC_UYVY:
    case FOURCC_YUY2:
    default:
        size = *w << 1;
        if (pitches)
            pitches[0] = size;
        size *= *h;
        break;
    }

    return size;
}

/**
 * Set up all our internal structures.
 */
static KdVideoAdaptorPtr
omapVideoOverlaySetup(OmapScreenInfo *omaps)
{
    KdVideoAdaptorPtr adapt;
    OmapPortPriv *port_info;
    OmapPlaneInfo *plane;
    int i;

    omaps->num_video_ports = 0;
    for (plane = omaps->omapc->planes; plane; plane = plane->next) {
        if (plane->type == OMAP_PLANE_OVERLAY &&
            (plane->colors & (OMAPFB_COLOR_YUV422 | OMAPFB_COLOR_YUY422)))
            omaps->num_video_ports++;
    }

    /* No usable video overlays. */
    if (omaps->num_video_ports == 0)
        return NULL;

    adapt = xcalloc(1, sizeof(KdVideoAdaptorRec) +
                       omaps->num_video_ports *
                        (sizeof(OmapPortPriv) + sizeof(DevUnion)));
    if (!adapt)
        return NULL;

    DummyEncoding[0].width = omaps->screen->width;
    DummyEncoding[0].height = omaps->screen->height;

    adapt->type = XvWindowMask | XvInputMask | XvImageMask;
    adapt->flags = (VIDEO_CLIP_TO_VIEWPORT | VIDEO_OVERLAID_IMAGES);
    adapt->name = "OMAP Video Overlay";
    adapt->nEncodings = 1;
    adapt->pEncodings = DummyEncoding;
    adapt->nFormats = NUM_FORMATS;
    adapt->pFormats = Formats;
    adapt->nPorts = omaps->num_video_ports;
    adapt->pPortPrivates = (DevUnion *)(&adapt[1]);
    
    port_info = (OmapPortPriv *)
                  (&adapt->pPortPrivates[omaps->num_video_ports]);

    plane = omaps->omapc->planes;
    for (i = 0; i < omaps->num_video_ports; i++) {
        while (plane->type != OMAP_PLANE_OVERLAY ||
               !(plane->colors & (OMAPFB_COLOR_YUV422 | OMAPFB_COLOR_YUY422)))
            plane = plane->next;

        port_info[i].plane = plane;
        port_info[i].omaps = omaps;
        port_info[i].visibility = VisibilityPartiallyObscured;
        port_info[i].drawable = NULL;
        REGION_INIT(pScreen, &port_info[i].clip, NullBox, 0);

        adapt->pPortPrivates[i].ptr = &port_info[i];

        plane = plane->next;
    }

    adapt->nAttributes = NUM_ATTRIBUTES;
    adapt->pAttributes = Attributes;
    adapt->nImages = NUM_IMAGES;
    adapt->pImages = Images;

    adapt->PutImage = omapPutImage;
    adapt->ReputImage = omapReputImage;
    adapt->StopVideo = omapVideoStop;
    adapt->GetPortAttribute = omapGetPortAttribute;
    adapt->SetPortAttribute = omapSetPortAttribute;
    adapt->QueryBestSize = omapQueryBestSize;
    adapt->QueryImageAttributes = omapQueryImageAttributes;
    adapt->ClipNotify = omapClipNotify;

    omaps->xv_adaptors = adapt;

    xvColorKey = MAKE_ATOM("XV_COLORKEY");
    xvVSync = MAKE_ATOM("XV_OMAP_VSYNC");

    return adapt;
}

#ifdef XV
/**
 * Set up everything we need for Xv.
 */
Bool omapVideoInit(OmapScreenInfo *omaps)
{
    KdVideoAdaptorPtr *adaptors, *newAdaptors = NULL;
    KdVideoAdaptorPtr newAdaptor = NULL;
    int numAdaptors = 0;

    omaps->video_region = REGION_CREATE(omaps->screen->pScreen, NullBox, 0);
    if (!omaps->video_region)
        FatalError("omapVideoInit: couldn't create video region\n");

    omaps->xv_adaptors = NULL;

    numAdaptors = KdXVListGenericAdaptors(omaps->screen, &adaptors);
    newAdaptor = omapVideoOverlaySetup(omaps);

    if (!newAdaptor)
        return FALSE;

    if (!numAdaptors) {
        numAdaptors = 1;
        adaptors = &newAdaptor;
    }
    else {
        newAdaptors = xcalloc(numAdaptors + 1, sizeof(KdVideoAdaptorPtr *));
        if (!newAdaptors)
            return FALSE;

        memcpy(newAdaptors, adaptors, numAdaptors * sizeof(KdVideoAdaptorPtr));
        newAdaptors[numAdaptors] = newAdaptor;
        adaptors = newAdaptors;
        numAdaptors++;
    }

    KdXVScreenInit(omaps->screen->pScreen, adaptors, numAdaptors);

    if (newAdaptors)
        xfree(newAdaptors);

    return TRUE;
}

/**
 * Shut down Xv, used on regeneration.
 */
void omapVideoFini(OmapScreenInfo *omaps)
{
    REGION_DESTROY(omaps->screen->pScreen, omaps->video_region);
}
#else
Bool omapVideoInit(OmapScreenInfo *omaps)
{
    omaps->video_region = NULL;
    return TRUE;
}

void omapVideoFini(OmapScreenInfo *omaps)
{
}
#endif
