/*
 * Copyright © 2004-2006 Nokia Corporation
 *
 * Permission to use, copy, modify, distribute and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the names of the authors and/or copyright holders
 * not be used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  The authors and
 * copyright holders make no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without any express
 * or implied warranty.
 *
 * THE AUTHORS AND COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES WITH REGARD TO
 * THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS, IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR
 * ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER
 * RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
 * CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Author: Daniel Stone <daniel.stone@nokia.com>
 */

#ifndef _OMAP_H_
#define _OMAP_H_

#ifdef HAVE_KDRIVE_CONFIG_H
#include <kdrive-config.h>
#endif

#include <stdint.h>
#include <scrnintstr.h>

#include "cursor.h"
#include "cursorstr.h"
#include "picturestr.h"
#include "kdrive.h"

#ifdef XV
#include "kxv.h"
#endif

#include "omapfb.h"

#define ENTER() DebugF("Enter %s\n", __FUNCTION__)
#define LEAVE() DebugF("Leave %s\n", __FUNCTION__)

/* Minimum time between framebuffer updates, in ms. */
#define OMAP_UPDATE_TIME 50

/* Is our video dispc or external? */
#define OMAP_EXT_NONE      0
#define OMAP_EXT_CANDIDATE (1 << 0)
#define OMAP_EXT_MIGRATED  (1 << 1) 
#define OMAP_EXT_BASE_MASK (OMAP_EXT_CANDIDATE | OMAP_EXT_MIGRATED)
#define OMAP_EXT_PENDING   (1 << 15)

#define OMAP_GET_EXT(x)    ((x)->ext_state & OMAP_EXT_BASE_MASK)

#define OMAP_DEFAULT_CKEY (0x007f << 5)
#define OMAP_DEFAULT_BG (0x0000)


typedef struct _omapCardInfo OmapCardInfo;
typedef struct _omapScreenInfo OmapScreenInfo;
typedef struct _omapPlaneInfo OmapPlaneInfo;
typedef struct _omapPortPriv OmapPortPriv;

#define getOmapCardInfo(kd)      ((OmapCardInfo *) ((kd)->card->driver))
#define omapCardInfo(kd)         OmapCardInfo *omapc = getOmapCardInfo(kd)

#define getOmapScreenInfo(kd)    ((OmapScreenInfo *) ((kd)->screen->driver))
#define omapScreenInfo(kd)       OmapScreenInfo *omaps = getOmapScreenInfo(kd)

#define getOmapPortPriv(omaps, n) ((OmapPortPriv *) ((omaps)->xv_adaptors->pPortPrivates[n].ptr))
#define omapPortPriv(omaps, n)    OmapPortPriv *pPortPriv = getOmapPortPriv(omaps, n)

typedef enum omapPlaneType {
    OMAP_PLANE_BASE,
    OMAP_PLANE_OVERLAY,
} OmapPlaneType;

typedef enum omapPlaneState {
    OMAP_STATE_STOPPED,
    OMAP_STATE_GRABBED,
    OMAP_STATE_ACTIVE,
} OmapPlaneState;

struct _omapPlaneInfo {
    /* Unique identifier. */
    int id;

    OmapPlaneType type;

    /* Intrinsic properties of the plane. */
    char *filename;
    int fd;
    CARD8 *fb;
    int fb_size;
    int pitch, bpp;
    int max_width, max_height;
    unsigned long colors;

    /* Whether or not the plane is enabled.  If a plane is not active and
     * not dirty, it can be restarted without being reconfigured. */
    OmapPlaneState state;

    /* Do we need to set the plane up again? */
    int dirty;

    /* Changeable; changing any of these requires setting the
     * dirty flag. */
    xRectangle src_area;
    xRectangle dst_area;
    enum omapfb_color_format format;

    /* OMAP_EXT_* flags. */
    int ext_state;

    /* Plane capabilities, from omapfb.h. */
    unsigned long caps;

    /* Colour key, in RGB565. */
    int colorkey;

    enum {
        OMAP_VSYNC_NONE,
        OMAP_VSYNC_TEAR,
        OMAP_VSYNC_FORCE,
    } vsync;

    /* Number of frames we've drawn, for profiling. */
    int frames;
    int frames_since;

    /* Pointer back to our base screen. */
    struct _omapScreenInfo *omaps;

    struct _omapPlaneInfo *next;
};

struct _omapCardInfo {
    OmapPlaneInfo *planes;

    /* The mode which was set at startup. */
    int orig_width, orig_height, orig_bpp;
};

struct _omapScreenInfo {
    OmapCardInfo *omapc;
    KdScreenInfo *screen;

    /* Number of updates kicked from the timer. */
    int updates;

    /* Timer gets disabled after 500ms of inactivity. */
    int empty_updates, timer_active;

    /* Should we avoid aggregating updates? */
    int individual_updates;

    /* Pointer to our base plane. */
    OmapPlaneInfo *plane;

    /* The current damaged area. */
    BoxRec dirty_area;

    /* Have we enabled pixel doubling? */
    int pixel_doubled;

    /* Do we need to block UI updates? */
    int block_updates;

    /* Are we using a shadow framebuffer? */
    int shadowfb;

    /* Use for tracking damage for window updates. */
    PixmapPtr pixmap;
    DamagePtr damage;
    RegionPtr tmp_region;
    RegionPtr video_region;

    int num_video_ports;
#ifdef XV
    KdVideoAdaptorPtr xv_adaptors;
#else
    void *xv_adaptors;
#endif
};

struct _omapPortPriv {
    /* General overlay information */
    RegionRec clip;
    OmapScreenInfo *omaps;

    /* Plane currently in use. */
    OmapPlaneInfo *plane;

    /* Are we downscaling? */
    Bool hscale, vscale;

    /* Current FourCC being sent from the client. */
    int fourcc;

    /* Backing drawable. */
    DrawablePtr drawable;

    /* Visibility of the backing drawable. */
    unsigned int visibility;

    /* Timer to make sure we don't hang with updates blocked. */
    OsTimerPtr timer;

    int src_w, src_h;
    int dst_x, dst_y, dst_w, dst_h;
};


Bool omapDrawInit(ScreenPtr pScreen);
void omapDrawEnable(ScreenPtr pScreen);
void omapDrawDisable(ScreenPtr pScreen);
void omapDrawFini(ScreenPtr pScreen);

Bool omapCreateDrawResources(OmapScreenInfo *omaps);
void omapRemoveDrawResources(OmapScreenInfo *omaps);

OmapPlaneInfo *omapSetupPlane(OmapCardInfo *omapc, char *filename,
                              OmapPlaneType type);
void omapPlaneDestroy(OmapPlaneInfo *plane);
int omapPlaneEnable(OmapPlaneInfo *plane);
void omapPlaneDisable(OmapPlaneInfo *plane);

void omapSyncEngine(OmapCardInfo *omapc);
void omapForceFullScreenUpdate(OmapScreenInfo *omaps, Bool force_all);
void omapFlushDamage(OmapPlaneInfo *plane, xRectangle *area);

Bool omapVideoInit(OmapScreenInfo *omaps);
void omapVideoFini(OmapScreenInfo *omaps);

extern KdCardFuncs omapFuncs;

#endif /* _OMAP_H_ */
