/*
 * This file is part of telepathy-feed
 *
 * Copyright (C) 2006 Nokia Corporation. All rights reserved.
 *
 * Contact: Onne Gorter <onne.gorter@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <string.h>
#include <glib.h>

#include <libgalago/galago.h>

#include <dbus/dbus-glib-bindings.h>

#include <libtelepathy/tp-connmgr.h>
#include <libtelepathy/tp-conn.h>
#include <libtelepathy/tp-conn-gen.h>
#include <libtelepathy/tp-conn-iface-presence-gen.h>
#include <libtelepathy/tp-chan.h>
#include <libtelepathy/tp-chan-iface-group-gen.h>

#include "galago.h"
#include "tp-helper.h"
#include "shared.h"

extern guint debug_flags;

static void presence_handle (gpointer key, gpointer value, gpointer user_data);

/* TODO:
 * - listen for connection crashing
 * - handle statuses
 */

/* We need this one since galago_person_create_account will happily return
 * already existing accounts, because the same username on two different
 * rosters is the same account for galago but not for the rest of the world
 * So we do our own ref-counting to avoid destruction of accounts that still have
 * presence
 */
static void
extra_ref (GObject *object)
{
  guint count;

  count = GPOINTER_TO_UINT (g_object_get_data (object, "extra-ref-count"));
  count++;
  g_object_set_data (object, "extra-ref-count", GUINT_TO_POINTER (count));
}

static void
extra_unref (GObject *object)
{
  guint count;
  count = GPOINTER_TO_UINT (g_object_get_data (object, "extra-ref-count"));

  if (--count <= 0)
    g_object_unref (object);
  else
    g_object_set_data (object, "extra-ref-count", GUINT_TO_POINTER (count));
}

/**
 * Get a GalagoService from a service name.
 */
static GalagoService*
get_service (const char *service_name)
{
  static GHashTable *hash = NULL;
  GalagoService *service;

  g_assert (service_name != NULL);

  if (hash == NULL) {
    hash = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, g_object_unref);
    g_assert (hash != NULL);
  }

  service = g_hash_table_lookup (hash, service_name);
  if (service == NULL) {
    service = galago_create_service(service_name, NULL, 0) ;
    g_hash_table_insert (hash, g_strdup (service_name), service);
    p_debug ("Created service %p for %s", service, service_name);
  }

  return service;
}


/**
 * Map Telepathy status names to the Galago status enum. Note that this is wrong
 * and we should be using Presence:GetStatus() to get the status IDs from there.
 */
static GalagoStatusType
name_to_status (const char *status)
{
  g_assert (status != NULL);

  switch (status[0]) {
  case 'o': /* offline */
    return GALAGO_STATUS_OFFLINE;

  case 'h': /* hidden */
    return GALAGO_STATUS_HIDDEN;

  case 'x': /* xa */
    return GALAGO_STATUS_EXTENDED_AWAY;

  case 'b': /* busy, brb */
  case 'd': /* dnd */
    return GALAGO_STATUS_AWAY;

  case 'a': /* available, away */
    switch (status[1]) {
    case 'v': /* available */
      return GALAGO_STATUS_AVAILABLE;

    case 'w': /* away */
      return GALAGO_STATUS_AWAY;

    default:
      break;
    }

  default:
    break;
  }

  return GALAGO_STATUS_UNSET;
}

static char*
get_service_username_from_tp_handle (TpConn* conn, guint handle)
{
  GError *error = NULL;
  char *retval = NULL;

  if (!tf_tp_conn_inspect_handle (DBUS_G_PROXY (conn), TP_CONN_HANDLE_TYPE_CONTACT, handle, &retval, &error)) {
    g_warning ("Cannot get name for handle %d: %s", handle, error->message);
    g_error_free (error);
    return NULL;
  }
  return retval;
}

static char*
get_service_protocol (TpConn *conn)
{
  GError *error = NULL;
  char *retval = NULL;

  if (!tp_conn_get_protocol (DBUS_G_PROXY (conn), &retval, &error)) {
      g_warning ("Cannot get protocol: %s\n", error->message);
      g_error_free (error);
      return NULL;
    }
  return retval;
}

static GalagoPresence*
get_galago_presence_from_tp_handle (FeedContext *context, guint handle)
{
  GalagoAccount *account;
  GalagoPresence *presence;

  g_assert (context);

  account = g_hash_table_lookup (context->account_hash, GINT_TO_POINTER (handle));
  if (!account) {
    GalagoService *service;
    GalagoPerson *person;
    char *id = NULL, *protocol = NULL;

    protocol = get_service_protocol (context->conn);
    if (!protocol)
      return NULL;

    service = get_service (protocol);
    g_free (protocol);

    id = get_service_username_from_tp_handle (context->conn, handle);
    if (!id)
      return NULL;

    person = galago_create_person (NULL);
    /* Galago will happily return existing accounts here, because the same username/id on two different
     * rosters is the same account for galago but not for the rest of the world
     * so we do our own ref counting for it in order that we don't destroy accounts that still have a presence
     */
    account = galago_service_create_account (service, person, id);
    g_assert (GALAGO_IS_ACCOUNT (account));
    extra_ref (G_OBJECT (account));

    g_hash_table_insert (context->account_hash, GINT_TO_POINTER (handle), account);
  }

  presence = galago_account_create_presence (account);

  return presence;
}

static void
populate_presence (gpointer key, gpointer value, gpointer user_data)
{
  char *id = key;
  GHashTable *params = value;
  GalagoPresence *presence = user_data;
  GalagoStatus *status;
  GValue *message;
  const char *s;

  /* TODO: get code from GetStatus */
  /* TODO at some point: pass idle time */
  status = galago_status_new (name_to_status (id), id, id, TRUE);

  if (params) {
    message = g_hash_table_lookup (params, "message");
    s = message ? g_value_get_string (message) : NULL;
    if (s && s[0] != '\0') {
      galago_object_set_attr_string (GALAGO_OBJECT (status), "message", s);
    }
  }

  galago_presence_add_status (presence, status);
}

#define NICE_DEBUGGING 1

#if NICE_DEBUGGING
static void
concat_vals(gpointer key, gpointer value, gpointer userdata)
{
        char *rv, *str;

        str = *((char **)userdata);
        rv = g_strdup_printf("%s%s%s", str ? str : "", str ? ", " : "", (char*)key);
        if (str)
                g_free (str);
        *((char**)userdata) = rv;
}

static char *
get_hashtable_as_string(GHashTable *ht)
{
        char *rv = NULL;
        g_hash_table_foreach(ht, concat_vals, (gpointer)&rv);
        return rv;
}
#endif

static void
presence_handle (gpointer key, gpointer value, gpointer user_data)
{
  GalagoPresence *presence;
  GHashTable *presences;
  GValueArray *values;
  FeedContext *context = user_data;
  guint handle;

  handle = GPOINTER_TO_UINT (key);

  if (handle == context->self_handle) {
    p_debug ("Ignoring our own presence");
  }

  values = value;
  presences = g_value_get_boxed (g_value_array_get_nth (values, 1));

/* Debugging: Have nice names instead of tp-handles */
#if NICE_DEBUGGING
  if (debug_flags & DEBUG_PRESENCE) {
    char *id, *status;

    id = get_service_username_from_tp_handle (context->conn, handle);
    if (!id)
      id = g_strdup_printf ("unknown handle %d", handle);
    status = get_hashtable_as_string (presences);
    p_debug ("Handling presence for %s: %s (context %p)", id, status, context);
    g_free (id);
    g_free (status);
  }
#else
  p_debug ("Handling presence for %d (context %p)", handle, context);
#endif

  presence = get_galago_presence_from_tp_handle (user_data, handle);
  if (!presence) {
    g_warning (G_STRLOC ": could not create presence from tp_handle %d", handle);
    return;
  }

  galago_presence_clear_statuses (presence);
  g_hash_table_foreach (presences, populate_presence, presence);
}

static void
on_presence_update (DBusGProxy *proxy, GHashTable *hash, gpointer user_data)
{
  g_return_if_fail (DBUS_IS_G_PROXY (proxy));

  p_debug ("Got presence update (context %p)", user_data);
  g_hash_table_foreach (hash, presence_handle, user_data);
  p_debug ("End of update.");
}

static void
handle_member_presence (FeedContext *context,
			GArray      *members,
			GArray      *pending)
{
  int i;
  GHashTable *hash = NULL;
  GError *error = NULL;
  
  p_debug ("Members - %d, pending - %d (%p)", 
	   members->len, pending->len, context);

  for (i = 0; i < pending->len; i++) {
    GalagoPresence *presence;
    GalagoStatus *status;
    guint id = g_array_index (pending, guint, i);

    p_debug ("Handling pending for %d (%p)", id, context);
    presence = get_galago_presence_from_tp_handle (context, id);

    if (!presence) {
      g_warning (G_STRLOC ": could not create presence from tp_handle %d", id);
      continue;
    }

    status = galago_status_new (GALAGO_STATUS_PENDING,
				"pending", "pending", 
				TRUE);
    galago_presence_add_status (presence, status);
  }

  p_debug ("Getting Presence for %d contact list members", members->len);
  if (!tp_conn_iface_presence_get_presence (context->presence_proxy, members, &hash, &error)) {
    g_warning ("Could not get presences for contact list members: %s", error->message);
    g_error_free (error);
    return;
  }
  g_hash_table_foreach (hash, presence_handle, context);
  p_debug ("Presences stored.");
}

static void
on_members_changed (DBusGProxy *proxy,
                   char *message,
                   GArray *added,
                   GArray *removed,
                   GArray *local_pending,
                   GArray *remote_pending,
                   guint actor,
                   guint reason,
                   gpointer user_data)
{
  FeedContext *context;
  GArray *pending;
  int i;

  context = user_data;

  g_return_if_fail (DBUS_IS_G_PROXY (proxy));
  g_return_if_fail (context);

  p_debug ("%d members to be removed", removed->len);
  for (i = 0; i < removed->len; i++) {
    guint id = g_array_index (removed, guint, i);

    p_debug ("Removing %d (%p)", id, context);
    g_hash_table_remove (context->account_hash, GINT_TO_POINTER (id));
  }

  pending = g_array_new(FALSE, FALSE, sizeof(guint));
  g_array_append_vals (pending, local_pending->data, local_pending->len);
  g_array_append_vals (pending, remote_pending->data, remote_pending->len);

  handle_member_presence (context, added, pending);

  g_array_free (pending, TRUE);
}

/* Disconnect the signals */
void
context_disconnected (FeedContext *context)
{
  p_debug ("Disconnecting context %p", context);

  g_return_if_fail (context->status == 2);
  context->status = (guint)-2;

  if (context->presence_proxy)
    dbus_g_proxy_disconnect_signal (context->presence_proxy, "PresenceUpdate",
                                    G_CALLBACK (on_presence_update), context);
  if (context->group_proxy)
    dbus_g_proxy_disconnect_signal (context->group_proxy, "MembersChanged",
                                    G_CALLBACK (on_members_changed), context);

  /* Do not unref proxies returned by tp_[chan|conn]_get_interface */
  context->group_proxy = NULL;
  context->presence_proxy = NULL;

  if (context->chan) {
    g_object_unref (context->chan);
    context->chan = NULL;
  }

  if (context->conn) {
    g_object_unref (context->conn);
    context->conn = NULL;
  }

  if (context->account_hash) {
    g_hash_table_destroy (context->account_hash);
    context->account_hash = NULL;
  }
}

void
context_connected (FeedContext *context)
{
  GError *error = NULL;
  GArray *members;
  GArray *local_pending;
  GArray *remote_pending;

  p_debug ("Connecting context %p", context);

  /* Our code assumes that a context is destroyed after tp-connection is closed
   * If we get here for a second time, we somehow missed the disconnect signal
   */
  g_return_if_fail (context->account_hash == NULL);
  
  if (!context->list_handle) {
    if (!tf_tp_conn_request_handle (DBUS_G_PROXY (context->conn),
  				    TP_CONN_HANDLE_TYPE_LIST,
				    "subscribe", &context->list_handle, &error)) {
      g_warning ("Cannot get handle: %s", error->message);
      g_error_free (error);
      return;
    }
  }
  
  if (!context->self_handle) {
    if (!tp_conn_get_self_handle (DBUS_G_PROXY (context->conn), &context->self_handle, &error)) {
      g_warning ("Cannot get self handle: %s", error->message);
      g_error_free (error);
      return;
    }
  }

  if (!context->chan) {
    context->chan = tp_conn_new_channel (gconnection, context->conn,
			                 dbus_g_proxy_get_bus_name (DBUS_G_PROXY (context->conn)),
			                 TP_IFACE_CHANNEL_TYPE_CONTACT_LIST,
			                 TP_CONN_HANDLE_TYPE_LIST, context->list_handle,
			                 FALSE);
    if (!context->chan) {
      g_warning ("Cannot open channel");
      return;
    }
    g_object_add_weak_pointer (G_OBJECT (context->chan),
			       (gpointer)&context->chan);
  }

  if (!context->group_proxy) {
    context->group_proxy = tp_chan_get_interface (context->chan, TELEPATHY_CHAN_IFACE_GROUP_QUARK);
    if (context->group_proxy == NULL) {
      g_warning ("Cannot get subscribe group interface for connection");
      return;
    }
    g_object_add_weak_pointer (G_OBJECT (context->group_proxy),
			       (gpointer)&context->group_proxy);
  }

  if (!context->presence_proxy) {
    context->presence_proxy = tp_conn_get_interface (context->conn, TELEPATHY_CONN_IFACE_PRESENCE_QUARK);
    if (context->presence_proxy == NULL) {
      g_warning ("Cannot get presence interface for connection");
      return;
    }
    g_object_add_weak_pointer (G_OBJECT (context->presence_proxy),
			       (gpointer)&context->presence_proxy);
  }

  if (!tp_chan_iface_group_get_all_members (context->group_proxy, &members, &local_pending, &remote_pending, &error)) {
    g_warning ("Cannot get subscribed members: %s", error->message);
    g_error_free (error);
    return;
  }

  /* If we have come this far, all calls should have succeeded
   * Now set up the signal handlers and let's go
   */
  context->account_hash = g_hash_table_new_full (NULL, NULL, NULL, (GDestroyNotify)extra_unref);
  dbus_g_proxy_connect_signal (context->presence_proxy, "PresenceUpdate",
			       G_CALLBACK (on_presence_update),
			       context, NULL);
  dbus_g_proxy_connect_signal (context->group_proxy, "MembersChanged",
			       G_CALLBACK (on_members_changed),
			       context, NULL);
  
  g_array_append_vals (local_pending, remote_pending->data, remote_pending->len);
  handle_member_presence (context, members, local_pending);

  g_array_free (members, TRUE);
  g_array_free (local_pending, TRUE);
  g_array_free (remote_pending, TRUE);
}
