/*
 * Copyright (c) 2003, 2004 Nokia
 * Author: tsavola@movial.fi
 *
 * This program is licensed under GPL (see COPYING for details)
 */

#ifndef MOUNT_H
#define MOUNT_H

#include "types.h"

#define PROC_MOUNTS "/proc/mounts"

typedef enum {
	/** mount info type: NFS */
	MTYPE_NFS  = 1,

	/** mount info type: path should be bound to a mount point */
	MTYPE_BIND = 2
} mtype_t;

struct mount_info_s;
struct mount_s;

/** Describes how stuff should be mounted. */
typedef struct mount_info_s {
	/** Mount type. */
	int16_t type;

	/** Mount point path. */
	char *point;

	/** Path to special file or whatever should be mounted... */
	char *device;

	/** Options for the mount command (-o). */
	char *opts;

	/** Device number of DEVICE (only applicable for MTYPE_NFS). */
	dev_t device_dev;
} mount_info_t;

/** An entry in a doubly linked list of mount points. */
typedef struct mount_s {
	struct mount_s *prev;
	struct mount_s *next;

	/** How many processes are using this mount? */
	unsigned int usage;

	/** When will the mount expire? Relevant when usage is 0. */
	time_t expiration;

	/** Properties of the mount entry. */
	struct mount_info_s info;

	/** Device number of POINT (only applicable for MTYPE_NFS). */
	dev_t point_dev;
} mount_t;

#define mntinfo_alloc()  calloc(1, sizeof (mount_info_t))
extern void mntinfo_free(mount_info_t *);
extern int mntinfo_copy(mount_info_t *, mount_info_t *);
extern mount_info_t *mntinfo_parse(const char *);
extern int mntinfo_stat_device(mount_info_t *);
extern void mntinfo_sort_vec(mount_info_t **);

#endif
