/**
 * @file ui_search.c everything about searching
 *
 * Copyright (C) 2003, 2004 Lars Lindner <lars.lindner@gmx.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#define _BSD_SOURCE
 
#include <string.h>
#include <strings.h>
#include <stdlib.h>
#include <stdio.h>
#include <hildon-widgets/gtk-infoprint.h>
#include "callbacks.h"
#include "interface.h"
#include "feed.h"
#include "folder.h"
#include "rule.h"
#include "vfolder.h"
#include "support.h"
#include "common.h"
#include "appdata.h"
#include "update.h"
#include "gtkhtml/gtkhtml_plugin.h"

#include <osso-log.h>

#define HISTORY_SIZE 5

#ifdef UPDATE_DEBUG_MUTEX
#define G_MUTEX_LOCK(mx) { ULOG_DEBUG("##Trying to lock mutex: %s, %d, %p", __FUNCTION__, __LINE__, g_thread_self()); g_mutex_lock(mx); ULOG_DEBUG("##Locked mutex: %s, %d, %p", __FUNCTION__, __LINE__, g_thread_self()); }
#define G_MUTEX_UNLOCK(mx) { ULOG_DEBUG("##Trying to unlock mutex: %s, %d, %p", __FUNCTION__, __LINE__, g_thread_self()); g_mutex_unlock(mx); ULOG_DEBUG("##Unlocked mutex: %s, %d, %p", __FUNCTION__, __LINE__, g_thread_self()); }
#define GDK_THREADS_ENTERD { ULOG_DEBUG("@@Trying gdk_threads_enter: %s, %d, %p", __FUNCTION__, __LINE__, g_thread_self()); if (g_thread_self() != mainThread) gdk_threads_enter(); ULOG_DEBUG("@@Success gdk_threads_enter: %s, %d, %p", __FUNCTION__, __LINE__, g_thread_self()); }
#define GDK_THREADS_LEAVED { ULOG_DEBUG("@@Trying gdk_threads_leave: %s, %d, %p", __FUNCTION__, __LINE__, g_thread_self()); if (g_thread_self() != mainThread) gdk_threads_leave(); ULOG_DEBUG("@@Success gdk_threads_leave: %s, %d, %p", __FUNCTION__, __LINE__, g_thread_self()); }
#else
#define G_MUTEX_LOCK(mx) g_mutex_lock(mx);
#define G_MUTEX_UNLOCK(mx) g_mutex_unlock(mx);
#define GDK_THREADS_ENTERD { if (g_thread_self() != mainThread) gdk_threads_enter(); }
#define GDK_THREADS_LEAVED { if (g_thread_self() != mainThread) gdk_threads_leave(); }
#endif

extern GThread *mainThread;
extern GtkWidget *mainwindow;
extern AppData *app_data;
//extern guint rss_search_found;
//extern GtkWidget * gtkhtml_htmlview;

feedPtr searchFeed = NULL;

/************************************************************************/
/*                        PUBLIC FUNCTIONS                              */
/************************************************************************/

/*------------------------------------------------------------------------------*/
/* search dialog callbacks							*/
/*------------------------------------------------------------------------------*/

gboolean start_search(gpointer data)
{
//    AppUIData *app_ui_data = (AppUIData *)data;

    gdk_threads_enter();
    /* search through the feed list now. This will call ui_feedlist__do_for_all_data() */
    ULOG_DEBUG("??????Start_search: vfolder_refresh start: %d, %d",
        app_data->app_ui_data->rss_search_found,
        app_data->app_ui_data->search_mode);

    vfolder_refresh(searchFeed);

    ULOG_DEBUG("!!!!!!Start_search: vfolder_refresh completed: %d, %d",
        app_data->app_ui_data->rss_search_found,
        app_data->app_ui_data->search_mode);
    /* If no items were found the counter must be manually updated to show the results, since
       this is normally done through feed_add_item() function, which is not called if
       no items are found */
    if ((app_data->app_ui_data->rss_search_found == 0) &&
            (app_data->app_ui_data->search_mode == SFM_SEARCH))
    	ui_mainwindow_construct_search_view();

    /* better call it here so that search_mode won't cause any errors
        or even progress bar problems, as search feed might require 
        progress bar to show downloading images from internet too
     */
    // if it wasn't interrupted
    if (app_data->app_ui_data->search_mode == SFM_SEARCH)
    {
        gtkhtml_finish_search_feed(); // close the stream
        switch_progressbar(PROGRESSBAR_NORMAL_MODE);
//        if (SFM_SEARCH == app_ui_data->search_mode) {
            gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->app), 
                _("rss_ib_search_completed"));
//        }
//        ui_on_search_finished();
    }    

    
    // reload the html page
    
    //displayed_node = (nodePtr)searchFeed;

    //gtkhtml_write_html( app_data->app_ui_data->html_scrollpane, "crap" , "crap", (nodePtr) searchFeed);
    
    ULOG_DEBUG("on_searchentry_activate: Setting feed displayed to TRUE");
    app_data->app_ui_data->feed_displayed = TRUE;
   
    ui_mainwindow_empty_search_field();

    gtk_widget_set_sensitive(GTK_WIDGET
            (app_data->app_ui_data->menuitem_search_results), TRUE);


    gdk_threads_leave();

    return FALSE;
}

void on_searchentry_activate(GtkButton * button, gpointer user_data)
{
    int i = 0;
    gchar *entry_text = NULL;

    /* This used to hang the program for unknown reason. However, 
       that is no longer the case so I suppose the platform had 
       an issue with this earlier. */

    AppUIData *app_ui_data;
     
    g_assert(app_data != NULL);
    
    app_ui_data = app_data->app_ui_data;

    g_assert(app_ui_data != NULL);
    
    ULOG_DEBUG(__FUNCTION__);

    if (SFM_CONNECTING == app_ui_data->search_mode) {
        gtk_widget_queue_draw(app_ui_data->search_button);

        request_iap_disconnect();
//        download_cancel_all();
        return;
    } else
    /* If a refresh is being processed, cancel it and go on with the search.
       search_mode is set back to SFM_NORMAL in on_refresh_finished() */
    if (SFM_REFRESH == app_ui_data->search_mode) {
        app_ui_data->search_mode = SFM_INTERRUPTED;

        /* Lets wait for the GUI events. Without this, the button press
           graphics dont get updated right */
/*        while (gtk_events_pending ())
	    gtk_main_iteration ();*/
        gtk_widget_queue_draw(app_ui_data->search_button);

	download_cancel_all(TRUE);
	
	/* Loading the feed directory uses the same SFM_REFRESH flag
	   as a normal refresh. feed_directory_loading is set if the feed directory
	   is being loaded and the stop button has a little bit different
	   functionality in that case */
// 	if(app_ui_data->feed_directory_loading == TRUE)
// 	{
// 	    app_ui_data->feed_directory_loading = FALSE;
// 	    on_refresh_finished(TRUE, TRUE);
// 	}
// 	else
// 	    on_refresh_finished(FALSE, FALSE);
	
	return;
    }
    /* If a search is on, search_mode is set to SFM_INTERRUPTED,
       which in turn forces the search to end and change the value
       of search_mode back to SFM_NORMAL */
    else if (SFM_SEARCH == app_ui_data->search_mode) {
        stop_search();
	   //app_ui_data->search_mode = SFM_INTERRUPTED;
	return;
    }
    /* If the user clicks the button repeatedly, do nothing. Let the program
       set search_mode back to SFM_NORMAL first. The search is processed
       only when search_mode is SFM_NORMAL */
    else if (SFM_INTERRUPTED == app_ui_data->search_mode)
	return;

    GtkWidget *searchentry = app_ui_data->search_field;

    gchar *searchstring;

    g_assert(searchentry != NULL);

    GtkWidget *entryfield = GTK_WIDGET(GTK_BIN(searchentry)->child);
    g_assert(entryfield != NULL);

    searchstring =
	trim_whitespaces(g_strdup
			 (gtk_entry_get_text(GTK_ENTRY(entryfield))));
    g_assert(searchstring != NULL);

    if ((0 == *searchstring) || (strcmp(searchstring, " ") == 0 )){
    	gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->app), _("rss_ib_enter_key_words"));
	   g_free(searchstring);
	return;
    }
    
    
    /* Remove duplicates */
    for(i =  0;i < app_ui_data->search_string_count; i++) {
        gtk_combo_box_set_active(GTK_COMBO_BOX(searchentry), i);
	entry_text = gtk_combo_box_get_active_text(GTK_COMBO_BOX(searchentry));
        if(!strcmp(entry_text, searchstring)) {
            gtk_combo_box_remove_text(GTK_COMBO_BOX(searchentry), i);
            app_ui_data->search_string_count--;
            g_free(entry_text);
            break;
        }
	g_free(entry_text);
    }
    gtk_combo_box_prepend_text(GTK_COMBO_BOX(searchentry), searchstring);
    app_ui_data->search_string_count++;

    while (app_ui_data->search_string_count > HISTORY_SIZE) {
	gtk_combo_box_remove_text(GTK_COMBO_BOX(searchentry), app_ui_data->search_string_count-1);
	ULOG_DEBUG("Removed the last child");
	app_ui_data->search_string_count--;
    }

    ui_itemlist_clear();
    ULOG_DEBUG("Itemlist cleared");

    if (NULL != searchFeed) {
	feed_free(searchFeed);
	searchFeed = NULL;
	ULOG_DEBUG("searchFeed freed");
    }

    ULOG_INFO("Search string: %s", searchstring);

    /* is this to make sure searchFeed will have the item list from root??? */
    g_assert(searchFeed == NULL);
    searchFeed = vfolder_new();
    ULOG_DEBUG("new searchFeed created");

    feed_set_title(searchFeed, searchstring);
    ULOG_DEBUG("searchFeed title set to: %s", searchstring);
    //tvh 2006May10 set time for searchFeed 
    g_get_current_time(&(searchFeed->lastPoll));
    //get_corrected_time(&(searchFeed->lastPoll));

    /* check for single/double quotes */
    int slen = strlen(searchstring);
    if ((slen > 1)
	&&
	(((index(searchstring, '\'') == searchstring) &&
	  (rindex(searchstring, '\'') == searchstring + slen - 1)
	 )
	 ||
	 ((index(searchstring, '\"') == searchstring) &&
	  (rindex(searchstring, '\"') == searchstring + slen - 1)
	 )
	)
	) {
	//string was surrounded by quotes, so add it as a single rule (without the quotes)
	ULOG_DEBUG("Quoted search string: %s", searchstring);
	searchstring[slen - 1] = '*';
//	searchstring[slen - 1] = 0;
//	gchar *tmp = searchstring + 1;
	*searchstring = '*';
	ULOG_DEBUG("Unquoted search string: %s", searchstring);
	vfolder_add_rule(searchFeed, "exact", searchstring, TRUE);
    } else {
	//one or more words without surrounding quotes, so add every words as an own rule
	gchar *tmp, *tmp2;
	tmp = searchstring;
	while ((tmp2 = (gchar *) strsep(&tmp, " \t")) != NULL) {
	    ULOG_DEBUG("Search word: %s (added as a rule)", tmp2);
	    vfolder_add_rule(searchFeed, "exact", tmp2, TRUE);
	}
    }

    /*NOTE: if the rules seem to crash or something, check whether vfolder_add_rule 
    copies the string or just uses it*/

    // the old one  vfolder_add_rule(searchFeed, "exact", searchstring, TRUE);
    ULOG_DEBUG("vfolder rule added");
    switch_progressbar(PROGRESSBAR_SEARCH_MODE);

    ui_mainwindow_set_search(TRUE);

    ui_feedlist_select(NULL);

    ui_itemlist_search_init(); /* unloading stuff */


    // let's load the page
    gtkhtml_start_write_search_feed((nodePtr) searchFeed);
    displayed_node = (nodePtr)searchFeed;

    /* re-display the number of items */
    ui_mainwindow_construct_search_view();

    g_idle_add(start_search, app_ui_data);

    g_free(searchstring);
}

void on_searchresults_menuitem_selected(GtkWidget * menuitem,
					gpointer user_data)
{
    if (NULL == searchFeed) {
      gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->main_menu_bar), _("rss_ib_no_previous_search_results"));
	return;
    }

    /* If already doing the search, nothing should be done here */
    if (app_data->app_ui_data->search_mode == SFM_SEARCH)
        return;

    ULOG_DEBUG("on_searchresults_menuitem_selected");
    ui_feedlist_select(NULL);
    //feeds might have been deleted. So we need this
    ui_mainwindow_construct_search_view();
    ui_mainwindow_set_search(TRUE);
    ui_itemlist_load((nodePtr) searchFeed);
}

void ui_search_clear_search_results()
{
    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);
    
    if(searchFeed != NULL)
    {
        feed_free(searchFeed);
	   searchFeed = NULL;
    }
    gtk_widget_set_sensitive(GTK_WIDGET
			     (app_data->app_ui_data->menuitem_search_results), FALSE);
    ui_mainwindow_set_search(FALSE);    
}
