/**
 * @file ui_mainwindow.c some functions concerning the main window 
 *
 * Copyright (C) 2004 Nathan J. Conrad <t98502@users.sourceforge.net>
 * Copyright (C) 2004 Lars Lindner <lars.lindner@gmx.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 * 
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef UI_MAINWINDOW_H
#define UI_MAINWINDOW_H

#include <gtk/gtk.h>

#include "appdata.h"

#if GTK_CHECK_VERSION(2,4,0)
#define TOOLBAR_ADD(toolbar, label, icon, tooltips, tooltip, function) \
 do { \
	GtkToolItem *item = gtk_tool_button_new(gtk_image_new_from_icon_name (icon, GTK_ICON_SIZE_LARGE_TOOLBAR), label); \
	gtk_tool_item_set_tooltip(item, tooltips, tooltip, NULL); \
	gtk_tool_item_set_homogeneous (item, FALSE); \
	gtk_tool_item_set_is_important (item, TRUE); \
	g_signal_connect((gpointer) item, "clicked", G_CALLBACK(function), NULL); \
	gtk_toolbar_insert(GTK_TOOLBAR(toolbar), \
				    item, \
				    -1); \
 } while (0);
#else
#define TOOLBAR_ADD(toolbar, label, icon, tooltips, tooltip, function)      \
 gtk_toolbar_append_item(GTK_TOOLBAR(toolbar), \
					label, \
					tooltip, \
					NULL, \
					gtk_image_new_from_icon_name (icon, GTK_ICON_SIZE_LARGE_TOOLBAR), \
					G_CALLBACK(function), NULL)

#endif

#define ADD_TOOLBAR_BUTTON(buttonvar,filename,callbackfunc)\
  buttonvar=GTK_TOOL_ITEM( gtk_tool_button_new(\
    gtk_image_new_from_icon_name(filename, HILDON_ICON_SIZE_TOOLBAR), NULL));\
  gtk_widget_show_all(GTK_WIDGET(buttonvar));\
  GTK_WIDGET_UNSET_FLAGS(buttonvar, GTK_CAN_FOCUS);\
  g_signal_connect(G_OBJECT(buttonvar), "clicked", G_CALLBACK(callbackfunc), app_ui_data);\
  gtk_toolbar_insert(GTK_TOOLBAR(app_ui_data->main_toolbar), GTK_TOOL_ITEM(buttonvar), -1);
#define ADD_TOOLBAR_SEPARATOR separator = gtk_separator_tool_item_new(); \
  gtk_widget_show(GTK_WIDGET(separator));\
  gtk_separator_tool_item_set_draw (GTK_SEPARATOR_TOOL_ITEM (separator), TRUE);\
  gtk_toolbar_insert (GTK_TOOLBAR(app_ui_data->main_toolbar),separator, -1);

typedef enum {
    CONFRESP_YES = 1,
    CONFRESP_NO,
    CONFRESP_OK,
    CONFRESP_CANCEL,
    CONFRESP_ERROR
} ConfirmationResponse;

extern GtkWidget *mainwindow;
extern GtkWidget *newsbox, *feedlist;

/* 2 or 3 pane mode flag from ui_mainwindow.c */
extern gboolean itemlist_mode;

/**
 * Create a new main window
 */
void ui_create_main_window(AppData * app_data);

/** Displays the statistics to the screen
  */
void ui_mainwindow_finish(void);

/**
 * Set whether to show search-view with number of found matches or not.
 * Requires searchview to have been created.
 *
 * @param search TRUE to show number of matches above htmlview.
 */
void ui_mainwindow_set_search(gboolean search);

/**
 * Meant to select between two and three pane view.
 * Three pane view is not used in OSSO version, 
 * so this function now simply
 * sets up htmlview inside search-view.
 *
 * @param threePane Always FALSE
 */
void ui_mainwindow_set_mode(gboolean threePane);

GtkWidget *ui_mainwindow_get_active_htmlview();

/** Sets the toolbar to a particular style
 * @param window main window containing toolbar
 * @param toolbar_style text string containing the type of style to use
 */
void ui_mainwindow_set_toolbar_style(GtkWindow * window,
				     const gchar * toolbar_style);

typedef void (*fileChoosenCallback) (const gchar * title,
				     gpointer user_data);

void ui_help_cb();

/**
 * Delete a feed or folder.
 * Confirm deletion and check that the feed or folder
 * can be deleted.
 *
 * @return TRUE if the deletion was successful, FALSE otherwise
 */
gboolean delete_feed(nodePtr);


/**
 * Set folders check menu item state.
 *
 * @param app_ui_data AppUIData object
 * @param state TRUE to activate, FALSE to deactivate
 */
void ui_set_folders_menu_item_state(AppUIData * app_ui_data,
				    gboolean state);


/**
 * Set fullscreen check menu item state.
 *
 * @param app_ui_data AppUIData object
 * @param state TRUE to activate, FALSE to deactivate
 */
void ui_set_fullscreen_menu_item_state(AppUIData * app_ui_data,
				       gboolean state);


/**
 * Set toolbar check menu item state.
 *
 * @param app_ui_data AppUIData object
 * @param state TRUE to activate, FALSE to deactivate
 */
void ui_set_toolbar_full_menu_item_state(AppUIData * app_ui_data,
					 gboolean state);


/**
 * Set toolbar check menu item state.
 *
 * @param app_ui_data AppUIData object
 * @param state TRUE to activate, FALSE to deactivate
 */
void ui_set_toolbar_normal_menu_item_state(AppUIData * app_ui_data,
					   gboolean state);


/**
 * Set folder list state (show folders or not).
 *
 * @param app_ui_data AppUIData object
 * @param state TRUE to show folders, FALSE not to show
 * @param update_menu_item TRUE to update menu item state,
 *   FALSE not to update
 */
void ui_set_folder_list_state(AppUIData * app_ui_data,
			      gboolean state, gboolean update_menu_item);


/**
 * Set fullscreen state.
 *
 * @param app_ui_data AppUIData object
 * @param state TRUE for fullscreen, FALSE not fullscreen
 * @param update_menu_item TRUE to update menu item state,
 *   FALSE not to update
 */
void ui_set_fullscreen_state(AppUIData * app_ui_data,
			     gboolean state, gboolean update_menu_item);


/**
 * Show confirmation note for deletion of a folder.
 *
 * @param app_ui_data AppUIData object
 * @param n_feeds number of feeds in the folder
 * @return user's choice:
 *   CONFRESP_OK for OK
 *   CONFRESP_CANCEL for CANCEL
 *   CONFRESP_ERROR on error
 */
ConfirmationResponse
ui_show_folder_delete_confirmation(AppUIData * app_ui_data, gint n_feeds);

/**
 * Show confirmation note for deletion of a feed.
 *
 * @param app_ui_data AppUIData object
 * @param feed_title title of the feed to delete
 * @return user's choice:
 *   CONFRESP_OK for OK
 *   CONFRESP_CANCEL for CANCEL
 *   CONFRESP_ERROR on error
 */
ConfirmationResponse
ui_show_feed_delete_confirmation(AppUIData * app_ui_data,
				 gchar * feed_title);

/** This function is called when the program is closed
  *
  * @param widget passed to on_quit but not used
  * @param event passed to on_quit but not used
  * @param user_data passed to on_quit but not used
  * @return if GConf key SHOW_TRAY_ICON is set to TRUE, this function returns
  * TRUE
  */			 
gboolean
on_close(GtkWidget * widget, GdkEvent * event, gpointer user_data);				 

/** window_state_event signal handler
  *
  * @param widget not used
  * @param event a structure for determining the event type
  * @param user_data not used
  * @return always FALSE
  */
gboolean
on_mainwindow_window_state_event(GtkWidget * widget,
				 GdkEvent * event, gpointer user_data);
				 
/** Updates the search view with the number of results found during search
  */				 
void ui_mainwindow_construct_search_view();				 

/** Shows the statistics on the right hand side of the 
  * screen and clears the itemlist
  */
void ui_mainwindow_show_statistics();

/** Saves the search history to a file
  */
void ui_mainwindow_save_search_entries();

/** Loads the search history from a file
  */
void ui_mainwindow_load_search_entries();

/** Empties the search field
  */
void ui_mainwindow_empty_search_field();
				 
void on_unsubscribe_clicked(GtkButton * button, gpointer user_data);

#endif
