/**
    @file ui_feed_directory.c

    Feed directory functionality and dialogs.

    Copyright (c) 2004-2006 Nokia Corporation. All rights reserved.
	
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
    02111-1307 USA
*/


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <libxml/parser.h>
#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include <hildon-widgets/gtk-infoprint.h>

#include <osso-log.h>

#define DBUS_API_SUBJECT_TO_CHANGE
#include "interface.h"
#include "ui_mainwindow.h"
#include "ui_itemlist.h"
#include "ui_mainwindow.h"
#include "ui_queue.h"
#include "ui_feed_directory.h"
#include "htmlview.h"
#include "debug.h"
#include "update.h"
#include "opml.h"
#include "support.h"
#include "osso-helplib.h"
#include "dbus.h"
#include "conf.h"

#define FEED_DIRECTORY_DIALOG_X_SIZE 500
#define FEED_DIRECTORY_DIALOG_Y_SIZE 300

GtkWidget *fd_dialog;		/* Feed directory dialog                       */
GtkWidget *dialog_vbox2;	/* Toplevel vbox for the feed directory dialog */
GtkWidget *dialog_vbox3;	/* Vbox for the checkboxes                     */
GtkWidget *combobox;		/* Combobox for selecting the category         */
GtkWidget *label;               /* The label for the category list             */
GtkWidget *dialog_scrolled;

gchar *fd_source; 
folderPtr fd_folderPtr;
//gchar * feed_directory_source_value;
GSList *category_list;		/* List of categories                          */

gboolean can_quit = TRUE;

extern AppData *app_data;

/************************************************************************/
/*                        PRIVATE FUNCTIONS                             */
/************************************************************************/

/** Collects the selected checkbox values to the list of feeds
  * @param widget the checkbox for collecting the data
  * @param user_data NULL
  */
static void ui_feed_directory_collect_selected(GtkWidget * widget,
					       gpointer user_data)
{
    fdNodePtr node = NULL;

    node = (fdNodePtr) gtk_object_get_user_data(GTK_OBJECT(widget));

    if (node)
        node->selected = GTK_TOGGLE_BUTTON(widget)->active;

    gtk_widget_hide(widget);
}

static gint
viewport_get_height (GtkViewport *viewport)
{
    GtkWidget *widget = GTK_WIDGET (viewport);
    GtkAllocation *allocation = &widget->allocation;
    gint border_width = GTK_CONTAINER (viewport)->border_width;
    gint result = 0;

    if (viewport->shadow_type != GTK_SHADOW_NONE)
        result = widget->style->ythickness;

    return MAX (1, allocation->height - result * 2 - border_width * 2);
}

static gboolean 
cb_focus_in(GtkWidget *widget, GdkEventFocus *event, gpointer user_data)
{
/*    ULOG_DEBUG("Focus in: (%d,%d)(%d,%d)", widget->allocation.x, widget->allocation.y,
        widget->allocation.width, widget->allocation.height);*/
    GtkScrolledWindow *w = dialog_scrolled;
    GtkViewport *vp = gtk_bin_get_child(GTK_BIN(w));
    GtkAdjustment *adj = gtk_scrolled_window_get_vadjustment(w);
    gint top = (gint)gtk_adjustment_get_value(adj), 
         height = viewport_get_height(vp);
/*    ULOG_DEBUG("Alloc: %d,%d", 
        top,
        height);*/
    if (widget->allocation.y < top)
        gtk_adjustment_set_value(adj, widget->allocation.y);
    else if (widget->allocation.y+widget->allocation.height > top+height)
        gtk_adjustment_set_value(adj, widget->allocation.y - height +
            widget->allocation.height);
    return FALSE;
}

/** Fills the feedlist according to the category selection so that
  * there is a checkbox for each feed
  *
  * @param category the selected category
  */
static void ui_feed_directory_fill_feedlist(gchar * category)
{
    GSList *iter;

    iter = category_list;
    if (category)
        while (iter) {
            fdNodePtr node;

            node = (fdNodePtr) iter->data;
            iter = iter->next;

            if (node->url == NULL && !strcmp(category, node->title))
                break;
        }

    gtk_container_foreach(GTK_CONTAINER(dialog_vbox3),
			  (GtkCallback) gtk_widget_hide, NULL);

    while (iter) {
        fdNodePtr node;

        node = (fdNodePtr) iter->data;
        iter = iter->next;

        if (node->url == NULL)
            break;

        if (node->title == NULL)
            node->title = g_strdup("");

        if (node->checkbutton == NULL) {
            GtkWidget *checkbutton = node->checkbutton =
            gtk_check_button_new_with_label(node->title);
            gtk_object_set_user_data(GTK_OBJECT(checkbutton),
                         node);
            gtk_widget_show(checkbutton);
            g_signal_connect(G_OBJECT(checkbutton), "focus-in-event", 
                G_CALLBACK(cb_focus_in), NULL);
            gtk_box_pack_start(GTK_BOX(dialog_vbox3), checkbutton, FALSE,
                       FALSE, 0);
        } else {
            gtk_widget_show(node->checkbutton);
        }
    }
}

/** Callback listening the combobox changed signal
  *
  * @param widget the combobox to listen
  * @param user_data NULL
  */
static void ui_feed_directory_combobox_changed(GtkWidget * widget,
					       gpointer user_data)
{
    gchar *category =
	gtk_combo_box_get_active_text(GTK_COMBO_BOX(combobox));
    ui_feed_directory_fill_feedlist(category);
    g_free(category);
}

/** Fills the combobox with the category list provided by an external server
*	@return 0 if the list is empty (because we get a wrong file)
  */
static int ui_feed_directory_fill_categories()
{
    int categories_found = 0;
    GSList *iter;
        
    ULOG_DEBUG("Testing if category list empty, size: %d",
	       g_slist_length(category_list));

    if (category_list == NULL)
	return 0;

    ULOG_DEBUG("Filling categories...");
    iter = category_list;
    while (iter) {
	fdNodePtr node;

	node = (fdNodePtr) iter->data;
    iter = iter->next;

	if (node->url != NULL || node->title == NULL)
	    continue;
        
	categories_found++;
	if (GTK_IS_COMBO_BOX(combobox))
	{
	     ULOG_DEBUG("Filling combobox with: %s", node->title);
	     gtk_combo_box_append_text(GTK_COMBO_BOX(combobox), node->title);
	}
    }
    
    if(categories_found == 0)
    {
        ui_feed_directory_fill_feedlist(NULL);
        gtk_widget_set_sensitive(combobox, FALSE);
        gtk_widget_set_sensitive(label, FALSE);
    }
    else 
        gtk_combo_box_set_active (GTK_COMBO_BOX(combobox),0);
	return 1;
}

/** Frees allocated memory
  */
static void ui_feed_directory_free()
{
    ULOG_DEBUG ("ui_feed_directory_free...\n");
    GSList *iter;

    if (category_list != NULL) {
    iter = category_list;
	while (iter) {
	    fdNodePtr node;

	    node = (fdNodePtr) iter->data;
        iter = iter->next;

	    if (node != NULL)
	    {
	        if (node->title != NULL)
		    g_free(node->title);
		    
	        if (node->url != NULL)
		    g_free(node->url);
		    
	     }
         //check button will be destroyed when parent dialog is destroyed
         g_free(node);
	}
	g_slist_free(category_list);
    }

//    if (fd_source)
  //      g_free(fd_source);
    /* Free fd_folderPtr */
    //can't free the folder itself. node only refers to it as a pointer
    /* NOTE: Use folder_free() */
    //g_free(fd_folderPtr);
    //if (fd_folderPtr)
    //    folder_free(fd_folderPtr);
}

/** Called when the dialog buttons are being pressed
  *
  * @param widget the dialog
  * @param user_data NULL
  */
static void ui_feed_directory_dialog_response(GtkWidget * widget,
					      gint response,
					      gpointer user_data)
{
    ULOG_DEBUG("ui_feed_directory_dialog_response.....");
    GSList *iter;
    
    folderPtr folder = (folderPtr) user_data;

    ULOG_DEBUG("Response %d",response);
    
    if(!can_quit)
        return;
	
    if (response == GTK_RESPONSE_OK) {
        gtk_container_foreach(GTK_CONTAINER(dialog_vbox3),
			      ui_feed_directory_collect_selected, NULL);

        iter = category_list;
        while (iter) {
	       fdNodePtr node;

            node = (fdNodePtr) iter->data;
            iter = iter->next;
            //Temporarily commented out this DEBUG. Is it needed though?    
	       //ULOG_DEBUG("Checking node: %s, selected: %d", node->title,
		      //     node->selected);
	       if (node->url != NULL && node->selected)
            /* Also pass 'folder' as the param */
		  ui_feedlist_new_subscription_for_fd(node->url, NULL,
					            FEED_REQ_RESET_TITLE |
					            FEED_REQ_RESET_UPDATE_INT |
					            FEED_REQ_AUTO_DISCOVER, folder, TRUE, FALSE);
    	}
      if (app_data->app_ui_data->connect_at_the_end)
      {
        request_iap_connect();
      }
    	ui_feedlist_update();
    }
    ui_feed_directory_free();

    //gtk_widget_hide(fd_dialog);
    gtk_widget_destroy(fd_dialog);
    app_data->app_ui_data->feed_directory_dialog=NULL;
}

/** Appends a node to the category list
  *
  * @param title the title of the feed or category to add
  * @param url the url of the feed or NULL for a category
  */
static void ui_feed_directory_append_node_to_category_list(gchar * title,
							   gchar * url)
{
    fdNodePtr node = NULL;

    node = g_new0(struct fd_node_ptr, 1);

    node->title = title;
    node->url = url;
    node->selected = FALSE;
    node->checkbutton = NULL;

    category_list = g_slist_prepend(category_list, node);
}

/** Process outline tags of type rss and collapse the hieararchical structure
  * into one level so that the category list can be built into a single combobox
  *
  * @param cur the current node
  */
static void ui_feed_directory_process_outline_tags(xmlNodePtr cur)
{
    gchar *title = NULL;
    gchar *url = NULL;
    xmlNodePtr child = NULL;
    static int depth = 0;

    /* Do not iterate the opml tree infinitely */
    if (depth >= FEED_DIRECTORY_MAX_DEPTH)
        return;

    depth++;

    g_assert(cur != NULL);

    /* process all <outline> tags */
    child = cur->xmlChildrenNode;
    while (child != NULL) {
        /* If the outline is of type rss, save the necessary data */
        if (!xmlStrcmp(child->name, BAD_CAST "outline")) {
            if (!child->xmlChildrenNode || depth < 2) {
                if (!(title = utf8_fix(xmlGetProp(child, BAD_CAST "text"))))
                    title = utf8_fix(xmlGetProp(child, BAD_CAST "title"));

                url = utf8_fix(xmlGetProp(child, BAD_CAST "xmlUrl"));

                if (title)
                    ui_feed_directory_append_node_to_category_list(title, url);
            }

            if (child->xmlChildrenNode)
                ui_feed_directory_process_outline_tags(child);
        }

        child = child->next;
    }
    depth--;
}

/** Parses the feed directory contents provided in OPML format
  *
  * @param doc the document to parse
  * @param cur the current node
  */
static xmlDocPtr ui_feed_directory_opml_parse(xmlDocPtr doc, xmlNodePtr cur)
{
    g_assert(doc != NULL);
    g_assert(cur != NULL);

    ULOG_DEBUG("ui_feed_directory_opml_parse: Parsing opml document");
        
    if (xmlStrcmp(cur->name, BAD_CAST "opml") &&
	xmlStrcmp(cur->name, BAD_CAST "oml") &&
	xmlStrcmp(cur->name, BAD_CAST "outlineDocument")) {
	ULOG_DEBUG
	    ("Could not find OPML header. Unable to parse feed directory.");
	xmlFreeDoc(doc);
		doc=NULL;
    }

    cur = cur->xmlChildrenNode;

    while (cur && xmlIsBlankNode(cur)) {
	cur = cur->next;
    }

    while (cur != NULL) {
	if (!xmlStrcmp(cur->name, BAD_CAST "body"))
	    ui_feed_directory_process_outline_tags(cur);
	cur = cur->next;
    }
	return doc;
}

/** Start parsing the feed directory contents
  *
  * @param data the feed directory contents
  * @param length the length of the contents
  */
static void ui_feed_directory_parse(gchar * data, size_t length)
{
    gchar *errors;
    xmlDocPtr doc = NULL;
    xmlNodePtr cur = NULL;
    
    ULOG_DEBUG("ui_feed_directory_parse");
    
    /* try to parse buffer with XML and to create a DOM tree */
    if (NULL == (doc = parseBuffer(data, length, &errors))) {
	ULOG_DEBUG("XML error occurred. Unable to parse feed directory. ");
	return;
    }
    if (NULL == (cur = xmlDocGetRootElement(doc))) {
	ULOG_DEBUG("Empty document. Unable to parse feed directory. ");

	if (doc != NULL)
	    xmlFreeDoc(doc);
	return;
    }
    while (cur && xmlIsBlankNode(cur)) {
	cur = cur->next;
    }
    if (NULL == cur->name) {
	ULOG_DEBUG("Invalid XML. Unable to parse feed directory. ");

	if (doc != NULL)
	    xmlFreeDoc(doc);
	return;
    }

    doc=ui_feed_directory_opml_parse(doc, cur);
    if (doc != NULL)
	{
	xmlFreeDoc(doc);
	}
}

/** Creates the feed directory dialog with its widgets
  * 
  * @return the dialog
  */
//GtkWidget *ui_feed_directory_create_dialog(struct request *request)
GtkWidget *ui_feed_directory_create_dialog(folderPtr folder)
{
    GtkWidget *dialog_vbox;
    //GtkWidget *viewport;
    GtkWidget *subscribe_button;
    GtkWidget *close_button;
        
    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);

    /*to pass the pointer to on_dialog_response, create this folderPtr */
    //folderPtr folder = request->folder;
  
    /* ****************** General dialog setup ****************** */

    fd_dialog = gtk_dialog_new();

    ossohelp_dialog_help_enable(
            GTK_DIALOG(fd_dialog),
            RSS_NEWS_READER_HELP_DIRECTORY,
            app_data->osso);

    gtk_window_set_title(GTK_WINDOW(fd_dialog),
			 _("rss_ti_feed_directory"));
//     gtk_window_set_position(GTK_WINDOW(fd_dialog), GTK_WIN_POS_CENTER);
    gtk_window_set_default_size(GTK_WINDOW(fd_dialog), FEED_DIRECTORY_DIALOG_X_SIZE, FEED_DIRECTORY_DIALOG_Y_SIZE);
    gtk_window_set_type_hint(GTK_WINDOW(fd_dialog),
			     GDK_WINDOW_TYPE_HINT_DIALOG);
    gtk_window_set_transient_for(GTK_WINDOW(fd_dialog),
				 GTK_WINDOW(app_data->app_ui_data->app));
    gtk_window_set_modal(GTK_WINDOW(fd_dialog), TRUE);
    gtk_dialog_set_has_separator(GTK_DIALOG(fd_dialog), FALSE);
    
    dialog_vbox = GTK_DIALOG(fd_dialog)->vbox;
    gtk_widget_show(dialog_vbox);

    dialog_scrolled = gtk_scrolled_window_new(NULL, NULL);
    gtk_widget_show(dialog_scrolled);
    gtk_container_set_border_width(GTK_CONTAINER(dialog_scrolled),
				   HILDON_MARGIN_DEFAULT);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(dialog_scrolled),
				   GTK_POLICY_AUTOMATIC,
				   GTK_POLICY_AUTOMATIC);
    gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW
					(dialog_scrolled),
					GTK_SHADOW_NONE);
    /*gtk_box_pack_start_defaults (GTK_BOX( dialog_vbox ), dialog_scrolled); */
    /*
    viewport =
	gtk_viewport_new(gtk_scrolled_window_get_hadjustment
			 (GTK_SCROLLED_WINDOW(dialog_scrolled)),
			 gtk_scrolled_window_get_vadjustment
			 (GTK_SCROLLED_WINDOW(dialog_scrolled)));

    gtk_viewport_set_shadow_type(GTK_VIEWPORT(viewport), GTK_SHADOW_NONE);
    gtk_widget_show(viewport);
    
    gtk_container_add(GTK_CONTAINER(dialog_scrolled), viewport);
    */
    dialog_vbox2 = gtk_hbox_new(FALSE, 0);
    gtk_widget_show(dialog_vbox2);

    gtk_box_pack_start(GTK_BOX(dialog_vbox), dialog_vbox2, FALSE, FALSE, 0);
    gtk_box_pack_end(GTK_BOX(dialog_vbox), dialog_scrolled, TRUE, TRUE, 0);
//    gtk_box_pack_end_defaults(GTK_BOX(dialog_vbox), dialog_scrolled);

    /* ****************** Setup dialog widgets ****************** */

    combobox = gtk_combo_box_new_text();
//    gtk_box_pack_end(GTK_BOX(dialog_vbox2), combobox, TRUE, TRUE, 10);

    gtk_widget_show(combobox);
    label =
	hildon_caption_new(NULL, _("rss_ia_category"), combobox, NULL,
			   HILDON_CAPTION_OPTIONAL);
    gtk_widget_show(label);

    gtk_box_pack_start(GTK_BOX(dialog_vbox2), label, TRUE, TRUE, 10);

    dialog_vbox3 = gtk_vbox_new(FALSE, 0);
//    gtk_container_add(GTK_CONTAINER(viewport), dialog_vbox3);
    gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(dialog_scrolled), dialog_vbox3);

    gtk_widget_show(dialog_vbox3);


    /* ****************** Setup dialog buttons ****************** */

    subscribe_button = gtk_button_new_with_label(_("rss_bd_subscribe"));
    close_button = gtk_button_new_with_label(_("rss_bd_close"));

    gtk_widget_show(subscribe_button);
    gtk_widget_show(close_button);

    gtk_dialog_add_action_widget(GTK_DIALOG(fd_dialog), subscribe_button,
				 GTK_RESPONSE_OK);
    gtk_dialog_add_action_widget(GTK_DIALOG(fd_dialog), close_button,
				 GTK_RESPONSE_CANCEL);

    GTK_WIDGET_SET_FLAGS(subscribe_button, GTK_CAN_DEFAULT);

    g_signal_connect(G_OBJECT(fd_dialog), "response",
		     G_CALLBACK(ui_feed_directory_dialog_response), (gpointer) folder);

    g_signal_connect(G_OBJECT(combobox), "changed",
		     G_CALLBACK(ui_feed_directory_combobox_changed), NULL);

    /*g_signal_connect ( G_OBJECT ( dialog_vbox3 ),  "event" ,
       G_CALLBACK ( ui_feed_directory_event_catcher    ), NULL ); */

    return fd_dialog;
}

/** Checks the status of downloading the contents of the feed directory
  * and starts parsing it on success
  * @param request the download request
  */
void ui_feed_directory_process(struct request *request)
{
    GtkWidget *fd_dialog;
    
    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);
    
    if(!app_data->app_ui_data->feed_directory_loading)
        return;
     
    app_data->app_ui_data->feed_directory_loading = FALSE;

/*    if(app_data->app_ui_data->search_mode == SFM_REFRESH)
       gtk_progress_bar_set_text(GTK_PROGRESS_BAR(app_data->app_ui_data->progress_bar),
				 _("rss_ia_refreshing"));	*/
	
    g_assert(NULL != request);
    
    ui_lock();

//    hide_progressbar_if_needed();

    ULOG_DEBUG("Download completed");
    if (401 == request->httpstatus) {
        gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->app), _("rss_ni_errors_loading_directory"));
        ULOG_DEBUG("Download status: 401");
    } else if (410 == request->httpstatus) {
        gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->app), _("rss_ni_errors_loading_directory"));
        ULOG_DEBUG("Download status: 410");
    } else if (304 == request->httpstatus) {
        ULOG_DEBUG("Download status: 304");
    } else if (NULL != request->data) {
        ULOG_DEBUG("Download status: OK");
    	/* This should probably be taken into consideration */
	   /* note this is to update the feed URL on permanent redirects
	      if(0 != strcmp(request->source, feed_get_source(fp))) {
	      feed_set_source(fp, request->source);
	      } */
    
	   /* parse the new downloaded feed into new_fp */
	   /*fhp = feed_parse(fp, request->data, request->size, request->flags & FEED_REQ_AUTO_DISCOVER); */
        /*tvh: 2006May31 */
       app_data->app_ui_data->feed_directory_dialog=fd_dialog = ui_feed_directory_create_dialog(request->folder);
	   
	   category_list = NULL;
	   ui_feed_directory_parse(request->data, request->size);
	   category_list = g_slist_reverse(category_list);
	   if (ui_feed_directory_fill_categories())
	   {
	   	   gtk_widget_show(GTK_WIDGET(fd_dialog));
              //now set the feed directory 
              ULOG_DEBUG("XXXXXXXXXXXXXXXXXXXXXXXXXXXXXX: %s", fd_source);
              setStringConfValue(FEED_DIRECTORY_SOURCE,fd_source);
	   }
	   else
	   {
		   gtk_widget_destroy(GTK_WIDGET(fd_dialog));
		   ui_feed_directory_free();
		   gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->app), _("rss_ni_errors_while_parsing_feed"));
	   }
    }
    else
    {
       gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->app), _("rss_ni_errors_loading_directory"));
	   ULOG_DEBUG("Feed directory not available. HTTP status: %d", request->httpstatus);
    }
    ui_unlock();
}

/************************************************************************/
/*                        PUBLIC FUNCTIONS                              */
/************************************************************************/

/** Checks if the type of the given outline is rss
  *
  * @param cur the current node
  * @return TRUE if the given outline is of type rss, otherwise FALSE
  */
gboolean ui_feed_directory_outline_type_is_rss(xmlNodePtr cur)
{
    gchar *value = NULL;

    if (NULL != (value = utf8_fix(xmlGetProp(cur, BAD_CAST "type"))))
	if (!strcmp(value, "rss")) {
	    g_free(value);
	    return TRUE;
	}

    if (value)
	g_free(value);

    return FALSE;
}

/* 2006June05: not used
void load_factory_feed_directory_source(void)
{
    ///usr/share/osso_rss_feed_reader/opml/feed_directory.txt
    //contains the feed directory from factory
    gchar *filename =
	g_strdup_printf("%s" G_DIR_SEPARATOR_S ".osso_rss_feed_reader"
			G_DIR_SEPARATOR_S "feed_directory.txt",
			g_get_home_dir());

    FILE *cache = fopen(filename, "rb");
    
    feed_directory_source_value = cache_read_str(cache);
    ULOG_DEBUG(">>>>>>>>>>>>> FEED_DIRECTORY_SOURCE_VALUE = %s", feed_directory_source_value );
    fclose(cache);
    g_free(filename);
}
*/
void on_menuitem_feed_directory_clicked(GtkWidget * menuitem,
                    gpointer user_data){

    folderPtr ptr = NULL;

    /* While searching, no feed should be added */
    if (!GTK_WIDGET_SENSITIVE(app_data->app_ui_data->menuitem_feed_directory)) {
    
        if (app_data->app_ui_data->search_mode == SFM_SEARCH)
            gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->app),
                 _("rss_ib_unable_add_feed_searching"));        
        else if (app_data->app_ui_data->search_mode != SFM_NORMAL)
            gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->app),
                 _("rss_ib_unable_add_feed"));        
        else 
            ULOG_DEBUG("oooooooooooooooooooooooooERRORooooooooooooooooooooooooooooo");

        return;    
    }

    /* 28-06-2006 Added low mem check */
    if (app_data->app_ui_data->low_mem)
    {
        gtk_infoprint( GTK_WINDOW(app_data->app_ui_data->app), 
                    dgettext("ke-recv", "memr_ib_operation_disabled") );
        return;                    
    }

    /* This value is either the default or the one set by the last
    opml_import action */
    gchar *feed_dir_source = getStringConfValue(FEED_DIRECTORY_SOURCE);
    //If it's never been touched before
    //if( 0 == strcmp(feed_dir_source, "") )
    if (strlen(feed_dir_source) == 0 )
    {
        /* spec changed so that there should be no default feed directory.
        * Due to the permission from feed server
        */
        //Menu item should be dimmed now
        if (GTK_WIDGET_SENSITIVE(menuitem)) {
            ULOG_DEBUG("Why is feed directory menu items sensitive?? while there's nothing in gconf?");
            ULOG_DEBUG("xooooooooooooooooooooooooooooooooooERRORooooooooooooooooooooooooooooooox");
        }


        g_free(feed_dir_source);
        return;

        /* default feed not used. So comment this out */
        /*
        setStringConfValue(FEED_DIRECTORY_SOURCE, FEED_DIRECTORY_SOURCE_VALUE);
        if (feed_directory_source_value != NULL)
            g_free(feed_directory_source_value);
        feed_dir_source = getStringConfValue(FEED_DIRECTORY_SOURCE);
        */
    }

    //now continue with the feed directory once we've fot the feed dir source
    ptr = ui_feedlist_get_target_folder();
    ui_feed_directory_button_clicked(NULL, NULL, feed_dir_source, ptr);
    g_free(feed_dir_source);
}
void ui_feed_directory_button_clicked(GtkWidget * widget,
                      gpointer user_data, gchar *fdsource, folderPtr folder)
{
    AppUIData *app_ui_data;
    
    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);

    ULOG_DEBUG("ui_feed_directory_button_clicked: with source = %s", fdsource);

    if (fdsource)
        fd_source = g_strdup(fdsource) ;
    else //also needs to reset fd_source
    {
        if (fd_source)     
            g_free(fd_source);
        fd_source = NULL;
    }
    
    //shouldn't call this
    //folder_free(fd_folderPtr);

    if (folder)
        fd_folderPtr = folder;
    else
        fd_folderPtr = NULL;

    if(app_data->app_ui_data->feed_directory_loading)
        return;
    
    app_ui_data = app_data->app_ui_data;
    
    app_ui_data->iap_action = OSSO_IAP_REFRESH_FEED_DIRECTORY;
/*    osso_iap_connect(OSSO_IAP_ANY, OSSO_IAP_REQUESTED_CONNECT,
		     app_ui_data);*/
    request_iap_connect();

    /* now set the gconf value to the latest opml file imported
    the use of this is: When clicking on menu->tools->feed directories...
    It will invoke the latest opml feed added. If user wants to have a fresh
    feed dir, he/she should use Import OPML menu item
    */
    /* Shouldn't set here. Until the feed directory is loaded successfully.
     * Then set it in ui_feed_directory_process(), once the feed directory dialog popped up
     */
    //setStringConfValue(FEED_DIRECTORY_SOURCE,fd_source);
}


void ui_feed_directory_get()
{
    struct request *request = NULL;
    AppUIData *app_ui_data;
    //gchar *source = NULL;
    
    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);
    g_assert(app_data->app_ui_data->search_button != NULL);
    g_assert(app_data->app_ui_data->button_image_stop != NULL);

    if (!fd_source)    
        fd_source = getStringConfValue(FEED_DIRECTORY_SOURCE);
    
    //if(!strcmp(fd_source, ""))
    if (strlen(fd_source) == 0)
    {
        g_free(fd_source);
        gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->app), _("rss_ni_errors_loading_directory"));
        ULOG_DEBUG("ui_feed_directory_get: errors loading the directory");
	return;
    }

    app_ui_data = app_data->app_ui_data;
    
    app_ui_data->feed_directory_loading = TRUE;
    
/*    if(app_data->app_ui_data->search_mode != SFM_REFRESH)
    {
	gtk_tool_button_set_icon_widget(GTK_TOOL_BUTTON
					(app_ui_data->search_button),
					app_ui_data->button_image_stop);
    ULOG_DEBUG("\n\n\n\n\n\n\nui_feed_directory_get: setting 0 progressbar \n\n\n\n\n");			
	gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR
					(app_ui_data->progress_bar), 0);
					
	gtk_container_remove(GTK_CONTAINER(app_ui_data->var_field),
				app_ui_data->search_field);
	gtk_container_add(GTK_CONTAINER(app_ui_data->var_field),
			app_ui_data->progress_bar);

    //always show toolbar
    gtk_widget_show (GTK_WIDGET(app_ui_data->main_toolbar));
    
	app_data->app_ui_data->search_mode = SFM_REFRESH;
	rss_dbus_signal_applet(OSSO_RSS_FEED_READER_REFRESH_STARTED);
    }
*/
    /*
    gtk_progress_bar_set_text(GTK_PROGRESS_BAR(app_ui_data->progress_bar),
				_("rss_ia_loading"));
    */
    
/*    gtk_progress_bar_set_text(GTK_PROGRESS_BAR(app_ui_data->progress_bar),
				_("rss_pb_toolbar_receiving_file"));*/
                
    switch_progressbar(PROGRESSBAR_RECEIVING_MODE);

    ULOG_DEBUG("ui_feed_directory_get: Creating a download request");
    
    request = download_request_new();

    request->callback = ui_feed_directory_process;
    request->source = fd_source;
    request->user_data = NULL;

    /* Add a parent folder as well */
    if (fd_folderPtr != NULL)
        request->folder = fd_folderPtr;
    else
        request->folder = NULL;

    request->feed = NULL;
    request->lastmodified = NULL;
    request->etag = NULL;
    request->flags = 0;
    request->priority = 1;

    ULOG_DEBUG("ui_feed_directory_get: Putting request to download queue");
    download_queue(request);
}
/*tvh: 2006May31: This is to create a feed directory when adding an opml file from
 * a add-feed dialog. In stead of the old confirmation dialog: "Add all xx feeds"
 * This feed directory is created.
 * Note : used in the feedParser func of the opml feed handler pointer : opml_parse()
 * in opml.c
 * couldn't think of a better name for the func :(
 */
void ui_feed_directory_from_add_feed_dialog( xmlDocPtr doc,
                         xmlNodePtr cur, folderPtr folder,
                         gchar *fd_source)
{
    ULOG_DEBUG("creating the feed directory dialog now, originally from a add feed dialog action");
    app_data->app_ui_data->feed_directory_dialog=fd_dialog = ui_feed_directory_create_dialog(folder);
       
    category_list = NULL;
    //ui_feed_directory_parse((gchar * )doc, strlen((gchar* )doc) );
    ui_feed_directory_opml_parse(doc, cur);
    category_list = g_slist_reverse(category_list);
    if (ui_feed_directory_fill_categories())
    {
          setStringConfValue(FEED_DIRECTORY_SOURCE, fd_source);
          ULOG_DEBUG("XXXXXXXXXXXXXXXXXXX: %s", fd_source);
          gtk_widget_show(GTK_WIDGET(fd_dialog));
    }
    else
    {
        gtk_widget_destroy(GTK_WIDGET(fd_dialog));
        ui_feed_directory_free();
       gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->app), _("rss_ni_errors_while_parsing_feed"));
   }
}
