/**
 * @file item.c common item handling
 *
 * Copyright (C) 2003, 2004 Lars Lindner <lars.lindner@gmx.net>
 * Copyright (C) 2004 Nathan J. Conrad <t98502@users.sourceforge.net>
 *	      
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <glib.h>
#include <time.h>
#include <string.h>		/* For memset() */
#include <stdlib.h>

#include "vfolder.h"
#include "item.h"
#include "support.h"
#include "common.h"
#include "htmlview.h"

#include "callbacks.h"
#include "metadata.h"
#include "debug.h"
#include "cache.h"
#include "html.h"

extern AppData *app_data;

/* function to create a new feed structure */
itemPtr item_new(void)
{
    itemPtr ip = NULL;

    ip = g_new0(struct item, 1);
    ip->newStatus = TRUE;

    return ip;
}

/************************************************************************/
/*                        PUBLIC FUNCTIONS                              */
/************************************************************************/

void item_copy(itemPtr from, itemPtr to)
{
    g_assert(from != NULL);

    item_set_title(to, from->title);
    item_set_source(to, from->source);
    item_set_real_source_url(to, from->real_source_url);
    item_set_real_source_title(to, from->real_source_title);
    item_set_description(to, from->description);
    item_set_id(to, from->id);

    g_assert(to != NULL);
    
    to->readStatus = from->readStatus;
    to->newStatus = FALSE;
    to->marked = from->marked;
    to->time = from->time;
    to->nr = from->nr;

    /* the following line enables state propagation in item.c */
    to->sourceFeed = from->fp;

    /* this copies metadata */
    metadata_list_free(to->metadata);
    //to->metadata = NULL;
    to->metadata = metadata_list_copy(from->metadata, NULL);
}

inline void item_set_title(itemPtr ip, const gchar * title)
{
    g_assert(ip != NULL);
    if (NULL != ip->title)
        g_free(ip->title);
    ip->title = g_strdup(title);
}

inline void item_set_description(itemPtr ip, const gchar * description)
{
    g_assert(ip != NULL);
    if (NULL!= ip->description)
        g_free(ip->description);
    ip->description = g_strdup(description);	
}


/*inline void item_increase_delete_count(itemPtr ip)
{
    g_assert(ip != NULL);
    ip->delete_count++;
}*/

inline void item_set_source(itemPtr ip, const gchar * source)
{
    g_assert(ip != NULL);
    if (ip->source != NULL)
        g_free(ip->source);
    ip->source = g_strdup(source);
}
inline void item_set_real_source_url(itemPtr ip, const gchar * source)
{
    g_assert(ip != NULL);
    if (NULL!= ip->real_source_url)
        g_free(ip->real_source_url);
    ip->real_source_url = g_strdup(source);
}
inline void item_set_real_source_title(itemPtr ip, const gchar * source)
{
    g_assert(ip != NULL);
    if (NULL != ip->real_source_title)
        g_free(ip->real_source_title);
    ip->real_source_title = g_strdup(source);
}
inline void item_set_time(itemPtr ip, const time_t t)
{
    g_assert(ip != NULL);
    ip->time = t;
}
inline void item_set_read_status(itemPtr ip, const gboolean readStatus)
{
    g_assert(ip != NULL);
    ip->readStatus = readStatus;
}

inline void item_set_id(itemPtr ip, const gchar * id)
{
    /*NOTE: should remove white spaces? */
    g_assert(ip != NULL);
    if (NULL!= ip->id)
        g_free(ip->id);
    ip->id = g_strdup(id);
}

inline void item_set_hidden(itemPtr ip, const gboolean hidden)
{
    g_assert(ip != NULL);
    ip->hidden = hidden;
}

inline const gchar *item_get_id(itemPtr ip)
{
    return (ip != NULL ? ip->id : NULL);
}
inline const gchar *item_get_title(itemPtr ip)
{
    return (ip != NULL ? ip->title : NULL);
}
inline const gchar *item_get_description(itemPtr ip)
{
    return (ip != NULL ? ip->description : NULL);
}
inline const gchar *item_get_source(itemPtr ip)
{
    return (ip != NULL ? ip->source : NULL);
}
inline const gchar *item_get_real_source_url(itemPtr ip)
{
    return (ip != NULL ? ip->real_source_url : NULL);
}
inline const gchar *item_get_real_source_title(itemPtr ip)
{
    return (ip != NULL ? ip->real_source_title : NULL);
}
inline const time_t item_get_time(itemPtr ip)
{
    return (ip != NULL ? ip->time : 0);
}
inline const gboolean item_get_read_status(itemPtr ip)
{
    return (ip != NULL ? ip->readStatus : FALSE);
}

/*inline const gboolean item_get_delete_count(itemPtr ip)
{
    g_assert(ip != NULL);
    return ip->delete_count;
}*/


inline const gboolean item_get_mark(itemPtr ip)
{
    g_assert(ip != NULL);
    return ip->marked;
}
inline const gboolean item_get_hidden(itemPtr ip)
{
    g_assert(ip != NULL);
    return ip->hidden;
}
inline const gboolean item_get_new_status(itemPtr ip)
{
    g_assert(ip != NULL);
    return ip->newStatus;
}

void item_set_mark(itemPtr ip, gboolean flag)
{
    itemPtr sourceItem = NULL;

    ip->marked = flag;
    /*
    if (ip->ui_data != NULL)
	   ui_update_item(ip);
    */
    if (ip->fp != NULL)
	ip->fp->needsCacheSave = TRUE;

    /* if this item belongs to a vfolder update the source feed */
    if (ip->sourceFeed != NULL) {
	feed_load(ip->sourceFeed);
	sourceItem = feed_lookup_item_by_nr(ip->sourceFeed, ip->nr);
	item_set_mark(sourceItem, flag);
	feed_unload(ip->sourceFeed);
    } else {
	vfolder_update_item(ip);	/* there might be vfolders using this item */
	vfolder_check_item(ip);	/* and check if now a rule matches */
    }
}

void item_set_new_status(itemPtr ip, const gboolean newStatus)
{
    if (newStatus != ip->newStatus) {
	ip->newStatus = newStatus;

	if (TRUE == newStatus)
	    feed_increase_new_counter((feedPtr) (ip->fp));
	else
	    feed_decrease_new_counter((feedPtr) (ip->fp));
    }
}

void item_set_unread(itemPtr ip)
{
    itemPtr sourceItem = NULL;

    if (TRUE == ip->readStatus) {
	feed_increase_unread_counter((feedPtr) (ip->fp));

	ip->readStatus = FALSE;
    /*
	if (ip->ui_data != NULL)
	    ui_update_item(ip);
    */
	if (ip->fp != NULL)
	    ip->fp->needsCacheSave = TRUE;

	/* if this item belongs to a vfolder update the source feed */
	if (ip->sourceFeed != NULL) {
	    feed_load(ip->sourceFeed);
	    sourceItem = feed_lookup_item_by_nr(ip->sourceFeed, ip->nr);
	    item_set_unread(sourceItem);
	    feed_unload(ip->sourceFeed);
	} else {
	    vfolder_update_item(ip);	/* there might be vfolders using this item */
	    vfolder_check_item(ip);	/* and check if now a rule matches */
	}
    }
}

void item_set_read(itemPtr ip)
{
    itemPtr sourceItem = NULL;

    if (FALSE == ip->readStatus) {
	feed_decrease_unread_counter((feedPtr) (ip->fp));

	ip->readStatus = TRUE;
    /*
	if (ip->ui_data)
	    ui_update_item(ip);
    */
	if (ip->fp != NULL)
	    ip->fp->needsCacheSave = TRUE;

	/* if this item belongs to a vfolder update the source feed */
	if (ip->sourceFeed != NULL) {
	    feed_load(ip->sourceFeed);
	    sourceItem = feed_lookup_item_by_nr(ip->sourceFeed, ip->nr);
	    item_set_read(sourceItem);
	    feed_unload(ip->sourceFeed);
	} else {
	    vfolder_update_item(ip);	/* there might be vfolders using this item */
	    vfolder_check_item(ip);	/* and check if now a rule matches */
	}
    }
}

void item_free(itemPtr ip)
{

    if (FALSE == ip->readStatus)
	feed_decrease_unread_counter(ip->fp);
    if (TRUE == ip->newStatus)
	feed_decrease_new_counter(ip->fp);

    g_free(ip->title);
    g_free(ip->source);
    g_free(ip->real_source_url);
    g_free(ip->real_source_title);
    g_free(ip->description);
    g_free(ip->id);
    //gtk_widget_destroy(GTK_WIDGET (ip->titleLabel));
    g_assert(NULL == ip->tmpdata);	/* should be free after rendering */
    metadata_list_free(ip->metadata);
    	
    //only assert when ip is not in vfolder
    if (ip->sourceFeed == NULL)
        g_assert(ip->ui_data == NULL);
    
    g_free(ip);
}

void item_free_ignore_metadata(itemPtr ip)
{
    g_free(ip->title);
    g_free(ip->source);
    g_free(ip->real_source_url);
    g_free(ip->real_source_title);
    g_free(ip->description);
    g_free(ip->id);
    g_assert(NULL == ip->tmpdata);	/* should be free after rendering */
   	
    g_assert(ip->ui_data == NULL);
    
    g_free(ip);
}

gchar *item_render(itemPtr ip)
{
    struct displayset displayset;
    gchar *buffer = NULL;
    gchar *tmp = NULL, *tmp2 = NULL;
    gboolean migration = FALSE;

    displayset.headtable = NULL;
    displayset.head = NULL;
    displayset.body = g_strdup(item_get_description(ip));
    displayset.foot = NULL;
    displayset.foottable = NULL;

    /* I hope this is unique enough... */
    if ((NULL != displayset.body) && (NULL != strstr(displayset.body, "class=\"itemhead\"")))
	migration = TRUE;

    if (FALSE == migration) {
	metadata_list_render(ip->metadata, &displayset);
	/* Head table */
	addToHTMLBufferFast(&buffer, HEAD_START);

	/*  -- Item line */
	if (item_get_source(ip) != NULL)
	    tmp =
		g_strdup_printf("%s",
				(item_get_title(ip) !=
				 NULL) ? item_get_title(ip) : "-");
	else
	    tmp =
		g_strdup((item_get_title(ip) !=
			  NULL) ? item_get_title(ip) : "-");

	if (item_get_mark(ip)) {
	    tmp2 = g_strdup_printf(HEAD_LINE_MARKED, tmp);
	} else {
	    tmp2 = g_strdup_printf(HEAD_LINE, tmp);
	}
	addToHTMLBufferFast(&buffer, tmp2);
	g_free(tmp);
	g_free(tmp2);

	/*  -- real source line */
	tmp = NULL;
	if (item_get_real_source_url(ip) != NULL)
	    tmp = g_strdup("");
	else if (item_get_real_source_title(ip) != NULL)
	    tmp = g_strdup(item_get_real_source_title(ip));

	if (NULL != tmp) {
	    tmp2 = g_strdup_printf(HEAD_LINE, tmp);
	    g_free(tmp);
	    addToHTMLBufferFast(&buffer, tmp2);
	    g_free(tmp2);
	}

	g_free(displayset.headtable);
	addToHTMLBufferFast(&buffer, HEAD_END);

	/* Head */
	if (displayset.head != NULL) {
	    addToHTMLBufferFast(&buffer, displayset.head);
	    g_free(displayset.head);
	}
    }

    if (displayset.body != NULL) {
	remove_newlines_and_extra_spaces(displayset.body);
	addToHTMLBufferFast(&buffer, displayset.body);
	g_free(displayset.body);
    }


    tmp = g_strdup_printf(ITEM_ACTIONS, ip->nr);
    addToHTMLBufferFast(&buffer, tmp);
    g_free(tmp);

    if (FALSE == migration) {
	if (displayset.foot != NULL) {
	    g_free(displayset.foot);
	}

	if (displayset.foottable != NULL) {
	    g_free(displayset.foottable);
	}
    }

    addToHTMLBufferFast(&buffer, ITEM_END);

    return buffer;
}

itemPtr item_parse_cache(xmlDocPtr doc, xmlNodePtr cur)
{
    itemPtr ip = NULL;
    gchar *tmp = NULL;

    g_assert(NULL != doc);
    g_assert(NULL != cur);

    ip = item_new();
    ip->newStatus = FALSE;

    cur = cur->xmlChildrenNode;
    while (cur != NULL) {

	if (cur->type != XML_ELEMENT_NODE ||
	    NULL == (tmp =
		     utf8_fix(xmlNodeListGetString
			      (doc, cur->xmlChildrenNode, 1)))) {
	    cur = cur->next;
	    continue;
	}

	if (!xmlStrcmp(cur->name, BAD_CAST "title"))
	    item_set_title(ip, tmp);

	else if (!xmlStrcmp(cur->name, BAD_CAST "description"))
	    item_set_description(ip, tmp);

	else if (!xmlStrcmp(cur->name, BAD_CAST "source"))
	    item_set_source(ip, tmp);

	else if (!xmlStrcmp(cur->name, BAD_CAST "real_source_url"))
	    item_set_real_source_url(ip, tmp);

	else if (!xmlStrcmp(cur->name, BAD_CAST "real_source_title"))
	    item_set_real_source_title(ip, tmp);

	else if (!xmlStrcmp(cur->name, BAD_CAST "id"))
	    item_set_id(ip, tmp);

	else if (!xmlStrcmp(cur->name, BAD_CAST "nr"))
	    ip->nr = atol(tmp);

	else if (!xmlStrcmp(cur->name, BAD_CAST "readStatus"))
	    item_set_read_status(ip, (0 == atoi(tmp)) ? FALSE : TRUE);

	else if (!xmlStrcmp(cur->name, BAD_CAST "mark"))
	    /* we don't call item_set_mark here because it would
	     * update the UI */
	    ip->marked = (1 == atoi(tmp)) ? TRUE : FALSE;

	else if (!xmlStrcmp(cur->name, BAD_CAST "time"))
	    item_set_time(ip, atol(tmp));

	else if (!xmlStrcmp(cur->name, BAD_CAST "attributes"))
	    ip->metadata = metadata_parse_xml_nodes(doc, cur);

	g_free(tmp);
	cur = cur->next;
    }

    return ip;
}

void item_save(itemPtr ip, xmlNodePtr feedNode)
{
    xmlNodePtr itemNode = NULL;
    gchar *tmp = NULL;

    if (NULL != (itemNode = xmlNewChild(feedNode, NULL, "item", NULL))) {

	/* should never happen... */
	if (NULL == item_get_title(ip))
	    item_set_title(ip, "");
	xmlNewTextChild(itemNode, NULL, "title", item_get_title(ip));

	if (NULL != item_get_description(ip))
	    xmlNewTextChild(itemNode, NULL, "description",
			    item_get_description(ip));

	if (NULL != item_get_source(ip))
	    xmlNewTextChild(itemNode, NULL, "source", item_get_source(ip));

	if (NULL != item_get_real_source_title(ip))
	    xmlNewTextChild(itemNode, NULL, "real_source_title",
			    item_get_real_source_title(ip));

	if (NULL != item_get_real_source_url(ip))
	    xmlNewTextChild(itemNode, NULL, "real_source_url",
			    item_get_real_source_url(ip));

	if (NULL != item_get_id(ip))
	    xmlNewTextChild(itemNode, NULL, "id", item_get_id(ip));

	tmp = g_strdup_printf("%ld", ip->nr);
	xmlNewTextChild(itemNode, NULL, "nr", tmp);
	g_free(tmp);

	tmp =
	    g_strdup_printf("%d",
			    (TRUE == item_get_read_status(ip)) ? 1 : 0);
	xmlNewTextChild(itemNode, NULL, "readStatus", tmp);
	g_free(tmp);

	tmp = g_strdup_printf("%d", (TRUE == item_get_mark(ip)) ? 1 : 0);
	xmlNewTextChild(itemNode, NULL, "mark", tmp);
	g_free(tmp);

	tmp = g_strdup_printf("%ld", item_get_time(ip));
	xmlNewTextChild(itemNode, NULL, "time", tmp);
	g_free(tmp);

	metadata_add_xml_nodes(ip->metadata, itemNode);

    } else {
	g_warning("could not write XML item node!\n");
    }
}

void
remove_link_from_cache(gchar *link, gpointer user_data)
{
    file_cache_remove(FILE_CACHE(user_data), link);
//    ULOG_DEBUG("Removing link: <%s>", link);
}

void
remove_links_from_cache(itemPtr ip)
{
    find_image_links(item_get_description(ip), remove_link_from_cache, 
        (gpointer)app_data->app_ui_data->img_cache);
}

/* From liferea, and not used in News reader 
void item_display(itemPtr ip)
{
    gchar *buffer = NULL, *tmp = NULL;

    tmp = item_render(ip);
    addToHTMLBufferFast(&buffer, tmp);
    g_free(tmp);
    ui_htmlview_finish_output(&buffer);
    if (feed_get_source(ip->fp) != NULL &&
    ip->fp->source[0] != '|' &&
    strstr(feed_get_source(ip->fp), "://") != NULL)
    //ui_htmlview_write(ui_mainwindow_get_active_htmlview(), buffer,
    //          feed_get_source(ip->fp), NULL);//TODO: re-consider NULL, displayed_node should be used
                                                //if can't find any
    else
    ui_htmlview_write(ui_mainwindow_get_active_htmlview(), buffer,
              NULL, NULL);
    g_free(buffer);
}
*/
