/**
 * @file conf.c Liferea configuration (gconf access and feedlist import)
 *
 * Copyright (C) 2003, 2004 Lars Lindner <lars.lindner@gmx.net>
 * Copyright (C) 2004 Nathan J. Conrad <t98502@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gconf/gconf.h>
#include <gconf/gconf-client.h>
#include <libxml/uri.h>
#include <string.h>
#include <time.h>
#include <osso-ic-gconf.h>
#include <osso-log.h>

#include <settings.h>

#include "support.h"
#include "callbacks.h"
#include "update.h"
#include "feed.h"
#include "folder.h"
#include "common.h"
#include "conf.h"
#include "debug.h"
#include "htmlview.h"
#include "ui_mainwindow.h"
#include "cache.h"

#define MAX_GCONF_PATHLEN	256
#define HOMEPAGE	"http://liferea.sf.net/"

extern AppData *app_data;
extern time_t rss_updated;

static GConfClient *gconfclient = NULL; 

static guint feedlist_save_timer;
static guint feedlistLoading;

/* configuration strings for the SnowNews HTTP code used from within netio.c */
char *useragent = NULL;
char *proxyname = NULL;
char *proxyusername = NULL;
char *proxypassword = NULL;
int proxyport = 0;

/* Function prototypes */
static void conf_proxy_reset_settings_cb(GConfClient * gconfclient,
					 guint cnxn_id, GConfEntry * entry,
					 gpointer user_data);

static void conf_internet_connection_changed_cb(GConfClient * gconfclient,
					        guint cnxn_id, GConfEntry * entry,
					        gpointer user_data);

/************************************************************************/
/*                        PRIVATE FUNCTIONS                             */
/************************************************************************/					 

/** Checks for errors
  *
  * @param err an error structure
  * @return TRUE if an error has been found, ELSE otherwise
  */
static gboolean is_gconf_error(GError ** err)
{
    if (*err != NULL) {
        ULOG_ERR("%s\n",(*err)->message);
	g_error_free(*err);
	*err = NULL;
	return TRUE;
    }

    return FALSE;
}

/** Resets proxy settings
  * 
  * @param gconfclient the gconf client
  * @param cnxn_id not used
  * @param entry a gconf entry
  * @param user_data not used
  */
static void conf_proxy_reset_settings_cb(GConfClient * gconfclient,
					 guint cnxn_id, GConfEntry * entry,
					 gpointer user_data)
{
    gchar *tmp = NULL;
    xmlURIPtr uri = NULL;

    g_free(proxyname);
    proxyname = NULL;
    proxyport = 0;

    g_free(proxyusername);
    proxyusername = NULL;
    g_free(proxypassword);
    proxypassword = NULL;

    /* first check for a configured GNOME proxy */
    if (getBooleanConfValue(USE_PROXY)) {
	proxyname = getStringConfValue(PROXY_HOST);
	proxyport = getNumericConfValue(PROXY_PORT);
	debug2(DEBUG_CONF,
	       "using GNOME configured proxy: \"%s\" port \"%d\"",
	       proxyname, proxyport);
	if (getBooleanConfValue(PROXY_USEAUTH)) {
	    proxyusername = getStringConfValue(PROXY_USER);
	    proxypassword = getStringConfValue(PROXY_PASSWD);
	}
    } else {
	/* otherwise there could be a proxy specified in the environment 
	   the following code was derived from SnowNews' setup.c */
	if (g_getenv("http_proxy") != NULL) {
	    /* The pointer returned by getenv must not be altered.
	       What about mentioning this in the manpage of getenv? */
	    debug0(DEBUG_CONF, "using proxy from environment");
	    do {
		uri = xmlParseURI(BAD_CAST g_getenv("http_proxy"));
		if (uri == NULL)
		    break;
		if (uri->server == NULL) {
		    xmlFreeURI(uri);
		    break;
		}
		proxyname = g_strdup(uri->server);
		proxyport = (uri->port == 0) ? 3128 : uri->port;
		if (uri->user != NULL) {
		    tmp = strtok(uri->user, ":");
		    tmp = strtok(NULL, ":");
		    if (tmp != NULL) {
			proxyusername = g_strdup(uri->user);
			proxypassword = g_strdup(tmp);
		    }
		}
		xmlFreeURI(uri);
	    } while (FALSE);
	}
    }

    ui_htmlview_set_proxy(proxyname, proxyport, proxyusername,
			  proxypassword);
    debug4(DEBUG_CONF, "Proxy settings are now %s:%d %s:%s",
	   proxyname != NULL ? proxyname : "NULL", proxyport,
	   proxyusername != NULL ? proxyusername : "NULL",
	   proxypassword != NULL ? proxypassword : "NULL");
}

/** Listens for changes in the internet connection and checks if the News Reader
  * should be refreshed or not
  *
  * @param gconfclient not used
  * @param cnxn_id not used
  * @param entry not used
  * @param user_data not used
  */
static void conf_internet_connection_changed_cb(GConfClient * gconfclient,
					 guint cnxn_id, GConfEntry * entry,
					 gpointer user_data)
{
    ULOG_DEBUG("\n*******\n\ninternet connection changed\n\n*********\n");
    
    AppUIData *app_ui_data = NULL;
    
    if(app_data == NULL)
        return;

    if(app_data->app_ui_data == NULL)
        return;
        
    app_ui_data = app_data->app_ui_data;
        
    struct tm *tm;
    
    tm = localtime (&rss_updated);
    ULOG_DEBUG("Last refresh: %d:%d",tm->tm_hour,tm->tm_min);
    ULOG_DEBUG("Search mode: %d",app_ui_data->search_mode);
    ULOG_DEBUG("Automatic updates: %d",getBooleanConfValue(RSS_SETTINGS_AUTOMATIC_UPDATES));
    
}

static void configuration_changed_cb(GConfClient * gconfclient,
					 guint cnxn_id, GConfEntry * entry,
					 gpointer user_data)
{
    ULOG_DEBUG(">>> Configuration changed ! <<<");
    
    AppUIData *app_ui_data = NULL;
    
    if(!app_data || !app_data->app_ui_data)
        return;
        
    app_ui_data = app_data->app_ui_data;
        
    GConfValue* gval = gconf_entry_get_value(entry);
    if (gval)
    {
        size_t size = gconf_value_get_int(gval);
        ULOG_DEBUG("Resizing image cache to: %d mb", size);
        file_cache_resize(app_ui_data->img_cache, 
            all_cache_size[find_cache_size(size)]*1024*1024);
    }
}

/** Saves the feedlist into a file
  *
  * @param user_data not used
  * @return always FALSE
  */
static gboolean conf_feedlist_schedule_save_cb(gpointer user_data)
{
    feedlist_save_timer = 0;
    conf_feedlist_save();
    return FALSE;
}

/************************************************************************/
/*                        PUBLIC FUNCTIONS                              */
/************************************************************************/

/* called once on startup */
void conf_init()
{
    int ualength = 0;
    const char *lang = NULL;

    g_assert(app_data != NULL);

    /* have to be called for multithreaded programs */
    xmlInitParser();

    /* the following code was copied from SnowNews and adapted to build
       a Liferea user agent... */

    /* Construct the User-Agent string of Liferea. This is done here in program init,
       because we need to do it exactly once and it will never change while the program
       is running. */
    if (g_getenv("LANG") != NULL) {
	lang = g_getenv("LANG");
	/* e.g. Liferea/0.3.8 (Linux; de_DE; (http://liferea.sf.net/) */
	ualength =
	    strlen("Liferea/") + strlen(VERSION) + 2 + strlen(lang) + 2 +
	    strlen(OSNAME) + 2 + strlen(HOMEPAGE) + 2;
	useragent = g_malloc(ualength);
	snprintf(useragent, ualength, "Liferea/%s (%s; %s; %s)", VERSION,
		 OSNAME, lang, HOMEPAGE);
    } else {
	/* "Liferea/" + VERSION + "(" OS + "; " + HOMEPAGE + ")" */
	ualength =
	    strlen("Liferea/") + strlen(VERSION) + 2 + strlen(OSNAME) + 2 +
	    strlen("http://liferea.sf.net/") + 2;
	useragent = g_malloc(ualength);
	snprintf(useragent, ualength, "Liferea/%s (%s; %s)", VERSION,
		 OSNAME, HOMEPAGE);
    }

    /* initialize GConf gconfclient */
    gconfclient = gconf_client_get_default();
    config_check_first_time(gconfclient);

    app_data->gconf_client = gconfclient;
    gconf_client_add_dir(gconfclient, "/system/http_proxy",
			 GCONF_CLIENT_PRELOAD_NONE, NULL);

    gconf_client_notify_add(gconfclient, "/system/http_proxy",
			    conf_proxy_reset_settings_cb, NULL, NULL,
			    NULL);

    gconf_client_add_dir(gconfclient, ICD_GCONF_CURRENT,
			 GCONF_CLIENT_PRELOAD_NONE, NULL);			    
			    
    gconf_client_notify_add(gconfclient, ICD_GCONF_CURRENT,
			    conf_internet_connection_changed_cb, NULL, NULL,
			    NULL);

    gconf_client_add_dir(gconfclient, RSS_SETTINGS_IMAGE_CACHE_SIZE,
			 GCONF_CLIENT_PRELOAD_NONE, NULL);

    gconf_client_notify_add(gconfclient, RSS_SETTINGS_IMAGE_CACHE_SIZE,
			    configuration_changed_cb, NULL, NULL,
			    NULL);
                
    find_iap_connection_type();

    /* Load settings into static buffers */
    conf_proxy_reset_settings_cb(NULL, 0, NULL, NULL);
}

/*----------------------------------------------------------------------*/
/* feed entry handling							*/
/*----------------------------------------------------------------------*/

gchar *conf_new_id()
{
    int i = 0;
    gchar *id = NULL, *filename = NULL;
    gboolean already_used = FALSE;

    id = g_new0(gchar, 10);
    do {
	for (i = 0; i < 7; i++)
	    id[i] = (char) g_random_int_range('a', 'z');
	id[7] = '\0';

	filename =
	    common_create_cache_filename("cache" G_DIR_SEPARATOR_S "feeds",
					 id, NULL);
	already_used = g_file_test(filename, G_FILE_TEST_EXISTS);
	g_free(filename);
    } while (already_used);

    return id;
}

/*----------------------------------------------------------------------*/
/* config loading on startup						*/
/*----------------------------------------------------------------------*/

void conf_load_subscriptions(void)
{
    gchar *filename = NULL;
    gchar *opml_filename = NULL;

    folderPtr root_folder = NULL;

    feedlistLoading = TRUE;
    filename =
	g_strdup_printf("%s" G_DIR_SEPARATOR_S ".osso_rss_feed_reader"
			G_DIR_SEPARATOR_S "feedlist.opml",
			g_get_home_dir());
    root_folder = ui_feedlist_get_root_folder();

    if (!g_file_test(filename, G_FILE_TEST_EXISTS)) {
	opml_filename =
	    g_strdup_printf("%s" G_DIR_SEPARATOR_S "osso_rss_feed_reader"
			    G_DIR_SEPARATOR_S "opml" G_DIR_SEPARATOR_S
			    "feedlist.opml", PACKAGE_DATA_DIR);
	if (g_file_test(opml_filename, G_FILE_TEST_EXISTS)) {
	    g_free(filename);
	    filename = opml_filename;
	} else
	    g_free(opml_filename);
    }
    
    import_OPML_feedlist(filename, root_folder, FALSE, TRUE);
    
    g_free(filename);
    feedlistLoading = FALSE;
}

void conf_feedlist_save()
{
    gchar *filename = NULL, *filename_real = NULL;

    if (feedlistLoading)
	return;

    if (feedlist_save_timer) {
	g_source_remove(feedlist_save_timer);
	feedlist_save_timer = 0;
    }

    debug_enter("conf_feedlist_save");
    filename =
	g_strdup_printf("%s" G_DIR_SEPARATOR_S "feedlist.opml~",
			common_get_cache_path());

    if (0 == export_OPML_feedlist(filename, TRUE)) {
	filename_real =
	    g_strdup_printf("%s" G_DIR_SEPARATOR_S "feedlist.opml",
			    common_get_cache_path());
	if (rename(filename, filename_real) < 0)
	    g_warning("Error renaming %s to %s\n", filename,
		      filename_real);
	g_free(filename_real);
    }
    g_free(filename);
    debug_exit("conf_feedlist_save");
}

void conf_feedlist_schedule_save()
{
    if (!feedlistLoading && !feedlist_save_timer) {
	debug0(DEBUG_CONF, "Scheduling feedlist save");
	feedlist_save_timer =
	    g_timeout_add(5000, conf_feedlist_schedule_save_cb, NULL);
    }
}

/* returns true if namespace is enabled in configuration */
gboolean getNameSpaceStatus(gchar * nsname)
{
    GConfValue *value = NULL;
    gchar *gconfpath = NULL;
    gboolean status = FALSE;

    g_assert(NULL != nsname);
    gconfpath = g_strdup_printf("%s/ns_%s", RSS_SETTINGS, nsname);
    value = gconf_client_get_without_default(gconfclient, gconfpath, NULL);
    if (NULL == value) {
	ULOG_INFO("RSS namespace %s not yet configured! Activating...",
		nsname);
	setNameSpaceStatus(nsname, TRUE);
	status = TRUE;
    } else {
	status = gconf_value_get_bool(value);
    }
    g_free(gconfpath);
    g_free(value);
    return status;
}

/* used to enable/disable a namespace in configuration */
void setNameSpaceStatus(gchar * nsname, gboolean enable)
{
    gchar *gconfpath = NULL;

    g_assert(NULL != nsname);

    gconfpath = g_strdup_printf("%s/ns_%s", RSS_SETTINGS, nsname);
    setBooleanConfValue(gconfpath, enable);
    g_free(gconfpath);
}

/*----------------------------------------------------------------------*/
/* generic configuration access methods					*/
/*----------------------------------------------------------------------*/

gboolean checkConfValueExist(gchar * valuename)
{
    GConfValue *value = NULL;
    value = gconf_client_get_without_default(gconfclient, valuename, NULL);
    if (NULL==value) {
        return FALSE; //value not initialized
    } else {
        gconf_value_free(value);
        return TRUE;
    }
}

void setStringConfValue(gchar * valuename, gchar * value)
{
    GError *err = NULL;
    GConfValue *gcv = NULL;

    g_assert(valuename != NULL);
    g_assert(gconfclient != NULL);

    gcv = gconf_value_new(GCONF_VALUE_STRING);
    gconf_value_set_string(gcv, value);
    gconf_client_set(gconfclient, valuename, gcv, &err);
    gconf_value_free(gcv);
    is_gconf_error(&err);
}

gchar *getStringConfValue(gchar * valuename)
{
    GConfValue *value = NULL;
    gchar *result = NULL;

    g_assert(valuename != NULL);
    g_assert(gconfclient != NULL);

    value = gconf_client_get_without_default(gconfclient, valuename, NULL);
    if (NULL == value) {
	result = g_strdup("");
    } else {
	result = (gchar *) g_strdup(gconf_value_get_string(value));
	gconf_value_free(value);
    }

    return result;
}

gint getNumericConfValueDef(gchar * valuename, gint def)
{
    GConfValue *value = NULL;
    gint result = def;

    g_assert(valuename != NULL);
    g_assert(gconfclient != NULL);

    value = gconf_client_get_without_default(gconfclient, valuename, NULL);
    if (NULL != value) {
	result = gconf_value_get_int(value);
	gconf_value_free(value);
    }
    debug2(DEBUG_CONF, "Reading %s to %d", valuename, result);
    return result;
}

void setNumericConfValue(gchar * valuename, gint value)
{
    gconf_client_set_int(gconfclient, valuename, value, NULL);
}

gint getNumericConfValue(gchar * valuename)
{
    return gconf_client_get_int(gconfclient, valuename, NULL);
}

void setBooleanConfValue(gchar * valuename, gboolean value)
{
    gconf_client_set_bool(gconfclient, valuename, value, NULL);
}

gboolean getBooleanConfValue(gchar * valuename)
{
    return gconf_client_get_bool(gconfclient, valuename, NULL);
}
