/**
 * @file callbacks.c misc UI stuff
 *
 * Most of the GUI code is distributed over the ui_*.c
 * files but what didn't fit somewhere else stayed here.
 * 
 * Copyright (C) 2003, 2004 Lars Lindner <lars.lindner@gmx.net>
 * Copyright (C) 2004 Christophe Barbe <christophe.barbe@ufies.org>	
 * Copyright (C) 2004 Nathan J. Conrad <t98502@users.sourceforge.net>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gtk/gtk.h>
#include <gdk/gdk.h>

#include <gdk/gdkkeysyms.h>

#define DBUS_API_SUBJECT_TO_CHANGE
#include <hildon-widgets/gtk-infoprint.h>
#include <hildon-widgets/hildon-caption.h>
#include <hildon-widgets/hildon-time-editor.h>
#include <hildon-widgets/hildon-note.h>
#include <hildon-widgets/hildon-sort-dialog.h>

#include <hildon-widgets/hildon-file-details-dialog.h>
#include <hildon-fm/hildon-widgets/hildon-file-chooser-dialog.h>

#include <osso-log.h>
#include <string.h>
#include <glib.h>

#include <settings.h>
#include <gtkhtml/gtkhtml.h>
#include <osso-helplib.h>

#include "debug.h"
#include "interface.h"
#include "support.h"
#include "folder.h"
#include "feed.h"
#include "item.h"
#include "conf.h"
#include "export.h"
#include "htmlview.h"
#include "common.h"
#include "callbacks.h"
#include "ui_mainwindow.h"
#include "ui_folder.h"
#include "ui_feedlist.h"
#include "ui_itemlist.h"
#include "ui_queue.h"
#include "ui_notification.h"
#include "ui_mainwindow.h"
#include "ui_prefs.h"
#include "state_save.h"
#include "update.h"
#include "dbus.h"
#include "main.h"
#include "gtkhtml/gtkhtml_plugin.h"
#include "gtkhtml/htmlimport.h"

#include <libgnomevfs/gnome-vfs.h>

#ifdef UPDATE_DEBUG_MUTEX
#define G_MUTEX_LOCK(mx) { ULOG_DEBUG("##Trying to lock mutex: %s, %d, %p", __FUNCTION__, __LINE__, g_thread_self()); g_mutex_lock(mx); ULOG_DEBUG("##Locked mutex: %s, %d, %p", __FUNCTION__, __LINE__, g_thread_self()); }
#define G_MUTEX_UNLOCK(mx) { ULOG_DEBUG("##Trying to unlock mutex: %s, %d, %p", __FUNCTION__, __LINE__, g_thread_self()); g_mutex_unlock(mx); ULOG_DEBUG("##Unlocked mutex: %s, %d, %p", __FUNCTION__, __LINE__, g_thread_self()); }
#define GDK_THREADS_ENTERD { ULOG_DEBUG("@@Trying gdk_threads_enter: %s, %d, %p", __FUNCTION__, __LINE__, g_thread_self()); if (g_thread_self() != mainThread) gdk_threads_enter(); ULOG_DEBUG("@@Success gdk_threads_enter: %s, %d, %p", __FUNCTION__, __LINE__, g_thread_self()); }
#define GDK_THREADS_LEAVED { ULOG_DEBUG("@@Trying gdk_threads_leave: %s, %d, %p", __FUNCTION__, __LINE__, g_thread_self()); if (g_thread_self() != mainThread) gdk_threads_leave(); ULOG_DEBUG("@@Success gdk_threads_leave: %s, %d, %p", __FUNCTION__, __LINE__, g_thread_self()); }
#else
#define G_MUTEX_LOCK(mx) g_mutex_lock(mx);
#define G_MUTEX_UNLOCK(mx) g_mutex_unlock(mx);
#define GDK_THREADS_ENTERD { if (g_thread_self() != mainThread) gdk_threads_enter(); }
#define GDK_THREADS_LEAVED { if (g_thread_self() != mainThread) gdk_threads_leave(); }
#endif

extern AppData *app_data;

extern time_t rss_updated;
extern long rss_unread;

extern GHashTable *feedHandler;

extern GtkWidget *newsbox;
extern GtkWidget *feedlist_container;
extern feedPtr searchFeed;
extern GtkWidget * gtkhtml_htmlview;
extern GThread *mainThread;

gchar *prev_find_text = NULL;
/** prev_text_has_a_match
    will be set to TRUE if prev_find_text appears in the page
    will be reset to FALSE if pre_find_text is also reset
*/
gboolean prev_text_has_a_match; 
/* all used icons */
GdkPixbuf *icons[MAX_ICONS];

/* icon names */
static gchar *iconNames[] = { "qgn_list_rss_feed",	/* ICON_READ */
    "qgn_list_rss_feed_invalid", /* invalid feed*/
    "qgn_list_gene_fldr_cls",	/* ICON_VFOLDER */
    "qgn_list_gene_fldr_opn",	/* ICON_OPEN */
    "qgn_list_gene_fldr_clp",  /*  closed folder icon - plus sign */
    "qgn_list_gene_fldr_exp",   /*  expanded folder icon - minus sign */
    NULL
};

/*------------------------------------------------------------------------------*/
/* generic GUI functions							*/
/*------------------------------------------------------------------------------*/

/************************************************************************/
/*                        PRIVATE FUNCTIONS                             */
/************************************************************************/


static void change_manage_delete_button_status(GtkWidget * manage_folders_dialog)
{
  GtkWidget *TView=lookup_widget(manage_folders_dialog, "folder_view");
  GtkTreeSelection *curent_selection=gtk_tree_view_get_selection(GTK_TREE_VIEW(TView));
  g_print("%s\n",__FUNCTION__);
  if (gtk_tree_selection_get_selected(curent_selection,NULL,NULL))
  {
    gtk_widget_set_sensitive(lookup_widget(manage_folders_dialog, "mf_renamebtn"), TRUE);
    nodePtr node = ui_feedlist_get_selected_from_tree(TView);
    if (app_data->app_ui_data->search_mode>=SFM_REFRESH)
    {
      if (ui_feedlist_check_if_folder_empty(node))
      {
        gtk_widget_set_sensitive(lookup_widget(manage_folders_dialog, "mf_deletebtn"), TRUE);
      }
      else
      {
        gtk_widget_set_sensitive(lookup_widget(manage_folders_dialog, "mf_deletebtn"), FALSE);
      }
    }
    else
    {
      gtk_widget_set_sensitive(lookup_widget(manage_folders_dialog, "mf_deletebtn"), TRUE);
    }
    
  }
  else
  {
    gtk_widget_set_sensitive(lookup_widget(manage_folders_dialog, "mf_deletebtn"), FALSE);
    gtk_widget_set_sensitive(lookup_widget(manage_folders_dialog, "mf_renamebtn"), FALSE);
  }
}


/** Filters the tree model so that only folders are shown
  *
  * @param model the tree model to filter
  * @param iter a tree iter
  * @param data not used
  * @return TRUE if the node pointed at by the iterator is a folder, FALSE otherwise
  */
static gboolean filter_show_folders_only(GtkTreeModel * model,
					 GtkTreeIter * iter, gpointer data)
{

    nodePtr ptr = NULL;

    if (model == NULL) {
	ULOG_WARN("filter_show_folders_only: WARNING: model is NULL.");
	return TRUE;
    }

    gtk_tree_model_get(model, iter, FS_PTR, &ptr, -1);
    if (ptr == NULL) {
        return FALSE;
    }

    if (ptr->type == FST_FOLDER) {
        if ((gboolean) data && folder_get_nonremovable((folderPtr) ptr))
            return (folderPtr) ptr == ui_feedlist_get_root_folder();
        else
            return TRUE;
    } else {
	return FALSE;
    }

}

/* free the icons[] array
*/
void free_icons()
{
    gint i;
    ULOG_DEBUG("freeing the icons array");
    for (i = 0; i < MAX_ICONS ; i++)
        if (icons[i] != NULL)
        {
            g_object_unref(icons[i]);
            icons[i] = NULL;
        }
}

extern GtkWidget *newsbox;

static gboolean set_pane_pos(gpointer data)
{
    gtk_paned_set_position(GTK_PANED(GTK_PANED(newsbox)), (gint)data);
    return FALSE;
}

void ui_init(gboolean startIconified, AppUIData * app_ui_data)
{
    gchar *path;
    gint size;
    GTimer *tm =g_timer_new();
//    int i;

    g_assert(app_ui_data != NULL);
    g_assert(app_ui_data->main_view != NULL);
    app_ui_data->zoom_level = getNumericConfValue(ZOOM_LEVEL);
    if (app_ui_data->zoom_level==0.0)
    {
      app_ui_data->zoom_level=1.0;
      ULOG_DEBUG("Default Zoom level");
    }
    else
    {
      app_ui_data->zoom_level=app_data->app_ui_data->zoom_level/100.0;
    }
    
    ULOG_DEBUG("UI_INIT %s: %f", "zoom", g_timer_elapsed(tm, NULL));
    conf_keep_feeds_in_memory = getBooleanConfValue(KEEP_FEEDS_IN_MEMORY);
    
    /* initialize cache */
    path = getStringConfValue(RSS_SETTINGS_IMAGE_CACHE_DIR);
    if (!*path)
    {
        path = g_strdup_printf("%s" G_DIR_SEPARATOR_S ".osso_rss_feed_reader/cache/images",
                        g_get_home_dir());
        ULOG_DEBUG("Default image cache path %s", path);
    }
    size = getNumericConfValue(RSS_SETTINGS_IMAGE_CACHE_SIZE);
    app_ui_data->img_cache = file_cache_new(path, size*1024*1024);
    g_free(path);
    ULOG_DEBUG("UI_INIT %s: %f", "cache", g_timer_elapsed(tm, NULL));

    mainwindow = GTK_WIDGET(app_ui_data->main_view);
    create_mainwindow(app_ui_data);

    ULOG_DEBUG("UI_INIT %s: %f", "mainwindow", g_timer_elapsed(tm, NULL));

    /* order important !!! */
    ui_feedlist_init(feedlist);
    //getItemStore();

    ULOG_DEBUG("UI_INIT %s: %f", "feedlist", g_timer_elapsed(tm, NULL));

    ui_folder_set_expansion(ui_feedlist_get_root_folder(), TRUE);
    //TODO: REMOVE THIS
    ui_mainwindow_set_mode(itemlist_mode);

    ULOG_DEBUG("UI_INIT %s: %f", "set_mode", g_timer_elapsed(tm, NULL));
    /*
    for (i = 0; i < MAX_ICONS -2 ; i++)
        icons[i] = create_pixbuf(iconNames[i]);
    */

    /* TODO: will Those icons will be unref'ed by the gtktree view ?*/
    /* create rss feed icons */
    icons[0] = create_pixbuf(iconNames[0]);
    /* create rss invalid feed icons*/
    icons[1] = create_pixbuf(iconNames[1]);
    /* create rss folder closed icon . This is sometimes used in dialog lable*/
    icons[2] = create_pixbuf(iconNames[2]);
    
    /* create composite folder closed with emblem */
    icons[3] = create_composite_pixbuf(iconNames[2], iconNames[4]);
    /* create composite folder opened with emblem */
    icons[4] = create_composite_pixbuf(iconNames[3], iconNames[5]);

    ULOG_DEBUG("UI_INIT %s: %f", "pixbuf", g_timer_elapsed(tm, NULL));

    app_ui_data->search_mode = SFM_NORMAL;
    app_ui_data->feed_displayed = FALSE;
    	
    /* set up icons that are build from stock */
    ui_dnd_setup_URL_receiver(mainwindow);	/* setup URL dropping support */
    ULOG_DEBUG("UI_INIT %s: %f", "dnd", g_timer_elapsed(tm, NULL));
    conf_load_subscriptions();
    ULOG_DEBUG("UI_INIT %s: %f", "subscriptions", g_timer_elapsed(tm, NULL));
    /* Set the interval for automatic updates */
/*    if(getBooleanConfValue(RSS_SETTINGS_AUTOMATIC_UPDATES))
        set_update_timer(FALSE);*/
    ui_mainwindow_finish();	/* Display the initial status info */
    ULOG_DEBUG("UI_INIT %s: %f", "finish", g_timer_elapsed(tm, NULL));
    gtk_widget_set_sensitive(GTK_WIDGET
			     (app_data->app_ui_data->unsubscribe_button),
			     FALSE);
    gtk_widget_set_sensitive(GTK_WIDGET
			     (app_data->app_ui_data->
			      feed_properties_button), FALSE);

    if(getBooleanConfValue(DISABLE_TOOLBAR))
    {
        GTK_CHECK_MENU_ITEM(app_ui_data->menuitem_normal)->active = FALSE;
        gtk_widget_hide(app_ui_data->main_toolbar);
    }
    else
        GTK_CHECK_MENU_ITEM(app_ui_data->menuitem_normal)->active = TRUE;
    
    if(getBooleanConfValue(DISABLE_TOOLBAR_FULLSCREEN))
        GTK_CHECK_MENU_ITEM(app_ui_data->menuitem_full)->active = FALSE;
    else
        GTK_CHECK_MENU_ITEM(app_ui_data->menuitem_full)->active = TRUE;	

    if(getBooleanConfValue(ENABLE_FIND_ON_PAGE))
    {
        GTK_CHECK_MENU_ITEM(app_ui_data->menuitem_find_on_page)->active = TRUE;
    	gtk_widget_show_all(GTK_WIDGET(app_ui_data->find_toolbar));
    }
    else
    {
        GTK_CHECK_MENU_ITEM(app_ui_data->menuitem_find_on_page)->active = FALSE;
    	gtk_widget_hide_all(GTK_WIDGET(app_ui_data->find_toolbar));
    }
 
    ULOG_DEBUG("UI_INIT %s: %f", "hide", g_timer_elapsed(tm, NULL));

    /* initialize sort */
    int key = getNumericConfValue(RSS_SETTINGS_SORT_KEY);
    if (key < 0 || key > 1) {
        key = 0;
        setNumericConfValue(RSS_SETTINGS_SORT_KEY, key);
    }
    key = getNumericConfValueDef(RSS_SETTINGS_SORT_ORDER, 1);
    if (key < 0 || key > 1) {
        key = 0;
        setNumericConfValue(RSS_SETTINGS_SORT_ORDER, key);
    }
    
/*    if (checkConfValueExist(RSS_SETTINGS_REUSE_BROWSER_WINDOW) == FALSE)
    	setBooleanConfValue(RSS_SETTINGS_REUSE_BROWSER_WINDOW, TRUE);

    if (checkConfValueExist(RSS_SETTINGS_NEWEST_POSTS_FIRST)== FALSE)
    	setBooleanConfValue(RSS_SETTINGS_NEWEST_POSTS_FIRST, TRUE);
       	
    if (checkConfValueExist(RSS_SETTINGS_AUTOMATIC_UPDATES_INTERVAL)== FALSE)
        setNumericConfValue(RSS_SETTINGS_AUTOMATIC_UPDATES_INTERVAL, DEFAULT_INTERVAL);*/
    ULOG_DEBUG("UI_INIT %s: %f", "default values", g_timer_elapsed(tm, NULL));

/*    if (!getBooleanConfValue(LAST_SHOWFOLDERS))
        toggle_folder_list(NULL, NULL);*/
/*    if (getBooleanConfValue(LAST_WINDOW_MAXIMIZED))
        toggle_fullscreen_and_update_menu(NULL, app_data->app_ui_data);*/

    if (startIconified)
    	gtk_widget_hide(GTK_WIDGET(app_ui_data->app));
    else
    	gtk_widget_show(GTK_WIDGET(app_ui_data->app));
    ULOG_DEBUG("UI_INIT %s: %f", "finished", g_timer_elapsed(tm, NULL));

    g_timer_destroy(tm);
    g_idle_add(set_pane_pos, (gpointer)gtk_paned_get_position(GTK_PANED(newsbox)));
}

/*------------------------------------------------------------------------------*/
/* simple callbacks which don't belong to item or feed list 			*/
/*------------------------------------------------------------------------------*/
static gint quit_count = 0;
static gint quit_source = 0;
gboolean quit_reset_state = FALSE;

gboolean on_quit_cb(gpointer data)
{
    ULOG_DEBUG("on_quit 1");
    download_cancel_all(FALSE);
    
    ULOG_DEBUG("on_quit 2");
    free_icons();
    free_find_string();

    ULOG_DEBUG("on_quit 3");
    ui_mainwindow_save_search_entries();
    
    ULOG_DEBUG("on_quit 4");
    ui_itemlist_clear();	/* to free all item list related data */
    
    ULOG_DEBUG("on_quit 5");
    save_application_data();

    ULOG_DEBUG("on_quit 6");
    gtk_main_quit();

    return FALSE;
}

gboolean on_quit(gboolean reset_state)
{
    quit_reset_state = reset_state;

    if (quit_count == 0)
    {
        quit_count++;
		
        g_signal_handlers_block_by_func((gpointer) app_data->app_ui_data->app,
             G_CALLBACK(key_release), app_data->app_ui_data);

        g_signal_handlers_block_by_func((gpointer) app_data->app_ui_data->app,
             G_CALLBACK(key_press), app_data->app_ui_data);
		
        quit_source = g_idle_add(on_quit_cb, NULL);
	return(TRUE);
    }
    return(FALSE);
}

void abort_quit()
{
    if (quit_source) {
        g_source_remove(quit_source);
        quit_count = 0;
        quit_source = 0;
    }
}


void 
change_zoom_cb (GtkWidget *widget, gpointer data)
{
    AppUIData *app_ui_data;
    ULOG_DEBUG(__FUNCTION__);

    g_assert(app_data != NULL);
    app_ui_data = app_data->app_ui_data;
    app_ui_data->zoom_level=(int)data/100.0;
    setNumericConfValue(ZOOM_LEVEL,(int)data);
    gtk_html_set_magnification(
        GTK_HTML(gtk_bin_get_child(GTK_BIN(app_ui_data->html_scrollpane))),
        app_ui_data->zoom_level);

    gchar tmp[60], tmp2[60];
    snprintf(tmp, sizeof(tmp), "rss_ib_zoom_%d", (int)data);
    snprintf(tmp2, sizeof(tmp2), _(tmp));
    gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->app), tmp2);
}


/**tvh: Icon render for the Folder tree view in the dialog mode
 * (manage folder dialog and change folder dialog )
 * Must use the icon, just for the folder icons being changed
 * as the dialogs use a filtered, but different model, and treeviews

*/
void
   dialog_folder_icon_render_func (GtkTreeViewColumn *col,
                           GtkCellRenderer   *renderer,
                           GtkTreeModel      *model,
                           GtkTreeIter       *iter,
                           gpointer           user_data)
{
    folderPtr ptr = NULL;
    GtkTreeIter child;
    gtk_tree_model_get(model, iter,
               FS_PTR, &ptr, -1);
    gboolean has_sub_folder = gtk_tree_model_iter_children(model, &child, iter);

    if (ptr->type == FST_FOLDER )
    {
        if (has_sub_folder == FALSE) 
             g_object_set(renderer, "pixbuf", icons[ICON_FOLDER_CLOSED], NULL);
        else
             g_object_set(renderer, "pixbuf", ui_folder_select_icon(ptr->dialog_mode_expanded), NULL);
    }
}

void populate_folders(GtkWidget * dialog, gboolean hide_factory)
{
    GtkWidget *folder_view = lookup_widget(dialog, "folder_view");

    GtkTreeModel *model = NULL;
    GtkCellRenderer *textRenderer = NULL;
    GtkCellRenderer *iconRenderer = NULL;
    GtkTreeViewColumn *column = NULL;
    GtkTreeModel *filter = NULL;
    folderPtr rootfolder ;
    
    ULOG_DEBUG("populate_folders: starting...");

    filter = (GtkTreeModel *) gtk_tree_model_filter_new(GTK_TREE_MODEL(feedstore), NULL);

    gtk_tree_model_filter_set_visible_func(GTK_TREE_MODEL_FILTER(filter),
					   filter_show_folders_only,
					   (gpointer)hide_factory, NULL);
    model = GTK_TREE_MODEL(filter);

/*	model = GTK_TREE_MODEL(feedstore);*/
    ULOG_DEBUG("populate_folders: ...1...");

    gtk_tree_view_set_model(GTK_TREE_VIEW(folder_view), GTK_TREE_MODEL(model) );

    /* Show all the folders. */
    gtk_tree_view_collapse_all(GTK_TREE_VIEW(folder_view));
    /* Collapse the root folder */
    rootfolder = (folderPtr) ui_feedlist_get_root_folder();
    rootfolder->dialog_mode_expanded = FALSE;

    /* folder_view should have its own handlers */
    g_signal_connect(G_OBJECT(folder_view), "row-expanded",
                   G_CALLBACK(ui_folder_mfdialog_collapse_expand_cb), NULL);
    g_signal_connect(G_OBJECT(folder_view), "row-collapsed",
                   G_CALLBACK(ui_folder_mfdialog_collapse_expand_cb), NULL);

    /* we only render the state and title */
    iconRenderer = gtk_cell_renderer_pixbuf_new();
    textRenderer = gtk_cell_renderer_text_new();

    g_object_set(G_OBJECT(textRenderer), "ellipsize", PANGO_ELLIPSIZE_MIDDLE, NULL);
    ULOG_DEBUG("populate_folders: ...2...");

    gtk_cell_renderer_set_fixed_size(textRenderer, CELL_RENDERER_WIDTH, -1);

    column = gtk_tree_view_column_new();

    gtk_tree_view_column_pack_start(column, iconRenderer, FALSE);
    gtk_tree_view_column_pack_start(column, textRenderer, TRUE);

    /* Must use the func to render correctly as we don't want to modify the
    original model just because of expanding/collapsing the folders in dialogs
    */
    gtk_tree_view_column_set_cell_data_func(column, iconRenderer, dialog_folder_icon_render_func, NULL, NULL);
    /*
     gtk_tree_view_column_add_attribute(column, iconRenderer, "pixbuf",
                                       FS_ICON);
    */

    gtk_tree_view_column_add_attribute(column, iconRenderer, "pixbuf",
				       FS_ICON);
    gtk_tree_view_column_add_attribute(column, textRenderer, "markup",
				       FS_LABEL);

    gtk_tree_view_column_set_resizable(column, TRUE);
    gtk_tree_view_append_column(GTK_TREE_VIEW(folder_view), column);
}

/** Update the progress bar if needed
 * This is called in two cases , for now
    1. Refreshing
    2. Receiving images
    3. TODO: searching
*/
void update_progress_bar()
{
    GtkProgressBar *pb = GTK_PROGRESS_BAR(app_data->app_ui_data->progress_bar);
    
    switch (app_data->app_ui_data->progressbar_mode){
    case PROGRESSBAR_REFRESH_MODE: 
        gtk_progress_bar_set_fraction(pb, (float)get_counter(C_RSS_CURRENT) /
            get_counter(C_RSS_MAX));
        break;
    case PROGRESSBAR_IMAGE_MODE:
    {
        if (!get_counter(C_IMG_MAX))
            gtk_progress_bar_set_text(pb, "");
        else {
            gchar *fmt = _("rss_pb_toolbar_receiving");
            gchar *text = NULL;
            text = g_strdup_printf(fmt, 
                    get_counter(C_IMG_CURRENT), get_counter(C_IMG_MAX));
            gtk_progress_bar_set_text(pb, text);
            g_free(text);
        }
        break;
    }
    case PROGRESSBAR_CONNECT_MODE:
        gtk_progress_bar_pulse(pb);
        gtk_progress_bar_set_text(pb,  _("rss_pb_toolbar_connecting"));
        break;
    case PROGRESSBAR_SEARCH_MODE:
    //TODO
        break;
    default:
        break;
    }

    gtk_widget_queue_draw(GTK_WIDGET(pb));
}

/*
    PROGRESSBAR_NORMAL_MODE,
    PROGRESSBAR_SEARCH_MODE,
    PROGRESSBAR_CONNECT_MODE,
    PROGRESSBAR_REFRESH_MODE,
    PROGRESSBAR_IMAGE_MODE,
    PROGRESSBAR_RECEIVING_MODE
*/

static void pb_change_sensitive_buttons(AppUIData *app_ui_data, gboolean enable, 
    gboolean nosearch)
{
    ULOG_DEBUG("\n\n\n\n\n@@@@@@@@@@@ Set sensitive buttons: %d, %d\n\n\n\n",
        enable, nosearch);
    gtk_widget_set_sensitive(GTK_WIDGET
            (app_ui_data->menuitem_refresh_feeds), enable);
    gtk_widget_set_sensitive(GTK_WIDGET
            (app_ui_data->refresh_button), enable);
    gtk_widget_set_sensitive(GTK_WIDGET
            (app_ui_data->menuitem_add), enable);
    gtk_widget_set_sensitive(GTK_WIDGET
            (app_ui_data->new_feed_button), enable);
    gtk_widget_set_sensitive(GTK_WIDGET
            (app_ui_data->menuitem_unsubscribe), enable);
    gtk_widget_set_sensitive(GTK_WIDGET
            (app_ui_data->unsubscribe_button), enable);
    if (app_ui_data->unsubscribe_dialog)
    {
      gtk_widget_set_sensitive(GTK_WIDGET
        (app_ui_data->unsubscribe_dialog), enable);
    }
    gtk_widget_set_sensitive(GTK_WIDGET
            (app_ui_data->menuitem_feed_directory), enable);

    gtk_widget_set_sensitive(GTK_WIDGET
        (app_ui_data->menuitem_open), nosearch);
    gtk_widget_set_sensitive(GTK_WIDGET
        (app_ui_data->menuitem_properties), nosearch);
    gtk_widget_set_sensitive(GTK_WIDGET
        (app_ui_data->feed_properties_button), nosearch);
    gtk_widget_set_sensitive(GTK_WIDGET
            (app_ui_data->menuitem_manage_folders), enable || nosearch);
    gtk_widget_set_sensitive(GTK_WIDGET
            (app_ui_data->manage_folders_button), enable || nosearch);

    if (enable)    
        ui_feedlist_enable_def_buttons();
}

static gboolean progress_update(gpointer data)
{
//    int i;
//    ULOG_DEBUG("progress_update");
    
    if (app_data->app_ui_data->progressbar_mode != PROGRESSBAR_CONNECT_MODE)
        app_data->app_ui_data->pidle = 0;
    else {
        GDK_THREADS_ENTERD;
//        gdk_threads_enter();
/*        dotsno = (dotsno+1)%4;
        gchar dots[8];
        for (i=0; i < dotsno*2; i+=2)
        {
            dots[i]=' ';
            dots[i+1]='.';
        }
//        memset(dots, '.', dotsno);
        dots[dotsno*2] = 0;
//        gchar *tmp = g_markup_printf_escaped("%s<b>%s</b>", "rss_ia_connecting", dots);
        gchar *tmp = g_strconcat(_("rss_ia_connecting"), dots);
        gtk_progress_bar_set_text(GTK_PROGRESS_BAR(
            app_data->app_ui_data->progress_bar), tmp);
        gtk_widget_queue_draw (app_data->app_ui_data->progress_bar);
        g_free(tmp);*/
/*        gtk_progress_bar_pulse(GTK_PROGRESS_BAR(
            app_data->app_ui_data->progress_bar));*/
        update_progress_bar();
        GDK_THREADS_LEAVED;
//        gdk_threads_leave();
    }

    return app_data->app_ui_data->pidle;
}

void switch_progressbar(progress_bar_mode mode)
{
    g_assert(app_data);
    g_assert(app_data->app_ui_data);

    AppUIData *app_ui_data = app_data->app_ui_data;
    GtkProgressBar *pb = GTK_PROGRESS_BAR(app_ui_data->progress_bar);
    gint smode = 0;
    
    g_assert(app_ui_data->search_button);
    g_assert(app_ui_data->button_image_stop);

    ULOG_DEBUG("<!><!> switch_progressbar: %d <!><!>", mode);
    
    //Filtering some invalid switch modes
    if (mode == app_ui_data->progressbar_mode)
        return;

    ULOG_DEBUG("<!><!> switch_progressbar: 1, %s", app_ui_data->network_iap);
    if ((mode == PROGRESSBAR_CONNECT_MODE) && app_ui_data->network_iap &&
            *app_ui_data->network_iap)
        return;
    
    ULOG_DEBUG("<!><!> switch_progressbar: 2, %d-%d, %d-%d",
        app_ui_data->progressbar_mode,
        PROGRESSBAR_RECEIVING_MODE,
        mode,
        PROGRESSBAR_REFRESH_MODE);
    if ((app_ui_data->progressbar_mode == PROGRESSBAR_RECEIVING_MODE) &&
        (mode == PROGRESSBAR_REFRESH_MODE))
        return;

    ULOG_DEBUG("<!><!> switch_progressbar: continue <!><!>");
    if (app_ui_data->progressbar_mode == PROGRESSBAR_NORMAL_MODE)
    {
        g_assert(app_ui_data->button_image_stop);

        //disable toolbar / menu items
        pb_change_sensitive_buttons(app_ui_data, FALSE,
                mode != PROGRESSBAR_SEARCH_MODE);

        //change toolbutton image
        gtk_tool_button_set_icon_widget(GTK_TOOL_BUTTON
                (app_ui_data->search_button), app_ui_data->button_image_stop);

        //remove from the container the searchfield and add progressbar
        gtk_container_remove(GTK_CONTAINER(app_ui_data->var_field),
                app_ui_data->search_field);
        gtk_container_add(GTK_CONTAINER(app_ui_data->var_field),
                GTK_WIDGET(pb));

        //show the toolbar if it's hidden 
        gtk_widget_show (GTK_WIDGET(app_ui_data->main_toolbar));
/*
        if (pulse)
            app_ui_data->search_mode = SFM_CONNECTING;*/
    }

    if (app_ui_data->dlg_btn_unsubscribe)
        gtk_widget_set_sensitive(app_ui_data->dlg_btn_unsubscribe, 
            mode == PROGRESSBAR_NORMAL_MODE);

    if (app_ui_data->pidle)
    {
        g_source_remove(app_ui_data->pidle);
        app_ui_data->pidle = 0;
    }

    app_ui_data->progressbar_mode = mode;

    switch (mode) {
    case PROGRESSBAR_NORMAL_MODE:
        app_ui_data->search_mode = SFM_NORMAL;

        //enable toolbar / menu items
        pb_change_sensitive_buttons(app_ui_data, TRUE, TRUE);

        //change toolbutton image
        gtk_tool_button_set_icon_widget(GTK_TOOL_BUTTON
                (app_ui_data->search_button), app_ui_data->button_image_search);

        //remove from the container the progressbar and add the searchfield
        gtk_container_remove(GTK_CONTAINER(app_ui_data->var_field),
                GTK_WIDGET(pb));
        gtk_container_add(GTK_CONTAINER(app_ui_data->var_field),
                app_ui_data->search_field);

        //hide the toolbar if needed
        if( (getBooleanConfValue(DISABLE_TOOLBAR_FULLSCREEN) &&
            (hildon_appview_get_fullscreen
                    (hildon_app_get_appview(app_ui_data->app)))
            ) ||
            (
            getBooleanConfValue(DISABLE_TOOLBAR) &&
                (FALSE == hildon_appview_get_fullscreen
	                (hildon_app_get_appview(app_ui_data->app)))
            )
          ) // then hide the toolbar
            gtk_widget_hide(GTK_WIDGET(app_ui_data->main_toolbar));


        smode = SFM_NORMAL;

        break;

    case PROGRESSBAR_SEARCH_MODE:
        gtk_progress_bar_set_text(pb, _("rss_ia_searching"));
        gtk_progress_bar_set_fraction(pb, 0);
                      
        smode = SFM_SEARCH;

        break;

    case PROGRESSBAR_CONNECT_MODE:
//TODO Szs change id
        gtk_progress_bar_set_text(pb,  _("rss_pb_toolbar_connecting"));
        gtk_progress_bar_set_pulse_step(pb, 0.2);

        if (!app_ui_data->pidle)
            app_ui_data->pidle = g_timeout_add(200, progress_update, NULL);

        smode = SFM_CONNECTING;

        break;

    case PROGRESSBAR_REFRESH_MODE:
        gtk_progress_bar_set_text(pb, _("rss_pb_toolbar_refreshing"));
        gtk_progress_bar_set_fraction(pb, 0);
        
        smode = SFM_REFRESH;
        
        break;

    case PROGRESSBAR_IMAGE_MODE:
    {
        gchar *fmt = _("rss_pb_toolbar_receiving");
        gchar *text = g_strdup_printf(fmt, 0, get_counter(C_IMG_MAX));
        gtk_progress_bar_set_text(pb, text);
        g_free(text);
        gtk_progress_bar_set_fraction(pb, 1);

        smode = SFM_REFRESH;
        
        break;
    }
    case PROGRESSBAR_RECEIVING_MODE:
        gtk_progress_bar_set_text(pb, _("rss_pb_toolbar_receiving_file"));
        gtk_progress_bar_set_fraction(pb, 1);

        smode = SFM_REFRESH;

        break;
    }

    ULOG_DEBUG("<!><!> switch_progressbar: new mode %d, %d <!><!>", smode, app_ui_data->search_mode);
    if (smode == SFM_CONNECTING && app_ui_data->search_mode != smode)
        rss_dbus_signal_applet(OSSO_RSS_FEED_READER_CONNECTING_STARTED);
    else
    if (smode == SFM_REFRESH && app_ui_data->search_mode != smode)
        rss_dbus_signal_applet(OSSO_RSS_FEED_READER_REFRESH_STARTED);

    app_ui_data->search_mode = smode;
}

/** Hide progressbar again if needed
  * Sometimes when downloading either images or xml files (feeds)
  * The progressbar has to be shown if it is hidden
*/
/*void hide_progressbar_if_needed(void)
{

    AppUIData *app_ui_data = app_data->app_ui_data;
    g_assert(app_ui_data != NULL);

    if( (getBooleanConfValue(DISABLE_TOOLBAR_FULLSCREEN) &&
        ( TRUE == hildon_appview_get_fullscreen
	                (hildon_app_get_appview(app_ui_data->app)))
            ) ||
            (
            getBooleanConfValue(DISABLE_TOOLBAR) &&
                (FALSE == hildon_appview_get_fullscreen
	                (hildon_app_get_appview(app_ui_data->app)))
            )
          ) // then hide the toolbar
        gtk_widget_hide(GTK_WIDGET(app_ui_data->main_toolbar));
    return;
}*/

gboolean dialog_key_press(GtkWidget *widget, GdkEventKey *event, gpointer data)
{
  (void)data;

  if (event->state & (GDK_CONTROL_MASK |
      GDK_SHIFT_MASK |
      GDK_MOD1_MASK |
      GDK_MOD3_MASK |
      GDK_MOD4_MASK |
      GDK_MOD5_MASK)) {
    return FALSE;
      }

      switch (event->keyval)
      {
        case GDK_Escape:
          return TRUE;
          break;
        default:
          break;
      }

      return FALSE;
}

void ui_show_save_nodevicemem_dialog()
{
  HildonNote *note = HILDON_NOTE(hildon_note_new_information(
      app_data->app_ui_data->app,
  dgettext("ke-recv","cerm_device_memory_full")
                                                            ));
  hildon_note_set_button_text(note, dgettext("hildon-common-strings","sfil_ni_not_enough_memory_ok"));
  g_signal_connect(G_OBJECT(note), "key_press_event",
                   G_CALLBACK(dialog_key_press), NULL);

  gtk_dialog_run(GTK_DIALOG(note));
  gtk_widget_destroy(GTK_WIDGET(note));

}

void on_refresh_finished(gboolean disable_banner,
			 gboolean do_not_signal_applet)
{
    AppUIData *app_ui_data;
    int search_mode = 0;
	static gboolean in_refresh_finished = FALSE;
	app_data->app_ui_data->auto_refresh_is_ongoing=FALSE;
	if (in_refresh_finished) return;
	
	in_refresh_finished = TRUE;
        
    ULOG_DEBUG("@@@@@@@@@@@@@on_refresh_finished");
    
	g_assert(app_data != NULL);
    
    app_ui_data = app_data->app_ui_data;
    app_ui_data->nospace = FALSE;

    g_assert(app_ui_data != NULL);

    ui_lock();
    
//    ui_feedlist_update_applet();
    conf_feedlist_save();
//    hide_progressbar_if_needed();
    
    search_mode = app_ui_data->search_mode;
    app_ui_data->search_mode = SFM_NORMAL;
    if (app_ui_data->manage_folders_displayed)
    {
      change_manage_delete_button_status(app_ui_data->dialog);
    }
    if (SFM_INTERRUPTED == search_mode) 
    {
        ULOG_DEBUG("Refresh interrupted");

        if(app_ui_data->add_feed_dialog != NULL)
            newdialog_return_to_normal();

        if (!do_not_signal_applet &&
                app_ui_data->start_on_background &&
//                !app_ui_data->network_disconnected)
                app_ui_data->network_iap)
        {
            do_not_signal_applet = TRUE;
            rss_dbus_signal_applet("refresh_interrupted");
        }
        else if((!do_not_signal_applet &&
                    !app_ui_data->start_on_background) ||
                (!do_not_signal_applet &&
                    app_ui_data->start_on_background &&
//                    app_ui_data->network_disconnected))
                    !app_ui_data->network_iap))
            {
                do_not_signal_applet = TRUE;
                rss_dbus_signal_applet("refresh_finished");
            }

/*        if (!disable_banner)
            gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->app), 
                _("rss_ib_refresh_interrupted"));*/
    }

    ULOG_DEBUG("search mode: %d", search_mode);

    if (SFM_REFRESH == search_mode ||
        SFM_INTERRUPTED == search_mode) 
    {
        /* This happens if there are errors in refreshing, for example
           the feed is unavailable. This is set in ui_feed:ui_feed_process_update_result */
 
        if(app_ui_data->errors_in_refreshing) 
        {   
            /* If the user has interrupted the refresh we dont
               display banners */
            if(search_mode != SFM_INTERRUPTED) 
            {
                if(app_ui_data->add_feed_dialog == NULL) 
                {
                    if(app_ui_data->start_on_background) 
                    {
                        do_not_signal_applet = TRUE;
                        rss_dbus_signal_applet(OSSO_RSS_FEED_READER_REFRESH_PREVENTED_SIGNAL);
                    }
                    else
                        if (app_data->app_ui_data->pending_requests_nr > 1)
                            gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->app), 
                                _("rss_ni_error_refreshing_all_feeds"));
                        else if (app_data->app_ui_data->pending_requests_nr == 1)
                            gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->app), 
                                _("rss_ni_error_resfreshing_one_feed"));
                } else {
                    gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->app), 
                        _("rss_ni_adding_failed"));
                    newdialog_return_to_normal();
                }
            }
            app_ui_data->errors_in_refreshing = FALSE;
        }
        
        switch_progressbar(PROGRESSBAR_NORMAL_MODE);
        //restore_search_button(FALSE);
        //download_remove_dequeue_source();
    }

    if (!do_not_signal_applet) 
    {
        ULOG_DEBUG("Refresh completed");
        rss_dbus_refresh_completed();
    }
    
    /*tvh: Hide toolbar again if needed,
       according to gconf value and which fullscreen mode we're in*/
    if( (
        getBooleanConfValue(DISABLE_TOOLBAR_FULLSCREEN) &&
            ( TRUE == hildon_appview_get_fullscreen
	             (hildon_app_get_appview(app_ui_data->app)))
        ) ||
        (
        getBooleanConfValue(DISABLE_TOOLBAR) &&
            (FALSE == hildon_appview_get_fullscreen
	             (hildon_app_get_appview(app_ui_data->app)))
        )
      ) // then hide the toolbar
        gtk_widget_hide(GTK_WIDGET(app_ui_data->main_toolbar));


/*    app_data->app_ui_data->rss_max_download = 0;
    app_data->app_ui_data->rss_current_download = 0;  	 	
    app_data->app_ui_data->images_max_download = 0;
    app_data->app_ui_data->images_current_download = 0;  	 	
    app_data->app_ui_data->downloading_images = FALSE;*/

/*    if(!app_ui_data->start_on_background &&
       !do_not_signal_applet)
        set_update_timer(TRUE);*/
    if(app_ui_data->start_on_background) {
        ULOG_DEBUG("Calling quit!");
        on_quit(TRUE);
        ULOG_DEBUG("Calling quit ended!");
    }

    app_data->app_ui_data->pending_requests_nr = 0;

    ui_unlock();

    in_refresh_finished = FALSE;
        
}

void on_refreshbtn_clicked_cb(nodePtr ptr)
{
    feed_schedule_update((feedPtr) ptr, FEED_REQ_DOWNLOAD_FAVICON);
}

//static gint dotsno = 0;

/*void show_stop_refresh_button(gboolean pulse)
{
    AppUIData *app_ui_data = NULL;
    
    ULOG_DEBUG("show_stop_refresh_button: %d", pulse);

    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);
    
    if (pulse && app_data->app_ui_data->network_iap)
        return;
        
    app_ui_data = app_data->app_ui_data;

    if(app_data->app_ui_data->search_mode != SFM_REFRESH && 
       app_data->app_ui_data->search_mode != SFM_CONNECTING)
    {
        g_assert(NULL != app_ui_data->button_image_stop);

        gtk_widget_set_sensitive(GTK_WIDGET
                (app_ui_data->menuitem_refresh_feeds), FALSE);
        gtk_widget_set_sensitive(GTK_WIDGET(app_ui_data->refresh_button),
                FALSE);
        gtk_widget_set_sensitive(GTK_WIDGET
                (app_ui_data->menuitem_add), FALSE);
        gtk_widget_set_sensitive(GTK_WIDGET(app_ui_data->new_feed_button),
                FALSE);
        gtk_widget_set_sensitive(GTK_WIDGET(app_ui_data->menuitem_unsubscribe),
                FALSE);
        gtk_widget_set_sensitive(GTK_WIDGET(app_ui_data->unsubscribe_button),
                FALSE);									
        gtk_tool_button_set_icon_widget(GTK_TOOL_BUTTON
                    (app_ui_data->search_button),
                    app_ui_data->button_image_stop);

        gtk_container_remove(GTK_CONTAINER(app_ui_data->var_field),
                app_ui_data->search_field);
        gtk_container_add(GTK_CONTAINER(app_ui_data->var_field),
            app_ui_data->progress_bar);
*/
        /*tvh: show the toolbar if it's hidden 
        Then in on_refresh_finished, hide it again if necessary 
        */
/*
 *         gtk_widget_show (GTK_WIDGET(app_ui_data->main_toolbar));

        if (pulse)
            app_ui_data->search_mode = SFM_CONNECTING;
    }

    if (pulse)
    {
        gtk_progress_bar_set_text(GTK_PROGRESS_BAR(app_ui_data->progress_bar),
                _("rss_ia_connecting"));
        gtk_progress_bar_set_pulse_step(GTK_PROGRESS_BAR(app_ui_data->progress_bar),
                0.2);
        dotsno = 0;

        if (!app_ui_data->pidle)
            app_ui_data->pidle = g_timeout_add(200, progress_update, NULL);
    } else {
        if (app_ui_data->search_mode != SFM_REFRESH) {
            app_ui_data->search_mode = SFM_REFRESH;
            rss_dbus_signal_applet(OSSO_RSS_FEED_READER_REFRESH_STARTED);
        }

        gtk_progress_bar_set_text(GTK_PROGRESS_BAR(app_ui_data->progress_bar),
                _("rss_ia_refreshing"));
    }

    gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR
                (app_ui_data->progress_bar), 0);
}*/

/*void restore_search_button(gboolean pulse)
{
    AppUIData *app_ui_data = NULL;
     
    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);

    ULOG_DEBUG("restore_search_button, %d", pulse);
        
    app_ui_data = app_data->app_ui_data;

    g_assert(NULL != app_ui_data->button_image_refresh);
    
    if (app_ui_data->pidle)
        app_ui_data->pidle = 0;
    
    if (app_ui_data->search_mode == SFM_REFRESH && pulse)
        return;

    gtk_widget_set_sensitive(GTK_WIDGET
                 (app_ui_data->menuitem_refresh_feeds), TRUE);
    gtk_widget_set_sensitive(GTK_WIDGET(app_ui_data->refresh_button),
                 TRUE);
    gtk_widget_set_sensitive(GTK_WIDGET
                (app_ui_data->menuitem_add), TRUE);
    gtk_widget_set_sensitive(GTK_WIDGET(app_ui_data->new_feed_button),
                TRUE);			
    gtk_widget_set_sensitive(GTK_WIDGET(app_ui_data->menuitem_unsubscribe),
                TRUE);
    gtk_widget_set_sensitive(GTK_WIDGET(app_ui_data->unsubscribe_button),
                TRUE);												     
    gtk_tool_button_set_icon_widget(GTK_TOOL_BUTTON
                    (app_ui_data->search_button),
                    app_ui_data->button_image_search);
    gtk_container_remove(GTK_CONTAINER(app_ui_data->var_field),
             app_ui_data->progress_bar);
    gtk_container_add(GTK_CONTAINER(app_ui_data->var_field),
              app_ui_data->search_field);
}*/

void on_refresh_action_response(GtkWidget * refresh_dialog,
				gint responseId, gpointer user_data)
{
    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);

    if (responseId == GTK_RESPONSE_CANCEL ||
            responseId == GTK_RESPONSE_DELETE_EVENT) {
        gtk_widget_destroy(GTK_WIDGET(refresh_dialog));
        app_data->app_ui_data->refresh_dialog = NULL;
        ULOG_INFO("Refresh was canceled");
        return;
    }

    GtkWidget *selected_feed_widget =
        lookup_widget(refresh_dialog, "radio_button_selected_feed");
    gboolean selectedFeedOnly =
        gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON
                        (selected_feed_widget));
    
    AppUIData *app_ui_data = app_data->app_ui_data;
    
    ULOG_INFO("Refresh invoked...");
    /*TODO: Show toolbar if currently hidden and hide in on_refresh_finished again*/
    if (selectedFeedOnly) {
        app_ui_data->iap_action = OSSO_IAP_REFRESH_FEED;
        app_ui_data->new_subscription = (feedPtr) ui_feedlist_get_selected();
        app_ui_data->new_subscription_flags = 0;
    } else 
        app_ui_data->iap_action = OSSO_IAP_REFRESH_ALL_NO_DIALOG;
    
    ULOG_DEBUG("Requesting connection...");
    request_iap_connect();
/*        ULOG_INFO("Selected feed only...");
        nodePtr ptr = ui_feedlist_get_selected();

        if (ptr != NULL && ptr->type == FST_FEED) {
            ULOG_DEBUG("Updating the selected feed");
            feed_schedule_update((feedPtr) ptr, FEED_REQ_DOWNLOAD_FAVICON);
        }
    } 
    else {
	ULOG_DEBUG("Updating all feeds");
	res = ui_feedlist_do_for_all_with_int(NULL,
					      ACTION_FILTER_FEED,
					      update_node);
    }*/
    gtk_widget_destroy(GTK_WIDGET(refresh_dialog));
    app_data->app_ui_data->refresh_dialog = NULL;
}

void on_refresh_action(AppUIData * app_ui_data)
{
    GtkWidget *refreshdialog = NULL;

    g_assert(NULL != app_ui_data);

    /* A refresh dialog has already been created so lets not create
       another one. This might happen when several iap requests are
       sent in succession */
    
    if(app_ui_data->refresh_dialog == NULL)
    {
		
        app_ui_data->refresh_dialog =
            refreshdialog = create_refresh_dialog();
        g_signal_connect(G_OBJECT(refreshdialog), "response",
		         G_CALLBACK(on_refresh_action_response), NULL);
        gtk_widget_show(refreshdialog);
    }
}

void on_new_folder_action_response(GtkWidget * new_folder_dialog,
				   gint responseId, gpointer user_data)
{
    GtkWidget *dialog = NULL;
    GtkWidget *folder_view = NULL;

    ULOG_DEBUG(__FUNCTION__);

    dialog = (GtkWidget *) user_data;
    if (dialog != NULL) {
	folder_view = lookup_widget(dialog, "folder_view");
	g_assert(folder_view != NULL);
	ULOG_DEBUG("attempting to grab focus");
	gtk_widget_grab_focus(folder_view);
    }
}

gboolean update_node(nodePtr ptr)
{
    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);
    feed_schedule_update((feedPtr) ptr, FEED_REQ_RESET_TITLE |
					FEED_REQ_RESET_UPDATE_INT | 
					FEED_REQ_AUTO_DISCOVER |
					FEED_REQ_DOWNLOAD_FAVICON);
    /*ULOG_DEBUG("update_node: search_mode = %d",
	       app_data->app_ui_data->search_mode);*/
    return app_data->app_ui_data->search_mode != SFM_INTERRUPTED;
}

GtkWidget* display_new_folder_dialog(GtkWidget * widget, GtkWindow *parent_window)
{
    GtkTreeView *folder_view = NULL;
    GtkTreeViewColumn *column = NULL;
    GtkTreePath *path = NULL;
    GtkTreeIter iter;
    nodePtr node = NULL;
    folderPtr folder = NULL;
    GValue value;

    ULOG_DEBUG(__FUNCTION__);

    memset(&value, 0, sizeof(value));
    if (widget != NULL) {
	ULOG_DEBUG("widget was set on call");
	folder_view = GTK_TREE_VIEW(lookup_widget(widget, "folder_view"));
	g_assert(folder_view != NULL && GTK_IS_TREE_VIEW(folder_view));
	gtk_tree_view_get_cursor(folder_view, &path, &column);
	if (path != NULL) {
	    ULOG_DEBUG("and we got path to cursor");
	    gtk_tree_model_get_iter(gtk_tree_view_get_model
				    (folder_view), &iter, path);
	    gtk_tree_model_get_value(gtk_tree_view_get_model
				     (folder_view), &iter, FS_PTR, &value);
	    node = (nodePtr) g_value_peek_pointer(&value);
	    if (node != NULL && node->type == FST_FOLDER) {
		folder = (folderPtr) node;
	    }
	    gtk_tree_path_free(path);
	    path = NULL;
	}
    }
    GtkWidget *nfdialog = create_newfolderdialog(folder, folder_view, parent_window);
    g_value_unset(&value);
    g_signal_connect(G_OBJECT(nfdialog), "response",
		     G_CALLBACK(on_new_folder_action_response), widget);
    gtk_widget_show(nfdialog);
    
    return nfdialog;
}



/*TODO: this will be removed later as SAVE & SAVEAS is not used
 * Actually it can be used for exporting opml file
 */
enum e_file_chooser {
    FILE_CHOOSER_LOAD,
    FILE_CHOOSER_SAVE,
    FILE_CHOOSER_SAVEAS
};

static
HildonFileSystemModel *get_file_system_model(AppUIData *app_ui, GtkWidget *ref_widget)
{
  g_return_val_if_fail(app_ui != NULL, NULL);
  
  if(!app_ui->model) {
    app_ui->model = g_object_new (HILDON_TYPE_FILE_SYSTEM_MODEL,
				  "ref_widget", ref_widget,
				  NULL);
  }

  return app_ui->model;
}

/**
  Show a Hildon file chooser dialog
   
  @param app_ui Application UI data
  @param save #FILE_CHOOSER_LOAD, #FILE_CHOOSER_SAVE, or
              #FILE_CHOOSER_SAVEAS
  @return filename (g_free'd by the caller), or NULL if problems
*/
static
const gchar *file_chooser( AppUIData *app_ui, enum e_file_chooser mode )
{
    g_assert(app_ui);
    
    HildonFileSystemModel *model= NULL;
    GtkWidget *fs= NULL;
    model= get_file_system_model(app_ui, GTK_WIDGET(app_ui->app));
    
    if (!model)
      return NULL;

    app_data->app_ui_data->fsdialog = fs = hildon_file_chooser_dialog_new_with_properties(
				  GTK_WINDOW(app_ui->app),
				  "file_system_model", model,
				  "action", GTK_FILE_CHOOSER_ACTION_OPEN,
				  NULL);
    gtk_file_chooser_set_select_multiple(GTK_FILE_CHOOSER(fs), FALSE);
    gtk_window_set_modal (GTK_WINDOW(fs), TRUE);

        if (!fs) {
        ULOG_ERR( "Failed creating HildonFileChooserDialog" );
        return NULL;
    }

    /* RSS News Reader-specific dialog texts
    */
    gtk_window_set_title( GTK_WINDOW (fs), _("rss_ti_feed_select_opml_file") );

    /*what's this? 
    g_object_set( G_OBJECT (fs), 
                  "empty-text", _("sfil_va_select_object_no_objects_sketches"),
                  NULL );
    */

    // Filtering .xml or .opml files only
    GtkFileFilter *filter= gtk_file_filter_new();

    gtk_file_filter_add_pattern( filter, "*."OPML_FILE_EXTENSION_1 );//.xml
    gtk_file_filter_add_pattern( filter, "*."OPML_FILE_EXTENSION_2 );//.opml

    gtk_file_chooser_add_filter( GTK_FILE_CHOOSER (fs), filter );
    gtk_file_chooser_set_filter( GTK_FILE_CHOOSER (fs), filter );

    /* Folder that the dialog opens in 
    */
    const gchar *uri= "/home/user/MyDocs";
    g_assert(uri);
    gtk_file_chooser_set_current_folder_uri (GTK_FILE_CHOOSER (fs), uri);

    
    ossohelp_dialog_help_enable(GTK_DIALOG(fs),
                                RSS_NEWS_READER_HELP_OPEN, 
                                app_data->osso);

//    dialog_enable_help(GTK_DIALOG(fs), RSS_NEWS_READER_HELP_OPEN);
    
    /* Run the dialog
    */
    if ( gtk_dialog_run( GTK_DIALOG (fs) ) != GTK_RESPONSE_OK ) {
        /* dialog cancelled, exit */
        gtk_widget_destroy(fs);
        app_data->app_ui_data->fsdialog = NULL;
        return NULL;
    }
    /* check that user did not just click OK
        without selecting anything
     */
    const gchar *fn= gtk_file_chooser_get_uri( GTK_FILE_CHOOSER (fs) );
    if (!fn) {
        gtk_widget_destroy(fs);
        app_data->app_ui_data->fsdialog = NULL;
        return NULL;
    }

    /* "Houston, we have selection.."
    */
    const gchar *uri_new = gtk_file_chooser_get_current_folder_uri(GTK_FILE_CHOOSER(fs));
	 if (!uri_new) {
	 }
     /*
     else if ( strcmp( app_ui->uri_, uri_new ) != 0 ) {        // changed the folder
        g_free(app_ui->uri_);
        app_ui->uri_= uri_new;        
        uri_new=NULL;
        const gchar *cwd = gtk_file_chooser_get_current_folder(GTK_FILE_CHOOSER(fs));
        if (!cwd) {
		 } else if (strcmp(app_ui->cwd_,cwd)!=0){
        g_free(app_ui->cwd_);
        app_ui->cwd_=cwd;
        g_free(cwd), cwd=NULL; 
       } 
      }
    */ 
    app_data->app_ui_data->fsdialog = NULL;
    gtk_widget_destroy(fs), fs=NULL;

    // append opml file extension if necessary 
    // How could this happen? Didn't the pattern filter work?
    
    /*
    if (!g_str_has_suffix( fn, "." OPML_FILE_EXTENSION_1 )) {
        gchar *tmp= g_strconcat( fn, "." OPML_FILE_EXTENSION_1, NULL );
        g_free(fn);
        fn= tmp;
    }
    else if (!g_str_has_suffix( fn, "." OPML_FILE_EXTENSION_2 )) {
        gchar *tmp= g_strconcat( fn, "." OPML_FILE_EXTENSION_2, NULL );
        g_free(fn);
        fn= tmp;
    }
    */
    return fn;
}


void display_file_chooser_dialog(GtkWidget *widget, gpointer user_data)
{
    ULOG_DEBUG("display_file_chooser_dialog ...........\n");
    struct fp_prop_ui_data *data = (struct fp_prop_ui_data *)user_data;
    
    g_assert(app_data != NULL);
    
    const gchar *filename = file_chooser( app_data->app_ui_data, FILE_CHOOSER_LOAD);
    if (!filename)
      return;
    ULOG_DEBUG("display_file_chooser_dialog: Browsed file name = %s \n", filename);

    //now update the opmlAdressEntry field
//    GtkWidget *entry = lookup_widget (GTK_WIDGET (app_ui_data->dialog),"opmlAddressEntry");
//    gtk_entry_set_text(GTK_ENTRY(opmlAdressEntry),filename);
    gchar *name_for_display=gnome_vfs_unescape_string(filename,NULL);
    gtk_entry_set_text(GTK_ENTRY(data->sourceEntry), name_for_display);
    g_free(name_for_display);
    data->sourceURI=g_strdup(filename);
}


void on_manage_folders_action_response(GtkWidget * manage_folders_dialog,
				       gint responseId, gpointer user_data)
{
    AppUIData *app_ui_data = NULL;
     
    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);
    

    app_ui_data = app_data->app_ui_data;

    GtkTreeView *treeview =
	GTK_TREE_VIEW(lookup_widget(manage_folders_dialog, "folder_view"));
  GtkTreeSelection *selection=gtk_tree_view_get_selection(treeview);
    nodePtr node = ui_feedlist_get_selected_from_tree(treeview);

    ULOG_DEBUG(__FUNCTION__);

    switch (responseId) {
    case RSS_RESPONSE_NEW_FOLDER:
	app_ui_data->nfdialog = display_new_folder_dialog(manage_folders_dialog, GTK_WINDOW(user_data));
	break;
    case RSS_RESPONSE_RENAME_FOLDER:
	ULOG_INFO("Rename folder");

	if ((node != NULL) && (node->type == FST_FOLDER)) {
	    folderPtr folder = (folderPtr) node;
	    ULOG_INFO("Selected folder for rename: %s", folder->title);
	    if (folder_get_nonremovable(folder))
        /*
		gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->app),
			      _("rss_ib_renaming_folder_not_allowed"));
        */
		gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->app), dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE,
		                "sfil_ib_unable_to_rename_selected_folder"));

	    else
		create_folder_rename_dialog(folder);
	} else {
	    ULOG_INFO("No selection, or selection is not a folder!");
	}

	break;

    case RSS_RESPONSE_DELETE_FOLDER:
	ULOG_INFO("Delete folder");

	if ((node != NULL) && (node->type == FST_FOLDER)) {
	    folderPtr folder = (folderPtr) node;
      GtkTreePath *path_to_select=NULL;
      path_to_select=ui_feedlist_get_previous_after_delete_folder(treeview);
	    if (ui_feedlist_confirm_and_delete_folder(folder))
      {
        if (path_to_select)
        {
          ULOG_INFO("Selecting new folder");
          gtk_widget_grab_focus(treeview);
          gtk_tree_selection_select_path(selection,path_to_select);
          
        }
      }
      if (path_to_select)
      {
        gtk_tree_path_free(path_to_select);
      }
    change_manage_delete_button_status(manage_folders_dialog);
	} else {
	    ULOG_INFO("No selection, or selection is not a folder!");
	}

	break;
    case GTK_RESPONSE_CANCEL:
    case GTK_RESPONSE_DELETE_EVENT:
	app_ui_data->dialog = NULL;
	gtk_widget_destroy(GTK_WIDGET(manage_folders_dialog));
    app_data->app_ui_data->manage_folders_displayed=FALSE;
	break;
    }
}

void on_change_folder_action_response(GtkWidget * change_folder_dialog,
				      gint responseId, gpointer user_data)
{
  AppUIData *app_ui_data = NULL;

  ULOG_DEBUG(__FUNCTION__);

  g_assert(app_data != NULL);
    
  app_ui_data = app_data->app_ui_data;
  
    struct fp_prop_ui_data *ui_data = (struct fp_prop_ui_data *) user_data;

    switch (responseId) {
    case RSS_RESPONSE_NEW_FOLDER:
	app_ui_data->nfdialog = display_new_folder_dialog(change_folder_dialog, GTK_WINDOW(ui_data->parent_window));
	break;
    case GTK_RESPONSE_OK:
	ULOG_INFO("change folder");
	GtkTreeView *treeview =
	    GTK_TREE_VIEW(lookup_widget
			  (change_folder_dialog, "folder_view"));
	nodePtr node = ui_feedlist_get_selected_from_tree(treeview);
	if ((node != NULL) && (node->type == FST_FOLDER)) {
	    folderPtr folder = (folderPtr) node;
	    ULOG_INFO("Selected folder to change to: %s", folder->title);

	    ui_data->parent_folder = folder;
	    gtk_label_set_text(GTK_LABEL(ui_data->label_widget),
			       folder->title);
	} else {
	    ULOG_INFO
		("No selection, or selection is not a folder! Cannot change.");
	}
	gtk_widget_destroy(GTK_WIDGET(change_folder_dialog));
	//ui_data is destroyed by the parent dialog

	break;
    case GTK_RESPONSE_CANCEL:
    case GTK_RESPONSE_DELETE_EVENT:
	gtk_widget_destroy(GTK_WIDGET(change_folder_dialog));
	break;
    }
}


void display_change_folder_dialog(GtkWidget * widget, gpointer user_data)
{
    GtkTreeView *folder_view = NULL;
    GtkTreeModel *folder_model = NULL;
    folderPtr current_folder = NULL;
    GtkTreePath *current_path = NULL;
    AppUIData *app_ui_data = NULL;
    struct fp_prop_ui_data *data = NULL;

    ULOG_DEBUG(__FUNCTION__);

    g_assert(app_data != NULL);
    
    app_ui_data = app_data->app_ui_data;
    data = (struct fp_prop_ui_data *) user_data;
    
    if (data->parent_folder != NULL) {
	current_folder = data->parent_folder;
    } else {
	current_folder = ui_feedlist_get_root_folder();
    }
    g_assert(current_folder != NULL);

    GtkWidget *cfdialog = data->parent_window = create_change_folder_dialog(GTK_WINDOW(data->dialog));
    populate_folders(cfdialog, TRUE);
    app_ui_data->cfdialog = cfdialog;

    /* Select the root folder. */
    folder_view = GTK_TREE_VIEW(lookup_widget(cfdialog, "folder_view"));
    g_assert(folder_view != NULL);
    folder_model = gtk_tree_view_get_model(folder_view);
    g_assert(folder_model != NULL);
    
    ui_feedlist_search_model_by_ptr(folder_model,
				    (nodePtr) current_folder,
				    &current_path, NULL);
    gtk_tree_view_set_cursor(folder_view, current_path, NULL, FALSE);
    gtk_tree_path_free(current_path);
    current_path = NULL;

    g_signal_connect(G_OBJECT(cfdialog), "response",
		     G_CALLBACK(on_change_folder_action_response),
		     user_data);
    ULOG_INFO("Change folder dialog created, showing ...\n");
    gtk_widget_show(cfdialog);
    ULOG_INFO("Change folder dialog created, on screen.\n");
}

void on_folder_cursor_changed(GtkTreeView *tv, GtkWidget *dlg)
{
    nodePtr node = ui_feedlist_get_selected_from_tree(tv);
    gboolean enable = FALSE, root = FALSE,can_enable=FALSE;
    
    if (node && (node->type == FST_FOLDER)) {
        enable = !folder_get_nonremovable((folderPtr) node);
        root = (folderPtr) node == ui_feedlist_get_root_folder();
      if (enable&&(app_data->app_ui_data->search_mode>=SFM_REFRESH))
      {
        can_enable=ui_feedlist_check_if_folder_empty((folderPtr)node);
      }
      else
      {
        can_enable=TRUE;
      }
    }
    
    gtk_widget_set_sensitive(lookup_widget(dlg, "mf_newbtn"), enable || root);
    gtk_widget_set_sensitive(lookup_widget(dlg, "mf_renamebtn"), enable&&can_enable);
    gtk_widget_set_sensitive(lookup_widget(dlg, "mf_deletebtn"), enable&&can_enable);
}

void on_managefoldersbtn_clicked(GtkButton * button, gpointer user_data)
{
    AppUIData *app_ui_data = NULL;
     
    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);
    
    app_ui_data = app_data->app_ui_data;

    ULOG_DEBUG(__FUNCTION__);

    if (SFM_SEARCH == app_ui_data->search_mode){
        ULOG_DEBUG("rss_ib_unable_managefolders_searching");
    	gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->app), _("rss_ib_unable_managefolders_searching"));
        return;
    }
    else {
        ULOG_DEBUG("wtf");
    }

    ULOG_DEBUG("on_managefoldersbtn_clicked: creating mng folder");
    GtkWidget *mfdialog = app_ui_data->dialog =
	create_manage_folders_dialog();
    app_ui_data->mfdialog = mfdialog;

    GtkWidget *fld = lookup_widget(mfdialog, "folder_view");
    g_signal_connect(G_OBJECT(fld), "cursor_changed",
             G_CALLBACK(on_folder_cursor_changed), mfdialog);

    ULOG_DEBUG("on_managefoldersbtn_clicked: populating mng folder");
    populate_folders(mfdialog, FALSE);
    ULOG_DEBUG("on_managefoldersbtn_clicked: DONE populating mng folder");

    g_signal_connect(G_OBJECT(mfdialog), "response",
		     G_CALLBACK(on_manage_folders_action_response), mfdialog);
    GtkWidget *btn = lookup_widget(mfdialog, "mf_newbtn");
    g_signal_connect(G_OBJECT(btn), "insensitive_press",
		     G_CALLBACK(on_insensitive_infoprint), 
             _("rss_ib_unable_to_create_folder_into"));
    btn = lookup_widget(mfdialog, "mf_renamebtn");
    g_signal_connect(G_OBJECT(btn), "insensitive_press",
		     G_CALLBACK(on_insensitive_infoprint), 
		     dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE, "sfil_ib_unable_to_rename_selected_folder"));
    btn = lookup_widget(mfdialog, "mf_deletebtn");
    g_signal_connect(G_OBJECT(btn), "insensitive_press",
		     G_CALLBACK(on_insensitive_infoprint), 
                     dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE, "sfil_ib_unable_to_delete_selected_folder"));
    ULOG_DEBUG("on_managefoldersbtn_clicked: signal connected");
    gtk_widget_show(mfdialog);
}

void automatic_updates_toggled_cb (GtkToggleButton *togglebutton,
                                       gpointer         user_data)
{
    GtkWidget *prefdialog = (GtkWidget *) user_data;
    GtkWidget *connTypePulldown =
	lookup_widget(prefdialog, "connTypePulldown");
    GtkWidget *connIntervalPulldown =
	lookup_widget(prefdialog, "connIntervalPulldown");
    gboolean active = gtk_toggle_button_get_active(togglebutton);

    gtk_widget_set_sensitive(connTypePulldown, active);
    gtk_widget_set_sensitive(connIntervalPulldown, active);
}

gboolean on_prefdialog_insensitive_press(GtkWidget * widget,
                                            GdkEventButton * event,
                                            gpointer user_data)
{
    ULOG_DEBUG("Event %d !!!!!!!", event->type);
    if (!GTK_WIDGET_SENSITIVE(widget))
        gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->app), 
            _("rss_ib_no_feed_selected"));
    return FALSE;
}    

void on_refreshbtn_clicked(GtkButton * button, gpointer user_data)
{
    on_refresh_menuitem_selected(NULL,user_data);
}

void on_refresh_menuitem_selected(GtkWidget * menuitem, gpointer user_data)
{
    AppUIData *app_ui_data = NULL;
    nodePtr ptr = NULL;

    ULOG_DEBUG("***refresh menuitem selected");
    app_ui_data = (AppUIData *) user_data;
    g_assert(app_ui_data != NULL);

    if (SFM_REFRESH == app_ui_data->search_mode) 
    {
        gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->app), 
            _("rss_ib_refresh_already_started"));
    } 
    else if (SFM_NORMAL != app_ui_data->search_mode) 
    {
        gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->app), 
            _("rss_ib_cannot_start_refresh"));
    }
    else if (app_ui_data->low_mem)
    {
        gtk_infoprint( GTK_WINDOW(app_data->app_ui_data->app),
                            dgettext("ke-recv", "memr_ib_operation_disabled") );
        return;
    }
    else 
    {
//        ptr = ui_feedlist_get_selected();
      
      if (gtk_widget_is_focus(gtk_bin_get_child( GTK_BIN(app_data->app_ui_data->html_scrollpane))))
      
      {
      
        ptr = (nodePtr) displayed_node;
    
      } else {
      
        ptr = (nodePtr) ui_feedlist_get_selected();
      
      }
      

//        ui_search_clear_search_results();

        if(displayed_node == (nodePtr)searchFeed)
            ui_mainwindow_show_statistics();

        ULOG_DEBUG("refresh clicked: ...checking iap_action");

        if(ptr == NULL || ptr->type != FST_FEED)
            app_ui_data->iap_action = OSSO_IAP_REFRESH_ALL_NO_DIALOG;
        else
            if(menuitem == (GtkWidget *)app_ui_data->context_menuitem_refresh)
            {
                ULOG_DEBUG("refresh clicked:...refreshing only one feed");
                app_ui_data->iap_action = OSSO_IAP_REFRESH_FEED;
                app_ui_data->new_subscription = (feedPtr) ptr;
                app_ui_data->new_subscription_flags = 0;
            } else {
                on_refresh_action(app_data->app_ui_data);
                return;
            }
    //            app_ui_data->iap_action = OSSO_IAP_NORMAL;

        ULOG_DEBUG("refresh clicked:...requesting connection");
        request_iap_connect();
    }
}

void toggle_folder_list(GtkWidget * button, gpointer user_data)
{
    AppUIData *app_ui_data = NULL;
     
    gboolean state = FALSE;

    ULOG_DEBUG("toggle_folder_list");
    
    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);
    
    app_ui_data = app_data->app_ui_data;
    state = !(GTK_WIDGET_VISIBLE(feedlist_container) != 0);


    gboolean is_active =
        gtk_toggle_tool_button_get_active(GTK_TOGGLE_TOOL_BUTTON (app_ui_data->folders_list_toggle_button));
    gtk_toggle_tool_button_set_active(GTK_TOGGLE_TOOL_BUTTON (app_ui_data->folders_list_toggle_button),
                                             !is_active);
    ui_set_folder_list_state(app_ui_data, state, FALSE);

    /* Action for the folder_list_button */

    debug_exit("toggle_folder_list");
}


void on_folder_list_button_clicked(GtkWidget * button, gpointer user_data)
{
    AppUIData *app_ui_data = NULL;
     
    gboolean state = FALSE;

    debug_enter("on_folder_list_button_clicked");
    
    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);
    
    app_ui_data = app_data->app_ui_data;
    
    state = !(GTK_WIDGET_VISIBLE(feedlist_container) != 0);

    ui_set_folder_list_state(app_ui_data, state, TRUE);

    debug_exit("on_folder_list_button_clicked");
}


void toggle_toolbar(GtkWidget * widget, gpointer user_data)
{
    AppUIData *app_ui_data = NULL;
    GtkWidget *toolbar;
    
    ULOG_DEBUG(__FUNCTION__);
    
    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);
     
    app_ui_data = app_data->app_ui_data;
    
    g_assert(app_ui_data->main_toolbar != NULL);
    
    toolbar = app_ui_data->main_toolbar;
        
    gboolean fullscreen =
	hildon_appview_get_fullscreen(hildon_app_get_appview
				      (app_ui_data->app)) != 0;
     
    setBooleanConfValue(DISABLE_TOOLBAR_FULLSCREEN, !GTK_CHECK_MENU_ITEM(
                        app_ui_data->menuitem_full)->active);
    setBooleanConfValue(DISABLE_TOOLBAR, !GTK_CHECK_MENU_ITEM(
                        app_ui_data->menuitem_normal)->active);	    
    
    if ((!fullscreen && widget == GTK_WIDGET(app_ui_data->menuitem_full))
	|| (fullscreen
	    && widget == GTK_WIDGET(app_ui_data->menuitem_normal)))
	return;
	
    if (GTK_WIDGET_VISIBLE(toolbar)) {
	gtk_widget_hide(toolbar);
    } else {
        gtk_widget_show(toolbar);
    }
}


void on_toggle_fullscreen_mode(GtkWidget * widget, gpointer user_data)
{
    AppUIData *app_ui_data = NULL;
    gboolean state = FALSE;

    g_assert(app_data != NULL);
    
    app_ui_data = app_data->app_ui_data;
    g_assert(app_ui_data != NULL);

    state = !(hildon_appview_get_fullscreen
	      (hildon_app_get_appview(app_ui_data->app))
	      != 0);
    ui_set_fullscreen_state(app_ui_data, state, FALSE);
}


void toggle_fullscreen_and_update_menu(GtkWidget * widget,
				       gpointer user_data)
{
    AppUIData *app_ui_data = NULL;
    gboolean state = FALSE;

    g_assert(app_data != NULL);
    
    app_ui_data = app_data->app_ui_data;
    g_assert(app_ui_data != NULL);

    state = !(hildon_appview_get_fullscreen
	      (hildon_app_get_appview(app_ui_data->app))
	      != 0);
    ui_set_fullscreen_state(app_ui_data, state, TRUE);
    
}

void on_cut(GtkAction * action, gpointer data)
{
    GtkWidget *focus = NULL;
    AppUIData *app_ui_data = NULL;

    app_ui_data = (AppUIData *) data;
    g_assert(app_ui_data != NULL);

    if (GTK_WIDGET_SENSITIVE(app_ui_data->menuitem_cut)) {
	if (NULL != app_ui_data->dialog)
	    focus = gtk_window_get_focus(GTK_WINDOW(app_ui_data->dialog));
	else
	    focus = gtk_window_get_focus(GTK_WINDOW(app_ui_data->app));

	if (GTK_IS_EDITABLE(focus))
	    gtk_editable_cut_clipboard(GTK_EDITABLE(focus));
	return;
    }
    gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->main_menu_bar), 
	dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE, "ckct_ib_unable_to_cut"));
}

void on_copy(GtkAction * action, gpointer data)
{

    GtkWidget *focus = NULL;
    AppUIData *app_ui_data = NULL;

    app_ui_data = (AppUIData *) data;
    g_assert(app_ui_data != NULL);

    if (GTK_WIDGET_SENSITIVE(app_ui_data->menuitem_copy)) {
    	if (NULL != app_ui_data->dialog)
    	    focus = gtk_window_get_focus(GTK_WINDOW(app_ui_data->dialog));
    	else
    	    focus = gtk_window_get_focus(GTK_WINDOW(app_ui_data->app));

    	if (GTK_IS_EDITABLE(focus)) //like dialog entry or stuff
    	{
    	    gtk_editable_copy_clipboard(GTK_EDITABLE(focus));
    	    gtk_infoprint(GTK_WINDOW(app_ui_data->app),
                      dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE, "ecoc_ib_edwin_copied"));
          	return;
    	}
        else { //check if gtkhtml widget has focus and then
               //make sure it has selection first 
        //not gtkeditable, not feedlist, it should be gtkhtml
        //if ( (focus != feedlist)
        if ((GTK_WIDGET(focus) != GTK_WIDGET(feedlist))        
                     && htmlwidget_has_selection (app_data->app_ui_data->html_scrollpane))
            {
                htmlwidget_copy_selection (GTK_WIDGET(app_data->app_ui_data->html_scrollpane));
        	    gtk_infoprint(GTK_WINDOW(app_ui_data->app),
                              dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE, "ecoc_ib_edwin_copied"));
              	return;
            }
            else
            {
                ULOG_DEBUG ("on_copy: ***ERRORS****\nSomething must have gone wrong here.\
                            htmlwidget has no selection. Dialog is not focused. \
                            Nothing is selected \n \
                            Yet copy is SENSITIVE. Recheck the code \n ");                          
            }
        }

    }
    gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->main_menu_bar), 
	dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE, "ckct_ib_unable_to_copy"));


    //delete the gtktextview stuff use gtkhtml in stead
/*
    GtkWidget *focus = NULL;
    AppUIData *app_ui_data = NULL;

    app_ui_data = (AppUIData *) data;
    g_assert(app_ui_data != NULL);

    if (GTK_WIDGET_SENSITIVE(app_ui_data->menuitem_copy)) {
	if (NULL != app_ui_data->dialog)
	    focus = gtk_window_get_focus(GTK_WINDOW(app_ui_data->dialog));
	else
	    focus = gtk_window_get_focus(GTK_WINDOW(app_ui_data->app));

	if (GTK_IS_EDITABLE(focus))
	{
	    gtk_editable_copy_clipboard(GTK_EDITABLE(focus));
	    gtk_infoprint(GTK_WINDOW(app_ui_data->app),
                  _("ecoc_ib_edwin_copied"));
	}
	return;
    }
    gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->app), _("rss_ib_unable_to_copy"));
*/
}

void on_paste(GtkAction * action, gpointer data)
{
    GtkWidget *focus = NULL;
    AppUIData *app_ui_data = NULL;

    app_ui_data = (AppUIData *) data;
    g_assert(app_ui_data != NULL);

    GtkClipboard *clipboard = gtk_clipboard_get(GDK_SELECTION_CLIPBOARD);
    if (gtk_clipboard_wait_is_text_available(clipboard)) {

	if (GTK_WIDGET_SENSITIVE(app_ui_data->menuitem_paste)) {
	    if (NULL != app_ui_data->dialog)
		focus =
		    gtk_window_get_focus(GTK_WINDOW(app_ui_data->dialog));
	    else
		focus = gtk_window_get_focus(GTK_WINDOW(app_ui_data->app));

	    if (GTK_IS_EDITABLE(focus))
		gtk_editable_paste_clipboard(GTK_EDITABLE(focus));
	    return;
	}
        gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->main_menu_bar), _("rss_ib_unable_to_paste"));
    } else
      gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->main_menu_bar), _("rss_ib_nothing_to_paste"));
}

void on_selectall(GtkAction * action, gpointer data)
{
    GtkWidget *focus = NULL;
    AppUIData *app_ui_data = NULL;

    app_ui_data = (AppUIData *) data;
    g_assert(app_ui_data != NULL);

    ULOG_DEBUG("on_selectall");
    if (GTK_WIDGET_SENSITIVE(app_ui_data->menuitem_selectall)) {
    	if (NULL != app_ui_data->dialog)
    	    focus = gtk_window_get_focus(GTK_WINDOW(app_ui_data->dialog));
    	else
    	    focus = gtk_window_get_focus(GTK_WINDOW(app_ui_data->app));

    	if (GTK_IS_EDITABLE(focus)) //like dialog entry or stuff
    	{
    	    gtk_editable_select_region(GTK_EDITABLE(focus), 0, -1);
            ULOG_DEBUG("gtk editable SELECTED ALL");
          	return;
    	}
        else { //must be the htmlwidget that has selection 
                //make sure it has selection first 
                //not gtkeditable, not feedlist, it should be gtkhtml
            ULOG_DEBUG("on_selectall: checking if focus is not feedlist...");
            if (GTK_WIDGET(focus) != GTK_WIDGET(feedlist))
            //if (focus != feedlist)
            {
                gtkhtml_selectall();
                ULOG_DEBUG("GKTHML selected all");
              	return;
            }
            else
            {
                ULOG_DEBUG ("on_copy: ***ERRORS****\nSomething must have gone wrong here.\
                            htmlwidget has no selection. Dialog is not focused. \
                            gtkeditable is empty \n \
                            Yet select all is SENSITIVE. Recheck the code \n ");                          
                 return;
                           
            }
        }

    }
    else 
      gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->main_menu_bar),
             dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE, "ckct_ib_nothing_to_select"));

    return;

}

void on_menuitem_edit(GtkWidget * widget, gpointer user_data)
{
    AppUIData *app_ui_data = (AppUIData *) user_data;
    GtkWidget *focus = NULL;
    GtkClipboard *clipboard = gtk_clipboard_get(GDK_SELECTION_CLIPBOARD);

    g_assert(NULL != user_data);
    g_assert(NULL != clipboard);

    /* first, pretend everything is not click-able */
    gtk_widget_set_sensitive(GTK_WIDGET(app_ui_data->menuitem_cut), FALSE);
    gtk_widget_set_sensitive(GTK_WIDGET(app_ui_data->menuitem_copy),
			     FALSE);
    gtk_widget_set_sensitive(GTK_WIDGET(app_ui_data->menuitem_paste),
			     FALSE);
    gtk_widget_set_sensitive(GTK_WIDGET(app_ui_data->menuitem_selectall),
			     FALSE);
			     
    if (NULL != app_ui_data->dialog)
    	focus = gtk_window_get_focus(GTK_WINDOW(app_ui_data->dialog));
    else
    	focus = gtk_window_get_focus(GTK_WINDOW(app_ui_data->app));

    if (GTK_IS_EDITABLE(focus)) {
    	if (gtk_editable_get_selection_bounds
	       (GTK_EDITABLE(focus), NULL, NULL)) {
	               gtk_widget_set_sensitive(GTK_WIDGET(app_ui_data->menuitem_cut),
	   		       TRUE);
	               gtk_widget_set_sensitive(GTK_WIDGET
				     (app_ui_data->menuitem_copy), TRUE);
	       }
	   if (gtk_clipboard_wait_is_text_available(clipboard)) {
	        gtk_widget_set_sensitive(GTK_WIDGET
				     (app_ui_data->menuitem_paste), TRUE);
	   }
	   gchar * tmp = gtk_editable_get_chars(GTK_EDITABLE(focus), 0, -1);
	   if (strlen(tmp) != 0) //there is something to be select-all'ed
          gtk_widget_set_sensitive(GTK_WIDGET(app_ui_data->menuitem_selectall),
			     TRUE);
	   g_free(tmp);
    }

    /* htmlwidget checking... */
    else {
        ULOG_DEBUG("on_menuitem_edit: gtkeditable is not the focus");
        //if ((focus != feedlist) &&
        if ( (GTK_WIDGET(focus) != GTK_WIDGET(feedlist)) &&
           htmlwidget_has_selection (app_data->app_ui_data->html_scrollpane))
        {
            ULOG_DEBUG("on_menuitem_edit: htmlwidget has selection.");
            //can set to COPY sensitive.
            gtk_widget_set_sensitive(GTK_WIDGET
				     (app_ui_data->menuitem_copy), TRUE);
          
        }
        //TODO: This will be safe.        
        //gboolean has_focus;
        //g_object_get(G_OBJECT(app_ui_data->htmlwidget), "focus", &has_focus, NULL);

        //not gtkeditable, not feedlist, it should be gtkhtml

        //if ( focus != feedlist)
        if ( GTK_WIDGET(focus) != GTK_WIDGET(feedlist))
            gtk_widget_set_sensitive(GTK_WIDGET(app_ui_data->menuitem_selectall),
	                   		     TRUE);
    }

}

/* Insensitiveize feed directory if neeeded */
void on_menuitem_tools(GtkWidget * widget, gpointer user_data)
{

    gchar *feed_dir_source = getStringConfValue(FEED_DIRECTORY_SOURCE);
    if (strlen(feed_dir_source) == 0 )
    {
        ULOG_DEBUG("on_menuitem_tools: setting feed dir to insensitive");
        gtk_widget_hide(GTK_WIDGET
                     (app_data->app_ui_data->menuitem_feed_directory));
    }
    else if (app_data->app_ui_data->search_mode == SFM_NORMAL)
        gtk_widget_show(GTK_WIDGET
                     (app_data->app_ui_data->menuitem_feed_directory));

    g_free(feed_dir_source);
    return;
}
void toggle_find_on_page(GtkWidget *widget, gpointer user_data)
{
    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL); 

    setBooleanConfValue(ENABLE_FIND_ON_PAGE, GTK_CHECK_MENU_ITEM(widget)->active);
    
    if(GTK_CHECK_MENU_ITEM(widget)->active)
        gtk_widget_show_all(GTK_WIDGET(app_data->app_ui_data->find_toolbar));
    else
        gtk_widget_hide_all(GTK_WIDGET(app_data->app_ui_data->find_toolbar));
}

void free_find_string()
{
    AppUIData *app_ui_data = NULL;
    
    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);
    
    app_ui_data = app_data->app_ui_data;
    
    if(app_ui_data->find_string != NULL &&
	   strcmp(app_ui_data->find_string,""))
	g_free(app_ui_data->find_string);

    app_ui_data->find_string = "";
}

void find_toolbar_search(GtkWidget * widget, gpointer data)
{
    gchar *find_text = NULL;
    gchar *infotext = NULL;
    
    ULOG_DEBUG("find_toolbar_search");

    g_assert(widget != NULL);
    
    g_object_get(G_OBJECT(widget), "prefix", &find_text, NULL);
    ULOG_DEBUG("prefix: %s", find_text);

    if (( NULL == find_text ) || (strcmp (find_text, "") == 0) )
    {    
        gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->app), 
            dgettext(HILDON_LIBS_L10N_PACKAGE, "ckct_ib_find_enter_the_text"));
        return;    
    }    
    /*
    if(strcmp(find_text,""))
    {

	if(strcmp(app_ui_data->find_string,find_text))
	{
	    free_find_string();
	    app_ui_data->find_string = g_strdup(find_text);
	    
	    gtk_text_buffer_get_start_iter(textbuffer, &iter);
	
	    find(iter, find_text, textbuffer, TRUE);
	}
	else
	{
	    gtk_text_buffer_get_iter_at_mark(textbuffer,
					&iter,
					gtk_text_buffer_get_insert
					(textbuffer));
					
            find(iter, find_text, textbuffer, FALSE);					
	}
    }
    else
    {
        free_find_string();
        gtk_infoprint(NULL, _("rss_ib_enter_text_to_be_found"));
    }
    */
    /*TODO: when is rewind = FALSE/TRUE. Add a global find_text to compare
    if this is first search or second search with the same str 
     TODO: remove find() */

    gboolean rewind = FALSE;
    gboolean matched = FALSE;
    if (prev_find_text == NULL)
    {
        prev_find_text = g_strdup(find_text);
        prev_text_has_a_match = FALSE;
        rewind = TRUE; //start from beginning if starting a new search
    }
    else if (strcmp(find_text,prev_find_text) != 0)
    {
        if ( NULL != prev_find_text)
            g_free(prev_find_text);
        prev_find_text = g_strdup(find_text);
        prev_text_has_a_match = FALSE;
        rewind = TRUE; //start from beginning if starting a new search
    }
    else if ((strcmp(find_text,prev_find_text) == 0) && (prev_text_has_a_match == FALSE))
        rewind = TRUE;
 

    gtkhtml_find_cb( GTK_WIDGET(app_data->app_ui_data ->html_scrollpane),
        find_text,rewind, &matched );

    if (matched)
        prev_text_has_a_match = TRUE;

    if (matched == FALSE)
    {
        if ( rewind == TRUE )
        { 
            gchar *fmt = dgettext(HILDON_LIBS_L10N_PACKAGE, 
                "ckct_ib_find_no_matches");
            infotext =  g_strdup_printf (fmt,find_text);
            gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->app), infotext);
            g_free(infotext);
        }
        else 
            gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->app), 
                dgettext(HILDON_LIBS_L10N_PACKAGE, "ckct_ib_find_no_matches"));

    }
    g_free(find_text);
    find_text = NULL;

}

void find_toolbar_close(GtkWidget * widget, gpointer data)
{
    ULOG_DEBUG("find toolbar close");
    AppUIData * app_ui_data = NULL;
     
    g_assert(app_data != NULL);
     
    app_ui_data = app_data->app_ui_data;

    g_assert(app_ui_data != NULL);
    
    if(app_ui_data->find_toolbar == NULL)
        return;
    
    GTK_CHECK_MENU_ITEM(app_ui_data->menuitem_find_on_page)->active=0;

    setBooleanConfValue(ENABLE_FIND_ON_PAGE, FALSE);
    
    gtk_widget_hide_all(GTK_WIDGET(app_ui_data->find_toolbar));
}

/*------------------------------------------------------------------------------*/
/* status bar callback, error box function, GUI update function			*/
/*------------------------------------------------------------------------------*/

void ui_update(void)
{
    if (!lifereaStarted)
	return;
    while (gtk_events_pending())
        gtk_main_iteration();
}

void ui_show_error_box(const char *format, ...)
{
    GtkWidget *dialog = NULL;
    va_list args;
    gchar *msg = NULL;

    if(format == NULL)
        return;

    va_start(args, format);
    msg = g_strdup_vprintf(format, args);
    va_end(args);

    dialog = gtk_message_dialog_new(GTK_WINDOW(mainwindow),
				    GTK_DIALOG_DESTROY_WITH_PARENT,
				    GTK_MESSAGE_ERROR,
				    GTK_BUTTONS_CLOSE, "%s", msg);
    gtk_dialog_run(GTK_DIALOG(dialog));
    gtk_widget_destroy(dialog);
    g_free(msg);
}

void ui_show_info_box(const char *format, ...)
{
    GtkWidget *dialog = NULL;
    va_list args;
    gchar *msg = NULL;

    if(format == NULL)
        return;
	
    va_start(args, format);
    msg = g_strdup_vprintf(format, args);
    va_end(args);

    dialog = gtk_message_dialog_new(GTK_WINDOW(mainwindow),
				    GTK_DIALOG_DESTROY_WITH_PARENT,
				    GTK_MESSAGE_INFO,
				    GTK_BUTTONS_CLOSE, "%s", msg);
    gtk_dialog_run(GTK_DIALOG(dialog));
    gtk_widget_destroy(dialog);
    g_free(msg);
}

/*------------------------------------------------------------------------------*/
/* exit handler									*/
/*------------------------------------------------------------------------------*/

void save_state(AppData *app_data)
{
    AppState *app_state = NULL;

    g_assert(app_data != NULL);
    
    reset_state_file(app_data->osso);
    
    app_state = create_app_state();
    construct_app_state(app_state);
    reader_save_state(app_data, app_state);
    destroy_app_state(app_state);
    app_state = NULL;
}

void restore_state(AppData *app_data)
{
    AppState *app_state = NULL;
    StateSaveResultCode stres = ST_SUCCESS;

    g_assert(app_data != NULL);

    app_state = create_app_state();
    stres = reader_load_state(app_data, app_state);
    
    ULOG_DEBUG("State save result code: %d", stres);
    if (stres == ST_SUCCESS) {
        ULOG_DEBUG("Updating app state");
	update_app_state(app_state);
    }
    destroy_app_state(app_state);
    app_state = NULL;
}

void on_untop(GtkWidget * widget, gpointer user_data)
{
    AppData *app_data = NULL;

    app_data = (AppData *) user_data;

    ULOG_DEBUG("**************************on_untop**************************");

    g_assert(app_data != NULL);
    
    g_assert(app_data->app_ui_data != NULL);

    app_data->app_ui_data->application_topped = FALSE;

    hildon_app_set_killable(app_data->app_ui_data->app, TRUE);
}

void on_top(GtkWidget * widget, gpointer user_data)
{   
    AppData *app_data = NULL;

    app_data = (AppData *) user_data;

    ULOG_DEBUG("************************on_top*****************************");
      
    g_assert(app_data != NULL);
    
    //ui_mainwindow_empty_search_field();
                
    g_assert(app_data->app_ui_data != NULL);
    
    app_data->app_ui_data->start_on_background = FALSE;
    app_data->app_ui_data->application_topped = TRUE;
    
    /* Main-window is hidden if we were started for refresh in background */
    gtk_widget_show(GTK_WIDGET(app_data->app_ui_data->app));

    hildon_app_set_killable(app_data->app_ui_data->app, FALSE);
}

gboolean on_key_press(GtkWidget * widget, GdkEventKey * event,
		      gpointer user_data)
{
    AppData *app_data = (AppData *) user_data;

    switch (event->keyval) {
    case GDK_F3:		/* For debugging only. */
	ULOG_DEBUG("emulate task away\n");
	on_untop(NULL, app_data);
	return TRUE;
    default:
	break;
    }
    return FALSE;
}

gboolean key_press_for_cancelling_dialog(GtkWidget *widget, GdkEventKey *event, gpointer data)
{
    (void) data;

    g_assert(app_data != NULL);
    g_assert(app_data->app_ui_data != NULL);

    if(event == NULL)
        return FALSE;

    switch (event->keyval) {
        case GDK_Escape:
            app_data->app_ui_data->dialog_closed = TRUE;
            return FALSE;
        default:
            return FALSE;
    }
    return FALSE;
}

void on_open_cmenuitem_selected(GtkWidget * menuitem, gpointer user_data)
{
    AppUIData *app_ui_data = (AppUIData *)user_data;
     
    g_assert(NULL != app_ui_data);

    if (!app_ui_data->context_url || !*app_ui_data->context_url)
        on_menu_open(GTK_MENU_ITEM(menuitem), user_data);
    else {
        invoke_url(app_ui_data->context_url);
    }
}

void on_new_folder_cmenuitem_selected(GtkWidget * menuitem, gpointer user_data)
{
    AppUIData *app_ui_data = (AppUIData *)user_data;
    nodePtr node = NULL;
     
    g_assert(NULL != app_ui_data);
    
    node = ui_feedlist_get_selected();
    if (node && node->type == FST_FEED)
        node = (nodePtr) ui_feedlist_get_parent(node);
            
    if (node) {
    GtkWidget *nfdialog = create_newfolderdialog((folderPtr)node, 
        GTK_TREE_VIEW(feedlist), GTK_WINDOW(app_ui_data->app));
    g_signal_connect(G_OBJECT(nfdialog), "response",
		     G_CALLBACK(on_new_folder_action_response), NULL);
    gtk_widget_show(nfdialog);
    }
}

void on_rename_cmenuitem_selected(GtkWidget * menuitem, gpointer user_data)
{
    AppUIData *app_ui_data = (AppUIData *)user_data;
    nodePtr node = NULL;
     
    g_assert(NULL != app_ui_data);
    
    node = ui_feedlist_get_selected();
    if (node && node->type == FST_FOLDER) {
	    if (folder_get_nonremovable((folderPtr)node))
		gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->app),
			      dgettext(HILDON_FM_L10N_PACKAGE, "sfil_ib_renaming_not_allowed"));
	    else
		create_folder_rename_dialog((folderPtr)node);
    }
}

void on_delete_cmenuitem_selected(GtkWidget * menuitem, gpointer user_data)
{
    AppUIData *app_ui_data = (AppUIData *)user_data;
    nodePtr node = NULL;
     
    g_assert(NULL != app_ui_data);
    
    node = ui_feedlist_get_selected();
    if (node && node->type == FST_FOLDER) {
	    ui_feedlist_confirm_and_delete_folder((folderPtr)node);
    }
}

void on_copy_link_menuitem_selected(GtkWidget * menuitem, gpointer user_data)
{
    GtkClipboard *clipboard = gtk_clipboard_get(GDK_SELECTION_CLIPBOARD);
    AppUIData *app_ui_data = (AppUIData *)user_data;
    gchar *url = NULL;

    g_assert(NULL != clipboard);
    g_assert(NULL != app_ui_data);
    
    clipboard = gtk_clipboard_get(GDK_SELECTION_CLIPBOARD);
    url = strip_url(app_ui_data->context_url);
    gtk_clipboard_set_text(clipboard, url, -1);
    gtk_infoprint(GTK_WINDOW(app_ui_data->app),
                      dgettext(HILDON_COMMON_STRINGS_L10N_PACKAGE, "ecoc_ib_edwin_copied"));
    g_free(url);
}

void on_send_post_menuitem_selected(GtkWidget * menuitem, gpointer user_data)
{
    g_assert(NULL != app_data);
    g_assert(NULL != app_data->app_ui_data);

    post_url(app_data->app_ui_data->context_url, user_data != NULL);
}

void on_load_image_menuitem_selected(GtkWidget * menuitem, gpointer user_data)
{
    AppUIData *app_ui_data = (AppUIData *)user_data;
    g_assert(NULL != app_ui_data);

    if (!app_ui_data->load_url && (app_ui_data->search_mode <= SFM_INTERRUPTED) &&
        app_ui_data->image_object && app_ui_data->image_url)
    {
        app_ui_data->load_url = g_strdup(app_ui_data->image_url);
        file_cache_remove(app_ui_data->img_cache, app_ui_data->load_url);
        html_image_edit_set_url(HTML_IMAGE(app_ui_data->image_object), 
            app_ui_data->load_url);
    }
}

gboolean ui_show_replace_file_dialog(gchar *file)
{
    gboolean answer = FALSE;
    gchar *base = g_path_get_basename(file);
    gchar *fmt = dgettext(HILDON_FM_L10N_PACKAGE, "docm_nc_replace_file");
    gchar *tmp = g_strdup_printf("%s\n%s", fmt, base);
    
    HildonNote *note = HILDON_NOTE(hildon_note_new_confirmation_add_buttons(
        GTK_WINDOW(app_data->app_ui_data->app),
        tmp,
        dgettext(HILDON_FM_L10N_PACKAGE, "docm_bd_replace_file_ok"),
        1,
        dgettext(HILDON_FM_L10N_PACKAGE, "docm_bd_replace_file_cancel"),
        0,
        NULL));
    answer = gtk_dialog_run(GTK_DIALOG(note));
    gtk_widget_destroy(GTK_WIDGET(note));
    
    g_free(tmp);
    g_free(base);
    return answer;
}

void on_save_image_menuitem_selected(GtkWidget * menuitem, gpointer user_data)
{
    AppUIData *app_ui_data = (AppUIData *)user_data;
    g_assert(NULL != app_ui_data);

    if (app_ui_data->image_url) {
    
        size_t size;
        gpointer data = file_cache_find(app_ui_data->img_cache, 
            app_ui_data->image_url, &size,FALSE);
      
        
        if (data) {
            GtkWidget *fs = NULL;
    
            fs = hildon_file_chooser_dialog_new(
                GTK_WINDOW(app_data->app_ui_data->app),
                GTK_FILE_CHOOSER_ACTION_SAVE);
    
            ossohelp_dialog_help_enable(
                    GTK_DIALOG(fs),
                    RSS_NEWS_READER_HELP_SAVE,
                    app_data->osso);
/*            dialog_enable_help(GTK_DIALOG(fs),
                    RSS_NEWS_READER_HELP_SAVE);*/

            gchar *buf = g_path_get_basename(app_ui_data->image_url);
            
            // Truncate the question mark and the content after it
            // because the filename is wrong (NB# 48234)
            if (strstr(buf, "?") != NULL)
            {
                      
              buf = g_strndup(buf, strlen(buf)-strlen(strstr(buf, "?")));
                      
            }
            
            gtk_file_chooser_set_current_name (GTK_FILE_CHOOSER (fs), buf);
            g_free(buf);
    
            gtk_file_chooser_set_current_folder_uri (GTK_FILE_CHOOSER (fs),
                "file:///home/user/MyDocs/.images");
    
            if (gtk_dialog_run (GTK_DIALOG (fs)) == GTK_RESPONSE_OK) {
                gchar *filename = NULL;
    
                gtk_widget_hide (fs);
                filename = gtk_file_chooser_get_uri(GTK_FILE_CHOOSER (fs));
                
                if (filename) {
                
                    GnomeVFSURI *uri = gnome_vfs_uri_new(filename);
                    GnomeVFSHandle *handle = NULL;
                        
                    ULOG_DEBUG("filename: %s", filename);
                    if (gnome_vfs_uri_exists(uri) &&
                        (!ui_show_replace_file_dialog(filename)))
                    {
                        gnome_vfs_uri_unref(uri);
                        g_free(filename);
                        gtk_widget_destroy (fs);
                        g_free(data);
                    
                        return;
                    }

                    if (gnome_vfs_create_uri(&handle, uri, 
                            GNOME_VFS_OPEN_WRITE |
                            GNOME_VFS_OPEN_TRUNCATE,
                            TRUE,
                            0644) == GNOME_VFS_OK) {
                        gnome_vfs_write(handle, data, size, NULL);
                        gnome_vfs_close(handle);
                        ULOG_DEBUG("saved: %s", filename);
                    }
                            
                    gnome_vfs_uri_unref(uri);
                    g_free(filename);
                }
            }
            
            gtk_widget_destroy (fs);
            g_free(data);
        }
    }
}

void on_clear_image_cache(GtkWidget * menuitem, gpointer user_data)
{
    AppUIData *app_ui_data = (AppUIData *)user_data;
    g_assert(NULL != app_ui_data);

    gboolean answer = FALSE;
    
    HildonNote *note = HILDON_NOTE(hildon_note_new_confirmation_add_buttons(
        GTK_WINDOW(app_data->app_ui_data->app),
        _("rss_nc_clear_image_cache"),
        _("rss_bd_ok"),
        1,
        _("rss_bd_cancel"),
        0,
        NULL));
    if (note) {
        answer = gtk_dialog_run(GTK_DIALOG(note));
        
        if (answer)
            file_cache_clear(app_ui_data->img_cache);
    } else
        ULOG_CRIT("Unable to create HildonNote");
    
    gtk_widget_destroy(GTK_WIDGET(note));
}

void on_menuitem_sort(GtkWidget * button, gpointer user_data)
{
    AppUIData *app_ui_data = (AppUIData *)user_data;
     
    g_assert(app_ui_data);

    HildonSortDialog *sort_dialog = HILDON_SORT_DIALOG(
            hildon_sort_dialog_new(GTK_WINDOW(app_ui_data->app)));
    
    ossohelp_dialog_help_enable(
            GTK_DIALOG(sort_dialog),
            RSS_NEWS_READER_HELP_SORT,
            app_data->osso);

//    dialog_enable_help(GTK_DIALOG(sort_dialog), RSS_NEWS_READER_HELP_SORT);
    
    gint response_id, add_sort_index;
    
    add_sort_index = hildon_sort_dialog_add_sort_key (sort_dialog,
            dgettext(HILDON_LIBS_L10N_PACKAGE, "ckdg_va_sort_date"));
    
    hildon_sort_dialog_add_sort_key (sort_dialog, 
        dgettext(HILDON_LIBS_L10N_PACKAGE, "ckdg_va_sort_name"));
  
    int sort_key = getNumericConfValue(RSS_SETTINGS_SORT_KEY);
    int sort_order = getNumericConfValue(RSS_SETTINGS_SORT_ORDER);

    hildon_sort_dialog_set_sort_key (sort_dialog, sort_key);
    hildon_sort_dialog_set_sort_order (sort_dialog, sort_order);
    
    gtk_widget_show (GTK_WIDGET (sort_dialog));
    
    response_id = gtk_dialog_run (GTK_DIALOG (sort_dialog));
    
    if (response_id == GTK_RESPONSE_OK)
    {
        setNumericConfValue(RSS_SETTINGS_SORT_KEY,
            hildon_sort_dialog_get_sort_key (sort_dialog));
    
        setNumericConfValue(RSS_SETTINGS_SORT_ORDER,
            hildon_sort_dialog_get_sort_order (sort_dialog));
        
	if (displayed_node == searchFeed && searchFeed)
	    ui_itemlist_load((nodePtr) searchFeed);
	else {
//	    nodePtr ptr = (nodePtr) ui_feedlist_get_selected();
            nodePtr ptr;
            
            if (gtk_widget_is_focus(gtk_bin_get_child( GTK_BIN(app_data->app_ui_data->html_scrollpane))))
      
            {
      
              ptr = (nodePtr) displayed_node;
    
            } else {
      
              ptr = (nodePtr) ui_feedlist_get_selected();
      
            }
            
            if (ptr && displayed_node && ((FST_FEED == ptr->type) || (FST_VFOLDER == ptr->type))) {
	        displayed_node = NULL;
    	        ui_feedlist_load_selected(ptr);
            }
	}
    }
    
    gtk_widget_destroy(GTK_WIDGET(sort_dialog));
}

/* Update the toolbar and menu*/
/*void    ui_on_search_finished(void)
{
    
    AppUIData *app_ui_data;
     
    g_assert(app_data != NULL);
    
    app_ui_data = app_data->app_ui_data;

    g_assert(app_ui_data != NULL);
    
    g_assert(NULL != app_ui_data->button_image_search);
				
    gtk_tool_button_set_icon_widget(GTK_TOOL_BUTTON
				    (app_ui_data->search_button),
				    app_ui_data->button_image_search);
    gtk_container_remove(GTK_CONTAINER(app_ui_data->var_field),
			 app_ui_data->progress_bar);
    gtk_container_add(GTK_CONTAINER(app_ui_data->var_field),
		      app_ui_data->search_field);
    gtk_widget_set_sensitive(GTK_WIDGET(app_ui_data->refresh_button),
			     TRUE);
    gtk_widget_set_sensitive(GTK_WIDGET
			     (app_ui_data->menuitem_refresh_feeds), TRUE);
    gtk_widget_set_sensitive(GTK_WIDGET
				(app_ui_data->menuitem_add), TRUE);
    gtk_widget_set_sensitive(GTK_WIDGET(app_ui_data->new_feed_button),
				TRUE);			
    gtk_widget_set_sensitive(GTK_WIDGET(app_ui_data->menuitem_unsubscribe),
				TRUE);
    gtk_widget_set_sensitive(GTK_WIDGET(app_ui_data->unsubscribe_button),
				TRUE);												     
    gtk_widget_set_sensitive(GTK_WIDGET
				(app_ui_data->menuitem_properties), TRUE);
    gtk_widget_set_sensitive(GTK_WIDGET(app_ui_data->feed_properties_button),
				TRUE);			
    gtk_widget_set_sensitive(GTK_WIDGET(app_ui_data->menuitem_manage_folders),
				TRUE);
    gtk_widget_set_sensitive(GTK_WIDGET(app_ui_data->manage_folders_button),
				TRUE);					
    gtk_widget_set_sensitive(GTK_WIDGET
				(app_ui_data->menuitem_feed_directory), TRUE);
    gtk_widget_set_sensitive(GTK_WIDGET
			     (app_ui_data->menuitem_search_results), TRUE);				
							     
    g_assert(SFM_SEARCH == app_ui_data->search_mode
	     || SFM_INTERRUPTED == app_ui_data->search_mode);

    if (SFM_SEARCH == app_ui_data->search_mode) {
    	gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->app), _("rss_ib_search_completed"));
    } else if (SFM_INTERRUPTED == app_ui_data->search_mode) {
	   gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->app), _("rss_ib_search_interrupted"));
    }
    
    app_ui_data->search_mode = SFM_NORMAL;
    ui_mainwindow_empty_search_field();
    //update toolbar to be hidden or not. Because user might have switched 
    //to different Fullscreen mode, and we wanted to show them the progress
    //bar even if it's in toolbar-hidden mode.
    hide_progressbar_if_needed();


}*/
/* handle for users pressing stop while searching
*/
void stop_search()
{
    ULOG_DEBUG("vvvvvvvvvvvvv....stop_search.....vvvvvvvvvvv......");
    //this is enough to stop the vfolder_apply_rules() for one feed
    app_data->app_ui_data->search_mode = SFM_INTERRUPTED;
    //Now we need to stop the html page too. TODO
    gtkhtml_finish_search_feed();
    ui_mainwindow_empty_search_field();
    
    if (SFM_INTERRUPTED == app_data->app_ui_data->search_mode) {
        gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->app), 
            _("rss_ib_search_interrupted"));
    }

    switch_progressbar(PROGRESSBAR_NORMAL_MODE);
//    ui_on_search_finished();
    return;    
}

/* ui_on_search_start: update toolbar and menuitem stuff
*/
/*void ui_on_search_started(void)
{

    AppUIData *app_ui_data;
     
    g_assert(app_data != NULL);
    
    app_ui_data = app_data->app_ui_data;

    g_assert(app_ui_data != NULL);
    
    g_assert(NULL != app_ui_data->search_button);
    g_assert(NULL != app_ui_data->button_image_stop);

    gtk_tool_button_set_icon_widget(GTK_TOOL_BUTTON
                    (app_ui_data->search_button),
                    app_ui_data->button_image_stop);
    gtk_progress_bar_set_text(GTK_PROGRESS_BAR(app_ui_data->progress_bar),
			      _("rss_ia_searching"));
    ULOG_DEBUG("\n\n\n\n\n on_search_entry_activate: Setting fraction\n\n\n\n\n");
    gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR
				  (app_ui_data->progress_bar), 0);
    gtk_container_remove(GTK_CONTAINER(app_ui_data->var_field),
			 app_ui_data->search_field);
    gtk_container_add(GTK_CONTAINER(app_ui_data->var_field),
		      app_ui_data->progress_bar);
    gtk_widget_set_sensitive(GTK_WIDGET(app_ui_data->refresh_button),
			     FALSE);
    gtk_widget_set_sensitive(GTK_WIDGET
			     (app_ui_data->menuitem_refresh_feeds), FALSE);
    gtk_widget_set_sensitive(GTK_WIDGET
			     (app_ui_data->menuitem_search_results), TRUE);
    gtk_widget_set_sensitive(GTK_WIDGET
				(app_ui_data->menuitem_add), FALSE);
    gtk_widget_set_sensitive(GTK_WIDGET(app_ui_data->new_feed_button),
				FALSE);			
    gtk_widget_set_sensitive(GTK_WIDGET(app_ui_data->menuitem_unsubscribe),
				FALSE);
    gtk_widget_set_sensitive(GTK_WIDGET(app_ui_data->unsubscribe_button),
				FALSE);												     
    gtk_widget_set_sensitive(GTK_WIDGET
				(app_ui_data->menuitem_properties), FALSE);
    ///
    //gtk_widget_set_sensitive(GTK_WIDGET
	//			(app_ui_data->menuitem_search_results), FALSE);
    //
    // No adding feed while searching 
    gtk_widget_set_sensitive(GTK_WIDGET
				(app_ui_data->menuitem_feed_directory), FALSE);
    gtk_widget_set_sensitive(GTK_WIDGET(app_ui_data->feed_properties_button),
				FALSE);			
    gtk_widget_set_sensitive(GTK_WIDGET(app_ui_data->menuitem_manage_folders),
				FALSE);
    gtk_widget_set_sensitive(GTK_WIDGET(app_ui_data->manage_folders_button),
				FALSE);												     

    app_ui_data->search_mode = SFM_SEARCH;
    app_ui_data->progressbar_mode = PROGRESSBAR_SEARCH_MODE;
}*/

void on_insensitive_infoprint(GtkWidget *widget, gchar *info)
{
    if (info) {
        gtk_infoprint(GTK_WINDOW(app_data->app_ui_data->app), 
            info);
    }
}

void insensitive_add_press_cb( GtkWidget *widget, gpointer data )
{
  hildon_banner_show_information(widget,NULL,_("rss_ib_unable_add_feed"));
}

/**
 * help_cb:
 * @dialog: Browser dialog
 * @help_id: Help id
 *
 * Callback uses to open the help in help application.
 */
/*void
osso_help_cb (GtkWidget * dialog, const gchar * help_id)
{
    ULOG_DEBUG ("Help ID: %s", help_id);
    osso_return_t retval;

    //    retval = ossohelp_show(osso_browser_ctx, help_id, 1);
    retval = ossohelp_show(app_data->app_ui_data->app, help_id, 1);
    if (retval == OSSO_OK)
        ULOG_DEBUG("Help invocation succesful\n");
    else
        ULOG_DEBUG("Error invoking help application\n");
}

void dialog_enable_help(GtkDialog* dialog, const gchar* help_id)
{
    if (dialog && help_id)
    {
        if (ossohelp_show(app_data->app_ui_data->app, help_id, 
            OSSO_HELP_SHOW_JUSTASK) == OSSO_OK)
        {
            gtk_dialog_help_enable(dialog);
            g_signal_connect(dialog,
                             "help",
                             G_CALLBACK(osso_help_cb),
                             (gpointer) help_id
                            );
        }
    }
}*/
