/**
 * @file cache.h File cache
 *
 * Copyright (c) 2005-06 Nokia Corporation. All rights reserved.
 * Contact: Ouyang Qi <qi.ouyang@nokia.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Initial developer(s): Zsolt Simon
 */

#ifndef _CACHE_H
#define _CACHE_H

#include <glib-object.h>

#define FILE_CACHE_TYPE         (file_cache_get_type ())
#define FILE_CACHE(obj)         (G_TYPE_CHECK_INSTANCE_CAST ((obj), \
                                    FILE_CACHE_TYPE, FileCache))
#define FILE_CACHE_CLASS(c)     (G_TYPE_CHECK_CLASS_CAST ((c), \
                                    FILE_CACHE_TYPE, FileCacheClass))
#define IS_FILE_CACHE(obj)      (G_TYPE_CHECK_INSTANCE_TYPE ((obj), \
                                    FILE_CACHE_TYPE))
#define IS_FILE_CACHE_CLASS(c)  (G_TYPE_CHECK_CLASS_TYPE ((c), FILE_CACHE_TYPE))

#define CACHE_ENTRY(c)          ((CacheEntry *)(c))

typedef struct _FileCache FileCache;
typedef struct _FileCacheClass FileCacheClass;
    
typedef struct _CacheEntry CacheEntry;
    
struct _FileCache {
    GObject parent;
    
    /* private */
    gchar *cache_dir;        //directory for storing the cached files

    size_t current_size;       //current size of the hash
    size_t max_size;           //maximum size of the hash
    
    GHashTable *hash;
    GQueue *queue;
    GMutex *lock;
};

struct _FileCacheClass {
    GObjectClass  parent_class;
};

struct _CacheEntry {
    gchar *url;     //used for the queue
    GList *link;    //used to quickly remove the item from the queue
    gchar *id;
    gchar *filename;
    gint size;
    gint refcount;
    GTimeVal time;
};

GType file_cache_get_type (void);

/**
 * Public functions
 */

/**
 * Creates a new cache directory
 * @param cache_dir the name of the directory or null to use the default ~/.cache
 * @param max_size maximum size of the cache or -1 to use the default 64Mb
 * @return newly created cache
 */
FileCache *file_cache_new(const gchar *cache_dir, size_t max_size);

/**
 * Delete the cache object. Also can be deleted by g_object_unref
 * @param cache pointer to the cache object
 */
void file_cache_destroy(FileCache *cache);

/**
 * Add an URL to the cache. If the URL already existed then it get replaced.
 * @param cache pointer to the cache object
 * @param url pointer to the url
 * @param id additional identifier, later can be removed items from the cache
 *           with this id
 * @param buffer pointer to the data which will be saved to the file
 * @param buffer_size length of the buffer
 * @return true if the item has been successfuly placed to the cache
 *         false if there is not enough disk space or the cache is full
 */
gboolean file_cache_add(FileCache *cache, const gchar *url, const gchar *id,
    const gpointer buffer, const size_t buffer_size);

/**
 * Remove an URL from the cache. The disk information also will be deleted.
 * @param cache pointer to the cache object
 * @param url pointer to the url
 * @return true if the item has been successfuly removed from the cache
 *         false if the url is not found
 */
gboolean file_cache_remove(FileCache *cache, const gchar *url);

/**
 * Remove items from the cache which has the given "id"
 * @param cache pointer to the cache object
 * @param id pointer to a character set
 */
void file_cache_remove_by_id(FileCache *cache, gchar *id);

/**
 * Clear the cache by removing each element
 * @param cache pointer to the cache object
 */
void file_cache_clear(FileCache *cache);

/**
 * Resize the cache. When the new_size is smaller than the actual cache size,
 * elements will be deleted from the cache.
 * @param cache pointer to the cache object
 * @param new_size will be the cache new size
 */
void file_cache_resize(FileCache *cache, size_t new_size);

/**
 * Returns true if there is enough disk space for the new cache
 * @param cache pointer to the cache object
 * @param new_size the desired new cache size
 * @return see above
 */
gboolean file_cache_can_resize(FileCache *cache, size_t new_size);

/**
 * Return the maximum size of the file cache
 * @param cache pointer to the cache object
 * @return the cache maximum size
 */
size_t file_cache_size(FileCache *cache);

/**
 * Find the URL in the cache. If it is found read the data from the disk,
 * and return a pointer to it. If not found, NULL is returned
 * BEWARE: the returned pointer must be freed after use with g_free
 * @param cache pointer to the cache object
 * @param url pointer to the url
 * @param size pointer to a gint, which will contain the size of the buffer.
 *             Can be NULL if the size doesn't matter
 * @return pointer to the data or NULL
 */
gpointer file_cache_find(FileCache *cache, const gchar *url, size_t *size,gboolean increas_statistic);

/**
 * If the url exists in cache, returns true.
 * @param cache pointer to the cache object
 * @param url pointer to the url
 * @return true if url was found
 */
gboolean file_cache_exists(FileCache *cache, const gchar *url);

#endif //_CACHE_H
