/**
    @file applet_cache.c

    Applet cache functionality.

    Copyright (c) 2004-2005 Nokia. All rights reserved.
	
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
    02111-1307 USA
*/


#define DBUS_API_SUBJECT_TO_CHANGE

#include <stdlib.h>
#include <sys/stat.h>
#include <stdio.h>
#include <dirent.h>
#include <unistd.h>
#include <glib.h>
#include <string.h>

#include <osso-log.h>

#include "applet.h"
#include "applet_cache.h"

#include "debug.h"

const glong C_MAGICHEAD = 0xCA204220;
const glong C_VERSION = 2;

/************************************************************************/
/*                        PRIVATE FUNCTIONS                             */
/************************************************************************/

/** Creates a filename of the form: $HOME/.osso_rss_feed_reader/folder/key.extension
  *
  * @param folder the folder part of the filename
  * @param key the key part of the filename
  * @param extension the extension part of the filename
  */
static gchar *common_create_cache_filename(const gchar * folder,
				    const gchar * key,
				    const gchar * extension)
{
    gchar *filename = NULL;
    gchar *lifereaUserPath = NULL;
    
    lifereaUserPath =
	g_strdup_printf("%s" G_DIR_SEPARATOR_S ".osso_rss_feed_reader",
			g_get_home_dir());
    
    filename =
	g_strdup_printf("%s" G_DIR_SEPARATOR_S "%s%s%s%s%s",
			lifereaUserPath,
			(folder != NULL) ? folder : "",
			(folder != NULL) ? G_DIR_SEPARATOR_S : "", key,
			(extension != NULL) ? "." : "",
			(extension != NULL) ? extension : "");

    g_free(lifereaUserPath);			
    return filename;
}

/** Loads a favicon for a feed
  *
  * @param hl a struct where a pointer to the favicon is stored
  * @param id the filename of the cache file. This is identical
  *           to the feed id, so it can be used to find
  *           the favicon
  * @return FALSE in case of errors
  */	
static gboolean favicon_load(HeadLine *hl, gchar *id)
{
    struct stat statinfo;
    gchar *pngfilename = NULL, *xpmfilename = NULL;
    GdkPixbuf *pixbuf = NULL;
    GError *error = NULL;
       
    /* try to load a saved favicon */
    
    id = g_path_get_basename(id);
    pngfilename =
	common_create_cache_filename("cache" G_DIR_SEPARATOR_S "favicons",
				     id, "png");
    xpmfilename =
	common_create_cache_filename("cache" G_DIR_SEPARATOR_S "favicons",
				     id, "xpm");
    g_free(id);

//    ULOG_DEBUG("Trying to load favicon: %s", pngfilename);
				     
    if (0 == stat((const char *) pngfilename, &statinfo)) {
	pixbuf = gdk_pixbuf_new_from_file(pngfilename, &error);
	if (pixbuf != NULL) {
	    if (hl->icon != NULL)
		g_object_unref(hl->icon);
	    hl->icon = gdk_pixbuf_scale_simple(pixbuf,
					       HILDON_ICON_PIXEL_SIZE_SMALL,
					       HILDON_ICON_PIXEL_SIZE_SMALL,
					       GDK_INTERP_BILINEAR);
	    g_object_unref(pixbuf);
	} else {		/* Error */
	    ULOG_DEBUG("Failed to load pixbuf file: %s: %s\n",
		       pngfilename, error->message);
            if(hl->icon != NULL) {
	        g_object_unref(hl->icon);
		hl->icon = NULL;
	    }
	    g_error_free(error);
	}

    } else {
	/* FIXME: remove this migration code when time comes */
	if (g_file_test(xpmfilename, G_FILE_TEST_EXISTS)) {
	    pixbuf = gdk_pixbuf_new_from_file(xpmfilename, &error);
	    if (pixbuf) {
		hl->icon =
		    gdk_pixbuf_scale_simple(pixbuf, 16, 16,
					    GDK_INTERP_BILINEAR);
		gdk_pixbuf_save(pixbuf, pngfilename, "png", NULL, NULL);
		g_object_unref(pixbuf);
	    } else {		/* Error */
		ULOG_DEBUG("Failed to load pixbuf file: %s: %s\n",
			   xpmfilename, error->message);
                if(hl->icon != NULL) {
	            g_object_unref(hl->icon);
		    hl->icon = NULL;
	        }
		g_error_free(error);
	    }
	    unlink(xpmfilename);
	}
    }
    g_free(pngfilename);
    g_free(xpmfilename);

    if(hl->icon != NULL)
        return TRUE;
    else
        return FALSE;
}

/** A function for comparing headlines
  *
  * @param a the first headline to compare
  * @param b the second headline to compare
  * @return the difference between the times
  */
static gint time_order(gconstpointer a, gconstpointer b)
{
    return ((HeadLine *) b)->time - ((HeadLine *) a)->time;
}

/** Reads a string from the cache
  *
  * @param cache the cache file
  * @return the read string
  */
static gchar *cache_read_str(FILE * cache)
{
   int len=0;
    fread(&len, sizeof(len), 1, cache);

    if (feof(cache)) {
	return NULL;
    }

    if (len == 0) {
      return NULL;
    }
    long int currentposition=0,filesize=0;
    currentposition=ftell(cache);
    fseek(cache,0,SEEK_END);
    filesize=ftell(cache);
    if ((len>=0)&&(filesize>=(currentposition+len)))
    {
      fseek(cache,currentposition,SEEK_SET);
      int readlen=0;
      gchar *string = g_malloc(len + 1);
      readlen=fread(string, len, 1, cache);
      string[len] = 0;
      return string;
    }
    else
    { //problem.... it would be too long
/*      fseek(cache,currentposition,SEEK_SET);
      gchar *string = g_malloc(100 + 1);
      while (!feof(cache))
      {
        fread(string, 100, 1, cache);
      }*/
//      fseek(cache,0,SEEK_END);
      return NULL;
    }
}

/** Skip a string
  *
  * @param cache the cache file
  * @return TRUE on success, FALSE on failure
  */
static gboolean cache_skip_str(FILE * cache)
{
    int len=0;
//    ULOG_DEBUG(__FUNCTION__);
    fread(&len, sizeof(len), 1, cache);

    if (feof(cache)) {
	return FALSE;
    }

    if (len == 0) {
	return TRUE;
    }

    return fseek(cache, len, SEEK_CUR) != -1;
}

/** Read an integer from the cache
  *
  * @param cache the cache file
  * @return the read value
  */
static int cache_read_int(FILE * cache)
{
    int data;
    fread(&data, sizeof(data), 1, cache);
    return data;
}

/** Read a long integer from the cache
  *
  * @param cache the cache file
  * @return the read value
  */
static long cache_read_long(FILE * cache)
{
    long data;
    fread(&data, sizeof(data), 1, cache);
    return data;
}

/** Read a boolean from the cache
  *
  * @param cache the cache file
  * @return the read value
  */
static gboolean cache_read_boolean(FILE * cache)
{
    char c;
    fread(&c, sizeof(c), 1, cache);

    if (c == 0)
	return FALSE;

    return TRUE;
}

/** Read a feed item
  *
  * @param cache the cache file
  * @param feed the feed to read from
  */
static HeadLine *cache_read_feed_item(FILE * cache, gchar * feed, gint ver, gint size)
{
    HeadLine *hl = g_new0(HeadLine, 1);
    gint pos = 0;
//    ULOG_DEBUG(__FUNCTION__);

    if (ver > 0)
        pos = cache_read_long(cache);

    hl->icon = NULL;
    
    /* Item title */
    hl->title = cache_read_str(cache);

    if (feof(cache)) {
        g_free(hl->title);
        g_free(hl);
        return NULL;
    }

    /* Item description */
    //cache_skip_str(cache);
    hl->description=cache_read_str(cache);

    /* Item source */
    cache_skip_str(cache);

    /* Item real soure title */
    cache_skip_str(cache);

    /* Item real source url */
    cache_skip_str(cache);

    /* Item ID */
    cache_skip_str(cache);

    /* Item number */
    hl->nr = (guint32) cache_read_long(cache);

    /* Item read status */
    cache_read_boolean(cache);

    /* Item marked status */
    cache_read_boolean(cache);
    
    /*Item delete_count */
//    cache_read_int(cache);

    /* Item posting time */
    hl->time = cache_read_long(cache);
 
    if (ver && pos < size)
        fseek(cache, pos, SEEK_SET);

/*#ifdef VERSION_3   
    count_enclosure=cache_read_int(cache);
    
    while (count_enclosure)
    {
      cache_skip_str(cache);
      cache_skip_str(cache);
			cache_read_int(cache);
      count_enclosure--;
      
    }
#endif*/
    /* Save a reference to feed */
    hl->feed = g_strdup(feed);

    return hl;
}

/** Load a feed
  *
  * @param filename the file to read from
  * @return the list of items
  */
#ifdef VERSION_3
static GList *applet_cache_feed_load(gchar * filename, gchar * feed)
#else
static GList *applet_cache_feed_load(gchar * filename)
#endif
{
    GList *items = NULL;
#ifndef VERSION_3
    gchar *feed = NULL;
#endif
    GdkPixbuf *pixbuf = NULL;    
    gboolean loading_succeeded = TRUE;
    gint ver = 0, pos = 0, tmp, size;

    /*ULOG_DEBUG("%s reading feed %s", __FUNCTION__, filename);*/

    FILE *cache = fopen(filename, "rb");
    if (cache == NULL) {
        return FALSE;
    }

    fseek(cache, 0, SEEK_END);
    size = ftell(cache);
    fseek(cache, 0, SEEK_SET);

    tmp = cache_read_long(cache);
    if (tmp == C_MAGICHEAD) {
      ver = cache_read_long(cache);
      pos = cache_read_long(cache);
    } else {
      fseek(cache, 0, SEEK_SET);
    }

    /* Feed title */
    cache_skip_str(cache);

    /* Feed source */
#ifndef VERSION_3
    feed = cache_read_str(cache);
#else
    cache_skip_str(cache);
#endif

    /* Feed description */
    cache_skip_str(cache);

    /* Feed favicon URL */
    cache_skip_str(cache);

    /* Feed available */
    cache_read_boolean(cache);
    /* Feed discontinued */
    cache_read_boolean(cache);

    /* Feed adding time */
    cache_read_long(cache);

    /* Feed last modified */
    cache_skip_str(cache);

    /* Feed newest post */
    cache_read_long(cache);
    
    /* Feed read time */
    cache_read_long(cache);

    if (ver && pos < size)
      fseek(cache, pos, SEEK_SET);

    while (!feof(cache)) {
        HeadLine *hl = cache_read_feed_item(cache, feed, ver, size);
        if (hl != NULL) {
            if(loading_succeeded) {
                if(pixbuf != NULL) {
                    hl->icon = pixbuf;
                    g_object_ref(hl->icon);
                }
                else
                    loading_succeeded = favicon_load(hl, filename);
                pixbuf = hl->icon;
            }
            items = g_list_append(items, hl);
        }
    }

    fclose(cache);
//    g_free(filename);
#ifndef VERSION_3
    g_free(feed);
#endif
    return items;
}

/** Read the number of unread posts and the time of the last refresh
  *
  * @param lifereaUserPath the user path
  * @param applet the applet structure
  */
/*static void general_load(gchar * lifereaUserPath, AppletInfo * applet)
{
    gchar *filename;

    return;

    g_assert(applet != NULL);
    
    filename =
	g_strdup_printf("%s" G_DIR_SEPARATOR_S "general", lifereaUserPath);
    FILE *cache = fopen(filename, "rb");
    g_free(filename);

    if (NULL == cache) {	
	applet->unread = 0;
	applet->updated = 0;
	return;
    }

    applet->unread = cache_read_long(cache);
    applet->updated = cache_read_long(cache);

    fclose(cache);
}*/

/************************************************************************/
/*                        PUBLIC FUNCTIONS                              */
/************************************************************************/

void test_if_cache_directory_exists()
{
    gchar *lifereaUserPath = NULL;

    lifereaUserPath =
	g_strdup_printf("%s" G_DIR_SEPARATOR_S ".osso_rss_feed_reader",
			g_get_home_dir());
    if (!g_file_test(lifereaUserPath, G_FILE_TEST_IS_DIR)) {
	if (0 != mkdir(lifereaUserPath, S_IRUSR | S_IWUSR | S_IXUSR)) {
	    ULOG_ERR("Cannot create cache directory %s!", lifereaUserPath);
	}
    }
    g_free(lifereaUserPath);
}

/*void do_general_load(AppletInfo *applet)
{
    gchar *lifereaUserPath = NULL;
    
    g_assert(applet != NULL);

    return;
    
    lifereaUserPath =
	g_strdup_printf("%s" G_DIR_SEPARATOR_S ".osso_rss_feed_reader",
			g_get_home_dir());
    if (!g_file_test(lifereaUserPath, G_FILE_TEST_IS_DIR)) {
	if (0 != mkdir(lifereaUserPath, S_IRUSR | S_IWUSR | S_IXUSR)) {
	    ULOG_ERR("Cannot create cache directory %s!", lifereaUserPath);
	}
    }
    general_load(lifereaUserPath, applet);
    g_free(lifereaUserPath);
}*/

GList *feed_load_all(AppletInfo * applet)
{
    GList *items = NULL;
    DIR *cacheDir = NULL;
    struct dirent *cacheFile = NULL;
    FILE *file = NULL;
    long int filesize;

//    applet->updated = 0;

    gchar *cachePath = NULL;
    gchar *feedCachePath = NULL;
    gchar *lifereaUserPath = NULL;
    gchar *feedlistPath = NULL;
    gchar *rss = NULL;
    gchar *slpt = NULL;
    long lpt;

    g_assert(applet != NULL);

    lifereaUserPath =
    g_strdup_printf("%s" G_DIR_SEPARATOR_S ".osso_rss_feed_reader",
        g_get_home_dir());
    if (!g_file_test(lifereaUserPath, G_FILE_TEST_IS_DIR)) {
        if (0 != mkdir(lifereaUserPath, S_IRUSR | S_IWUSR | S_IXUSR)) {
            ULOG_ERR("Cannot create cache directory %s!", lifereaUserPath);
        }
    }

    cachePath =
        g_strdup_printf("%s" G_DIR_SEPARATOR_S "cache", lifereaUserPath);
    if (!g_file_test(cachePath, G_FILE_TEST_IS_DIR)) {
        if (0 != mkdir(cachePath, S_IRUSR | S_IWUSR | S_IXUSR)) {
            ULOG_ERR("Cannot create cache directory %s!", cachePath);
        }
    }

    feedCachePath =
        g_strdup_printf("%s" G_DIR_SEPARATOR_S "feeds", cachePath);
    if (!g_file_test(feedCachePath, G_FILE_TEST_IS_DIR)) {
        if (0 != mkdir(feedCachePath, S_IRUSR | S_IWUSR | S_IXUSR)) {
            ULOG_ERR("Cannot create cache directory %s!", feedCachePath);
        }
    }

    feedlistPath =
        g_strdup_printf("%s" G_DIR_SEPARATOR_S "feedlist.opml", lifereaUserPath);
    file = fopen(feedlistPath, "r");
    if (file) {
        fseek(file,0,SEEK_END);
        filesize=ftell(file);
        fseek(file,0,SEEK_SET);
        if (filesize > 0 && filesize < 1024 * 1024) {
            rss = g_malloc(filesize + 1);
            fread(rss, filesize, 1, file);
            rss[filesize] = 0;
        }
        fclose(file);
    }
    g_free(feedlistPath);
    if (!rss)
        rss = g_strdup("");

//    general_load(lifereaUserPath, applet);

    cacheDir = opendir(feedCachePath);
    if (NULL == cacheDir)
    {
        g_free(cachePath);
        g_free(feedCachePath);
        g_free(lifereaUserPath);
        g_free(rss);
        closedir(cacheDir);
        return NULL;
    }
    TDB("=========================");
    applet->updated=0;
    while ((cacheFile = readdir(cacheDir)) != NULL) {
        gchar *path = g_strdup_printf("%s" G_DIR_SEPARATOR_S "%s",
                           feedCachePath,
                           cacheFile->d_name);
        if (g_file_test(path, G_FILE_TEST_IS_REGULAR)) {
            if ((slpt=strstr(rss, cacheFile->d_name))) {
                if ((slpt=strstr(slpt, "lastPollTime")) && (slpt = strchr(slpt, '\"'))) {
                    lpt = atol(slpt+1);
                    
                    if ((applet->updated < (time_t) lpt) )
                        applet->updated = (time_t) lpt;
                }
                    
                TDB("Parsing feed: %s", cacheFile->d_name);
#ifdef VERSION_3
                items = g_list_concat(items, applet_cache_feed_load(path, cacheFile->d_name));
#else
                items = g_list_concat(items, applet_cache_feed_load(path));
#endif
            } else {
                unlink(path);
                gchar *imgpath = g_strdup_printf("%s" G_DIR_SEPARATOR_S "favicons"
                                   G_DIR_SEPARATOR_S "%s.png",
                                   cachePath,
                                   cacheFile->d_name);
                TDB("Removing: %s", path);
                TDB("Removing image: %s", imgpath);
                unlink(imgpath);
                g_free(imgpath);
            }
        }
        g_free(path);
    }
    TDB("--------------------");
    g_free(cachePath);
    g_free(feedCachePath);
    g_free(lifereaUserPath);
    g_free(rss);
    closedir( cacheDir );

    items = g_list_sort(items, (GCompareFunc) time_order);

    return items;
}
