/**
    @file applet.c

    Copyright (c) 2004-2005 Nokia. All rights reserved.
	
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
    02111-1307 USA
*/


#define DBUS_API_SUBJECT_TO_CHANGE

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif


#include "applet.h"
#include "applet_cache.h"

/* Locale includes */
#include <libintl.h>
#include <locale.h>

#include <glib/gi18n-lib.h>

#include <libxml/xmlerror.h>
#include <libxml/uri.h>
#include <libxml/parser.h>
#include <libxml/entities.h>
#include <libxml/HTMLparser.h>


#include <osso-log.h>

#include <osso-ic-dbus.h>
#include <osso-ic-gconf.h>

#include <libosso.h>

#include <hildon-home-plugin/hildon-home-plugin-interface.h>
#include <hildon-widgets/hildon-defines.h>
#include <hildon-widgets/gtk-infoprint.h>
#include <hildon-widgets/hildon-banner.h>
#include <hildon-widgets/hildon-note.h>

#include <gdk/gdkkeysyms.h>

#include <stdlib.h>
#include <string.h>

#include <langinfo.h>

#include <gdk/gdk.h>
/*This was 7 in SP*/
#define MAX_HEADLINES 9
#define REFRESH_BUTTON_SIZE 60
#define REQUESTED_WIDTH 390

#define TZ_SIZE 64 // maximum size of a timezone string
#define LOCALTIME_FILE	"/etc/localtime"
#define ZONEINFO_DIR	"/usr/share/zoneinfo/"

#define NEWLINE_POSITION 36

#ifdef VERSION_3
#include <settings.h>
#endif
#include "debug.h"

static void realize(GtkWidget *widget, gpointer user_data);

extern int setenv (__const char *__name, __const char *__value, int __replace)
     __THROW;

extern int unsetenv (__const char *__name) __THROW;

/* Needed global for internet connectivity callback */
AppletInfo *rss_appl_inf;
gint htimeout;

static gboolean have_last_selection;
static GtkTreeIter last_selection;
static gint last_pos = -1;

#ifdef VERSION_3
GtkWindow *window = NULL;
#endif

static void _applet_set_information(AppletInfo * applet);
static gboolean _applet_timeout_cb(gpointer user_data);
static gboolean _applet_do_refresh(AppletInfo * applet, gboolean timed);
static gboolean _applet_dbus_display(AppletInfo * applet,
				     gchar * feed, guint32 item);
static void Unselect_all(void);


/************************************************************************/
/*                        PRIVATE FUNCTIONS                             */
/************************************************************************/

gboolean dialog_key_press(GtkWidget *widget, GdkEventKey *event, gpointer data)
{
  (void)data;

  if (event->state & (GDK_CONTROL_MASK |
      GDK_SHIFT_MASK |
      GDK_MOD1_MASK |
      GDK_MOD3_MASK |
      GDK_MOD4_MASK |
      GDK_MOD5_MASK)) {
    return FALSE;
      }

      switch (event->keyval)
      {
        case GDK_Escape:
        return TRUE;
          break;
        default:
          break;
      }

      return FALSE;
}

void ui_show_save_nomem_dialog(AppletInfo *applet)
{
  HildonNote *note = HILDON_NOTE(hildon_note_new_information(
      gtk_widget_get_ancestor(GTK_WIDGET(applet->appletframe), GTK_TYPE_WINDOW),
  dgettext("hildon-common-strings","sfil_ni_not_enough_memory")
                                                            ));
  hildon_note_set_button_text(note, dgettext("hildon-common-strings","sfil_ni_not_enough_memory_ok"));
  g_signal_connect(G_OBJECT(note), "key_press_event",
                   G_CALLBACK(dialog_key_press), NULL);

  gtk_dialog_run(GTK_DIALOG(note));
  gtk_widget_destroy(GTK_WIDGET(note));

//  hildon_appview_set_title(app_ui_data->main_view, "");
}

void ui_show_save_nodevicemem_dialog(AppletInfo *applet)
{
  HildonNote *note = HILDON_NOTE(hildon_note_new_information(
      gtk_widget_get_ancestor(GTK_WIDGET(applet->appletframe), GTK_TYPE_WINDOW),
  dgettext("ke-recv","cerm_device_memory_full")
                                                            ));
  hildon_note_set_button_text(note, dgettext("hildon-common-strings","sfil_ni_not_enough_memory_ok"));
  g_signal_connect(G_OBJECT(note), "key_press_event",
                   G_CALLBACK(dialog_key_press), NULL);

  gtk_dialog_run(GTK_DIALOG(note));
  gtk_widget_destroy(GTK_WIDGET(note));

//  hildon_appview_set_title(app_ui_data->main_view, "");
}

/* Depending on the state of hw, do something */
void hw_event_handler(osso_hw_state_t * state, gpointer data)
{
  (void) data;

  g_assert(state != NULL);
    
  ULOG_DEBUG("Hardware event received: %d %d %d %d", state->shutdown_ind,
             state->memory_low_ind, state->save_unsaved_data_ind, state->system_inactivity_ind);
    
  if (state->shutdown_ind) {
  
  }
  if (state->memory_low_ind) {
    /* Memory low */
    
    rss_appl_inf->startrss = TRUE;
    
  } else {
    
    rss_appl_inf->startrss = FALSE;
    
  }
  if (state->save_unsaved_data_ind) {
  }
  if (state->system_inactivity_ind) {
    /* Minimum activity */
  }
}

gboolean Scroll_to_correct(gpointer data)
{
  GtkRequisition sreq;
  GtkAllocation nreq;
  gint pos = (gint)data;

  ULOG_DEBUG("Curently selected: %d", pos);
  gtk_widget_size_request(rss_appl_inf->rss_news, &sreq);
  nreq = rss_appl_inf->scrolledwindow->allocation;
  if (pos >= 0 && sreq.height > nreq.height)
  {
    gtk_scrolled_window_set_policy(rss_appl_inf->scrolledwindow,GTK_POLICY_NEVER,GTK_POLICY_ALWAYS);
    gtk_layout_set_size(rss_appl_inf->layout, nreq.width-30, sreq.height);
    gtk_widget_set_size_request(rss_appl_inf->rss_news, nreq.width-30,-1);

    GtkAdjustment *adjust=gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(rss_appl_inf->scrolledwindow));
    if (pos == 1 && rss_appl_inf->displayed_lines == 3)
      gtk_adjustment_set_value(adjust,sreq.height-nreq.height-rss_appl_inf->line_height);
    else if (pos >= rss_appl_inf->displayed_lines/2)
      gtk_adjustment_set_value(adjust,sreq.height-nreq.height);
    else
      gtk_adjustment_set_value(adjust,0.0);
    g_object_set(G_OBJECT(adjust), "step-increment", (gdouble)rss_appl_inf->line_height, NULL);
    gtk_scrolled_window_set_vadjustment(GTK_SCROLLED_WINDOW(rss_appl_inf->scrolledwindow), adjust);
  }
  else
  {
    gtk_scrolled_window_set_policy(rss_appl_inf->scrolledwindow,GTK_POLICY_NEVER,GTK_POLICY_NEVER);
    gtk_layout_set_size(rss_appl_inf->layout, nreq.width, nreq.height);
    gtk_widget_set_size_request(rss_appl_inf->rss_news, nreq.width, nreq.height);
  }
  return FALSE;
}

static void
render_feed (GtkTreeModel *model,
             GtkTreeIter *iter,
             gboolean selected)
{
  if (selected)
  {
    gchar *markup = NULL, *title = NULL, *desc = NULL;

    gtk_tree_model_get(model, iter, TITLE_COLUMN, &title,
                  DESCRIPTION_COLUMN, &desc, -1);
    if (title) {
        
      gchar *end=NULL;
      if (!g_utf8_validate(desc,-1,&end))
      {
        end[0]='\0';
        
      }
      if (desc[strlen(desc)-1]=='\n')
      {
        desc[strlen(desc)-1]='\0';
      }
        gchar *tmp = g_markup_printf_escaped ("\n<small>%s</small>",
                                    desc);
        markup = g_strconcat(title, tmp, NULL);
        g_free(tmp);
    }

    gtk_tree_store_set(model, iter, RENDER_COLUMN, markup ? markup : "", -1);
    ULOG_DEBUG("render feed: %s", markup);

    g_free (markup);
    g_free (title);
    g_free (desc);
  }
  else
  {
    gchar *title = NULL;

    gtk_tree_model_get(model, iter, TITLE_COLUMN, &title,
                  -1);
    gtk_tree_store_set(model, iter, RENDER_COLUMN, title, -1);
    ULOG_DEBUG("render feed: %s", title);
    
    g_free(title);
  }
}

gboolean 
expand_collapse_selection(gboolean collapse_only)
{
  GtkTreeIter iter;
  GtkTreeModel *model = NULL;
  GtkTreeSelection *selection;
  gint pos;

  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(rss_appl_inf->rss_news));
  if (gtk_tree_selection_get_selected(selection, &model, &iter) && !collapse_only) {
    gchar *listpath = gtk_tree_model_get_string_from_iter(model, &iter);
    pos = atoi(listpath);
    g_free(listpath);
  } else
    pos = -1;

  ULOG_DEBUG("Last selection: %d, %d", pos, last_pos);
  if (pos == last_pos)
    return FALSE;

  if (have_last_selection)
    render_feed(model, &last_selection, FALSE);

  last_pos = pos;
  have_last_selection = pos != -1;

  if (have_last_selection) {
    GtkAllocation nreq = rss_appl_inf->scrolledwindow->allocation;
    last_selection = iter;

    gtk_widget_set_size_request(rss_appl_inf->rss_news, nreq.width-30,-1);
    render_feed(model, &last_selection, TRUE);
  }

  g_idle_add(Scroll_to_correct, (gpointer)last_pos);

  return TRUE;
}

void update_font_width(AppletInfo *rss_appl_inf)
{
  PangoContext *context;
  PangoFontMetrics *metrics;
  PangoFontDescription *font_desc, *font2;

  font_desc = pango_font_description_copy (rss_appl_inf->rss_news->style->font_desc);
  g_object_get(rss_appl_inf->textRenderer, "font-desc", &font2, NULL);
  pango_font_description_merge (font_desc, font2, TRUE);

  context = gtk_widget_get_pango_context (rss_appl_inf->rss_news);

  metrics = pango_context_get_metrics (context,
                                       font_desc,
                                       pango_context_get_language (context));
  rss_appl_inf->line_height = (pango_font_metrics_get_ascent(metrics) +
          pango_font_metrics_get_descent(metrics)) / PANGO_SCALE + 2;
  pango_font_metrics_unref (metrics);

  ULOG_DEBUG("Character height: %d", rss_appl_inf->line_height);
  pango_font_description_set_size (font_desc,
                                   pango_font_description_get_size (font_desc) / (float)1.2);

  metrics = pango_context_get_metrics (context,
                                       font_desc,
                                       pango_context_get_language (context));
  rss_appl_inf->avg_char_width = pango_font_metrics_get_approximate_char_width(metrics) /
                                    PANGO_SCALE + 2;
  ULOG_DEBUG("Avarage character width: %d", rss_appl_inf->avg_char_width);
  pango_font_metrics_unref (metrics);
  pango_font_description_free (font_desc);
  pango_font_description_free (font2);
}

void Launch_item(GtkTreeModel *model,gpointer user_data)
{
  GtkTreeIter iter;
  GtkTreeSelection *selection=NULL;
  gchar *feed=NULL;
  guint32 itemnr=0;
  ULOG_DEBUG(__FUNCTION__);
  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(rss_appl_inf->rss_news));
  if (gtk_tree_selection_get_selected(selection, &model, &iter))
  {
    gtk_tree_model_get(model, &iter, FEED_COLUMN, &feed,
		   			ITEMNR_COLUMN, &itemnr, -1);
    ULOG_DEBUG("feed: %s, number: %d",feed,itemnr);
	  if (_applet_dbus_display(user_data, feed, itemnr) == FALSE) {
			ULOG_ERR("Sending DBus message to display news item failed");
	  }
    g_free(feed);
  }
  else
  {
    ULOG_DEBUG("None is selected");
  }
}



void Selecting(void)
{
//  ULOG_DEBUG(__FUNCTION__);
  return;
  //gtk_widget_activate(rss_appl_inf->rss_news);
  //gtk_widget_grab_focus(rss_appl_inf->rss_news);
  gtk_widget_activate(rss_appl_inf->rss_refresh);
  gtk_widget_grab_focus(rss_appl_inf->rss_refresh);
}

/** Sets a timezone
  *
  * @param tz timezone
  * @return zero on success, -1 if there was insufficient space in the environment
  */
static int set_timezone(const char *tz) {
  int i = setenv("TZ", tz, 1);
  tzset();
  return i;
}

/**
  * Restore a timezone stored by _save_timezone.
  *
  * @param tz timezone
  */
static void restore_timezone(const char *tz)
{
  tz ? set_timezone(tz) : unsetenv("TZ");
}

/**
  *Store timezone to a string.
  *
  * @param tz timezone
  */
static void save_timezone(char *tz)
{
    char *t;
    g_assert(tz);
    if ((t = getenv("TZ"))) {
      g_assert(TZ_SIZE > strlen(t));
      strncpy(tz, t, TZ_SIZE);
    }
}

/** Gets local time
  *
  * @param t time
  * @param zone timezone
  * @param ret the localtime
  * @return localtime
  */
static struct tm *get_local_tm(time_t t, const gchar *zone, struct tm *ret) {
  gchar temp_tz[TZ_SIZE];

  save_timezone(temp_tz);
  set_timezone(zone);
  localtime_r(&t, ret);
  restore_timezone(temp_tz);

  return ret; 
}

/** Converts time_to to struct tm
  *
  * @param t time
  * @param zone timezone
  * @param ret the localtime
  * @return localtime
  */
static struct tm *clock_epoch_to_local(time_t t, const gchar *zone, struct tm *ret) {
  g_assert(zone);
  g_assert(ret);

  get_local_tm(t, zone, ret);
  return ret;
}

/** Gets the local timezone
  *
  * @return the local timezone */
static gchar* clock_get_local_timezone(void)
{
	gchar** parts;
	gchar* link = NULL;
	gchar* identifier = NULL;
	
	/* Test that localtime-file exists and it is link */
	if (!g_file_test(LOCALTIME_FILE, G_FILE_TEST_IS_SYMLINK |
		                         G_FILE_TEST_EXISTS))
	{
		return NULL;
	}
	
	/* Read link dest from localtime-file (assume it's utf-8) */
	link = g_file_read_link(LOCALTIME_FILE, NULL);
	
	/* Remove zoneinfo_dir from dest */
	parts = g_strsplit(link, ZONEINFO_DIR, -1);
	g_free(link);
	identifier = g_strjoinv("", parts);
	g_strfreev(parts);
	
	/* Return the result */
	return identifier; // newly allocated	
}

/** Gets the current time
  *
  * @return the current time in seconds after the epoch
  */
static time_t clock_get_time(void) {
  return time(NULL);
}

/************************************************************************/
/*                        PUBLIC FUNCTIONS                              */
/************************************************************************/

gchar *remove_newlines_and_extra_spaces(gchar * string)
{
//  ULOG_DEBUG(__FUNCTION__);
    if (string == NULL)
	return NULL;

    gchar *p = string, *q = string;
    gchar first_character = 1;
    gboolean last_was_space = FALSE;
    
    while (*p != 0) {
	gchar c = *p++;
	if(c != '\x0A' && c != '\x0D') {
	    if (((c != ' ') && (c != '\t')) || (last_was_space == FALSE)) {
		*q++ = c;
	    }

	    if ((c == ' ') || (c == '\t'))
		last_was_space = TRUE;
	    else
		last_was_space = FALSE;
	} else {
//             if(c == '\x0D') {
//                 ULOG_DEBUG("Adding newline character");
//                 *q++ = '\n';
//                 last_was_space = TRUE;
//             }
	    if(!last_was_space && !first_character) {
	        *q++ = ' ';
	        last_was_space = TRUE;
            }
	}
	first_character = 0;
    }
    *q = 0;
    
    return string;
}


void unhtmlizeHandleSAXStartElement(void *user_data, const xmlChar * name,
				    const xmlChar ** atts)
{
    result_buffer *buffer = (result_buffer *) user_data;
    gint old_length = 0;
//  ULOG_DEBUG(__FUNCTION__);
    if ((strcmp(name, "br") == 0) || (strcmp(name, "BR") == 0) ||
	(strcmp(name, "p") == 0) || (strcmp(name, "P") == 0)
	) {
	old_length = buffer->length;
	buffer->length++;
	buffer->data = g_renew(gchar, buffer->data, buffer->length + 1);
	strncpy(buffer->data + old_length, "\n", 1);
	buffer->data[buffer->length] = 0;
    }
}


void unhtmlizeHandleCharacters(void *user_data, const xmlChar * string,
			       int length)
{
  
    result_buffer *buffer = (result_buffer *) user_data;
    gint old_length = 0;
//    ULOG_DEBUG(__FUNCTION__);
    old_length = buffer->length;
    buffer->length += length;
    buffer->data = g_renew(gchar, buffer->data, buffer->length + 1);
    strncpy(buffer->data + old_length, (gchar *) string, length);
    buffer->data[buffer->length] = 0;

}

/* Conversion function which should be applied to all read XML strings, 
   to ensure proper UTF8. This is because we use libxml2 in recovery
   mode which can produce invalid UTF-8. 
   
   The valid or a corrected string is returned. The original XML 
   string is modified */
gchar *utf8_fix(xmlChar * string)
{
    const gchar *invalid_offset = NULL;
  
//  ULOG_DEBUG(__FUNCTION__);

    if (NULL == string)
	return NULL;

    if (!g_utf8_validate(string, -1, &invalid_offset)) {
	/* if we have an invalid string we try to shorten
	   it until it is valid UTF-8 */
	/*debug0(DEBUG_PARSING, "parser delivered invalid UTF-8!");
	debug1(DEBUG_PARSING, "	>>>%s<<<\n", string);
	debug1(DEBUG_PARSING, "first invalid char is: >>>%s<<<\n",
	       invalid_offset);
	debug0(DEBUG_PARSING, "removing invalid bytes");*/

	do {
	    memmove((void *) invalid_offset, invalid_offset + 1,
		    strlen(invalid_offset + 1) + 1);
	} while (!g_utf8_validate(string, -1, &invalid_offset));

	/*debug0(DEBUG_PARSING, "result is:\n");
	debug1(DEBUG_PARSING, "	>>>%s<<<\n", string);*/
    }

    return string;
}


/* Converts a UTF-8 strings containing any HTML stuff to 
   a string without any entities or tags containing all
   text nodes of the given HTML string. The original 
   string will be freed. 
   
   NOTE TO SELF: this is used from various different places.
   Check what needs to be done to have external links processed
   in this and converted to GtkEventBoxes in GtkTextBuffer.
   
*/
gchar *unhtmlize(gchar * string)
{
    htmlSAXHandlerPtr sax_p = NULL;
    htmlParserCtxtPtr ctxt = NULL;
    gchar *result = NULL;
    result_buffer *buffer = NULL;
//    ULOG_DEBUG(__FUNCTION__);

    if (NULL == string)
	return NULL;

    string = utf8_fix(string);

    /* only do something if there are any entities or tags */
    if (NULL == (strpbrk(string, "&<>")))
	return string;

    buffer = g_new0(result_buffer, 1);
    sax_p = g_new0(htmlSAXHandler, 1);
    sax_p->characters = unhtmlizeHandleCharacters;
    sax_p->startElement = unhtmlizeHandleSAXStartElement;

    /* in older versions htmlSAXParseDoc was used which caused
       strange crashes when freeing the parser context... */

    ctxt =
	htmlCreatePushParserCtxt(sax_p, buffer, string, strlen(string), "",
				 XML_CHAR_ENCODING_UTF8);
    htmlParseChunk(ctxt, string, 0, 1);
    htmlFreeParserCtxt(ctxt);
    result = buffer->data;
    g_free(buffer);
    g_free(sax_p);

    if (result == NULL || !g_utf8_strlen(result, -1)) {
	/* Something went wrong in the parsing.
	 * Use original string instead */
	g_free(result);
	return string;
    } else {
	g_free(string);
	return result;
    }
}

/** Gets the current time considering the local timezone
  *
  * @return the current time in seconds after the epoch in the local timezone
  */
time_t get_current_time()
{
#ifndef VERSION_3
    struct tm htmp;
    time_t htime = clock_get_time();
    
    gchar *zone = clock_get_local_timezone();
//    ULOG_DEBUG(__FUNCTION__);
    
    clock_epoch_to_local(htime, zone, &htmp);
    
    g_free(zone);
    
    return mktime(&htmp);
#else
    return time(NULL);
#endif    
}

time_t get_local_time_from_time_t(time_t time)
{
    struct tm htmp;
//    ULOG_DEBUG(__FUNCTION__);
    
    ULOG_DEBUG("Get system time: %ld", time);
#ifndef VERSION_3    
    gchar *zone = clock_get_local_timezone();
    
    clock_epoch_to_local(time, zone, &htmp);
    
    time_t localized_time = mktime(&htmp);
    
    ULOG_DEBUG("Get localized time: %ld", localized_time);
    
    g_free(zone);
    
    return localized_time;
#else
    return time;
#endif
}

/** Gets localized time
  *
  * @param time the time for which the time string is produced
  */
gchar *get_localized_time(time_t time)
{
    struct tm *tm = NULL;
    
    gchar *time_string = NULL;
    gchar *time_format = g_strdup(nl_langinfo(T_FMT));
    gchar ts_with_secs[TIMESTRLEN + 1] = {""};
    gchar ts_without_secs[TIMESTRLEN + 1] = {""};
        
    int i = 0, j = 0;
    int nr_colon = 0;
    gboolean ignore = FALSE;
    gboolean add_space = FALSE;
//    ULOG_DEBUG(__FUNCTION__);
    tm = localtime(&time);
    
    strftime(ts_with_secs, TIMESTRLEN, time_format, tm);

    for(i = 0;i<TIMESTRLEN + 1; i++)
    {
        if(ts_with_secs[i] == ':')
	    nr_colon++;
	
	if(nr_colon == 2)
	{
	    ignore = TRUE;
	    nr_colon = 0;
	}
	  
	if(ignore && ts_with_secs[i] != ':' && (ts_with_secs[i] == ' ' || ts_with_secs[i] == '\0' || ts_with_secs[i] > 57)) 
        {
            if(ts_with_secs[i] > 57)
                add_space = TRUE;
	    ignore = FALSE;
        }

	if(!ignore)
        {
            if(add_space)
            {
                ts_without_secs[j++] = ' ';
                add_space = FALSE;
            }
	    ts_without_secs[j++] = ts_with_secs[i];
        }
	
	if(ts_with_secs[i] == '\0')
	    break;
    }
    
    g_free(time_format);
    
    time_string = g_strdup(ts_without_secs);
    time_string = g_strstrip(time_string);
    
    return time_string;
}

/** Gets localized date
  *
  * @param time the time for which the date string is produced
  * @param years set to TRUE to display years
  */
gchar *get_localized_date(time_t time, gboolean years)
{
    struct tm *tm = NULL;

    gchar *date_format = g_strdup(nl_langinfo(D_FMT));
    gchar *df_no_years = NULL;
    gchar **df_split = NULL;
    gchar date_string[TIMESTRLEN + 1];
//    ULOG_DEBUG(__FUNCTION__);
    tm = localtime(&time);
    
    strftime(date_string, TIMESTRLEN, date_format, tm);
    
    if(!years)
    {
        df_split = g_strsplit_set(date_string, " /-.", -1);
	
	if(g_strv_length(df_split) >= 2)
	{
	    gchar *separator = NULL;
	    
	    if(g_strstr_len(date_string, TIMESTRLEN + 1, " ") != NULL)
	        separator = g_strdup(" ");
            else if(g_strstr_len(date_string, TIMESTRLEN + 1, "/") != NULL)
	        separator = g_strdup("/");
	    else if(g_strstr_len(date_string, TIMESTRLEN + 1, "-") != NULL)
	        separator = g_strdup("-");
	    else if(g_strstr_len(date_string, TIMESTRLEN + 1, ".") != NULL)
                separator = g_strdup(".");
		
	    if(separator != NULL)
	    {
                df_no_years = g_strconcat(df_split[0], separator, df_split[1], NULL);
		g_free(separator);
	    }
	    else
	        df_no_years = g_strdup("");
	}
	else
	    df_no_years = g_strdup("");
	
	g_strfreev(df_split);
	g_free(date_format);
	
	return df_no_years;
    }
    
    g_free(date_format);

    return g_strdup(date_string);
}

#ifndef WITHOUT_CONNECTIVITY

/** Gets a string value from gconf
  *
  * @param valuename the name of the value to get
  */
static gchar *getStringConfValue(gchar * valuename)
{
    GConfValue *value = NULL;
    gchar *result = NULL;
//    ULOG_DEBUG(__FUNCTION__);

    g_assert(valuename != NULL);
    g_assert(rss_appl_inf->client != NULL);

    value = gconf_client_get_without_default(rss_appl_inf->client, valuename, NULL);
    if (NULL == value) {
	result = g_strdup("");
    } else {
	result = (gchar *) g_strdup(gconf_value_get_string(value));
	gconf_value_free(value);
    }

    return result;
}

#endif

/* Applet GConf monitoring code begins here */

/** Set update timeout
  *
  * @param applet the applet structure
  * @return the set timeout
  */
static int _applet_set_timeout(AppletInfo * applet, gboolean constant_interval)
{
#ifndef VERSION_3
    int interval = 0;
    time_t last_refresh;
    time_t current_time;
        
//    ULOG_DEBUG(__FUNCTION__);

    g_assert(applet != NULL);

    /* Remove old timeout function if set */
    if (applet->timeout > 0) {
	ULOG_INFO("Removing old RSS Refresh timeout");
	g_source_remove(applet->timeout);
	applet->timeout = 0;
    }
    ULOG_DEBUG("Setting timeout applet->update %d applet->online %d",applet->update,applet->online);
    ULOG_DEBUG("Interval %d",applet->interval);
    /* Set up new interval */
    if (TRUE == applet->update) {
        current_time = get_current_time();
#ifndef VERSION_3
	last_refresh = current_time - get_local_time_from_time_t(applet->updated);
#else
  last_refresh = current_time - applet->updated;
#endif
	
	interval = applet->interval * 60;
	
	if(!constant_interval)
	    interval -= last_refresh;
	
	if(interval < 0)
	    interval = 0;
	    
	applet->timeout =
	    g_timeout_add(interval * 1000, _applet_timeout_cb,
			  applet);
	ULOG_INFO("RSS Refresh timeout installed as source %d with timeout",
		  applet->timeout);
    }
    return applet->timeout;
#else
    return 0;
#endif
}

gboolean call_realize(gpointer data);

gboolean call_realize(gpointer data)
{
  TDB("Calling realize from idle\n");
  rss_appl_inf->prev_height++;
  realize(NULL,NULL);
  return FALSE;
}

/** A callback function for handling IAP events
  *
  * @param event the incoming IAP event
  * @param arg arguments for the event
  */
void _applet_iap_cb(struct iap_event_t *event, void *arg)
{
    ULOG_DEBUG(__FUNCTION__);

    g_assert(NULL != event);
    g_assert(NULL != rss_appl_inf);

    if (OSSO_IAP_CONNECTED == event->type) {
	rss_appl_inf->online = TRUE;

	ULOG_DEBUG("iap cb before if");
	if (rss_appl_inf->update) {
	    /*if (rss_appl_inf->updated + (MIN_INTERVAL * 60) < time(NULL)) {
		if (_applet_do_refresh(rss_appl_inf, TRUE) == FALSE)
		    ULOG_ERR
			("Refresh DBus message failed when going online");
	    }*/
            ULOG_DEBUG("iap cb in if");
	    _applet_set_timeout(rss_appl_inf, FALSE);
	}
    } else if (OSSO_IAP_DISCONNECTED == event->type) {
	rss_appl_inf->online = FALSE;
	if (rss_appl_inf->timeout > 0) {
	    g_source_remove(rss_appl_inf->timeout);
	    rss_appl_inf->timeout = 0;
	}
    }
}

void _applet_internet_connection_changed_cb(GConfClient * client,
					    guint cnxn_id, GConfEntry * entry,
					    gpointer user_data)
{
    (void) client;
    (void) cnxn_id;
    (void) entry;
    (void) user_data;
   ULOG_DEBUG(__FUNCTION__);
#ifdef WITHOUT_CONNECTIVITY
    if(rss_appl_inf->update)
        _applet_set_timeout(rss_appl_inf, FALSE);
#else    
    if(*getStringConfValue(ICD_GCONF_CURRENT) &&
       rss_appl_inf->update)
        _applet_set_timeout(rss_appl_inf, FALSE);
#endif
	
}						

/** A callback function that is called after the time set
  * for the automatic refresh has been expired
  *
  * @param user_data the applet structure
  * @return always TRUE;
  */
static gboolean _applet_timeout_cb(gpointer user_data)
{
    AppletInfo *applet = (AppletInfo *) user_data;

    //ULOG_DEBUG(__FUNCTION__);

    g_assert(user_data != NULL);
    
#ifdef WITHOUT_CONNECTIVITY
    if (_applet_do_refresh(applet, TRUE) == FALSE)
	    ULOG_ERR("Timer triggered refresh DBus message failed");
#else    
//    printf("Timeout: check connectivity...\n");
    gconf_client_clear_cache(applet->client);
    if(*getStringConfValue(ICD_GCONF_CURRENT))
//        printf("Timeout: Do refresh...\n");
        if (_applet_do_refresh(applet, TRUE) == FALSE)
	    ULOG_ERR("Timer triggered refresh DBus message failed");
#endif
    	    
    return FALSE;
}

/** Read update related data from gconf
  *
  * @param client gconf client
  * @param cnxn_id not used
  * @param entry not used
  * @param user_data the applet structure
  */
static void _applet_gconf_read_cb(GConfClient * client, guint cnxn_id,
				  GConfEntry * entry, gpointer user_data)
{
    AppletInfo *applet;
     
//    ULOG_DEBUG(__FUNCTION__);
    
    g_assert(client != NULL);
    g_assert(user_data != NULL);
    
    applet = (AppletInfo *) user_data;

    //ULOG_DEBUG(__FUNCTION__);
    
    applet->update =
	gconf_client_get_bool(client, RSS_SETTINGS_AUTOMATIC_UPDATES,
			      NULL);
    applet->interval =
	gconf_client_get_int(client,
			     RSS_SETTINGS_AUTOMATIC_UPDATES_INTERVAL,
			     NULL);

     if(applet->interval < MIN_INTERVAL)
        applet->interval = MIN_INTERVAL;			     			     
	
    /* There is no sensible way to detect or handle error here */
    _applet_set_timeout(applet, FALSE);
}

/** Initialize gconf for reading the update related data
  *
  * @param applet the applet structure
  */
static void _applet_gconf_init(AppletInfo * applet)
{
    GError *error = NULL;

    ULOG_DEBUG(__FUNCTION__);

    GConfClient *client = gconf_client_get_default();
    g_assert(NULL != client);
    g_assert(NULL != applet);
    
    applet->client = client;
    applet->timeout = 0;

    gconf_client_add_dir(client, RSS_SETTINGS, GCONF_CLIENT_PRELOAD_NONE,
			 &error);
    if (NULL != error) {
	ULOG_ERR
	    ("Could not add %s directory to GConf clients watch list, error: %s",
	     RSS_SETTINGS, error->message);
        g_free(error);
    }	     

    if (NULL == error) {
	/* Should we save notification numbers and unregister later? */
	applet->gconf_notify_id1=gconf_client_notify_add(client, RSS_SETTINGS_AUTOMATIC_UPDATES,
				_applet_gconf_read_cb, applet, NULL,
				&error);
	if (NULL != error) {
	    ULOG_ERR("Could not set notify for GCong key %s, error: %s",
		     RSS_SETTINGS_AUTOMATIC_UPDATES, error->message);
            g_free(error);		     
        }		     
	
	applet->gconf_notify_id2=gconf_client_notify_add(client,
				RSS_SETTINGS_AUTOMATIC_UPDATES_INTERVAL,
				_applet_gconf_read_cb, applet, NULL,
				&error);
	if (NULL != error) {
	    ULOG_ERR("Could not set notify for GCong key %s, error: %s",
		     RSS_SETTINGS_AUTOMATIC_UPDATES_INTERVAL,
		     error->message);
            g_free(error);		     
	}
		   
	gconf_client_add_dir(client, ICD_GCONF_CURRENT,
			 GCONF_CLIENT_PRELOAD_NONE, &error);
	
        if(NULL == error) {			 
            applet->gconf_notify_id3=gconf_client_notify_add(client, ICD_GCONF_CURRENT,
			        _applet_internet_connection_changed_cb, NULL, NULL,
			        &error);			     
				
            if(NULL != error) {
	        ULOG_DEBUG("Could not set notify for internet connection changes: %s",
	                   error->message);
	        g_free(error);
	    }			    				
	} else {
	    ULOG_DEBUG("Could not set directory for internet connection changes: %s",
	               error->message);
	    g_free(error);
	}
    }

    if (NULL != error)
	g_free(error);

    /* Try to install timeout even if we got errors setting up change listeners */
    _applet_gconf_read_cb(client, 0, NULL, applet);
}

/** Deinitialize gconf
  *
  * @param applet the applet structure
  */
static void _applet_gconf_deinit(AppletInfo * applet)
{
    ULOG_DEBUG(__FUNCTION__);
    GError *error = NULL;

    g_assert(applet != NULL);
    if (applet->gconf_notify_id1>0)
    {
    gconf_client_notify_remove(applet->client,applet->gconf_notify_id1);
    }
    gconf_client_remove_dir(applet->client, RSS_SETTINGS, 
			 &error);
    if (applet->gconf_notify_id2>0)
    {
    gconf_client_notify_remove(applet->client,applet->gconf_notify_id2);
    }
    if (applet->gconf_notify_id3>0)
    {
    gconf_client_notify_remove(applet->client,applet->gconf_notify_id3);
    }
    gconf_client_remove_dir(applet->client, ICD_GCONF_CURRENT,
			  &error);
    g_free(error);
    g_object_unref(applet->client);
}

/** Shows or hides the empty label or the news. This is a rather
  * ugly way to do this, a better way would be to use just gtk_widget_show
  * and gtk_widget_hide and not to add and remove stuff from containers.
  * Unfortunately this is not possible for the following reason: the
  * _applet_set_headlines function gets called when the applet
  * is started. Now if we are using this gtk_widget_show/hide scheme
  * and there are news available the empty label would be hidden.
  * Now after the initialization ALL the widgets in the applet
  * are shown, also the empty label we just hid..that's why we have
  * to play around with adding and removing from containers
  *
  * @param show if the empty label should be shown or not
  */
static void _applet_show_empty_label(AppletInfo * applet, gboolean show)
{
  ULOG_DEBUG(__FUNCTION__);
  if (show)
  {
/*    if (applet->scrolledwindow->parent == applet->applet_vbox)
      gtk_container_remove(GTK_CONTAINER(applet->applet_vbox),
                         applet->scrolledwindow);*/
    if (applet->empty_label->parent != applet->applet_vbox)
    {
      gtk_box_pack_start(GTK_BOX(applet->applet_vbox), applet->empty_label, FALSE, FALSE, 0);
      gtk_widget_show(applet->empty_label);
    }
    GtkTreeSelection *rss_select =
             gtk_tree_view_get_selection(GTK_TREE_VIEW(applet->rss_news));
    gtk_tree_selection_set_mode(rss_select, GTK_SELECTION_NONE);
//    gtk_widget_hide(applet->scrolledwindow);
  }
  else
  {
    if (applet->empty_label->parent == applet->applet_vbox)
    {
      g_idle_add(call_realize,NULL);
      gtk_container_remove(GTK_CONTAINER(applet->applet_vbox),
                       applet->empty_label);
    }
    GtkTreeSelection *rss_select =
             gtk_tree_view_get_selection(GTK_TREE_VIEW(applet->rss_news));
    gtk_tree_selection_set_mode(rss_select, GTK_SELECTION_SINGLE);
    
//    gtk_widget_show(applet->scrolledwindow);
/*    if (applet->scrolledwindow->parent != applet->applet_vbox)
    {
      gtk_box_pack_end(GTK_BOX(applet->applet_vbox), applet->scrolledwindow, TRUE, TRUE, 0);
      gtk_widget_show(applet->scrolledwindow);
    }*/
  }
  Selecting();
}

gchar *put_newlines(gchar *description)
{
	gint i=rss_appl_inf->chars_per_line-1,currentpos=0;
	gchar *string_return=NULL;
	ULOG_DEBUG(__FUNCTION__);
  if (description==NULL)
  {
    return NULL;
  }
  ULOG_DEBUG("The string: %s", description);
	if (strlen(description)>rss_appl_inf->chars_per_line)
	{
    /*We have to search the first space, looking backward*/
		while(i&&(description[i]!=' '))
		{
			i--;
		}
		if (i>0)
		{
			description[i]='\n';
      currentpos=i+1;
      //string_return=description;
			//return description;
		}
		else
		{/*there was no space inside. we wrap the text simply*/
			gchar *temp=NULL;
			
			temp=g_strdup(description+rss_appl_inf->chars_per_line);
			description[rss_appl_inf->chars_per_line]='\n';
			string_return=g_strjoin(NULL,description,temp,NULL);
			g_free(temp);
			g_free(description);
      currentpos=rss_appl_inf->chars_per_line+1;
      description=string_return;
		}
		if (strlen(description+currentpos)>rss_appl_inf->chars_per_line)
			{/*We creat the third line*/
				gchar *temp2=NULL;
				int j=currentpos+rss_appl_inf->chars_per_line;
				/*We have to search the first space, looking backward*/
				while((j>=currentpos)&&(description[j]!=' '))
				{
					j--;
				}
				if (j!=currentpos)
				{/*If there was, then we change that to '\n'*/
					description[j]='\n';
          currentpos = j;
				}
				else
				{/*otherwise, we wrap the text simply*/
					temp2=g_strdup(description+currentpos+rss_appl_inf->chars_per_line);
					description[currentpos+rss_appl_inf->chars_per_line]='\n';
					string_return=g_strjoin(NULL,description,temp2,NULL);
					g_free(description);
					g_free(temp2);
          currentpos+=rss_appl_inf->chars_per_line+1;
					description=string_return;
				}
        if (strlen(description+currentpos)>rss_appl_inf->chars_per_line + 10)
          description[currentpos+rss_appl_inf->chars_per_line+10] = 0;
			}
		
	}
	else
	{
		return description;
	}
  return description;
}

/** Set the list of feed headlines for the applet
  *
  * @param applet the applet structure
  */
static void _applet_set_headlines(AppletInfo * applet)
{
    GList *items = NULL;
    GList *list_iter = NULL;
    GtkTreeIter iter,child;
    GdkPixbuf *pixbuf = NULL;
    GdkPixbuf *default_pixbuf = NULL;
    GtkIconTheme *icon_theme = gtk_icon_theme_get_default();
    HeadLine *headline = NULL;
    gchar *item_title = NULL;
    gchar *item_desciption = NULL;
    gchar *tmp;
    int num = 0;

    ULOG_DEBUG(__FUNCTION__);

    g_assert(NULL != applet);
    g_assert(NULL != applet->headlines);

    items = feed_load_all(applet);

    default_pixbuf = gtk_icon_theme_load_icon(icon_theme, RSS_ICON_ITEM,
				      HILDON_ICON_PIXEL_SIZE_SMALL, 0,
				      NULL);

    have_last_selection = FALSE;
    gtk_tree_store_clear(applet->headlines);
//    printf("Have last selection1: %d\n", have_last_selection);
    ULOG_DEBUG("222");
    
    if(items != NULL)
    {
        if(g_list_length(items) == 0)
            _applet_show_empty_label(applet, TRUE);
        else
           _applet_show_empty_label(applet, FALSE);
    }
    else
        _applet_show_empty_label(applet, TRUE);
    	    
    list_iter = g_list_first(items);
    while (NULL != list_iter) {
        headline = (HeadLine *) list_iter->data;
        if (headline)
        {
//            if (num++ < MAX_HEADLINES) {
          num++;
            if ((num <= applet->displayed_lines)&&(num <= MAX_HEADLINES)) {
                if (headline->title)
                {
                    item_title = g_markup_escape_text(headline->title, -1);
                    item_title = g_strstrip(item_title);
                    item_title = g_strdelimit(item_title, "\r\n\t", ' ');
                }
                else
                {
                    item_title=g_strdup(" ");
                }
            
                /*item_desciption = g_markup_escape_text(headline->description, -1);
                item_desciption = g_strstrip(item_desciption);
                item_desciption = g_strdelimit(item_desciption, "\r\n\t", ' ');*/
                //item_desciption=unhtmlize(item_desciption);
                if (headline->description)
                {
                    item_desciption=unhtmlize(headline->description);
                    headline->description = NULL;
                    item_desciption=remove_newlines_and_extra_spaces(item_desciption);
                    item_desciption=put_newlines(item_desciption);
/*                    item_desciption=g_markup_printf_escaped("<small>%s</small>",tmp);
                    g_free(tmp);*/
                }
                else
                {
                    item_desciption=g_strdup("");
                }

                if(headline->icon == NULL)
                    pixbuf = default_pixbuf;
                else
                    pixbuf = headline->icon;
            
                /*gtk_list_store_append(applet->headlines, &iter);
                gtk_list_store_set(applet->headlines, &iter,
                           ICON_COLUMN, pixbuf,
                           TITLE_COLUMN, item_title,
                           FEED_COLUMN, headline->feed,
                           ITEMNR_COLUMN, headline->nr, -1);*/
                    /*Adding the title*/
                gtk_tree_store_append(applet->headlines, &iter,NULL);
                gtk_tree_store_set(applet->headlines, &iter,
                           ICON_COLUMN, pixbuf,
                           RENDER_COLUMN, item_title,
                           TITLE_COLUMN, item_title,
                           DESCRIPTION_COLUMN, item_desciption,
                           FEED_COLUMN, headline->feed,
                           ITEMNR_COLUMN, headline->nr, -1);
                /*Adding the description as a child*/
/*                gtk_tree_store_append(applet->headlines, &child,&iter);
                gtk_tree_store_set(applet->headlines, &child,
                           //ICON_COLUMN, pixbuf,
                           TITLE_COLUMN, item_desciption,
                           FEED_COLUMN, headline->feed,
                           ITEMNR_COLUMN, headline->nr, -1);*/
                g_free(item_title);
                g_free(item_desciption);
            }
            g_free(headline->feed);
            g_free(headline->title);
            g_free(headline->description);
            if(headline->icon != NULL)
                g_object_unref(headline->icon);
            g_free(headline);
        }
        list_iter = g_list_next(list_iter);
    }

//    printf("Have last selection2: %d\n", have_last_selection);
    g_list_free(items);

    if (default_pixbuf != NULL)
	g_object_unref(default_pixbuf);

    _applet_set_information(applet);
    Scroll_to_correct((gpointer)(-1));

//  Unselect_all();
}

/** This is called after the refresh is completed in the main application
  *
  * @param applet the applet structure
  */
static void _applet_refresh_completed(AppletInfo * applet)
{
    ULOG_DEBUG(__FUNCTION__);

    g_assert(applet != NULL);
    g_assert(applet->rss_image != NULL);

    /*gtk_image_set_from_icon_name(GTK_IMAGE(applet->rss_image),
				 "qgn_toolb_gene_refresh",
				 GTK_ICON_SIZE_LARGE_TOOLBAR);*/
	gtk_container_remove(GTK_CONTAINER(applet->rss_refresh), GTK_WIDGET(gtk_bin_get_child(GTK_BIN(applet->rss_refresh))));
	gtk_container_add(GTK_CONTAINER(applet->rss_refresh), GTK_WIDGET(applet->rss_image));	
  _applet_set_headlines(applet);
#ifndef VERSION_3
    _applet_set_timeout(rss_appl_inf, TRUE);
#else
    _applet_set_information(applet);
#endif
  
  Selecting();
}

/** Creates an empty file
  *
  * @param file the filename to create
  */
static int _applet_create_temporary_file(gchar *file)
{
    gchar *filename = NULL;
    FILE * tmpf;
    //ULOG_DEBUG(__FUNCTION__);
    filename = g_strdup_printf("%s" G_DIR_SEPARATOR_S
	    		       ".osso_rss_feed_reader"
			       G_DIR_SEPARATOR_S "%s",
			       g_get_home_dir(), file);
    test_if_cache_directory_exists();			       
    tmpf = fopen(filename, "wb");
    if (tmpf == NULL) {
	    g_free(filename);
	    return 0;
    }
    fclose(tmpf);
    g_free(filename);
    return 1;
}

/** The D-BUS messages are handled here
  *
  * @param connection the D-BUS connection
  * @param signal the incoming D-BUS message
  * @param user_data the applet structure
  * @return returns DBUS_HANDLER_RESULT_HANDLED on success and
  *         DBUS_HANDLER_RESULT_NOT_YET_HANDLED on failure
  */
static DBusHandlerResult _applet_handle_signal_cb(DBusConnection *
						  connection,
						  DBusMessage * signal,
						  gpointer user_data)
{
    DBusHandlerResult msg_handled = DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
    AppletInfo *applet = (AppletInfo *) user_data;

//    ULOG_DEBUG(__FUNCTION__);
    ULOG_DEBUG("DBus message: %s\n%s", dbus_message_get_member(signal),
      dbus_message_get_interface(signal));

    g_assert(NULL != connection);
    g_assert(NULL != signal);

    if (applet == NULL) {
	/* Why would this happen? */
	ULOG_DEBUG("Applet data is NULL");
	return msg_handled;
    }

    if (dbus_message_is_method_call(signal, "com.nokia.statusbar",
        "statusbar_insensitive"))
    {
      ULOG_DEBUG(">>>>>Eureka");
    } else
    if (dbus_message_is_signal(signal, NEWS_INTERFACE_NAME,
			       OSSO_RSS_FEED_READER_REFRESH_FINISHED_SIGNAL))
    {
	ULOG_DEBUG("signal: refresh_finished");
	_applet_refresh_completed(applet);
	msg_handled = DBUS_HANDLER_RESULT_HANDLED;
  applet->refreshing = FALSE;
        //gtk_banner_close(NULL);
#ifndef VERSION_3
      if (rss_appl_inf->refresh_banner)
      {
        gtk_widget_destroy(rss_appl_inf->refresh_banner);
        rss_appl_inf->refresh_banner=NULL;
        rss_appl_inf->refresh_banner_displayed=FALSE;
      }
#endif
    } else
	if (dbus_message_is_signal(signal, NEWS_INTERFACE_NAME,
				   OSSO_RSS_FEED_READER_REFRESH_INTERRUPTED_SIGNAL))
    {
	ULOG_DEBUG("signal: refresh_interrupted");
        _applet_refresh_completed(applet);
#ifndef VERSION_3
      if (rss_appl_inf->refresh_banner)
      {
        gtk_widget_destroy(rss_appl_inf->refresh_banner);
        rss_appl_inf->refresh_banner=NULL;
        rss_appl_inf->refresh_banner_displayed=FALSE;
      }
        //if(!applet->backgrounded)
          //hildon_banner_show_information(applet->appl_frame,NULL,_("home_ib_refresh_interrupted"));
        hildon_banner_show_information(gtk_widget_get_ancestor(applet->appl_frame,GTK_TYPE_WINDOW)
         ,NULL,_("home_ib_refresh_interrupted"));
	    //gtk_infoprint(NULL, _("home_ib_refresh_interrupted"));
#endif
	msg_handled = DBUS_HANDLER_RESULT_HANDLED;
        applet->refreshing = FALSE;
        //gtk_banner_close(NULL);
        
    } else
	if (dbus_message_is_signal(signal, NEWS_INTERFACE_NAME,
				   OSSO_RSS_FEED_READER_REFRESH_PREVENTED_SIGNAL))
    {
#ifndef VERSION_3
        //if(!applet->backgrounded)
          //hildon_banner_show_information(applet->appl_frame,NULL,_("rss_ni_error_refreshing_all_feeds"));
        hildon_banner_show_information(gtk_widget_get_ancestor(applet->appl_frame,GTK_TYPE_WINDOW)
        ,NULL,_("rss_ni_error_refreshing_all_feeds"));
          //  gtk_infoprint(NULL, _("rss_ni_error_refreshing_all_feeds"));
#endif
    ULOG_DEBUG("signal: refresh_prevented");
	g_assert(NULL != applet->rss_image);
	/*gtk_image_set_from_icon_name(GTK_IMAGE(applet->rss_image),
				     "qgn_toolb_gene_refresh",
				     GTK_ICON_SIZE_LARGE_TOOLBAR);*/
	gtk_container_remove(GTK_CONTAINER(applet->rss_refresh), GTK_WIDGET(gtk_bin_get_child(GTK_BIN(applet->rss_refresh))));
	gtk_container_add(GTK_CONTAINER(applet->rss_refresh), GTK_WIDGET(applet->rss_image));	
	msg_handled = DBUS_HANDLER_RESULT_HANDLED;
        applet->refreshing = FALSE;
#ifndef VERSION_3
        if (rss_appl_inf->refresh_banner)
        {
        gtk_widget_destroy(rss_appl_inf->refresh_banner);
        //gtk_banner_close(NULL);
          rss_appl_inf->refresh_banner=NULL;
          rss_appl_inf->refresh_banner_displayed=FALSE;
        }
#endif
        _applet_refresh_completed(applet);
    } else
	if (dbus_message_is_signal(signal, NEWS_INTERFACE_NAME,
				   OSSO_RSS_FEED_READER_FOLDER_READ_SIGNAL))
    {
	ULOG_DEBUG("signal: folder_read");
#ifdef VERSION_3
    if (!applet->refreshing)
#endif
	_applet_set_information(applet);
	/*_applet_refresh_completed(applet);*/
	msg_handled = DBUS_HANDLER_RESULT_HANDLED;
    }
#ifdef VERSION_3
    else
	if (dbus_message_is_signal(signal, NEWS_INTERFACE_NAME,
				   OSSO_RSS_FEED_READER_CONNECTING_STARTED))
    {
	gtk_container_remove(GTK_CONTAINER(rss_appl_inf->rss_refresh), GTK_WIDGET(gtk_bin_get_child(GTK_BIN(rss_appl_inf->rss_refresh))));
	gtk_container_add(GTK_CONTAINER(rss_appl_inf->rss_refresh), GTK_WIDGET(rss_appl_inf->rss_image_stop));
	gtk_widget_show_all(rss_appl_inf->rss_refresh);
	msg_handled = DBUS_HANDLER_RESULT_HANDLED;
	applet->refreshing = TRUE;
    gtk_label_set_markup(GTK_LABEL(applet->rss_updated),
        _("home_ia_connecting"));
	gtk_widget_queue_draw(GTK_WIDGET(applet->rss_updated));
    }
#endif
    else
	if (dbus_message_is_signal(signal, NEWS_INTERFACE_NAME,
				   OSSO_RSS_FEED_READER_REFRESH_STARTED))
    {
#ifndef VERSION_3
    if (!rss_appl_inf->refresh_banner_displayed)
       {
          rss_appl_inf->refresh_banner=hildon_banner_show_animation(gtk_widget_get_ancestor(applet->appl_frame,GTK_TYPE_WINDOW)
        ,NULL, _("home_ia_refreshing"));
          rss_appl_inf->refresh_banner_displayed=TRUE;
        }
#else
    gtk_label_set_markup(GTK_LABEL(applet->rss_updated),
        _("home_ia_receiving"));
	gtk_widget_queue_draw(GTK_WIDGET(applet->rss_updated));
#endif
	gtk_container_remove(GTK_CONTAINER(rss_appl_inf->rss_refresh), GTK_WIDGET(gtk_bin_get_child(GTK_BIN(rss_appl_inf->rss_refresh))));
	gtk_container_add(GTK_CONTAINER(rss_appl_inf->rss_refresh), GTK_WIDGET(rss_appl_inf->rss_image_stop));
	gtk_widget_show_all(rss_appl_inf->rss_refresh);
	msg_handled = DBUS_HANDLER_RESULT_HANDLED;
	applet->refreshing = TRUE;
    }
    Selecting();
    return msg_handled;
}


gint dbus_req_handler(const gchar * interface,
		      const gchar * method,
		      GArray * arguments,
		      gpointer data, osso_rpc_t * retval)
{
    (void) interface;
    return TRUE;
}


/** Gets a D-BUS connection
  * 
  * @param applet the applet structure
  * @param signal_handler the handler for the D-BUS connection
  * @return TRUE on success, FALSE on failure
  */
static gboolean _applet_dbus_connect(AppletInfo * applet,
				     DBusHandleMessageFunction
				     signal_handler)
{
    DBusError err;
    ULOG_DEBUG(__FUNCTION__);
	g_assert(applet);
	applet->osso=NULL;
	applet->connection=NULL;
    dbus_error_init(&err);

    if (applet != NULL && applet->connection == NULL) {
        applet->osso = osso_initialize("osso_rss_feed_reader_applet", PACKAGE_VERSION, FALSE, NULL);
				g_assert(applet->osso!=NULL);
        applet->connection = (DBusConnection *)osso_get_dbus_connection(applet->osso);
				g_assert(applet->connection!=NULL);
// 	applet->connection = dbus_bus_get(DBUS_BUS_ACTIVATION, &err);
                                
        /* Set handling changes in HW states. Note: not tested */
        osso_return_t ret = osso_hw_set_event_cb(applet->osso,
                 NULL, hw_event_handler, applet);

        if (ret != OSSO_OK) {
          ULOG_CRIT("Could not set callback for HW monitoring");
        }
                                

	if (applet->connection == NULL || dbus_error_is_set(&err)) {
	    dbus_error_free(&err);
	    return FALSE;
	}

	gchar *rule = NULL;

	rule = g_strdup_printf(APPLET_DBUS_SIGNAL_LISTENER_FMT,
			       NEWS_SERVICE_NAME, NEWS_INTERFACE_NAME);

	dbus_bus_add_match(applet->connection, rule, &err);

	if (dbus_error_is_set(&err)) {
	    ULOG_ERR("Failed to set up DBus rule %s, error: %s", rule,
		     err.message);
	    dbus_error_free(&err);
	    g_free(rule);
	    return FALSE;
	} else {
	    dbus_connection_add_filter(applet->connection,
				       signal_handler, applet, NULL);
	}
	g_free(rule);

	/* Finally, register the connection handling to main loop */
	dbus_connection_setup_with_g_main(applet->connection, NULL);
    }

    return TRUE;
}

/** The applet is refreshed by calling this function
  *
  * @param applet the applet structure
  * @param timed if the refresh should be timed or not
  * @return TRUE on success, FALSE on failure
  */
static gboolean _applet_dbus_refresh(AppletInfo * applet, gboolean timed)
{
    ULOG_DEBUG(__FUNCTION__);
	gboolean timing=timed;

    if (applet != NULL && applet->connection != NULL) {
      
      if (rss_appl_inf->startrss == TRUE)
      {
          
        ui_show_save_nomem_dialog(rss_appl_inf->appletframe);
            
        return TRUE;
          
      }
      
	DBusMessage *message = NULL;
	message =
	    dbus_message_new_method_call
	    (NEWS_SERVICE_NAME,
	     NEWS_SERVICE_OBJECT,
	     NEWS_INTERFACE_NAME, OSSO_RSS_FEED_READER_REFRESH_SIGNAL);
	if (message == NULL) {
	    ULOG_ERR("Creating DBus refresh call failed");
	    return FALSE;
	}
	dbus_message_set_auto_start(message, TRUE);
	if (!dbus_message_append_args(message,
				      DBUS_TYPE_BOOLEAN, &timing,
				      DBUS_TYPE_INVALID)) {
	    ULOG_ERR("Could not append args to message");
	}
	if (!_applet_create_temporary_file("background")) {
		ULOG_ERR("Could not create temporary file \"background\"");
    if(!applet->backgrounded)
      //hildon_banner_show_information(applet->appl_frame,NULL,_("rss_ni_error_refreshing_all_feeds"));
           ui_show_save_nodevicemem_dialog(rss_appl_inf->appletframe);
		  //   gtk_infoprint(NULL, _("rss_ni_error_refreshing_all_feeds"));
		dbus_message_unref(message);
		return TRUE;
	}
	if (!(dbus_connection_send(applet->connection, message, NULL))) {
	    ULOG_ERR("Sending DBus refresh message failed");
	    dbus_message_unref(message);
	    return FALSE;
	}
	if (message != NULL) {
	    dbus_message_unref(message);
	}
    } else {
   ULOG_ERR
	    ("Refresh message failed because applet is not properly initialized");
    }
    return TRUE;
}

/** Calls the main application to display a selected feed
  *
  * @param applet the applet structure
  * @param feed the feed to display
  * @param item the item to display
  * @return TRUE on success, FALSE on failure
  */
static gboolean _applet_dbus_display(AppletInfo * applet,
				     gchar * feed, guint32 item)
{
    ULOG_DEBUG(__FUNCTION__);

    g_assert(feed != NULL);
    
    if (applet != NULL && applet->connection != NULL) {
        if (rss_appl_inf->startrss == TRUE)
        {
          
            ui_show_save_nomem_dialog(rss_appl_inf->appletframe);
            
            return TRUE;
          
        }

        if (OSSO_OK != osso_rpc_async_run(applet->osso,
            NEWS_SERVICE_NAME,
            NEWS_SERVICE_OBJECT,
            NEWS_INTERFACE_NAME, 
            OSSO_RSS_FEED_READER_DISPLAY_SIGNAL,
            NULL, NULL,
            DBUS_TYPE_STRING, feed,
            DBUS_TYPE_UINT32, item,
            DBUS_TYPE_INVALID, NULL))
              ULOG_DEBUG("Could not send the display message");

    } else {
	ULOG_ERR
	    ("Display message failed because applet is not properly initialized");
    }

    return TRUE;
}

/** Call the main application to do a refresh on all the feeds
  *
  * @param applet the applet structure
  * @param timed if the refresh should be timed or not
  * @return TRUE on success, FALSE on failure
  */
static gboolean _applet_do_refresh(AppletInfo * applet, gboolean timed)
{
    ULOG_DEBUG(__FUNCTION__);

    g_assert(NULL != applet);

    return _applet_dbus_refresh(applet, timed);
}

/** Set the information for the applet, such as the time of the last refresh
  *
  * @param applet the applet structure
  */
static void _applet_set_information(AppletInfo * applet)
{
    gchar *str = NULL,*tmp = NULL, *timestr = NULL;
    time_t time_now;
    struct tm *timestruct = NULL;
    gint yday = 0, year = 0;
    gboolean updated = FALSE;
    time_t local_time;
    
    ULOG_DEBUG(__FUNCTION__);

    g_assert(NULL != applet);
    /* We don't need this in the latest specification*/
    /*str = g_strdup_printf("%s", _("home_ia_n_unread_posts"));
    str2 = g_strdup_printf(str, applet->unread);
    gtk_label_set_markup(GTK_LABEL(applet->rss_unread), str2);
    g_free(str);
    g_free(str2);*/
    
//    do_general_load(applet);

    time_now = get_current_time();
    timestruct = localtime(&time_now);
    yday = timestruct->tm_yday;
    year = timestruct->tm_year;
    
    updated = !(applet->updated == 0);
#ifndef VERSION_3    
    local_time = get_local_time_from_time_t(applet->updated);
#else
    local_time =applet->updated;
#endif
    
    timestruct = localtime(&local_time);
    
    if (timestruct->tm_yday == yday && timestruct->tm_year == year) {
	tmp = get_localized_time(local_time);
    } else {
        tmp = get_localized_date(local_time, TRUE);
    }
    timestr = g_locale_to_utf8(tmp, -1, NULL, NULL, NULL);
    g_free(tmp);
    
    str = g_strdup_printf("%s %s", _("home_ia_last_updated"),
			  updated ? timestr : _("home_ia_last_updated_never"));
    gtk_label_set_markup(GTK_LABEL(applet->rss_updated), str);
    hildon_gtk_widget_set_logical_font(
        applet->rss_updated,
        "osso-TitleFont" );
    hildon_gtk_widget_set_logical_color(
        applet->rss_updated,
        GTK_RC_FG,
        GTK_STATE_NORMAL,
        "DialogTitleTextColor" );
    g_free(str);
    g_free(timestr);
}

/** A function to handle cycling up through the tree view with the keyboard
  * @param tr the tree view to cycle
  * @return TRUE on success, FALSE on failure
  */
static gboolean _applet_tree_view_cycle_up(GtkWidget * tr)
{
    GtkTreeIter iter;
    GtkTreePath *path = NULL;
    GtkTreeModel *model = NULL;
    GtkTreeView *treeview = NULL;
	GtkTreeSelection *selection;
	
	ULOG_DEBUG(__FUNCTION__);

    if (tr == NULL)
	return FALSE;

    //treeview = GTK_TREE_VIEW(tr);
    treeview = GTK_TREE_VIEW(rss_appl_inf->rss_news);
    model = gtk_tree_view_get_model(treeview);
	selection = gtk_tree_view_get_selection(treeview);
    if (!gtk_tree_selection_get_selected(selection,&model,&iter))
    {//none selected
      gtk_tree_model_get_iter_first(model,&iter);
    }
    path=gtk_tree_model_get_path(model,&iter);
    if (!gtk_tree_path_prev(path))
    {//don't need to loop
      /*GtkTreeIter temp;
      temp=iter;
      while (gtk_tree_model_iter_next(model,&temp))
      {
        iter=temp;
      }
      if (path)
      {
        gtk_tree_path_free(path);
      }
      path=gtk_tree_model_get_path(model,&iter);*/
    }
    gtk_tree_selection_select_path(selection, path);
    expand_collapse_selection(FALSE);
//    Scroll_to_correct();
/*    gtk_tree_view_sele
    Collapse_old_row();
    Expand_row(path);
  Selecting();*/
	return TRUE;

}

/** A function to handle cycling down through the tree view with the keyboard
  * @param tr the tree view to cycle
  * @return TRUE on success, FALSE on failure
  */
static gboolean _applet_tree_view_cycle_down(GtkWidget * tr)
{
    //int toplevel_tree_size = 0;
    //int *indices = NULL;
    GtkTreeIter iter;
    //GtkTreePath *path = NULL;
    GtkTreeModel *model = NULL;
    GtkTreeView *treeview = NULL;
	GtkTreeSelection *selection;
  gboolean move_needed=TRUE;

	ULOG_DEBUG(__FUNCTION__);
    if (tr == NULL)
	return FALSE;
	
    //treeview = GTK_TREE_VIEW(tr);
    treeview = GTK_TREE_VIEW(rss_appl_inf->rss_news);
    model = gtk_tree_view_get_model(treeview);
	  selection = gtk_tree_view_get_selection(treeview);
    if (!gtk_tree_selection_get_selected(selection,&model,&iter))
    {
      gtk_tree_model_get_iter_first(model,&iter);
      move_needed=FALSE;
    }
    //path=gtk_tree_model_get_path(model,&iter);
    if (move_needed==TRUE)
    {
      if (!gtk_tree_model_iter_next(model,&iter))
      {// don't need to loop
        //gtk_tree_model_get_iter_first(model,&iter);
      }
    }
    gtk_tree_selection_select_iter(selection, &iter);
    expand_collapse_selection(FALSE);
//    Scroll_to_correct();
/*    Collapse_old_row();
    Expand_row(gtk_tree_model_get_path(model,&iter));
  Selecting();*/
	return TRUE;

}


/** A callback function that is called after the refresh button has been clicked
  *
  * @param widget not used
  * @param user_data the applet structure
  */
static void _applet_refresh_button_cb(GtkWidget * widget, gpointer user_data)
{
    AppletInfo *applet = (AppletInfo *) user_data;

    ULOG_DEBUG(__FUNCTION__);

    g_assert(user_data != NULL);

    if (_applet_do_refresh(applet, FALSE) == FALSE) {
	ULOG_ERR("User requested refresh DBus message failed");
    }
}

/** A callback function to listen to key strokes to enable cycling in the tree view
  *
  * @param widget the widget to listen to
  * @param event information about the key that has been hit
  * @param user_data the tree view to cycle
  * @return TRUE if the signal should not be propagated further, FALSE otherwise
  */
static gboolean _applet_tree_view_key_release(GtkWidget * widget,
					      GdkEventKey * event,
					      gpointer user_data)
{
	ULOG_DEBUG(__FUNCTION__);
     if (event == NULL)
        return FALSE;

    if (event->state & (GDK_CONTROL_MASK |
			GDK_SHIFT_MASK |
			GDK_MOD1_MASK |
			GDK_MOD3_MASK | GDK_MOD4_MASK | GDK_MOD5_MASK)) {
	return TRUE;
    }

    if (user_data == NULL)
	return FALSE;

	
    if (GDK_KEY_PRESS == event->type) {
	switch (event->keyval) {
	case GDK_Up:
        return _applet_tree_view_cycle_up(rss_appl_inf->rss_news);
//        return FALSE;
//	    return _applet_tree_view_cycle_up(GTK_WIDGET(user_data));
	    break;

	case GDK_Down:
        return _applet_tree_view_cycle_down(rss_appl_inf->rss_news);
//        return FALSE;
//	    return _applet_tree_view_cycle_down(GTK_WIDGET(user_data));
	    break;

	case GDK_Left:
	case GDK_Right:
	    return TRUE;
  case GDK_Return:
  {
    ULOG_DEBUG("Return released");
    Launch_item( gtk_tree_view_get_model(GTK_TREE_VIEW(rss_appl_inf->rss_news))
      ,user_data);
    return TRUE;
  }
	    
	default:
	    break;
	}
    }
    Selecting();
    return TRUE;
}

static gboolean _applet_key_release(GtkWidget * widget,
					      GdkEventKey * event,
					      gpointer user_data)
{
	ULOG_DEBUG(__FUNCTION__);
    if (event == NULL)
        return FALSE;
  if (event->state & (GDK_CONTROL_MASK |
			GDK_SHIFT_MASK |
			GDK_MOD1_MASK |
			GDK_MOD3_MASK | GDK_MOD4_MASK | GDK_MOD5_MASK)) {
	return TRUE;
    }

	
    switch (event->keyval) {
	case GDK_Escape:
	    ULOG_DEBUG("Escape released");
	    if(rss_appl_inf->refreshing)
	    {
	        _applet_refresh_button_cb(NULL, rss_appl_inf);
		rss_appl_inf->refreshing = FALSE;
	    }
	    break;
	    
	default:
	    break;
    }
    Selecting();
    return TRUE;
}					      

/** A callback function that is called after a headline is clicked
  *
  * @param widget the tree view containing the headlines
  * @param event not used
  * @param user_data the applet structure
  * @return always FALSE
  */
static gboolean _applet_headline_clicked_cb(GtkWidget * widget,
				     GdkEventButton * event,
				     gpointer user_data)
{
  GtkTreeModel *model;
  GtkTreeSelection *selection;
  GtkTreeIter iter;
  gint pos;

  ULOG_DEBUG(__FUNCTION__);

  g_assert(NULL != widget);
  g_assert(NULL != event);
  g_assert(NULL != user_data);

  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(rss_appl_inf->rss_news));
  if (!expand_collapse_selection(FALSE) && 
      gtk_tree_selection_get_selected(selection, &model, &iter))
    Launch_item(model,user_data);

  return TRUE;
}

static void refresh_pressed_cb(GtkWidget * widget, GdkEventButton *event,gpointer user_data)
{
	ULOG_DEBUG(__FUNCTION__);
	gtk_container_remove(GTK_CONTAINER(widget), GTK_WIDGET(gtk_bin_get_child(GTK_BIN(widget))));
	if (rss_appl_inf->refreshing)
	{
		gtk_container_add(GTK_CONTAINER(widget), GTK_WIDGET(rss_appl_inf->rss_image_stop_pressed));
	}
	else
	{
		gtk_container_add(GTK_CONTAINER(widget), GTK_WIDGET(rss_appl_inf->rss_image_pressed));
	}
	gtk_widget_show_all(widget);
	_applet_refresh_button_cb(NULL,rss_appl_inf);
  Selecting();
	return;
}

static void refresh_released_cb(GtkWidget * widget, GdkEventButton *event,gpointer user_data)
{
	ULOG_DEBUG(__FUNCTION__);
	gtk_container_remove(GTK_CONTAINER(widget), GTK_WIDGET(gtk_bin_get_child(GTK_BIN(widget))));
	if (rss_appl_inf->refreshing)
	{
		gtk_container_add(GTK_CONTAINER(widget), GTK_WIDGET(rss_appl_inf->rss_image_stop));
	}
	else
	{
		gtk_container_add(GTK_CONTAINER(widget), GTK_WIDGET(rss_appl_inf->rss_image));
	}
	gtk_widget_show_all(widget);
  Selecting();
	return;
}


static void Unselect_all(void)
{
  GtkTreeSelection *selection;
  ULOG_DEBUG(__FUNCTION__);
  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(rss_appl_inf->rss_news));
  expand_collapse_selection(TRUE);
  gtk_tree_selection_unselect_all (selection);
  ULOG_DEBUG("Unselect end");
//  Scroll_to_correct();
//  gtk_scrolled_window_set_policy(rss_appl_inf->scrolledwindow,GTK_POLICY_NEVER,GTK_POLICY_NEVER);
//  gtk_layout_set_size(rss_appl_inf->layout, RSS_APPLET_WIDTH-30,RSS_APPLET_SCROLLED_WINDOW_HEIGHT);
//  gtk_widget_set_size_request(rss_appl_inf->rss_news,RSS_APPLET_WIDTH-30,-1);
//  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (rss_appl_inf->scrolledwindow), GTK_POLICY_NEVER, GTK_POLICY_NEVER);
}


gboolean focus_lost(GtkWidget * widget, GdkEventFocus *event,gpointer user_data)
{
	ULOG_DEBUG(__FUNCTION__);
/*  GtkAdjustment *adjust=gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(rss_appl_inf->scrolledwindow));
  gtk_adjustment_set_value(adjust,0.0);
  gtk_scrolled_window_set_vadjustment(GTK_SCROLLED_WINDOW(rss_appl_inf->scrolledwindow), adjust);*/
/*  Collapse_old_row();*/
  Unselect_all();
	/*GtkTreeSelection *selection;
	selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(widget));
	gtk_tree_selection_unselect_all (selection);*/
	/*gtk_tree_view_collapse_all(GTK_TREE_VIEW(rss_appl_inf->rss_news));
	lastexpanded=NULL;*/
	return TRUE;
}



gboolean focus_lost2(GtkWidget * widget, GdkEventCrossing *event,gpointer user_data)
{
	ULOG_DEBUG(__FUNCTION__);
  ULOG_DEBUG("x: %f y: %f",event->x, event->y);
  gint xcord=0,ycord=0;
  gtk_widget_get_pointer(rss_appl_inf->appl_frame,&xcord,&ycord);
  ULOG_DEBUG("getcord:x: %f y: %f",xcord, ycord);
  
  update_font_width(rss_appl_inf);
  if (rss_appl_inf->textColumn)
  ULOG_DEBUG("Column: %d, max_chars: %d",
    gtk_tree_view_column_get_width(rss_appl_inf->textColumn),
    (gtk_tree_view_column_get_width(rss_appl_inf->textColumn) /
          rss_appl_inf->avg_char_width));
  GtkAllocation sreq;
  sreq = rss_appl_inf->appl_frame->allocation;
  ULOG_DEBUG("widget:x: %d y: %d",sreq.width, sreq.height);
  /*if ((event->x<=0)||(event->y<=RSS_APPLET_SCROLLED_WINDOW_HEIGHT-RSS_APPLET_HEIGHT)||
    (event->x>=RSS_APPLET_WIDTH-30)||(event->y>=RSS_APPLET_HEIGHT-(RSS_APPLET_HEIGHT-RSS_APPLET_SCROLLED_WINDOW_HEIGHT)))*/
  if ((xcord-1<=0)||(ycord-1<=0)||((gint)xcord>=sreq.width)||((gint)ycord>=sreq.height))
  {
    ULOG_DEBUG("Entered...");
/*    GtkAdjustment *adjust=gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(rss_appl_inf->scrolledwindow));
    gtk_adjustment_set_value(adjust,0.0);
    gtk_scrolled_window_set_vadjustment(GTK_SCROLLED_WINDOW(rss_appl_inf->scrolledwindow), adjust);*/
    
//      Collapse_old_row();
    Unselect_all();

  /*GtkTreeSelection *selection;
	selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(rss_appl_inf->rss_news));
	gtk_tree_selection_unselect_all (selection);*/
	/*gtk_tree_view_collapse_all(GTK_TREE_VIEW(rss_appl_inf->rss_news));
	lastexpanded=NULL;*/
	return TRUE;
  }
    ULOG_DEBUG("Omited...%d", xcord+1>=sreq.width);
  Selecting();
  return FALSE;
}

#ifdef VERSION_3
static void on_menuitem_settings(GtkWidget *widget, gpointer user_data)
{
    /*if (!window && user_data)
        window = gtk_widget_get_ancestor(GTK_WIDGET(user_data), GTK_TYPE_WINDOW);
    
    if (window && GTK_IS_WINDOW(window))*/
        execute_rss_settings(rss_appl_inf->osso, NULL, TRUE);
}

void on_menuitem_close(GtkWidget * widget, gpointer user_data)
{
  gtk_widget_destroy(GTK_WIDGET(user_data));
}

static gboolean context_menu_query(GtkWidget *widget, GdkEvent *event)
{
  g_signal_stop_emission_by_name (widget, "tap-and-hold-query");
  return FALSE;
}

static gboolean context_menu_query2(GtkWidget *widget, GdkEvent *event)
{
  g_signal_stop_emission_by_name (widget, "tap-and-hold-query");
  return TRUE;
}
#endif

static void realize(GtkWidget *widget, gpointer user_data)
{
  GtkAllocation sreq, nreq;
  gint width = gtk_tree_view_column_get_width(rss_appl_inf->textColumn);
//  gtk_tree_view_column_get_height
  sreq = rss_appl_inf->appl_frame->allocation;
  if ((rss_appl_inf->prev_height != sreq.height && rss_appl_inf->prev_width) ||
      (rss_appl_inf->prev_width !=  sreq.width && width)) {

    nreq = rss_appl_inf->scrolledwindow->allocation;
//    printf("########realize\n");
    update_font_width(rss_appl_inf);
    rss_appl_inf->prev_width = sreq.width;
    rss_appl_inf->prev_height = sreq.height;

    rss_appl_inf->chars_per_line = (sreq.width - 74) / rss_appl_inf->avg_char_width;
    rss_appl_inf->displayed_lines = nreq.height / rss_appl_inf->line_height;
    
    ULOG_DEBUG(">>Column: %d, max_chars: %d, lines: %d",
      width, rss_appl_inf->chars_per_line, rss_appl_inf->displayed_lines);
    ULOG_DEBUG(">>widget:x: %d y: %d",sreq.width, sreq.height);
    ULOG_DEBUG(">>widget rss:x: %d y: %d",nreq.width, nreq.height);
        

    _applet_set_headlines(rss_appl_inf);

//    gtk_layout_set_size(rss_appl_inf->layout, nreq.width, nreq.height);
//    gtk_widget_set_size_request(rss_appl_inf->rss_news,nreq.width,-1);
//    Unselect_all();
  }
}
                                            
/************************************************************************/
/*                        PUBLIC FUNCTIONS                              */
/************************************************************************/

/* Applet interface functions under here */


void *hildon_home_applet_lib_initialize(void *state_data,
					gint * state_size,
					GtkWidget ** widget)
{
	
  g_type_init();

  bindtextdomain(GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR);

  ULOG_DEBUG("Initializing News reader applet");
    
  rss_appl_inf = g_new0(AppletInfo, 1);
  rss_appl_inf->avg_char_width = 10;

  GtkWidget *scrolledwindow;
  scrolledwindow = gtk_scrolled_window_new (NULL, NULL);
  gtk_widget_show (scrolledwindow);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow), GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
  GtkWidget *news_vbox = gtk_vbox_new(FALSE, 0/*HILDON_MARGIN_DEFAULT*/);
	
  GtkWidget *applet_frame = gtk_widget_new(
        GTK_TYPE_FRAME,
        "visible",        TRUE,
        "name",           "osso-rss-feed-reader",
        "shadow-type",    GTK_SHADOW_NONE,
        "border-width",   0,
        "can-focus",      FALSE,
        NULL );
  rss_appl_inf->appletframe = applet_frame;

  
  GtkWidget *event_box =g_object_new (
        GTK_TYPE_EVENT_BOX,
        "visible",           TRUE,
        "above-child",       FALSE,
        "visible-window",    FALSE,
        "can-focus",         FALSE,
        "border-width",      0,
        NULL);
        
    /* Outer container */
/*  GtkWidget *rss_vbox = g_object_new (
        GTK_TYPE_VBOX,
        "visible",           TRUE,
        "homogeneous",       FALSE,
        "spacing",           0,
        "border-width",      HILDON_MARGIN_DOUBLE,
        //"border-width",      0,
        NULL);*/
  //GtkWidget *applet_vbox = gtk_vbox_new(FALSE, HILDON_MARGIN_DOUBLE);
  GtkWidget *applet_vbox = gtk_vbox_new(FALSE, 0);
  g_object_set(applet_vbox,
        "homogeneous",       FALSE,
        "spacing",           0,
        "border-width",      HILDON_MARGIN_DOUBLE,
        NULL);

/*  GtkWidget *label_and_news_box = g_object_new (
        GTK_TYPE_VBOX,
        "visible",           TRUE,
        "homogeneous",       FALSE,
        "spacing",           0,
        //"border-width",      HILDON_MARGIN_DOUBLE,
        "border-width",      0,
        NULL);*/
    GtkWidget *empty_label = gtk_label_new(_("rss_ia_no_posts"));
    g_object_ref(empty_label);
    GtkWidget *rss_refresh = g_object_new (
        GTK_TYPE_EVENT_BOX,
        "visible",           TRUE,
        "above-child",       FALSE,
        "visible-window",    FALSE,
        "can-focus",         FALSE,
        "border-width",      0,
        NULL);

	/*Load the images to the structure*/
	rss_appl_inf->rss_image=gtk_image_new_from_icon_name("qgn_toolb_gene_refresh",
				     GTK_ICON_SIZE_LARGE_TOOLBAR);
	rss_appl_inf->rss_image_pressed=gtk_image_new_from_icon_name("qgn_toolb_gene_refresh_pressed",
				     GTK_ICON_SIZE_LARGE_TOOLBAR);
	rss_appl_inf->rss_image_stop=gtk_image_new_from_icon_name("qgn_toolb_gene_stop",
				     GTK_ICON_SIZE_LARGE_TOOLBAR);
	rss_appl_inf->rss_image_stop_pressed=gtk_image_new_from_icon_name("qgn_toolb_gene_stop_pressed",
				     GTK_ICON_SIZE_LARGE_TOOLBAR);
	/*Increase the ref number, otherwise it will be destroyed, when we puts out from the event_box*/
	g_object_ref(rss_appl_inf->rss_image);
	g_object_ref(rss_appl_inf->rss_image_pressed);
	g_object_ref(rss_appl_inf->rss_image_stop);
	g_object_ref(rss_appl_inf->rss_image_stop_pressed);
    
  GtkWidget *rss_updated = gtk_label_new(NULL);
  gtk_misc_set_alignment(GTK_MISC(rss_updated), 0, 0.5);
  GtkWidget *update_hbox = gtk_hbox_new(FALSE, HILDON_MARGIN_DOUBLE);
	gtk_widget_set_size_request(scrolledwindow, RSS_APPLET_WIDTH,
 				RSS_APPLET_SCROLLED_WINDOW_HEIGHT);
	GtkTreeStore *headlines =
	gtk_tree_store_new(N_COLUMNS, GDK_TYPE_PIXBUF,
			   G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING,
			   G_TYPE_INT);
  GtkWidget *rss_news =
	gtk_tree_view_new_with_model(GTK_TREE_MODEL(headlines));
	
	gtk_widget_show (rss_news);
//	g_object_ref(scrolledwindow);
	gtk_widget_show_all(scrolledwindow);
	
    
  have_last_selection = FALSE;
  GtkCellRenderer *iconRenderer = gtk_cell_renderer_pixbuf_new();
  g_object_set (iconRenderer, "yalign", 0.0, NULL);
  GtkCellRenderer *textRenderer = gtk_cell_renderer_text_new();
  rss_appl_inf->textRenderer = textRenderer;
  g_object_set (textRenderer, "yalign", 0.0, NULL);

  GtkTreeViewColumn *column = gtk_tree_view_column_new();
  gtk_tree_view_append_column(GTK_TREE_VIEW(rss_news), column);

  gtk_tree_view_column_pack_start(column, iconRenderer, FALSE);
  gtk_tree_view_column_add_attribute(column, iconRenderer, "pixbuf",
                        ICON_COLUMN);
  gtk_tree_view_column_set_expand (column, FALSE);

  rss_appl_inf->textColumn = column = gtk_tree_view_column_new();
  gtk_tree_view_append_column(GTK_TREE_VIEW(rss_news), column);
/*  gtk_tree_view_column_set_sizing (column, GTK_TREE_VIEW_COLUMN_FIXED);
  gtk_tree_view_column_set_expand (column, TRUE);
  gtk_tree_view_column_add_attribute(column, textRenderer, "markup",
                                       RENDER_COLUMN);*/
  gtk_tree_view_column_pack_start(column, textRenderer, TRUE);
  gtk_tree_view_column_add_attribute(column, textRenderer, "markup",
                                       RENDER_COLUMN);
  g_object_set(textRenderer,  "ellipsize",PANGO_ELLIPSIZE_END, NULL);

  /* Hiding the expander column*/
/*  column = gtk_tree_view_column_new();
  gtk_tree_view_append_column(GTK_TREE_VIEW(rss_news), column);
  gtk_tree_view_set_expander_column (GTK_TREE_VIEW(rss_news),column);
  gtk_tree_view_column_set_visible(column,FALSE);
  //gtk_tree_view_get_expander_column(GTK_TREE_VIEW(rss_news))
  hiddencolumn=column;*/

  GtkTreeSelection *rss_select =
           gtk_tree_view_get_selection(GTK_TREE_VIEW(rss_news));
  gtk_tree_selection_set_mode(rss_select, GTK_SELECTION_SINGLE);

  /*Changes the order of the next two, and start and end changed*/
       
  rss_appl_inf->state_data = state_data;
  rss_appl_inf->state_size = (int) &state_size;
    
  gtk_widget_show(applet_vbox);
	/*It is added for SP+ background*/
	gtk_widget_show_all(applet_frame);
	//gtk_widget_set_app_paintable(applet_frame,1);
	//gtk_widget_realize(applet_frame);
	//gdk_window_set_back_pixmap(applet_frame->window,gdkpixmap,0);
	gtk_widget_set_size_request(applet_frame, RSS_APPLET_WIDTH,
 				RSS_APPLET_HEIGHT);

   GtkWidget *layout=NULL;
   layout=gtk_layout_new(NULL,NULL);
   gtk_layout_put(GTK_LAYOUT(layout),rss_news,0,0);
   GtkWidget *align=NULL;
   align=gtk_alignment_new(0, 0, 1, 1);

   g_object_set(GTK_ALIGNMENT(align),
        "top-padding",      HILDON_MARGIN_DOUBLE+4,
        NULL);
   g_object_set(GTK_ALIGNMENT(align),
        "bottom-padding",      HILDON_MARGIN_DOUBLE,
        NULL);
   gtk_container_add(GTK_CONTAINER(align), scrolledwindow);

/*
  g_object_set(scrolledwindow,
        "border-width",      HILDON_MARGIN_DOUBLE,
        NULL);
		*/
   //GtkRequisition sreq;
   //gtk_widget_size_request(rss_news,&sreq);
/*   gtk_layout_set_size(layout, RSS_APPLET_WIDTH-30,RSS_APPLET_SCROLLED_WINDOW_HEIGHT);
   gtk_widget_set_size_request(rss_news,RSS_APPLET_WIDTH-30,-1);*/

   gtk_container_add (GTK_CONTAINER (scrolledwindow), layout);
//   gtk_container_add (GTK_CONTAINER (scrolledwindow), rss_news);
//   GtkAdjustment *adj = gtk_layout_get_vadjustment(layout);
//   gtk_container_add (GTK_CONTAINER (scrolledwindow), rss_news);
//   gtk_box_pack_end(GTK_BOX(news_vbox), label_and_news_box, FALSE, FALSE, 0);
   gtk_container_add(GTK_CONTAINER(rss_refresh), rss_appl_inf->rss_image);
   gtk_box_pack_start(GTK_BOX(update_hbox), rss_refresh, FALSE, FALSE, 0);
   gtk_box_pack_end(GTK_BOX(update_hbox), rss_updated, TRUE, TRUE, 0);
   gtk_container_add(GTK_CONTAINER(event_box), GTK_WIDGET(update_hbox));
   gtk_box_pack_start(GTK_BOX(applet_vbox), event_box, FALSE, FALSE, 0);
   gtk_box_pack_end(GTK_BOX(applet_vbox), align, TRUE, TRUE, 0);
//   gtk_box_pack_end(GTK_BOX(applet_vbox), news_vbox, TRUE, TRUE, 0);
/*   gtk_box_pack_start(GTK_BOX(rss_vbox), update_hbox, FALSE, FALSE, 0);
	 gtk_box_pack_end(GTK_BOX(rss_vbox), news_vbox, FALSE, FALSE, 0);*/
//   gtk_box_pack_start(GTK_BOX(applet_vbox), event_box, FALSE, FALSE, 0);
//   gtk_box_pack_start(GTK_BOX(applet_vbox), rss_vbox, FALSE, FALSE, 0);
   gtk_container_add(GTK_CONTAINER(applet_frame), applet_vbox);
    
    *widget = applet_frame;
#ifdef VERSION_3
    GtkMenu *context_menu = GTK_MENU(gtk_menu_new());

    GtkWidget *menuitem = gtk_menu_item_new_with_label(_("rss_me_settings"));
    gtk_widget_show(menuitem);
    gtk_container_add(GTK_CONTAINER(context_menu), menuitem);
    g_signal_connect((gpointer) menuitem, "activate",
		     G_CALLBACK(on_menuitem_settings), applet_frame);

    GtkWidget *menuitem_separator = gtk_separator_menu_item_new();
    gtk_widget_show(menuitem_separator);
    gtk_container_add(GTK_CONTAINER(context_menu), menuitem_separator);
    gtk_widget_set_sensitive(menuitem_separator, FALSE);

    menuitem = gtk_menu_item_new_with_label(_("rss_me_close"));
    gtk_widget_show(menuitem);
    gtk_container_add(GTK_CONTAINER(context_menu), menuitem);
    g_signal_connect((gpointer) menuitem, "activate",
		     G_CALLBACK(on_menuitem_close), applet_frame);

  gtk_widget_tap_and_hold_setup (GTK_WIDGET(applet_frame),
                                               context_menu, NULL, GTK_TAP_AND_HOLD_PASS_PRESS);
  g_signal_connect((gpointer) rss_news, "tap-and-hold-query",
                G_CALLBACK(context_menu_query), applet_frame);
  g_signal_connect((gpointer) layout, "tap-and-hold-query",
                G_CALLBACK(context_menu_query2), applet_frame);
#endif
  rss_appl_inf->displayed_lines = 1;
  rss_appl_inf->headlines = headlines;
  rss_appl_inf->rss_news = rss_news;
  rss_appl_inf->rss_updated = rss_updated;
  rss_appl_inf->rss_refresh = rss_refresh;
  rss_appl_inf->empty_label = empty_label;
	rss_appl_inf->scrolledwindow=scrolledwindow;
//  rss_appl_inf->label_and_news_box = label_and_news_box;
  rss_appl_inf->refreshing = FALSE;
//  rss_appl_inf->event_box = event_box;
  rss_appl_inf->appl_frame = applet_frame;
  rss_appl_inf->applet_vbox = applet_vbox;
  rss_appl_inf->layout=layout;
    
//    _applet_set_headlines(rss_appl_inf);
    _applet_gconf_init(rss_appl_inf);
    _applet_dbus_connect(rss_appl_inf, (DBusHandleMessageFunction)
			 _applet_handle_signal_cb);
       
   g_signal_connect(G_OBJECT(applet_vbox), "key-release-event",
		     G_CALLBACK(_applet_key_release), NULL);		     
	/*g_signal_connect(G_OBJECT(event_box), "button-press-event",
		     G_CALLBACK(refresh_pressed_cb), NULL);*/
	g_signal_connect(G_OBJECT(rss_refresh), "button-press-event",
		     G_CALLBACK(refresh_pressed_cb), NULL);
    /*g_signal_connect(G_OBJECT(event_box), "button-release-event",
		     G_CALLBACK(refresh_released_cb), NULL);*/
	g_signal_connect(G_OBJECT(rss_refresh), "button-release-event",
		     G_CALLBACK(refresh_released_cb), NULL);

/*    g_signal_connect(G_OBJECT(rss_news), "button-press-event",
		     G_CALLBACK(_applet_headline_pressed_cb),
		     rss_appl_inf);*/
    g_signal_connect(G_OBJECT(rss_news), "button-release-event",
		     G_CALLBACK(_applet_headline_clicked_cb),
		     rss_appl_inf);
      
    g_signal_connect(G_OBJECT(rss_news), "key-press-event",
		     //G_CALLBACK(_applet_tree_view_key_release), rss_news);
         G_CALLBACK(_applet_tree_view_key_release), rss_appl_inf);
/*    g_signal_connect(G_OBJECT(rss_news), "realize",
        G_CALLBACK(realize), rss_appl_inf);*/
    g_signal_connect(G_OBJECT(applet_frame), "size-allocate",
        G_CALLBACK(realize), rss_appl_inf);
	//g_signal_connect(G_OBJECT(rss_news), "focus-out-event",
  g_signal_connect(G_OBJECT(rss_news), "focus-out-event",
		     G_CALLBACK(focus_lost), NULL);
  g_signal_connect(G_OBJECT(rss_news), "leave-notify-event",
		     G_CALLBACK(focus_lost2), NULL);
  g_signal_connect(G_OBJECT(rss_refresh), "leave-notify-event",
		     G_CALLBACK(focus_lost2), NULL);
/*  g_signal_connect(G_OBJECT(event_box), "leave-notify-event",
		     G_CALLBACK(focus_lost2), NULL);*/
  Unselect_all();
  Selecting();

  update_font_width(rss_appl_inf);

  if (osso_iap_cb((osso_iap_cb_t) _applet_iap_cb) != OSSO_OK)
	ULOG_ERR("Could not set callback for connectivity");

#ifdef VERSION_3
  
  htimeout = g_timeout_add(5000, _setup_alarmd, NULL);
  
#endif

    return rss_appl_inf;
}

#ifdef VERSION_3
gint _setup_alarmd(gpointer data)
{
  
  ULOG_ERR("ENTER alarm setup timeout\n");
  
  if (setup_alarm_event_now(FALSE) == FALSE)
  {
    ULOG_ERR("Must run again...\n");
    return TRUE;
  } else {
    ULOG_ERR("Operation finished\n");
    return FALSE;
  }
  
  ULOG_ERR("LEAVE alarm setup timeout'n");
  
}
#endif

void hildon_home_applet_lib_deinitialize(void *applet_data)
{
    AppletInfo *appl_inf = (AppletInfo *) applet_data;
		DBusError err;
    ULOG_DEBUG(__FUNCTION__);
		dbus_error_init(&err);
    ULOG_DEBUG(__FUNCTION__);
    
    if (appl_inf)
    {
    if (appl_inf->timeout > 0) {
	    ULOG_INFO("Removing old RSS Refresh timeout");
	    g_source_remove(appl_inf->timeout);
	    appl_inf->timeout = 0;
    }
#ifndef VERSION_3
    if (rss_appl_inf->refresh_banner)
    {
      gtk_widget_destroy(rss_appl_inf->refresh_banner);
      rss_appl_inf->refresh_banner=NULL;
      rss_appl_inf->refresh_banner_displayed=FALSE;
    }
#else
    close_rss_settings();
#endif

#ifdef VERSION_3
  
  if (htimeout != 0)
    gtk_timeout_remove(htimeout);
  
#endif

    _applet_gconf_deinit(appl_inf);

    g_object_unref(appl_inf->empty_label);
    //g_object_unref(appl_inf->rss_news);
	g_object_unref(appl_inf->rss_image);
	g_object_unref(appl_inf->rss_image_pressed);
	g_object_unref(appl_inf->rss_image_stop);
	g_object_unref(appl_inf->rss_image_stop_pressed);
	gchar *rule = NULL;

	rule = g_strdup_printf(APPLET_DBUS_SIGNAL_LISTENER_FMT,
			       NEWS_SERVICE_NAME, NEWS_INTERFACE_NAME);

	dbus_bus_remove_match(appl_inf->connection, rule, &err);
	if (dbus_error_is_set(&err)) {
	    ULOG_ERR("Failed to set up DBus rule %s, error: %s", rule,
		     err.message);
	    dbus_error_free(&err);
	    g_free(rule);
      osso_deinitialize(appl_inf->osso);
	    return;
	} else {
	    dbus_connection_remove_filter(appl_inf->connection,
				       (DBusHandleMessageFunction)
			 _applet_handle_signal_cb, appl_inf);
	}
	dbus_error_free(&err);
	g_free(rule);
  /*dbus_connection_close(appl_inf->connection);
  dbus_connection_unref(appl_inf->connection);*/
  /*osso_rpc_unset_cb_f(appl_inf->osso,
			OSSO_RSS_FEED_READER_APPLET_SERVICE,
			OSSO_RSS_FEED_READER_APPLET_OBJECT_PATH,
			OSSO_RSS_FEED_READER_APPLET_INTERFACE,
			dbus_req_handler, appl_inf);*/
	osso_deinitialize(appl_inf->osso);
    
    /* We should probably statesave here? */
     if(appl_inf->state_data != NULL)
         g_free(appl_inf->state_data);
    if(appl_inf != NULL);	
        g_free(appl_inf);
  }
}


/*int hildon_home_applet_lib_get_requested_width(void *applet_data)
{
    ULOG_DEBUG(__FUNCTION__);

    return REQUESTED_WIDTH;
}*/


void hildon_home_applet_lib_background(void *applet_data)
{
  ULOG_DEBUG(__FUNCTION__);
    /*if (rss_appl_inf->refresh_banner && GTK_IS_WIDGET(rss_appl_inf->refresh_banner))
    {
       rss_appl_inf->refresh_banner=NULL;
       gtk_widget_destroy(rss_appl_inf->refresh_banner);
    }*/
    //gtk_banner_close(NULL);

    if(rss_appl_inf != NULL)
        rss_appl_inf->backgrounded = TRUE;
    else
        ULOG_DEBUG("rss_appl_inf NULL, cannot update applet info");

    return;
}

void hildon_home_applet_lib_foreground(void *applet_data)
{
    ULOG_DEBUG(__FUNCTION__);
    
    if(rss_appl_inf != NULL) {
        rss_appl_inf->backgrounded = FALSE;
        /*if(rss_appl_inf->refreshing)
            rss_appl_inf->refresh_banner=hildon_banner_show_animation(rss_appl_inf->appl_frame,NULL, _("home_ia_refreshing"));
            //gtk_banner_show_animation(NULL, _("home_ia_refreshing"));*/
    }
    else
        ULOG_DEBUG("rss_appl_inf NULL, cannot update applet info");

    if (!rss_appl_inf->refreshing)
        _applet_set_information(rss_appl_inf);
    Unselect_all();
    
    return;
}


int hildon_home_applet_lib_save_state(void *applet_data, void **state_data,
				      int *state_size)
{

    ULOG_DEBUG(__FUNCTION__);

    /* Whats this used for? */
//     *state_data = (void *) g_malloc(sizeof(char *) * 20);
//     if (*state_data == NULL) {
// 	return -1;
//     }
//     *state_size = 20;
//     return 1;
    return -1;
}

GtkWidget *hildon_home_applet_lib_settings(void *applet_data,
					     GtkWindow * parent)
{
#ifdef VERSION_3
     ULOG_DEBUG(">>>>>>>>>>>>>>>>>>>>>> SETTINGS <<<<<<<<<<<<<<<<<<<< %p",
        parent);

    window = parent;
    
    GtkWidget *settings;

    settings = gtk_menu_item_new_with_label
            (_("rss_ap_news_reader"));
    g_signal_connect (settings, "activate",
                      G_CALLBACK (on_menuitem_settings), NULL);

    return settings;
#else
  GtkWidget *dialog = NULL;
  return dialog;
#endif
}
