/**
    @file interface.c
	
    General user interface functions

    Copyright (c) 2004-2006 Nokia Corporation.
	
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
    02111-1307 USA
*/


#ifdef HAVE_CONFIG_H
#include <aconf.h>
#endif

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <libgnomevfs/gnome-vfs-result.h>
#include <libgnomevfs/gnome-vfs.h>
#include <libosso.h>
#include <hildon-widgets/hildon-defines.h>
#include <hildon-widgets/hildon-app.h>
#include <hildon-widgets/hildon-appview.h>
#include <hildon-widgets/hildon-number-editor.h>
#include <hildon-widgets/hildon-file-chooser-dialog.h>
#include <hildon-widgets/hildon-file-details-dialog.h>
#include <hildon-widgets/hildon-file-system-model.h>
#include <hildon-widgets/hildon-get-password-dialog.h>
#include <hildon-widgets/hildon-caption.h>
#include <hildon-widgets/hildon-banner.h>

#include <hildon-widgets/hildon-note.h>


#include <osso-helplib.h>

#include "ErrorCodes.h"

#include "i18n.h"
#include "utility.h"
#include "interface.h"
#include "ui.h"
#include "callbacks.h"
#include "constant.h"
#include "pdfviewer.h"
#include "debug.h"

#define HELP_ICON_SIZE HILDON_ICON_SIZE_26

#define CURRENT_PAGE_BUTTON_MIN_WIDTH 80
//#define CURRENT_PAGE_BUTTON_MIN_WIDTH 55
#define CURRENT_ZOOM_WIDGET_CHARS      6

#define debug(x) x

/*******************************************************************************
 **** Private data
 **/

/*******************************************************************************
 **** Prototypes for private functions
 **/

/* Application constructors */
static void        add_custom_actions    (GtkActionGroup *actions, 
					  AppUIData      *app_ui_data);
static GtkToolbar *build_toolbar         (GtkActionGroup *actions,
					  AppUIData      *app_ui_data);
static void        build_application_area(AppUIData      *app_ui_data);

/* Switch to page widget maintainers */
static void ui_set_current_page                    (AppUIData *app_ui_data, 
						    gint       value);
static void ui_set_total_pages                     (AppUIData  *app_ui_data, 
						    gint       value);
/* Recent menu maintainers */
static void ui_add_recent_item                     (AppUIData   *app_ui_data, 
						    const gchar *filename);

/* Menu/ToolBar Item logic controllers */
void ui_enable_document_controls                   (AppUIData   *app_ui_data, 
						    gboolean    enable);
static void ui_set_callbacks_for_dimmed_menu_items (AppUIData   *app_ui_data);
static void ui_set_dimmed_menu_item_callback       (AppUIData   *app_ui_data,
						    const gchar *path,
						    InsensitiveMenuPressCallback
						    callback, 
						    gpointer     data);
/* PopUp widget constructor */
static void ui_setup_popup_menu                    (AppUIData   *app_ui_data);

/* Creates and/or gives reference on HildonFileSystemModel */
static HildonFileSystemModel *get_file_system_model(GtkWidget   *ref_widget);


/*******************************************************************************
 **** Public Functions
 ***/


/**
	Creates application user interface.
	Creates corresponding actions, UI manager, menus and the toolbar.
	
	@param app HildonApp
	@return void
*/
void ui_create_main_window(AppData * app_data)
{
    HildonApp      *app;
    HildonAppView  *app_view;
    GtkSettings    *settings;
    GtkUIManager   *ui_manager;
    GtkActionGroup *actions;
    GtkAction      *action;
    GtkMenu        *menu_appview;
    GtkWidget      *menu_created;
    GError         *error     = NULL;
    gpointer        user_data = NULL;
    AppUIData      *app_ui_data;

    app_ui_data = app_data->app_ui_data;
    g_return_if_fail(app_ui_data != NULL);
    
    /* enabling some flags by default */
    PDF_FLAGS_SET(app_ui_data->flags,
		  PDF_FLAGS_SELECT_KEY_ALLOWED);
    
    /* Create app_view and application */
    app_view =
	HILDON_APPVIEW(hildon_appview_new(_("pdfv_ap_pdf_viewer_name")));
    g_return_if_fail(app_view != NULL);
    app_ui_data->app_view = app_view;

    
    app = HILDON_APP(hildon_app_new_with_appview(app_view));
    if( app == NULL) {
      gtk_object_destroy(GTK_OBJECT(app_ui_data->app_view));
      return;
    }
    app_ui_data->app = app;

    hildon_app_set_appview(app, app_view);

    /* Set window title */
    hildon_app_set_title(app, _("pdfv_ap_pdf_viewer_name"));

    /* Allow two part titles */
    hildon_app_set_two_part_title(app, TRUE);

    /* Allow fullscreen shortcut key? */
    hildon_appview_set_fullscreen_key_allowed(app_view, TRUE);

    /* Disable icon display in menus */
    settings = gtk_settings_get_default();
    g_return_if_fail(settings != NULL);

    gtk_settings_set_long_property(settings, "gtk-menu-images", 0,
				   GETTEXT_PACKAGE);

    /* set toolbar icon size */
    gtk_settings_set_long_property(settings, "gtk-toolbar-icon-size",
				   HILDON_ICON_SIZE_26, GETTEXT_PACKAGE);

    /* Set user_data to make it accessible to action callback(s) */
    user_data = app_ui_data;

    /* Create new action group */
    actions = gtk_action_group_new("Actions");
    g_assert(actions);
    /* Translation domain need to be set _before_ adding actions */
    gtk_action_group_set_translation_domain(actions, GETTEXT_PACKAGE);
    gtk_action_group_add_actions(actions, action_entries, n_action_entries,
				 user_data);
    gtk_action_group_add_toggle_actions(actions, toggle_action_entries,
					n_toggle_action_entries,
					user_data);

    /* Add any custom actions */
    add_custom_actions(actions, app_ui_data);

    /* show empty recent menu */
    action = gtk_action_group_get_action(actions, ACTION_MENU_RECENT);
    if (action) {
	g_object_set(action, "hide-if-empty", FALSE, NULL);
    }

    /* Create new UI manager */
    ui_manager = gtk_ui_manager_new();
    g_assert(ui_manager);
    app_ui_data->ui_manager = ui_manager;
    gtk_ui_manager_insert_action_group(ui_manager, actions, 0);

    if (!gtk_ui_manager_add_ui_from_string
	(ui_manager, ui_info, -1, &error)) {
	OSSO_LOG_ERR("Building ui_manager failed: %s", error->message);
	g_error_free(error);
	g_assert(FALSE);
    }

    /* Transfer menu items to app_view menu */
    app_ui_data->menu = menu_appview = hildon_appview_get_menu(app_view);
    g_assert(menu_appview);
    menu_created = gtk_ui_manager_get_widget(ui_manager, "/MenuBar");

    g_assert(menu_created);
    gtk_container_foreach(GTK_CONTAINER(menu_created),
			  (GtkCallback) gtk_widget_reparent, menu_appview);

    /* Update recent menu */
    ui_update_recent_menu(app_ui_data);

    /* Update show images menu item */
    action = gtk_action_group_get_action(actions,
					 "pdfv_me_menu_screen_show_images");
    g_object_set(action, "sensitive", TRUE, NULL);
    gtk_toggle_action_set_active
      ( GTK_TOGGLE_ACTION(action),
        PDF_FLAGS_IS_SET(app_ui_data->flags, PDF_FLAGS_SHOW_IMAGES) );

    /* Bind toolbar with app_view */
    app_ui_data->toolbar = build_toolbar(actions, app_ui_data);
    g_assert(app_ui_data->toolbar);
    /* deprecated */
    /* hildon_appview_set_toolbar(app_ui_data->app_view, app_ui_data->toolbar); */
    gtk_box_pack_start(GTK_BOX(app_ui_data->app_view->vbox),
		       GTK_WIDGET(app_ui_data->toolbar), TRUE, TRUE, 0);

    /* Build application area */
    build_application_area(app_ui_data);

    /* Finally some signal-handlers */
    
    /* Set callbacks for dimmed menu items. */
    ui_set_callbacks_for_dimmed_menu_items(app_ui_data);

    /* catching all HW key signals - separatelly */
    gtk_widget_add_events(GTK_WIDGET(app),
			  GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK);

    g_signal_connect(G_OBJECT(app), "key_release_event",
		     G_CALLBACK(key_release), (gpointer) app_ui_data);
         
   g_signal_connect(G_OBJECT(menu_appview), "expose-event",
		     G_CALLBACK(menu_opened), (gpointer) app_ui_data);
   /*g_signal_connect(G_OBJECT(menu_created), "expose-event",
		     G_CALLBACK(menu_opened), (gpointer) app_ui_data);*/
         
    
    g_signal_connect(G_OBJECT(app), "key_press_event",
		     G_CALLBACK(key_press),   (gpointer) app_ui_data);

    /* catch top/untop events for state saving */
    g_signal_connect(G_OBJECT(app), "topmost_status_lose",
		     G_CALLBACK(on_untop), app_ui_data);
    g_signal_connect(G_OBJECT(app), "topmost_status_acquire",
		     G_CALLBACK(on_top),   app_ui_data);

    /* catch delete event */
    g_signal_connect(G_OBJECT(app), "delete_event", 
		     G_CALLBACK(on_delete_event), app_ui_data);

    /* Show all widgets */
    gtk_widget_show_all(GTK_WIDGET(app));

    /* set the UI */
    ui_update(app_ui_data);
}


/**
 	Runs file chooser dialog.
	
	@param app_ui_data Application UI data structure
	@param action Which file chooser action to perform
	@return void
*/
void ui_run_file_chooser(AppUIData * app_ui_data,
			 GtkFileChooserAction action)
{
    GtkWidget             *file_chooser = NULL;
    GnomeVFSURI           *uri          = NULL;
    gchar                 *filename;
    gchar                 *uri_str      = NULL;
    gint                   response;
    PDFViewerResult        result       = RESULT_INVALID_INTERFACE;
    HildonFileSystemModel *model        = NULL;
    GtkFileFilter         *filter       = NULL;

    g_return_if_fail(app_ui_data);
    g_return_if_fail((action == GTK_FILE_CHOOSER_ACTION_OPEN) ||
		     (action == GTK_FILE_CHOOSER_ACTION_SAVE));

    if( (model = get_file_system_model(GTK_WIDGET(app_ui_data->app)) ) == NULL)
      goto done;

    /* TODO add check for fullscreen mode */
    file_chooser = hildon_file_chooser_dialog_new_with_properties
                    (GTK_WINDOW(app_ui_data->app),
		     "file_system_model", model,
		     "action",            action,
		     NULL);

    gtk_file_chooser_set_select_multiple(GTK_FILE_CHOOSER(file_chooser),
					 FALSE);
    gtk_window_set_modal                (GTK_WINDOW(file_chooser), 
					 TRUE);

    /* set mime filter */
    filter = get_filter_for_supported_formats();

    if(filter != NULL) {
      gtk_file_chooser_add_filter (GTK_FILE_CHOOSER (file_chooser), filter);
      gtk_file_chooser_set_filter (GTK_FILE_CHOOSER (file_chooser), filter);
    } else {
      /* there's no registred mime type by the application! */
      goto done;
    }

    /* set window title */
    if (action == GTK_FILE_CHOOSER_ACTION_OPEN) {
      gtk_window_set_title(GTK_WINDOW(file_chooser),
#ifndef VERSION_3
			   _("pdfy_ti_open_document")); /*NOLOC*/
#else
         _("pdfv_ti_open_document"));
#endif
    } else if (action == GTK_FILE_CHOOSER_ACTION_SAVE) {
      gtk_window_set_title(GTK_WINDOW(file_chooser),
			   _("ckdg_ti_save_object_doc"));
    } else {
      /* this action is not known! */
      goto done;
    }

    /* Set the directory that the dialog ockdg_ti_save_object_docpens in */
    if (  (uri = pdf_viewer_get_uri()) != NULL) {
        uri_str = gnome_vfs_uri_to_string(uri, GNOME_VFS_URI_HIDE_NONE);
        
        if (action == GTK_FILE_CHOOSER_ACTION_OPEN) {
	  /* Set current filename in dialog */
	  gtk_file_chooser_set_uri(GTK_FILE_CHOOSER(file_chooser),
				   uri_str);

	} else if (action == GTK_FILE_CHOOSER_ACTION_SAVE) {
	  /* set the folder in dialog where the file was opened from */
	  gtk_file_chooser_set_current_folder_uri(GTK_FILE_CHOOSER(file_chooser),
						  uri_str);
	  
	  /* set the file name in dialog */
	  if( (filename = get_basename_for_display(uri_str)) == NULL) {
	    g_free(uri_str);
	    goto done;
	  }

	  gtk_file_chooser_set_current_name(GTK_FILE_CHOOSER(file_chooser),
					    filename);
	  
	  g_free(filename);
	}

	g_free(uri_str);
    } else {
	/* Filename is not available, use factory default folder */
	filename =
	    g_build_filename(g_get_home_dir(),
			     SETTINGS_FACTORY_DEFAULT_FOLDER, NULL);
	gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(file_chooser),
					    filename);
	g_free(filename);
    }
    

    /* 
       Ensure that user did not just click OK
       the dialog box remains open in case of errors, 
       but only in case of save! 
    */
    do {
      /* Run the dialog */
      response = gtk_dialog_run(GTK_DIALOG(file_chooser));
      if (response == GTK_RESPONSE_OK) {
	filename =
	  gtk_file_chooser_get_uri(GTK_FILE_CHOOSER(file_chooser));
	
	/* ensure that user did not just click OK */
	if (filename != NULL) {
	  if (action == GTK_FILE_CHOOSER_ACTION_OPEN) {
	    /* destroy the dialog */
	    gtk_widget_destroy(file_chooser);
	    file_chooser = NULL;
    
	    /* destory HildonFileSystemModel */
	    g_object_unref(model);
	    model = NULL;
	    
	    /* start opening the file */
	    ui_open_document(app_ui_data, filename, NULL);
	    
	    /* close the open file dialog in any situation */
	    g_free(filename);

	    break;
	  } else if (action == GTK_FILE_CHOOSER_ACTION_SAVE) {
	    /* save the file */
	    if (file_chooser != NULL) gtk_widget_destroy(file_chooser);
	    file_chooser = NULL;

	    result = pdf_viewer_save(filename);

	    TDB("save file: %s, %d, %d\n", filename, result, RESULT_SAVE_OK);
	    if (result == RESULT_SAVE_OK) {
                ui_add_recent_item(app_ui_data, filename);
                ui_update(app_ui_data);
	    }

	    /* show the result */
	    ui_show_result(app_ui_data, result);
	    
	    g_free(filename);
	    break;
	  }
	}

	/* unset 'ok' and every other buttons in file dialog box */
	gtk_widget_grab_focus(GTK_WIDGET(file_chooser));

      } else {
	/* user cancelled the dialog */
	goto done;
      }
    } while( (result != RESULT_SAVE_OK) ); /* assures that the save dialog closes iff it's saved! */ 
    

/*    if( action == GTK_FILE_CHOOSER_ACTION_OPEN )
      ui_show_result(app_ui_data, result);*/

 done:
    /* destroy the dialog */
    if (file_chooser != NULL) gtk_widget_destroy(file_chooser);
    
    /* destory HildonFileSystemModel */
    if (model != NULL) g_object_unref(model);
}

static GtkWidget *global_password_dialog = NULL;

/**
 	Open document with given file name.
	Presents user with a password dialog if necessary.
	
	@param app_ui_data Application UI data structure
	@param filename filename
	@param password password (optional, can be NULL)
	@return pdf_viewer_open return code
*/
static PDFViewerResult _ui_open_document(AppUIData   *app_ui_data, 
				 const gchar *filename,
				 const gchar *password)
{
    GtkWidget *password_dialog = NULL;
    const gchar     *pass            = NULL;
    gchar     *uri             = NULL;
    gint       response;
    int        retries;
    PDFViewerResult result;
//    GtkWidget   *widget = NULL;

    /* check if the currently opened document is the same as the upcoming one */
    if( pdf_viewer_get_uri() != NULL) {
      gint strcmp_res;

      uri = gnome_vfs_uri_to_string(pdf_viewer_get_uri(), 
				    GNOME_VFS_URI_HIDE_NONE);

      /* the file is already opened */
      strcmp_res = strcmp(uri, filename);
      
      /* destroy the prev file's URI */
      g_free(uri);

      if(strcmp_res == 0)
	 return RESULT_LOAD_OK;
    }

    /* open document with the xpdf engine */
    retries = 0;
    do {
      if (retries == 0) {
	/* show empty document & bring it up, then start rendering
	   to get reasonable response time */
/*	PDFViewerState saved_state = app_ui_data->app_data->state;
	app_ui_data->app_data->state = PDF_VIEWER_STATE_EMPTY;
	ui_update(app_ui_data);
	gtk_window_present(GTK_WINDOW(app_ui_data->app));
	app_ui_data->app_data->state = saved_state;*/
        pdf_viewer_empty_document();

	/* open document initially with supplied password, if any */
	result = pdf_viewer_open(filename, password);
        if (result==RESULT_COPY_STARTED)
        {
          return RESULT_COPY_STARTED;
        }
      } else {

	/* get password using the password dialog */
	password_dialog =
	  hildon_get_password_dialog_new(GTK_WINDOW
					 (app_ui_data->app), FALSE);
	gtk_window_set_modal(GTK_WINDOW(password_dialog), TRUE);

    global_password_dialog = password_dialog;
GDK_THR_ENTER;
	response = gtk_dialog_run(GTK_DIALOG(password_dialog));
	GDK_THR_LEAVE;
	pass = hildon_get_password_dialog_get_password
	             (HILDON_GET_PASSWORD_DIALOG(password_dialog));

    gboolean gpd;
    if ((gpd = (global_password_dialog == password_dialog)))
        global_password_dialog = 0;

	if (response != GTK_RESPONSE_OK) {
	  /* user cancelled the password dialog */
	  gtk_widget_destroy(password_dialog);
      if (gpd)
        ui_enable_document_open(app_ui_data, TRUE);

	  return RESULT_ENCRYPTED_FILE;
	}

	/* attempt to open pdf document with supplied password */
	result = pdf_viewer_open(filename, pass);
  if (result==RESULT_COPY_STARTED)
  {
    return RESULT_COPY_STARTED;
  }

	gtk_widget_destroy(password_dialog);

      }
      
      /* check if document was opened successfully */
      if (result == RESULT_LOAD_OK) {
	
	/* don't add documents to recent list from insecure place */
	if(!g_str_has_prefix(filename, "file:///var/tmp")
	   || !g_str_has_prefix(filename, "file:///tmp") ) {
	  ui_add_recent_item(app_ui_data, filename);
	}

	/* update the UI */
	ui_update(app_ui_data);

	break;
      } else {
        gtk_banner_close(GTK_WINDOW(app_ui_data->app));

        if(result != RESULT_ENCRYPTED_FILE) {
            /* we only need to show the info banner/note */
            break;
        } 
      }

      /* errEncrypted */
      if (retries == 0) {
	/* first round, display an infobanner that document is encrypted */
	ui_show_banner(GTK_WIDGET(app_ui_data->app),
		       _("pdfv_ib_encrypted_file"));
      } else {
	/* incorrect password */
	ui_show_banner(GTK_WIDGET(app_ui_data->app),
		       _("pdfv_ib_incorrect_password"));
      }
    
    } while (retries++ < MAX_PASSWORD_RETRIES);

    if (result != RESULT_LOAD_OK)
        ui_enable_document_open(app_ui_data, TRUE);

    return result;
}

typedef struct {
    AppUIData *app_ui_data;
    gchar *filename;
    gchar *password;
} tOpenDoc;

static gint idle_id = 0;

static gboolean idle_open(gpointer data)
{
    tOpenDoc *p = (tOpenDoc *)data;
    PDFViewerResult result;
    idle_id = 0;
    if (global_password_dialog) {
        GDK_THR_ENTER;
        gtk_dialog_response(GTK_DIALOG(global_password_dialog),
            GTK_RESPONSE_CANCEL);
        gtk_widget_hide(global_password_dialog);
        GDK_THR_LEAVE;
        global_password_dialog = 0;
    }
    result = _ui_open_document(p->app_ui_data, p->filename, p->password);
    if (result!=RESULT_COPY_STARTED)
    {
      GDK_THR_ENTER;
      ui_show_result(p->app_ui_data, result);
      GDK_THR_LEAVE;
    }
    return FALSE;
}

void idle_delete(gpointer data)
{
    tOpenDoc *p = (tOpenDoc *)data;
  if (!p->app_ui_data->copy_from_gw)
  {
    TDB("Freeing filename, password\n");
    g_free(p->filename);
    g_free(p->password);
    g_free(p);
  }
}

void ui_open_document(AppUIData   *app_ui_data, 
                 const gchar *filename,
                 const gchar *password)
{
  TDB("%s: %s %s\n",__FUNCTION__,filename,password);
    tOpenDoc *p = NULL;
  if (!app_ui_data->copy_from_gw)
  {
    p=g_new(tOpenDoc, 1);
    p->app_ui_data = app_ui_data;
    p->filename = g_strdup(filename);
    p->password = g_strdup(password);
    app_ui_data->open_document_structure=p;
  }
  else
  {
    p=app_ui_data->open_document_structure;
    app_ui_data->open_document_structure=NULL;
  }
    if (idle_id)
        g_source_remove(idle_id);
    idle_id = g_idle_add_full(G_PRIORITY_DEFAULT_IDLE, idle_open, p, idle_delete);
}


/**
	Runs switch to page dialog.
	
	@param app_ui_data Application UI data structure
	@return void
*/
void ui_run_switch_page_dialog(AppUIData * app_ui_data)
{
    GtkWidget *dialog, *vbox, *label, *number_editor;
    GtkWidget *caption_control = NULL;
    gchar     *text            = NULL;
    gint       result;
    gint       idle_id;
    gboolean  valid_number     = FALSE;

    gint min   = PDF_PAGE_INIT + 1,
         max   = pdf_viewer_get_num_pages(),
	 value = pdf_viewer_get_current_page();

    
    /* Create new stock dialog with ok and cancel buttons */
    app_ui_data->switch_page_dialog=dialog = gtk_dialog_new_with_buttons(_("pdfv_ti_switch_page_title"),
					 GTK_WINDOW(app_ui_data->app),
					 GTK_DIALOG_MODAL |
					 GTK_DIALOG_DESTROY_WITH_PARENT |
					 GTK_DIALOG_NO_SEPARATOR,
					 _("pdfv_bd_switch_page_ok"),
					 GTK_RESPONSE_OK,
					 _("pdfv_bd_switch_page_cancel"),
					 GTK_RESPONSE_CANCEL, NULL);
    g_assert(dialog);

    ossohelp_dialog_help_enable(GTK_DIALOG(dialog), PDFV_HELP_SWITCH_TO_PAGE,
				app_ui_data->app_data->osso_context);
    
    /* Disable resize, already modal */
    gtk_window_set_resizable(GTK_WINDOW(dialog), FALSE);
    vbox = GTK_DIALOG(dialog)->vbox;

    /* "Current page" label, 1st argument current page,
       2nd argument last page */
    text = g_strdup_printf(SUPPRESS_FORMAT_WARNING(_("pdfv_fi_switch_page_amount")), value, max);
    label = gtk_label_new(text);
    gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);
    g_free(text);
    
    gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, TRUE, 5);

    /* Number editor */
    number_editor = hildon_number_editor_new(min, max);
    g_assert(number_editor);
    
    /* Print custom message if user gives out-of-range value */ 
    g_object_set_data(G_OBJECT(number_editor), "idle-id", (gpointer)0);
    g_signal_connect(G_OBJECT(number_editor), "range-error", 
		     G_CALLBACK(on_number_editor_error), app_ui_data);
    

    hildon_number_editor_set_value(HILDON_NUMBER_EDITOR(number_editor),
				   value);

    caption_control = hildon_caption_new( NULL, 
					  _("pdfv_fi_switch_page_description"),
					  number_editor, 
					  NULL, 
					  HILDON_CAPTION_OPTIONAL);

    gtk_box_pack_start(GTK_BOX(vbox), caption_control, TRUE, TRUE, 5);

    /* Show all widgets and run dialog */
    gtk_widget_show_all(dialog);
    do {
      result = gtk_dialog_run(GTK_DIALOG(dialog));

      idle_id = (gint) g_object_get_data(G_OBJECT(number_editor), "idle-id");
      if (result == GTK_RESPONSE_OK) {
//        valid_number = FALSE;
        TDB(">>OK\n");
	valid_number = idle_id == 0;
	if (valid_number) {
          result =
            hildon_number_editor_get_value(HILDON_NUMBER_EDITOR
                (number_editor));
          /* Dialog was accepted, navigate to desired page */
          valid_number = ( result != pdf_viewer_get_current_page() );
	} else {
	  g_source_remove(idle_id);
	  g_object_set_data(G_OBJECT(number_editor), "idle-id", (gpointer)0);
          gtk_widget_grab_focus(GTK_WIDGET(number_editor));
        }
      } else {
	/* the number is not true, but button was cancel so we quit */
//	valid_number = TRUE;
        if (idle_id)
          g_source_remove(idle_id);
        break;
      }
    }
    while (valid_number == FALSE);
    /* else cancelled */
    gtk_widget_destroy(dialog);
    app_ui_data->switch_page_dialog=NULL;

    if (valid_number) {
//      ui_show_progress_banner(GTK_WINDOW(app_ui_data->app), _("pdfv_ib_opening"));
      pdf_viewer_navigate_page(result);
      ui_set_current_page(app_ui_data, pdf_viewer_get_current_page());
//      gtk_banner_close(GTK_WINDOW(app_ui_data->app));
    }
    
}

/**
	Runs details dialog.

	@todo How to display pdf document metainformation:
	title, author, subject, keywords. See UI spec 9.3.
	
	@param app_ui_data Application UI data structure
	@return void
*/
void ui_run_details_dialog(AppUIData * app_ui_data)
{
    GtkWidget   *dialog, *label;
    GnomeVFSURI *uri = pdf_viewer_get_uri();
    char        *uri_str = NULL;
    HildonFileSystemModel *model = NULL;
    GtkTreeIter  iter;

    gchar *info     = NULL,
          *title    = NULL,
          *author   = NULL,
          *subject  = NULL, 
          *keywords = NULL,
          *tmp      = NULL;
    
    GtkWidget *tblDetail;
    GtkWidget *lTitle;
    GtkWidget *lAuthor;
    GtkWidget *lSubject;
    GtkWidget *lKeywords;
    GtkWidget *lTitleTxt;
    GtkWidget *lAuthorTxt;
    GtkWidget *lSubjectTxt;
    GtkWidget *lKeywordsTxt;
    GtkWidget *lspace1;
    GtkWidget *lspace2;

    g_return_if_fail (app_ui_data != NULL);
    g_return_if_fail (app_ui_data->app != NULL);

    if (uri == NULL)
      return;

    /* create text uri from GnomeVFSURI */
    uri_str = gnome_vfs_uri_to_string(uri, GNOME_VFS_URI_HIDE_NONE);

    if(uri_str == NULL)
      return;

    /* retrieve document information */
    title    = pdf_viewer_get_info(DOCUMENT_DETAILS_TITLE);
    author   = pdf_viewer_get_info(DOCUMENT_DETAILS_AUTHOR);
    subject  = pdf_viewer_get_info(DOCUMENT_DETAILS_SUBJECT);
    keywords = pdf_viewer_get_info(DOCUMENT_DETAILS_KEYWORDS);

    /* replace null strings with empty ones */
    if (!title)
        title = g_strdup(" ");
    if (!author)
        author = g_strdup(" ");
    if (!subject)
        subject = g_strdup(" ");
    if (!keywords)
        keywords = g_strdup(" ");

    tblDetail = gtk_table_new (6, 2, FALSE);
    gtk_widget_show (tblDetail);
    gtk_table_set_col_spacings (GTK_TABLE (tblDetail), HILDON_MARGIN_DEFAULT);

    gtk_container_set_border_width(GTK_CONTAINER(tblDetail),
                        HILDON_MARGIN_DEFAULT);

    tmp = g_strdup_printf("%s: ", _("pdfv_fi_document_details_title"));
    lTitle = gtk_label_new (tmp);
    g_free(tmp);
    gtk_widget_show (lTitle);
    gtk_table_attach (GTK_TABLE (tblDetail), lTitle, 0, 1, 1, 2,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_FILL), 0, 0);
    gtk_misc_set_alignment (GTK_MISC (lTitle), 1.0, 0.0);

    tmp = g_strdup_printf("%s: ", _("pdfv_fi_document_details_author"));
    lAuthor = gtk_label_new (tmp);
    g_free(tmp);
    gtk_widget_show (lAuthor);
    gtk_table_attach (GTK_TABLE (tblDetail), lAuthor, 0, 1, 2, 3,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_FILL), 0, 0);
    gtk_misc_set_alignment (GTK_MISC (lAuthor), 1.0, 0.0);

    tmp = g_strdup_printf("%s: ", _("pdfv_fi_document_details_subject"));
    lSubject = gtk_label_new (tmp);
    g_free(tmp);
    gtk_widget_show (lSubject);
    gtk_table_attach (GTK_TABLE (tblDetail), lSubject, 0, 1, 3, 4,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_FILL), 0, 0);
    gtk_misc_set_alignment (GTK_MISC (lSubject), 1.0, 0.0);

    tmp = g_strdup_printf("%s: ", _("pdfv_fi_document_details_keywords"));
    lKeywords = gtk_label_new (tmp);
    g_free(tmp);
    gtk_widget_show (lKeywords);
    gtk_table_attach (GTK_TABLE (tblDetail), lKeywords, 0, 1, 4, 5,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_FILL), 0, 0);
    gtk_misc_set_alignment (GTK_MISC (lKeywords), 1.0, 0.0);

    lTitleTxt = gtk_label_new (title);
    gtk_widget_show (lTitleTxt);
    gtk_table_attach (GTK_TABLE (tblDetail), lTitleTxt, 1, 2, 1, 2,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_FILL), 0, 0);
    gtk_label_set_line_wrap (GTK_LABEL (lTitleTxt), TRUE);
    gtk_misc_set_alignment (GTK_MISC (lTitleTxt), 0, 0.0);
    
    lAuthorTxt = gtk_label_new (author);
    gtk_widget_show (lAuthorTxt);
    gtk_table_attach (GTK_TABLE (tblDetail), lAuthorTxt, 1, 2, 2, 3,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_FILL), 0, 0);
    gtk_label_set_line_wrap (GTK_LABEL (lAuthorTxt), TRUE);
    gtk_misc_set_alignment (GTK_MISC (lAuthorTxt), 0, 0.0);

    lSubjectTxt = gtk_label_new (subject);
    gtk_widget_show (lSubjectTxt);
    gtk_table_attach (GTK_TABLE (tblDetail), lSubjectTxt, 1, 2, 3, 4,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_FILL), 0, 0);
    gtk_label_set_line_wrap (GTK_LABEL (lSubjectTxt), TRUE);
    gtk_misc_set_alignment (GTK_MISC (lSubjectTxt), 0, 0.0);

    lKeywordsTxt = gtk_label_new (keywords);
    gtk_widget_show (lKeywordsTxt);
    gtk_table_attach (GTK_TABLE (tblDetail), lKeywordsTxt, 1, 2, 4, 5,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_FILL), 0, 0);
    gtk_label_set_line_wrap (GTK_LABEL (lKeywordsTxt), TRUE);
    gtk_misc_set_alignment (GTK_MISC (lKeywordsTxt), 0, 0.0);

    lspace1 = gtk_label_new ("");
    gtk_widget_show (lspace1);
    gtk_table_attach (GTK_TABLE (tblDetail), lspace1, 0, 1, 0, 1,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_EXPAND), 0, 0);
    gtk_misc_set_alignment (GTK_MISC (lspace1), 0, 0.5);

    lspace2 = gtk_label_new ("");
    gtk_widget_show (lspace2);
    gtk_table_attach (GTK_TABLE (tblDetail), lspace2, 0, 1, 5, 6,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_EXPAND), 0, 0);
    gtk_misc_set_alignment (GTK_MISC (lspace2), 0, 0.5);

    /* TODO construct temporary information output */
/*    info = g_strdup_printf("%s: %s\n%s: %s\n%s: %s\n%s: %s",
			   _("pdfv_fi_document_details_title"), title,
			   _("pdfv_fi_document_details_author"), author,
			   _("pdfv_fi_document_details_subject"), subject,
			   _("pdfv_fi_document_details_keywords"),
			   keywords);*/

    g_free(title);
    g_free(author);
    g_free(subject);
    g_free(keywords);
    
    /* get HildonFileSystemModel */
    model = get_file_system_model(GTK_WIDGET(app_ui_data->app));

    if(!model){
      goto done;
    }
    
    /* load the uri and set the GtkTreeIter */
    if( !hildon_file_system_model_load_uri(model, uri_str, &iter) ) {
      ui_show_banner(GTK_WIDGET(app_ui_data->app), _("pdfv_ib_menu_not_available"));
      goto done;
    }
    
    /* create file details dialog */
    app_ui_data->details_dialog=dialog = hildon_file_details_dialog_new_with_model
                        (GTK_WINDOW(app_ui_data->app), model);
    
    if(!dialog)
      goto done;

    gtk_window_set_modal(GTK_WINDOW(dialog), TRUE);
    hildon_file_details_dialog_set_file_iter
      (HILDON_FILE_DETAILS_DIALOG(dialog), &iter);
    
/*    label = gtk_label_new(info);
    gtk_label_set_line_wrap(label, TRUE);*/
    
    g_object_set( G_OBJECT(dialog),
		  "title", _("pdfv_me_menu_document_details"),
		  "show_tabs", TRUE,
		  "additional-tab", tblDetail,
		  "enable-read-only-checkbox", FALSE,
		  NULL );

    gtk_dialog_run(GTK_DIALOG(dialog));
    gtk_widget_destroy(dialog);
    app_ui_data->details_dialog=NULL;
    
 done:
    if(model) g_object_unref(model);
    g_free(uri_str);
//    g_free(info);
}


/**
	Toggles fullscreen mode and updates user interface accordingly,
	including main menu and context sensitive menu toggles.


	@param app_ui_data Application UI data structure
	@param fullscreen TRUE if switch to fullscreen, FALSE if normal
	@return void
*/
void ui_toggle_fullscreen(AppUIData * app_ui_data, gboolean fullscreen)
{
    GtkWidget *widget;
    GtkAction *action;
    gboolean   active;
    
    g_return_if_fail(app_ui_data != NULL);
    g_return_if_fail(app_ui_data->ui_manager != NULL);

    /* attempt to retrieve the full screen menu item widget */
    widget = gtk_ui_manager_get_widget(app_ui_data->ui_manager,
				       ACTION_PATH_FULL_SCREEN);

    g_return_if_fail(widget != NULL);

    /* exit if full screen menu item widget is insensitive */
    if (GTK_WIDGET_SENSITIVE(widget) == FALSE) {
	return;
    }

    /* attempt to retrieve the full screen menu action */
    action =
	gtk_ui_manager_get_action(app_ui_data->ui_manager,
				  ACTION_PATH_FULL_SCREEN);

    g_assert(action != NULL);

    /* retrieve active state of the menu toggle
       only enable or disable fullscreen mode if necessary */
    active = gtk_toggle_action_get_active(GTK_TOGGLE_ACTION(action));
    if (active != fullscreen) {
	/* Toggle fullscreen state. The actual fullscreen mode setting
	   is performed via a callback in ui_set_fullscreen(). */
	gtk_toggle_action_set_active(GTK_TOGGLE_ACTION(action), !active);
    }
}


/**
	Sets fullscreen mode on or off.

	NOTE this function must _only_ be called from the on_screen_full_screen()
	callback. All other users, please use ui_toggle_fullscreen() instead.

	@param app_ui_data Application UI data structure
	@param fullscreen TRUE if switch to fullscreen, FALSE if normal
	@return void
*/
void ui_set_fullscreen(AppUIData * app_ui_data, gboolean fullscreen)
{
   if (PDF_FLAGS_IS_SET(app_ui_data->flags, PDF_FLAGS_FULLSCREEN) != fullscreen){
	PDF_FLAGS_TOGGLE(app_ui_data->flags, PDF_FLAGS_FULLSCREEN);
  pdf_viewer_move_after_fullscreen_togle();
	hildon_appview_set_fullscreen(app_ui_data->app_view, fullscreen);
	/* hide toolbars when going fullscreen */
	if (fullscreen) {
	    gtk_widget_hide_all(GTK_WIDGET(app_ui_data->toolbar));
	} else {
	    gtk_widget_show_all(GTK_WIDGET(app_ui_data->toolbar));
	}

	/* 
	   enable or disable scrollbars as necessary
	   this is done via the pfdviewer component 
	*/
  
	pdf_viewer_toggle_fullscreen();
    }
}


/**
	Updates current page label number.
	Gets current page number from pdfviewer.

	@param app_ui_data AppUIData structure
	@return void
*/
void ui_update_current_page(AppUIData * app_ui_data)
{
    ui_set_current_page(app_ui_data, pdf_viewer_get_current_page());
}


/**
	Opens recent item

	@param app_ui_data AppUIData structure
	@param index recent item index
	@return void
*/
void ui_open_recent_item(AppUIData * app_ui_data, gint index)
{
    PDFViewerResult result;
    GSList *iter;
    RecentItem *recent_item;

    g_assert((index > 0) && (index <= MAX_RECENT_ITEMS));
    iter = g_slist_nth(app_ui_data->app_data->recent_list, index - 1);
    g_assert(iter);
    recent_item = (RecentItem *) iter->data;
    
    /* attempt to open document */

    ui_open_document(app_ui_data, recent_item->uri, NULL);
}


/**
   Update every menu/toolbar item on the UI
   according to the application's state

   @param AppUIData
*/
void ui_update (AppUIData *app_ui_data)
{
  PDFViewerState cur_state;

  gchar *file       = NULL;
  gchar *title      = NULL;

  /* check input */
  g_return_if_fail(app_ui_data != NULL);
  g_return_if_fail(app_ui_data->app_data != NULL);

  /* get application state */
  cur_state = app_ui_data->app_data->state;

  if(cur_state == PDF_VIEWER_STATE_EMPTY) {
    /* set the title empty */
    hildon_appview_set_title(HILDON_APPVIEW(app_ui_data->app_view), "");
    
    /* set page numbers */
    ui_set_current_page(app_ui_data, PDF_PAGE_INIT);
    ui_set_total_pages (app_ui_data, PDF_PAGE_INIT);
    
    /* disable page controls */
    ui_enable_page_controls(app_ui_data, DIM_ALL, FALSE);

    /* disable scrolling/panning */
    gtk_layout_set_size(GTK_LAYOUT(app_ui_data->layout), 1, 1);
    ui_enable_scrollbars(app_ui_data, FALSE, FALSE);
    
  } else {
    /* document was successfully opened, update user interface */
    ui_set_current_page(app_ui_data,
			pdf_viewer_get_current_page());
    ui_set_total_pages(app_ui_data, pdf_viewer_get_num_pages());
    
    /* update recent files list */
    ui_update_recent_menu(app_ui_data);

    /* get the newly opened document's location */
    file = gnome_vfs_uri_to_string(pdf_viewer_get_uri(),
				   GNOME_VFS_URI_HIDE_NONE);
    
    /* get the base name (w/o path) */
    title = get_basename_for_display(file);
    g_free(file);

    /* lower every char in title */
    file = g_ascii_strdown(title, -1);

    /* check if it has .pdf extension */
    if (g_str_has_suffix(file, PDFV_FILE_EXTENSION_DOT)) {
      title[strlen(title)-strlen(PDFV_FILE_EXTENSION_DOT)] = '\0';
    }
    g_free(file);
    
    /* update application title */
    hildon_appview_set_title(HILDON_APPVIEW(app_ui_data->app_view), title);
    g_free(title);
    	
    /* enable document menu items */
    //ui_enable_document_controls(app_ui_data, TRUE);
  }
}

/*******************************************************************************
 **** Private functions
 **/


/**
	Registers any custom actions.
	These actions are not created automatically using
	gtk_action_group_add_actions().
	
	@param actions Initialized action group
	@param app_ui_data AppUIData structure
	@return void
*/
static void add_custom_actions(GtkActionGroup * actions, AppUIData * app_ui_data)
{
    GtkAction *action;

    g_assert(actions);

    /* Create new custom action and add it to action group */
    action = gtk_action_new(ACTION_PAGE_SWITCH_TO,
			    _(ACTION_PAGE_SWITCH_TO), NULL, NULL);
    g_signal_connect(G_OBJECT(action),
		     "activate",
		     G_CALLBACK(on_page_switch_to), app_ui_data);
    
    /* g_object_set(action, "sensitive", FALSE, NULL); */
    gtk_action_group_add_action(actions, GTK_ACTION(action));
}


/**
	Builds main toolbar.
	Uses ui manager created toolbar as a basis and adds
	any custom widgets to it.

	AppUIData ui_manager needs to be initialized.
	
	@param ui_manager UI manager
	@param actions Initialized action group
	@return initialized toolbar if successfull; NULL otherwise
*/
static GtkToolbar *build_toolbar(GtkActionGroup * actions,
				 AppUIData * app_ui_data)
{
    GtkToolbar *toolbar = NULL;
    GtkToolItem *tool_item, *aligner;
    GtkWidget *button, *label, *image, *box;
    GtkAction *action;
    gint i, nitems;

    /* Get toolbar widget from ui manager */
    toolbar =
	GTK_TOOLBAR(gtk_ui_manager_get_widget
		    (app_ui_data->ui_manager, "/ToolBar"));

    //GTK_WIDGET_UNSET_FLAGS (, GTK_CAN_FOCUS);
    if (!toolbar)
	return NULL;

    /* Set toolbar properties */
    gtk_toolbar_set_orientation(toolbar, GTK_ORIENTATION_HORIZONTAL);
    gtk_toolbar_set_style(toolbar, GTK_TOOLBAR_ICONS);

    /* Set GtkToolButtons unfocusable */
    nitems = gtk_toolbar_get_n_items(toolbar);
    for (i = 0; i < nitems; i++) {
	tool_item = gtk_toolbar_get_nth_item(toolbar, i);
        gtk_tool_item_set_homogeneous(tool_item, FALSE);
	if (!GTK_IS_SEPARATOR_TOOL_ITEM(tool_item)) {
/*	    gtk_tool_item_set_expand(tool_item, TRUE);*/
	    GTK_WIDGET_UNSET_FLAGS(gtk_bin_get_child(GTK_BIN(tool_item)),
				   GTK_CAN_FOCUS);

	    gtk_tool_item_set_expand(tool_item, FALSE);
	    /* use icon theme
	       TODO there has to be a more elegant way to
	       bind GtkActionEntries with GtkIconTheme */
	    if (GTK_IS_TOOL_BUTTON(tool_item)
		&& (gtk_tool_button_get_stock_id(GTK_TOOL_BUTTON(tool_item))
		    != NULL)) {
		/* for each tool button:
		   1. get custom stock id as defined in ui.h GtkActionEntry
		   2. create a new image widget using an
		   icon theme aware function and stock id
		   3. set (replace) image widget with new image */

		image =
		    gtk_image_new_from_icon_name
		    (gtk_tool_button_get_stock_id(GTK_TOOL_BUTTON(tool_item)),
		     HILDON_ICON_SIZE_26);
		if (image != NULL) {
		    gtk_widget_show(image);
		    gtk_tool_button_set_icon_widget(GTK_TOOL_BUTTON
						    (tool_item), image);
		}

	    }
	} else {
	  gtk_tool_item_set_expand(tool_item, TRUE);
      gtk_tool_item_set_homogeneous(tool_item, FALSE);
	  if ((i != TOOLBAR_POS_VISIBLE_SEPARATOR_A)
	      && (i != TOOLBAR_POS_VISIBLE_SEPARATOR_B)
	      && (i != TOOLBAR_POS_VISIBLE_SEPARATOR_C)) {
	    gtk_separator_tool_item_set_draw(
		  GTK_SEPARATOR_TOOL_ITEM(tool_item), FALSE );
	  }
	}
    }

    /* Create "current page" widget */
    tool_item = gtk_tool_item_new();
    gtk_tool_item_set_expand(tool_item, FALSE);
    gtk_tool_item_set_homogeneous(tool_item, FALSE);
    button = gtk_button_new();
    GTK_WIDGET_UNSET_FLAGS(button, GTK_CAN_FOCUS);

    /* set minimum horizontal size for button to avoid other
       toolbar buttons moving around with longer page numbers */
    gtk_widget_set_size_request(button, CURRENT_PAGE_BUTTON_MIN_WIDTH, -1);
    gtk_button_set_alignment(GTK_BUTTON(button), (gfloat)1.0, (gfloat)0.5);
    gtk_button_set_relief(GTK_BUTTON(button), GTK_RELIEF_NONE);
    label = gtk_label_new(NULL);
    gtk_container_add(GTK_CONTAINER(button), label);
    gtk_container_add(GTK_CONTAINER(tool_item), button);
    gtk_toolbar_insert(GTK_TOOLBAR(toolbar),
		       tool_item, TOOLBAR_POS_CURRENT_PAGE_WIDGET);
    gtk_label_set_width_chars(GTK_LABEL(label), 4);
    
    /* with newer GTK, just use action's short-label */
    // app_ui_data->current_page = label;

    app_ui_data->current_page_item = GTK_WIDGET(tool_item);
    app_ui_data->current_page_button = GTK_WIDGET(button);

    /* Get previously created custom action and bind it to custom widget */
    action = gtk_action_group_get_action(actions, ACTION_PAGE_SWITCH_TO);
    app_ui_data->current_page = action;
    g_assert(action);
    gtk_action_connect_proxy(action, button);

    /* Create "total pages" widget */
    tool_item = gtk_tool_item_new();
    label = gtk_label_new(NULL);
    gtk_container_add(GTK_CONTAINER(tool_item), label);
    gtk_toolbar_insert(GTK_TOOLBAR(toolbar), tool_item,
		       TOOLBAR_POS_CURRENT_PAGE_WIDGET + 1);
    app_ui_data->total_pages = label;
    gtk_label_set_width_chars(GTK_LABEL(label), 6);
//    gtk_widget_set_size_request(label, CURRENT_PAGE_BUTTON_MIN_WIDTH, -1);

    /* align it to the right */
/*    aligner = gtk_separator_tool_item_new();
    gtk_tool_item_set_expand(aligner, TRUE);
    gtk_separator_tool_item_set_draw(GTK_SEPARATOR_TOOL_ITEM(aligner), FALSE);

    gtk_toolbar_insert(GTK_TOOLBAR(toolbar),
                       aligner,
		       TOOLBAR_POS_CURRENT_PAGE_WIDGET);*/
 
    /* Set initial page numbers */
    ui_set_current_page(app_ui_data, PDF_PAGE_INIT);
    ui_set_total_pages(app_ui_data, PDF_PAGE_INIT);

    /* create Current Zoom widget */
    tool_item = gtk_tool_item_new();
//    gtk_widget_set_size_request(tool_item, 80, -1);
    gtk_tool_item_set_homogeneous(tool_item, FALSE);
    gtk_tool_item_set_expand(tool_item, FALSE);
    box = gtk_hbox_new(FALSE, 0);
    label = gtk_label_new(NULL);
    /* width of the widget */
//    gtk_label_set_width_chars(GTK_LABEL(label), CURRENT_ZOOM_WIDGET_CHARS);
    gtk_box_pack_end (GTK_BOX (box), label, TRUE, FALSE, 0);
    gtk_label_set_single_line_mode (GTK_LABEL (label), TRUE);
    gtk_widget_set_size_request(GTK_WIDGET(box), 80, -1);

    gtk_container_add(GTK_CONTAINER(tool_item), box);
    gtk_toolbar_insert(GTK_TOOLBAR(toolbar),
		       tool_item,
		       TOOLBAR_POS_CURRENT_ZOOM_WIDGET);

    app_ui_data->current_zoom_value = label;
    app_ui_data->current_zoom_item = tool_item;

    
/*    aligner = gtk_separator_tool_item_new();
    gtk_separator_tool_item_set_draw(GTK_SEPARATOR_TOOL_ITEM(aligner), FALSE);
    gtk_toolbar_insert(GTK_TOOLBAR(toolbar),
                       aligner,
		       TOOLBAR_POS_CURRENT_ZOOM_WIDGET);*/

    /* show images button widget */
    app_ui_data->image_button = 
      GTK_WIDGET(gtk_toolbar_get_nth_item(toolbar, TOOLBAR_POS_SHOW_IMAGES));

    /*create images for dynamic toolbar */
    app_ui_data->img_images_on  = 
      gtk_image_new_from_icon_name( "qgn_indi_gene_images_attachment",
				    HILDON_ICON_SIZE_26);
    g_object_ref( app_ui_data->img_images_on );
    
    app_ui_data->img_images_off = 
      gtk_image_new_from_icon_name( "qgn_indi_gene_images_attachment_purged",
				    HILDON_ICON_SIZE_26);

    g_object_ref( app_ui_data->img_images_off );
    return toolbar;
}

/**
	Builds application area.
	Application area consists of ScrolledWindow, Layout and Image.
	Ensures that application area is built only once.
	
	@param app HildonApp
	@return void
*/
static void build_application_area(AppUIData * app_ui_data)
{
    GtkWidget *scrolled_window, *layout, *page_image;
    GtkWidget *popup = NULL;
    GtkAdjustment *vadj, *hadj;

    g_assert(app_ui_data);

    /* Application area: ScrolledWindow - Layout - Image */
    scrolled_window = gtk_table_new(2, 2, FALSE);

    g_assert(scrolled_window);

    gtk_container_add(GTK_CONTAINER(app_ui_data->app_view),
		      scrolled_window);

    /* It is crucial to set size for the layout, otherwise
       scrolledwindow cannot calculate scrollbar adjustments */
    layout = gtk_layout_new(NULL, NULL);
    g_assert(layout);
    gtk_container_set_border_width(GTK_CONTAINER(layout), 0);
    
    /* moved gtk_widget_add_events and g_signal_connects
       immediately after creating GtkLayout, otherwise
       button_release_event mystically disappears */

    /* Add events for panning */
    gtk_widget_add_events(layout,
			  (GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK
			   | GDK_BUTTON_MOTION_MASK |
			   GDK_POINTER_MOTION_MASK |
			   GDK_POINTER_MOTION_HINT_MASK));

    g_signal_connect(G_OBJECT(layout), "event",
		     G_CALLBACK(on_screen_event), app_ui_data);

    g_signal_connect(G_OBJECT(layout), "button_press_event",
		     G_CALLBACK(on_screen_press), app_ui_data);

    g_signal_connect(G_OBJECT(layout), "button_release_event",
		     G_CALLBACK(on_screen_release), app_ui_data);

    g_signal_connect(G_OBJECT(layout), "motion_notify_event",
		     G_CALLBACK(on_screen_motion), app_ui_data);


    /* setup popup menu 
       TODO: check if this is OK
    */
    popup =
      gtk_ui_manager_get_widget(app_ui_data->ui_manager, "/Popup");
    
    if (popup) {
/*      gtk_widget_tap_and_hold_setup(layout, popup, NULL, 
				    GTK_TAP_AND_HOLD_PASS_PRESS);*/
        g_signal_connect_after((gpointer) layout, "tap-and-hold",
               G_CALLBACK(on_screen_tap_and_hold_event), popup);
        gtk_widget_tap_and_hold_setup(layout, NULL, NULL,
               GTK_TAP_AND_HOLD_PASS_PRESS);
      gtk_menu_attach_to_widget(popup,layout,NULL);
    }
    

    /* TODO: check if needed
       g_signal_connect( G_OBJECT(layout), "size_allocate",
       G_CALLBACK( ), app_ui_data ); */


    /* attaching the layout to the GtkTable */
    gtk_table_attach(GTK_TABLE(scrolled_window), layout, 0, 1, 0, 1,
		     GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 0, 0);

    /* creating scrollbars, attaching to the GtkTable */
    app_ui_data->vscroll = gtk_vscrollbar_new(NULL);
    gtk_table_attach(GTK_TABLE(scrolled_window), app_ui_data->vscroll, 1,
		     2, 0, 1, GTK_FILL | GTK_SHRINK, GTK_FILL | GTK_SHRINK,
		     0, 0);
    app_ui_data->hscroll = gtk_hscrollbar_new(NULL);
    gtk_table_attach(GTK_TABLE(scrolled_window), app_ui_data->hscroll, 0,
		     1, 1, 2, GTK_FILL | GTK_SHRINK, GTK_FILL | GTK_SHRINK,
		     0, 0);

    /* associating the scrollbars with layout's adjustments */
    vadj = gtk_layout_get_vadjustment(GTK_LAYOUT(layout));
    gtk_range_set_adjustment(GTK_RANGE(app_ui_data->vscroll), vadj);
    hadj = gtk_layout_get_hadjustment(GTK_LAYOUT(layout));
    gtk_range_set_adjustment(GTK_RANGE(app_ui_data->hscroll), hadj);

    g_object_set(vadj, "step-increment", HSCROLLBAR_INCREMENT, NULL);
    g_object_set(hadj, "step-increment", VSCROLLBAR_INCREMENT, NULL);

    /* adding signal events for scrollbars' value change 
       - for partial rendering */
    g_signal_connect(G_OBJECT(hadj), "value-changed",
		     G_CALLBACK(on_screen_scroll),
		     GUINT_TO_POINTER(SCROLL_HOR));
    g_signal_connect(G_OBJECT(vadj), "value-changed",
		     G_CALLBACK(on_screen_scroll),
		     GUINT_TO_POINTER(SCROLL_VER));

    /* TODO change to appropriate widget */
    page_image = 
      gtk_image_new_from_pixbuf(NULL);
      
    gtk_layout_put(GTK_LAYOUT(layout), page_image, 0, 0);

    gtk_layout_set_size(GTK_LAYOUT(layout),
			page_image->allocation.width + 1,
			page_image->allocation.height + 1);
    
    /* Store ui references to AppData */
    app_ui_data->scrolled_window = scrolled_window;
    app_ui_data->layout          = layout;
    app_ui_data->page_image      = page_image;
}

/**
	Sets current page label number

	@param app_ui_data AppUIData structure
	@param value current page
	@return void
*/
static void ui_set_current_page(AppUIData * app_ui_data, gint value)
{
    gchar *text = g_strdup_printf("%d", value);
    g_object_set(app_ui_data->current_page, "short-label", text, NULL);
    g_free(text);
}


/**
	Sets total pages label number

	@param app_ui_data AppUIData structure
	@param value total pages
	@return void
*/
static void ui_set_total_pages(AppUIData * app_ui_data, gint value)
{
    gchar *text = g_strdup_printf("/ %d", value);
    g_object_set(app_ui_data->total_pages, "label", text, NULL);
    g_free(text);
}


/**
	Update recent items menu

	@param app_ui_data AppUIData structure
	@return void
*/
void ui_update_recent_menu(AppUIData * app_ui_data)
{
    GSList    *iter = NULL;	// = app_ui_data->app_data->recent_list;
    GtkWidget *widget;
    GtkLabel  *label;
    gchar     *path, *basename, *str_label, *pc;
    gint       n;
    gint       nr_of_hidden_entries = 0;
    RecentItem *recent_item;

    for (n = 1; n <= MAX_RECENT_ITEMS; n++) {
      
	path =
	    g_strdup_printf
	    ("/MenuBar/pdfv_me_main_menu_recent/ignore_this_%d",
	     n);
	widget = gtk_ui_manager_get_widget(app_ui_data->ui_manager, path);
	g_free(path);
	if (!widget) {
	    OSSO_LOG_WARNING
		("ui_update_recent_menu failed, cannot access menu item %d",
		 n);
	    break;
	}
	if (app_ui_data->app_data->recent_list
	    && (iter =
		g_slist_nth(app_ui_data->app_data->recent_list, n - 1))) {

  	    recent_item = (RecentItem *)iter->data;
	    if( (basename = get_basename_for_display(recent_item->uri)) == NULL) {
	      return;
	    }
	    pc = g_utf8_strrchr(basename, -1, '.');
	    if (pc) {
	        // check the legality of strlen here!
	        str_label =
		  g_strndup(basename, strlen(basename) - strlen(pc));
	    } else
		str_label = g_strdup(basename);
	    g_free(basename);

	    /* truncate and add ellipsis to long entries */
	    if (strlen(str_label) > MAX_RECENT_ITEM_TITLE_LENGTH) {
		str_label[MAX_RECENT_ITEM_TITLE_LENGTH] = '\0';
		str_label[MAX_RECENT_ITEM_TITLE_LENGTH - 1] = '.';
		str_label[MAX_RECENT_ITEM_TITLE_LENGTH - 2] = '.';
		str_label[MAX_RECENT_ITEM_TITLE_LENGTH - 3] = '.';
	    }

	    /* set menuitem label */
	    label = GTK_LABEL(gtk_bin_get_child(GTK_BIN(widget)));
	    g_object_set(label, "use-underline", FALSE, NULL);
	    g_object_set(label, "label", str_label, NULL);
	    g_free(str_label);
	    gtk_widget_show(widget);
	} else {
	    gtk_widget_hide(widget);
	    nr_of_hidden_entries++;
	}
    }
    if (nr_of_hidden_entries == 6) {
	widget = gtk_ui_manager_get_widget(app_ui_data->ui_manager,
					   "/MenuBar/pdfv_me_main_menu_recent");
	if (widget != NULL)
	    gtk_widget_set_sensitive(widget, 0);
    } else {
	widget = gtk_ui_manager_get_widget(app_ui_data->ui_manager,
					   "/MenuBar/pdfv_me_main_menu_recent");
	if (widget != NULL)
	    gtk_widget_set_sensitive(widget, 1);
    }
}

/**
	Refreshes recent items menu on opening the menu

	@param app_ui_data AppUIData structure
	@return void
*/
void ui_refresh_recent_menu(AppUIData * app_ui_data)
{
  GSList      *list        = NULL;
  GnomeVFSResult vfs_res;
  RecentItem  *recent_item = NULL;
  list = app_ui_data->app_data->recent_list;
  while (list)
  {
    GnomeVFSFileInfo *file_info = gnome_vfs_file_info_new();
    recent_item = (RecentItem *)list->data;
    gchar *path = recent_item->uri;
    if (!g_str_has_prefix(path,"obex"))
    {
      vfs_res = gnome_vfs_get_file_info(path,
          file_info,
          GNOME_VFS_FILE_INFO_FOLLOW_LINKS);
      if (vfs_res == GNOME_VFS_OK ||
          file_info->type == GNOME_VFS_FILE_TYPE_REGULAR)
      {
      }
      else
      {
        app_ui_data->app_data->recent_list=g_slist_remove(app_ui_data->app_data->recent_list,recent_item);
        g_free(recent_item->uri);
        gnome_vfs_monitor_cancel(recent_item->monitor_handle);
        g_free(recent_item);
      }
    }
    else
    {
    }
    
    gnome_vfs_file_info_unref(file_info);
    list=g_slist_next(list);
  
  }
  ui_update_recent_menu(app_ui_data);
}


/**
	Add recent item to the beginning of the list

	@param app_ui_data AppUIData structure
	@param filename filename
	@return void
*/
static void ui_add_recent_item(AppUIData   *app_ui_data, 
			       const gchar *filename)
{
    GSList     *list = NULL, 
               *iter;
    RecentItem *recent_item = NULL;
    GnomeVFSResult vfs_result;

    list = app_ui_data->app_data->recent_list;

    if (list == NULL) {
      /* empty list adding it to the list */
      recent_item = g_new0(RecentItem, 1);
      recent_item->uri = g_strdup(filename);
      
      if (strncmp(recent_item->uri, "obex:", 5) == 0)
        vfs_result = GNOME_VFS_OK;
     else
        vfs_result = gnome_vfs_monitor_add(&(recent_item->monitor_handle),
					 filename,
					 GNOME_VFS_MONITOR_FILE,
					 on_file_changed,
					 app_ui_data->app_data);
      
      if (vfs_result == GNOME_VFS_OK)
	list = g_slist_append(list, recent_item);
      else
	OSSO_LOG_CRIT("Couldn't add monitor for file"
		      "not added to recent list!");
    } else {
      /* check if filename is already on the list */
      iter = g_slist_find_custom(list, filename, (GCompareFunc) recentcmp);
      if (iter) {
	/* found it, move it to top */
	recent_item  = (RecentItem *)iter->data;
	list = g_slist_remove(list, iter->data);
	list = g_slist_prepend(list, recent_item);
      } else {
	/* add filename to the beginning of the list */
	recent_item = g_new0(RecentItem, 1);
	recent_item->uri = g_strdup(filename);

        if (strncmp(recent_item->uri, "obex:", 5) == 0)
          vfs_result = GNOME_VFS_OK;
        else
          vfs_result = gnome_vfs_monitor_add(&(recent_item->monitor_handle),
					   filename,
					   GNOME_VFS_MONITOR_FILE,
					   on_file_changed,
					   app_ui_data->app_data);
      
	if (vfs_result == GNOME_VFS_OK)
	  list = g_slist_prepend(list, recent_item);
	else
	  OSSO_LOG_CRIT("Couldn't add monitor for file"
			"not added to recent list!");
	
	/* remove excess item if necessary */
	iter = g_slist_nth(list, MAX_RECENT_ITEMS);
	if (iter) {
	  recent_item = (RecentItem *)iter->data;
          if (recent_item->monitor_handle)
            gnome_vfs_monitor_cancel(recent_item->monitor_handle);
	
	  g_free(recent_item->uri);
	  
	  g_free(recent_item);
	  
	  list = g_slist_remove(list, iter->data);
	}
      }
    }
    
    /* update list head pointer */
    app_ui_data->app_data->recent_list = list;
}


/**
	Enables or disabled document controls.
	Currently, sets certain menu items sensitive or insensitive.

	@param app_ui_data AppUIData structure
	@return void
*/
void ui_enable_document_controls(AppUIData * app_ui_data, gboolean enable)
{
    GtkWidget *widget;
    GtkAction *action;
    static gboolean prev_enable = -1;

//    TDB("ui_enable_document_controls: %d\n", enable);

    if (enable == prev_enable) {
	/* primitive state caching */
	return;
    }

    /* use gtk_widget_set_sensitive instead of g_object_set
       Also, page controls (next, prev, zoom etc.) are now
       handled in ui_enable_page_controls() */

    /* 'Save as...' */
    widget = gtk_ui_manager_get_widget(app_ui_data->ui_manager,
				       "/MenuBar/pdfv_me_main_menu_document/pdfv_me_menu_document_save");
    if (widget != NULL)
	gtk_widget_set_sensitive(widget, enable);

    /* 'Send document via email' menu/popup */
    widget = gtk_ui_manager_get_widget(app_ui_data->ui_manager,
#ifdef  VERSION_3
    "/MenuBar/pdfv_me_main_menu_document/pdfv_me_menu_document_send_main");
#else
    "/MenuBar/pdfv_me_main_menu_document/pdfv_me_menu_document_send");
#endif
    if (widget != NULL)
	gtk_widget_set_sensitive(widget, enable);

    widget = gtk_ui_manager_get_widget(app_ui_data->ui_manager,
#ifdef  VERSION_3
				       "/Popup/pdfv_me_menu_document_send_main");
#else
				       "/Popup/pdfv_me_menu_document_send");
#endif    
    if (widget != NULL)
	gtk_widget_set_sensitive(widget, enable);
#ifdef  VERSION_3
    /* 'Send document via bluetooth' menu/popup */
/*    widget = gtk_ui_manager_get_widget(app_ui_data->ui_manager,
				       "/MenuBar/pdfv_me_main_menu_document/pdfv_me_menu_document_send_main/pdfv_me_menu_document_send_bluetooth");
    if (widget != NULL)
	gtk_widget_set_sensitive(widget, enable);

    widget = gtk_ui_manager_get_widget(app_ui_data->ui_manager,
				       "/Popup/pdfv_me_menu_document_send_main/pdfv_me_menu_document_send_bluetooth");
    if (widget != NULL)
	gtk_widget_set_sensitive(widget, enable);*/
#endif
    
    /* 'Document Details' */
    widget = gtk_ui_manager_get_widget(app_ui_data->ui_manager,
				       "/MenuBar/pdfv_me_main_menu_document/pdfv_me_menu_document_details");
    if (widget != NULL)
	gtk_widget_set_sensitive(widget, enable);

    widget = gtk_ui_manager_get_widget(app_ui_data->ui_manager,
				       "/Popup/pdfv_me_menu_document_details");
    if (widget != NULL)
	gtk_widget_set_sensitive(widget, enable);

    /* 'Fit to page' */
    widget = gtk_ui_manager_get_widget(app_ui_data->ui_manager,
				       "/MenuBar/pdfv_me_main_menu_view/pdfv_me_menu_screen_zoom"
				       "/pdfv_me_menu_screen_zoom_page");
    if (widget != NULL)
	gtk_widget_set_sensitive(widget, enable);

    /* 'Fit to width' */
    widget = gtk_ui_manager_get_widget(app_ui_data->ui_manager,
				       "/MenuBar/pdfv_me_main_menu_view/pdfv_me_menu_screen_zoom"
				       "/pdfv_me_menu_screen_zoom_width");
    if (widget != NULL)
	gtk_widget_set_sensitive(widget, enable);

    /* Show/hide images */
    widget = gtk_ui_manager_get_widget(app_ui_data->ui_manager,
				       "/MenuBar/pdfv_me_main_menu_view"
				       "/pdfv_me_menu_screen_show_images");
    if (widget != NULL)
	gtk_widget_set_sensitive(widget, enable);

    widget = gtk_ui_manager_get_widget(app_ui_data->ui_manager,
				       "/ToolBar/"
				       "pdfv_me_menu_screen_show_images");
    if (widget != NULL)
	gtk_widget_set_sensitive(widget, enable);

}

void ui_enable_document_open(AppUIData * app_ui_data, gboolean enable)
{
    GtkWidget *widget;

    /* Recent menu */
    /*widget = gtk_ui_manager_get_widget(app_ui_data->ui_manager,
                       "/MenuBar/pdfv_me_main_menu_recent");
    if (widget != NULL)
        gtk_widget_set_sensitive(widget, enable);*/
  /*Enable recent, but only if it is not empty */
  app_ui_data->rendering=FALSE;
  on_recent(NULL,app_ui_data);

    widget=gtk_ui_manager_get_widget(app_ui_data->ui_manager,
                         "/MenuBar/pdfv_me_main_menu_document/"
                         "pdfv_me_menu_document_open");
    gtk_widget_set_sensitive(widget, enable);
    widget=gtk_ui_manager_get_widget(app_ui_data->ui_manager,
                          "/ToolBar/"
                          "pdfv_me_menu_document_open");
    gtk_widget_set_sensitive(widget, enable);
}

/**
   Adds callback functions for every UI item, which can be dimmed

   @param AppUIData
*/
static void ui_set_callbacks_for_dimmed_menu_items(AppUIData * app_ui_data)
{
    g_assert(app_ui_data != NULL);

    ui_set_dimmed_menu_item_callback(app_ui_data,
				     "/ToolBar/pdfv_me_menu_document_open",
				     insensitive_press_menu_default,
				     app_ui_data);

    g_signal_connect(G_OBJECT(app_ui_data->current_page_button), "insensitive_press",
	     G_CALLBACK(insensitive_press_menu_default), app_ui_data);

    ui_set_dimmed_menu_item_callback(app_ui_data,
				     "/MenuBar/pdfv_me_main_menu_document/"
				     "pdfv_me_menu_document_save",
				     insensitive_press_menu_document_save,
				     app_ui_data);

    ui_set_dimmed_menu_item_callback(app_ui_data,
				     "/MenuBar/pdfv_me_main_menu_document/pdfv_me_menu_document_details",
				     insensitive_press_menu_document_details,
				     app_ui_data);

    /* document send */
    ui_set_dimmed_menu_item_callback(app_ui_data,
#ifdef  VERSION_3
				     "/MenuBar/pdfv_me_main_menu_document/pdfv_me_menu_document_send_main/pdfv_me_menu_document_send",
#else
"/MenuBar/pdfv_me_main_menu_document/pdfv_me_menu_document_send",
#endif
				     insensitive_press_menu_document_send,
				     app_ui_data);
    
    ui_set_dimmed_menu_item_callback(app_ui_data,
#ifdef  VERSION_3
				     "/Popup/pdfv_me_menu_document_send_main/pdfv_me_menu_document_send",
#else
				     "/Popup/pdfv_me_menu_document_send",
#endif
				     insensitive_press_menu_document_send,
				     app_ui_data);
#ifdef  VERSION_3
    ui_set_dimmed_menu_item_callback(app_ui_data,
				     "/MenuBar/pdfv_me_main_menu_document/pdfv_me_menu_document_send_main/pdfv_me_menu_document_send_bluetooth",
				     insensitive_press_menu_document_send,
				     app_ui_data);
    
    ui_set_dimmed_menu_item_callback(app_ui_data,
				     "/Popup/pdfv_me_menu_document_send_main/pdfv_me_menu_document_send_bluetooth",
				     insensitive_press_menu_document_send,
				     app_ui_data);

    ui_set_dimmed_menu_item_callback(app_ui_data,
				     "/MenuBar/pdfv_me_main_menu_document/pdfv_me_menu_document_send_main",
				     insensitive_press_menu_document_send,
				     app_ui_data);
    
    ui_set_dimmed_menu_item_callback(app_ui_data,
				     "/Popup/pdfv_me_menu_document_send_main",
				     insensitive_press_menu_document_send,
				     app_ui_data);
#endif
    /* prev */
    ui_set_dimmed_menu_item_callback(app_ui_data,
				     "/MenuBar/pdfv_me_main_menu_page/pdfv_me_menu_page_previous",
				     insensitive_press_menu_page_previous,
				     app_ui_data);

    ui_set_dimmed_menu_item_callback(app_ui_data,
				     "/ToolBar/pdfv_me_menu_page_previous",
				     insensitive_press_menu_page_previous,
				     app_ui_data);
    

    ui_set_dimmed_menu_item_callback(app_ui_data,
				     "/Popup/pdfv_me_menu_page_previous",
				     insensitive_press_menu_page_previous,
				     app_ui_data);
    /* first */
    ui_set_dimmed_menu_item_callback(app_ui_data,
				     "/MenuBar/pdfv_me_main_menu_page/pdfv_me_menu_page_first",
				     insensitive_press_menu_page_first,
				     app_ui_data);

    ui_set_dimmed_menu_item_callback(app_ui_data,
				     "/ToolBar/pdfv_me_menu_page_first",
				     insensitive_press_menu_page_first,
				     app_ui_data);

    /* next */
    ui_set_dimmed_menu_item_callback(app_ui_data,
				     "/MenuBar/pdfv_me_main_menu_page/pdfv_me_menu_page_next",
				     insensitive_press_menu_page_next,
				     app_ui_data);

    ui_set_dimmed_menu_item_callback(app_ui_data,
				     "/ToolBar/pdfv_me_menu_page_next",
				     insensitive_press_menu_page_next,
				     app_ui_data);

    ui_set_dimmed_menu_item_callback(app_ui_data,
				     "/Popup/pdfv_me_menu_page_next",
				     insensitive_press_menu_page_next,
				     app_ui_data);

    /* last */
    ui_set_dimmed_menu_item_callback(app_ui_data,
				     "/MenuBar/pdfv_me_main_menu_page/pdfv_me_menu_page_last",
				     insensitive_press_menu_page_last,
				     app_ui_data);

    ui_set_dimmed_menu_item_callback(app_ui_data,
				     "/ToolBar/pdfv_me_menu_page_last",
				     insensitive_press_menu_page_last,
				     app_ui_data);

    /* switch */
    ui_set_dimmed_menu_item_callback(app_ui_data,
				     "/MenuBar/pdfv_me_main_menu_page/"
				     "pdfv_me_menu_page_switch_to",
				     insensitive_press_menu_default,
				     app_ui_data);
    /* zoom in */
    ui_set_dimmed_menu_item_callback(app_ui_data,
				     "/MenuBar/pdfv_me_main_menu_view/"
				     "pdfv_me_menu_screen_zoom/"
				     "pdfv_me_menu_screen_zoom_in",
				     insensitive_press_menu_screen_zoom_in,
				     app_ui_data);

    ui_set_dimmed_menu_item_callback(app_ui_data,
				     "/ToolBar/pdfv_me_menu_screen_zoom_in",
				     insensitive_press_menu_screen_zoom_in,
				     app_ui_data);
    
    ui_set_dimmed_menu_item_callback(app_ui_data,
				     "/Popup/pdfv_me_menu_screen_zoom_in",
				     insensitive_press_menu_screen_zoom_in,
				     app_ui_data);
    
    /* zoom out */
    ui_set_dimmed_menu_item_callback(app_ui_data,
				     "/MenuBar/pdfv_me_main_menu_view/"
				     "pdfv_me_menu_screen_zoom/"
				     "pdfv_me_menu_screen_zoom_out",
				     insensitive_press_menu_screen_zoom_out,
				     app_ui_data);

    ui_set_dimmed_menu_item_callback(app_ui_data,
				     "/ToolBar/pdfv_me_menu_screen_zoom_out",
				     insensitive_press_menu_screen_zoom_out,
				     app_ui_data);
    
    ui_set_dimmed_menu_item_callback(app_ui_data,
				     "/Popup/pdfv_me_menu_screen_zoom_out",
				     insensitive_press_menu_screen_zoom_out,
				     app_ui_data);
    
    /* fullscreen */
    ui_set_dimmed_menu_item_callback(app_ui_data,
				     "/MenuBar/pdfv_me_main_menu_view/"
				     "pdfv_me_menu_screen_full_screen",
				     insensitive_press_menu_default,
				     app_ui_data);

    /* fit to page */
    ui_set_dimmed_menu_item_callback(app_ui_data,
				     "/MenuBar/pdfv_me_main_menu_view/"
				     "pdfv_me_menu_screen_zoom/"
				     "pdfv_me_menu_screen_zoom_page",
				     insensitive_press_menu_default,
				     app_ui_data);

    /* fit to width */
    ui_set_dimmed_menu_item_callback(app_ui_data,
				     "/MenuBar/pdfv_me_main_menu_view/"
				     "pdfv_me_menu_screen_zoom/"
				     "pdfv_me_menu_screen_zoom_width",
				     insensitive_press_menu_default,
				     app_ui_data);

    /* show image */
    ui_set_dimmed_menu_item_callback(app_ui_data,
				     "/ToolBar/pdfv_me_menu_screen_show_images",
				     insensitive_press_menu_default,
				     app_ui_data);
    /* help */
    /*    ui_set_dimmed_menu_item_callback(app_ui_data,
				     "/MenuBar/pdfv_me_main_menu_tools/"
				     "pdfv_me_menu_tools_help",
				     insensitive_press_menu_default,
    			     app_ui_data);
    */
    /* recent menu */
    ui_set_dimmed_menu_item_callback(app_ui_data,
				     "/MenuBar/pdfv_me_main_menu_recent",
				     insensitive_press_menu_default,
				     app_ui_data);
    /* show images */
    ui_set_dimmed_menu_item_callback(app_ui_data,
				     "/MenuBar/pdfv_me_main_menu_view"
				     "/pdfv_me_menu_screen_show_images",
				     insensitive_press_menu_default,
				     app_ui_data);

    /* details */
    ui_set_dimmed_menu_item_callback(app_ui_data,
				     "/Popup/pdfv_me_menu_document_details",
				     insensitive_press_menu_document_details,
				     app_ui_data);

}

/**
   Connects "insensitive_press" signal to the given UI items

   @param AppUIData
   @param path of the tool item (see ui.h)
   @param callback function
   @param data passed to the callback function (appuidata)
*/
static void ui_set_dimmed_menu_item_callback(AppUIData * app_ui_data,
					     const gchar * path,
					     InsensitiveMenuPressCallback
					     callback, gpointer data)
{
    GtkWidget *item = NULL;

    g_assert(path != NULL && app_ui_data != NULL
    	     && app_ui_data->ui_manager != NULL);
    item = gtk_ui_manager_get_widget(app_ui_data->ui_manager, path);
    g_assert(item != NULL);
    g_signal_connect(G_OBJECT(item),
		     "insensitive_press", G_CALLBACK(callback), data);
}

/**
   Sets the defined Popup menu in ui.h for the application's layout
   
   @param AppUIData
*/
static void ui_setup_popup_menu(AppUIData * app_ui_data) 
{
    GtkWidget *popup = NULL;
    
    g_return_if_fail(app_ui_data != NULL);
    
/*    popup =
      gtk_ui_manager_get_widget(app_ui_data->ui_manager, "/Popup");
    
    if (popup) {
      gtk_widget_tap_and_hold_setup(app_ui_data->layout, popup, NULL,
				    GTK_TAP_AND_HOLD_PASS_PRESS);
    }*/
}


/**
   Gets file system model 
  
   @param ref_widget - any widget on the screen
   @return HildonFileSystemModel object
*/
static HildonFileSystemModel* get_file_system_model(GtkWidget *ref_widget)
{
 return HILDON_FILE_SYSTEM_MODEL(g_object_new(HILDON_TYPE_FILE_SYSTEM_MODEL,
					      "ref_widget", ref_widget, NULL));
}


/**
   Enables page controls
   
   @param app_ui_data - application UI data
   @param dim - what to undim/dim
   @param enable - undim or dim
*/
void ui_enable_page_controls(AppUIData * app_ui_data, PDFDim dim,
			     gboolean enable)
{
    GtkWidget *widget;

    g_return_if_fail(app_ui_data != NULL);

//    TDB("ui_enable_page_controls: %d, %d\n", dim, enable);

    switch (dim) {

    case DIM_ZOOM_IN:
	widget = gtk_ui_manager_get_widget(app_ui_data->ui_manager,
					   "/MenuBar/pdfv_me_main_menu_view"
					   "/pdfv_me_menu_screen_zoom"
					   "/pdfv_me_menu_screen_zoom_in");
	if (widget != NULL)
	    gtk_widget_set_sensitive(widget, enable);

	widget = gtk_ui_manager_get_widget(app_ui_data->ui_manager,
					   "/ToolBar/pdfv_me_menu_screen_zoom_in");
	if (widget != NULL)
	    gtk_widget_set_sensitive(widget, enable);

	widget = gtk_ui_manager_get_widget(app_ui_data->ui_manager,
					   "/Popup/pdfv_me_menu_screen_zoom_in");
	if (widget != NULL)
	    gtk_widget_set_sensitive(widget, enable);
	break;

    case DIM_ZOOM_OUT:
	widget = gtk_ui_manager_get_widget(app_ui_data->ui_manager,
					   "/MenuBar/pdfv_me_main_menu_view"
					   "/pdfv_me_menu_screen_zoom"
					   "/pdfv_me_menu_screen_zoom_out");
	if (widget != NULL)
	    gtk_widget_set_sensitive(widget, enable);

	widget = gtk_ui_manager_get_widget(app_ui_data->ui_manager,
					   "/ToolBar/pdfv_me_menu_screen_zoom_out");
	if (widget != NULL)
	    gtk_widget_set_sensitive(widget, enable);

	widget = gtk_ui_manager_get_widget(app_ui_data->ui_manager,
					   "/Popup/pdfv_me_menu_screen_zoom_out");
	if (widget != NULL)
	    gtk_widget_set_sensitive(widget, enable);
	break;
    case DIM_ERROR:
        /* error case of out-of-memory / corrupted page */
        
        /* zoom functionality */
        ui_enable_page_controls(app_ui_data, DIM_ZOOM_IN,   enable);
	ui_enable_page_controls(app_ui_data, DIM_ZOOM_OUT,  enable);

	/* fit to page */
	widget = gtk_ui_manager_get_widget(app_ui_data->ui_manager,
					   "/MenuBar/pdfv_me_main_menu_view"
					   "/pdfv_me_menu_screen_zoom"
					   "/pdfv_me_menu_screen_zoom_page");
	if (widget != NULL)
	  gtk_widget_set_sensitive(widget, enable);


	/* fit to width */
	widget = gtk_ui_manager_get_widget(app_ui_data->ui_manager,
					   "/MenuBar/pdfv_me_main_menu_view"
					   "/pdfv_me_menu_screen_zoom"
					   "/pdfv_me_menu_screen_zoom_width");
	if (widget != NULL)
	  gtk_widget_set_sensitive(widget, enable);

	/* show images button */
	widget = gtk_ui_manager_get_widget(app_ui_data->ui_manager,
					   "/MenuBar/pdfv_me_main_menu_view"
					   "/pdfv_me_menu_screen_show_images");

	if (widget != NULL)
	  gtk_widget_set_sensitive(widget, enable);
	
        break;
    case DIM_PREV:
	widget = gtk_ui_manager_get_widget(app_ui_data->ui_manager,
					   "/ToolBar/pdfv_me_menu_page_previous");
	if (widget != NULL)
	    gtk_widget_set_sensitive(widget, enable);

	widget = gtk_ui_manager_get_widget(app_ui_data->ui_manager,
					   "/ToolBar/pdfv_me_menu_page_first");
	if (widget != NULL)
	    gtk_widget_set_sensitive(widget, enable);

	widget = gtk_ui_manager_get_widget(app_ui_data->ui_manager,
					   "/MenuBar/pdfv_me_main_menu_page"
					   "/pdfv_me_menu_page_previous");
	if (widget != NULL)
	    gtk_widget_set_sensitive(widget, enable);

	widget = gtk_ui_manager_get_widget(app_ui_data->ui_manager,
					   "/MenuBar/pdfv_me_main_menu_page/pdfv_me_menu_page_first");
	if (widget != NULL)
	    gtk_widget_set_sensitive(widget, enable);

	widget = gtk_ui_manager_get_widget(app_ui_data->ui_manager,
					   "/Popup/pdfv_me_menu_page_previous");
	if (widget != NULL)
	    gtk_widget_set_sensitive(widget, enable);
	break;

    case DIM_NEXT:
	widget = gtk_ui_manager_get_widget(app_ui_data->ui_manager,
					   "/ToolBar/pdfv_me_menu_page_next");
	if (widget != NULL)
	    gtk_widget_set_sensitive(widget, enable);

	widget = gtk_ui_manager_get_widget(app_ui_data->ui_manager,
					   "/ToolBar/pdfv_me_menu_page_last");
	if (widget != NULL)
	    gtk_widget_set_sensitive(widget, enable);

	widget = gtk_ui_manager_get_widget(app_ui_data->ui_manager,
					   "/MenuBar/pdfv_me_main_menu_page/pdfv_me_menu_page_next");
	if (widget != NULL)
	    gtk_widget_set_sensitive(widget, enable);

	widget = gtk_ui_manager_get_widget(app_ui_data->ui_manager,
					   "/MenuBar/pdfv_me_main_menu_page/pdfv_me_menu_page_last");
	if (widget != NULL)
	    gtk_widget_set_sensitive(widget, enable);

	widget = gtk_ui_manager_get_widget(app_ui_data->ui_manager,
					   "/Popup/pdfv_me_menu_page_next");
	if (widget != NULL)
	    gtk_widget_set_sensitive(widget, enable);
	break;

    case DIM_SWITCH_TO:
	if (app_ui_data->current_page_item)
	    gtk_widget_set_sensitive(app_ui_data->current_page_item,
				     enable);

	widget = gtk_ui_manager_get_widget(app_ui_data->ui_manager,
					   "/MenuBar/pdfv_me_main_menu_page"
					   "/pdfv_me_menu_page_switch_to");
	if (widget != NULL)
	    gtk_widget_set_sensitive(widget, enable);
	break;
    case DIM_ALL:
	/* call other dim functions */
	ui_enable_page_controls(app_ui_data, DIM_ZOOM_IN,   enable);
	ui_enable_page_controls(app_ui_data, DIM_ZOOM_OUT,  enable);
	ui_enable_page_controls(app_ui_data, DIM_PREV,      enable);
	ui_enable_page_controls(app_ui_data, DIM_NEXT,      enable);
	ui_enable_page_controls(app_ui_data, DIM_SWITCH_TO, enable);
	
	/* change the state of any other document controls as well */
	ui_enable_document_controls(app_ui_data, enable);
	break;
    }
}

/**
   Enables / disables scrollbars
   
   @param app_ui_data
   @param henable - enable horizontal scrollbar
   @param venable - enable vertical scrollbar
*/
void ui_enable_scrollbars(AppUIData* app_ui_data,
			  gboolean henable, gboolean venable)
{
    g_return_if_fail(app_ui_data != NULL);
    g_return_if_fail(app_ui_data->hscroll != NULL);
    g_return_if_fail(app_ui_data->vscroll != NULL);

    henable ? gtk_widget_show(app_ui_data->hscroll) :
	gtk_widget_hide(app_ui_data->hscroll);
    venable ? gtk_widget_show(app_ui_data->vscroll) :
	gtk_widget_hide(app_ui_data->vscroll);
}


/**
   Loads application state
   
   @param app_ui_data - strucuture
   @param app_state - structure
*/
void ui_load_state(AppUIData * app_ui_data, AppState * app_state)
{
  GtkWidget *widget = NULL;

  /* check input */
  g_return_if_fail(app_ui_data != NULL);
  g_return_if_fail(app_state != NULL);
  
  /* adjust UI to the state */
  app_ui_data->dpi = app_state->dpi;

  /* set show images flag */
  if(app_state->show_images != PDF_FLAGS_IS_SET(app_ui_data->flags, 
						PDF_FLAGS_SHOW_IMAGES) ) {
    PDF_FLAGS_TOGGLE(app_ui_data->flags, PDF_FLAGS_SHOW_IMAGES);

    /* update the menuitem */
    widget = gtk_ui_manager_get_widget(app_ui_data->ui_manager,
				       "/MenuBar/pdfv_me_main_menu_view/"
				       "pdfv_me_menu_screen_show_images");

    if(widget) gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(widget),
					       app_state->show_images );

    
    ui_set_images_mode( app_ui_data, 
			app_state->show_images );
  }
  
  /* set fullscreen mode on or off */
  ui_toggle_fullscreen(app_ui_data, app_state->fullscreen);
  
}

/**
   Starts help
   
   @param app_ui_data structure
   @param help_topic - topic to open
*/
void ui_run_help(AppUIData * app_ui_data, gchar * help_topic)
{
    osso_return_t  result;
    AppData       *app_data;

    g_assert(app_ui_data != NULL);
    g_assert(app_ui_data->app_data != NULL);
    app_data = app_ui_data->app_data;
    g_assert(app_data->osso_context != NULL);
    g_assert(help_topic != NULL);

    result = ossohelp_show(app_data->osso_context, help_topic, 0);

    if (result != OSSO_OK) {
	OSSO_LOG_DEBUG("Failed opening help for '%s' (result %d)",
		       help_topic, (int) result);
    }
}

/**
   Gets if tool item is sensitive
   
   @param app_ui_data - structure
   @param PDFdim which item
   @return sensitive or not
*/
gboolean ui_get_toolitem_is_sensitive(AppUIData * app_ui_data, PDFDim dim)
{
    gchar *path = NULL;

    g_assert(app_ui_data);
    g_return_val_if_fail(app_ui_data->ui_manager != NULL, FALSE);

    switch (dim) {
    case DIM_PREV:
	path = "/ToolBar/pdfv_me_menu_page_previous";
	break;
    case DIM_NEXT:
	path = "/ToolBar/pdfv_me_menu_page_next";
	break;
    case DIM_ZOOM_IN:
	path = "/ToolBar/pdfv_me_menu_screen_zoom_in";
	break;
    case DIM_ZOOM_OUT:
	path = "/ToolBar/pdfv_me_menu_screen_zoom_out";
	break;
    default:
	g_assert_not_reached();
    }
    

    return ( (path != NULL) ? 
	     GTK_WIDGET_IS_SENSITIVE(gtk_ui_manager_get_widget
				     (app_ui_data->ui_manager, path))
	     : FALSE );
}

/**
   Shows info banner
   
   @param app_window - application window
   @param msg - text message
*/
void ui_show_banner(GtkWidget * widget, const gchar *msg) {
  
  //g_assert(widget);

  if(msg) {
    hildon_banner_show_information(widget,NULL, msg);

    /* keep the UI responsive */
/*    while (gtk_events_pending ()) {
      gtk_main_iteration ();
    }*/
  }
}

/**
   Shows progress banner
   
   @param app_window - application windows
   @param msg - text message
*/
GtkWidget *ui_show_progress_banner(GtkWindow * app_window, const gchar *msg) {

  g_assert(app_window);

  if(msg) {
    GtkWidget *w = hildon_banner_show_animation(app_window, NULL, msg);
    TDB("====Show progress banner==== \n");
    gtk_widget_realize(w);
    gdk_window_process_all_updates();
    gdk_window_invalidate_rect(w->window, NULL, TRUE);
    gdk_window_process_all_updates();
    return w;
    /* keep the UI responsive */
/*    while (gtk_events_pending ()) {
      gtk_main_iteration ();
    }*/
  } else
    return NULL;
}

/**
   Shows infonote
   
   @param app_window - application window
   @param msg - text message
*/
void ui_show_note(GtkWindow * app_window, const gchar *msg) {
  GtkWidget *note;

  g_assert(app_window);

  if(msg) {
    note = hildon_note_new_information(app_window, msg);

    gtk_dialog_run(GTK_DIALOG(note));
    gtk_widget_destroy(GTK_WIDGET(note));
  }
}

/**
   Shows result
   
   @param app_ui_data - structure
   @param result - the result to show
*/
void ui_show_result(AppUIData *app_ui_data, PDFViewerResult result) {
  
  gchar *msg = NULL;
  
  g_return_if_fail(app_ui_data != NULL);

  switch(result) {
  case RESULT_INVALID_INTERFACE:
    msg = _("pdfv_ib_incorrect_interface");
    break;
  case RESULT_SAVE_OK:
    msg = _("pdfv_error_successfulsave");
    break;
  case RESULT_SAVE_NOT_ALLOWED:
    msg = _("pdfv_error_notallowed");
    break;
  case RESULT_SAVE_FAILED:
    msg = _("pdfv_error_savefail");
    break;
  case RESULT_UNSUPPORTED_FORMAT:
    ui_show_note(GTK_WINDOW(app_ui_data->app),
		 _("pdfv_ni_unsupported_file"));
    break;
  case RESULT_CORRUPTED_FILE:
    ui_show_note(GTK_WINDOW(app_ui_data->app), 
		 _("pdfv_error_documentcorrupted"));
    break;
  case RESULT_INSUFFICIENT_MEMORY:
    ui_show_note(GTK_WINDOW(app_ui_data->app),
		 _("pdfv_ni_not_enough_memory"));
    break;
  case RESULT_INVALID_URI:
    msg = _("pdfv_ni_document_not_found");
    break;
  case RESULT_NO_SPACE_ON_DEVICE:
    msg = _("pdfv_error_notenoughmemorysave");
    break;
  case RESULT_INTERRUPTED_MMC_OPEN:
    msg =  _("pdfv_error_memorycard");
    break;
  default:
    msg = NULL;
  }

  if(msg) ui_show_banner(GTK_WIDGET(app_ui_data->app), msg);
}

/**
   Sets zoom
   
   @param app_ui_data
   @param value - new zoom value
*/
void ui_set_current_zoom(AppUIData * app_ui_data, int value)
{
    gchar *tmp = g_strdup_printf(_("pdfv_tb_zoom_level"), value);
    gchar *text = g_strdup_printf( value < 100 ? " %s " : "%s",
        tmp);
    
    gtk_widget_queue_resize_no_redraw(app_ui_data->current_zoom_value);
    g_object_set(app_ui_data->current_zoom_value, "label", text, NULL);
    g_free(text);
    g_free(tmp);
}

/**
   Sets if images are shown or not
   
   @param app_ui_data
   @param sign - if images are shown or not
*/
void ui_set_images_mode( AppUIData *app_ui_data, int sign )
{
  GtkWidget * image;

  image = sign ? app_ui_data->img_images_on : app_ui_data->img_images_off;
  if (image != NULL) {
    gtk_widget_show(image);
    gtk_tool_button_set_icon_widget( GTK_TOOL_BUTTON(app_ui_data->image_button), image);
    gtk_widget_show_all( app_ui_data->image_button );
  }
}


/* EOF */
