/**
    @file callbacks.c

    Callbacks for the user interface.

    Copyright (c) 2004-2006 Nokia Corporation.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
    02111-1307 USA
*/


#include <string.h>
#include <stdlib.h>
#include <math.h>

#include <gtk/gtk.h>
#include <libosso.h>
#include <hildon-widgets/hildon-app.h>
#include <gdk/gdkkeysyms.h>
#include <ossoemailinterface.h>
#ifdef VERSION_3
#include <conbtdialogs-dbus.h>
#include <dbus/dbus-glib.h>
#endif

#include "callbacks.h"
#include "interface.h"
#include "utility.h"
#include "pdfviewer.h"
#include "constant.h"
#include "state_save.h"
#include "debug.h"


#define NO_RESET_STATE_FILE

/**
	Action callbacks for document menu
*/

void on_document_open(GtkAction * action, gpointer user_data)
{
    AppUIData *app_ui_data;
  
    g_return_if_fail(user_data != NULL);
    
    app_ui_data = (AppUIData *) user_data;
    if (!PDF_FLAGS_IS_SET(app_ui_data->flags, PDF_FLAGS_RENDERING))
    {
        ui_run_file_chooser(app_ui_data,
			GTK_FILE_CHOOSER_ACTION_OPEN);
    }
}

void on_document_save(GtkAction * action, gpointer user_data)
{
    AppUIData *app_ui_data;
  
    g_return_if_fail(user_data != NULL);
    
    app_ui_data = (AppUIData *) user_data;

  
    /* indicate save dialog */
    ui_run_file_chooser(app_ui_data,
			GTK_FILE_CHOOSER_ACTION_SAVE);
}

#ifdef VERSION_3
gboolean rpc_send_via_bluetooth(gchar *path)
{
    DBusGProxy *proxy = NULL;
    DBusGConnection *sys_conn = NULL;
    GError *error = NULL;
    gboolean result = TRUE;
    gchar **files = NULL;

    /*sys_conn = osso_get_sys_dbus_connection(ctx);*/
    sys_conn = dbus_g_bus_get(DBUS_BUS_SYSTEM, &error);

    if(sys_conn == NULL)
    {
        return FALSE;
    }

    files = g_new0(gchar*, 2);
    *files = g_strdup(path);
    files[1] = NULL;

    /* Open connection for btdialogs service */
    proxy = dbus_g_proxy_new_for_name(sys_conn,
                           CONBTDIALOGS_DBUS_SERVICE,
                           CONBTDIALOGS_DBUS_PATH,
                           CONBTDIALOGS_DBUS_INTERFACE);

    /* Send send file request to btdialogs service */
    if (!dbus_g_proxy_call(proxy, CONBTDIALOGS_SEND_FILE_REQ,
                             &error, G_TYPE_STRV, files,
                             G_TYPE_INVALID, G_TYPE_INVALID))
    {
        g_print("Error: %s\n", error->message);
        g_clear_error(&error);
        result = FALSE;
    }

    g_strfreev(files);

    g_object_unref(G_OBJECT(proxy));
    dbus_connection_unref(sys_conn);
    return result;
}

void on_document_send_bt(GtkAction * action, gpointer user_data)
{
    osso_return_t  retval;
    AppUIData     *app_ui_data;
    GnomeVFSURI   *uri  = pdf_viewer_get_uri();
    gchar         *file = NULL;

    g_return_if_fail(user_data != NULL);

    app_ui_data = (AppUIData *) user_data;

    if(!uri) return;
    
    file = gnome_vfs_uri_to_string(uri, GNOME_VFS_URI_HIDE_NONE);
    
    

    rpc_send_via_bluetooth(file);

    

    if(file)
      g_free(file);
}
#endif

void on_document_send(GtkAction * action, gpointer user_data)
{
    osso_return_t  retval;
    GSList        *list = NULL;
    AppUIData     *app_ui_data;
    GnomeVFSURI   *uri  = pdf_viewer_get_uri();
    gchar         *file = NULL;

    g_return_if_fail(user_data != NULL);

    app_ui_data = (AppUIData *) user_data;

    if(!uri) return;
    
    file = gnome_vfs_uri_to_string(uri, GNOME_VFS_URI_HIDE_NONE);
    
    list = g_slist_append(list, file);

    retval =
	osso_email_files_email(app_ui_data->app_data->osso_context, list);

    g_slist_free(list);

    if(file)
      g_free(file);
}


void on_document_details(GtkAction * action, gpointer user_data)
{    
    AppUIData *app_ui_data;
  
    g_return_if_fail(user_data != NULL);
    
    app_ui_data = (AppUIData *) user_data;
    
    ui_run_details_dialog(app_ui_data);

}

void on_recent( GtkAction *action, gpointer *data ) {
  /* nothing more than recent list update */
    AppUIData *app_ui_data = NULL;

    (void) action;

    app_ui_data = (AppUIData *) data;
    ui_refresh_recent_menu(app_ui_data);
}

gboolean        menu_opened                  (GtkWidget      *widget,
                                            GdkEventExpose *event,
                                            gpointer        user_data)
{
  AppUIData *app_ui_data = NULL;


  app_ui_data = (AppUIData *) user_data;
  on_recent( NULL,user_data );
  
  if (app_ui_data->rendering)
  {
    GtkWidget *recent_menu=NULL;
    recent_menu = gtk_ui_manager_get_widget(app_ui_data->ui_manager,
					   "/MenuBar/pdfv_me_main_menu_recent");
	if (recent_menu != NULL)
	    gtk_widget_set_sensitive(recent_menu, FALSE);
  }
  
  return FALSE;
}

/**
	Action callbacks for page menu
*/

void on_page_previous(GtkAction * action, gpointer user_data)
{
    AppUIData *app_ui_data;
    
    g_return_if_fail(user_data != NULL);

    app_ui_data = (AppUIData *) user_data;
    
    if( !PDF_FLAGS_IS_SET(app_ui_data->flags, PDF_FLAGS_RENDERING) ) {
        pdf_viewer_navigate(DOC_NAVI_PREVIOUS);
        ui_update_current_page(app_ui_data);
    }
}


void on_page_next(GtkAction * action, gpointer user_data)
{
    AppUIData *app_ui_data;

    g_return_if_fail(user_data != NULL);

    app_ui_data = (AppUIData *) user_data;

    if( !PDF_FLAGS_IS_SET(app_ui_data->flags, PDF_FLAGS_RENDERING) ) {
        pdf_viewer_navigate(DOC_NAVI_NEXT);
        ui_update_current_page(app_ui_data);
    }
}


void on_page_first(GtkAction * action, gpointer user_data)
{
    AppUIData *app_ui_data;

    g_return_if_fail(user_data != NULL);
    
    app_ui_data = (AppUIData *) user_data;

    if( !PDF_FLAGS_IS_SET(app_ui_data->flags, PDF_FLAGS_RENDERING) ) {
        pdf_viewer_navigate(DOC_NAVI_FIRST);
        ui_update_current_page(app_ui_data);
    }
}


void on_page_last(GtkAction * action, gpointer user_data)
{
    AppUIData *app_ui_data;

    g_return_if_fail(user_data != NULL);

    app_ui_data = (AppUIData *) user_data;

    if( !PDF_FLAGS_IS_SET(app_ui_data->flags, PDF_FLAGS_RENDERING) ) {
        pdf_viewer_navigate(DOC_NAVI_LAST);
        ui_update_current_page(app_ui_data);
    }
}


void on_page_switch_to(GtkAction * action, gpointer user_data)
{
    AppUIData *app_ui_data;
    
    g_return_if_fail(user_data != NULL);
    
    app_ui_data = (AppUIData *) user_data;
  
    if( !PDF_FLAGS_IS_SET(app_ui_data->flags, PDF_FLAGS_RENDERING) ) {
      ui_run_switch_page_dialog(app_ui_data);
    }
}


/**
	Action callbacks for screen menu
*/

void on_screen_full_screen(GtkAction * action, gpointer user_data)
{
    gboolean active =
	gtk_toggle_action_get_active(GTK_TOGGLE_ACTION(action));
    ui_set_fullscreen((AppUIData *) user_data, active);
}


void on_screen_zoom_in(GtkAction * action, gpointer user_data)
{
    AppUIData *app_ui_data = (AppUIData *) user_data;

    g_return_if_fail(app_ui_data != NULL);
	
    if( !PDF_FLAGS_IS_SET(app_ui_data->flags, PDF_FLAGS_RENDERING) ) {
	pdf_viewer_zoom(DOC_ZOOM_IN);
    }
}


void on_screen_zoom_out(GtkAction * action, gpointer user_data)
{
    AppUIData *app_ui_data = (AppUIData *) user_data;

    g_return_if_fail(app_ui_data != NULL);

    if( !PDF_FLAGS_IS_SET(app_ui_data->flags, PDF_FLAGS_RENDERING) ) {
	pdf_viewer_zoom(DOC_ZOOM_OUT);
    }
}


void on_screen_zoom_page(GtkAction * action, gpointer user_data)
{
    AppUIData *app_ui_data = (AppUIData *) user_data;

    g_return_if_fail(app_ui_data != NULL);
    
    if( !PDF_FLAGS_IS_SET(app_ui_data->flags, PDF_FLAGS_RENDERING) ) {    
	pdf_viewer_zoom(DOC_ZOOM_PAGE);
    }
}


void on_screen_zoom_width(GtkAction * action, gpointer user_data)
{
    AppUIData *app_ui_data = (AppUIData *) user_data;

    g_return_if_fail(app_ui_data != NULL);
   
    if( !PDF_FLAGS_IS_SET(app_ui_data->flags, PDF_FLAGS_RENDERING) ) {    
	pdf_viewer_zoom(DOC_ZOOM_WIDTH);
    }
}


void on_screen_show_images(GtkAction * action, gpointer user_data)
{
    AppUIData *app_ui_data = (AppUIData *) user_data;
    gboolean active =
	gtk_toggle_action_get_active(GTK_TOGGLE_ACTION(action));
    
    g_return_if_fail(app_ui_data != NULL);

    if( active != PDF_FLAGS_IS_SET(app_ui_data->flags,
				   PDF_FLAGS_SHOW_IMAGES) ) {
	 
      gboolean images;
									
      PDF_FLAGS_TOGGLE(app_ui_data->flags, PDF_FLAGS_SHOW_IMAGES);
    
      images = PDF_FLAGS_IS_SET(app_ui_data->flags, PDF_FLAGS_SHOW_IMAGES);

      ui_set_images_mode( app_ui_data, images );
		
    }
    
    if(pdf_viewer_get_num_pages() > 0)
      pdf_viewer_toggle_images();
}


/**
	Action callbacks for tools menu
*/
void on_tools_help(GtkAction * action, gpointer user_data)
{
    ui_run_help((AppUIData *) user_data, PDFV_HELP_MAINVIEW);
}


/**
	Other action callbacks
*/

void on_recent_document(GtkAction * action, gpointer user_data)
{
    int index;
    const gchar *name = gtk_action_get_name(action);

    /* NOTE action name must be in format 'pdfv_me_menu_recent_document1' */
    g_return_if_fail( (name != NULL)
		      && (strlen(name) > 0)
		      && g_ascii_isdigit(name[strlen(name) - 1]) );

    index = atoi(&name[strlen(name) - 1]);
    ui_open_recent_item((AppUIData *) user_data, index);
}

void on_close(GtkAction * action, gpointer user_data)
{
    AppUIData *app_ui_data = (AppUIData *) user_data;
    TDB("On close\n");
    app_ui_data->close_called=TRUE;
    g_return_if_fail(app_ui_data != NULL);
    
    /* destroy state information */
    destroy_state_data(app_ui_data->app_data);
    /* Close application */
    gtk_main_quit();
}


void on_switch_to_page_help(GtkAction * action, gpointer user_data)
{
    ui_run_help((AppUIData *) user_data, PDFV_HELP_SWITCH_TO_PAGE);
}


/**
	Action callback for undefined actions.
	This function is invoked by all undefined menu and toolbar interactions.
	
	@param action Corresponding action
	@param user_data AppUIData
	@return void
*/
void on_undefined_action(GtkAction * action, gpointer user_data)
{
    /* Print debug message */
    OSSO_LOG_DEBUG("Unhandled action \"%s\"", gtk_action_get_name(action));
}


/**
	Other user interface callbacks
*/

inline int key_press_disabled( AppUIData* app_ui_data, GdkEventKey* event )
{
  const unsigned long mask = GDK_CONTROL_MASK | GDK_SHIFT_MASK | GDK_MOD1_MASK |
  		             GDK_MOD3_MASK | GDK_MOD4_MASK | GDK_MOD5_MASK;
       
  if ( event->state & mask ) return( 1 ); //bits are setted

  /* 
     rendering a document or no document is opened disabling HW keys,
     except if it's SELECT HW button.
   */
  if ( PDF_FLAGS_IS_SET(app_ui_data->flags, PDF_FLAGS_RENDERING) ) return( 2 ); //rendering
      
  if ( pdf_viewer_get_uri() != NULL) return( 0 ); //all keys enabled
      
  if (event->keyval == GDK_Return ) return( 0 );
  if (event->keyval == GDK_Escape ) return( 0 );

  return( 3 ); //uri == null
}

/* Callback for keypresses */
gboolean key_press(GtkWidget * widget, GdkEventKey * event, gpointer data)
{
    AppUIData *app_ui_data;
    GtkAdjustment *adj = NULL;
    GtkWidget *w = NULL;

    int rc;

    app_ui_data = (AppUIData *) data;

    switch (event->keyval) {
    /* F6 = toggle full screen mode */
    case GDK_F6:
        if (PDF_FLAGS_IS_SET(app_ui_data->flags, PDF_FLAGS_SELECT_KEY_ALLOWED) && !PDF_FLAGS_IS_SET(app_ui_data->flags, PDF_FLAGS_RENDERING) /*&& app_ui_data->app_data->state != PDF_VIEWER_STATE_LOADING*/ )
        {
          PDF_FLAGS_UNSET(app_ui_data->flags, PDF_FLAGS_SELECT_KEY_ALLOWED);
          ui_toggle_fullscreen(app_ui_data, !PDF_FLAGS_IS_SET
                                (app_ui_data->flags,
                         PDF_FLAGS_FULLSCREEN));
        }
      
    return TRUE;
    }

    rc = key_press_disabled( app_ui_data, event );
    if ( rc ) return( TRUE );

    switch (event->keyval) {

    /* scrolling UP or LEFT is almost the same, just different GtkAdjustments */
    /* UP = scroll up 20% */
    case GDK_KP_Up:
    case GDK_Up:
    OSSO_LOG_DEBUG("Up");
    adj = gtk_layout_get_vadjustment(GTK_LAYOUT(app_ui_data->layout));

    /* LEFT = scroll left 20% */
    case GDK_KP_Left:
    case GDK_Left:
    if (!adj) {
        OSSO_LOG_DEBUG("Left");
        adj =
        gtk_layout_get_hadjustment(GTK_LAYOUT
                       (app_ui_data->layout));

        /* 
           document has more than one pages and it's width fits into
           the screen then switch to the previous page.
        */
        if ( adj->upper <= adj->page_size ) {
           w = gtk_ui_manager_get_widget
        ( app_ui_data->ui_manager,
          "/ToolBar/pdfv_me_menu_page_previous" );
          
          if ((w != NULL) && (GTK_WIDGET_IS_SENSITIVE(w))) {
        on_page_previous(NULL, data);
          } else {
        if (pdf_viewer_get_current_page() == 1)
        {
          ui_show_banner( GTK_WIDGET(app_ui_data->app),
                _("pdfv_ib_first_page_reached") );
        }
          }

          return TRUE;
        }
    }
    /* scroll the page by 20% */
    if (adj->lower <= adj->value - 0.2 * adj->page_size)
        gtk_adjustment_set_value(adj,
                     adj->value - 0.2 * adj->page_size);
    else
        gtk_adjustment_set_value(adj, adj->lower);

    return TRUE;

    /* scrolling DOWN or RIGHT is almost the same, just different GtkAdjustments */
    /* DOWN = scroll down 20% */
    case GDK_KP_Down:
    case GDK_Down:
    OSSO_LOG_DEBUG("Down");
    adj = gtk_layout_get_vadjustment(GTK_LAYOUT(app_ui_data->layout));

    /* RIGHT = scroll to the right 20% */
    case GDK_KP_Right:
    case GDK_Right:
    if (!adj) {
        OSSO_LOG_DEBUG("Right");
        adj =
        gtk_layout_get_hadjustment(GTK_LAYOUT
                       (app_ui_data->layout));

        /* 
           document has more than one pages and it's width fits into
           the screen then switch to the next page.
        */
        if ( adj->upper <= adj->page_size ) {
          w = gtk_ui_manager_get_widget
        ( app_ui_data->ui_manager,
          "/ToolBar/pdfv_me_menu_page_next" );
          
          if ((w != NULL) && (GTK_WIDGET_IS_SENSITIVE(w))) {
        on_page_next(NULL, data);
          } else {
        if (pdf_viewer_get_num_pages() == pdf_viewer_get_current_page())
        {
          ui_show_banner( GTK_WIDGET(app_ui_data->app),
                _("pdfv_ib_last_page_reached") );
        }
          }

          return TRUE;
        }
    }
    /* scroll the page by 20% */
    if (adj->value + adj->page_size * 0.2 <=
        adj->upper - adj->page_size)
        gtk_adjustment_set_value(adj,
                     adj->value + adj->page_size * 0.2);
    else
        gtk_adjustment_set_value(adj, adj->upper - adj->page_size);

    return TRUE;

    /* display infoprint when maximum or minimum zoom
       level has been reached. See Bugzilla 18393 */

    /* F8 = zoom out */
    case GDK_F8:
      if (PDF_FLAGS_IS_SET(app_ui_data->flags, PDF_FLAGS_SELECT_KEY_ALLOWED))
      {
        PDF_FLAGS_UNSET(app_ui_data->flags, PDF_FLAGS_SELECT_KEY_ALLOWED);
        w = gtk_ui_manager_get_widget( app_ui_data->ui_manager,
            "/ToolBar/pdfv_me_menu_screen_zoom_out" );
        if ((w != NULL) && (GTK_WIDGET_IS_SENSITIVE(w))) {
            pdf_viewer_zoom(DOC_ZOOM_OUT);
        } else if (pdf_viewer_get_zoom_percent() <= 50) {
            ui_show_banner( GTK_WIDGET(app_ui_data->app),
                _("pdfv_ib_minimum_zoom") );
        }
        return TRUE;
      }
      break;

    /* F7 = zoom in */
    case GDK_F7:
      if (PDF_FLAGS_IS_SET(app_ui_data->flags, PDF_FLAGS_SELECT_KEY_ALLOWED))
      {
        PDF_FLAGS_UNSET(app_ui_data->flags, PDF_FLAGS_SELECT_KEY_ALLOWED);
        w = gtk_ui_manager_get_widget( app_ui_data->ui_manager,
            "/ToolBar/pdfv_me_menu_screen_zoom_in" );
        if ((w != NULL) && (GTK_WIDGET_IS_SENSITIVE(w))) {
            pdf_viewer_zoom(DOC_ZOOM_IN);
        } else if (pdf_viewer_get_zoom_percent() >= 400) {
            ui_show_banner( GTK_WIDGET(app_ui_data->app),
                _("pdfv_ib_maximum_zoom") );
        }
        return TRUE;
      }
      break;

    /*
      RETURN/SELECT =
      if no document present Open file dialog
      else full screen mode 
    */
    case GDK_KP_Enter:
    case GDK_Return:
        if ((pdf_viewer_get_num_pages() == 0)&&
          (GTK_WIDGET_IS_SENSITIVE(gtk_ui_manager_get_widget(app_ui_data->ui_manager,
                         "/MenuBar/pdfv_me_main_menu_document/"
                         "pdfv_me_menu_document_open")))) {
      ui_run_file_chooser(app_ui_data, GTK_FILE_CHOOSER_ACTION_OPEN);
    } else {

      if(PDF_FLAGS_IS_SET(app_ui_data->flags, PDF_FLAGS_SELECT_KEY_ALLOWED)) {
       
        PDF_FLAGS_UNSET(app_ui_data->flags, PDF_FLAGS_SELECT_KEY_ALLOWED);

        ui_toggle_fullscreen(app_ui_data, !PDF_FLAGS_IS_SET
                                  (app_ui_data->flags,
                           PDF_FLAGS_FULLSCREEN));
      }
    }
    return TRUE;
  /* ESC = if fullscreen: back to normal */
    case GDK_Escape:
       if(PDF_FLAGS_IS_SET(app_ui_data->flags, PDF_FLAGS_FULLSCREEN)) {
	    ui_toggle_fullscreen(app_ui_data, FALSE);
	    return FALSE;
	  }
	  break;
    default:
        break;
    }

    return FALSE;
}


/* 
	Navigation keys		Arrow keys	GDK_Left, GDK_Right, GDK_Up, GDK_Down
	Cancel (Escape)		Esc			GDK_Escape
	Menu key			F4			GDK_F4
	Home				F5			GDK_F5
	Fullscreen			F6			GDK_F6
	Plus (Zoom in)		F7			GDK_F7
	Minus (Zoom out)	F8			GDK_F8
*/


/* non-repeating key handlers */
gboolean key_release(GtkWidget * widget, GdkEventKey * event,
		     gpointer data)
{
    GtkWidget *w = NULL;
    AppUIData *app_ui_data = (AppUIData *) data;

    g_return_val_if_fail(app_ui_data != NULL, FALSE);
    
    switch (event->keyval) {
    case GDK_F6:
        PDF_FLAGS_SET(app_ui_data->flags, PDF_FLAGS_SELECT_KEY_ALLOWED);
	return TRUE;
    }

    /* 
       rendering a document or no document is opened disabling HW keys 
    */
    if ( PDF_FLAGS_IS_SET(app_ui_data->flags, PDF_FLAGS_RENDERING) 
	 || ( (pdf_viewer_get_uri() == NULL) && (event->keyval != GDK_F6 && event->keyval != GDK_Escape) )) {
	return TRUE;
    }

    if (event->state & (GDK_CONTROL_MASK | GDK_SHIFT_MASK | GDK_MOD1_MASK |
			GDK_MOD3_MASK | GDK_MOD4_MASK | GDK_MOD5_MASK)) {
	return TRUE;
    }

    switch (event->keyval) {
    case GDK_F6:
    case GDK_KP_Enter:
    case GDK_Return:
    case GDK_F7:
    case GDK_F8:
        PDF_FLAGS_SET(app_ui_data->flags, PDF_FLAGS_SELECT_KEY_ALLOWED);
	return TRUE;

	

    case GDK_plus:
	OSSO_LOG_DEBUG("plus");
	break;

    case GDK_KP_Add:
	OSSO_LOG_DEBUG("Add");
	break;

    case GDK_minus:
	OSSO_LOG_DEBUG("Minus");
	break;

    case GDK_KP_Subtract:
	OSSO_LOG_DEBUG("Substract");
	break;

    default:
        break;
    }

    return FALSE;
}


void on_screen_scroll(GtkAdjustment * adjustment, gpointer user_data)
{
    pdf_viewer_scroller_changed((PDFScroll) GPOINTER_TO_UINT(user_data));
}


void on_screen_tap_and_hold_event(GtkWidget * widget, gpointer data)
{
    gtk_grab_remove(widget);

    gtk_menu_popup(GTK_MENU(data), NULL, NULL, NULL, NULL, 0,
           gtk_get_current_event_time());
}


gboolean on_screen_event(GtkWidget * widget, GdkEvent * event,
			 gpointer user_data)
{
    AppUIData *app_ui_data = (AppUIData *) user_data;

    g_return_val_if_fail( app_ui_data != NULL, TRUE);

    if (app_ui_data->app_data->state <= PDF_VIEWER_STATE_LOADING &&
	    event->type == GDK_BUTTON_PRESS)
	return TRUE;

    return FALSE;
}


gboolean on_screen_press(GtkWidget * widget, GdkEventButton * event,
			 gpointer user_data)
{
    AppUIData *app_ui_data = (AppUIData *) user_data;

    g_return_val_if_fail( app_ui_data != NULL, TRUE);

    /* Initialize values for potential panning */
    app_ui_data->lastx = (gint) event->x;
    app_ui_data->lasty = (gint) event->y;

    /* We are not panning yet */
    PDF_FLAGS_UNSET(app_ui_data->flags, PDF_FLAGS_PANNING);
    PDF_FLAGS_SET  (app_ui_data->flags, PDF_FLAGS_BUTTON_DOWN);

    gtk_grab_add(widget);

    return FALSE;
}


gboolean on_screen_release(GtkWidget * widget, GdkEventButton * event,
			   gpointer user_data)
{
    AppUIData *app_ui_data = (AppUIData *) user_data;

    g_return_val_if_fail( app_ui_data != NULL, TRUE);

    /* Stop panning */
    PDF_FLAGS_UNSET(app_ui_data->flags, 
		    PDF_FLAGS_PANNING | PDF_FLAGS_BUTTON_DOWN);

    gtk_grab_remove(widget);
    
    return FALSE;
}


gboolean on_screen_motion(GtkWidget * widget, GdkEventMotion * event,
			  gpointer user_data)
{
    AppUIData *app_ui_data = (AppUIData *) user_data;
    GtkAdjustment *adj;
    gdouble value;
    gint x, y;
    gint dx, dy;
    
    g_return_val_if_fail( app_ui_data != NULL, TRUE);

    /* ignore event if button is not pressed, or page is rendering */
    /* TODO could add test to ensure that document is loaded */
    if ( !PDF_FLAGS_IS_SET(app_ui_data->flags, PDF_FLAGS_BUTTON_DOWN)
	 || PDF_FLAGS_IS_SET(app_ui_data->flags, PDF_FLAGS_RENDERING)) {
	return TRUE;
    }

    if (event->is_hint) {
	/* hint event, retrieve pointer location */
	gdk_window_get_pointer(event->window, &x, &y, NULL);
    } else {
	/* use motion event coordinates */
	x = (gint) event->x;
	y = (gint) event->y;
    }

    /* calculate delta values */
    dx = x - app_ui_data->lastx;
    dy = y - app_ui_data->lasty;

    if ( !PDF_FLAGS_IS_SET(app_ui_data->flags, PDF_FLAGS_PANNING) ) {
	/* not panning yet, check if threshold exceeded */
	if (ABS(dx) > PANNING_THRESHOLD || ABS(dy) > PANNING_THRESHOLD) {
	    /* start panning */
	    PDF_FLAGS_SET(app_ui_data->flags, PDF_FLAGS_PANNING);

	    /* don't move just yet */
	    app_ui_data->lastx = x;
	    app_ui_data->lasty = y;
	}
    } else {
	/* panning */

	/* retrieve and set vertical adjustment,
	   ensure that range is not exceeded */
	adj = gtk_layout_get_vadjustment(GTK_LAYOUT(app_ui_data->layout));
	value = adj->value - (gdouble) dy;
	if (value < adj->lower)
	    value = adj->lower;
	else if (value > (adj->upper - adj->page_size))
	    value = adj->upper - adj->page_size;
	gtk_adjustment_set_value(adj, value);

	/* retrieve and set horizontal adjustment */
	adj = gtk_layout_get_hadjustment(GTK_LAYOUT(app_ui_data->layout));
	value = adj->value - (gdouble) dx;
	if (value < adj->lower)
	    value = adj->lower;
	else if (value > (adj->upper - adj->page_size))
	    value = adj->upper - adj->page_size;
	gtk_adjustment_set_value(adj, value);

	/* NOTE on_screen_scroll handler is invoked automatically
	   when gtk_adjustment_set_value is called */
    }

    return FALSE;
}

void insensitive_press_menu_document_save(GtkWidget * widget,
					  gpointer data)
{
    AppUIData *app_ui_data = NULL;

    app_ui_data = (AppUIData *) data;
    if (app_ui_data == NULL || app_ui_data->app == NULL) {
	return;
    }

    ui_show_banner(widget,
		   _("pdfv_ib_menu_not_available"));
}


void insensitive_press_menu_document_send(GtkWidget * widget,
					  gpointer data)
{
    AppUIData *app_ui_data = NULL;

    app_ui_data = (AppUIData *) data;
    if (app_ui_data == NULL || app_ui_data->app == NULL) {
	return;
    }
    ui_show_banner(widget,
		   _("pdfv_ib_menu_not_available"));
}


void insensitive_press_menu_document_details(GtkWidget * widget,
					     gpointer data)
{
    AppUIData *app_ui_data = NULL;

    app_ui_data = (AppUIData *) data;
    if (app_ui_data == NULL || app_ui_data->app == NULL) {
	return;
    }
    if (pdf_viewer_get_num_pages() > 0) {
	ui_show_banner(widget,
		       _("pdfv_ib_menu_not_available"));
    } else {
        ui_show_banner(widget,
		       _("pdfv_ib_menu_not_available"));
    }
}


void insensitive_press_menu_page_previous(GtkWidget * widget,
					  gpointer data)
{
    AppUIData *app_ui_data = NULL;

    app_ui_data = (AppUIData *) data;
    if (app_ui_data == NULL || app_ui_data->app == NULL) {
	return;
    }

    if (pdf_viewer_get_num_pages() > 0 && pdf_viewer_get_current_page() == 1) {
	ui_show_banner(widget,
		       _("pdfv_ib_first_page_reached"));
    } else {
	ui_show_banner(widget,
		       _("pdfv_ib_menu_not_available"));
    }
}

void insensitive_press_menu_page_first(GtkWidget * widget, gpointer data)
{
    AppUIData *app_ui_data = NULL;

    app_ui_data = (AppUIData *) data;
    if (app_ui_data == NULL || app_ui_data->app == NULL) {
	return;
    }

    if (pdf_viewer_get_num_pages() > 0 && pdf_viewer_get_current_page() == 1) {
	ui_show_banner(widget,
		      _("pdfv_ib_first_page_reached"));
    } else {
	ui_show_banner(widget,
		       _("pdfv_ib_menu_not_available"));
    }
}

void insensitive_press_menu_page_next(GtkWidget * widget, gpointer data)
{
    AppUIData *app_ui_data = NULL;

    app_ui_data = (AppUIData *) data;
    if (app_ui_data == NULL || app_ui_data->app == NULL) {
	return;
    }

    if (pdf_viewer_get_num_pages() > 0 && 
        pdf_viewer_get_current_page() == pdf_viewer_get_num_pages()) {
	ui_show_banner(widget,
		       _("pdfv_ib_last_page_reached"));
    } else {
	ui_show_banner(widget,
		       _("pdfv_ib_menu_not_available"));
    }
}

void insensitive_press_menu_page_last(GtkWidget * widget, gpointer data)
{
    AppUIData *app_ui_data = NULL;

    app_ui_data = (AppUIData *) data;
    if (app_ui_data == NULL || app_ui_data->app == NULL) {
	return;
    }

    if (pdf_viewer_get_num_pages() > 0 && 
        pdf_viewer_get_current_page() == pdf_viewer_get_num_pages()) {
	ui_show_banner(widget,
		       _("pdfv_ib_last_page_reached"));
    } else {
	ui_show_banner(widget,
		       _("pdfv_ib_menu_not_available"));
    }
}

void insensitive_press_menu_screen_zoom_in(GtkWidget * widget,
					   gpointer data)
{
    AppUIData *app_ui_data = NULL;

    app_ui_data = (AppUIData *) data;
    if (app_ui_data == NULL 
	|| app_ui_data->app == NULL
	|| app_ui_data->app_data == NULL) {
	return;
    }

    /* state == STATE_LOADED ? */
    if ( !app_ui_data->app_data->low_memory 
	 && !PDF_FLAGS_IS_SET(app_ui_data->flags, PDF_FLAGS_PAGE_ERROR)
	 && (pdf_viewer_get_num_pages() > 0) &&
         pdf_viewer_get_current_zoom() == 288 ){

	ui_show_banner(widget,
		       _("pdfv_ib_maximum_zoom"));

    } else {
	ui_show_banner(widget,
		       _("pdfv_ib_menu_not_available"));
    }
}


void insensitive_press_menu_screen_zoom_out(GtkWidget * widget,
					    gpointer data)
{
    AppUIData *app_ui_data = NULL;

    app_ui_data = (AppUIData *) data;
    if (app_ui_data == NULL || app_ui_data->app == NULL) {
	return;
    }

    if ( !app_ui_data->app_data->low_memory 
	 && !PDF_FLAGS_IS_SET(app_ui_data->flags, PDF_FLAGS_PAGE_ERROR)
	 && (pdf_viewer_get_num_pages() > 0
         && pdf_viewer_get_current_zoom() == 36 ) ) {

	ui_show_banner(widget,
		       _("pdfv_ib_minimum_zoom"));

    } else {
	ui_show_banner(widget,
		       _("pdfv_ib_menu_not_available"));
    }
}

void insensitive_press_menu_default(GtkWidget *widget,
				    gpointer data)
{
    AppUIData *app_ui_data = NULL;
    
    app_ui_data = (AppUIData *)data;
    
    g_return_if_fail( app_ui_data != NULL );
    g_return_if_fail( app_ui_data->app != NULL );

    ui_show_banner(widget,
		   _("pdfv_ib_menu_not_available"));	   
}

/**
	Handles top event.

	@param widget Pointer to window widget
	@param data Pointer to AppUIData
*/
void on_top(GtkWidget * widget, gpointer data)
{
    AppUIData *app_ui_data = (AppUIData *) data;

    /* topmost status acquired */
    g_return_if_fail(app_ui_data != NULL);
    g_return_if_fail(app_ui_data->app_data != NULL);
    g_return_if_fail(app_ui_data->app_data->osso_context != NULL);

    /* destory the state file */
    //    destroy_state_data(app_ui_data->app_data);

    PDF_FLAGS_UNSET(app_ui_data->flags, PDF_FLAGS_BACKGROUNDED);

    hildon_app_set_killable(app_ui_data->app, FALSE);

}


/**
	Handles untop event.

	@param widget Pointer to window widget
	@param data Pointer to AppUIData
*/
void on_untop(GtkWidget *widget, gpointer data)
{
    /* topmost status lost, save state */

    AppUIData *app_ui_data = (AppUIData *) data;
        
    if (app_ui_data->app_data == NULL ||
	app_ui_data->app_data->osso_context == NULL) {
	return;
    }
    
    /* Save application's state */
    save_app_state(app_ui_data->app_data);

    /* Key repeat flags need to be reset on focus out */
    PDF_FLAGS_SET (app_ui_data->flags,
		   PDF_FLAGS_SELECT_KEY_ALLOWED  |
                   PDF_FLAGS_BACKGROUNDED);
    
    hildon_app_set_killable(app_ui_data->app, TRUE);
}

gboolean on_delete_event(GtkWidget *widget, GdkEvent *event, gpointer user_data)
{
    AppUIData *app_ui_data = (AppUIData *) user_data;
    TDB("%s\n",__FUNCTION__);
    app_ui_data->close_called=TRUE;
    g_return_val_if_fail(app_ui_data != NULL, FALSE);

    destroy_state_data(app_ui_data->app_data);
    gtk_main_quit();
    return TRUE;
}

static gboolean set_default_value(gpointer hildonnumbereditor)
{
  TDB("Setting default number\n");
  hildon_number_editor_set_value((HildonNumberEditor *)hildonnumbereditor, pdf_viewer_get_current_page());
  TDB("Setting default number: Ok\n");
  return FALSE;
}

static void find_focus_all(GtkWidget *widget, gpointer data)
{
  *(gboolean *) data |= GTK_WIDGET_HAS_FOCUS(widget);
}

typedef struct {
  HildonNumberEditor *hildonnumbereditor;
  AppUIData *app_ui_data;
} SNumberError;


gboolean show_number_editor_error(gpointer data)
{
  SNumberError *number = (SNumberError *)data;
  guint min   = PDF_PAGE_INIT+1,
        max   = pdf_viewer_get_num_pages(),
        value = pdf_viewer_get_current_page();
  gchar *info = NULL;
GDK_THR_ENTER;
  if (!G_IS_OBJECT(number->hildonnumbereditor))
  {
    GDK_THR_LEAVE;
    return FALSE;
  }

  hildon_number_editor_set_value(number->hildonnumbereditor, value);
  if( (info = g_strdup_printf(SUPPRESS_FORMAT_WARNING(_("pdfv_error_pagenumber")), min, max)) == NULL)
  {
    GDK_THR_LEAVE;
    return FALSE;
  }

  gtk_infoprint(GTK_WINDOW(number->app_ui_data->app), info);
  g_free(info);
  g_object_set_data(G_OBJECT(number->hildonnumbereditor), "idle-id", (gpointer)0);
  GDK_THR_LEAVE;
  return FALSE;
}

gboolean on_number_editor_error(HildonNumberEditor *hildonnumbereditor,
				gint                arg1,
				gpointer            user_data)
{
  AppUIData *app_ui_data = (AppUIData*) user_data;
  gboolean   focus       = FALSE;
  gboolean   result      = TRUE;
  SNumberError *error;

  g_return_val_if_fail(app_ui_data != NULL, FALSE);
  
  if (g_object_get_data(G_OBJECT(hildonnumbereditor), "idle-id"))
    return TRUE;

  error = g_new(SNumberError, 1);
  error->app_ui_data = app_ui_data;
  error->hildonnumbereditor = hildonnumbereditor;
  g_object_set_data(G_OBJECT(hildonnumbereditor), "idle-id", 
    (gpointer)g_idle_add_full(G_PRIORITY_DEFAULT_IDLE, show_number_editor_error, error, (GDestroyNotify)g_free));

  return TRUE;
}

/**
	Handles on_file_changed event

	@param handle Gnome vfs monitor handle
	@param monitor_uri uri to monitor
	@param info_uri, not used
	@param event_type the type of the file event
	@param user_data pointer to AppData as gpointer
*/
void on_file_changed(GnomeVFSMonitorHandle *handle,
		     const gchar *monitor_uri,
		     const gchar *info_uri,
		     GnomeVFSMonitorEventType event_type,
		     gpointer     user_data)
{
    GSList      *list        = NULL;
    GSList      *iter        = NULL;
    RecentItem  *recent_item = NULL;
    AppData     *app_data    = (AppData *) user_data;
    gchar       *loaded_uri  = NULL;
    GtkWidget   *widget = NULL;    
    g_return_if_fail(app_data != NULL);
    
    /* if there's no recent_list, what are we monitoring? */
    list = app_data->recent_list;
    g_return_if_fail(list != NULL);
    
    switch(event_type) {
    case GNOME_VFS_MONITOR_EVENT_DELETED:
      
      /* check if this document is currently loaded */
      if (app_data->state == PDF_VIEWER_STATE_LOADED) {
	loaded_uri = gnome_vfs_uri_to_string(pdf_viewer_get_uri(),
					     GNOME_VFS_URI_HIDE_NONE);
      
	if (strcmp(monitor_uri, loaded_uri) == 0) {
      	  /* Dim 'save as' item */
	  widget=gtk_ui_manager_get_widget(app_data->app_ui_data->ui_manager,
		      "/MenuBar/pdfv_me_main_menu_document/"
		      "pdfv_me_menu_document_save");
	  gtk_widget_set_sensitive(widget, FALSE);
	  /* as per spec, a deleted file which is open in an application should be closed */
	  pdf_viewer_unload();
	}

	g_free(loaded_uri);
      }

      /* remove file from recent list */
      iter = g_slist_find_custom(list, monitor_uri, (GCompareFunc) recentcmp);
      if (iter) {
	recent_item = (RecentItem *)iter->data;

	/* cancel monitor */
        if (recent_item->monitor_handle)
          gnome_vfs_monitor_cancel(recent_item->monitor_handle);
	
	g_free(recent_item->uri);

	g_free(recent_item);
	
	list = g_slist_remove(list, iter->data);

	/* update list head pointer */
	app_data->recent_list = list;

	/* update recent list */
	ui_update_recent_menu(app_data->app_ui_data);
      }

      OSSO_LOG_DEBUG("file has been deleted! info: %s, monitor: %s\n", info_uri, monitor_uri);

      break;
    case GNOME_VFS_MONITOR_EVENT_CREATED:
    case GNOME_VFS_MONITOR_EVENT_CHANGED: 
      OSSO_LOG_DEBUG("file has been changed! info: %s, monitor: %s\n", info_uri, monitor_uri);
      break;
    default:
      OSSO_LOG_DEBUG("file has been something! info: %s, monitor: %s\n", info_uri, monitor_uri);
      break;
    }
}

/* EOF */
