/**
    @file rpc.c

    DBUS functionality.

    Copyright (c) 2004-2006 Nokia Corporation.
	
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
    02111-1307 USA
*/


#include <gtk/gtk.h>
#include <libosso.h>

#include "rpc.h"
#include "appdata.h"
#include "constant.h"
#include "utility.h"
#include "ui/interface.h"
#include "debug.h"

#include "pdfviewer.h"

static osso_context_t *osso_context;

/**
	Initialize osso
	Do initialization for osso, create osso context, set topping callback,
	dbus-message handling callbacks, and hw-event callbacks	
	
	@param app_data Application data
	@returns TRUE if successful, FALSE otherwise
*/
gboolean init_osso(AppData * app_data)
{
//    osso_context_t *osso_context;
    osso_return_t result;
    AppUIData *app_ui_data;
    
    /* check input */
    g_return_val_if_fail( app_data != NULL, FALSE);
    g_return_val_if_fail( app_data->app_ui_data != NULL, FALSE);

    app_ui_data = app_data->app_ui_data;

    /* Initialize osso and ensure that the initialization was successful */
    osso_context = osso_initialize(OSSO_PDFVIEWER_PACKAGE,
				   PACKAGE_VERSION, TRUE, NULL);

    /* initialize logging */
    ULOG_OPEN(PACKAGE_NAME " " PACKAGE_VERSION);

    if (osso_context == NULL) {
	OSSO_LOG_CRIT("OSSO context initialization failed");
	return FALSE;
    }

    /* set exit callback */
    result = osso_application_set_exit_cb(osso_context,
					  osso_exit_callback, app_ui_data);

    if (result != OSSO_OK) {
	ULOG_CRIT("Error setting exit callback (%d)", result);
	TDB(("Error setting exit callback!!!\n"));
	return FALSE;
    }

    /* Set handler for D-BUS messages from session bus */
    result = osso_rpc_set_cb_f(osso_context,
			       OSSO_PDFVIEWER_SERVICE,
			       OSSO_PDFVIEWER_OBJECT_PATH,
			       OSSO_PDFVIEWER_INTERFACE,
			       dbus_req_handler, app_ui_data);
    if (result != OSSO_OK) {
        TDB(("result != OSSO_OK!!!\n"));
	OSSO_LOG_ERR
	    ("Error setting callback for receiving D-BUS messages (%d)",
	     result);
	return FALSE;
    }
    TDB(("passed!!!\n"));

    /* Set handler for changes in HW states
       NULL means that we are interested in all state changes  */
    result = osso_hw_set_event_cb(osso_context,
				  NULL, hw_event_handler, app_ui_data);

    TDB(("2\n"));
    if (result != OSSO_OK) {
    TDB(("3\n"));
	OSSO_LOG_ERR("Error setting callback for HW state monitoring (%d)",
		     result);
	return FALSE;
    }

    /* Success */
    app_data->osso_context = osso_context;
    TDB(("4\n"));
    return TRUE;
}


void osso_exit_callback(gboolean die_now, gpointer data)
{
    if (die_now) {
	gtk_main_quit();
    }
}


/**
	Deinitialize osso
	Deinitialize osso specific data TODO: Check of return values from osso
	
	@param app_data Application data
	@returns TRUE if successful, FALSE otherwise
*/
gboolean deinit_osso(AppData * app_data)
{
    osso_context_t *osso_context;
    osso_return_t   result;
    AppUIData      *app_ui_data;
    gboolean        return_value;

    /* check input */
    g_return_val_if_fail(app_data != NULL, FALSE);
    g_return_val_if_fail(app_data->osso_context != NULL, FALSE);
    g_return_val_if_fail(app_data->app_ui_data != NULL, FALSE);
    
    osso_context = app_data->osso_context;
    app_ui_data  = app_data->app_ui_data;

    return_value = TRUE;	/* Indicate success */

    /* Unset all callbacks */

    /* Unset handler for changes in HW states */
    result = osso_hw_unset_event_cb(osso_context, NULL);

    if (result != OSSO_OK) {
	OSSO_LOG_ERR
	    ("Error unsetting callback for HW state monitoring (%d)",
	     result);
	return_value = FALSE;
    }

    /* Unset handler for D-BUS messages from session bus */
    result = osso_rpc_unset_cb_f(osso_context,
				 OSSO_PDFVIEWER_SERVICE,
				 OSSO_PDFVIEWER_OBJECT_PATH,
				 OSSO_PDFVIEWER_INTERFACE,
				 dbus_req_handler, app_ui_data);

    if (result != OSSO_OK) {
	OSSO_LOG_ERR
	    ("Error unsetting callback for receiving D-BUS messages (%d)",
	     result);
	return_value = FALSE;
    }

    /* deinitialize logging */
    LOG_CLOSE();

    /* Deinitialize osso */
    osso_deinitialize(osso_context);
    app_data->osso_context = NULL;

    return return_value;
}


/**
	Sends a DBUS message
	
	@todo Currently cannot specify target app or pass arguments.
	
	@param method Method to be called.
	@param retval The return value of the method.
	@param app_data Application specific data
*/
osso_return_t send_dbus_message(const gchar * method,
				osso_rpc_t * retval, AppData * app_data)
{
    return osso_rpc_run(app_data->osso_context,
			"com.nokia.app_launcher",
			"/com/nokia/app_launcher",
			"app_launcher", method, retval, DBUS_TYPE_INVALID);
}


/**
	Private functions
*/


/** 
	Receive D-BUS messages and handles them
	
	@param interface The interface of the called method.
	@param method The method that was called.
	@param arguments A GArray of osso_rpc_t_structures.
	@param data An application specific pointer.
	@param retval The return value of the method.
	@returns gint value
*/
gint dbus_req_handler(const gchar * interface,
		      const gchar * method,
		      GArray * arguments,
		      gpointer data, osso_rpc_t * retval)
{
    (void) interface;
    if (arguments == NULL) {
	    fprintf(stderr,"[pdf] dbus req handler arguments == NULL\n");
	    return OSSO_ERROR;
    }
    return dbus_message_handler(method, arguments, data, retval);
}


/**
	Handles incoming D-BUS message
	
	@param method The method that was called.
	@param arguments A GArray of osso_rpc_t structures.
	@param data An application specific pointer.
	@param retval The return value of the method.
	@returns gint value
*/
gint dbus_message_handler(const gchar *method,
			  GArray      *arguments,
			  gpointer     data, 
			  osso_rpc_t  *retval)
{

    if (g_ascii_strcasecmp(method, "top_application") == 0) {
	retval->type = DBUS_TYPE_BOOLEAN;
	retval->value.b = TRUE;
	return OSSO_OK; 
    }
	
    AppUIData *app_ui_data = (AppUIData *) data;
    osso_rpc_t val = g_array_index(arguments, osso_rpc_t, 0);
    PDFViewerResult pdf_result;

    /* check input */
    g_return_val_if_fail(app_ui_data != NULL, OSSO_ERROR);
    g_return_val_if_fail(method != NULL, OSSO_ERROR);

    /* handle mime_open message */
    if (g_ascii_strcasecmp(method, DBUS_METHOD_MIME_OPEN) == 0) {
	if ((val.type == DBUS_TYPE_STRING)
	    && (val.value.s != NULL)) {
		
	    /* set startup mode */ 
    	    app_ui_data->app_data->mode = STARTUP_MODE_URI_REQUEST;

	    /* bringing front the application on MIME_OPEN */
	    gtk_window_present( GTK_WINDOW(app_ui_data->app) );
      if (app_ui_data->details_dialog)
      {
        gtk_dialog_response(GTK_DIALOG(app_ui_data->details_dialog),GTK_RESPONSE_OK);
      }
      
      if (app_ui_data->switch_page_dialog)
      {
        gtk_dialog_response(GTK_DIALOG(app_ui_data->switch_page_dialog),GTK_RESPONSE_CANCEL);
      }

	    /* attempt to open document */
	    ui_open_document(app_ui_data, (gchar*)val.value.s, NULL);

//	    if (pdf_result == RESULT_LOAD_OK) {
		/* success */
		retval->type = DBUS_TYPE_BOOLEAN;
		retval->value.b = TRUE;
		return OSSO_OK;
//	    }
	}
    } else if (g_ascii_strcasecmp(method, DBUS_METHOD_DISPLAY_INFOPRINT) ==
	       0) {
	/* displays an infoprint with predefined string */
	osso_system_note_infoprint(app_ui_data->app_data->osso_context,
				   DBUS_METHOD_DISPLAY_INFOPRINT, retval);

	/* success */
	retval->type = DBUS_TYPE_BOOLEAN;
	retval->value.b = TRUE;
	return OSSO_OK;
    } else if (g_ascii_strcasecmp(method, "top_application") == 0) {
	retval->type = DBUS_TYPE_BOOLEAN;
	retval->value.b = TRUE;
	return OSSO_OK; 
    } else {
	OSSO_LOG_ERR("Unknown DBUS method: %s\n", method);
    }

    /* failure */
    retval->type = DBUS_TYPE_BOOLEAN;
    retval->value.b = FALSE;
    return OSSO_ERROR;
}


/**
	Osso topping callback
	
	@param arguments Extra parameters
	@param app_data Application specific data
*/
osso_application_top_cb_f *osso_top_callback(const gchar * arguments,
					     AppUIData * app_ui_data)
{
    /* check input */
    g_return_val_if_fail(app_ui_data != NULL, NULL);
    g_return_val_if_fail(app_ui_data->app != NULL, NULL);
    GDK_THR_ENTER;
  
    gtk_window_present(GTK_WINDOW(app_ui_data->app));
  GDK_THR_LEAVE;
    return NULL;
}


/** 
	Handles hardware events.
	
	@param state State occurred.
	@param data Application specific data.
	*/
void hw_event_handler(osso_hw_state_t * state, gpointer data)
{
    AppUIData *app_ui_data = (AppUIData *)data;
    
    g_return_if_fail(app_ui_data != NULL);

    if (state->shutdown_ind) {
	gtk_main_quit();
	OSSO_LOG_DEBUG("hw_event_handler, state = reboot");
    }
    if (state->memory_low_ind) {
        app_ui_data->app_data->low_memory = TRUE;
      
  	OSSO_LOG_DEBUG("hw_event_handler, state = memlow");
    }
    if (state->save_unsaved_data_ind) {

	OSSO_LOG_DEBUG("hw_event_handler, state = batlow");
    }
    if (state->system_inactivity_ind) {

	OSSO_LOG_DEBUG("hw_event_handler, state = minact");
    }
}


/* EOF */
