/**
    @file main.c

    Application main.

    Copyright (c) 2004-2006 Nokia Corporation.
	
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
    02111-1307 USA
*/


#include <libintl.h>
#include <locale.h>
#include <gtk/gtk.h>
#include <libgnomevfs/gnome-vfs.h>
#include <libosso.h>
#include <signal.h>
#include <stdlib.h>
#include <dbus/dbus-glib.h>

#include "rpc.h"
#include "ui/interface.h"
#include "utility.h"
#include "pdfviewer.h"
#include "configuration.h"
#include "settings.h"
#include "state_save.h"
#include "debug.h"

/*******************************************************************************
 **** Private data
 **/
static AppData *app_data = NULL;


/*******************************************************************************
 **** Private functions
 **/

GThread *mainThread = NULL;
gboolean just_exit=FALSE;

/**

*/
static void free_recent_list_item(gpointer data, gpointer user_data)
{
    RecentItem *recent_item = (RecentItem*) data;
    
    g_return_if_fail(recent_item != NULL);

    /* cancel the monitor on the file */
    if (recent_item->monitor_handle)
        gnome_vfs_monitor_cancel(recent_item->monitor_handle);
    
    /* free the string used for uri */
    g_free(recent_item->uri);

    /* free the pointer itself */
    g_free(recent_item);
}


/**
   Free application data and substructures.
	
   @param app_data Application data
*/
static void free_app_data()
{
    if (!app_data)
	return;

    TDB(("18-1\n"));
    if (app_data->recent_list) {
        /* free each element */ 
        TDB(("18-*1\n"));
        g_slist_foreach(app_data->recent_list, (GFunc) free_recent_list_item, NULL);
	/* free the list */
        TDB(("18-*2\n"));
	g_slist_free(app_data->recent_list);
    }
    TDB(("18-2\n"));
    if (app_data->mmc_uri) {
        g_free(app_data->mmc_uri);
        app_data->mmc_uri = NULL;
    }
    TDB(("18-3\n"));
    if (app_data->app_ui_data) {
	g_free(app_data->app_ui_data);
	app_data->app_ui_data = NULL;
    }
    
    TDB(("18-4\n"));
    g_free(app_data);

}

/* Handler for POSIX signals */
static void signal_handler(int sig)
{
    static gboolean handling = FALSE;

    if (handling) {
        return;
    }
    handling = TRUE;
    just_exit=TRUE;
    /*
      if killable -> data already saved
    if(app_data)
      save_app_state(app_data);
    */
    
    /* gtk_main_quit is not safe here! There can be recursive loops
       that would all need to be quit. So we destroy everything,
       and exit. */

    if(app_data->app_ui_data->app)
      gtk_object_destroy(GTK_OBJECT(app_data->app_ui_data->app));

    if(app_data->gc_client)
      g_object_unref(app_data->gc_client);
    
    free_app_data();
    gtk_main_quit();

    //exit(EXIT_SUCCESS);
}


/*******************************************************************************
 **** Public function
 **/

/**
	Application main.
	Initializes internationalization, libosso, app and appview.
	Calls user interface creation functions and gtk_main.

	Follows the component initialization order of
	osso-filemanager main.
		
	@param argc Number of command line arguments
	@param argv Command line arguments
	@return 0 if successful; otherwise error code
*/
int main(int argc, char *argv[])
{
    gboolean          result;
    struct sigaction  sa;
    sigset_t          empty_mask;
    g_thread_init (NULL);
    dbus_g_thread_init();
    gdk_threads_init ();
    
    mainThread = g_thread_self();


    /* Allocate application data structures */
    app_data = g_new0(AppData, 1);
    if(app_data == NULL) {
      OSSO_LOG_CRIT("Failed memory allocation: AppData");
      exit(1);
    }
    
    /* there was no low memory ind from HW yet. */ 
    app_data->low_memory = FALSE;

    TDB("i1\n");    
    app_data->app_ui_data = g_new0(AppUIData, 1);
    if(app_data->app_ui_data == NULL) {
      OSSO_LOG_CRIT("Failed memory allocation: AppUIData");
      free_app_data();
      exit(1);
    }
    
    TDB("i2\n");
    /* Add reference back to parent structure (AppData) */
    app_data->app_ui_data->app_data = app_data;
    
    /* Initialize osso */
    result = init_osso(app_data);
    if (!result) {
	free_app_data(app_data);
	return OSSO_ERROR;
    }

    TDB("i3\n");
    /* Initialize GnomeVFS */
    result = gnome_vfs_init();

    if (!result) {
	OSSO_LOG_CRIT("Failed initializing GnomeVFS");
	deinit_osso(app_data);
	free_app_data(app_data);
	return OSSO_ERROR;
    }

    TDB("i4\n");
    /* Initialize GConf and read application configuration */
    if (!init_settings(app_data))
    {
	OSSO_LOG_CRIT("Failed initializing GConf");
	deinit_osso(app_data);
	free_app_data(app_data);
	return OSSO_ERROR;
    }
    TDB("i5\n");
    if (!get_configuration(argc, argv, app_data)) {
	OSSO_LOG_CRIT("Failed reading configuration");
	deinit_osso(app_data);
	free_app_data(app_data);
	return OSSO_ERROR;
    }
    TDB("i6\n");
#ifdef ENABLE_NLS
    /* Initialize localization */
    /* Gettext does not seem to work properly without
       the following function call */
    setlocale(LC_ALL, "");

    bindtextdomain(GETTEXT_PACKAGE, LOCALEDIR);
    bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");
    textdomain(GETTEXT_PACKAGE);
#endif				/* ENABLE_NLS */

    TDB("i7\n");
    /* Signals */
    sigemptyset (&empty_mask);
    sa.sa_handler = signal_handler;
    sa.sa_mask = empty_mask;
    sa.sa_flags = 0;
    sigaction (SIGTERM, &sa, NULL);
    sigaction (SIGINT, &sa, NULL);
    

    

    TDB("i8\n");
    /* Initialize GTK */
    gtk_init(&argc, &argv);

    TDB(("10\n"));
    /* Create application UI */
    ui_create_main_window(app_data);
    
    /* Initialize engine */
    pdf_viewer_init(app_data->app_ui_data);

    TDB(("13\n"));
    GDK_THR_ENTER;
    gtk_main();
    GDK_THR_LEAVE;
    if (!just_exit)
    {
    TDB(("14\n"));
    /* Save configuration */
    save_configuration(app_data);

    TDB(("15\n"));
    /* Deinitialize engine */
    pdf_viewer_deinit();

    TDB(("16\n"));
    /* Deinitialize GConf */
    deinit_settings();

    TDB(("17\n"));
    /* Deinitialize osso */
    deinit_osso(app_data);

    TDB(("18\n"));
    /* Free memory */
    free_app_data(app_data);
    
    TDB(("19\n"));
    /* utility.c */
    free_mime_filters();
    free_application_mime_types();

    TDB(("20\n"));
    if (gnome_vfs_initialized()) {
        gnome_vfs_shutdown();
    }
    }
    TDB(("The End\n"));
    /* Exit successfully, regardless of any errors */
    exit( EXIT_SUCCESS);
}


/* EOF */
