/**
  @file completion.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2004 Nokia. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 
*/
#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#ifdef HAVE_LIBREADLINE

#include <stdlib.h>
#include <ctype.h>
#include <stdio.h>
#include <string.h>

#include <readline/readline.h>
#include <readline/history.h>

#include "obc-main.h"
#include "fl.h"
#include "bt.h"
#include "completion.h"

static ObcContext *obc_ctx = NULL;

static gboolean need_quote(const char *str)
{
    int i;

    for (i = 0; str[i] != '\0'; i++) {
        switch (str[i]) {
            case ' ':
            case '\t':
            case '*':
            case '?':
                return TRUE;
        }
    }

    return FALSE;
}

static Completion get_completion(const char *text)
{
    int i;

    for (i = 0; cmd_table[i].name != NULL; i++) {
        const char *name = cmd_table[i].name;
        if (strncmp(rl_line_buffer, name, strlen(name)) == 0)
            return cmd_table[i].completion;
    }

    return CPL_NONE;
}

static char *obc_completion_cmd(const char *text, int state)
{
    static int i, len;
    const char *name;

    if (state == 0) {
        i = 0;
        len = strlen(text);
    }

    while ((name = cmd_table[i].name) != NULL) {
        i++;
        if (strncmp(name, text, len) == 0)
            return strdup(name);
    }

    return NULL;
}

static char *obc_completion_bt(const char *text, int state)
{
    static int i, len;
    BtDev *dev;

    if (obc_ctx->dev_cache == NULL)
        return NULL;

    if (state == 0) {
        i = 0;
        len = strlen(text);
    }

    while ((dev = g_slist_nth_data(obc_ctx->dev_cache, i)) != NULL) {
        i++;
        if (dev->name && strncmp(dev->name, text, len) == 0) {
            char *ret;
            if (need_quote(dev->name)) {
                char *tmp;
                tmp = g_shell_quote(dev->name);
                ret = strdup(tmp);
                g_free(tmp);
            }
            else
                ret = strdup(dev->name);
            return ret;
        }
        if (strncmp(dev->bda, text, len) == 0)
            return strdup(dev->bda);
    }
    return NULL;
}

static char *obc_completion_remote(const char *text, int state)
{
    static int i, len;
    FlEntry *e;

    if (state == 0) {
        i = 0;
        len = strlen(text);
        if (obc_ctx->fl_cache == NULL)
            obc_ctx->fl_cache = fl_list_get(obc_ctx, NULL, FALSE, NULL);
    }

    while ((e = g_slist_nth_data(obc_ctx->fl_cache, i)) != NULL) {
        i++;
        if (e->name == NULL)
            continue;
        if (strncmp(e->name, text, len) == 0) {
            char *ret;
            if (need_quote(e->name)) {
                char *tmp;
                tmp = g_shell_quote(e->name);
                ret = strdup(tmp);
                g_free(tmp);
            }
            else
                ret = strdup(e->name);
            return ret;
        }
    }

    return NULL;
}

static char **obc_completion(const char *text, int start, int end)
{
    if (start == 0)
        return rl_completion_matches(text, obc_completion_cmd);
    
    switch (get_completion(text)) {
        case CPL_REM:
            return rl_completion_matches(text, obc_completion_remote);
        case CPL_BT:
            return rl_completion_matches(text, obc_completion_bt);
        case CPL_CMD:
            return rl_completion_matches(text, obc_completion_cmd);
        case CPL_LOC:
        case CPL_NONE:
        default:
            return NULL;
    }
}

#else
# include "completion.h"
#endif /* HAVE_LIBREADLINE */

void obc_completion_init(ObcContext *ctx)
{
#ifdef HAVE_LIBREADLINE
    obc_ctx = ctx;
    rl_attempted_completion_function = obc_completion;
#endif
}
