/**
    @file startup_gamemanager.c

    Implements GameManager module.

    Copyright (c) 2004, 2005 Nokia Corporation.
	
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
    02111-1307 USA
*/


#include <glib/gstdio.h>
#include <hildon-widgets//hildon-file-handling-note.h>
#include "startup_gamemanager.h"

/* State information */
GameType game_type = gt_none;
GameState game_state = gs_unloaded;

/* The following function is from INDT's games-startup code. The copyrights 
belongs to them. The INDT's games-startup was released under LGPL license*/

void 
startup_app_set_state_change_cb (AppUIData  *app,
                                 startup_app_state_change_cb *state_change_cb,
                                 gpointer     data)
{
  g_return_if_fail (app != NULL);
  app->state_change_cb = state_change_cb;
  app->state_change_cb_data = data;
}

/* The following function is from INDT's games-startup code. The copyrights 
belongs to them. The INDT's games-startup was released under LGPL license*/

void
startup_app_set_top_cb (AppUIData  *app,
                        startup_app_top_cb *top_cb,
                        gpointer     data)
{
  g_return_if_fail (app != NULL);
  app->top_cb = top_cb;
  app->top_cb_data = data;
}

/* The following function is from INDT's games-startup code. The copyrights 
belongs to them. The INDT's games-startup was released under LGPL license*/

/* Startup the plugin, and initialize the structure */
StartupPlugin *
startup_plugin_load (const gchar *path)
{
  StartupPlugin *plugin;
  GModule *handle;
  gboolean (*startup_init_plugin) ();

  g_return_val_if_fail (path != NULL, NULL);

  handle = g_module_open (path, 0);
  g_assert (handle);

  if (!g_module_symbol (handle,
        "startup_init_plugin",
        (gpointer) &startup_init_plugin)) {
    g_module_close (handle);
			g_print("Symbol can't be found\n");
    return NULL;
  }
  
  plugin = g_new0 (StartupPlugin, 1);
  g_assert (plugin);

  plugin->priv = g_new0 (StartupPluginPrivate, 1);
  g_assert (plugin->priv);
  plugin->priv->handle = handle;

  if (startup_init_plugin (plugin)) 
    return plugin;

  g_module_close (plugin->priv->handle);
  g_free (plugin->priv);
  g_free (plugin);
    
  return NULL;
}

void Fill_Plugin_Settings(void)
{
	AppData *appdata=get_app_data();
	AppUIData *ui=appdata->app_ui_data;
	ui->plugin->gs->ui->app=g_new0(StartupApp,1);
	ui->plugin->gs->ui->app->osso=appdata->app_osso_data->osso;
	  ui->plugin->gs->ui->app->state=get_game_state();
	  ui->plugin->gs->ui->app->top_cb=ui->top_cb;
	  ui->plugin->gs->ui->app->top_cb_data=ui->top_cb_data;
	  ui->plugin->gs->ui->app->state_change_cb=ui->state_change_cb;
	  ui->plugin->gs->ui->app->state_change_cb_data=ui->state_change_cb_data;
	  ui->plugin->gs->ui->app->service=ui->service;
	  ui->plugin->gs->ui->app->path=ui->path;
	  ui->plugin->gs->ui->app->iface=ui->iface;
	ui->plugin->gs->ui->app->config=ui->indt_startup_config;
	ui->plugin->gs->ui->plugin=ui->plugin;
	ui->plugin->gs->ui->hildon_app=ui->app;
	ui->plugin->gs->ui->hildon_appview=ui->main_view;
	ui->plugin->gs->ui->controls_state_label=ui->controls_state_label;
	ui->plugin->gs->ui->play_button=ui->play_button;
	ui->plugin->gs->ui->restart_button=ui->restart_button;
}

/* The following function is from INDT's games-startup code. The copyrights 
belongs to them. The INDT's games-startup was released under LGPL license. 
This function is modified under LGPL license*/

/* Initialize the INDT specific settings */
AppUIData *
Initialize_INDT_settings (AppUIData *ui)
{ 
  //StartupUI *ui;
	//AppData *ui;
	
  const gchar *plugin_path;
  /*const gchar *game_gettext_package;
  const gchar *msgid;*/
  g_return_val_if_fail (ui != NULL, NULL);

  //ui = g_new0 (StartupUI, 1);
	//ui = g_new0 (AppData, 1);
  g_assert (ui);

  //ui->app_ui_data = app;
  startup_app_set_top_cb (ui, startup_ui_top_cb, (gpointer) ui);
  startup_app_set_state_change_cb (ui, startup_ui_state_change_cb, (gpointer) ui);
  plugin_path = startup_config_read_entry (startup_app_get_config (ui), "PluginPath", NULL);
  if (plugin_path){
    ui->plugin = startup_plugin_load (plugin_path);
    //ui->plugin->gs->ui = ui; /*this structure differs from our, so it won't b only 1 line
	  ui->plugin->gs->ui=g_new0 (StartupUI, 1);
	  
    ui->plugin->gs->startup_ui_menu_action_cb = startup_ui_menu_action_cb;
    ui->plugin->gs->startup_ui_state_change_cb = startup_ui_state_change_cb;
  } else 
    ui->plugin = NULL;
  return ui;
}

void game_play(void)
{
  AppData *app_data = get_app_data();
	//osso_rpc_t ret;
  
  game_state = gs_running;
  if (set_game_state_async(GAME_RUN)==TRUE) {
    /* Reset save file location */
    settings_set_string(SETTINGS_CHESS_SAVE_FILE,"");

    if (app_data!=NULL)
      app_data->app_ui_data->chess_saved = FALSE;
  } else {
    /* Setting game state unloaded */
    game_state = gs_unloaded;

    /* TODO Should user be informed somehow? */
  }
}

void game_continue(void)
{
  AppData *app_data = get_app_data();

  game_state = gs_running;
  if (set_game_state_async(GAME_CONTINUE)) {
    if (app_data!=NULL)
      app_data->app_ui_data->chess_saved = FALSE;
  } else {
    game_state = gs_unloaded;
  }
}

void game_restart(void)
{
  AppData *app_data = get_app_data();
  
  game_state = gs_running;  
  if (set_game_state_async(GAME_RESTART)) {
    /* Reset save file location */
    settings_set_string(SETTINGS_CHESS_SAVE_FILE,"");

    if (app_data!=NULL)
      app_data->app_ui_data->chess_saved = FALSE;
  } else {
    game_state = gs_unloaded;
  }
}

void game_close(void)
{
  if (game_state!=gs_closed && game_state!=gs_unloaded) {
    if (set_game_state(GAME_CLOSE, NULL)) {
      game_state = gs_closed;
    } else {
      game_state = gs_unloaded;
    }
  }
}

void game_load(void)
{
  osso_rpc_t retval;
  AppData *app_data = get_app_data();
  gchar *loadfile = settings_get_string(SETTINGS_CHESS_SAVE_FILE);
  
  /* register to monitoring */
  GnomeVFSResult gnomevfs_res;  
  if (gnomevfs_handle != NULL) {
    gnomevfs_res = gnome_vfs_monitor_cancel(gnomevfs_handle);
  }
  gchar *foldername = g_path_get_dirname(loadfile);
  gnomevfs_res = gnome_vfs_monitor_add(&gnomevfs_handle,
					foldername,
					GNOME_VFS_MONITOR_DIRECTORY,
					chess_saved_file_monitor,
					(gpointer)app_data->app_ui_data);
  g_free(foldername);
  /* end register to monitoring */ 
  
  game_state = gs_running;
  if (set_game_state(GAME_LOAD, &retval)) {
    /* game_state = gs_closed; */
    if (app_data != NULL)
      app_data->app_ui_data->chess_saved = FALSE;
    
    if ((retval.type == DBUS_TYPE_BOOLEAN) && (retval.value.b == FALSE)) {
      game_state = gs_unloaded;
      
      ui_show_load_err(GAME_LOAD_FILE_UNSUPPORTED, app_data->app_ui_data, loadfile);
    } else {
      get_settings(app_data->app_ui_data);
    }
  } else {
    game_state = gs_unloaded;
    /* ui_load_game_fail(app_data->app_ui_data); */
  }
	osso_rpc_free_val (&retval);
  
  if (loadfile != NULL)
    g_free(loadfile);
  
  ui_repaint(get_app_data());
}

void game_save(void)
{
  AppData *app_data = get_app_data();
  gdouble cpos = 0.1;

  game_state = gs_running;

  /* Set where the MMC is mounted */
  const gchar *mmc_env = NULL;      
  mmc_env = g_getenv(MMC_MOUNTPOINT_ENV);
  if (mmc_env) {
    if (app_data->app_ui_data->mmc_uri)
      g_free(app_data->app_ui_data->mmc_uri);
    app_data->app_ui_data->mmc_uri = g_strconcat (URI_FILE_PREFIX, mmc_env, NULL);
  }

  /* Monitor MMC state */
  guint volume_monitor_id = 0;
  app_data->app_ui_data->is_mmc = FALSE;
  GnomeVFSVolumeMonitor *volume_monitor;
  volume_monitor = gnome_vfs_get_volume_monitor();
  volume_monitor_id = g_signal_connect(G_OBJECT(volume_monitor), "volume_unmounted", 
    G_CALLBACK(volume_unmounted_cb), app_data->app_ui_data);

  /* Display saving progress dialog */
  ui_show_saving_dialog(app_data->app_ui_data);

  if (!ui_set_saving_dialog_progress(app_data->app_ui_data, cpos)) return;
  app_data->app_ui_data->chess_save_done=GAME_SAVE_SAVING;
  while(gtk_events_pending())
    gtk_main_iteration();

  g_remove(CHESS_SAVE_FILE_TMP);
  
  /* Send SAVE request to osso-chess-ui and waiting the result */
  if (set_game_state_async(GAME_SAVE)) {
    while (app_data->app_ui_data->chess_save_done==GAME_SAVE_SAVING) {
      if (!ui_set_saving_dialog_progress(app_data->app_ui_data, cpos)) {
        gtk_signal_disconnect(G_OBJECT(volume_monitor), volume_monitor_id);
        ui_repaint(app_data);
        game_state = gs_paused;
        return;
      }
      while(gtk_events_pending())
        gtk_main_iteration();
      usleep(500000);
      cpos+=0.05;
      if (cpos>0.95) cpos=0.95;
    }    
    cpos=1.0;
    /* game_state = gs_paused; */

    if (app_data->app_ui_data->chess_save_done==GAME_SAVE_CANCEL) {
      gtk_signal_disconnect(G_OBJECT(volume_monitor), volume_monitor_id);
      ui_repaint(app_data);
      game_state = gs_paused;
      return;
    }

    if (!ui_set_saving_dialog_progress(app_data->app_ui_data, cpos)) {
      gtk_signal_disconnect(G_OBJECT(volume_monitor), volume_monitor_id);
      ui_repaint(app_data);
      game_state = gs_paused;
      return;
    }
    while(gtk_events_pending())
      gtk_main_iteration();
    usleep(100000);

    if (app_data->app_ui_data->chess_save_done==GAME_SAVE_OK) {
      if (app_data->app_ui_data->is_mmc) { /* MMC cover open while saving to MMC */
        ui_hide_saving_dialog(app_data->app_ui_data);
        while(gtk_events_pending())
          gtk_main_iteration();
        
        ui_show_save_nommc_dialog(app_data->app_ui_data);
        app_data->app_ui_data->chess_saved = FALSE;
      } else {
        gchar *tmp_text, *dst_text;
        GnomeVFSURI *tmp_uri, *dst_uri;
        GnomeVFSResult vfs_res;
        gchar *uri_scheme;
      
        /* Copy saved file from temp to destination */
        tmp_text = gnome_vfs_get_uri_from_local_path(CHESS_SAVE_FILE_TMP);
        
        dst_text = settings_get_string(SETTINGS_CHESS_SAVE_FILE);
        uri_scheme = gnome_vfs_get_uri_scheme(dst_text);
        if (!uri_scheme)
          dst_text = gnome_vfs_get_uri_from_local_path(dst_text);
        else
          g_free(uri_scheme);

        tmp_uri = gnome_vfs_uri_new(tmp_text);
        dst_uri = gnome_vfs_uri_new(dst_text);

        vfs_res = gnome_vfs_xfer_uri(tmp_uri, dst_uri,
            GNOME_VFS_XFER_DEFAULT,
            GNOME_VFS_XFER_ERROR_MODE_ABORT,
            GNOME_VFS_XFER_OVERWRITE_MODE_REPLACE,
            NULL, NULL);

        gnome_vfs_uri_unref(tmp_uri);
        gnome_vfs_uri_unref(dst_uri);
        g_free(tmp_text);
        g_free(dst_text);

        ui_hide_saving_dialog(app_data->app_ui_data);
        while(gtk_events_pending())
          gtk_main_iteration();

        while(gtk_events_pending())
          gtk_main_iteration();
        usleep(10000);

        /* Return if user press Cancel while saving */
        if (app_data->app_ui_data->chess_save_done==GAME_SAVE_CANCEL) {
          gtk_signal_disconnect(G_OBJECT(volume_monitor), volume_monitor_id);
          ui_repaint(app_data);
          game_state = gs_paused;
          return;
        }
            
        /* Copy file error handling */
        switch (vfs_res) {
          case GNOME_VFS_OK:
            show_infoprint(_("sfil_ib_saved"));
            app_data->app_ui_data->chess_saved = TRUE;
            break;
          case GNOME_VFS_ERROR_ACCESS_DENIED:
          case GNOME_VFS_ERROR_READ_ONLY:
          case GNOME_VFS_ERROR_READ_ONLY_FILE_SYSTEM:
          case GNOME_VFS_ERROR_NO_MEMORY:
          case GNOME_VFS_ERROR_NO_SPACE:
          default:
            app_data->app_ui_data->chess_saved = FALSE;
            ui_show_save_nomem_dialog(app_data->app_ui_data);
        }
      } /* End check MMC cover open while saving to MMC */
    } else {
      ui_hide_saving_dialog(app_data->app_ui_data);
      while(gtk_events_pending())
        gtk_main_iteration();
        
      ui_show_save_nomem_dialog(app_data->app_ui_data);
    }
    game_state = gs_paused;
    
    /* Remove tmp file */
    gnome_vfs_unlink(CHESS_SAVE_FILE_TMP);
  } else {
    ui_hide_saving_dialog(app_data->app_ui_data);
    while(gtk_events_pending())
      gtk_main_iteration();
          
    game_state = gs_unloaded;
  }
  app_data->app_ui_data->chess_save_done=GAME_SAVE_NONE;
  
  gtk_signal_disconnect(G_OBJECT(volume_monitor), volume_monitor_id);
  ui_repaint(app_data);
}

/* Game state changes, set state and update UI */
void game_state_changed(GameState state)
{
  osso_log (LOG_DEBUG, "%s\n", __FUNCTION__);
  game_state = state;
  ui_repaint(get_app_data());
}

void assign_game_type(GameType type)
{
  game_type=type;
}

/* Set the game state */
void set_gamestate(GameState state)
{
	game_state=state;
	return;
}

/* Set game type to one of supported ones */
void set_game_type(char *type)
{
  char *pos = NULL;
	game_type=gt_indt;
	return;

  /* Defaulting chess */
  if (type == NULL) {
    game_type=gt_chess;
    return;
  }

  /* Game is Chess? */
  pos=g_strstr_len(type, strlen(type), GAME_CHESS);
  if (pos!=NULL) {
    game_type=gt_chess;
    return;
  }

  /* Game is Mahjong? */
  pos=g_strstr_len(type, strlen(type), GAME_MAHJONG);
  if (pos!=NULL) {
    game_type=gt_mahjong;
    return;
  }

  /* Game is LMarbles? */
  pos=g_strstr_len(type, strlen(type), GAME_LMARBLES);
  if (pos!=NULL) {
    game_type=gt_lmarbles;
    return;
  }
  
  /* Game is Sweper? */
  pos=g_strstr_len(type, strlen(type), GAME_SWEEPER);
  if (pos!=NULL) {
    game_type=gt_indt;
    return;
  }

  /* Defaulting indt */
  //game_type=gt_chess;
  game_type=gt_indt;
}

/* Return selected game type */
GameType get_game_type(void)
{
  return game_type;
}

/* Return game state */
GameState get_game_state(void)
{
  return game_state;
}

/* Return selected game object path */
GString *game_get_object_path(void)
{
  GString *path = NULL;

  switch(game_type) {
    case gt_chess:
      path = g_string_new(OBJECT_PATH_CHESS);
      break;
    case gt_mahjong:
      path = g_string_new(OBJECT_PATH_MAHJONG);
      break;
    case gt_lmarbles:
      path = g_string_new(OBJECT_PATH_LMARBLES);
      break;
	case gt_indt:
		path=g_string_new(get_app_data()->app_ui_data->path);
	break;
    default:
      path = g_string_new(OBJECT_PATH_INVALID);
      break;
  }

  g_assert(path);

  return path;
}

/* Return selected game object path */
GString *game_get_service(void)
{
  GString *path = NULL;

  switch(game_type) {
    case gt_chess:
      path = g_string_new(SERVICE_CHESS);
      break;
    case gt_mahjong:
      path = g_string_new(SERVICE_MAHJONG);
      break;
    case gt_lmarbles:
      path = g_string_new(SERVICE_LMARBLES);
      break;
	case gt_indt:
		path=g_string_new(get_app_data()->app_ui_data->service);
	break;
    default:
      path = g_string_new(SERVICE_INVALID);
      break;
  }

  g_assert(path);

  return path;
}

/* Return selected game object path */
GString *game_get_iface(void)
{
  GString *path = NULL;

  switch(game_type) {
    case gt_chess:
      path = g_string_new(IFACE_CHESS);
      break;
    case gt_mahjong:
      path = g_string_new(IFACE_MAHJONG);
      break;
    case gt_lmarbles:
      path = g_string_new(IFACE_LMARBLES);
      break;
	case gt_indt:
		path=g_string_new(get_app_data()->app_ui_data->iface);
	break;
    default:
      path = g_string_new(IFACE_INVALID);
      break;
  }

  g_assert(path);

  return path;
}

void game_mime_open(gchar *filename) 
{
  AppUIData *app_ui_data = get_app_data()->app_ui_data;
	
	if (get_game_type()!=gt_indt)
	{
  	if ((app_ui_data->chess_save_done==GAME_SAVE_SAVING) || (get_game_state() != gs_running)) {
    	/* Show startup screen */
    	gtk_window_present(GTK_WINDOW(app_ui_data->app));
    	while(gtk_events_pending())
      	gtk_main_iteration();
  	}
	}
	else
	{
		/* The loading takes too much time. Make ensensitive all during the load */
		app_ui_data->plugin->info->plugin_cb(NULL,(gpointer)MA_GAME_PLAYING_START);
		gtk_widget_set_sensitive(app_ui_data->play_button,FALSE);
		while(gtk_events_pending())
      	gtk_main_iteration();
		app_ui_data->plugin->info->plugin_cb(NULL,(gpointer)MA_WAIT_FOR_SAVE);
		if ((get_game_state() != gs_running)) {
    	/* Show startup screen */
    	gtk_window_present(GTK_WINDOW(app_ui_data->app));
    	while(gtk_events_pending())
      	gtk_main_iteration();
  	}
	}
	
  if (get_game_state() == gs_running) {
    return;
  }
    
  /* Show startup screen */
  /* 
  gtk_window_present(GTK_WINDOW(app_ui_data->app));
  while(gtk_events_pending())
    gtk_main_iteration(); 
  */
  
  if (get_game_type() == gt_chess) {   
    /* Save the selected file name */
    settings_set_string(SETTINGS_CHESS_SAVE_FILE, filename);
      
    /* Update recent menu */
    app_ui_data->chess_recent_items = settings_get_list_string(SETTINGS_CHESS_RECENT_ITEMS);
    add_recent_item(app_ui_data, filename);
    ui_update_recent_items(app_ui_data);
      
    /* Load game settings */
    set_settings(app_ui_data);

    while(gtk_events_pending())
      gtk_main_iteration();

    game_state = gs_running;
    
    /* Attempt to load game */
    game_load();        
  }
	else if (get_game_type() == gt_indt)
	{
		settings_set_string(SETTINGS_CHESS_SAVE_FILE, filename);
		app_ui_data->plugin->info->plugin_cb(NULL,(gpointer)MA_SETTING_RECENT_ITEMS);
		game_state = gs_running;
		app_ui_data->plugin->info->plugin_cb(NULL,(gpointer)MA_LOAD_MIME);
    app_ui_data->plugin->info->plugin_cb(NULL,(gpointer)MA_GAME_PLAYING);
		//gtk_widget_set_sensitive(app_ui_data->play_button,TRUE);
	}
}
