/**
  @file plugin.c
  
  The plugin for the games-startup

  Copyright (c) 2004, 2005 Nokia Corporation.
	
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
    02111-1307 USA
*/

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <sys/vfs.h>
//#include "../chessui_i18n.hh"
#include <libgen.h>
#include <gtk/gtk.h>

#include <gconf/gconf.h>
#include <gconf/gconf-client.h>

#include <libgnomevfs/gnome-vfs.h>

#include <glib.h>
#include <glib/gstdio.h>
#include <glib-object.h>
#include <gdk/gdkkeysyms.h>

#include <osso-helplib.h>

/* Hildon */
#include <hildon-widgets/hildon-app.h>
#include <hildon-widgets/hildon-appview.h>
#include <hildon-widgets/gtk-infoprint.h>
#include <hildon-widgets/hildon-note.h>
#include <hildon-widgets/hildon-file-chooser-dialog.h>
#include <hildon-widgets/hildon-file-handling-note.h>
#include <hildon-widgets/hildon-caption.h>
#include <startup_plugin.h>
#ifdef VERSION_3
#define D_(s) dgettext ("hildon-fm", s)
#endif
#define COMBOBOX_HEIGHT 30
#define COMBOBOX_WIDTH 150
#define LABEL_HEIGHT 50
#define LABEL_WIDTH -1

#define SETTINGS_ENABLE_SOUND              "/apps/osso/chess/enable_sound"
#define SETTINGS_SHOW_LEGAL_MOVES          "/apps/osso/chess/show_legal_moves"
#define SETTINGS_PLAYER_COLOR              "/apps/osso/chess/player_color_white"
#define SETTINGS_OPPONENT_HUMAN            "/apps/osso/chess/opponent_type_human"
#define SETTINGS_LEVEL                     "/apps/osso/chess/difficulty_level"

#define SETTINGS_CHESS_SAVE_FOLDER_DEFAULT ".games"
#define OSSO_CHESS_HELP_PATH "//chess/a/1"
#define OSSO_CHESS_HELP_LOADFILE           "//chess/a/3"
#define OSSO_CHESS_HELP_SAVEFILE           "//chess/a/2"
#define CHESS_FILE_GLOB                    "*.game"
#define SETTINGS_CHESS_SAVE_FILE           "/apps/osso/chess/save_file"
#define SETTINGS_CHESS_MAX_RECENT_ITEMS    6
#define SETTINGS_CHESS_RECENT_ITEMS        "/apps/osso/chess/recent_items"
#define DEFAULT_STATE_SAVE_FILE            "/tmp/osso-chess-save"
#define CHESS_PAUSE_SAVE_FILE_TMP "/tmp/chess_state_file.pgn"
#define CHESS_SAVE_FILE_TMP "/tmp/osso_chess.tmp"
#define CHESS_SERVICE "com.nokia.osso_chess"
#define CHESS_OBJPATH "/com/nokia/osso_chess"
#define CHESS_IFACE "com.nokia.osso_chess"

#define MENU_RECENT_MENU_WIDTH 15

#define GAME_SAVE_METHOD     "game_save"
#define GAME_LOAD_METHOD     "game_load"

#define URI_FILE_PREFIX   "file://"
#define MMC_MOUNTPOINT_ENV "MMC_MOUNTPOINT"
gboolean is_mmc;
gchar *mmc_uri;


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#ifdef ENABLE_NLS
#include <libintl.h>
//#define _(String) gettext(String)
#ifdef gettext_noop
#define N_(String) gettext_noop(String)
#else
#define N_(String) (String)
#endif
#else /* NLS is disabled */
#define _(String) (String)
#define N_(String) (String)
#define textdomain(String) (String)
#define gettext(String) (String)
#define dgettext(Domain,String) (String)
#define dcgettext(Domain,String,Type) (String)
#define bindtextdomain(Domain,Directory) (Domain) 
#define bind_textdomain_codeset(Domain,Codeset) (Codeset) 
#endif /* ENABLE_NLS */

#define _(String)                 dgettext("osso-games", String)
#define __(String)                dgettext("hildon-common-strings", String)

#define CHESS_MENU_ITEMS          3

#define CHESS_MENU_HUMAN_WHITE         0
#define CHESS_MENU_HUMAN_BLACK         1
#define CHESS_MENU_OPPONENT_EASY       2
#define CHESS_MENU_OPPONENT_MEDIUM     3
#define CHESS_MENU_OPPONENT_HARD       4
#define CHESS_MENU_OPPONENT_EXPERT     5
#define CHESS_MENU_OPPONENT_HUMAN      6
#define CHESS_MENU_SOUND               7
#define CHESS_MENU_SHOW_LEGAL_MOVES    8
//#define CHESS_MENU_9     9

/* menu_action identifiers */
#define MA_GAME_PLAY      1
#define ME_GAME_OPEN     20
#define ME_GAME_SAVE     21
#define ME_GAME_SAVE_AS  22

#define MA_GAME_RECENT_1 0
#define MA_GAME_RECENT_2 1
#define MA_GAME_RECENT_3 2
#define MA_GAME_RECENT_4 3
#define MA_GAME_RECENT_5 4
#define MA_GAME_RECENT_6 5
#define MA_GAME_PLAYING_START 30
#define MA_GAME_PLAYING 31
#define MA_WAIT_FOR_SAVE 32
#define MA_SETTING_RECENT_ITEMS 33
#define MA_LOAD_MIME 34
#define GAME_PAUSED      2

enum { chess_human_invalid=-1, chess_human_white=0, chess_human_black };
enum { chess_opponent_1=0, chess_opponent_2, chess_opponent_3,
       chess_opponent_4, chess_opponent_human };

enum {
  GAME_SAVE_NONE=1,
  GAME_SAVE_SAVING,
  GAME_SAVE_OK,
  GAME_SAVE_ERR,
  GAME_SAVE_CANCEL
};

enum {
  GAME_SAVE=5,
  GAME_LOAD
};

enum {
  GAME_LOAD_FILE_UNSUPPORTED=1,
  GAME_LOAD_FILE_NOT_FOUND
};

GConfClient *gcc = NULL;
GtkWidget *human_box = NULL; 
GtkWidget *opponent_box = NULL;
GtkWidget *sound_check = NULL;
GtkWidget *show_legal_moves_check = NULL;
//GnomeVFSMonitorHandle *gnomevfs_handle = NULL;
gchar *chess_title  = NULL;
osso_context_t *osso;

//Menu
GtkWidget *menu_items[CHESS_MENU_ITEMS];
GtkWidget *settings_menu;
GtkWidget *settings_item;
GtkWidget *human_item;
GtkWidget *help_item;
GtkWidget *human_menu;
GtkWidget *opponent_item;
GtkWidget *opponent_menu;
GtkWidget *human_white_item;
GtkWidget *human_black_item;
GtkWidget *opponent_easy_item;
GtkWidget *opponent_medium_item;
GtkWidget *opponent_hard_item;
GtkWidget *opponent_expert_item;
GtkWidget *opponent_human_item;
GtkWidget *sound_item;
GtkWidget *show_legal_moves_item;
GtkWidget *recent_item;
GtkWidget *recent_menu;
GtkWidget *saving_progress_bar;
GtkWidget *human_label;
GtkWidget *opponent_label;
GtkWidget *sound_label;
GtkWidget *show_legal_moves_label;
	
GSList * group = NULL;
GSList * opponent_group = NULL;

gboolean is_new_file;

GnomeVFSHandle *read_handle=NULL;
   

static GtkWidget  *load_plugin                      (void);
static void       unload_plugin                     (void);
static void       write_config                      (void);
static void       update_menu                       (void);
static void       game_load_cb                      (void);
static void       game_save_cb                      (void);
static void       game_save_as_cb                   (void);
static void       plugin_ui_update_recent_items            (gboolean check_needed);
static void       game_save                         (void);
static void       plugin_ui_hide_saving_dialog             (void);
static void       ui_show_nomem_dialog              (void);
static void       plugin_ui_show_saving_dialog             (void);
static gboolean   ui_set_saving_dialog_progress     (gfloat       progress);
static GtkWidget  **load_menu                       (guint       *nitems);
static void       add_recent_item                   (gchar       *file);
static void       game_load                         (gchar       *file);
static gboolean   ui_show_replace_file_dialog       (gchar       *file);
static gboolean   ui_show_file_chooser              (gboolean     open);
static void       game_load_recent_cb               (GtkWidget   *menu_item,
                                                     int          num);
static void       plugin_callback                   (GtkWidget   *menu_item,
                                                     gpointer     data);
static void       chess_human_cb                    (GtkWidget   *widget, 
                                                     gpointer     data);
static void       chess_opponent_cb                 (GtkWidget   *widget,
                                                     gpointer     data);
static void       chess_sound_cb                    (GtkWidget   *widget,
                                                     gpointer     data);
static void       chess_show_cb                     (GtkWidget   *widget, 
                                                     gpointer     data);
static void       plugin_cb                         (GtkWidget   *menu_item,
                                                     gpointer     cb_data);
static gboolean   settings_set_list_string          (const gchar *key, 
                                                     GSList      *val);
static gchar*     settings_get_string               (const gchar *key);
/*static void       chess_saved_file_monitor          (GnomeVFSMonitorHandle    *handle,
			                                         const gchar              *monitor_uri,
                                                     const gchar              *info_uri,
                                                     GnomeVFSMonitorEventType  event_type,
			                                         gpointer                  user_data);*/ /* Not needed, if we keep open a file descriptor*/
//static void       remove_whitespace                 (char *str);
static void       plugin_volume_unmounted_cb        (GnomeVFSVolumeMonitor *vfsvolumemonitor,
                                                     GnomeVFSVolume *volume, gpointer user_data);
/*static void       plugin_dbus_message_async_result  (const gchar *interface,
			                                         const gchar *method,
			                                         osso_rpc_t *retval, gpointer data);*/
/*static osso_return_t plugin_send_dbus_message_async  (const gchar *service, 
				                                      const gchar *object_path,
				                                      const gchar *iface,
				                                      const gchar *method,
													  GArray *args, 
													  gpointer data
													  );*/
//static gboolean   plugin_send_message_async				  (const guint state);													  
static void       plugin_ui_show_save_nommc_dialog	  (void);
static void       plugin_ui_show_load_err             (int err_type, gchar *loadfile);
static gboolean   plugin_settings_get_bool_fallback	  (gchar *key, gboolean fall);
static void       plugin_get_settings                 (void);
static GSList*    plugin_settings_get_list_string     (const gchar *key);
static void 	  plugin_remove_recent_item			  ( gchar *file);
static GConfValue *plugin_settings_get				  (const gchar *key);
static gboolean   plugin_settings_get_bool			  (const gchar *key);
static gint 	  plugin_settings_get_int_fallback	  (gchar *key, gint fall);
static gchar* 	  plugin_settings_get_string		  (const gchar *key);
static gint 	  plugin_settings_get_int			  (const gchar *key);
static void 	  plugin_show_infoprint				  (gchar *msg);
static void 	  plugin_set_settings				  (void);
static void 			set_all_insensitive							(void);
static void 			plugin_on_insensitive_menu_press(GtkWidget * widget, gchar * message);
static void 			game_help_cb (GtkMenuItem *menuitem, gpointer user_data);
static void 			plugin_ui_cancel_saving(void);
static void				plugin_saving_dialog_cb(GtkDialog *dialog, gint button, gpointer user_data);
static gboolean 	plugin_settings_set_bool(const gchar *key, const gboolean val);
static gboolean 	plugin_settings_set_string(const gchar *key, const gchar *val);
static gboolean 	plugin_settings_set_int(const gchar *key, const gint val);
static void 			plugin_mime_open(void);
static void 			plugin_set_recent_item(void);
static void 			plugin_wait_for_save(void);
static gboolean plugin_settings_set_list_string(const gchar *key, GSList *val);
#ifdef VERSION_3
gboolean    plugin_select_separator  (GtkTreeModel *model,GtkTreeIter *iter,gpointer data);
gboolean    plugin_sound_keypress_cb      (GtkWidget   *widget,GdkEventKey *event,gpointer     user_data);
gboolean    plugin_opponent_keypress_cb      (GtkWidget   *widget,GdkEventKey *event,gpointer     user_data);
#endif

static StartupPluginInfo plugin_info = {
  load_plugin,
  unload_plugin,
  write_config,
  load_menu,
  update_menu,
  plugin_cb
};


typedef struct _StartupApp        StartupApp;
struct _StartupApp {
  StartupConfig *config;
  osso_context_t *osso;
  guint state;

  startup_app_top_cb *top_cb;
  gpointer top_cb_data;
  startup_app_state_change_cb *state_change_cb;
  gpointer state_change_cb_data;

  const gchar *service;
  const gchar *path;
  const gchar *iface;
};



GameStartupInfo gs;

STARTUP_INIT_PLUGIN(plugin_info, gs, FALSE, TRUE)

static int changed = FALSE;
//static GtkWidget *chess_recent_menu;
static GSList *chess_recent_items;
static gchar *chess_save_folder;
static gboolean chess_saved;
static gint chess_save_done;
static GtkWidget *chess_saving_dialog;


static gboolean plugin_settings_set_list_string(const gchar *key, GSList *val)
{
  osso_log(LOG_DEBUG, "Setting key %s value to list of chars\n", key);
  return gconf_client_set_list(gcc, key, GCONF_VALUE_STRING, val, NULL);
}

static void plugin_on_insensitive_menu_press(GtkWidget *widget, gchar *message)
{
  widget=NULL;
  gtk_infoprint( GTK_WINDOW (gs.ui->hildon_app), message);
}

/* Set int type value */
static gboolean plugin_settings_set_int(const gchar *key, const gint val)
{
  return gconf_client_set_int(gcc, key, val, NULL);
}


/* Set string type value */
static gboolean plugin_settings_set_string(const gchar *key, const gchar *val)
{
  return gconf_client_set_string(gcc, key, val, NULL);
}


static gboolean plugin_settings_set_bool(const gchar *key, const gboolean val)
{
  return gconf_client_set_bool(gcc, key, val, NULL);
}


/* Get int type key value */
static gint plugin_settings_get_int(const gchar *key)
{
  return gconf_client_get_int(gcc, key, NULL);
}

/* Get string type key value */
static gchar* plugin_settings_get_string(const gchar *key)
{
  return gconf_client_get_string(gcc, key, NULL);
}

/**
 Helper function to get an int entry, returning defined value, if not found.
 @param key GConf key to be get.
 @param fall Fallback to this, if not found, or invalid.
 @return Value got from GConf or the value specified in fall.
*/
static gint plugin_settings_get_int_fallback(gchar *key, gint fall)
{
  GConfValue *gc_val = plugin_settings_get(key);
  if (gc_val)
  {
    if (gc_val->type == GCONF_VALUE_INT) {
      g_free(gc_val);
      return plugin_settings_get_int(key);
    }
    g_free(gc_val);
  }
  return fall;
}

/* Get boolean type key value */
static gboolean plugin_settings_get_bool(const gchar *key)
{
  return gconf_client_get_bool(gcc, key, NULL);
}

/* Get stringlist type key value */
static GSList *plugin_settings_get_list_string(const gchar *key)
{
  return gconf_client_get_list(gcc, key, GCONF_VALUE_STRING, NULL);
}

/* Get key value */
static GConfValue *plugin_settings_get(const gchar *key)
{
  return gconf_client_get(gcc, key, NULL);
}

/**
 Helper function to get an bool entry, returning defined value, if not found.
 @param key GConf key to be get.
 @param fall Fallback to this, if not found, or invalid.
 @return Value got from GConf or the value specified in fall.
*/
static gboolean plugin_settings_get_bool_fallback(gchar *key, gboolean fall)
{
  GConfValue *gc_val = plugin_settings_get(key);
  
  if (gc_val)
  {
    if (gc_val->type == GCONF_VALUE_BOOL) {
      g_free(gc_val);
      return plugin_settings_get_bool(key);
    }
    g_free(gc_val);
  }
  return fall;
}

/**
  Get settings from GConf.

 */
static void plugin_get_settings(void)
{
  gboolean white = TRUE;
  gboolean human = FALSE;
  gboolean sound = FALSE;
  gboolean moves = FALSE;
  gint player = -1;
  gint difficulty = -1;


    /* Get settings */
    white = plugin_settings_get_bool_fallback(SETTINGS_PLAYER_COLOR, TRUE);
    sound = plugin_settings_get_bool_fallback(SETTINGS_ENABLE_SOUND, TRUE);

    difficulty = plugin_settings_get_int_fallback(SETTINGS_LEVEL, 4);

    human = plugin_settings_get_bool(SETTINGS_OPPONENT_HUMAN);
    moves = plugin_settings_get_bool(SETTINGS_SHOW_LEGAL_MOVES);

    /* Sound */
    gtk_toggle_button_set_active(
      GTK_TOGGLE_BUTTON( sound_check), sound );

    /* Legal moves */
    gtk_toggle_button_set_active(
      GTK_TOGGLE_BUTTON( show_legal_moves_check), moves );

    /* Human color */
    if (white == TRUE) {
      player = chess_human_white;
    } else {
      player = chess_human_black;
    }
    gtk_combo_box_set_active(
      GTK_COMBO_BOX( human_box ), player );
    
    if (difficulty==4)
    {
      difficulty=5;
    }

    /* Opponent difficulty and type */
    gtk_combo_box_set_active(
      GTK_COMBO_BOX( opponent_box ), difficulty );

}

#ifdef VERSION_3

gboolean    plugin_opponent_keypress_cb      (GtkWidget   *widget,
                                            GdkEventKey *event,
                                            gpointer     user_data)
{
  if (event->keyval==GDK_Down)
  {
    gtk_widget_grab_focus(sound_label);
    return TRUE;
  }
  return FALSE;
}

gboolean    plugin_sound_keypress_cb      (GtkWidget   *widget,
                                            GdkEventKey *event,
                                            gpointer     user_data)
{
  if (event->keyval==GDK_Up)
  {
    gtk_widget_grab_focus(opponent_label);
    return TRUE;
  }
  return FALSE;
}


gboolean    plugin_select_separator  (GtkTreeModel *model,
                                             GtkTreeIter *iter,
                                             gpointer data)
{
   GtkTreePath *path;
   gboolean result;
 
   path = gtk_tree_model_get_path (model, iter);
   result = gtk_tree_path_get_indices (path)[0] == 4;
   gtk_tree_path_free (path);
   return result;
}
#endif


/* Load the plugin, create the UI */
static GtkWidget *
load_plugin (void)
{
  GtkWidget *game_hbox, *game_vbox;
  GtkSizeGroup *groupleft= GTK_SIZE_GROUP( gtk_size_group_new( GTK_SIZE_GROUP_BOTH ) ),*groupright= GTK_SIZE_GROUP( gtk_size_group_new( GTK_SIZE_GROUP_BOTH ) );
  int enable_sound, show_legal_moves, color_white,  difficulty_level;
  gboolean gnomevfs_ok;  
	
	if (!gnome_vfs_initialized()) {
    gnomevfs_ok = gnome_vfs_init();
    if (!gnomevfs_ok) {
        puts("GnomeVFS init failed");
        return NULL;
    }
  }

  bindtextdomain (GETTEXT_PACKAGE, CHESSLOCALEDIR);
  bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
  gcc = gconf_client_get_default();
  //enable_sound = gconf_client_get_bool (gcc, SETTINGS_ENABLE_SOUND, NULL);
	enable_sound = plugin_settings_get_bool_fallback(SETTINGS_ENABLE_SOUND,TRUE);
  //show_legal_moves = gconf_client_get_bool (gcc, SETTINGS_SHOW_LEGAL_MOVES, NULL);
	show_legal_moves =plugin_settings_get_bool_fallback(SETTINGS_SHOW_LEGAL_MOVES,FALSE);
  //color_white = !gconf_client_get_bool (gcc, SETTINGS_PLAYER_COLOR, NULL);
	color_white = !plugin_settings_get_bool_fallback(SETTINGS_PLAYER_COLOR,TRUE);
  //opponent_type = gconf_client_get_bool (gcc, SETTINGS_OPPONENT_HUMAN, NULL);
  //difficulty_level = gconf_client_get_int (gcc, SETTINGS_LEVEL, NULL);
  difficulty_level = plugin_settings_get_int_fallback(SETTINGS_LEVEL,0);

  game_vbox = gtk_vbox_new (FALSE, 0);
  g_assert (game_vbox);

  game_hbox = gtk_hbox_new (TRUE, 0);
  g_assert (game_hbox);

  //hbox = gtk_hbox_new(FALSE,0);
  //g_assert(hbox);
  
  

  human_box = gtk_combo_box_new_text ();
  g_assert (human_box);

  gtk_combo_box_append_text (GTK_COMBO_BOX (human_box), 
    _("game_va_chess_settings_human_white")); //  "White");
  gtk_combo_box_append_text (GTK_COMBO_BOX (human_box), 
    _("game_va_chess_settings_human_black")); //  "Black");
  gtk_widget_set_size_request(human_box,
    COMBOBOX_WIDTH, COMBOBOX_HEIGHT);
  gtk_combo_box_set_active (GTK_COMBO_BOX (human_box), color_white);
  human_label =hildon_caption_new(groupleft,_("game_fi_chess_settings_human"),human_box,NULL,HILDON_CAPTION_OPTIONAL);
	g_assert(human_label);
  //hildon_caption_set_separator(HILDON_CAPTION(human_label)," ");
  

  gtk_box_pack_start (GTK_BOX (game_hbox), game_vbox, FALSE, FALSE, 2);
  gtk_box_pack_start (GTK_BOX (game_vbox), human_label, FALSE, FALSE, 2);

  //hbox = gtk_hbox_new(FALSE, 0);
  
  sound_check = gtk_check_button_new ();
  g_assert (sound_check);
  sound_label = hildon_caption_new(groupright,_("game_fi_chess_settings_sound"),sound_check,NULL,HILDON_CAPTION_OPTIONAL);
	g_assert (sound_label);
  //hildon_caption_set_separator(HILDON_CAPTION(sound_label)," ");
  
  //gtk_box_pack_start (GTK_BOX (game_hbox),  sound_label, FALSE, FALSE, 2);

  

  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(sound_check), enable_sound);

  //gtk_box_pack_start (GTK_BOX (game_vbox), game_hbox, FALSE, FALSE, 2);

  //game_hbox = gtk_hbox_new (FALSE, 0);
  //g_assert (game_hbox);
  //hbox = gtk_hbox_new (FALSE, 0);
  //g_assert(hbox);
  
  
  opponent_box = gtk_combo_box_new_text ();
  g_assert (opponent_box);
#ifdef VERSION_3
  gtk_combo_box_set_row_separator_func(GTK_COMBO_BOX(opponent_box),plugin_select_separator,NULL,NULL);
#endif

  gtk_combo_box_append_text (GTK_COMBO_BOX (opponent_box),
    _("game_va_chess_settings_opponent_easy")); //  "Easy");
  gtk_combo_box_append_text (GTK_COMBO_BOX (opponent_box),
    _("game_va_chess_settings_opponent_medium")); //  "Medium");
  gtk_combo_box_append_text (GTK_COMBO_BOX (opponent_box),
    _("game_va_chess_settings_opponent_hard")); //  "hard");
  gtk_combo_box_append_text (GTK_COMBO_BOX (opponent_box),
    _("game_va_chess_settings_opponent_expert")); //  "Expert");
#ifdef VERSION_3
  gtk_combo_box_append_text (GTK_COMBO_BOX (opponent_box),
    "---"); //  "Expert");
#endif
  gtk_combo_box_append_text (GTK_COMBO_BOX (opponent_box),
    _("game_va_chess_settings_opponent_human")); //  "Human player");
  gtk_widget_set_size_request(opponent_box,
    COMBOBOX_WIDTH, COMBOBOX_HEIGHT);
  /*if (opponent_type) {
    gtk_combo_box_set_active (GTK_COMBO_BOX (opponent_box), 4);
  } else {*/
  if (difficulty_level==4)
  {
    difficulty_level=5;
  }
    gtk_combo_box_set_active (GTK_COMBO_BOX (opponent_box), difficulty_level);
  //}
	opponent_label =hildon_caption_new(groupleft,_("game_fi_chess_settings_opponent"),opponent_box,NULL,HILDON_CAPTION_OPTIONAL);
	g_assert(opponent_label);
	//hildon_caption_set_separator(HILDON_CAPTION(opponent_label)," ");
  gtk_widget_set_size_request(opponent_label,
    LABEL_WIDTH, LABEL_HEIGHT);

	gtk_box_pack_start (GTK_BOX (game_vbox),   opponent_label, FALSE, FALSE, 2);

  //hbox = gtk_hbox_new(FALSE, 0);
  game_vbox = gtk_vbox_new (FALSE, 0);
	g_assert(game_vbox);
	
	gtk_box_pack_start (GTK_BOX (game_hbox), game_vbox, FALSE, FALSE, 2);
	gtk_box_pack_start (GTK_BOX (game_vbox),  sound_label, FALSE, FALSE, 2);
  
  show_legal_moves_check = gtk_check_button_new ();
  g_assert (show_legal_moves_check);

  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(show_legal_moves_check), show_legal_moves);
	show_legal_moves_label = hildon_caption_new(groupright,_("game_fi_chess_settings_legal_moves"),show_legal_moves_check,NULL,HILDON_CAPTION_OPTIONAL);
	g_assert (show_legal_moves_label);
	//hildon_caption_set_separator(HILDON_CAPTION(show_legal_moves_label)," ");
  gtk_widget_set_size_request(show_legal_moves_label,
    LABEL_WIDTH, LABEL_HEIGHT);
		
  gtk_box_pack_start (GTK_BOX (game_vbox), show_legal_moves_label, FALSE, FALSE, 2);
  //gtk_box_pack_start (GTK_BOX (game_vbox), game_hbox, FALSE, FALSE, 2);

  //gtk_widget_show_all (game_vbox);
	gtk_widget_show_all (game_hbox);
  g_signal_connect (G_OBJECT(human_box), "changed",
      G_CALLBACK(chess_human_cb), NULL);

  g_signal_connect (G_OBJECT(opponent_box), "changed",
      G_CALLBACK(chess_opponent_cb), NULL);
#ifdef VERSION_3     
  g_signal_connect (G_OBJECT(opponent_box), "key-press-event",
      G_CALLBACK(plugin_opponent_keypress_cb), NULL);
#endif
  g_signal_connect (G_OBJECT(sound_check), "clicked",
      G_CALLBACK(chess_sound_cb), NULL);
#ifdef VERSION_3      
  g_signal_connect (G_OBJECT(sound_check), "key-press-event",
      G_CALLBACK(plugin_sound_keypress_cb), NULL);
#endif
  g_signal_connect (G_OBJECT(show_legal_moves_check), "clicked",
      G_CALLBACK(chess_show_cb), NULL);
  plugin_settings_set_string(SETTINGS_CHESS_SAVE_FILE, "");
  chess_recent_items =plugin_settings_get_list_string(SETTINGS_CHESS_RECENT_ITEMS);
  //return game_vbox;  
	return game_hbox;  
}

static void
unload_plugin (void)
{
	if (read_handle)
	{
		gnome_vfs_close(read_handle);
    read_handle=NULL;
	}
	write_config();
}

/* The following function is from INDT's games-startup code. The copyrights 
belongs to them. The INDT's games-startup was released under LGPL license. 
This function is modified under LGPL license*/

/* Write the state of the UI */
static void 
write_config (void)
{
  int combo;

  combo = gtk_combo_box_get_active (GTK_COMBO_BOX (opponent_box));	  
    gconf_client_set_bool (gcc, SETTINGS_ENABLE_SOUND, 
	gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (sound_check)), NULL);
    gconf_client_set_bool (gcc, SETTINGS_SHOW_LEGAL_MOVES, 
	gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (show_legal_moves_check)), NULL);
  //if (combo == 4) {
    //gconf_client_set_int (gcc, SETTINGS_LEVEL, 0, NULL);
    //gconf_client_set_bool (gcc, SETTINGS_OPPONENT_HUMAN, 1, NULL);
    //gconf_client_set_bool (gcc, SETTINGS_PLAYER_COLOR, 0, NULL);
  //} else {
  if (combo==5)
  {
    combo=4;
  }
    gconf_client_set_int (gcc, SETTINGS_LEVEL, combo, NULL);
    //gconf_client_set_bool (gcc, SETTINGS_OPPONENT_HUMAN, 0, NULL);
    gconf_client_set_bool (gcc, SETTINGS_PLAYER_COLOR, 
    	!gtk_combo_box_get_active (GTK_COMBO_BOX (human_box)), NULL);
	if (combo == 4) {
		gconf_client_set_bool (gcc, SETTINGS_OPPONENT_HUMAN, 1, NULL);
	}
	else
	{
		gconf_client_set_bool (gcc, SETTINGS_OPPONENT_HUMAN, 0, NULL);
	}
  //}
}

static void game_help_cb (GtkMenuItem *menuitem, gpointer user_data)
{
  StartupApp *app=NULL;
  menuitem=NULL;
  user_data=NULL;
	app=gs.ui->app;
  ossohelp_show(app->osso,
      OSSO_CHESS_HELP_PATH,
      0);
}


/* The following function is from INDT's games-startup code. The copyrights 
belongs to them. The INDT's games-startup was released under LGPL license. 
This function is modified under LGPL license*/

/* Creates the menu to add that to the main application */
static GtkWidget **
load_menu (guint *nitems)
{

  *nitems = CHESS_MENU_ITEMS;
  
  settings_item = gtk_menu_item_new_with_label (_("game_me_chess_main_menu_settings"));
  settings_menu = gtk_menu_new ();    
  menu_items[0] = settings_item;

  recent_item = gtk_menu_item_new_with_label (_("game_me_chess_main_menu_recent"));
  recent_menu = gtk_menu_new ();    
  menu_items[1] = recent_item;
	
  help_item = gtk_menu_item_new_with_label (_("game_me_chess_main_menu_help"));
  g_signal_connect (G_OBJECT (help_item), "activate",
                      G_CALLBACK (game_help_cb),
                      NULL);    
  menu_items[2] = help_item;
	

  gtk_menu_item_set_submenu (GTK_MENU_ITEM (settings_item), settings_menu);
  gtk_menu_item_set_submenu (GTK_MENU_ITEM (recent_item), recent_menu);
  gtk_widget_set_sensitive(recent_item, FALSE);


  //human settings
  human_menu = gtk_menu_new ();
  human_item = gtk_menu_item_new_with_label (_("game_me_chess_menu_settings_human"));
  gtk_menu_item_set_submenu (GTK_MENU_ITEM (human_item), human_menu);
  gtk_menu_append (GTK_MENU (settings_menu), human_item);

  human_white_item = gtk_radio_menu_item_new_with_label (group, _("game_me_chess_menu_settings_human_white"));
  gtk_menu_append (GTK_MENU (human_menu), human_white_item);

  group = gtk_radio_menu_item_group(GTK_RADIO_MENU_ITEM(human_white_item));

  human_black_item = gtk_radio_menu_item_new_with_label (group, _("game_me_chess_menu_settings_human_black"));
  gtk_menu_append (GTK_MENU (human_menu), human_black_item);

  g_signal_connect (G_OBJECT (human_white_item), "toggled",
                            G_CALLBACK (plugin_callback),
                            (gpointer) CHESS_MENU_HUMAN_WHITE);
  g_signal_connect (G_OBJECT (human_black_item), "toggled",
                            G_CALLBACK (plugin_callback),
                            (gpointer) CHESS_MENU_HUMAN_BLACK);
  
  gtk_menu_append (GTK_MENU (settings_menu), gtk_menu_item_new());
  
 
  //opponent settings
  opponent_menu = gtk_menu_new ();
  opponent_item = gtk_menu_item_new_with_label (_("game_me_chess_menu_settings_opponent"));   
  gtk_menu_item_set_submenu (GTK_MENU_ITEM (opponent_item), opponent_menu);
  gtk_menu_append (GTK_MENU (settings_menu), opponent_item);

  opponent_easy_item = gtk_radio_menu_item_new_with_label (opponent_group,
                                        _("game_me_chess_menu_settings_opponent_1"));
  gtk_menu_append (GTK_MENU (opponent_menu), opponent_easy_item);
  opponent_group = gtk_radio_menu_item_group(GTK_RADIO_MENU_ITEM(opponent_easy_item));

  opponent_medium_item = gtk_radio_menu_item_new_with_label (opponent_group, 
                                        _("game_me_chess_menu_settings_opponent_2"));
  gtk_menu_append (GTK_MENU (opponent_menu), opponent_medium_item);
  opponent_group = gtk_radio_menu_item_group(GTK_RADIO_MENU_ITEM(opponent_medium_item));

  opponent_hard_item = gtk_radio_menu_item_new_with_label (opponent_group, 
                                        _("game_me_chess_menu_settings_opponent_3"));
  gtk_menu_append (GTK_MENU (opponent_menu), opponent_hard_item);
  opponent_group = gtk_radio_menu_item_group(GTK_RADIO_MENU_ITEM(opponent_hard_item));

  opponent_expert_item = gtk_radio_menu_item_new_with_label (opponent_group, 
                                        _("game_me_chess_menu_settings_opponent_4"));
  gtk_menu_append (GTK_MENU (opponent_menu), opponent_expert_item);
  opponent_group = gtk_radio_menu_item_group(GTK_RADIO_MENU_ITEM(opponent_expert_item));

  gtk_menu_append (GTK_MENU (opponent_menu), gtk_menu_item_new());

  opponent_human_item = gtk_radio_menu_item_new_with_label (opponent_group, 
                                    _("game_me_chess_menu_settings_opponent_human"));
  gtk_menu_append (GTK_MENU (opponent_menu), opponent_human_item);

  g_signal_connect (G_OBJECT (opponent_easy_item),
                    "toggled",
                    G_CALLBACK (plugin_callback),
                    (gpointer) CHESS_MENU_OPPONENT_EASY);
  g_signal_connect (G_OBJECT (opponent_medium_item), 
                    "toggled",
                    G_CALLBACK (plugin_callback),
                    (gpointer) CHESS_MENU_OPPONENT_MEDIUM);
  g_signal_connect (G_OBJECT (opponent_hard_item),
                    "toggled",
                    G_CALLBACK (plugin_callback),
                    (gpointer) CHESS_MENU_OPPONENT_HARD);
  g_signal_connect (G_OBJECT (opponent_expert_item),
                    "toggled",
                    G_CALLBACK (plugin_callback),
                    (gpointer) CHESS_MENU_OPPONENT_EXPERT);
  g_signal_connect (G_OBJECT (opponent_human_item),
                    "toggled",
                    G_CALLBACK (plugin_callback),
                    (gpointer) CHESS_MENU_OPPONENT_HUMAN);
  //sound settings
  sound_item = gtk_check_menu_item_new_with_label(
                 _("game_me_chess_menu_settings_sound"));
  gtk_menu_append (GTK_MENU (settings_menu), sound_item);
  g_signal_connect (G_OBJECT (sound_item),
                    "toggled",
                    G_CALLBACK (plugin_callback),
                    (gpointer) CHESS_MENU_SOUND);
  gtk_check_menu_item_set_state (GTK_CHECK_MENU_ITEM(sound_item),
		  gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (sound_check)));

  //show legal moves settings
  show_legal_moves_item = gtk_check_menu_item_new_with_label(
                            _("game_me_chess_menu_settings_legal_moves"));
  gtk_menu_append (GTK_MENU (settings_menu), show_legal_moves_item);
  g_signal_connect (G_OBJECT (show_legal_moves_item),
                    "toggled",
                    G_CALLBACK (plugin_callback),
                    (gpointer) CHESS_MENU_SHOW_LEGAL_MOVES);
  gtk_check_menu_item_set_state (GTK_CHECK_MENU_ITEM(show_legal_moves_item),
		  gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (show_legal_moves_check)));
		  g_signal_connect(G_OBJECT(recent_menu),
		 "insensitive_press",
		 G_CALLBACK(plugin_on_insensitive_menu_press),
		 _("game_ib_not_available"));
	g_signal_connect(G_OBJECT(recent_item),
		 "insensitive_press",
		 G_CALLBACK(plugin_on_insensitive_menu_press),
		 _("game_ib_not_available"));
  plugin_ui_update_recent_items(TRUE);
  return menu_items;
}

/* The following function is from INDT's games-startup code. The copyrights 
belongs to them. The INDT's games-startup was released under LGPL license.*/

/** Update the menu if something is changed in the UI */
static void
update_menu (void)
{
  chess_human_cb (human_box, NULL);
  chess_opponent_cb (opponent_box, NULL);
  chess_sound_cb (sound_check, NULL);
  chess_show_cb (show_legal_moves_check, NULL);
}

/* The following function is from INDT's games-startup code. The copyrights 
belongs to them. The INDT's games-startup was released under LGPL license. */

/** Called when a menu is activated */
static void
plugin_callback (GtkWidget *menu_item,
                 gpointer  data)
{
  menu_item=NULL;
  switch ((int) data) {
    case CHESS_MENU_HUMAN_WHITE:
      if (!changed) {
        changed = TRUE;
        gtk_combo_box_set_active (GTK_COMBO_BOX (human_box), (int) data);
      }
      break;
    case CHESS_MENU_HUMAN_BLACK:
      if (!changed) {
        changed = TRUE;
/*        if (gtk_combo_box_get_active (GTK_COMBO_BOX (opponent_box)) == CHESS_MENU_OPPONENT_HUMAN-2){
          gtk_combo_box_set_active (GTK_COMBO_BOX (human_box), CHESS_MENU_HUMAN_WHITE);
          gtk_check_menu_item_set_state(GTK_CHECK_MENU_ITEM(human_white_item), TRUE);  
        } else {*/
        if ((int) data==4)
        {
          gtk_combo_box_set_active (GTK_COMBO_BOX (human_box), (int) 5);
        }
        else
        { 
          gtk_combo_box_set_active (GTK_COMBO_BOX (human_box), (int) data);
        }
        //}
      }
	  break;
    case CHESS_MENU_OPPONENT_EASY:
    case CHESS_MENU_OPPONENT_MEDIUM:
    case CHESS_MENU_OPPONENT_HARD:
    case CHESS_MENU_OPPONENT_EXPERT:
      if (!changed) {
        changed = TRUE;
        gtk_combo_box_set_active (GTK_COMBO_BOX (opponent_box), (int) data - 2);
        /*if ((int) data == CHESS_MENU_OPPONENT_HUMAN){
          gtk_combo_box_set_active (GTK_COMBO_BOX (human_box), CHESS_MENU_HUMAN_WHITE);
          gtk_check_menu_item_set_state(GTK_CHECK_MENU_ITEM(human_white_item), TRUE); 
		}*/
      }
      break;
    case CHESS_MENU_OPPONENT_HUMAN:
      if (!changed) {
        changed = TRUE;
        gtk_combo_box_set_active (GTK_COMBO_BOX (opponent_box), (int) data - 1);
      }
      break;
     case CHESS_MENU_SOUND:
       if (!changed) {
          changed = TRUE;
          gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (sound_check), 
                                        gtk_check_menu_item_get_active (
                                          GTK_CHECK_MENU_ITEM(sound_item)));
       }
       break;
     case CHESS_MENU_SHOW_LEGAL_MOVES:
       if (!changed) {
          changed = TRUE;
          gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (show_legal_moves_check),
                                        gtk_check_menu_item_get_active (
                                          GTK_CHECK_MENU_ITEM(show_legal_moves_item)));
       }
       break;
  }

  changed = FALSE;
}

/* The following function is from INDT's games-startup code. The copyrights 
belongs to them. The INDT's games-startup was released under LGPL license. 
This function is modified under LGPL license*/

/** When the human setting is changed, then change the menu too */
static void 
chess_human_cb (GtkWidget *widget, 
                gpointer   data)
{
   if(!changed) {
      changed = TRUE;
      gint active = gtk_combo_box_get_active (GTK_COMBO_BOX (widget));
      if (active == CHESS_MENU_HUMAN_BLACK)
                 //&& (gtk_combo_box_get_active (GTK_COMBO_BOX (opponent_box)) != CHESS_MENU_OPPONENT_HUMAN-2)) {
	  {
        gtk_check_menu_item_set_state(GTK_CHECK_MENU_ITEM(human_black_item), TRUE);
      } else {
        gtk_check_menu_item_set_state(GTK_CHECK_MENU_ITEM(human_white_item), TRUE);
        gtk_combo_box_set_active (GTK_COMBO_BOX (human_box), (int) data);
      }
   }
   
   changed = FALSE;
}

/* The following function is from INDT's games-startup code. The copyrights 
belongs to them. The INDT's games-startup was released under LGPL license. 
This function is modified under LGPL license*/

/** When the opponent setting is changed, then change the menu too */
static void 
chess_opponent_cb (GtkWidget *widget, 
                   gpointer   data)
{
  data=NULL;
  if (!changed) {
    changed = TRUE;
    gint active = gtk_combo_box_get_active (GTK_COMBO_BOX (widget));

    switch (active) {
      default:
      case 0:
        gtk_check_menu_item_set_state(GTK_CHECK_MENU_ITEM(opponent_easy_item), TRUE);
        break;
      case 1:
        gtk_check_menu_item_set_state(GTK_CHECK_MENU_ITEM(opponent_medium_item), TRUE);
        break;
      case 2:
        gtk_check_menu_item_set_state(GTK_CHECK_MENU_ITEM(opponent_hard_item), TRUE);
        break;
      case 3:
        gtk_check_menu_item_set_state(GTK_CHECK_MENU_ITEM(opponent_expert_item), TRUE);
        break;
      case 5:
        gtk_check_menu_item_set_state(GTK_CHECK_MENU_ITEM(opponent_human_item), TRUE);
		//gtk_combo_box_set_active (GTK_COMBO_BOX (human_box), CHESS_MENU_HUMAN_WHITE);
        //gtk_check_menu_item_set_state(GTK_CHECK_MENU_ITEM(human_white_item), TRUE); 
        break;
    }
  }
  changed = FALSE;
}

/* The following function is from INDT's games-startup code. The copyrights 
belongs to them. The INDT's games-startup was released under LGPL license. */

/** When the sound setting is changed, then change the menu too */
static void 
chess_sound_cb (GtkWidget *widget, 
                gpointer   data)
{
  data=NULL;
  if (!changed) {
    changed = TRUE;
    gtk_check_menu_item_set_state (GTK_CHECK_MENU_ITEM(sound_item), gtk_toggle_button_get_active(
                                                 GTK_TOGGLE_BUTTON (widget)));
  }
  changed = FALSE;           
}

/* The following function is from INDT's games-startup code. The copyrights 
belongs to them. The INDT's games-startup was released under LGPL license. */

/** When the show legal moves setting is changed, then change the menu too */
static void 
chess_show_cb (GtkWidget *widget, 
               gpointer   data)
{
  data=NULL;
  if (!changed) {
    changed = TRUE;
    gtk_check_menu_item_set_state (GTK_CHECK_MENU_ITEM(show_legal_moves_item), 
                                   gtk_toggle_button_get_active(
                                     GTK_TOGGLE_BUTTON (widget)));
  }
  changed = FALSE;           
}

/* Callback function for the load menu */
static void game_load_cb(void)
{
  if (ui_show_file_chooser(TRUE) == TRUE) {
    plugin_set_settings();
    plugin_ui_update_recent_items(FALSE);
    //game_load();
  }
  //ui_update_status((AppUIData*)data);
}

/* Callback to the save menu */
static void game_save_cb(void)
{
  gchar *file = settings_get_string(SETTINGS_CHESS_SAVE_FILE);
  GnomeVFSFileInfo *file_info = gnome_vfs_file_info_new();
  GnomeVFSResult vfs_res = file ? gnome_vfs_get_file_info(file,
      file_info,
      GNOME_VFS_FILE_INFO_FOLLOW_LINKS) : GNOME_VFS_ERROR_GENERIC;
	if (read_handle)
	{
		gnome_vfs_close(read_handle);
		read_handle=NULL;
	}
  plugin_set_settings();
  if (vfs_res == GNOME_VFS_OK &&
      file_info->type == GNOME_VFS_FILE_TYPE_REGULAR) {
    is_new_file = FALSE;
    game_save();
  }
  else if (ui_show_file_chooser(FALSE) == TRUE) {
    is_new_file = TRUE;
    game_save();
  }

  gnome_vfs_file_info_unref(file_info);

  //ui_update_status((AppUIData*)data);

  if (file != NULL) 
    g_free(file);
}

/* Callback to the save_as menu */
static void game_save_as_cb(void)
{
  gchar *old_filename = settings_get_string(SETTINGS_CHESS_SAVE_FILE);
	if (read_handle)
	{
		gnome_vfs_close(read_handle);
		read_handle=NULL;
	}
  
  if (ui_show_file_chooser(FALSE) == TRUE) {
    gchar *new_filename = plugin_settings_get_string(SETTINGS_CHESS_SAVE_FILE);
    if (g_strcasecmp(old_filename, new_filename) != 0) {
      is_new_file = TRUE;
    } else {
      is_new_file = FALSE;
    }
    if (new_filename) {
      g_free(new_filename);
    }
    
    game_save();
		
  }
  //ui_update_status((AppUIData*)data);
  
  if (old_filename) {
    g_free(old_filename);
  }
}


/* The filechooser dialog */
static gboolean 
ui_show_file_chooser(gboolean open)
{
  GtkWidget *fs = NULL;
  GtkFileFilter *ff = NULL;
  gchar *buf;
  StartupApp *app=NULL;
	app=gs.ui->app;
  
   
  /* Get & set default folder */
  if (!chess_save_folder || !*chess_save_folder)
  {
    if (chess_save_folder)
      g_free(chess_save_folder);
    chess_save_folder = g_build_filename((const gchar*)getenv("MYDOCSDIR"),
        SETTINGS_CHESS_SAVE_FOLDER_DEFAULT, NULL);
  }
  
  /* Make dialog */
  fs = hildon_file_chooser_dialog_new(
    GTK_WINDOW(gs.ui->hildon_app),
    open?
    GTK_FILE_CHOOSER_ACTION_OPEN:
    GTK_FILE_CHOOSER_ACTION_SAVE
    );

  ossohelp_dialog_help_enable(
            GTK_DIALOG(fs),
            open ? OSSO_CHESS_HELP_LOADFILE : OSSO_CHESS_HELP_SAVEFILE,
            app->osso);
  

  if (open)
    g_object_set(G_OBJECT(fs),
      "empty-text",
      _("sfil_va_select_object_no_objects_games"),
      NULL
      );

  if (fs == NULL) return FALSE;
  g_assert(fs);

  gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (fs),
      chess_save_folder);

  if (!open) {
    buf=plugin_settings_get_string(SETTINGS_CHESS_SAVE_FILE);
    //buf = g_strdup_printf("%s.game", _("sfil_va_save_object_name_stub_chess"));
    if (!buf||!strlen(buf))
    {
      if (buf)
        g_free(buf);
      buf = g_strdup_printf("%s.game", _("sfil_va_save_object_name_stub_chess"));
    }
    else
    {
      gchar *fname=NULL;
      fname=gnome_vfs_unescape_string(g_strrstr(buf,"/")+1,NULL);
      g_free(buf);
      buf=fname;
    }
    gtk_file_chooser_set_current_name (GTK_FILE_CHOOSER (fs), buf);
    g_free(buf);
  }

  /* Set dialog title */
  if (open) {
    gtk_window_set_title (GTK_WINDOW (fs), _("game_ti_load_game_title"));
  } else {
    gtk_window_set_title (GTK_WINDOW (fs), _("game_ti_save_game_title"));
  }

  /* Set file filter */
  if (open) {
    ff = gtk_file_filter_new();
    gtk_file_filter_add_pattern(ff, CHESS_FILE_GLOB);
    gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(fs), ff);
    gtk_file_chooser_set_filter(GTK_FILE_CHOOSER(fs), ff);
  }

  /* Run the dialog, and expect OK. Otherwise just destroy the dialog. */
  if (gtk_dialog_run (GTK_DIALOG (fs)) == GTK_RESPONSE_OK) {
    gchar *filename = NULL;
    gchar *pathname = NULL;
    gtk_widget_hide (fs);
    filename = gtk_file_chooser_get_uri(
      GTK_FILE_CHOOSER (fs));

    //remove_whitespace(filename);

    if (filename == NULL)
    {
      return FALSE;
    }
    
    if (!open)
    {
	GnomeVFSURI *test_uri = gnome_vfs_uri_new(filename);
      
      if (gnome_vfs_uri_exists(test_uri)) {
        if (!ui_show_replace_file_dialog(filename))
        {
          gnome_vfs_uri_unref(test_uri);
          g_free(filename);
          gtk_widget_destroy (fs);

          return FALSE;
        }
      }
      gnome_vfs_uri_unref(test_uri);
    }

    pathname = gtk_file_chooser_get_current_folder(
      GTK_FILE_CHOOSER (fs));

    if (pathname != NULL)
    {
      g_free(chess_save_folder);
      chess_save_folder = pathname;
    }
      /* Set filename */
      if (!plugin_settings_set_string(SETTINGS_CHESS_SAVE_FILE, filename) ) {
        /* Free */
		  
        g_free(filename);
        gtk_widget_destroy (fs);
        return FALSE;
      }
      
      if (open)
      {
        add_recent_item(filename);
      }

    /* register to monitoring */
/*    GnomeVFSResult gnomevfs_res;
    if (gnomevfs_handle != NULL) {
      gnomevfs_res = gnome_vfs_monitor_cancel(gnomevfs_handle);
    }
    gchar *foldername = g_path_get_dirname(filename);
    gnomevfs_res = gnome_vfs_monitor_add(&gnomevfs_handle,
  					foldername,
  					GNOME_VFS_MONITOR_DIRECTORY,
  					chess_saved_file_monitor,
  					NULL);
    g_free(foldername);*/

    if (open) {
      game_load(filename);
    } 
    /* Free */
    g_free(filename);
    gtk_widget_destroy (fs);

    return TRUE;
  }

  gtk_widget_destroy (fs);

  return FALSE;
}



/* Update the recent menu */
static void 
plugin_ui_update_recent_items(gboolean check_needed)
{ 
  chess_recent_items=plugin_settings_get_list_string(SETTINGS_CHESS_RECENT_ITEMS);
  GSList *recent = chess_recent_items;
  GnomeVFSResult vfs_res;
  int count = 0;
  //gtk_widget_set_sensitive(recent_item, TRUE);
  gtk_widget_hide_all(recent_menu);
  if (check_needed)
  {
	while (recent)
  {
    GnomeVFSFileInfo *file_info = gnome_vfs_file_info_new();
    gchar *path = (gchar *)recent->data;
    if (!g_str_has_prefix(path,"obex"))
    {
    vfs_res = gnome_vfs_get_file_info(path,
        file_info,
        GNOME_VFS_FILE_INFO_FOLLOW_LINKS);
    if (vfs_res != GNOME_VFS_OK ||
        file_info->type != GNOME_VFS_FILE_TYPE_REGULAR)
    {
      GSList *temp = recent;
      recent = g_slist_next(recent);
      g_free(temp->data);
      chess_recent_items = 
        g_slist_delete_link(chess_recent_items, temp);
		
    }
    else
    {
      recent = g_slist_next(recent);
		count++;
    }
    }
    else
    {
      recent = g_slist_next(recent);
		  count++;
    }
    gnome_vfs_file_info_unref(file_info);
    
  }
  }
  else
  {
    
  }
	recent = chess_recent_items;
  
  if (recent)
  {
    count=g_slist_length(recent);
  }

  /*gtk_widget_set_sensitive(recent_menu,
      recent ? TRUE : FALSE);*/
  /*gtk_widget_set_sensitive(chess_recent_menu,
      recent ? TRUE : FALSE);*/
  if (count)
  {
	  if (recent_item)
	  {
	    gtk_widget_set_sensitive(recent_item, TRUE);
	  }
    count=0;
    while (recent) {
      gchar *path = strchr((gchar *)recent->data, '/');
	    if (path)
	    {
        gchar *file = g_basename(path);
        gchar *temp;
        file=gnome_vfs_unescape_string(file,NULL);
        if (file&&(temp = strrchr(file, '.')))
        *temp = '\0';
        
        if (file)
        {
        GtkWidget *item = gtk_menu_item_new_with_label (file);
        gtk_menu_append (GTK_MENU (recent_menu), item);  
        g_signal_connect (G_OBJECT (item), "activate",
                      G_CALLBACK (plugin_cb),
                      (gpointer) count);    
            gtk_widget_show (item);
          }
          if (file)
          {
        g_free(file);
          }
        
      }
      recent = g_slist_next(recent);
      count++;
    }
  }
  else
  {
	 //if (recent_item)
	 {
	  gtk_widget_set_sensitive(recent_item, FALSE);
	 }
	  return;
  }
}


/* Show the replace dialog */
static gboolean 
ui_show_replace_file_dialog(gchar *file)
{
  gboolean answer = FALSE;
  gchar *fname=NULL,*bname=NULL;
  bname=g_path_get_basename(file);
  fname=gnome_vfs_unescape_string(bname,NULL);
  g_free(bname);
#ifndef VERSION_3
  gchar *tmp = g_strdup_printf( _("docm_nc_replace_file%s"), fname); /*NOLOC*/
#else
  gchar *tmp = g_strdup_printf( D_("docm_nc_replace_file"));
#endif
  g_free(fname);
  HildonNote *note = HILDON_NOTE(hildon_note_new_confirmation_add_buttons(
        GTK_WINDOW(gs.ui->hildon_app),
	tmp,
#ifndef VERSION_3
        _("docm_bd_replace_file_ok"),
#else
        D_("docm_bd_replace_file_ok"),
#endif
        1,
#ifndef VERSION_3
        _("docm_bd_replace_file_cancel"),
#else
        D_("docm_bd_replace_file_cancel"),
#endif
        0,
        NULL));
  answer = gtk_dialog_run(GTK_DIALOG(note));
  gtk_widget_destroy(GTK_WIDGET(note));

  g_free(tmp);
  return answer;
}

static void set_all_insensitive(void)
{
  gtk_widget_set_sensitive(human_label,FALSE);
	gtk_widget_set_sensitive(opponent_label,FALSE);
	gtk_widget_set_sensitive(sound_label,FALSE);
	gtk_widget_set_sensitive(show_legal_moves_label,FALSE);
	if (GTK_WIDGET_VISIBLE(gs.ui->play_button))
	{
		gtk_widget_set_sensitive(gs.ui->play_button,FALSE);
	} 
	if (GTK_WIDGET_VISIBLE(gs.ui->restart_button))
	{
		gtk_widget_set_sensitive(gs.ui->restart_button,FALSE);
	}

}

static void set_all_sensitive(void)
{
  gtk_widget_set_sensitive(human_label,TRUE);
	gtk_widget_set_sensitive(opponent_label,TRUE);
	gtk_widget_set_sensitive(sound_label,TRUE);
	gtk_widget_set_sensitive(show_legal_moves_label,TRUE);
}

/* The following function is from INDT's games-startup code. The copyrights 
belongs to them. The INDT's games-startup was released under LGPL license. */

static void plugin_wait_for_save(void)
{
	if ((chess_save_done==GAME_SAVE_SAVING)) {
    /* Show startup screen */
    gtk_window_present(GTK_WINDOW(gs.ui->hildon_appview));
    while(gtk_events_pending())
      gtk_main_iteration();
  }
	return;
}

static void plugin_set_recent_item(void)
{
	gchar *filename=NULL;
	filename=settings_get_string(SETTINGS_CHESS_SAVE_FILE);
	/* Update recent menu */
  chess_recent_items = plugin_settings_get_list_string(SETTINGS_CHESS_RECENT_ITEMS);
		
  add_recent_item(filename);
	g_free(filename);
      
  /* Load game settings */
  plugin_set_settings();

  while(gtk_events_pending())
    gtk_main_iteration();
	return;
}

static void plugin_mime_open(void)
{
	gchar *filename=NULL;
	filename=settings_get_string(SETTINGS_CHESS_SAVE_FILE);
	game_load(filename);
	g_free(filename);
	return;
}

/* Called when a menu item activated */
static void 
plugin_cb(GtkWidget *menu_item, gpointer cb_data)
{
  switch( (int) cb_data ) {
    case ME_GAME_OPEN:
      game_load_cb();
      break;
    case ME_GAME_SAVE:
      game_save_cb();
      break;
    case ME_GAME_SAVE_AS:
      game_save_as_cb();
      break;
    case MA_GAME_RECENT_1:
      game_load_recent_cb(menu_item,1);
      break;
    case MA_GAME_RECENT_2:
      game_load_recent_cb(menu_item,2);
      break;
    case MA_GAME_RECENT_3:
      game_load_recent_cb(menu_item,3);
      break;
    case MA_GAME_RECENT_4:
      game_load_recent_cb(menu_item,4);
      break;
    case MA_GAME_RECENT_5:
      game_load_recent_cb(menu_item,5);
      break;
    case MA_GAME_RECENT_6:
      game_load_recent_cb(menu_item,6);
      break; 
	case MA_GAME_PLAYING_START:
      set_all_insensitive();
      break; 
	case MA_GAME_PLAYING:
      set_all_sensitive();
      break; 
	
	case MA_WAIT_FOR_SAVE:
      plugin_wait_for_save();
      break; 
	case MA_SETTING_RECENT_ITEMS:
      plugin_set_recent_item();
      break; 
	case MA_LOAD_MIME:
      plugin_mime_open();
      break; 
  }
}

/* Remove a recent item from the slist*/
static void plugin_remove_recent_item( gchar *file)
{
  GSList *item = g_slist_find_custom(chess_recent_items,
      		file, (GCompareFunc)strcmp);
  if (item)
  {
    chess_recent_items =
      g_slist_remove_link(chess_recent_items, item);
    g_slist_free_1(item);
  }
  plugin_settings_set_list_string(SETTINGS_CHESS_RECENT_ITEMS,
      chess_recent_items);
}

/* Show a infopint */
static void plugin_show_infoprint(gchar *msg)
{
  
    gtk_infoprint( GTK_WINDOW(gs.ui->hildon_app), msg);
}

/* Show load error dialog */
static void plugin_ui_show_load_err(int err_type, gchar *loadfile)
{ 
  plugin_remove_recent_item(loadfile);
  plugin_ui_update_recent_items(FALSE); 

  switch (err_type) {
    case GAME_LOAD_FILE_UNSUPPORTED: 
    {
      gchar *filename = gnome_vfs_format_uri_for_display(loadfile);
      gchar *basename = g_path_get_basename(filename);
      gchar *temp = g_strrstr(basename, ".");
      if (temp) *temp = '\0';
      gchar *msg = g_strdup_printf(_("game_ni_unsupported_file%s"), basename);

      hildon_appview_set_title(gs.ui->hildon_appview, "");  
      plugin_show_infoprint(msg);
      
      g_free(msg);
      g_free(basename);
      g_free(filename);
      break;
    }
    case GAME_LOAD_FILE_NOT_FOUND:
    {
      HildonNote *note = HILDON_NOTE(hildon_note_new_information(
#ifndef VERSION_3
        GTK_WINDOW(gs.ui->hildon_app), _("sfil_ni_unable_to_open_file_not_found") ));
      hildon_note_set_button_text(note, _("sfil_ni_unable_to_open_file_not_found_ok"));
#else
      GTK_WINDOW(gs.ui->hildon_app), __("sfil_ni_unable_to_open_file_not_found") ));
      hildon_note_set_button_text(note, __("sfil_ni_unable_to_open_file_not_found_ok"));
#endif
      gtk_dialog_run(GTK_DIALOG(note));
      gtk_widget_destroy(GTK_WIDGET(note));
      break;
    }
  }
}

/* Load the game */
void game_load(gchar *filename)
{
  StartupApp *app=NULL;
	app=gs.ui->app;
  osso_rpc_t retval;
	osso_return_t ret;
	if (!plugin_settings_set_string(SETTINGS_CHESS_SAVE_FILE, filename))
	{
		exit(0);
	}
  gchar *loadfile = g_strdup(filename);
	if (read_handle)
	{
		gnome_vfs_close(read_handle);
		read_handle=NULL;
	}
  GnomeVFSResult result;
  result = gnome_vfs_open (&read_handle, loadfile, GNOME_VFS_OPEN_READ);
	if (result != GNOME_VFS_OK)
  {
    plugin_ui_show_load_err(GAME_LOAD_FILE_NOT_FOUND, loadfile);
    read_handle=NULL;
    return;
  }
  /* register to monitoring */
/*  GnomeVFSResult gnomevfs_res;  
  if (gnomevfs_handle != NULL) {
    gnomevfs_res = gnome_vfs_monitor_cancel(gnomevfs_handle);
  }
  gchar *foldername = g_path_get_dirname(loadfile);
  gnomevfs_res = gnome_vfs_monitor_add(&gnomevfs_handle,
					foldername,
					GNOME_VFS_MONITOR_DIRECTORY,
					chess_saved_file_monitor,
					NULL);
  g_free(foldername);*/
  /* end register to monitoring */ 
  ret=osso_rpc_run(app->osso,
  		      CHESS_SERVICE,
		      CHESS_OBJPATH,
		      CHESS_IFACE,
		      GAME_LOAD_METHOD,
		      &retval,
		      0
//		      args
		      );
	
	  if (OSSO_OK==ret)
	  {
			
    
    if ((retval.type == DBUS_TYPE_BOOLEAN) && (retval.value.b == FALSE)) {
      plugin_ui_show_load_err(GAME_LOAD_FILE_UNSUPPORTED, loadfile);
			if (loadfile != NULL)
    	g_free(loadfile);
			osso_rpc_free_val (&retval);
      plugin_settings_set_string(SETTINGS_CHESS_SAVE_FILE,"");
      if (read_handle)
      {
      gnome_vfs_close(read_handle);
        read_handle=NULL;
      }
      gs.startup_ui_state_change_cb(gs.ui->app, GAME_CLOSED,
                                 GAME_CLOSED, gs.ui);
			return;
    } else {
			
      plugin_get_settings();
    }
  } else {
		
    /* ui_load_game_fail(app_data->app_ui_data); */
  }
  
  if (loadfile != NULL)
    g_free(loadfile);
	osso_rpc_free_val (&retval);
  set_all_sensitive();
  /*gs.startup_ui_state_change_cb(gs.ui->app, GAME_CONTINUE,
                                 GAME_CONTINUE, gs.ui);  */
  /*gs.startup_ui_state_change_cb(gs.ui->app, GAME_PAUSED,
                                 GAME_PAUSED, gs.ui);*/
  /*gs.startup_ui_state_change_cb(gs.ui->app, GAME_CONTINUE,
                                 GAME_CONTINUE, gs.ui);*/
  //gs.startup_ui_menu_action_cb(gs.ui, MA_GAME_PLAY, NULL);
}

/** Callback function when the volume unmounted */
static void plugin_volume_unmounted_cb(GnomeVFSVolumeMonitor *vfsvolumemonitor,
  GnomeVFSVolume *volume, gpointer user_data)
{
  gchar *dst_text = NULL;
  gchar *volume_text = NULL;
  GnomeVFSURI *dst_uri;
  gchar *uri_scheme;
  vfsvolumemonitor=NULL;
  user_data=NULL;


  dst_text = settings_get_string(SETTINGS_CHESS_SAVE_FILE);
  uri_scheme = gnome_vfs_get_uri_scheme(dst_text);
  if (!uri_scheme)
    dst_text = gnome_vfs_get_uri_from_local_path(dst_text);
  else
    g_free(uri_scheme);
  dst_uri = gnome_vfs_uri_new(dst_text);    
  g_free(dst_text);
  
  dst_text = gnome_vfs_uri_to_string(dst_uri, GNOME_VFS_URI_HIDE_NONE);

  if (dst_text != NULL) {
    /* the mount point of the changed volume */
    volume_text = gnome_vfs_volume_get_activation_uri(volume);
  }
  
  /* check if unmounting this volume has anything to do with us */
  if(volume_text && dst_text && (g_str_has_prefix(dst_text, volume_text))) {
    is_mmc = (mmc_uri 
      && g_str_has_prefix(volume_text, mmc_uri));
  }
  
  if (dst_text) g_free (dst_text);
  if (volume_text) g_free(volume_text);
}

/** D-bus async msg return handler */
/*static void plugin_dbus_message_async_result(const gchar *interface,
			  const gchar *method,
			  osso_rpc_t *retval, gpointer data)
{

  if (strcmp(method, GAME_SAVE_METHOD) == 0) {
    if (retval->type == DBUS_TYPE_BOOLEAN) {
      if (retval->value.b == TRUE)
	     chess_save_done = GAME_SAVE_OK;
      else
        chess_save_done = GAME_SAVE_ERR;
    } else {
        chess_save_done = GAME_SAVE_ERR;
    }
  }
}*/

/** Send an async message to the game */
/*static osso_return_t plugin_send_dbus_message_async(const gchar *service, 
				const gchar *object_path,
				const gchar *iface,
				const gchar *method,
				GArray *args, 
				gpointer data
				)
{
  StartupApp *app=NULL;
	app=gs.ui->app;
  return osso_rpc_async_run(app->osso,
  		      service,
		      object_path,
		      iface,
		      method,
		      plugin_dbus_message_async_result,
		      data,
		      0
//		      args
		      );
}*/


/** Send an async message to the game */
/*static gboolean plugin_send_message_async(const guint state)
{
  osso_return_t ret;
  GString *method;

method = g_string_new(GAME_SAVE_METHOD);
  g_assert(method);*/

  /* Send message */
  /*ret = plugin_send_dbus_message_async(
    "com.nokia.osso_chess",
    "/com/nokia/osso_chess",
    "com.nokia.osso_chess",
    method->str,
    NULL,
    NULL
  );

  if (method != NULL)
    g_string_free(method, TRUE);

  if (ret == OSSO_OK) 
    return TRUE;

  return FALSE;
}*/

/** Show nommc dialog */
static void plugin_ui_show_save_nommc_dialog(void)
{
  HildonNote *note = HILDON_NOTE(hildon_note_new_information(
        GTK_WINDOW(gs.ui->hildon_app),
        _("ckct_ni_cannot_save_mmc_cover_open")
  	));
  hildon_note_set_button_text(note, _("ckct_bd_cannot_save_mmc_cover_open_ok"));
  
  gtk_dialog_run(GTK_DIALOG(note));
  gtk_widget_destroy(GTK_WIDGET(note));

  hildon_appview_set_title(gs.ui->hildon_appview, "");
  plugin_settings_set_string(SETTINGS_CHESS_SAVE_FILE,"");
}

static void game_save(void)
{
  gdouble cpos = 0.1;
  gchar *tmp_text, *dst_text;
  GnomeVFSURI *tmp_uri, *dst_uri;
  GnomeVFSResult vfs_res;
  gchar *uri_scheme;
  
  /* Set where the MMC is mounted */
  const gchar *mmc_env = NULL;      
  mmc_env = g_getenv(MMC_MOUNTPOINT_ENV);
  if (mmc_env) {
    mmc_uri = g_strconcat (URI_FILE_PREFIX, mmc_env, NULL);
  }

  /* Monitor MMC state */
  guint volume_monitor_id = 0;
  is_mmc = FALSE;
  GnomeVFSVolumeMonitor *volume_monitor;
  volume_monitor = gnome_vfs_get_volume_monitor();
  volume_monitor_id = g_signal_connect(G_OBJECT(volume_monitor), "volume_unmounted", 
    G_CALLBACK(plugin_volume_unmounted_cb), NULL);
  dst_text = settings_get_string(SETTINGS_CHESS_SAVE_FILE);
  
  /* Checking if there is enough free space */
  struct statfs info;
  gchar *unuried_fname=NULL;
  unuried_fname=gnome_vfs_get_local_path_from_uri(dst_text);
  if (unuried_fname)
  {
  g_strrstr(unuried_fname,"/")[0]='\0';
  if (statfs(unuried_fname,&info)==0)
  {
    GnomeVFSFileInfo *file_info = gnome_vfs_file_info_new();
    GnomeVFSResult vfs_res = gnome_vfs_get_file_info(CHESS_PAUSE_SAVE_FILE_TMP,
      file_info,
      GNOME_VFS_FILE_INFO_FOLLOW_LINKS);
    if (vfs_res == GNOME_VFS_OK ||
        file_info->type == GNOME_VFS_FILE_TYPE_REGULAR)
    {
      //if (file_info->size>(info.f_bavail*info.f_bsize))
      if (0.02>=((double)info.f_bavail/(double)info.f_blocks))
      {//no more free space
        hildon_appview_set_title(gs.ui->hildon_appview, "");
        //plugin_show_infoprint(__("sfil_ni_not_enough_memory"));
        ui_show_nomem_dialog();
        plugin_ui_update_recent_items(TRUE);
        g_free(unuried_fname);
        return;
      }
    }
    else
    {//error while getting the fileinfo, or it is not a regular file
      plugin_show_infoprint(_("sfil_ni_unable_to_open_file_not_found"));
      plugin_ui_update_recent_items(TRUE);
      g_free(unuried_fname);
      return;
    }
  }
  else
  {// error gettint fs info
    plugin_show_infoprint(unuried_fname);
    plugin_ui_update_recent_items(TRUE);
    g_free(unuried_fname);
    return;
  }
  g_free(unuried_fname);
  }
  /* Display saving progress dialog */
  plugin_ui_show_saving_dialog();

  if (!ui_set_saving_dialog_progress( cpos)) return;
  chess_save_done=GAME_SAVE_SAVING;
  while(gtk_events_pending())
    gtk_main_iteration();

  
  /* Copy saved file from temp to destination */
  tmp_text = gnome_vfs_get_uri_from_local_path(CHESS_PAUSE_SAVE_FILE_TMP);
        
  
  uri_scheme = gnome_vfs_get_uri_scheme(dst_text);
  if (!uri_scheme)
    dst_text = gnome_vfs_get_uri_from_local_path(dst_text);
  else
    g_free(uri_scheme);

  tmp_uri = gnome_vfs_uri_new(tmp_text);
  dst_uri = gnome_vfs_uri_new(dst_text);

  vfs_res = gnome_vfs_xfer_uri(tmp_uri, dst_uri,
            GNOME_VFS_XFER_DEFAULT,
            GNOME_VFS_XFER_ERROR_MODE_ABORT,
            GNOME_VFS_XFER_OVERWRITE_MODE_REPLACE,
            NULL, NULL);

        
  if (is_mmc) { /* MMC cover open while saving to MMC */
   plugin_ui_hide_saving_dialog();
   while(gtk_events_pending())
     gtk_main_iteration();
        
   plugin_ui_show_save_nommc_dialog();
   chess_saved = FALSE;
   plugin_ui_hide_saving_dialog();
   while(gtk_events_pending())
      gtk_main_iteration();

    while(gtk_events_pending())
     gtk_main_iteration();
    usleep(10000);
    return;
  }

  /* Return if user press Cancel while saving */
  if (chess_save_done==GAME_SAVE_CANCEL) {
     gtk_signal_disconnect(G_OBJECT(volume_monitor), volume_monitor_id);
     return;
  }
  plugin_ui_hide_saving_dialog();
  while(gtk_events_pending())
      gtk_main_iteration();            
  
  /* Copy file error handling */
  switch (vfs_res) {
    case GNOME_VFS_OK:
#ifndef VERSION_3      
        plugin_show_infoprint(_("sfil_ib_saved"));
#else
        plugin_show_infoprint(__("sfil_ib_saved"));
#endif
	  		GnomeVFSResult result;
				result = gnome_vfs_open (&read_handle, dst_text, GNOME_VFS_OPEN_READ);
				g_assert(result == GNOME_VFS_OK);
        chess_saved = TRUE;
        add_recent_item(dst_text);
        plugin_ui_update_recent_items(FALSE);
        break;
      case GNOME_VFS_ERROR_ACCESS_DENIED:
      case GNOME_VFS_ERROR_READ_ONLY:
      case GNOME_VFS_ERROR_READ_ONLY_FILE_SYSTEM:
      case GNOME_VFS_ERROR_NO_MEMORY:
      case GNOME_VFS_ERROR_NO_SPACE:
      default:
        chess_saved = FALSE;
        plugin_ui_update_recent_items(TRUE);
        ui_show_nomem_dialog();
      }
	gnome_vfs_uri_unref(tmp_uri);
  gnome_vfs_uri_unref(dst_uri);
  g_free(tmp_text);
  g_free(dst_text);
  chess_save_done=GAME_SAVE_NONE;
  gtk_signal_disconnect(G_OBJECT(volume_monitor), volume_monitor_id);
}
  


static void plugin_ui_cancel_saving(void) {
//  AppData *app_data = get_app_data();
  
  /*app_data->app_ui_data->*/chess_save_done = GAME_SAVE_CANCEL;
  
  /* wait chess engine return result after saved */ 	  
  while (/*app_data->app_ui_data->*/chess_save_done == GAME_SAVE_CANCEL) {
    while(gtk_events_pending())
      gtk_main_iteration();
    usleep(10000);
  }
  plugin_ui_hide_saving_dialog(/*app_data->app_ui_data*/);  

  /* Was handled by game_save() */
  if (/*app_data->app_ui_data->*/chess_save_done==GAME_SAVE_NONE) {
    return;
  }
  
  //ui_repaint(app_data);
  
  /* Delete un-wanted file if file is new */
  if (is_new_file) {
    gchar *filename = settings_get_string(SETTINGS_CHESS_SAVE_FILE);

    hildon_appview_set_title(gs.ui->hildon_appview, "");
    plugin_settings_set_string(SETTINGS_CHESS_SAVE_FILE,"");   	          
  
    if (filename) {
      gnome_vfs_unlink(filename);
      plugin_ui_update_recent_items(TRUE);
      g_free(filename);
    }
    
  }
}



static void plugin_saving_dialog_cb(GtkDialog *dialog, gint button, gpointer user_data)
{
  dialog=NULL;
  user_data=NULL;
  if (button == GTK_RESPONSE_CANCEL) {
    if (chess_save_done == GAME_SAVE_SAVING) {
        plugin_ui_cancel_saving();
    }
  }
}



/* The following function is from INDT's games-startup code. The copyrights 
belongs to them. The INDT's games-startup was released under LGPL license. */

/** Show the saving dialog */
static void
plugin_ui_show_saving_dialog(void) 
{
  saving_progress_bar=gtk_progress_bar_new();
  /*chess_saving_dialog=hildon_file_handling_note_new_saving(
  				GTK_WINDOW(gs.ui->hildon_app));*/
  chess_saving_dialog=hildon_note_new_cancel_with_progress_bar(GTK_WINDOW(gs.ui->hildon_app),
  __("sfil_ib_saving"),GTK_PROGRESS_BAR(saving_progress_bar));
  g_signal_connect(G_OBJECT(chess_saving_dialog), "response",
    G_CALLBACK(plugin_saving_dialog_cb), NULL);

  g_signal_connect(G_OBJECT(chess_saving_dialog), "key_press_event",
    G_CALLBACK(plugin_saving_dialog_cb), NULL);
  gtk_widget_show_all(chess_saving_dialog);
}

/* The following function is from INDT's games-startup code. The copyrights 
belongs to them. The INDT's games-startup was released under LGPL license. */

/** Set the saving dialog progress */
gboolean ui_set_saving_dialog_progress(gfloat progress)
{
  if (chess_saving_dialog==NULL) {
    return FALSE;
  }
  gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(saving_progress_bar),progress);
  return TRUE;
}

/* The following function is from INDT's games-startup code. The copyrights 
belongs to them. The INDT's games-startup was released under LGPL license. */

/** Hide the saving dialog */
void plugin_ui_hide_saving_dialog(void)
{
  if (chess_saving_dialog==NULL) return;
  gtk_widget_hide_all(chess_saving_dialog);
  gtk_widget_destroy(GTK_WIDGET(chess_saving_dialog));
  chess_saving_dialog=NULL;
  saving_progress_bar=NULL;
}

/* The following function is from INDT's games-startup code. The copyrights 
belongs to them. The INDT's games-startup was released under LGPL license. */

/** Show no memory dialog */
void ui_show_nomem_dialog(void)
{
  HildonNote *note = HILDON_NOTE(hildon_note_new_information(
        GTK_WINDOW(gs.ui->hildon_app),
#ifndef VERSION_3
        _("sfil_ni_not_enough_memory")
#else
        __("sfil_ni_not_enough_memory")
#endif
  	));
#ifndef VERSION_3
  hildon_note_set_button_text(note, _("sfil_ni_not_enough_memory_ok"));
#else
  hildon_note_set_button_text(note, __("sfil_ni_not_enough_memory_ok"));
#endif
  
  gtk_dialog_run(GTK_DIALOG(note));
  gtk_widget_destroy(GTK_WIDGET(note));

  hildon_appview_set_title(gs.ui->hildon_appview, "");
  plugin_settings_set_string(SETTINGS_CHESS_SAVE_FILE,"");
}

/* The following function is from INDT's games-startup code. The copyrights 
belongs to them. The INDT's games-startup was released under LGPL license. */

/** Add a recent item to the menu*/
static void 
add_recent_item(gchar *file)
{
  gchar *filename = g_path_get_basename(file);
  gchar *temp = g_strrstr(filename, ".");
  if (temp)
    *temp = '\0';
  chess_title = gnome_vfs_unescape_string(filename,NULL);
  hildon_appview_set_title(gs.ui->hildon_appview, chess_title);
  g_free(chess_title);
  g_free(filename);
  GSList *item = g_slist_find_custom(chess_recent_items,
      file,
      (GCompareFunc)strcmp);
  if (item)
  {
    chess_recent_items =
      g_slist_remove_link(chess_recent_items, item);
    chess_recent_items =
      g_slist_prepend(chess_recent_items, item->data);
    g_slist_free_1(item);
  } else {
    chess_recent_items =
      g_slist_prepend(chess_recent_items, g_strdup(file));
    if (g_slist_length(chess_recent_items) >
        SETTINGS_CHESS_MAX_RECENT_ITEMS) {
      GSList *end = g_slist_last(chess_recent_items);
      g_free(end->data);
      chess_recent_items =
        g_slist_delete_link(chess_recent_items, end);
    }
  }
  plugin_settings_set_list_string(SETTINGS_CHESS_RECENT_ITEMS,
      chess_recent_items);
  plugin_ui_update_recent_items(FALSE);
}

/* The following function is from INDT's games-startup code. The copyrights 
belongs to them. The INDT's games-startup was released under LGPL license. */

/** Set a list string */
gboolean settings_set_list_string(const gchar *key, GSList *val)
{
  return gconf_client_set_list(gcc, key, GCONF_VALUE_STRING, val, NULL);
}

/* The following function is from INDT's games-startup code. The copyrights 
belongs to them. The INDT's games-startup was released under LGPL license. */


/* The following function is from INDT's games-startup code. The copyrights 
belongs to them. The INDT's games-startup was released under LGPL license. */

/* Get string type key value */
gchar* settings_get_string(const gchar *key)
{
  //osso_log(LOG_DEBUG,"Getting key %s value as string\n", key);
  return gconf_client_get_string(gcc, key, NULL);
}

/* The following function is from INDT's games-startup code. The copyrights 
belongs to them. The INDT's games-startup was released under LGPL license. */

/** Set the setings on the UI*/
static void plugin_set_settings(void)
{
  gboolean white = TRUE;
  gboolean human = FALSE;
  gboolean sound = TRUE;
  gboolean legal_moves = FALSE;
  gint player = -1;
  gint difficulty = -1;

  
    /* Sound */
    sound = gtk_toggle_button_get_active(
      GTK_TOGGLE_BUTTON( sound_check) );

    /* Human color */
    player = gtk_combo_box_get_active(
      GTK_COMBO_BOX( human_box ) );
    if (player == chess_human_white) white = TRUE;
    else white = FALSE;

    /* Opponent difficulty and type */
    difficulty = gtk_combo_box_get_active(
      GTK_COMBO_BOX( opponent_box ) );
    if (difficulty==5)
    {
      difficulty=4;
    }

    if (difficulty == chess_opponent_human) human = TRUE;
    else human = FALSE;

    /* Show legal moves */
    legal_moves = gtk_toggle_button_get_active(
      GTK_TOGGLE_BUTTON( show_legal_moves_check) );

    /* Set settings */
    plugin_settings_set_bool(SETTINGS_PLAYER_COLOR,
      white);
    plugin_settings_set_bool(SETTINGS_ENABLE_SOUND,
      sound);
    plugin_settings_set_bool(SETTINGS_SHOW_LEGAL_MOVES,
      legal_moves);
    plugin_settings_set_int(SETTINGS_LEVEL,
      difficulty);
    plugin_settings_set_bool(SETTINGS_OPPONENT_HUMAN,
      human);
    
}

/* Callback for the recent loading */
void game_load_recent_cb(GtkWidget *widget, int num)
{
  gchar *result = g_slist_nth_data(chess_recent_items, num - 1);
  widget=NULL;

  if (!result)
    return;
  if (!plugin_settings_set_string(SETTINGS_CHESS_SAVE_FILE, result))
    return;

  /* Check file existing */
  GnomeVFSURI *src_uri;
  gchar *src_text;
  gchar *uri_scheme;
  gboolean file_not_found = FALSE;

  src_text = g_strdup(result);
  uri_scheme = gnome_vfs_get_uri_scheme(src_text);
  if (!uri_scheme)
    src_text = gnome_vfs_get_uri_from_local_path(src_text);
  else
    g_free(uri_scheme);

  src_uri = gnome_vfs_uri_new(src_text);
  if (!gnome_vfs_uri_exists(src_uri)) {
    file_not_found = TRUE;
    plugin_ui_show_load_err(GAME_LOAD_FILE_NOT_FOUND, result);
  }
  g_free(src_text);
  gnome_vfs_uri_unref(src_uri);
  if (file_not_found)
  {
    plugin_remove_recent_item(result);
    plugin_ui_update_recent_items(FALSE);
    return;
  }
  /* End check file existing */  
  
  add_recent_item( result);

  plugin_set_settings();

  game_load(result);
  
}


/*
Notes: When renaming a file, GNOME_VFS_MONITOR_EVENT_DELETED is emitted for the old 
name and GNOME_VFS_MONITOR_EVENT_CREATED is emitted for the new name. Must have a 
monitor added for the location of the new file in order to receive that 
GNOME_VFS_MONITOR_EVENT_CREATED notification.

Only process CHESS_FILE_GLOB file extension
...
Not needed if we keep open a file descriptor
...
*/
/*void chess_saved_file_monitor( GnomeVFSMonitorHandle *handle,
			const gchar *monitor_uri,
			const gchar *info_uri,
			GnomeVFSMonitorEventType event_type,
			gpointer user_data)
{

  gchar *file_name;
  gchar *file_ext, *chess_file_glob;
  gchar *deleted_file_name;
  
  static gboolean rename_current_file = FALSE;
  monitor_uri=NULL;
  user_data=NULL;

  if (handle != gnomevfs_handle)
    return;
    
  if (info_uri == NULL)
    return;
  else
    file_name = gnome_vfs_format_uri_for_display(info_uri);
*/
  /* Get current event file extension */
/*  file_ext = g_strrstr(file_name, ".");
  if (!file_ext) {
    g_free(file_name);
    return;
  }
*/
  /* Get chess saved file extension */
/*  chess_file_glob = g_strrstr(CHESS_FILE_GLOB, ".");
  if (!chess_file_glob) {
    g_free(file_name);
    return;
  }
  
  if (strcmp(file_ext, chess_file_glob) != 0) {
    g_free(file_name);  
    return;
  }
  
  switch (event_type) {
      case GNOME_VFS_MONITOR_EVENT_DELETED:
        deleted_file_name = settings_get_string(SETTINGS_CHESS_SAVE_FILE);
        if (strcmp(deleted_file_name, file_name) == 0) {// current open file was renamed
          rename_current_file = TRUE;
        } else {
          rename_current_file = FALSE;
        }
        g_free(deleted_file_name);

        break;
      case GNOME_VFS_MONITOR_EVENT_CREATED:
        if (rename_current_file) {
          plugin_settings_set_string(SETTINGS_CHESS_SAVE_FILE, file_name);
          rename_current_file = FALSE;
        }
        break;
      case GNOME_VFS_MONITOR_EVENT_CHANGED:
      case GNOME_VFS_MONITOR_EVENT_STARTEXECUTING:
      case GNOME_VFS_MONITOR_EVENT_STOPEXECUTING:
      case GNOME_VFS_MONITOR_EVENT_METADATA_CHANGED:
      	break;
  }

  g_free(file_name);
    
  return;
}*/
