/*
 * This file is part of osso-backup
 *
 * Copyright (C) 2005 Nokia Corporation.
 *
 * Contact: Andrey Kochanov <andrey.kochanov@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <config.h>
#include <stdio.h>
#include <string.h>
#include <libgnomevfs/gnome-vfs.h>

#include "ob-backup-locations.h"
#include "ob-utils.h"

typedef struct {
	char     *path;
	gboolean  is_dir;
	gboolean  is_auto;
	gboolean  is_excluded;
} TestEntry;

static TestEntry entries[] = {
	/* path                     is_dir   is_auto    is_excluded */  
	{ "/MyDocs",                TRUE,    FALSE,   FALSE },
	{ "/MyDocs/",               TRUE,    FALSE,   FALSE },
	{ "/MyDocs/foobar",         TRUE,    TRUE,    FALSE },
	{ "/MyDocs/hey",            TRUE,    FALSE,   FALSE },
	{ "/MyDocs/hey/you",        TRUE,    TRUE,    FALSE },
	{ "/MyDocs/",               TRUE,    FALSE,   FALSE },
	{ "/etc/mail/mail.conf",    FALSE,   FALSE,   FALSE },
	{ "/etc/foo/bar",           TRUE,    FALSE,   FALSE },
	{ "/etc/foo/",              TRUE,    FALSE,   FALSE },
	{ "/etc/foo/baz",           TRUE,    FALSE,   FALSE },
	{ "/MyDocs/Media",          TRUE,    FALSE,   TRUE  },
	{ "/MyDocs/Media",          TRUE,    FALSE,   FALSE },
	{ "/MyDocs/Media/song.mp3", FALSE,   FALSE,   FALSE },
	{ "/MyDocs/Media/foo.mp3",  FALSE,   TRUE,    FALSE },
	{ "/stuff/things/file.doc", FALSE,   FALSE,   FALSE }
};

static char *
get_absolute_path (const char *relative_path)
{
	char *current_dir;
	char *absolute_path;
	
	current_dir = g_get_current_dir ();
	absolute_path = g_build_filename (current_dir, relative_path, NULL);
	g_free (current_dir);

	return absolute_path;
}

static gboolean
check_is_included (ObBackupLocations *locations,
		   const char        *path,
		   gboolean           answer)
{
	gboolean ret;

	ret = ob_backup_locations_is_included (locations, path);
	
	if (answer) {
		g_print (" %s is included: ", path);
	} else {
		g_print (" %s is not included: ", path);
	}

	if (ret == answer) {
		g_print ("OK\n");
	} else {
		g_print ("FAILED\n");
		g_error ("Test failed.\n");
	}
	
	return ret == answer;
}

static gboolean
check_is_auto_restore (ObBackupLocations *locations,
		       const char        *path,
		       gboolean           answer)
{
	gboolean ret;

	ret = ob_backup_locations_is_auto_restore (locations, path);
	
	if (answer) {
		g_print (" %s is auto-restore: ", path);
	} else {
		g_print (" %s is not auto-restore: ", path);
	}

	if (ret == answer) {
		g_print ("OK\n");
	} else {
		g_print ("FAILED\n");
		g_error ("Test failed.\n");
	}
	
	return ret == answer;
}

static void
test_config (void)
{
	ObBackupLocations *locations;
	int                i;
	GList             *list;

	_ob_backup_locations_init_empty ();
	locations = ob_backup_locations_get ();

	for (i = 0; i < G_N_ELEMENTS (entries); i++) {
		entries[i].path = ob_utils_parse_path (entries[i].path);
				
		_ob_backup_locations_add_location (locations,
						   entries[i].path,
						   OB_CATEGORY_OTHER,
						   entries[i].is_dir,
						   entries[i].is_auto,
						   entries[i].is_excluded);
	}
	
	_ob_backup_locations_remove_redundancies (locations);

	g_print ("Check inclusion:\n");
	
	check_is_included (locations, "/MyDocs/Whatever/Folder", TRUE);
	check_is_included (locations, "/Docs", FALSE);
	check_is_included (locations, "/Docs/file.txt", FALSE);
	check_is_included (locations, "/MyDocs/file.txt", TRUE);
	check_is_included (locations, "/MyDocs", TRUE);

	g_print ("\n");

	g_print ("Check auto-restore:\n");

	check_is_auto_restore (locations, "/MyDocs/Media/foo.mp3", TRUE);
        check_is_auto_restore (locations, "/MyDocs/file.txt", FALSE);
	check_is_auto_restore (locations, "/MyDocs", FALSE);
	check_is_auto_restore (locations, "/Docs", FALSE);
	check_is_auto_restore (locations, "/Docs/", FALSE);
	check_is_auto_restore (locations, "/Docs/test", FALSE);

	check_is_auto_restore (locations, "/MyDocs/foobar", TRUE);
	check_is_auto_restore (locations, "/MyDocs/foobar/baz", TRUE);
	check_is_auto_restore (locations, "/MyDocs/foobar/baz/test", TRUE);
	check_is_auto_restore (locations, "/MyDocs/foobar/baz/", TRUE);

	check_is_auto_restore (locations, "/MyDocs/hey/test.txt", FALSE);
	check_is_auto_restore (locations, "/MyDocs/hey/you/test.txt", TRUE);
	check_is_auto_restore (locations, "/MyDocs/hey/you", TRUE);
	check_is_auto_restore (locations, "/MyDocs/hey/you/you", TRUE);
	check_is_auto_restore (locations, "/MyDocs/hey/youyou", FALSE);
	check_is_auto_restore (locations, "/MyDocs/hey/yo", FALSE);

	g_print ("\n");

	g_print ("Check list of locations: ");
	list = ob_backup_locations_get_uris (locations, OB_CATEGORY_ALL, FALSE);

	if (g_list_length (list) != 4) {
		g_print ("Location list doesn't have the right number of entries.\n");
		g_error ("Test failed\n");
	}
 
	g_print ("OK\n");

	g_list_foreach (list, (GFunc) g_free, NULL);
	g_list_free (list);

	ob_backup_locations_shutdown ();
}

static void
test_parser (void)
{
	char              *absolute_path;
	ObBackupLocations *locations;
	
	g_print ("\n");

	absolute_path = get_absolute_path (CONFIG_TESTDIR);
	ob_backup_locations_init (absolute_path);
	locations = ob_backup_locations_get ();
	g_free (absolute_path);

	/* Read the configuration files in the test directory. */
	g_print ("Reading files in %s\n", CONFIG_TESTDIR);
	
	g_print ("Check inclusion:\n");
	
	check_is_included (locations, "/home/user/email/Addressbook.xml", TRUE);
	check_is_included (locations, "/home/user/email/File.xml", FALSE);
	check_is_included (locations, "/home/user/email/folders/mail.foo", TRUE);
	
	g_print ("\n");
	
	g_print ("Check auto-restore:\n");
	
	check_is_auto_restore (locations, "/home/user/email/folders/mail.bar", FALSE);
	check_is_auto_restore (locations, "/non-existent.data", FALSE);
	check_is_auto_restore (locations, "/browser/browser.ini", TRUE);
	
	ob_backup_locations_shutdown ();
}

int
main (int argc, char **argv)
{
	gnome_vfs_init ();
	
	test_config ();
	test_parser ();

	g_print ("\nAll tests passed.\n");
	
	gnome_vfs_shutdown ();
	
	return 0;
}

