/*
 * This file is part of osso-backup
 *
 * Copyright (C) 2005-2006 Nokia Corporation.
 *
 * Contact: Andrey Kochanov <andrey.kochanov@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <config.h>
#include <glib/gi18n.h>
#include <gtk/gtk.h>
#include <hildon-widgets/hildon-caption.h>
#include <hildon-widgets/gtk-infoprint.h>

#include "ob-types.h"
#include "ob-category.h"
#include "ob-utils.h"
#include "ob-selective-dialog.h"

struct _ObSelectiveDialog {
	GtkWidget     *dialog;

	GtkWidget     *selective_vbox;
	GtkWidget     *category_list;

	GtkWidget     *combo;
	GtkWidget     *protected_checkbox;
	GtkWidget     *protected_caption;

	ObUiOperation  operation;
	ObBackend     *backend;

	GtkWidget     *check_boxes[OB_NUM_CATEGORIES - 1];
	GtkWidget     *labels[OB_NUM_CATEGORIES - 1];

	GtkWidget     *ok_button;

	gboolean       was_protected;

	ObMemoryCardType card_type;
};

static void
selective_dialog_combo_box_changed_cb (GtkComboBox       *combo,
				       ObSelectiveDialog *dialog)
{
	int active;
	int i;

	active = gtk_combo_box_get_active (combo);

	if (active == 0) {
		gtk_widget_set_sensitive (dialog->selective_vbox, FALSE);
	} else {
		gtk_widget_set_sensitive (dialog->selective_vbox, TRUE);

		/* Move focus to the first item in the list. */
		for (i = 0; i < OB_NUM_CATEGORIES - 1; i++) {
			if (GTK_WIDGET_IS_SENSITIVE (dialog->check_boxes[i])) {
				gtk_widget_grab_focus (dialog->check_boxes[i]);
				break;
			}
		}
	}
}

static GtkWidget *
select_dialog_create_category_list (ObSelectiveDialog *dialog)
{
	gint       i, row, col;
	GtkWidget *table;

	table = gtk_table_new (2, OB_NUM_CATEGORIES - 1, FALSE);
	gtk_table_set_col_spacings (GTK_TABLE (table), OB_UI_MARGIN_DEFAULT);

	/* -1 since we don't want "other" in the list. */
	for (col = 0, row = 0, i = 0; i < OB_NUM_CATEGORIES - 1; i++, row++) {
		GtkWidget *hbox;
		gint       category;

		if (row >= 3) {
			row = 0;
			col++;
		}

		category = ob_category_from_index (i);

		dialog->check_boxes[i] = gtk_check_button_new ();

		g_object_set_data (G_OBJECT (dialog->check_boxes[i]), "ob-category",
				   GINT_TO_POINTER (category));

		gtk_toggle_button_set_active (
			GTK_TOGGLE_BUTTON (dialog->check_boxes[i]), TRUE);

		dialog->labels[i] = gtk_label_new (ob_category_to_string (category));
		gtk_misc_set_alignment (GTK_MISC (dialog->labels[i]), 0.0, 0.5);

		hbox = gtk_hbox_new (FALSE, OB_UI_MARGIN_DEFAULT);
		gtk_box_pack_start (GTK_BOX (hbox), dialog->check_boxes[i], FALSE, FALSE, 0);
		gtk_box_pack_start (GTK_BOX (hbox), dialog->labels[i], TRUE, TRUE, 0);

		gtk_table_attach (GTK_TABLE (table), hbox,
				  col, col + 1, row, row + 1,
				  GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
	}

	gtk_widget_show_all (table);
	
	return table;
}

static void
ok_button_insensitive_press_cb (GtkWidget *button, ObSelectiveDialog *dialog)
{
	gtk_infoprint (GTK_WINDOW (dialog->dialog), _("back_fi_nomemory_card"));
}

static void
memory_card_removed_cb (ObBackend         *backend,
			ObMemoryCard      *card,
			ObMemoryCardType   type,
			ObSelectiveDialog *dialog)
{
	if (type == dialog->card_type) {
		gtk_dialog_response (GTK_DIALOG (dialog->dialog),
				     OB_UI_RESPONSE_CARD_REMOVED);
	}
}

/* Note: This is a workaround for the fact that the combobox in the modified
 * GTK+ version has ellipsizing enabled, and there is now way to disable it. So
 * in order to fit all translations in the dialog, we need to explicitly set the
 * size of the combo box. */
static void
selective_dialog_combo_box_style_set_cb (GtkWidget         *widget,
					 GtkStyle          *previous_style,
					 ObSelectiveDialog *dialog_data)
{
	GtkTreeModel *model;
	PangoLayout  *layout;
	gint          width;
	gint          focus_width, focus_pad;
	gint          arrow_width;
	GtkTreeIter   iter;

	model = gtk_combo_box_get_model (GTK_COMBO_BOX (widget));
	if (!gtk_tree_model_get_iter_first (model, &iter)) {
		return;
	}
	
	gtk_widget_style_get (widget,
			      "arrow-width", &arrow_width,
			      "focus-line-width", &focus_width,
			      "focus-padding", &focus_pad,
			      NULL);
	
	layout = gtk_widget_create_pango_layout (widget, NULL);
	pango_layout_set_ellipsize (layout, PANGO_ELLIPSIZE_NONE);

	width = 0;

	do {
		gint   w;
		gchar *str;
		
		gtk_tree_model_get (model, &iter,
				    0, &str,
				    -1);

		pango_layout_set_text (layout, str, -1);
		g_free (str);

		pango_layout_get_pixel_size (layout, &w, NULL);
		width = MAX (width, w);
	} while (gtk_tree_model_iter_next (model, &iter));

	g_object_unref (layout);

	if (width > 0) {
		/* This is not the exact width, but there is no way to get the
		 * real width so an approximation will have to do.
		 */
		
		width += 2 * widget->style->xthickness;

		width += arrow_width;
		width += 2 * (focus_width + focus_pad);

		width += 2 * GTK_CONTAINER (widget)->border_width;

		/* This is to compensate for some of the width's we can't really
		 * get, like padding around a cell frame or something similar.
		 */
		width += 2 * 4;
		
		gtk_widget_set_size_request (widget, width, -1);
	}
}

ObSelectiveDialog *
ob_selective_dialog_new (GtkWindow        *parent,
			 ObUiOperation     operation,
			 ObBackend        *backend,
			 ObMemoryCardType  card_type)
{
	ObSelectiveDialog *dialog;
	GtkWidget         *caption;
	GtkWidget         *vbox;
	const char        *title;
	const char        *caption_text;

	dialog = g_new0 (ObSelectiveDialog, 1);

        if (operation == OB_UI_OPERATION_BACKUP) { 
                title = _("back_ti_backup_filter");
        } else {
                title = _("back_ti_restore_filter");
        }
	
	dialog->card_type = card_type;
	dialog->operation = operation;
	dialog->dialog = ob_ui_dialog_new (title, parent, NULL);
	dialog->backend = backend;
	dialog->was_protected = FALSE;
	
	dialog->ok_button = gtk_dialog_add_button (GTK_DIALOG (dialog->dialog),
						   _("back_bd_dia003_ok"),
						   GTK_RESPONSE_OK);
	gtk_widget_set_sensitive (
		dialog->ok_button, 
		ob_backend_get_memory_card (dialog->backend, card_type) != NULL);

	g_signal_connect (dialog->backend, "memory_card_removed",
			  G_CALLBACK (memory_card_removed_cb), dialog);
	
	g_signal_connect (dialog->ok_button, "insensitive_press", 
			  G_CALLBACK (ok_button_insensitive_press_cb), dialog);

	gtk_dialog_add_button (GTK_DIALOG (dialog->dialog),
			       _("back_bd_dia003_cancel"), GTK_RESPONSE_CANCEL);

	vbox = gtk_vbox_new (FALSE, OB_UI_WIDGET_SPACING);

	if (operation == OB_UI_OPERATION_BACKUP) {
		caption_text = _("back_fi_dia003_backup");
	} else {
		caption_text = _("back_fi_dia003_restore");
	}

	dialog->combo = gtk_combo_box_new_text ();

	gtk_combo_box_append_text (GTK_COMBO_BOX (dialog->combo), 
				   _("back_fi_dia003_all_data"));
	gtk_combo_box_append_text (GTK_COMBO_BOX (dialog->combo), 
				   _("back_fi_dia003_selected_data"));

	g_signal_connect (dialog->combo, "changed",
			  G_CALLBACK (selective_dialog_combo_box_changed_cb),
			  dialog);

	g_signal_connect (dialog->combo, "style_set",
			  G_CALLBACK (selective_dialog_combo_box_style_set_cb),
			  dialog);
	
	caption = hildon_caption_new (NULL, caption_text,
				      dialog->combo, NULL, HILDON_CAPTION_OPTIONAL);

	gtk_box_pack_start (GTK_BOX (vbox), caption, FALSE, FALSE, 0);

	gtk_box_pack_start (GTK_BOX (vbox), gtk_hseparator_new (), FALSE, FALSE, 0);

	dialog->selective_vbox = gtk_vbox_new (FALSE, OB_UI_WIDGET_SPACING);

	dialog->category_list = select_dialog_create_category_list (dialog);

	gtk_box_pack_start (GTK_BOX (dialog->selective_vbox), dialog->category_list, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), dialog->selective_vbox, TRUE, TRUE, 0);

	gtk_box_pack_start (GTK_BOX (vbox), gtk_hseparator_new (), FALSE, FALSE, 0);

	if (operation == OB_UI_OPERATION_BACKUP) {
		GtkWidget *alignment;

		alignment = gtk_alignment_new (0.5, 0.5, 0, 1);
		
		dialog->protected_checkbox = gtk_check_button_new ();

		caption = hildon_caption_new (NULL, _("back_fi_password_protect"),
					      dialog->protected_checkbox,
					      NULL, HILDON_CAPTION_OPTIONAL);

		dialog->protected_caption = caption;
		
		gtk_container_add (GTK_CONTAINER (alignment), caption);
		gtk_box_pack_start (GTK_BOX (vbox), alignment, FALSE, FALSE, 0);
	} else {
		dialog->protected_checkbox = NULL;
	}
	
	gtk_combo_box_set_active (GTK_COMBO_BOX (dialog->combo), 0);

        gtk_box_pack_start_defaults (GTK_BOX (GTK_DIALOG (dialog->dialog)->vbox), vbox);
        
	/* Setup help */
	ob_utils_setup_dialog_help (GTK_DIALOG (dialog->dialog), "features_backup_filter");

	gtk_widget_show_all (dialog->dialog);

	return dialog;
}

int
ob_selective_dialog_run (ObSelectiveDialog *dialog)
{
	int resp_id;

	while (1) {
		resp_id = gtk_dialog_run (GTK_DIALOG (dialog->dialog));

		if (resp_id != GTK_RESPONSE_OK) {
			return resp_id;
                }

		if (ob_selective_dialog_get_categories (dialog) == 0) {
			gtk_infoprint (GTK_WINDOW (dialog->dialog),
				       _("back_fi_select_oneoption"));
		} else {
			return resp_id;
                }
        }
}

void
ob_selective_dialog_set_available_categories (ObSelectiveDialog *dialog,
					      int                categories)
{
	int i;

	if (dialog->operation == OB_UI_OPERATION_BACKUP) {
		/* All categories are available. */
		return;
	}

	for (i = 0; i < OB_NUM_CATEGORIES - 1; i++) {
		int category;
		int sensitive;

		category = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (dialog->check_boxes[i]),
							       "ob-category"));
		sensitive = categories & category;

		gtk_widget_set_sensitive (dialog->check_boxes[i], sensitive);
		gtk_widget_set_sensitive (dialog->labels[i], sensitive);

		gtk_toggle_button_set_active (
			GTK_TOGGLE_BUTTON (dialog->check_boxes[i]), sensitive);
	}
}

gint
ob_selective_dialog_get_categories (ObSelectiveDialog *dialog)
{
	int i;
	int ret_val = 0;

	if (gtk_combo_box_get_active (GTK_COMBO_BOX (dialog->combo)) == 0) {
		return OB_CATEGORY_ALL;
	}

	for (i = 0; i < OB_NUM_CATEGORIES - 1; i++) {
		if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog->check_boxes[i])))
			ret_val |= GPOINTER_TO_INT (g_object_get_data (G_OBJECT (dialog->check_boxes[i]), 
								       "ob-category"));
	}

	return ret_val;
}

gboolean
ob_selective_dialog_get_protect (ObSelectiveDialog *dialog)
{
	gboolean protected;
	
	if (dialog->operation == OB_UI_OPERATION_RESTORE) {
		/* Not used for the restore dialog */
		return FALSE;
	}

	protected = gtk_toggle_button_get_active (
		GTK_TOGGLE_BUTTON (dialog->protected_checkbox));

	return protected;
}

void
ob_selective_dialog_destroy (ObSelectiveDialog *dialog)
{
	gtk_widget_destroy (dialog->dialog);

        g_signal_handlers_disconnect_by_func (dialog->backend,
                                              memory_card_removed_cb,
                                              dialog);

	g_free (dialog);
}

