/*
 * This file is part of osso-backup
 *
 * Copyright (C) 2005-2006 Nokia Corporation.
 *
 * Contact: Andrey Kochanov <andrey.kochanov@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <config.h>
#include <glib/gi18n.h>
#include <gtk/gtk.h>
#include <string.h>
#include <hildon-widgets/hildon-get-password-dialog.h>
#include <hildon-widgets/hildon-note.h>
#include <hildon-widgets/gtk-infoprint.h>
#include <hildon-widgets/hildon-file-system-info.h>
#include <hildon-widgets/hildon-appview.h>

#include "ob-backend.h"
#include "ob-backup-info.h"
#include "ob-ui.h"
#include "ob-utils.h"
#include "ob-icons.h"
#include "ob-selective-dialog.h"
#include "ob-restore-ui.h"
#include "ob-error-utils.h"
#include "ob-log.h"

enum {
	COL_NAME,
	COL_DATE,
        COL_SIZE,
        COL_INFO,
	NUM_COLS
};

static void
restore_ui_memory_card_removed_cb (ObBackend        *backend,
				   ObMemoryCard     *card,
				   ObMemoryCardType  type,
				   GtkDialog        *dialog)
{
	gpointer p;

	/*ob_log_info ("RestoreUI: %s memory card removed...", 
		     type == OB_MEMORY_CARD_INTERNAL ? "Internal" : "External");
	*/
	p = g_object_get_data (G_OBJECT (dialog), "memory_card_type");
	if (p) {
		ObMemoryCardType memory_card_type;

		memory_card_type = GPOINTER_TO_INT (p);

		/*ob_log_info ("RestoreUI: %s memory card is what we interested in", 
			     memory_card_type == OB_MEMORY_CARD_INTERNAL ? "Internal" : "External");
		*/
		/* If the memory card type we want to know about is
		 * not the same as the one we have the signal for, we
		 * silently ignore it. 
		 */
		if (memory_card_type != type) {
			/*ob_log_info ("RestoreUI: Not sending OB_UI_RESPONSE_CARD_REMOVED signal");*/
			return;
		}
	}

	gtk_dialog_response (dialog, OB_UI_RESPONSE_CARD_REMOVED);
}

static void
restore_ui_show_card_removed_dialog (GtkWindow *parent)
{
	GtkWidget   *dialog;
	gboolean     usb_cable;
	const gchar *str;
	
	usb_cable = ob_utils_get_is_usb_inserted ();
	if (usb_cable) {
		str = dgettext ("hildon-fm", _("sfil_ib_mmc_usb_connected"));
	} else {
		str = _("back_fi_restore_memory_card_removed");
	}
	
	dialog = hildon_note_new_information (parent, str);
	gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_widget_destroy (dialog);
}

/* Returns the basename of the URI, localized if there is a translation. The
 * basename is the last part of the URI. The reason for all this complexity is
 * that there is no good API for translating a while path, just the basename of
 * a URI.
 */
static char *
restore_ui_get_localized_basename (GnomeVFSURI *uri)
{
	const char           *path;
	char                 *uri_str;
	HildonFileSystemInfo *info;
	char                 *name;

	path = gnome_vfs_uri_get_path (uri);
	if (strcmp (path, "/") == 0) {
		return g_strdup ("/");
	}

	uri_str = g_strconcat ("file://", path, NULL);
	info = hildon_file_system_info_new (uri_str, NULL);
	g_free (uri_str);
	if (!info) {
		return g_path_get_basename (gnome_vfs_uri_get_path (uri));
	}

	name = g_strdup (hildon_file_system_info_get_display_name (info));

	hildon_file_system_info_free (info);

	return name;
}

static char *
restore_ui_get_localized_folder_name (GnomeVFSURI *uri)
{
	GnomeVFSURI *parent, *tmp_uri;
	char        *name;
	GSList      *list, *l;
	char        *result;

	/* Try to get localized versions of all the components of the URI. */
	parent = NULL;
	list = NULL;
	while (uri) {
		char *tmp;

		name = restore_ui_get_localized_basename (uri);

		list = g_slist_prepend (list, name);

		tmp = g_path_get_basename (gnome_vfs_uri_get_path (uri));
		if (strcmp (tmp, "MyDocs") == 0) {
			/* Stop at MyDocs if we get that. */
			break;
		}

		/* Continue with the parent. */
		tmp_uri = parent;
		parent = gnome_vfs_uri_get_parent (uri);

		if (tmp_uri) {
			gnome_vfs_uri_unref (tmp_uri);
		}

		uri = parent;
	}

	if (parent) {
		gnome_vfs_uri_unref (parent);
	}

	result = NULL;
	for (l = list; l; l = l->next) {
		char *tmp;

		name = l->data;

		if (result) {
			tmp = g_build_filename (result, name, NULL);
			g_free (result);
		} else {
			tmp = g_strdup (name);
		}
		result = tmp;

		g_free (name);
	}
	g_slist_free (list);

	return result;
}

static int
handle_conflict_file_file (GnomeVFSURI      *uri,
			   time_t            existing_time,
			   time_t            backup_time,
                           ObProgressDialog *progress_dialog)
{
	GtkWidget   *dialog;
	GtkWidget   *vbox;
	GtkWidget   *label;
	GtkWidget   *table;
	int          resp_id, retval;
	gchar       *timestamp;
	gchar       *str;
	gchar       *path;
	const gchar *i18n_hack;

	dialog = gtk_dialog_new_with_buttons (
		_("back_ti_restore_query"),
		GTK_WINDOW (progress_dialog->dialog),
		GTK_DIALOG_MODAL,
		_("back_bd_overwrite"), GTK_RESPONSE_YES,
		_("back_bd_overwrite_all"), OB_UI_RESPONSE_YES_TO_ALL,
		_("back_bd_skip"), GTK_RESPONSE_NO,
		_("back_bd_restore_dialog_cancel"), GTK_RESPONSE_CANCEL,
		NULL);

	gtk_dialog_set_has_separator (GTK_DIALOG (dialog), FALSE);

	gtk_container_set_border_width (GTK_CONTAINER (dialog),
                                        OB_UI_MARGIN_DEFAULT);
	
	vbox = gtk_vbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), vbox, FALSE, FALSE, 0);

	label = gtk_label_new (_("back_fi_dia009a_text"));
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
	gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, FALSE, 0);

	path = restore_ui_get_localized_folder_name (uri);
	label = gtk_label_new (path);
	gtk_misc_set_alignment (GTK_MISC (label), 0.5, 0.5);
	gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, FALSE, 0);
        g_free (path);

	label = gtk_label_new (_("back_fi_dia009b_text"));
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
	gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, FALSE, 0);

        table = gtk_table_new (2, 3, FALSE);
	gtk_table_set_col_spacing (GTK_TABLE (table), 0, OB_UI_MARGIN_DOUBLE);
	gtk_table_set_col_spacing (GTK_TABLE (table), 1, OB_UI_MARGIN_DOUBLE);
	gtk_box_pack_start (GTK_BOX (vbox), table, FALSE, FALSE, 0);

	/* FIXME: when the layout guide is fixed, we should just use
	 * ob_utils_get_timestamp() instead and put the one string in there
	 * directly.
	 */
	timestamp = ob_utils_get_timestamp_string (existing_time);
	i18n_hack = _("back_fi_newer_version_prompt");
	str = g_strdup_printf (i18n_hack, timestamp);

	label = gtk_label_new (str);

	g_free (str);
	g_free (timestamp);

	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), label,
			  0, 1, 0, 1,
			  GTK_EXPAND | GTK_FILL, 0, 0, 0);

	/* FIXME: when the layout guide is fixed, we should just use
	 * ob_utils_get_timestamp() instead and put the one string in there
	 * directly.
	 */
	timestamp = ob_utils_get_timestamp_string (backup_time);
	i18n_hack = _("back_fi_backed_up_version_prompt");
	str = g_strdup_printf (i18n_hack, timestamp);

	label = gtk_label_new (str);

	g_free (str);
	g_free (timestamp);

	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), label,
			  0, 1, 1, 2,
			  GTK_EXPAND | GTK_FILL, 0, 0, 0);

	gtk_widget_show_all (vbox);

	/* Setup help */
	ob_utils_setup_dialog_help (GTK_DIALOG (dialog), "features_backup_restoreconflict");

	progress_dialog->conflict_dialog = dialog;
	resp_id = gtk_dialog_run (GTK_DIALOG (dialog));
	progress_dialog->conflict_dialog = NULL;

	switch (resp_id) {
	case GTK_RESPONSE_YES:
		retval = OB_CONFLICT_RESPONSE_YES;
		break;
	case OB_UI_RESPONSE_YES_TO_ALL:
		retval = OB_CONFLICT_RESPONSE_YES_ALL;
		break;

	case GTK_RESPONSE_NO:
		retval = OB_CONFLICT_RESPONSE_NO;
		break;
	case GTK_RESPONSE_CANCEL:
	case GTK_RESPONSE_DELETE_EVENT:
		retval = OB_CONFLICT_RESPONSE_CANCEL;

		gtk_dialog_set_response_sensitive (
			GTK_DIALOG (progress_dialog->dialog),
			GTK_RESPONSE_CANCEL, FALSE);

		break;
	default:
		ob_log_error ("Got invalid response in conflict dialog.");
		retval = OB_CONFLICT_RESPONSE_CANCEL;
	}

	gtk_widget_destroy (dialog);

	return retval;
}

/* Dir in way of file. */
static int
handle_conflict_dir_file (GnomeVFSURI      *uri,
			  time_t            existing_time,
			  time_t            backup_time,
                          ObProgressDialog *progress_dialog)
{
	GtkWidget *dialog;
	int        response, retval;

	/* "Rename" renames the file, "No" does not overwrite the existing
	 * directory, "Cancel" cancels the restore process.
	 */

	dialog = ob_ui_message_dialog_new (GTK_WINDOW (progress_dialog->dialog),
                                           GTK_MESSAGE_QUESTION,
					   _("back_ti_restore_query"),
                                           _("back_fi_dia009_text_2"),
                                           _("back_bd_rename"), GTK_RESPONSE_YES, NULL, 
                                           _("back_bd_skip"), GTK_RESPONSE_NO, NULL,
                                           _("back_bd_restore_dialog_cancel"), GTK_RESPONSE_CANCEL, NULL,
                                           NULL);

	progress_dialog->conflict_dialog = dialog;
	response = gtk_dialog_run (GTK_DIALOG (dialog));
	progress_dialog->conflict_dialog = NULL;

	switch (response) {
	case GTK_RESPONSE_YES:
		retval = OB_CONFLICT_RESPONSE_YES;
		break;
	case GTK_RESPONSE_NO:
		retval = OB_CONFLICT_RESPONSE_NO;
		break;
	case GTK_RESPONSE_CANCEL:
	case GTK_RESPONSE_DELETE_EVENT:
		retval = OB_CONFLICT_RESPONSE_CANCEL;

		gtk_dialog_set_response_sensitive (
			GTK_DIALOG (progress_dialog->dialog),
			GTK_RESPONSE_CANCEL, FALSE);

		break;
	default:
		ob_log_error ("Got invalid response in conflict dialog.");
		retval = OB_CONFLICT_RESPONSE_CANCEL;
	}

	gtk_widget_destroy (dialog);

	return retval;
}

/* File in way of dir. */
static int
handle_conflict_file_dir (GnomeVFSURI      *uri,
			  time_t            existing_time,
			  time_t            backup_time,
                          ObProgressDialog *progress_dialog)
{
	GtkWidget *dialog;
	int        response, retval;

	/* "Rename" renames the folder, "No" does not restore the folder,
	 * "Cancel" cancels the restore process.
	 */

	dialog = ob_ui_message_dialog_new (GTK_WINDOW (progress_dialog->dialog),
                                           GTK_MESSAGE_QUESTION,
					   _("back_ti_restore_query"),
					   _("back_fi_dia009_text_3"),
                                           _("back_bd_rename"), GTK_RESPONSE_YES, NULL,
                                           _("back_bd_skip"), GTK_RESPONSE_NO, NULL,
                                           _("back_bd_restore_dialog_cancel"), GTK_RESPONSE_CANCEL, NULL,
                                           NULL);

	progress_dialog->conflict_dialog = dialog;
	response = gtk_dialog_run (GTK_DIALOG (dialog));
	progress_dialog->conflict_dialog = NULL;

	switch (response) {
	case GTK_RESPONSE_YES:
		retval = OB_CONFLICT_RESPONSE_YES;
		break;

	case GTK_RESPONSE_NO:
		retval = OB_CONFLICT_RESPONSE_NO;
		break;
	case GTK_RESPONSE_CANCEL:
	case GTK_RESPONSE_DELETE_EVENT:
		retval = OB_CONFLICT_RESPONSE_CANCEL;

		gtk_dialog_set_response_sensitive (
			GTK_DIALOG (progress_dialog->dialog),
			GTK_RESPONSE_CANCEL, FALSE);

		break;
	default:
		ob_log_error ("Got invalid response in conflict dialog.");
		retval = OB_CONFLICT_RESPONSE_CANCEL;
	}

        gtk_widget_destroy (dialog);

        return retval;
}

typedef struct {
        GtkWidget *dialog;
        GtkWidget *protected_label;
        GtkWidget *content_label;
} RestoreSelectionDialog;

static GtkResponseType
restore_ui_confirm_restore (GtkWindow    *parent, 
			    ObBackend    *backend, 
			    ObBackupInfo *info)
{
        GtkWidget        *dialog;
        gchar            *name;
        gchar            *msg;
        gint              resp_id;
	guint             id;
	ObMemoryCardType  memory_card_type;
	const gchar      *i18n_hack;

	/* Formatted logical string. */
        name = ob_backup_info_get_name (info);

	/* FIXME: The added argument here is to keep us from crashing when used
	 * with a faulty translation. We need to remove this when the
	 * translations are up to date.
	 */
	i18n_hack = _("back_fi_are_you_sure_you_would_like");
        msg = g_strdup_printf (i18n_hack, name);
        g_free (name);

        dialog = hildon_note_new_confirmation (parent, msg);
        hildon_note_set_button_text (HILDON_NOTE (dialog),
				     _("back_bd_007_restore_button"));

        g_free (msg);

	memory_card_type = ob_backup_info_get_card_type (info);
	g_object_set_data (G_OBJECT (dialog), "memory_card_type", 
			   GINT_TO_POINTER (memory_card_type));

	id = g_signal_connect (backend, "memory_card_removed",
			       G_CALLBACK (restore_ui_memory_card_removed_cb),
			       dialog);

        resp_id = gtk_dialog_run (GTK_DIALOG (dialog));

        g_signal_handler_disconnect (backend, id);

        gtk_widget_destroy (dialog);

        return resp_id;
}

static void
backend_progress_cb (ObBackend        *backend,
                     gint              files,
                     gint              total,
                     gint              complete,
		     ObCategory        category,
                     ObProgressDialog *dialog)
{
        ob_ui_progress_dialog_update (dialog, files, complete, category);
}

static void
backend_finalizing_cb (ObBackend        *backend,
		       ObProgressDialog *dialog)
{
        ob_ui_progress_dialog_set_string (dialog, _("back_va_finalizing_restore"));
}

static int
backend_conflict_cb (ObBackend        *backend,
                     ObConflictType    type,
                     GnomeVFSURI      *uri,
                     long              existing_time,
                     long              backup_time,
                     ObProgressDialog *dialog)
{
	switch (type) {
        case OB_CONFLICT_TYPE_FILE_FILE:
                return handle_conflict_file_file (uri,
						  existing_time,
						  backup_time,
						  dialog);
        case OB_CONFLICT_TYPE_DIR_FILE:
                return handle_conflict_dir_file (uri,
						 existing_time,
						 backup_time,
						 dialog);
        case OB_CONFLICT_TYPE_FILE_DIR:
                return handle_conflict_file_dir (uri,
						 existing_time,
						 backup_time,
						 dialog);
        default:
                g_assert_not_reached ();
                return OB_CONFLICT_RESPONSE_CANCEL;
        }
}

static void
backend_error_cb (ObBackend *backend, GError *error, ObProgressDialog *dialog)
{
        dialog->error = g_error_copy (error);
	gtk_dialog_response (GTK_DIALOG (dialog->dialog), OB_UI_RESPONSE_ERROR);
}

static void
backend_conflict_aborted_cb (ObBackend *backend, ObProgressDialog *dialog)
{
	if (dialog->conflict_dialog) {
		gtk_dialog_response (GTK_DIALOG (dialog->conflict_dialog),
				     GTK_RESPONSE_CANCEL);
	}
}

static void
backend_finished_cb (ObBackend *backend, ObProgressDialog *dialog)
{
        gtk_dialog_response (GTK_DIALOG (dialog->dialog),
                             OB_UI_RESPONSE_FINISHED);
}

static void
backend_cancelled_cb (ObBackend *backend, ObProgressDialog *dialog)
{
        /* Note: This is for cancellation due to reasons other than the user
         * pressing cancel, and when cancel is pressed in a conflict dialog.
         */
        gtk_dialog_response (GTK_DIALOG (dialog->dialog),
                             OB_UI_RESPONSE_CANCELLED);
}

static gboolean
restore_ui_run_restore (ObMainWindow *window,
			ObBackend    *backend,
                        GtkWindow    *parent,
                        ObBackupInfo *info,
                        int           categories)
{
        ObProgressDialog *progress_dialog;
        int               resp_id;
        GError           *error = NULL;
        size_t            size;
        GtkWidget        *info_dialog;
        char             *str;
        char             *size_str;
	char             *password;
	const char       *error_msg;
	int               processed_files, total_files;
	guint             id;
	GtkWindow        *main_window;
	const gchar      *i18n_hack;

        resp_id = restore_ui_confirm_restore (parent, backend, info);
        if (resp_id != GTK_RESPONSE_OK) {
		if (resp_id == OB_UI_RESPONSE_CARD_REMOVED) {
			restore_ui_show_card_removed_dialog (parent);
		}

                return FALSE;
        }

	password = NULL;
        if (ob_backup_info_get_is_protected (info)) {
                GtkWidget        *pw_dialog;
		ObMemoryCardType  memory_card_type;

	try_again:
                pw_dialog = hildon_get_password_dialog_new (parent, FALSE);
		ob_utils_setup_dialog_help (GTK_DIALOG (pw_dialog), "features_backup_getpassword");

		memory_card_type = ob_backup_info_get_card_type (info);
		g_object_set_data (G_OBJECT (pw_dialog), "memory_card_type", 
				   GINT_TO_POINTER (memory_card_type));

		id = g_signal_connect (backend, "memory_card_removed",
				       G_CALLBACK (restore_ui_memory_card_removed_cb),
				       pw_dialog);

                resp_id = gtk_dialog_run (GTK_DIALOG (pw_dialog));

		g_signal_handler_disconnect (backend, id);

                if (resp_id != GTK_RESPONSE_OK) {
                        gtk_widget_destroy (pw_dialog);

			if (resp_id == OB_UI_RESPONSE_CARD_REMOVED) {
				restore_ui_show_card_removed_dialog (parent);
			}

                        return FALSE;
                }

                password = g_strdup (hildon_get_password_dialog_get_password (
					     HILDON_GET_PASSWORD_DIALOG (pw_dialog)));

		/* Check if the password is correct before starting a restore
		 * operation. This makes error handling a lot easier.
		 */
		if (!ob_backup_info_check_password (info, password)) {
			gtk_infoprint (parent, _("back_ib_password_incorrect"));

			g_free (password);
			password = NULL;

			gtk_widget_destroy (pw_dialog);

			goto try_again;
		}

		gtk_widget_destroy (pw_dialog);
        }

	/* Close applications with the application killer. */
	ob_utils_kill_apps ();

	main_window = ob_main_window_get_window (window);
	if (!ob_main_window_is_fullscreen (window)) {
		gtk_window_fullscreen (main_window);
	}

	/* Make sure we are done before killing apps, also makes things look
	 * smoother.
	 */
	while (gtk_events_pending ()) {
		gtk_main_iteration ();
	}

        progress_dialog = ob_ui_progress_dialog_new (
		parent,
		OB_UI_OPERATION_RESTORE,
		ob_backup_info_get_num_files_for_categories (info, categories));

        gtk_widget_show_all (progress_dialog->dialog);

        g_signal_connect (backend, "finalizing",
                          G_CALLBACK (backend_finalizing_cb),
                          progress_dialog);
        g_signal_connect (backend, "progress",
                          G_CALLBACK (backend_progress_cb),
                          progress_dialog);
        g_signal_connect (backend, "conflict",
                          G_CALLBACK (backend_conflict_cb),
                          progress_dialog);
        g_signal_connect (backend, "conflict_aborted",
                          G_CALLBACK (backend_conflict_aborted_cb),
                          progress_dialog);
        g_signal_connect (backend, "error",
                          G_CALLBACK (backend_error_cb),
                          progress_dialog);
        g_signal_connect (backend, "finished",
                          G_CALLBACK (backend_finished_cb),
                          progress_dialog);
        g_signal_connect (backend, "cancelled",
                          G_CALLBACK (backend_cancelled_cb),
                          progress_dialog);

        if (!ob_backend_start_restore (backend, info, password,
                                       categories, &error)) {
		resp_id = OB_UI_RESPONSE_ERROR;
	} else {
		resp_id = gtk_dialog_run (GTK_DIALOG (progress_dialog->dialog));
		if (resp_id == OB_UI_RESPONSE_ERROR) {
			error = g_error_copy (progress_dialog->error);
		}
	}

	g_free (password);

        g_signal_handlers_disconnect_by_func (backend,
                                              backend_progress_cb,
                                              progress_dialog);
        g_signal_handlers_disconnect_by_func (backend,
                                              backend_conflict_cb,
                                              progress_dialog);
        g_signal_handlers_disconnect_by_func (backend,
                                              backend_error_cb,
                                              progress_dialog);
        g_signal_handlers_disconnect_by_func (backend,
                                              backend_finished_cb,
                                              progress_dialog);
        g_signal_handlers_disconnect_by_func (backend,
                                              backend_cancelled_cb,
                                              progress_dialog);
        g_signal_handlers_disconnect_by_func (backend,
                                              backend_conflict_aborted_cb,
                                              progress_dialog);

	/* Keep these before we destroy the dialog. */
	processed_files = MAX (0, progress_dialog->processed_files);
	total_files = progress_dialog->total_files;

	gtk_dialog_set_response_sensitive (GTK_DIALOG (progress_dialog->dialog),
					   GTK_RESPONSE_CANCEL, FALSE);

        switch (resp_id) {
        case OB_UI_RESPONSE_FINISHED:
		ob_ui_progress_dialog_destroy (progress_dialog);

		size = ob_backend_get_processed_size (backend);
		size_str = ob_utils_get_size (size);

		/* Formatted logical string. */
		i18n_hack = _("back_fi_not013_restore_successfully_completed");
                str = g_strdup_printf (i18n_hack, size_str);

                info_dialog = hildon_note_new_information (parent, str);
                g_free (str);
                g_free (size_str);

                gtk_dialog_run (GTK_DIALOG (info_dialog));
                gtk_widget_destroy (info_dialog);
                break;
        case OB_UI_RESPONSE_ERROR:
		error_msg = ob_error_utils_get_message (error);

		/* Log the raw message. */
		ob_log_error (error->message);

		ob_ui_progress_dialog_destroy (progress_dialog);

                info_dialog = hildon_note_new_information (parent,
                                                           error_msg);
                gtk_dialog_run (GTK_DIALOG (info_dialog));
                gtk_widget_destroy (info_dialog);
                break;

        case GTK_RESPONSE_DELETE_EVENT:
        case GTK_RESPONSE_CANCEL:
        case OB_UI_RESPONSE_CANCELLED:
		/* OB_UI_RESPONSE_CANCELLED happens when the user pressed cancel in a conflict
		 * dialog or when the operation is cancelled from outside.
		 */
                ob_backend_cancel (backend);

		/* When the backend responds, we can close the progress
		 * dialog. This way it doesn't look like the operation is
		 * cancelled before it is.
		 */
		ob_ui_progress_dialog_destroy (progress_dialog);

		/* Make sure that we didn't get an error after cancelling. */
		error = ob_backend_get_error (backend);
		if (error) {
			error_msg = ob_error_utils_get_message (error);

			/* Log the raw message. */
			ob_log_error (error->message);
			
			info_dialog = hildon_note_new_information (parent,
								   error_msg);
			gtk_dialog_run (GTK_DIALOG (info_dialog));
			gtk_widget_destroy (info_dialog);

			error = NULL;
			break;
		}

		/* Formatted logical string. */
		i18n_hack = _("back_fi_not029_msg");
                str = g_strdup_printf (i18n_hack, processed_files, total_files);
                info_dialog = hildon_note_new_information (parent, str);
                gtk_dialog_run (GTK_DIALOG (info_dialog));
                gtk_widget_destroy (info_dialog);
                g_free (str);
                break;
        }

	if (error) {
		g_error_free (error);
	}

	ob_utils_send_reboot_message ();
	gtk_main_quit ();

        if (resp_id == OB_UI_RESPONSE_FINISHED) {
                return TRUE;
        } else {
                return FALSE;
        }
}

void
ob_restore_ui_run (ObBackend    *backend,
		   ObBackupInfo *info, 
		   ObMainWindow *window)
{
	GtkWindow         *parent;
        ObSelectiveDialog *dialog;
        int                categories;
        int                resp_id;
	ObMemoryCardType   card_type;
	gboolean           ret;

	parent = ob_main_window_get_window (window);

	ob_backup_info_ref (info);

	card_type = ob_backup_info_get_card_type (info);

	dialog = ob_selective_dialog_new (parent,
                                          OB_UI_OPERATION_RESTORE,
					  backend, 
					  card_type);
        ob_selective_dialog_set_available_categories (
		dialog, ob_backup_info_get_categories (info));

        resp_id = ob_selective_dialog_run (dialog);

        if (resp_id != GTK_RESPONSE_OK) {
                ob_selective_dialog_destroy (dialog);
		ob_backup_info_unref (info);

		if (resp_id == OB_UI_RESPONSE_CARD_REMOVED) {
			restore_ui_show_card_removed_dialog (parent);
		}

                return;
        }

        categories = ob_selective_dialog_get_categories (dialog);
        ob_selective_dialog_destroy (dialog);

        ret = restore_ui_run_restore (window, backend, parent,
				      info, categories);

	/* If the restore was performed, we won't get here, since the app will
	 * be quitted.
	 */

	ob_backup_info_unref (info);
}

gboolean 
ob_restore_ui_startup (ObBackend *backend)
{
	GtkWidget   *dialog;
	const gchar *format;
	gchar       *str;
	gint         response;
	guint        num_backups = 0;
	
	num_backups += ob_backend_count_backups (backend, OB_MEMORY_CARD_INTERNAL);
 	num_backups += ob_backend_count_backups (backend, OB_MEMORY_CARD_EXTERNAL); 

	if (num_backups == 0) {
		return FALSE;
	}
	
	format = ngettext ("startup_ia_backup_found",
			   "startup_ia_backup_found_plural",
			   num_backups);
	
	str = g_strdup_printf (format);
	dialog = hildon_note_new_confirmation (NULL, str);
	g_free (str);
	
	hildon_note_set_button_text (HILDON_NOTE (dialog),
				     _("startup_bd_backup_found_ok"));

	response = gtk_dialog_run (GTK_DIALOG (dialog));

	gtk_widget_destroy (dialog);
	
	if (response != GTK_RESPONSE_OK) {
		ob_backend_add_restored_flag ();
		return FALSE;
	}

	return TRUE;
}
