/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/*
 * This file is part of osso-backup
 *
 * Copyright (C) 2005-2006 Nokia Corporation.
 *
 * Contact: Andrey Kochanov <andrey.kochanov@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <config.h>
#include <string.h>
#include <glib.h>
#include <libgnomevfs/gnome-vfs.h>

#include "ob-backup-info.h"
#include "ob-memory-card.h"
#include "ob-vfs-utils.h"

static void memory_card_clear_tmp_files (GnomeVFSURI *uri);


struct _ObMemoryCard {
	guint        ref_count;

	/* Points to the mountpoint. */
	GnomeVFSURI *uri;

	/* Cache of backups on this card. */
	GList       *backups;

	char        *name;
};

ObMemoryCard *
ob_memory_card_new (GnomeVFSURI *uri,
		    const char  *name)
{
	ObMemoryCard *card;

	g_return_val_if_fail (uri != NULL, NULL);
	g_return_val_if_fail (name != NULL, NULL);
	
	card = g_new0 (ObMemoryCard, 1);
	card->ref_count = 1;

	card->uri = gnome_vfs_uri_ref (uri);
	card->name = g_strdup (name);

	return card;
}

ObMemoryCard *
ob_memory_card_ref (ObMemoryCard *card)
{
	card->ref_count++;

	return card;
}

void
ob_memory_card_unref (ObMemoryCard *card)
{
	card->ref_count--;

	if (card->ref_count > 0) {
		return;
	}

	gnome_vfs_uri_unref (card->uri);
	g_free (card->name);

	ob_backup_info_list_free (card->backups);
	
	g_free (card);
}

const char *
ob_memory_card_get_name (ObMemoryCard *card)
{
	g_return_val_if_fail (card != NULL, NULL);

	return card->name;
}

GnomeVFSURI *
ob_memory_card_get_uri (ObMemoryCard *card)
{
	g_return_val_if_fail (card != NULL, NULL);

	return card->uri;
}

/* Returns a const list of ObBackupInfos. Don't free it. */
GList *
ob_memory_card_get_backups (ObMemoryCard *card)
{
	GnomeVFSURI *backups_uri;
	
	g_return_val_if_fail (card != NULL, NULL);

	if (card->backups) {
		return card->backups;
	}

	backups_uri = gnome_vfs_uri_append_path (card->uri, "backups");
	
	card->backups = ob_backup_info_list_get (backups_uri);

	gnome_vfs_uri_unref (backups_uri);

	return card->backups;
}

/* Invalidates the cache of backups on the card. */
void
ob_memory_card_clear_cache (ObMemoryCard *card)
{
	g_return_if_fail (card != NULL);

	if (card->backups) {
		ob_backup_info_list_free (card->backups);
		card->backups = NULL;
	}	
}

static void
memory_card_clear_tmp_files (GnomeVFSURI *uri)
{
	GnomeVFSResult  result;
	GList          *files = NULL, *l;

	result = gnome_vfs_directory_list_load (
		&files, 
		gnome_vfs_uri_get_path (uri), 
		GNOME_VFS_FILE_INFO_DEFAULT |
		GNOME_VFS_FILE_INFO_FORCE_FAST_MIME_TYPE);
	
	for (l = files; l; l = l->next) {
		GnomeVFSFileInfo *info;
		GnomeVFSURI      *child_uri;

		info = l->data;

		if (g_str_has_prefix (info->name, ".gsf")) {
			child_uri = gnome_vfs_uri_append_path (uri, info->name);
			gnome_vfs_unlink_from_uri (child_uri);
			gnome_vfs_uri_unref (child_uri);
		}
	}
	
	gnome_vfs_file_info_list_free (files);
}

void
ob_memory_card_clear_tmp_files (ObMemoryCard *card)
{
	GnomeVFSURI    *backups_uri;
	GnomeVFSResult  result;
	GList          *files = NULL, *l;

	g_return_if_fail (card != NULL);
	
	backups_uri = gnome_vfs_uri_append_path (card->uri, "backups");

	result = gnome_vfs_directory_list_load (
		&files, 
		gnome_vfs_uri_get_path (backups_uri), 
		GNOME_VFS_FILE_INFO_DEFAULT |
		GNOME_VFS_FILE_INFO_FORCE_FAST_MIME_TYPE);
	
	for (l = files; l; l = l->next) {
		GnomeVFSFileInfo *info;
		GnomeVFSURI      *uri;
		
		info = l->data;
		
		if (info->type == GNOME_VFS_FILE_TYPE_DIRECTORY &&
		    strcmp (info->name, ".") != 0 &&
		    strcmp (info->name, "..") != 0) {
			uri = gnome_vfs_uri_append_path (backups_uri,
							 info->name);
			memory_card_clear_tmp_files (uri);
			gnome_vfs_uri_unref (uri);
		}
	}

	gnome_vfs_file_info_list_free (files);
	gnome_vfs_uri_unref (backups_uri);
}

/* Gets the free space on the card. It must be inserted. */
GnomeVFSResult
ob_memory_card_get_free_space (ObMemoryCard     *card,
			       GnomeVFSFileSize *size)
{
	g_return_val_if_fail (card != NULL, GNOME_VFS_ERROR_BAD_PARAMETERS);
	g_return_val_if_fail (size != NULL, GNOME_VFS_ERROR_BAD_PARAMETERS);

	return gnome_vfs_get_volume_free_space (card->uri, size);
}
