/*
 * This file is part of osso-backup
 *
 * Copyright (C) 2005-2006 Nokia Corporation.
 *
 * Contact: Andrey Kochanov <andrey.kochanov@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <config.h>
#include <glib.h>
#include <glib/gi18n.h>
#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>
#include <libgnomevfs/gnome-vfs.h>
#include <hildon-widgets/gtk-infoprint.h>
#include <hildon-widgets/hildon-window.h>
#include <hildon-widgets/hildon-note.h>
#include <osso-helplib.h>

#include "ob-backend.h"
#include "ob-backup-info.h"
#include "ob-backup-ui.h"
#include "ob-log.h"
#include "ob-ui.h"
#include "ob-utils.h"
#include "ob-icons.h"
#include "ob-restore-ui.h"
#include "ob-main-window.h"

struct _ObMainWindow {
	GtkWidget    *main_window;
	
	GtkWidget    *dialog;

	ObBackend    *backend;
	ObMemoryCard *internal_mem_card;
	ObMemoryCard *external_mem_card;

	GtkWidget    *general_image;
	GtkWidget    *mem_card_image;

	GtkWidget    *general_backup_label;
	GtkWidget    *general_backup_date_label;

	GtkWidget    *card_backups_label;

	GtkWidget    *backups_treeview;
	GtkWidget    *no_backups_label;

	GtkWidget    *content_label;

	GtkWidget    *backup_button;
	GtkWidget    *restore_button;
	GtkWidget    *delete_button;

	GtkWidget    *ok_button;

	GtkWidget    *menu_create_new_backup;
	GtkWidget    *menu_restore;
	GtkWidget    *menu_delete;

	gboolean      first_category;
	
	guint         scroll_idle_id;

	/* When TRUE instead of using a HildonApp we use a GtkDialog
	 * because the spec says that the fresh flash restore app
	 * should use a dialog not a Hildon window.
	 */
	gboolean      startup_dialog;

	gboolean      fullscreen;
};

static void       main_window_backup_insensitive_press_cb (GtkWidget        *button,
							   ObMainWindow     *window);
static void       main_window_restore_insensitive_press_cb (GtkWidget        *button,
							   ObMainWindow     *window);
static void       main_window_delete_insensitive_press_cb (GtkWidget        *button,
							   ObMainWindow     *window);
static void       main_window_restore_button_clicked      (GtkButton        *button,
							   ObMainWindow     *window);
static void       main_window_backup_button_clicked       (GtkButton        *button,
							   ObMainWindow     *window);
static void       main_window_add_category_string         (ObMainWindow     *window,
							   GString          *c_string,
							   gint              category,
							   gint              categories);
static void       main_window_backup_changed_cb           (GtkTreeSelection *selection,
							   ObMainWindow     *window);
static GtkWidget *main_window_add_menu_item               (ObMainWindow     *window,
							   GtkWidget        *menu,
							   const gchar      *label,
							   GCallback         callback);
static void       main_window_backup                      (ObMainWindow     *window);
static void       main_window_restore_selected            (ObMainWindow     *window);
static void       main_window_delete_selected             (ObMainWindow     *window);
static void       main_window_menu_create_new_backup_cb   (GtkWidget        *menuitem,
							   ObMainWindow     *window);
static void       main_window_menu_restore_cb             (GtkWidget        *menuitem,
							   ObMainWindow     *window);
static void       main_window_menu_delete_cb              (GtkWidget        *menuitem,
							   ObMainWindow     *window);
static void       main_window_menu_help_cb                (GtkWidget        *menuitem,
							   ObMainWindow     *window);
static void       main_window_create_menu                 (ObMainWindow     *window);
static void       main_window_create_ui                   (ObMainWindow     *window);
static GList *    main_window_info_list_get               (ObMainWindow     *window);
static GtkWidget *main_window_create_last_info_box        (ObMainWindow     *window);
static GtkWidget *main_window_create_card_info_box        (ObMainWindow     *window);
static GtkListStore *main_window_create_backup_list_model (ObMainWindow     *window);
static GtkWidget *main_window_create_backup_list          (ObMainWindow     *window);
static GtkWidget *main_window_create_button_box           (ObMainWindow     *window);
static GtkWidget *main_window_create_content_info_box     (ObMainWindow     *window);
static void       main_window_update_menu                 (ObMainWindow     *window);
static void       main_window_update_last_info_box        (ObMainWindow     *window);
static void       main_window_update_card_info_box        (ObMainWindow     *window);
static void       main_window_update_backup_list          (ObMainWindow     *window);
static void       main_window_delete_button_clicked       (GtkButton        *button,
							   ObMainWindow     *window);
static void       main_window_response_cb                 (GtkWidget        *widget,
							   gint              response,
							   ObMainWindow     *window);
static gboolean   main_window_destroy_cb                  (GtkWidget        *widget,
							   ObMainWindow     *window);
static void       main_window_realize_cb                  (GtkWidget        *widget,
							   ObMainWindow     *window);
static gboolean   main_window_key_press_event_cb          (GtkWidget        *widget,
							   GdkEventKey      *event,
							   ObMainWindow     *window);
static void       main_window_memory_card_inserted        (ObBackend        *backend,
							   ObMemoryCard     *card,
							   ObMemoryCardType  type,
							   ObMainWindow     *window);
static void       main_window_memory_card_removed         (ObBackend        *backend,
							   ObMemoryCard     *card,
							   ObMemoryCardType  type,
							   ObMainWindow     *window);
static void       main_window_backups_changed             (ObBackend        *backend,
							   ObMemoryCard     *card,
							   ObMemoryCardType  type,
							   ObMainWindow     *window);

enum {
	COL_PROTECTED,
	COL_NAME,
	COL_DATE,
	COL_TIMESTAMP,
        COL_SIZE,
        COL_INFO,
	NUM_COLS
};

static void
main_window_backup_insensitive_press_cb (GtkWidget    *button, 
					 ObMainWindow *window)
{
	gboolean   memory_card_available;
	gboolean   usb_cable;
	GtkWidget *parent;

	usb_cable = ob_utils_get_is_usb_inserted ();
	memory_card_available = (window->internal_mem_card || window->external_mem_card);

	parent = gtk_widget_get_toplevel (GTK_WIDGET (window->main_window));
	
	if (usb_cable) {
		gtk_infoprint (
			GTK_WINDOW (parent),
			dgettext ("hildon-fm", _("sfil_ib_mmc_usb_connected")));
	}
	else if (!memory_card_available) {
		gtk_infoprint (GTK_WINDOW (parent), _("back_fi_nomemory_card"));
	}
}

static void
main_window_restore_insensitive_press_cb (GtkWidget    *button, 
					  ObMainWindow *window)
{
	GtkWidget *parent;
	gboolean   usb_cable;
	gboolean   memory_card_available;

	usb_cable = ob_utils_get_is_usb_inserted ();
	memory_card_available = (window->internal_mem_card || window->external_mem_card);

	parent = gtk_widget_get_toplevel (GTK_WIDGET (window->main_window));

	if (usb_cable) {
		gtk_infoprint (
			GTK_WINDOW (parent),
			dgettext ("hildon-fm", _("sfil_ib_mmc_usb_connected")));
	}
	else if (!memory_card_available) {
		gtk_infoprint (GTK_WINDOW (parent), _("back_fi_nomemory_card"));
	} else {
		gtk_infoprint (GTK_WINDOW (parent), _("back_fi_no_backups"));
	}		
}

static void
main_window_delete_insensitive_press_cb (GtkWidget    *button, 
					 ObMainWindow *window)
{
	GtkWidget *parent;
	gboolean   usb_cable;
	gboolean   memory_card_available;

	parent = gtk_widget_get_toplevel (GTK_WIDGET (window->main_window));

	usb_cable = ob_utils_get_is_usb_inserted ();
	memory_card_available = (window->internal_mem_card || window->external_mem_card);

	if (usb_cable) {
		gtk_infoprint (
			GTK_WINDOW (parent),
			dgettext ("hildon-fm", _("sfil_ib_mmc_usb_connected")));
	}
	else if (!memory_card_available) {
		gtk_infoprint (GTK_WINDOW (parent), _("back_fi_nomemory_card"));
	} else {
		gtk_infoprint (GTK_WINDOW (parent), _("back_fi_no_backups"));
	}
}

static void
main_window_add_category_string (ObMainWindow *window,
				 GString      *c_string,
				 gint          category,
				 gint          categories)
{
        gboolean     added;
        const gchar *str;

        added = FALSE;

        if (!(category & categories)) {
                return;
        }

        switch (category) {
        case OB_CATEGORY_EMAILS:
                str = _("back_fi_dia004_messages");
                break;
        case OB_CATEGORY_CONTACTS:
                str = _("back_fi_dia004_contacts");
                break;
        case OB_CATEGORY_DOCUMENTS:
                str = _("back_fi_dia004_documents");
                break;
        case OB_CATEGORY_MEDIA:
                str = _("back_fi_dia004_media");
                break;
        case OB_CATEGORY_BOOKMARKS:
                str = _("back_fi_dia004_bookmarks");
                break;
        case OB_CATEGORY_SETTINGS:
                str = _("back_fi_dia004_settings");
                break;
        case OB_CATEGORY_OTHER:
                str = _("back_fi_other");
                break;
        default:
                str = NULL;
                g_assert_not_reached ();
        }

        if (!window->first_category) {
		/* I18N: This is the delimiter between the backed up categories
		 * in the restore UI. E.g: "Contents: contacts, documents,
		 * media". Not translated for now.
		 */
                c_string = g_string_append (c_string, ", ");
        }

        window->first_category = FALSE;

        g_string_append (c_string, str);
}

static void
main_window_backup_changed_cb (GtkTreeSelection *selection,
			       ObMainWindow     *window)
{
        GtkTreeModel *model;
        GtkTreeIter   iter;
        ObBackupInfo *info;
        gboolean      protected;
        GString      *c_string;
        gint          cat = 0;

	main_window_update_menu (window);

        if (!gtk_tree_selection_get_selected (selection, &model, &iter)) {
		/* This usually occurs when backing up fails due to
		 * not enough memory on the mmc for example and we
		 * refresh the GtkListStore. 
		 */ 
		if (window->startup_dialog) {
			gtk_widget_set_sensitive (window->ok_button, FALSE);
		} else {
			gtk_widget_set_sensitive (window->restore_button, FALSE);
			gtk_widget_set_sensitive (window->delete_button, FALSE);
		}
		
		c_string = g_string_new (_("back_fi_dia001_content"));
		g_string_append_c (c_string, ' ');

		gtk_label_set_text (GTK_LABEL (window->content_label), c_string->str);
		
		g_string_free (c_string, TRUE);
		
                return;
        }

	if (window->startup_dialog) {
		gtk_widget_set_sensitive (window->ok_button, TRUE);
	} else {
		gtk_widget_set_sensitive (window->restore_button, TRUE);
		gtk_widget_set_sensitive (window->delete_button, TRUE);
	}

        gtk_tree_model_get (model, &iter, COL_INFO, &info, -1);
	
	if (info) {
		protected = ob_backup_info_get_is_protected (info);
		cat = ob_backup_info_get_categories (info);
	}

        c_string = g_string_new (_("back_fi_dia001_content"));
        g_string_append_c (c_string, ' ');

	window->first_category = TRUE;
	main_window_add_category_string (window,
					 c_string,
					 OB_CATEGORY_EMAILS,
					 cat);
	main_window_add_category_string (window,
					 c_string,
					 OB_CATEGORY_CONTACTS,
					 cat);
	main_window_add_category_string (window,
					 c_string,
					 OB_CATEGORY_DOCUMENTS,
					 cat);
	main_window_add_category_string (window,
					 c_string,
					 OB_CATEGORY_MEDIA,
					 cat);
	main_window_add_category_string (window,
					 c_string,
					 OB_CATEGORY_BOOKMARKS,
					 cat);
	main_window_add_category_string (window, 
					 c_string,
					 OB_CATEGORY_SETTINGS,
					 cat);

        gtk_label_set_text (GTK_LABEL (window->content_label), c_string->str);

        g_string_free (c_string, TRUE);

	if (info) {
		ob_backup_info_unref (info);
	}
}

static void
main_window_backup (ObMainWindow *window)
{
        if (ob_backup_ui_run (window->backend, window)) {
		main_window_update_menu (window);
		main_window_update_last_info_box (window);
		main_window_update_card_info_box (window);
		main_window_update_backup_list (window);
	}
}

static void
main_window_restore_selected (ObMainWindow *window)
{
	GtkTreeModel     *model;
	GtkTreeSelection *selection;
	GtkTreeIter       iter;
	ObBackupInfo     *info = NULL;
	
	model = gtk_tree_view_get_model (GTK_TREE_VIEW (window->backups_treeview));
	selection = gtk_tree_view_get_selection 
		(GTK_TREE_VIEW (window->backups_treeview));
	
	if (!gtk_tree_selection_get_selected (selection, NULL, &iter)) {
		ob_log_warning ("No backup selected to proceed with restore, "
				"this is not supposed to happen");
		return;
	}
	
	gtk_tree_model_get (model, &iter, COL_INFO, &info, -1);
	
	ob_restore_ui_run (window->backend, info, window);

	if (info) {
		ob_backup_info_unref (info);
	}
}

static void
main_window_delete_selected (ObMainWindow *window)
{
	GtkWidget        *dialog;
	gchar            *str;
        gint              response;
	GtkTreeModel     *model;
	GtkTreeSelection *selection;
	GtkTreeIter       iter;
	ObBackupInfo     *info = NULL;
	const gchar      *i18n_hack;

	model = gtk_tree_view_get_model 
		(GTK_TREE_VIEW (window->backups_treeview));
	selection = gtk_tree_view_get_selection 
		(GTK_TREE_VIEW (window->backups_treeview));
	
	if (!gtk_tree_selection_get_selected (selection, NULL, &iter)) {
		ob_log_warning ("No backup selected to proceed with delete, "
				"this is not supposed to happen");
		return;
	}
	
	gtk_tree_model_get (model, &iter, COL_INFO, &info, -1);

	i18n_hack = _("back_fi_delete_backup");
	str = g_strdup_printf (i18n_hack, ob_backup_info_get_name (info));
        dialog = hildon_note_new_confirmation (ob_main_window_get_window (window), str);
	g_free (str);

        hildon_note_set_button_text (HILDON_NOTE (dialog),
				     _("back_bd_delete_backup_ok"));

	response = gtk_dialog_run (GTK_DIALOG (dialog));

	gtk_widget_destroy (dialog);

	if (response == GTK_RESPONSE_OK) {
		ob_backend_remove_backup (window->backend, info);

		if (window->internal_mem_card) {
			ob_memory_card_clear_cache (window->internal_mem_card);
		} 

		if (window->external_mem_card) {
			ob_memory_card_clear_cache (window->external_mem_card);
		}
		
		main_window_update_menu (window);
		main_window_update_last_info_box (window);
		main_window_update_card_info_box (window);
		main_window_update_backup_list (window);
	}

	ob_backup_info_unref (info);
}

static void
main_window_menu_create_new_backup_cb (GtkWidget    *menuitem,
				       ObMainWindow *window)
{
	main_window_backup (window);
}

static void
main_window_menu_restore_cb (GtkWidget    *menuitem,
			     ObMainWindow *window)
{
	main_window_restore_selected (window);
}

static void
main_window_menu_delete_cb (GtkWidget    *menuitem,
			    ObMainWindow *window)
{
	main_window_delete_selected (window);
}

static void
main_window_menu_help_cb (GtkWidget    *menuitem,
			  ObMainWindow *window)
{
	ob_utils_show_help ("features_backup_mainview", FALSE);
}

static GtkWidget *
main_window_add_menu_item (ObMainWindow *window,
			   GtkWidget    *menu,
			   const gchar  *label,
			   GCallback     callback)
{
	GtkWidget *menu_item;

	menu_item = gtk_menu_item_new_with_label (label);

	gtk_widget_show (menu_item);
	gtk_menu_shell_append (GTK_MENU_SHELL (menu), menu_item);

	if (callback) {
		g_signal_connect (menu_item, "activate",
				  callback,
				  window);
	}

	return menu_item;
}

static void
main_window_create_menu (ObMainWindow *window)
{
	GtkMenu   *menu;
	GtkWidget *submenu;
	GtkWidget *item;

	menu = GTK_MENU (gtk_menu_new ());

	/* Backup submenu item. */
	item = main_window_add_menu_item (window,
					  GTK_WIDGET (menu),
					  _("back_me_backup"),
					  NULL);
	
	submenu = gtk_menu_new ();

	window->menu_create_new_backup = 
		main_window_add_menu_item (window,
					   GTK_WIDGET (submenu),
					   _("back_me_backup_create_new_backup"),
					   G_CALLBACK (main_window_menu_create_new_backup_cb));
	
	window->menu_restore = 
		main_window_add_menu_item (window,
					   GTK_WIDGET (submenu),
					   _("back_me_backup_restore"),
					   G_CALLBACK (main_window_menu_restore_cb));

	window->menu_delete = 
		main_window_add_menu_item (window,
					   GTK_WIDGET (submenu),
					   _("back_me_backup_delete"),
					   G_CALLBACK (main_window_menu_delete_cb));

	g_signal_connect (window->menu_create_new_backup, "insensitive_press", 
			  G_CALLBACK (main_window_backup_insensitive_press_cb),
			  window);
	
	g_signal_connect (window->menu_restore, "insensitive_press", 
			  G_CALLBACK (main_window_restore_insensitive_press_cb),
			  window);
	
	g_signal_connect (window->menu_delete, "insensitive_press", 
			  G_CALLBACK (main_window_delete_insensitive_press_cb),
			  window);

	gtk_menu_item_set_submenu (GTK_MENU_ITEM (item), submenu);
	
	/* Other items. */
	main_window_add_menu_item (window,
				   GTK_WIDGET (menu),
				   _("back_me_help"),
				   G_CALLBACK (main_window_menu_help_cb));

	main_window_add_menu_item (window,
				   GTK_WIDGET (menu),
				   _("back_me_close"),
				   G_CALLBACK (gtk_main_quit));

	hildon_window_set_menu (HILDON_WINDOW (window->main_window), 
				GTK_MENU (menu));
}

static void
main_window_create_ui (ObMainWindow *window)
{
	GtkWindow *parent;
	GtkWidget *vbox;
	GtkWidget *card_info_box;
	GtkWidget *content_info_box;
	GtkWidget *treeview;
	GtkWidget *separator_bottom;

	parent = ob_main_window_get_window (window);

	card_info_box = main_window_create_card_info_box (window);
	content_info_box = main_window_create_content_info_box (window);

	treeview = main_window_create_backup_list (window);

	separator_bottom = gtk_hseparator_new ();

	vbox = gtk_vbox_new (FALSE, OB_UI_MARGIN_DEFAULT);
	gtk_container_set_border_width (GTK_CONTAINER (vbox), OB_UI_MARGIN_DEFAULT);

	window->no_backups_label = ob_ui_label_new (_("back_fi_no_backups_present"));
	
	if (!window->startup_dialog) {
		GtkWidget *last_info_box;
		GtkWidget *button_box;
		GtkWidget *separator_top;

		last_info_box = main_window_create_last_info_box (window);
		button_box = main_window_create_button_box (window);

		separator_top = gtk_hseparator_new ();

		gtk_box_pack_start (GTK_BOX (vbox), last_info_box, FALSE, TRUE, 0);
		gtk_box_pack_start (GTK_BOX (vbox), separator_top, FALSE, FALSE, 0);
		gtk_box_pack_start (GTK_BOX (vbox), card_info_box, FALSE, TRUE, 0);

		/* The label that is shown when there are no backups. */ 
		gtk_box_pack_start (GTK_BOX (vbox), window->no_backups_label, FALSE, TRUE, 0);

		gtk_box_pack_start (GTK_BOX (vbox), treeview, TRUE, TRUE, 0);
		gtk_box_pack_start (GTK_BOX (vbox), separator_bottom, FALSE, FALSE, 0);
		gtk_box_pack_start (GTK_BOX (vbox), content_info_box, FALSE, TRUE, 0);
		gtk_box_pack_start (GTK_BOX (vbox), button_box, FALSE, TRUE, 0);

		gtk_container_add (GTK_CONTAINER (parent), vbox);
	} else {
		gtk_box_pack_start (GTK_BOX (vbox), card_info_box, FALSE, TRUE, 0);

		/* The label that is shown when there are no backups. */ 
		gtk_box_pack_start (GTK_BOX (vbox), window->no_backups_label, FALSE, TRUE, 0);

		gtk_box_pack_start (GTK_BOX (vbox), treeview, TRUE, TRUE, 0);
		gtk_box_pack_start (GTK_BOX (vbox), separator_bottom, FALSE, FALSE, 0);
		gtk_box_pack_start (GTK_BOX (vbox), content_info_box, FALSE, TRUE, 0);

		gtk_box_pack_start (GTK_BOX (GTK_DIALOG (parent)->vbox), 
				    vbox, TRUE, TRUE, 0);

		/* Setup help */
		ob_utils_setup_dialog_help (GTK_DIALOG (parent), "features_backup_selectbackup");
	}

	gtk_widget_show_all (vbox);

	main_window_update_menu (window);
	main_window_update_last_info_box (window);
	main_window_update_card_info_box (window);
	main_window_update_backup_list (window);
}

static GtkWidget *
main_window_create_last_info_box (ObMainWindow *window)
{
	GtkWidget *hbox;

	hbox = gtk_hbox_new (FALSE, OB_UI_MARGIN_DEFAULT);

	window->general_image = gtk_image_new_from_icon_name (OB_ICON_BACKUP, 
							      HILDON_ICON_SIZE_26);
	gtk_box_pack_start (GTK_BOX (hbox), window->general_image,
			    FALSE, FALSE, 0);

	window->general_backup_label = ob_ui_label_new (NULL);
	gtk_label_set_ellipsize (GTK_LABEL (window->general_backup_label),
				 PANGO_ELLIPSIZE_END);
	gtk_box_pack_start (GTK_BOX (hbox),
			    window->general_backup_label,
			    TRUE, TRUE, 0);

	window->general_backup_date_label = ob_ui_label_new (NULL);
	gtk_label_set_ellipsize (GTK_LABEL (window->general_backup_date_label),
				 PANGO_ELLIPSIZE_NONE);
	gtk_box_pack_start (GTK_BOX (hbox),
			    window->general_backup_date_label,
			    FALSE, FALSE, 0);

	return hbox;
}

static GtkWidget *
main_window_create_card_info_box (ObMainWindow *window)
{
	GtkWidget *hbox;

	hbox = gtk_hbox_new (FALSE, OB_UI_MARGIN_DEFAULT);

	window->mem_card_image = gtk_image_new_from_icon_name (OB_ICON_MMC, 
							       HILDON_ICON_SIZE_26);
	gtk_box_pack_start (GTK_BOX (hbox), window->mem_card_image, 
			    FALSE, FALSE, 0);

	window->card_backups_label = ob_ui_label_new (NULL);
	gtk_box_pack_start (GTK_BOX (hbox),
			    window->card_backups_label,
			    FALSE, TRUE, 0);

	return hbox;
}

static GtkWidget *
main_window_create_content_info_box (ObMainWindow *window)
{
	GtkWidget *hbox;

	hbox = gtk_hbox_new (FALSE, OB_UI_MARGIN_DEFAULT);
	
	window->content_label = ob_ui_label_new (_("back_fi_dia001_content"));
	gtk_label_set_line_wrap (GTK_LABEL (window->content_label), TRUE);

        gtk_box_pack_start (GTK_BOX (hbox), window->content_label,
			    FALSE, FALSE, 0);
	
	return hbox;
}

static GtkListStore *
main_window_create_backup_list_model (ObMainWindow *window)
{
        GtkListStore *model;
	
	model = gtk_list_store_new (NUM_COLS,
				    GDK_TYPE_PIXBUF,
				    G_TYPE_STRING,
                                    G_TYPE_STRING,
				    G_TYPE_INT,
                                    G_TYPE_STRING,
                                    OB_TYPE_BACKUP_INFO);
	
	gtk_tree_sortable_set_sort_column_id (GTK_TREE_SORTABLE (model), 
					      COL_TIMESTAMP, GTK_SORT_DESCENDING);
	
	return model;
}

static GtkWidget *
main_window_create_backup_list (ObMainWindow *window)
{
        GtkListStore      *model;
        GtkTreeViewColumn *col;
        GtkCellRenderer   *cell;
	GtkWidget         *sw;
        GtkTreeSelection  *selection;
	GdkPixbuf         *pixbuf;

	model = main_window_create_backup_list_model (window);
        window->backups_treeview = gtk_tree_view_new_with_model (GTK_TREE_MODEL (model));
	g_object_unref (model);
	
        gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (window->backups_treeview), FALSE);

        cell = gtk_cell_renderer_pixbuf_new ();
        col = gtk_tree_view_column_new_with_attributes ("", cell,
							"pixbuf", COL_PROTECTED,
							NULL);

	/* 26 is the width and height of the protected image we will use here */
	g_object_set (col, 
		      "sizing", GTK_TREE_VIEW_COLUMN_FIXED,
		      "min-width", OB_ICON_SIZE_26 + (OB_UI_MARGIN_DEFAULT * 2), 
		      NULL);
	g_object_set (cell, "xpad", OB_UI_MARGIN_DEFAULT, NULL);
	gtk_tree_view_append_column (GTK_TREE_VIEW (window->backups_treeview), col);

        cell = gtk_cell_renderer_pixbuf_new ();
        col = gtk_tree_view_column_new_with_attributes ("", cell,
							NULL);
	gtk_tree_view_append_column (GTK_TREE_VIEW (window->backups_treeview), col);

	pixbuf = gtk_icon_theme_load_icon (gtk_icon_theme_get_default (),
					   OB_ICON_BACKUP,
					   OB_ICON_SIZE_26,
					   GTK_ICON_LOOKUP_NO_SVG,
					   NULL);
	g_object_set (cell, "pixbuf", pixbuf, NULL);
	if (pixbuf) {
		g_object_unref (pixbuf);
	}

	/* Name. */
        cell = gtk_cell_renderer_text_new ();
	g_object_set (cell,
		      "xpad", OB_UI_MARGIN_DEFAULT,
		      "ellipsize", PANGO_ELLIPSIZE_END,
		      NULL);
        col = gtk_tree_view_column_new_with_attributes ("", cell,
                                                        "text", COL_NAME,
                                                        NULL);
	gtk_tree_view_column_set_expand (col, TRUE);
	gtk_tree_view_column_set_sizing (col, GTK_TREE_VIEW_COLUMN_AUTOSIZE);
        gtk_tree_view_append_column (GTK_TREE_VIEW (window->backups_treeview), col);

	/* Date. */
        cell = gtk_cell_renderer_text_new ();
	g_object_set (cell,
		      "xalign", 1.0,
		      "xpad", OB_UI_MARGIN_DEFAULT,
		      "ellipsize", PANGO_ELLIPSIZE_NONE,
		      NULL);
        col = gtk_tree_view_column_new_with_attributes ("", cell,
                                                        "text", COL_DATE,
                                                        NULL);
	gtk_tree_view_column_set_sort_column_id (col, COL_TIMESTAMP);
	gtk_tree_view_column_set_sizing (col, GTK_TREE_VIEW_COLUMN_AUTOSIZE);
        gtk_tree_view_append_column (GTK_TREE_VIEW (window->backups_treeview), col);

	/* Size. */
        cell = gtk_cell_renderer_text_new ();
	g_object_set (cell,
		      "xalign", 1.0,
		      "xpad", OB_UI_MARGIN_DEFAULT,
		      "ellipsize", PANGO_ELLIPSIZE_NONE,
		      NULL);
        col = gtk_tree_view_column_new_with_attributes ("", cell,
                                                        "text", COL_SIZE,
                                                        NULL);
	gtk_tree_view_column_set_sizing (col, GTK_TREE_VIEW_COLUMN_AUTOSIZE);
        gtk_tree_view_append_column (GTK_TREE_VIEW (window->backups_treeview), col);

	sw = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
					GTK_POLICY_NEVER,
					GTK_POLICY_AUTOMATIC);

	gtk_container_set_border_width (GTK_CONTAINER (sw), OB_UI_MARGIN_DEFAULT);
	gtk_container_add (GTK_CONTAINER (sw), window->backups_treeview);

	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (window->backups_treeview));
        gtk_tree_selection_set_mode (selection, GTK_SELECTION_SINGLE);

        g_signal_connect (selection, "changed",
                          G_CALLBACK (main_window_backup_changed_cb),
                          window);

	return sw;
}

static GtkWidget *
main_window_create_button_box (ObMainWindow *window)
{
	/*GtkWidget *alignment;*/
	GtkWidget *hbox;

	window->backup_button  = gtk_button_new_with_label (_("back_bd_backup"));
	window->restore_button = gtk_button_new_with_label (_("back_bd_restore"));
	window->delete_button   = gtk_button_new_with_label (_("back_bd_delete"));

	/* Yes, looks odd but this is what the layout guide says. */
	gtk_widget_set_size_request (window->backup_button, -1, 47);
	gtk_widget_set_size_request (window->restore_button, -1, 47);
	gtk_widget_set_size_request (window->delete_button, -1, 47);
	
	if (window->startup_dialog) {
		gtk_widget_set_sensitive (window->ok_button, FALSE);
	} else {
		gtk_widget_set_sensitive (window->restore_button, FALSE);
		gtk_widget_set_sensitive (window->delete_button, FALSE);
	}

	/*alignment = gtk_alignment_new (0.5, 0.5, 0, 1);*/

	hbox = gtk_hbox_new (TRUE, OB_UI_MARGIN_DEFAULT);
	/*gtk_container_add (GTK_CONTAINER (alignment), hbox);*/

	gtk_box_pack_start (GTK_BOX (hbox), window->backup_button, TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), window->restore_button, TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), window->delete_button, TRUE, TRUE, 0);

	g_signal_connect (window->backup_button, "clicked",
			  G_CALLBACK (main_window_backup_button_clicked),
			  window);
	g_signal_connect (window->restore_button, "clicked",
			  G_CALLBACK (main_window_restore_button_clicked),
			  window);
	g_signal_connect (window->delete_button, "clicked",
			  G_CALLBACK (main_window_delete_button_clicked),
			  window);

	g_signal_connect (window->backup_button, "insensitive_press", 
			  G_CALLBACK (main_window_backup_insensitive_press_cb), window);

	g_signal_connect (window->restore_button, "insensitive_press", 
			  G_CALLBACK (main_window_restore_insensitive_press_cb), window);

	g_signal_connect (window->delete_button, "insensitive_press", 
			  G_CALLBACK (main_window_delete_insensitive_press_cb), window);

	return hbox; /*alignment;*/
}

static GList *
main_window_info_list_get (ObMainWindow *window)
{
	GList *internal_list = NULL;
	GList *external_list = NULL;

	if (window->internal_mem_card) {
		internal_list = ob_memory_card_get_backups (window->internal_mem_card);
	}

	if (window->external_mem_card) {
		external_list = ob_memory_card_get_backups (window->external_mem_card);
	}

	if (internal_list && !external_list) {
		return g_list_copy (internal_list);
	}
	else if (!internal_list && external_list) {
		return g_list_copy (external_list);
	}
	else if (!internal_list && !external_list) {
		return NULL;
	}

	return g_list_concat (g_list_copy (internal_list), g_list_copy (external_list));
}

static void
main_window_update_menu (ObMainWindow *window)
{
	GtkTreeSelection *selection;
	gboolean          memory_card_available;
	gboolean          is_selection;
	gint              num_memory_cards;

	if (window->startup_dialog) {
		/* We have no menu for the startup dialog */
		return;
	}

	num_memory_cards = ob_backend_count_memory_cards (window->backend);
	memory_card_available = (window->internal_mem_card || window->external_mem_card);

	if (!memory_card_available) {
		gtk_widget_set_sensitive (window->menu_create_new_backup, FALSE);
		gtk_widget_set_sensitive (window->menu_restore, FALSE);
		gtk_widget_set_sensitive (window->menu_delete, FALSE);

		return;
	}

        selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (window->backups_treeview));
        is_selection = gtk_tree_selection_get_selected (selection, NULL, NULL);
	
	gtk_widget_set_sensitive (window->menu_create_new_backup, TRUE);

	gtk_widget_set_sensitive (window->menu_restore, is_selection);
	gtk_widget_set_sensitive (window->menu_delete, is_selection);
}

static void
main_window_update_last_info_box (ObMainWindow *window)
{
	gboolean     has_last_backup;
	gchar       *mem_card_name;
	gchar       *backup_name;
	time_t       timestamp;
	const gchar *i18n_hack;

	if (window->startup_dialog) {
		return;
	}

	has_last_backup = ob_utils_get_last_backup (&backup_name,
						    &mem_card_name,
						    &timestamp);
	
	if (has_last_backup) {
		char *date_str;
		char *str;

		i18n_hack = _("back_fi_dia001_last_backup_made");
		str = g_strdup_printf (i18n_hack, backup_name);

		gtk_label_set_text (GTK_LABEL (window->general_backup_label), str);
		g_free (str);

		date_str = ob_utils_get_date_string (timestamp);
		gtk_label_set_text (GTK_LABEL (window->general_backup_date_label), date_str);
                g_free (date_str);
		gtk_widget_show (window->general_backup_date_label);
	} else {
		gtk_label_set_text (GTK_LABEL (window->general_backup_label), 
				    _("back_fi_dia001_no_backups_made"));
		gtk_widget_hide (window->general_backup_date_label);
	} 
	
	if (has_last_backup) {
		g_free (mem_card_name);
		g_free (backup_name);
	}
}

static void
main_window_update_card_info_box (ObMainWindow *window)
{
	gchar    *str;
	GList    *backups;
	gboolean  memory_card_available;
	gint      num_backups;
	gint      num_memory_cards;

	num_memory_cards = ob_backend_count_memory_cards (window->backend);
	memory_card_available = (window->internal_mem_card || window->external_mem_card);

	if (!window->startup_dialog) {
		gtk_widget_set_sensitive (window->backup_button, 
					  memory_card_available);
	}
	
	if (!memory_card_available) {
		gtk_label_set_text (GTK_LABEL (window->card_backups_label),
				    _("back_fi_dia001_no_memory_card_present"));
		
		if (window->startup_dialog) {
			gtk_widget_set_sensitive (window->ok_button, FALSE);
		} else {
			gtk_widget_set_sensitive (window->restore_button, FALSE);
		}

		return;
	}

	/* Clear table */
	backups = main_window_info_list_get (window);
	num_backups = g_list_length (backups);
	g_list_free (backups);

	if (num_memory_cards > 0) {
		const gchar *tmp;
		
		/* Work around warning caused by this logical id mess. */
		tmp = ngettext ("back_fi_dia001_this_backup_was_taken_on",
				"back_fi_dia001_this_backup_was_taken_on_plural",
				num_memory_cards);
		
		/* FIXME: Remove num_backups here when translations are
		 * updated, there is no %d in the string anymore.
		 */
		str = g_strdup_printf (tmp, num_backups);
	} else {
		str = g_strdup (_("back_fi_dia001_no_memory_card_present"));
	}

	gtk_label_set_text (GTK_LABEL (window->card_backups_label), str);
	g_free (str);
}

static void
main_window_update_backup_list (ObMainWindow *window)
{
	GtkListStore     *store;
	GtkTreeSelection *selection; 
	GtkTreePath      *path; 
	GList            *backups;
	GList            *l;
	GdkPixbuf        *pixbuf;
	GdkPixbuf        *pixbuf_protected = NULL;
	GtkAdjustment    *vadj;
	gint              num_backups, num_memory_cards;

	backups = main_window_info_list_get (window);

	num_memory_cards = ob_backend_count_memory_cards (window->backend);
	if (num_memory_cards > 0) {
		num_backups = g_list_length (backups);
		if (num_backups == 0) {
			gtk_widget_show (window->no_backups_label);
		} else {
			gtk_widget_hide (window->no_backups_label);
		}		
	} else {
		gtk_widget_hide (window->no_backups_label);
	}		

	store = main_window_create_backup_list_model (window);
	
        for (l = backups; l; l = l->next) {
                ObBackupInfo     *info;
		GnomeVFSFileSize  size;
		char             *name;
                char             *date_str;
                char             *size_str;
		time_t            t;
                GtkTreeIter       iter;

                info = OB_BACKUP_INFO (l->data);

                name = ob_backup_info_get_name (info);

		t = ob_backup_info_get_timestamp (info);
                date_str = ob_utils_get_date_string (t);

		size = ob_backup_info_get_size (info);

		/* I18N: formatting, have no logical id for this one. */
		size_str = ob_utils_get_size (size);

		if (ob_backup_info_get_is_protected (info)) {
			if (!pixbuf_protected) {
				pixbuf_protected = gtk_icon_theme_load_icon 
					(gtk_icon_theme_get_default (),
					 OB_ICON_PROTECTED,
					 OB_ICON_SIZE_26,
					 GTK_ICON_LOOKUP_NO_SVG,
					 NULL);
			}

			pixbuf = pixbuf_protected;
		} else {
			pixbuf = NULL;
		}

		/* We use this instead of append + set because Maemo
		 * GTK+ use BROWSE selection behaviour (where there is
		 * always something selected) and it means that we are
		 * called back with no data set on append otherwise).
		 */
		gtk_list_store_insert_with_values (store, &iter, 0,
						   COL_PROTECTED, pixbuf,
						   COL_NAME, name,
						   COL_DATE, date_str,
						   COL_TIMESTAMP, t, 
						   COL_SIZE, size_str,
						   COL_INFO, info,
						   -1);

		g_free (name);
                g_free (date_str);
                g_free (size_str);
        }

	g_list_free (backups);

	if (pixbuf_protected) {
		g_object_unref (pixbuf_protected);
	}

	/* Use the new model. */
	gtk_tree_view_set_model (GTK_TREE_VIEW (window->backups_treeview), GTK_TREE_MODEL (store));
	g_object_unref (store);

        selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (window->backups_treeview));

	path = gtk_tree_path_new_first ();
	gtk_tree_selection_select_path (selection, path);
	gtk_tree_path_free (path);

	gtk_widget_grab_focus (window->backups_treeview);

	/* Make sure to scroll to the top to show the selected item. */
	vadj = gtk_tree_view_get_vadjustment (GTK_TREE_VIEW (window->backups_treeview));
	gtk_adjustment_set_value (vadj, 0);
}

static void
main_window_backup_button_clicked (GtkButton *button, ObMainWindow *window)
{
	main_window_backup (window);
}

static void
main_window_restore_button_clicked (GtkButton *button, ObMainWindow *window)
{
	main_window_restore_selected (window);
}

static void
main_window_delete_button_clicked (GtkButton *button, ObMainWindow *window)
{
	main_window_delete_selected (window);
}

static void
main_window_response_cb (GtkWidget    *widget,
			 gint          response,
			 ObMainWindow *window)
{
        if (response == GTK_RESPONSE_OK) {
		main_window_restore_selected (window);
		return;
        }

	gtk_widget_destroy (widget);
}

static gboolean
main_window_destroy_cb (GtkWidget    *widget,
			ObMainWindow *window)
{
	if (window->internal_mem_card) {
		ob_memory_card_unref (window->internal_mem_card);
	}

	if (window->external_mem_card) {
		ob_memory_card_unref (window->external_mem_card);
	}

	if (window->backend) {
		g_object_unref (window->backend);
	}

	if (window->scroll_idle_id) {
		g_source_remove (window->scroll_idle_id);
	}
	
	g_free (window);

	gtk_main_quit ();

	return FALSE;
}

static void
main_window_realize_cb (GtkWidget    *widget,
			ObMainWindow *window)
{
	GdkWindow *root;

	/* This is only used for dialog mode to make the dialog system modal. */	
	root = gdk_screen_get_root_window (gtk_widget_get_screen (window->dialog));
	
	gdk_window_set_transient_for (window->dialog->window, root);
	gdk_window_set_modal_hint (window->dialog->window, TRUE);
}

static gboolean
main_window_key_press_event_cb (GtkWidget    *widget,
				GdkEventKey  *event,
				ObMainWindow *window)
{
	/* Only close on Escape when we're in dialog mode. */
	if (window->startup_dialog && event->keyval == GDK_Escape) {
		gtk_main_quit ();
		return TRUE;
	}

	if (event->keyval == HILDON_HARDKEY_FULLSCREEN &&
	    window->main_window) {
		if (window->fullscreen) {
			gtk_window_unfullscreen (
				GTK_WINDOW (window->main_window));
			window->fullscreen = FALSE;
		} else {
			gtk_window_fullscreen (GTK_WINDOW (window->main_window));
			window->fullscreen = TRUE;
		}
		return TRUE;
	}
	
	return FALSE;
}

static void
main_window_memory_card_inserted (ObBackend        *backend,
				  ObMemoryCard     *card,
				  ObMemoryCardType  type,
				  ObMainWindow     *window)
{
	if (type == OB_MEMORY_CARD_INTERNAL) {
		if (window->internal_mem_card) {
			ob_memory_card_unref (window->internal_mem_card);
		}

		window->internal_mem_card = ob_memory_card_ref (card);
	} else {
		if (window->external_mem_card) {
			ob_memory_card_unref (window->external_mem_card);
		}

		window->external_mem_card = ob_memory_card_ref (card);
	}
	
	main_window_update_menu (window);
	main_window_update_last_info_box (window);
	main_window_update_card_info_box (window);
	main_window_update_backup_list (window);
}

static void
main_window_memory_card_removed (ObBackend        *backend,
				 ObMemoryCard     *card,
				 ObMemoryCardType  type,
				 ObMainWindow     *window)
{
	if (type == OB_MEMORY_CARD_INTERNAL) {
		if (window->internal_mem_card) {
			ob_memory_card_unref (window->internal_mem_card);
			window->internal_mem_card = NULL;
		}
	} else {
		if (window->external_mem_card) {
			ob_memory_card_unref (window->external_mem_card);
			window->external_mem_card = NULL;
		}
	}

	main_window_update_menu (window);
	main_window_update_last_info_box (window);
	main_window_update_card_info_box (window);
	main_window_update_backup_list (window);
}

static void
main_window_backups_changed (ObBackend        *backend,
			     ObMemoryCard     *card,
			     ObMemoryCardType  type,
			     ObMainWindow     *window)
{
	main_window_update_menu (window);
	main_window_update_last_info_box (window);
	main_window_update_card_info_box (window);
	main_window_update_backup_list (window);
}

static gboolean
main_window_scroll_idle_cb (gpointer data)
{
	ObMainWindow  *window = data;
	GtkAdjustment *vadj;

	window->scroll_idle_id = 0;
	
	/* Make sure to scroll to the top to show the selected item. */
	vadj = gtk_tree_view_get_vadjustment (GTK_TREE_VIEW (window->backups_treeview));
	gtk_adjustment_set_value (vadj, 0);

	return FALSE;
}

ObMainWindow *
ob_main_window_get (gboolean startup_dialog)
{
	static ObMainWindow *window = NULL;

	if (!window) {
		ObBackend *backend;

		/* Handle the case when don't get a backend if there is no dbus
		 * setup.
		 */
		backend = ob_backend_new ();

		if (!backend) {
			return NULL;
		}

		if (startup_dialog && !ob_restore_ui_startup (backend)) {
			g_object_unref (backend);
			return NULL;
		}

		window = g_new0 (ObMainWindow, 1);

		window->startup_dialog = startup_dialog;

		if (window->startup_dialog) {
			window->dialog = ob_ui_dialog_new 
				(_("back_ti_dia004_restore_selection"),
				 NULL,
				 _("back_bd_dia004_ok"), GTK_RESPONSE_OK, &window->ok_button,
				 _("back_bd_dia004_cancel"), GTK_RESPONSE_CANCEL, NULL,
				 NULL);

			gtk_widget_set_size_request (window->dialog, 580, -1);

			g_signal_connect (window->dialog,
					  "response", 
					  G_CALLBACK (main_window_response_cb),
					  window);
			
			g_signal_connect_after (window->dialog,
						"realize",
						G_CALLBACK (main_window_realize_cb),
						window);
		} else {
			window->main_window = hildon_window_new ();
			g_set_application_name ("");
			gtk_window_set_title (GTK_WINDOW (window->main_window),
					      _("back_ap_feature_title"));
			
			main_window_create_menu (window);
		}

		window->backend = backend;

		window->internal_mem_card = ob_backend_get_memory_card 
			(window->backend, OB_MEMORY_CARD_INTERNAL);
	
		if (window->internal_mem_card) {
			ob_memory_card_ref (window->internal_mem_card);
		}

		window->external_mem_card = ob_backend_get_memory_card 
			(window->backend, OB_MEMORY_CARD_EXTERNAL);

		if (window->external_mem_card) {
			ob_memory_card_ref (window->external_mem_card);
		}
		
		main_window_create_ui (window);
		
		g_signal_connect (ob_main_window_get_window (window), "destroy",
				  G_CALLBACK (main_window_destroy_cb),
				  window);
		g_signal_connect (window->backend, "memory_card_inserted",
				  G_CALLBACK (main_window_memory_card_inserted),
				  window);
		g_signal_connect (window->backend, "memory_card_removed",
				  G_CALLBACK (main_window_memory_card_removed),
				  window);
		g_signal_connect (window->backend, "backups_changed",
				  G_CALLBACK (main_window_backups_changed),
				  window);
		g_signal_connect (ob_main_window_get_window (window), "key_press_event",
				  G_CALLBACK (main_window_key_press_event_cb),
				  window);
	}

	/* This seems to be needed since for some reason the treeview scrolls
	 * down.
	 */
	window->scroll_idle_id = g_idle_add (main_window_scroll_idle_cb, window);
	
	return window;
}

ObBackend *
ob_main_window_get_backend (ObMainWindow *window)
{
	g_return_val_if_fail (window != NULL, NULL);

	return window->backend;
}

void
ob_main_window_show (ObMainWindow *window)
{
	GtkWindow *parent;

	g_return_if_fail (window != NULL);

	parent = ob_main_window_get_window (window);
	gtk_widget_show (GTK_WIDGET (parent));

	gtk_widget_grab_focus (window->backups_treeview);
}

GtkWindow *
ob_main_window_get_window (ObMainWindow *window)
{
	g_return_val_if_fail (window != NULL, NULL);

	if (window->startup_dialog) {
		return GTK_WINDOW (window->dialog);
	} else {
		return GTK_WINDOW (window->main_window);
	}
}

gboolean 
ob_main_window_is_fullscreen (ObMainWindow *window)
{
	return window->fullscreen;
}


