/*
 * This file is part of osso-backup
 *
 * Copyright (C) 2005-2006 Nokia Corporation.
 *
 * Contact: Andrey Kochanov <andrey.kochanov@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <config.h>
#include <string.h>
#include <glib.h>

#include "ob-types.h"
#include "ob-event.h"

/*
 * Event handling.
 */


struct _ObEvent {
	ObEventType type;

	union {
		struct {
			int            files;
			int            total;
			int            complete;
			ObCategory     category;
			ObProgressType progress_type;
		} progress;
		
		struct {
			ObConflictType  type;
			GnomeVFSURI    *uri;
			time_t          old_time;
			time_t          new_time;
		} conflict;

		struct {
			GError *error;
		} error;

		struct {
			GnomeVFSFileSize processed_size;
			int              processed_files;
		} finished;
		
		/* No data for "cancel" event. */
	} v;
};


void
ob_event_free (ObEvent *event)
{
	switch (event->type) {
	case OB_EVENT_PROGRESS:
		break;
	case OB_EVENT_FINISHED:
		break;
	case OB_EVENT_CANCELLED:
		break;
	case OB_EVENT_ERROR:
		g_error_free (event->v.error.error);
		break;
	case OB_EVENT_CONFLICT:
		gnome_vfs_uri_unref (event->v.conflict.uri);
		break;

	default:
		g_assert_not_reached ();
		break;
	}
	
	g_free (event);
}

ObEvent *
ob_event_new_progress (ObProgressType progress_type,
		       int            files,
		       int            total,
		       int            complete,
		       ObCategory     category)
{
	ObEvent *event;

	event = g_new0 (ObEvent, 1);
	event->type = OB_EVENT_PROGRESS;

	event->v.progress.progress_type = progress_type;
	event->v.progress.files = files;
	event->v.progress.total = total;
	event->v.progress.complete = complete;
	event->v.progress.category = category;

	return event;
}

ObEvent *
ob_event_new_error (GError *error)
{
	ObEvent *event;

	event = g_new0 (ObEvent, 1);
	event->type = OB_EVENT_ERROR;
			
	event->v.error.error = g_error_copy (error);

	return event;
}

ObEvent *
ob_event_new_conflict (ObConflictType  type,
		       GnomeVFSURI    *uri,
		       time_t          old_time,
		       time_t          new_time)
{
	ObEvent *event;

	event = g_new0 (ObEvent, 1);
	event->type = OB_EVENT_CONFLICT;

	event->v.conflict.type = type;
	event->v.conflict.uri = gnome_vfs_uri_ref (uri);
	event->v.conflict.old_time = old_time;
	event->v.conflict.new_time = new_time;

	return event;
}

ObEvent *
ob_event_new_finished (GnomeVFSFileSize processed_size,
		       int              processed_files)
{
	ObEvent *event;

	event = g_new0 (ObEvent, 1);
	event->type = OB_EVENT_FINISHED;

	event->v.finished.processed_size = processed_size;
	event->v.finished.processed_files = processed_files;
	
	return event;
}

ObEvent *
ob_event_new_cancelled (void)
{
	ObEvent *event;

	event = g_new0 (ObEvent, 1);
	event->type = OB_EVENT_CANCELLED;

	return event;
}

ObEventType
ob_event_get_type (ObEvent *event)
{
	return event->type;
}

gboolean
ob_event_get_progress (ObEvent        *event,
		       ObProgressType *progress_type,
		       int            *files,
		       int            *total,
		       int            *complete,
		       ObCategory     *category)
{
	if (event->type != OB_EVENT_PROGRESS) {
		return FALSE;
	}

	*progress_type = event->v.progress.progress_type;
	*files = event->v.progress.files;
	*total = event->v.progress.total;
	*complete = event->v.progress.complete;
	*category = event->v.progress.category;

	return TRUE;
}

gboolean
ob_event_get_error (ObEvent  *event,
		    GError  **error)
{
	if (event->type != OB_EVENT_ERROR) {
		return FALSE;
	}

	*error = g_error_copy (event->v.error.error);

	return TRUE;
}

gboolean
ob_event_get_conflict (ObEvent         *event,
		       ObConflictType  *type,
		       GnomeVFSURI    **uri,
		       time_t          *old_time,
		       time_t          *new_time)
{
	if (event->type != OB_EVENT_CONFLICT) {
		return FALSE;
	}

	*type = event->v.conflict.type;
	*uri = gnome_vfs_uri_ref (event->v.conflict.uri);
	*old_time = event->v.conflict.old_time;
	*new_time = event->v.conflict.new_time;

	return TRUE;
}

gboolean
ob_event_get_finished (ObEvent          *event,
		       GnomeVFSFileSize *size,
		       int              *files)
{
	if (event->type != OB_EVENT_FINISHED) {
		return FALSE;
	}

	*size = event->v.finished.processed_size;
	*files = event->v.finished.processed_files;
	
	return TRUE;
}

