/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * This file is part of osso-backup
 *
 * Copyright (C) 2005-2006 Nokia Corporation.
 *
 * Contact: Andrey Kochanov <andrey.kochanov@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <config.h>
#include <string.h>
#include <glib/gi18n.h>

#include "ob-error.h"
#include "ob-error-utils.h"
#include "ob-utils.h"
#include "ob-log.h"

typedef struct {
	gint   code;
	gchar *message;
} ErrorMessage;

static const ErrorMessage messages[] = {
	{ OB_ERROR_BACKUP_NO_MEMORY_CARD,
	  N_("back_fi_backup_memory_card_not_found")
	},

	{ OB_ERROR_RESTORE_NO_MEMORY_CARD,
	 N_("back_fi_restore_memory_card_not_found")
	},

	{ OB_ERROR_RESTORE_WRONG_PASSWORD,
	  N_("back_ib_password_incorrect")
	},

	{ OB_ERROR_BACKUP_NO_SPACE,
	  N_("back_fi_backup_memory_card_insufficient_space")
	},

	{ OB_ERROR_RESTORE_NO_SPACE,
	  N_("back_fi_restore_insufficient_space")
	},
	
	{ OB_ERROR_BACKUP_MEMORY_CARD_REMOVED,
	  N_("back_fi_backup_memory_card_removed")
	},

	{ OB_ERROR_RESTORE_MEMORY_CARD_REMOVED,
	  N_("back_fi_restore_memory_card_removed")
	},

	{ OB_ERROR_BACKUP_READONLY,
	  N_("back_fi_backup_memory_card_protected")
	},

	{ OB_ERROR_BACKUP_GENERIC,
	  N_("back_fi_backup_general_error")
	},
	
	{ OB_ERROR_RESTORE_GENERIC,
	  N_("back_fi_restore_general_error")
	},

	{ OB_ERROR_GENERIC,
	  N_("back_fi_general_error")
	}
};

const gchar *
ob_error_utils_get_message (GError *error)
{
	gint i;

	/* Handle no error pointer here, just to be absolutely sure that we
	 * always get a string, even in the case of a programming error.
	 */
	if (!error) {
		return _("back_fi_general_error");
	}

	/* We fall back to the most generic message if the error is not
	 * known.
	 */
	if (error->domain != OB_ERROR) {
		return _("back_fi_general_error");
	}

	/* We need to special case card-removal errors, since the reason can be
	 * that a USB cable was inserted.
	 */
	if (error->code == OB_ERROR_BACKUP_MEMORY_CARD_REMOVED) {
		gboolean usb_cable;
		
		usb_cable = ob_utils_get_is_usb_inserted ();
		if (usb_cable) {
			return dgettext ("hildon-fm", _("sfil_ib_mmc_usb_connected"));
		}
	}
	else if (error->code == OB_ERROR_RESTORE_MEMORY_CARD_REMOVED) {
		gboolean usb_cable;
		
		usb_cable = ob_utils_get_is_usb_inserted ();
		if (usb_cable) {
			return dgettext ("hildon-fm", _("sfil_ib_mmc_usb_connected"));
		}
	}
	
	for (i = 0; i < G_N_ELEMENTS (messages); i++) {
		if (messages[i].code == error->code) {
			return _(messages[i].message);
		}
	}

	return _("back_fi_general_error");
}

gchar *
ob_error_utils_get_no_space_message (GError *error,
				     GnomeVFSFileSize available_size,
				     GnomeVFSFileSize required_size)
{
	gchar       *available_size_str;
	gchar       *required_size_str;
	gchar       *msg;
	const gchar *i18n_hack;
	
	g_return_val_if_fail (error->code == OB_ERROR_BACKUP_NO_SPACE, NULL);

	available_size_str = ob_utils_get_size (available_size);
	required_size_str = ob_utils_get_size (required_size);

	i18n_hack = _("back_fi_backup_memory_card_insufficient_space");
	msg = g_strdup_printf (i18n_hack, available_size_str, required_size_str);
	
	g_free (available_size_str);
	g_free (required_size_str);
	
	return msg;
}		

