/*
 * This file is part of osso-backup
 *
 * Copyright (C) 2005-2006 Nokia Corporation.
 *
 * Contact: Andrey Kochanov <andrey.kochanov@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <config.h>
#include <glib/gi18n.h>
#include <gtk/gtk.h>
#include <string.h>
#include <hildon-widgets/gtk-infoprint.h>
#include <hildon-widgets/hildon-set-password-dialog.h>
#include <hildon-widgets/hildon-note.h>
#include <hildon-widgets/hildon-caption.h>

#include "ob-backend.h"
#include "ob-error.h"
#include "ob-ui.h"
#include "ob-utils.h"
#include "ob-selective-dialog.h"
#include "ob-ui.h"
#include "ob-backup-ui.h"
#include "ob-dbus-service.h"
#include "ob-config.h"
#include "ob-error-utils.h"
#include "ob-log.h"

static void backup_ui_memory_card_removed_cb (ObBackend        *backend,
					      ObMemoryCard     *card,
					      ObMemoryCardType  type,
					      GtkDialog        *dialog);


static void
backup_ui_show_card_removed_dialog (GtkWindow *parent)
{
	GtkWidget   *dialog;
	gboolean     usb_cable;
	const gchar *str;
	
	usb_cable = ob_utils_get_is_usb_inserted ();
	if (usb_cable) {
		str = dgettext ("hildon-fm", _("sfil_ib_mmc_usb_connected"));
	} else {
		str = _("back_fi_backup_memory_card_removed");
	}
	
	dialog = hildon_note_new_information (parent, str);
	gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_widget_destroy (dialog);
}

static void
send_backup_signal (const gchar *signal_name)
{
	DBusConnection *conn;
	DBusMessage    *message;

	ob_log_info ("Sending backup signal %s", signal_name);

	conn = ob_utils_get_dbus_connection ();
	if (conn) {
		message = dbus_message_new_signal (
			OB_DBUS_PATH, OB_DBUS_INTERFACE, signal_name);
		dbus_connection_send (conn, message, NULL);
		dbus_connection_flush (conn);

		dbus_message_unref (message);
	} else {
		ob_log_warning ("Could not get session bus.");
	}
	
	conn = dbus_bus_get (DBUS_BUS_SYSTEM, NULL);
	if (conn) {
		message = dbus_message_new_signal (
			OB_DBUS_PATH, OB_DBUS_INTERFACE, signal_name);
		dbus_connection_send (conn, message, NULL);
		dbus_connection_flush (conn);

		dbus_message_unref (message);
	} else {
		ob_log_warning ("Could not get system bus.");
	}
}

static void
backend_progress_cb (ObBackend        *backend,
                     int               files, 
                     int               total,  
                     int               complete, 
		     ObCategory        category,
                     ObProgressDialog *dialog)
{
        ob_ui_progress_dialog_update (dialog, files, complete, category);
}
 
static void
backend_error_cb (ObBackend *backend, GError *error, ObProgressDialog *dialog)
{
        dialog->error = g_error_copy (error);
        gtk_dialog_response (GTK_DIALOG (dialog->dialog), OB_UI_RESPONSE_ERROR);
}

static void
backend_finished_cb (ObBackend *backend, ObProgressDialog *dialog)
{
        gtk_dialog_response (GTK_DIALOG (dialog->dialog), 
                             OB_UI_RESPONSE_FINISHED);
}

static void
backend_cancelled_cb (ObBackend *backend, ObProgressDialog *dialog)
{
	/* Note: This is for cancellation due to reasons other than the user
	 * pressing cancel.
	 */
        gtk_dialog_response (GTK_DIALOG (dialog->dialog), 
                             OB_UI_RESPONSE_CANCELLED);
}

/* Make the backup dialog system model. */
static void
backup_ui_realize_cb (GtkWidget *window,
		      gpointer   data)
{
	GdkWindow *root;

	root = gdk_screen_get_root_window (gtk_widget_get_screen (window));
	
	gdk_window_set_transient_for (window->window, root);
	gdk_window_set_modal_hint (window->window, TRUE);
}

/* The replaced_size is the uncompressed size of any previous backup of the same
 * name that will be replaced. It's used to estimate if the backup will fit on
 * the MMC card.
 */
static void
backup_ui_run_backup (ObBackend        *backend,
		      ObMemoryCardType  type,
                      GtkWindow        *parent,
                      const char       *backup_name,
                      const char       *password,
                      int               categories,
		      GnomeVFSFileSize  replaced_size)
{
        ObProgressDialog *progress_dialog;
        gint              resp_id;
	GError           *error = NULL;
        GtkWidget        *info_dialog;
	gchar            *str;
	gchar            *size_str;
	ObMemoryCard     *mem_card;
	time_t            timestamp;
	const char       *error_msg;
	char             *error_msg_no_space;
	int               num_files;
	GnomeVFSFileSize  size;
	ObCategoryFiles  *files;
	int               processed_files, total_files;
	const gchar      *i18n_hack;
	GnomeVFSFileSize  available_size;
	GnomeVFSFileSize  required_size;

	/* NOTE: Ideally, the file counting should be done in the archiver
	 * thread and be thus be cancellable.
	 */

	/* Count the files that are in the locations conf. This is for files
	 * that aren't in "MyDocs".
	 */
	ob_backend_count_data (backend,
			       categories,
			       &num_files,
			       &size);

	/* Get the files managed under "MyDocs". */
	files = ob_category_files_new ();
	ob_category_files_read (files, categories);

	size += ob_category_files_get_size (files, categories);
	num_files += ob_category_files_get_num_files (files, categories);

	/*ob_log_info ("BackupUI: total files:%d, total size: %d KB\n", num_files, (int) size / 1024);*/
	
	/* Don't try to backup if there is no data. */
        if (num_files == 0) {
                info_dialog = hildon_note_new_information (
                        parent,
                        _("back_fi_nodata_tobackup"));
                gtk_dialog_run (GTK_DIALOG (info_dialog));
                gtk_widget_destroy (info_dialog);

		ob_category_files_free (files);

                return;
        }
        
        progress_dialog = ob_ui_progress_dialog_new (parent, 
                                                     OB_UI_OPERATION_BACKUP,
                                                     num_files);

        g_signal_connect (backend, "progress",
                          G_CALLBACK (backend_progress_cb),
                          progress_dialog);
        g_signal_connect (backend, "error",
                          G_CALLBACK (backend_error_cb),
                          progress_dialog);
        g_signal_connect (backend, "finished",
                          G_CALLBACK (backend_finished_cb),
                          progress_dialog);
	g_signal_connect (backend,
			  "cancelled",
			  G_CALLBACK (backend_cancelled_cb),
			  progress_dialog);

        g_signal_connect_after (progress_dialog->dialog,
				"realize",
				G_CALLBACK (backup_ui_realize_cb),
				NULL);

	timestamp = time (NULL);

	if (replaced_size > size) {
		required_size = 0;
	} else {
		required_size = size - replaced_size;
	}

	/* We need some margin since this is just an estimate. */
	required_size += 64*1024;
	
	if (!ob_backend_space_on_memory_card (backend,
					      type,
					      required_size,
					      &available_size)) {
		error = g_error_new (OB_ERROR,
				     OB_ERROR_BACKUP_NO_SPACE,
				     "No space");
		
		resp_id = OB_UI_RESPONSE_ERROR;
	}
	else if (!ob_backend_start_backup (backend, type,
					   files, backup_name, password,
					   timestamp, categories,
					   num_files, size,
					   /* Note: This is left from when
					    * updating backups was possible:
					    */
					   NULL,
					   &error)) {
		resp_id = OB_UI_RESPONSE_ERROR;
	} else {
		resp_id = gtk_dialog_run (GTK_DIALOG (progress_dialog->dialog));
		if (resp_id == OB_UI_RESPONSE_ERROR) {
			error = g_error_copy (progress_dialog->error);
		}
	}

        g_signal_handlers_disconnect_by_func (backend, 
                                              backend_progress_cb, 
                                              progress_dialog);
        g_signal_handlers_disconnect_by_func (backend,
                                              backend_error_cb,
                                              progress_dialog);
        g_signal_handlers_disconnect_by_func (backend,
                                              backend_finished_cb,
                                              progress_dialog);
        g_signal_handlers_disconnect_by_func (backend,
                                              backend_cancelled_cb,
                                              progress_dialog);

	total_files = progress_dialog->total_files;

	gtk_dialog_set_response_sensitive (GTK_DIALOG (progress_dialog->dialog),
					   GTK_RESPONSE_CANCEL, FALSE);
	
        switch (resp_id) {
        case OB_UI_RESPONSE_FINISHED:
		ob_ui_progress_dialog_destroy (progress_dialog);

		mem_card = ob_backend_get_memory_card (backend, type);
		if (mem_card) {		
			ob_utils_set_last_backup (backup_name,
						  ob_memory_card_get_name (mem_card),
						  timestamp);
		}

		/* Make sure to add the app killing delay to get it included in
		 * the timing. */
		size = ob_backend_get_processed_size (backend);
		size_str = ob_utils_get_size (size);

		/* Formatted logical string. */
		i18n_hack = _("back_fi_not011_backup_successfully_completed");
		str = g_strdup_printf (i18n_hack, size_str);
		
                info_dialog = hildon_note_new_information (parent, str);
		g_free (str);
		g_free (size_str);

                gtk_dialog_run (GTK_DIALOG (info_dialog));
		gtk_widget_destroy (info_dialog);
		break;

        case OB_UI_RESPONSE_ERROR:
		/* Special case this error. */
		if (error->code == OB_ERROR_BACKUP_NO_SPACE) {
			error_msg_no_space = ob_error_utils_get_no_space_message (
				error, available_size, required_size);
			error_msg = error_msg_no_space;
		} else {
			error_msg = ob_error_utils_get_message (error);
			error_msg_no_space = NULL;
		}

		/* Log the raw message. */
		ob_log_error (error->message);
		
		ob_ui_progress_dialog_destroy (progress_dialog);

                info_dialog = hildon_note_new_information (parent,
                                                           error_msg);
		
                gtk_dialog_run (GTK_DIALOG (info_dialog));
		gtk_widget_destroy (info_dialog);

		g_free (error_msg_no_space);

                break;

        case OB_UI_RESPONSE_CANCELLED:
		/* Note: This is for when we are cancelled from outside for
		 * backup.
		 */
        case GTK_RESPONSE_CANCEL:
        case GTK_RESPONSE_DELETE_EVENT:
		ob_backend_cancel (backend);

		/* When the backend responds, we can close the progress
		 * dialog. This way it doesn't look like the operation is
		 * cancelled before it is.
		 */
		ob_ui_progress_dialog_destroy (progress_dialog);

		/* Make sure that we didn't get an error after cancelling. */
		error = ob_backend_get_error (backend);
		if (error) {
			/* Special case this error. */
			if (error->code == OB_ERROR_BACKUP_NO_SPACE) {
				error_msg_no_space = ob_error_utils_get_no_space_message (
					error, available_size, required_size);
				error_msg = error_msg_no_space;

			} else {
				error_msg = ob_error_utils_get_message (error);
				error_msg_no_space = NULL;
			}

			/* Log the raw message. */
			ob_log_error (error->message);
			
			info_dialog = hildon_note_new_information (parent,
								   error_msg);
			gtk_dialog_run (GTK_DIALOG (info_dialog));
			gtk_widget_destroy (info_dialog);

			g_free (error_msg_no_space);
			error = NULL;
			break;
		}

		processed_files = ob_backend_get_processed_files (backend);
		if (processed_files > 0) {
			mem_card = ob_backend_get_memory_card (backend, type);
			if (mem_card) {
				ob_utils_set_last_backup (backup_name,
							  ob_memory_card_get_name (mem_card),
							  timestamp);
			}
		}
		
		/* Formatted logical string. */
		i18n_hack = _("back_fi_not028_msg");
		str = g_strdup_printf (i18n_hack, processed_files, total_files);

                info_dialog = hildon_note_new_information (parent, str);
                gtk_dialog_run (GTK_DIALOG (info_dialog));
		gtk_widget_destroy (info_dialog);
		g_free (str);
                break;
        }

	if (error) {
		g_error_free (error);
	}

	ob_category_files_free (files);
}

static void
backup_ui_memory_card_removed_cb (ObBackend        *backend,
				  ObMemoryCard     *card,
				  ObMemoryCardType  type,
				  GtkDialog        *dialog)
{
	gpointer p;

	ob_log_info ("BackupUI: %s memory card removed...", 
		     type == OB_MEMORY_CARD_INTERNAL ? "Internal" : "External");
	
	p = g_object_get_data (G_OBJECT (dialog), "memory_card_type");
	if (p) {
		ObMemoryCardType memory_card_type;

		memory_card_type = GPOINTER_TO_INT (p);

		ob_log_info ("BackupUI: %s memory card is what we interested in", 
			     memory_card_type == OB_MEMORY_CARD_INTERNAL ? "Internal" : "External");
			
		/* If the memory card type we want to know about is
		 * not the same as the one we have the signal for, we
		 * silently ignore it. 
		 */
		if (memory_card_type != type) {
			ob_log_info ("BackupUI: Not sending OB_UI_RESPONSE_CARD_REMOVED signal");
			return;
		}
	}

	gtk_dialog_response (dialog, OB_UI_RESPONSE_CARD_REMOVED);
}

static void
backup_ui_memory_card_inserted_cb (ObBackend        *backend,
				   ObMemoryCard     *card,
				   ObMemoryCardType  type,
				   GtkDialog        *dialog)
{
	gtk_dialog_response (dialog, OB_UI_RESPONSE_CARD_INSERTED);
}

static gboolean
backup_ui_ask_replace_existing (ObMainWindow *window, 
				ObBackend    *backend,
                                ObBackupInfo *info)
{
	GtkWindow        *parent;
	GtkWidget        *dialog;
	gchar            *date_str;
	gchar            *time_str;
	gchar            *str;
	int               resp_id;
	guint             id;
	ObMemoryCardType  memory_card_type;
	const gchar      *i18n_hack;

	parent = ob_main_window_get_window (window);

        date_str = ob_utils_get_date_string (ob_backup_info_get_timestamp (info));
        time_str = ob_utils_get_time_string (ob_backup_info_get_timestamp (info));

	/* Formatted logical string. */
	i18n_hack = _("back_fi_not026_msg");
	str = g_strdup_printf (i18n_hack, date_str, time_str);
	
        dialog = hildon_note_new_confirmation (parent, str);
        hildon_note_set_button_text (HILDON_NOTE (dialog),
				     _("back_bd_dia026_continue_button"));

	g_free (str);
        g_free (date_str);
        g_free (time_str);

	memory_card_type = ob_backup_info_get_card_type (info);
	g_object_set_data (G_OBJECT (dialog), "memory_card_type", 
			   GINT_TO_POINTER (memory_card_type));
	
	id = g_signal_connect (backend, "memory_card_removed",
			       G_CALLBACK (backup_ui_memory_card_removed_cb),
			       dialog);

	resp_id = gtk_dialog_run (GTK_DIALOG (dialog));
        g_signal_handler_disconnect (backend, id);

	gtk_widget_destroy (dialog);
	
	if (resp_id == OB_UI_RESPONSE_CARD_REMOVED) {
		backup_ui_show_card_removed_dialog (parent);
	}
	
	return (resp_id == GTK_RESPONSE_OK);
}

static void
backup_ui_name_entry_changed_cb (GtkWidget *entry,
				 GtkDialog *dialog)
{
	gchar    *name, *stripped;
	gint      len;
	gboolean  sensitive;

	name = g_strdup (gtk_entry_get_text (GTK_ENTRY (entry)));

	stripped = g_strstrip (name);
	len = strlen (stripped);

	if (len > 0) {
		sensitive = TRUE;
	} else {
		sensitive = FALSE;
	}

	g_free (name);
	
	gtk_dialog_set_response_sensitive (dialog, GTK_RESPONSE_OK, sensitive);

	if (len >= 50) {
		GtkWidget *parent;
		
		parent = gtk_widget_get_toplevel (GTK_WIDGET (entry));
		
		gtk_infoprint (GTK_WINDOW (parent),
			       dgettext ("hildon-common-strings",
					"ckdg_ib_maximum_characters_reached"));
	}
}
			
static gboolean
backup_ui_name_entry_focus_cb (GtkEntry *name_entry)
{
  	gtk_widget_grab_focus (GTK_WIDGET (name_entry)); 
	gtk_editable_select_region (GTK_EDITABLE (name_entry), 0, -1);
	return FALSE;
}

static void
backup_ui_name_entry_destroy_cb (GtkWidget *name_entry,
				 gpointer   user_data)
{
	gpointer p;
	guint    id;

	p = g_object_get_data (G_OBJECT (name_entry), "idle_id");
	id = GPOINTER_TO_UINT (p);

	if (id > 0) {
		g_source_remove (id);
	}
}

static void
backup_ui_ok_insensitive_press_cb (GtkButton *button,
				   GtkEntry  *name_entry)
{
	GtkWidget *parent;
	guint      id;

	parent = gtk_widget_get_toplevel (GTK_WIDGET (button));

	gtk_infoprint (GTK_WINDOW (parent),
		       dgettext ("hildon-common-strings", "ckdg_ib_enter_name"));

	gtk_entry_set_text (name_entry, _("back_ia_dia002_name"));

	id = g_idle_add ((GSourceFunc)backup_ui_name_entry_focus_cb, name_entry);
	g_object_set_data (G_OBJECT (name_entry), "idle_id", GUINT_TO_POINTER (id));
}

/* Get a new password (the set_password dialog, with two entries), returns the
 * response id and sets the password if successful.
 */
static int
backup_ui_get_new_password (ObBackend         *backend,
			    ObMemoryCardType   type,
			    ObMainWindow      *window,
			    gchar            **password)
{
	GtkWidget *pw_dialog;
	guint      id;
	gint       resp_id;
	
	*password = NULL;
	
	pw_dialog = hildon_set_password_dialog_new (ob_main_window_get_window (window), 
						    FALSE);
	ob_utils_setup_dialog_help (GTK_DIALOG (pw_dialog), "features_backup_setpassword");

	g_object_set_data (G_OBJECT (pw_dialog), "memory_card_type", 
			   GINT_TO_POINTER (type));
	
	id = g_signal_connect (backend, "memory_card_removed",
			       G_CALLBACK (backup_ui_memory_card_removed_cb),
			       pw_dialog);
	
	resp_id = gtk_dialog_run (GTK_DIALOG (pw_dialog));

	g_signal_handler_disconnect (backend, id);

	if (resp_id != GTK_RESPONSE_OK) {
		gtk_widget_destroy (pw_dialog);
		return resp_id;
	}
	
	*password = g_strdup (hildon_set_password_dialog_get_password (
				      HILDON_SET_PASSWORD_DIALOG (pw_dialog)));
	
	gtk_widget_destroy (pw_dialog);

	return resp_id;
}

/* Note: This is a workaround for the fact that the combobox in the modified
 * GTK+ version has ellipsizing enabled, and there is now way to disable it. So
 * in order to fit all translations in the dialog, we need to explicitly set the
 * size of the combo box. */
static void
backup_ui_combo_box_style_set_cb (GtkWidget *widget,
				  GtkStyle  *previous_style,
				  gpointer   user_data)
{
	GtkTreeModel *model;
	PangoLayout  *layout;
	gint          width;
	gint          focus_width, focus_pad;
	gint          arrow_width;
	GtkTreeIter   iter;

	model = gtk_combo_box_get_model (GTK_COMBO_BOX (widget));
	if (!gtk_tree_model_get_iter_first (model, &iter)) {
		return;
	}
	
	gtk_widget_style_get (widget,
			      "arrow-width", &arrow_width,
			      "focus-line-width", &focus_width,
			      "focus-padding", &focus_pad,
			      NULL);
	
	layout = gtk_widget_create_pango_layout (widget, NULL);
	pango_layout_set_ellipsize (layout, PANGO_ELLIPSIZE_NONE);

	width = 0;

	do {
		gint   w;
		gchar *str;
		
		gtk_tree_model_get (model, &iter,
				    0, &str,
				    -1);

		pango_layout_set_text (layout, str, -1);
		g_free (str);

		pango_layout_get_pixel_size (layout, &w, NULL);
		width = MAX (width, w);
	} while (gtk_tree_model_iter_next (model, &iter));

	g_object_unref (layout);

	if (width > 0) {
		/* This is not the exact width, but there is no way to get the
		 * real width so an approximation will have to do.
		 */
		
		width += 2 * widget->style->xthickness;

		width += arrow_width;
		width += 2 * (focus_width + focus_pad);

		width += 2 * GTK_CONTAINER (widget)->border_width;

		/* This is to compensate for some of the width's we can't really
		 * get, like padding around a cell frame or something similar.
		 */
		width += 2 * 4;
		
		gtk_widget_set_size_request (widget, width, -1);
	}
}

static int
sort_backups_by_date_func (gconstpointer a, gconstpointer b)
{
	ObBackupInfo *info_a, *info_b;
	time_t        time_a, time_b;

	info_a = (ObBackupInfo *) a;
	info_b = (ObBackupInfo *) b;

	time_a = ob_backup_info_get_timestamp (info_a);
	time_b = ob_backup_info_get_timestamp (info_b);

	if (time_a < time_b) {
		return 1;
	}
	else if (time_a > time_b) {
		return -1;
	} else {
		return 0;
	}
}

/* Returns FALSE if nothing was done, i.e. the backup list doesn't need to be
 * updated.
 */
gboolean
ob_backup_ui_run (ObBackend    *backend, 
		  ObMainWindow *window)
{
	GtkWindow         *parent;
        GtkWidget         *dialog;
        GtkWidget         *caption;
	GtkWidget         *combo = NULL;
        GtkWidget         *location_label;
        GtkWidget         *name_entry;
	GtkWidget         *ok_button;
        GtkWidget         *vbox;
        int                resp_id;
        gchar             *backup_name;
	ObMemoryCard      *mem_card_internal;
	ObMemoryCard      *mem_card_external;
	ObMemoryCard      *mem_card = NULL;
	ObMemoryCardType   type = OB_MEMORY_CARD_INTERNAL;
	GtkSizeGroup      *group;
	GList             *backups,*l;
	gchar             *name = NULL;
	ObBackupInfo      *backup_info;
	gboolean           protect;
	gchar             *password;
	gint               categories;
	ObSelectiveDialog *selective_dialog;
	guint              id_removed;
	guint              id_inserted;
#ifdef SET_FLIGHT_MODE
	gboolean           flight_mode;
#endif
	GnomeVFSFileSize   replaced_size;
	gchar             *folded_backup_name;
	
	parent = ob_main_window_get_window (window);

        dialog = ob_ui_dialog_new (_("back_ti_dia002_title"),
				   parent,
                                   _("back_bd_dia002_ok"), GTK_RESPONSE_OK, &ok_button,
                                   _("back_bd_dia002_cancel"), GTK_RESPONSE_CANCEL, NULL,
                                   NULL);

	ob_utils_setup_dialog_help (GTK_DIALOG (dialog), "features_backup_backupname");

	vbox = gtk_vbox_new (FALSE, 0);
        gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox),
                                     vbox, FALSE, FALSE, 0);

        name_entry = gtk_entry_new ();
	g_signal_connect (name_entry,
			  "changed",
			  G_CALLBACK (backup_ui_name_entry_changed_cb),
			  dialog);
	g_signal_connect (name_entry,
			  "destroy",
			  G_CALLBACK (backup_ui_name_entry_destroy_cb),
			  dialog);
	g_signal_connect (ok_button, "insensitive-press", 
			  G_CALLBACK (backup_ui_ok_insensitive_press_cb),
			  name_entry);

	gtk_entry_set_max_length (GTK_ENTRY (name_entry), 50);

	group = gtk_size_group_new (GTK_SIZE_GROUP_BOTH);

	caption = hildon_caption_new (group, _("back_fi_dia002_bkp_name"),
				      name_entry, NULL, HILDON_CAPTION_OPTIONAL);
	gtk_box_pack_start (GTK_BOX (vbox), caption, FALSE, FALSE, 0);

	/* If we have more than one memory card, we present a combo
	 * box to select the location to save to, otherwise we use the
	 * old method of showing a label with the only location we can
	 * save to.
	 */

locations_changed:
	mem_card_internal = ob_backend_get_memory_card (backend, 
							OB_MEMORY_CARD_INTERNAL);
	mem_card_external = ob_backend_get_memory_card (backend, 
							OB_MEMORY_CARD_EXTERNAL);

	if (!mem_card_internal && !mem_card_external) {
 		gtk_widget_destroy (dialog); 

		backup_ui_show_card_removed_dialog (parent);

		return FALSE;
	}

	if (mem_card_internal) {
		type = OB_MEMORY_CARD_INTERNAL;
		mem_card = mem_card_internal;
	} else {
		type = OB_MEMORY_CARD_EXTERNAL;
		mem_card = mem_card_external;
	}

	name = ob_backend_get_available_name (backend, -1);
	if (name) {
		gtk_entry_set_text (GTK_ENTRY (name_entry), name);
		g_free (name);
	}

	if (ob_backend_count_memory_cards (backend) > 1) {
		GList        *all_backups = NULL;
		ObBackupInfo *latest_backup;
		gint          active_card;

		combo = gtk_combo_box_new_text ();

		if (mem_card_internal) {
			gtk_combo_box_append_text (GTK_COMBO_BOX (combo), 
						   ob_memory_card_get_name (mem_card_internal));

			all_backups = g_list_copy (
				ob_memory_card_get_backups (mem_card_internal));
		}

		if (mem_card_external) {
			gtk_combo_box_append_text (GTK_COMBO_BOX (combo), 
						   ob_memory_card_get_name (mem_card_external));

			all_backups = g_list_concat (
				all_backups,
				g_list_copy (ob_memory_card_get_backups (mem_card_external)));
		}

		if (!all_backups) {
			active_card = 0;
		} else {
			all_backups = g_list_sort (all_backups, sort_backups_by_date_func);
			latest_backup = all_backups->data;

			if (ob_backup_info_get_card_type (latest_backup) ==
			    OB_MEMORY_CARD_INTERNAL) {
				active_card = 0;
			} else {
				active_card = 1;
			}

			g_list_free (all_backups);
		}			

		gtk_combo_box_set_active (GTK_COMBO_BOX (combo), active_card);

		g_signal_connect (combo, "style_set",
				  G_CALLBACK (backup_ui_combo_box_style_set_cb),
				  NULL);
		
		caption = hildon_caption_new (group, _("back_fi_dia002_location"),
					      combo, NULL, HILDON_CAPTION_OPTIONAL);
	} else {
		/* Get internal memory card */
		location_label = ob_ui_label_new (ob_memory_card_get_name (mem_card));
		caption = hildon_caption_new (group, _("back_fi_dia002_location"),
					      location_label, NULL, HILDON_CAPTION_OPTIONAL);
	}

	gtk_box_pack_start (GTK_BOX (vbox), caption, FALSE, FALSE, 0);

        gtk_widget_show_all (dialog);

change_name:
	/* This seems to be needed to be able to type in the entry... No idea
	 * why, might be a bug in the patched GTK.
	 */
	gtk_widget_grab_focus (name_entry);
	gtk_editable_select_region (GTK_EDITABLE (name_entry), 0, -1);

        backup_name = NULL;

	/* We DON'T set the memory_card_type object data here because
	 * if ANY memory card is removed, we want to know about it.
	 */
	id_removed = g_signal_connect (backend, "memory_card_removed",
				       G_CALLBACK (backup_ui_memory_card_removed_cb),
				       dialog);
	id_inserted = g_signal_connect (backend, "memory_card_inserted",
					G_CALLBACK (backup_ui_memory_card_inserted_cb),
					dialog);
	
	resp_id = gtk_dialog_run (GTK_DIALOG (dialog));

        g_signal_handler_disconnect (backend, id_removed);
        g_signal_handler_disconnect (backend, id_inserted);

	if (resp_id == OB_UI_RESPONSE_CARD_REMOVED || 
	    resp_id == OB_UI_RESPONSE_CARD_INSERTED) {
		gtk_widget_destroy (caption);
		goto locations_changed;
	}

	g_object_unref (group);
	
	if (resp_id == GTK_RESPONSE_OK) {
		const gchar *str;
		gchar       *tmp;
		
		str = gtk_entry_get_text (GTK_ENTRY (name_entry));

		/* Check for invalid characters */
		if (strchr (str, '?') || 
		    strchr (str, '"') || 
 		    strchr (str, '/') ||
 		    strchr (str, '\\') ||
 		    strchr (str, '*') ||
 		    strchr (str, '<') ||
 		    strchr (str, '>')) {
			const char *format;
			gchar      *message;

			format = dgettext ("hildon-common-strings",
					   "ckdg_ib_illegal_characters_entered");

			message = g_strdup_printf (format, "?\"/\\*<>");
			gtk_infoprint (GTK_WINDOW (parent), message);
			g_free (message);

			goto change_name;
		}

		tmp = g_strdup (str);
		backup_name = g_strdup (g_strstrip (tmp));
		g_free (tmp);

		/* Get selected location */
		if (ob_backend_count_memory_cards (backend) > 1) {
			if (gtk_combo_box_get_active (GTK_COMBO_BOX (combo)) == 0) {
				type = OB_MEMORY_CARD_INTERNAL;
			} else {
				type = OB_MEMORY_CARD_EXTERNAL;
			}
			
			mem_card = ob_backend_get_memory_card (backend, type);
		}
	}

        gtk_widget_destroy (dialog);

	if (resp_id == OB_UI_RESPONSE_CARD_REMOVED) {
		backup_ui_show_card_removed_dialog (parent);
	}
	
	if (resp_id != GTK_RESPONSE_OK) {
		return FALSE;
	}

	/* First ask if we should replace any existing backup with the same
	 * name.
	 */
	backups = ob_memory_card_get_backups (mem_card);
	backup_info = NULL;

	folded_backup_name = g_utf8_casefold (backup_name, -1);
	
	for (l = backups; l; l = l->next) {
		gchar *tmp;
		
		backup_info = OB_BACKUP_INFO (l->data);
		
		name = ob_backup_info_get_name (backup_info);

		/* Try to casefold, but fallback to the original name in case
		 * someone copied in non-UTF8 files.
		 */
		tmp = g_utf8_casefold (name, -1);
		if (tmp) {
			g_free (name);
			name = tmp;
		}

		if (strcmp (name, folded_backup_name) == 0) {
			/* Got conflicting name. */
			ob_backup_info_ref (backup_info);
			g_free (name);
			break;
		}
		
		g_free (name);
		backup_info = NULL;
	}

	g_free (folded_backup_name);
	
	if (backup_info) {
		if (!backup_ui_ask_replace_existing (window, backend, backup_info)) {
			ob_backup_info_unref (backup_info);
			return FALSE;
		}

		/* Get the uncompressed size on the card, and save it so that we
 		 * can use it to estimate further on if the new backup will fit
 		 * on the memory card. It's not exact, but if we compare the
 		 * uncompressed size on the card with the uncompressed size of
 		 * the data to backup, it will be fairly accurate.
 		 */
 		replaced_size = ob_backup_info_get_size_for_categories (
 			backup_info, OB_CATEGORY_ALL);

		ob_backup_info_unref (backup_info);
		backup_info = NULL;
	} else {
		replaced_size = 0;
  	}
	
	selective_dialog = ob_selective_dialog_new (parent,
						    OB_UI_OPERATION_BACKUP,
						    backend,
						    type);
	
 try_again:
	resp_id = ob_selective_dialog_run (selective_dialog);

	if (resp_id != GTK_RESPONSE_OK) {
		ob_selective_dialog_destroy (selective_dialog);
		if (backup_info) {
			ob_backup_info_unref (backup_info);
		}

		if (resp_id == OB_UI_RESPONSE_CARD_REMOVED) {
			backup_ui_show_card_removed_dialog (parent);
		}

		return FALSE;
	}
	
	protect = ob_selective_dialog_get_protect (selective_dialog);
	categories = ob_selective_dialog_get_categories (selective_dialog);

	password = NULL;
	if (protect) {
		resp_id = backup_ui_get_new_password (backend, type, window, &password);
		if (resp_id != GTK_RESPONSE_OK) {
			if (resp_id == OB_UI_RESPONSE_CARD_REMOVED) {
				ob_selective_dialog_destroy (selective_dialog);
				if (backup_info) {
					ob_backup_info_unref (backup_info);
				}
				
				backup_ui_show_card_removed_dialog (parent);
				return FALSE;
			}
			
			goto try_again;
		}
	}

	ob_selective_dialog_destroy (selective_dialog);

#ifdef SET_FLIGHT_MODE
	flight_mode = ob_utils_get_flight_mode ();
	if (!flight_mode) {
		ob_utils_set_flight_mode (TRUE);
	}
#endif
	
	send_backup_signal (OB_DBUS_SIGNAL_BACKUP_START);

	backup_ui_run_backup (backend, type, parent,
			      backup_name, password, categories,
			      replaced_size);

#ifdef SET_FLIGHT_MODE
	if (!flight_mode) {
		ob_utils_set_flight_mode (FALSE);
	}
#endif

	/* Send backup finished signal. */
	send_backup_signal (OB_DBUS_SIGNAL_BACKUP_FINISH);

	if (backup_info) {
		ob_backup_info_unref (backup_info);
	}
	
	g_free (backup_name);
	g_free (password);

	return TRUE;
}
