/*
 * This file is part of osso-backup
 *
 * Copyright (C) 2005 Nokia Corporation.
 *
 * Contact: Andrey Kochanov <andrey.kochanov@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <config.h>
#include <stdio.h>
#include <string.h>
#include <gsf/gsf-input-impl.h>
#include <gsf/gsf-impl-utils.h>
#include <gsf/gsf-utils.h>
#include <openssl/aes.h>

#include "gsf-input-crypt.h"
#include "ob-utils.h"

#define d(x) 

static GObjectClass *parent_class;

struct _GsfInputCrypt {
	GsfInput   input;

	GsfInput  *source;

	guint8    *buf;
	size_t     buf_size;

	/* The amount of padding at the end of the file. */
	int        padding;

	guchar    *hash;
	AES_KEY    aes;
};

typedef struct {
	GsfInputClass input_class;
} GsfInputCryptClass;


static gboolean
check_input (GsfInputCrypt *input)
{
	gsf_off_t size;
	char      pad_buf[32];
	char      pad_char;
	int       padding;
	
	if (gsf_input_seek (input->source, 0, G_SEEK_SET)) {
		return FALSE;
	}

	size = gsf_input_size (GSF_INPUT (input->source));
	
	/* We have always at least 16 bytes for the IV and one block for
	 * data.
	 */
	if (size < 32) {
		return FALSE;
	}

	/* Seek to the end to get the amount of padding. */
	if (gsf_input_seek (input->source, -32, G_SEEK_END)) {
		return FALSE;
	}

	/* Subtract the IV from the size. */
	size -= 16; 

	gsf_input_read (input->source, 32, pad_buf);

	/* Decrypt the last block in the file to get the padding. The second
	 * last block is the IV block for the last one.
	 */
	ob_utils_decrypt_data (&input->aes,
			       pad_buf,
			       pad_buf + 16,
			       16);
	
	/* Get the last char. */
	pad_char = pad_buf[31];
	
	/* Subtract the padding from the size. */
	if (pad_char >= '0' && pad_char <= '9') {
		padding = pad_char - '0';
	}
	else if (pad_char >= 'A' && pad_char <= 'H') {
		padding = pad_char - 'A' + 10;
	} else {
		return FALSE;
	}

	size -= padding;

	d(g_print ("size reduced to %d\n", (int) size));

	input->padding = padding;
	
	gsf_input_set_size (GSF_INPUT (input), size);

	if (gsf_input_seek (input->source, 0, G_SEEK_SET)) {
		return FALSE;
	}

	return TRUE;
}

static gboolean
init_crypt (GsfInputCrypt *crypt, GError **err)
{
	gsf_off_t cur_pos;

	if (!crypt->hash) {
		/* No password, just pass the data through. */

		gsf_input_set_size (GSF_INPUT (crypt),
				    gsf_input_size (crypt->source));

		return TRUE;
	}
	
	/* Initialize the AES data. */
	AES_set_decrypt_key (crypt->hash, 128, &crypt->aes);

	cur_pos = gsf_input_tell (crypt->source);

	if (!check_input (crypt)) {
		if (err != NULL) {
			*err = g_error_new (gsf_input_error (), 0,
					    "Crypted file invalid");
		}
		if (gsf_input_seek (crypt->source, cur_pos, G_SEEK_SET)) {
			g_warning ("attempt to restore position failed");
		}
		return FALSE;
	}

	return TRUE;
}

/**
 * gsf_input_crypt_new :
 * @source : The underlying data source.
 * @password: Password to protect data with, or NULL if no password should be used.
 * @err	   : optionally NULL.
 *
 * Adds a reference to @source.
 *
 * Returns a new file or NULL.
 **/
GsfInput *
gsf_input_crypt_new (GsfInput *source, const char *password, GError **err)
{
	GsfInputCrypt *crypt;
	
	g_return_val_if_fail (GSF_IS_INPUT (source), NULL);

	crypt = g_object_new (GSF_INPUT_CRYPT_TYPE, NULL);
	
	crypt->source = g_object_ref (source);

	if (password) {
		crypt->hash = ob_utils_get_128_bits_hash (password);
	}

	if (!init_crypt (crypt, err)) {
		g_object_unref (crypt);
		return NULL;
	}

	return GSF_INPUT (crypt);
}

static void
gsf_input_crypt_finalize (GObject *obj)
{
	GsfInputCrypt *crypt = (GsfInputCrypt *) obj;

	if (crypt->source != NULL) {
		g_object_unref (crypt->source);
		crypt->source = NULL;
	}

	g_free (crypt->hash);
	g_free (crypt->buf);

	parent_class->finalize (obj);
}

static GsfInput *
gsf_input_crypt_dup (GsfInput *src_input, GError **err)
{
	GsfInputCrypt const *src = (GsfInputCrypt *) src_input;
	GsfInputCrypt *dst = g_object_new  (GSF_INPUT_CRYPT_TYPE, NULL);

	dst->source = gsf_input_dup (src->source, NULL);
	dst->hash = g_memdup (src->hash, 16);

	if (!init_crypt (dst, err)) {
		g_object_unref (dst);
		return NULL;
	}

	return GSF_INPUT (dst);
}

static guint8 const *
gsf_input_crypt_read (GsfInput *input, size_t num_bytes, guint8 *buffer)
{
	GsfInputCrypt *crypt = GSF_INPUT_CRYPT (input);
	size_t         real_offset, real_num_bytes;
	int            pad;
	guchar        *crypted_data;
 	guchar         xor[16];

	if (buffer == NULL) {
		if (crypt->buf_size < num_bytes) {
			crypt->buf_size = MAX (num_bytes, 256);
			if (crypt->buf != NULL)
				g_free (crypt->buf);
			crypt->buf = g_new (guint8, crypt->buf_size);
		}
		buffer = crypt->buf;
	}

	/* Pass through when not protected. */
	if (!crypt->hash) {
		return gsf_input_read (crypt->source, num_bytes, buffer);
	}
	
	d(g_print ("read, %d bytes at %d\n", num_bytes, (int) input->cur_offset));

	/* Pad the start and end to read full 16 byte intervals starting at
	 * multiples of 16.
	 */
	pad = input->cur_offset % 16;
	real_offset = input->cur_offset - pad;
	real_num_bytes = num_bytes + pad;
	
	pad = real_num_bytes % 16;	
	if (pad > 0) {
		real_num_bytes += 16 - pad;
	}

	d(g_print ("read %d, %d (%d, %d)\n",
		 (int) input->cur_offset, (int) num_bytes,
		 (int) real_offset, (int) real_num_bytes));
	
	crypted_data = g_malloc (real_num_bytes);
	
	if (gsf_input_seek (crypt->source, real_offset, G_SEEK_SET)) {
		g_warning ("Could not seek when decrypting.");
	}
	
	/* First read the IV. */
	gsf_input_read (crypt->source, 16, xor);
	/*g_print ("IV: [%s]\n", g_strndup (xor, 16));*/

	/* Sanity check, don't try to read outside the data including
	 * padding.
	 */
	if (real_offset + real_num_bytes > gsf_input_size (input) + crypt->padding) {
		g_warning ("reading too much\n");
		return NULL;
	}

	gsf_input_read (crypt->source, real_num_bytes, crypted_data);
	/*hexdump (crypted_data);*/

	ob_utils_decrypt_data (&crypt->aes,
			       xor,
			       crypted_data,
			       real_num_bytes);
	
	memcpy (buffer, &crypted_data[input->cur_offset - real_offset], num_bytes);

	g_free (crypted_data);

	return buffer;
}

static gboolean
gsf_input_crypt_seek (GsfInput *input, gsf_off_t offset, GSeekType whence)
{
	GsfInputCrypt *crypt = GSF_INPUT_CRYPT (input);

	/* Just pass through when no password. */
	if (!crypt->hash) {
		return gsf_input_seek (crypt->source, offset, whence);
	}
	
	return FALSE;
}

static void
gsf_input_crypt_init (GObject *obj)
{
	GsfInputCrypt *crypt = GSF_INPUT_CRYPT (obj);

	crypt->source = NULL;
	crypt->buf = NULL;
	crypt->buf_size	= 0;
}

static void
gsf_input_crypt_class_init (GObjectClass *gobject_class)
{
	GsfInputClass *input_class = GSF_INPUT_CLASS (gobject_class);

	parent_class = g_type_class_peek_parent (gobject_class);

	gobject_class->finalize = gsf_input_crypt_finalize;

	input_class->Dup = gsf_input_crypt_dup;
	input_class->Read = gsf_input_crypt_read;
	input_class->Seek = gsf_input_crypt_seek;
}

GSF_CLASS (GsfInputCrypt, gsf_input_crypt,
	   gsf_input_crypt_class_init, gsf_input_crypt_init, GSF_INPUT_TYPE)


