/*
 * Copyright (C) 2007 Tuomas Kulve <tuomas@kulve.fi>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 * USA.
 */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <assert.h>            /* assert */
#include <errno.h>             /* errno */
#include <strings.h>           /* index */
#include <string.h>            /* strlen, strcmp, strerror */
#include <stdio.h>             /* fprintf */
#include <stdlib.h>            /* exit */
#include <stdint.h>            /* standard data types */
#include <unistd.h>            /* stat, execl */
#include <sys/types.h>         /* stat */
#include <sys/stat.h>          /* stat */

#include <glib.h>              /* GHashTable */

enum ACTION {
    ACTION_INVALID,
    ACTION_START,
    ACTION_STOP,
    ACTION_BOOTON,
    ACTION_BOOTOFF
};



static void print_usage(void);
static gchar *validate_service_name(const gchar *srv);
static enum ACTION parse_action(const gchar *srv);
static gboolean service_exists(const gchar *srv);
static gboolean execute(const gchar *srv, enum ACTION action);



int main(int argc, char *argv[])
{
    gchar *service = NULL;
    enum ACTION action = ACTION_INVALID;

    if (argc != 3) {
        print_usage();
    }

    /* validate action */
    action = parse_action(argv[2]);
    if (action == ACTION_INVALID) {
        fprintf(stderr, "ERROR: invalid action: %s\n", argv[2]);
        print_usage();
    }

    /* validate service name for invalid characters */
    service = validate_service_name(argv[1]);
    if (service == NULL) {
        fprintf(stderr, "ERROR: invalid service name: %s\n", argv[1]);
        print_usage();
    }

    /* check existance of the service file */
    if (service_exists(service) == FALSE) {
        fprintf(stderr, "ERROR: service not found: %s\n", service);
        g_free(service);
        exit(EXIT_FAILURE);
    }

    g_debug("parsed: action: %d, service: %s", action, service);

    if (setuid(0) == -1) {
        fprintf(stderr, "ERROR: failed setuid to ROOT: %s\n",
                strerror(errno));
        g_free(service);
        exit(EXIT_FAILURE);
    }

    if (execute(service, action) == FALSE) {
        g_free(service);
        exit(EXIT_FAILURE);
    }

    g_free(service);
    return 0;
}



static void print_usage(void)
{
    fprintf(stderr,
            "\n"
            "Usage: service_kicker <service> <action>\n"
            "\n"
            "Action can be: start, stop, booton, bootoff\n"
            "\n"
            "Start and stop actions start and stop service under\n"
            "/etc/init.d/. "
            "Booton and bootoff actions enables and disables\nstarting of the "
            "service during the boot up.\n"
            "\n");

    exit(EXIT_FAILURE);
}



static gchar *validate_service_name(const gchar *srv)
{
    gchar invalid_chars[] = {':', '%', '/', ';', '#'};
    gint i, j, srv_len;
    
    srv_len = strlen(srv);
    for(i = 0; i < srv_len; ++i) {
        for(j = 0; j < sizeof(invalid_chars); ++j) {
            if (srv[i] == invalid_chars[j]) {
                return NULL;
            }
        }
    }
    
    return g_strdup(srv);
}



static enum ACTION parse_action(const gchar *srv)
{
    if (strcmp(srv, "start") == 0) {
        return ACTION_START;
    } else if (strcmp(srv, "stop") == 0) {
        return ACTION_STOP;
    } else if (strcmp(srv, "booton") == 0) {
        return ACTION_BOOTON;
    } else if (strcmp(srv, "bootoff") == 0) {
        return ACTION_BOOTOFF;
    } else {
        return ACTION_INVALID;
    }
}



static gboolean service_exists(const gchar *srv)
{
    struct stat st;
    gchar *filename;

    filename = g_strdup_printf("/etc/init.d/%s", srv);

    if (stat(filename, &st) == -1) {
        fprintf(stderr, "ERROR: failed to stat %s: %s\n",
                filename, strerror(errno));
        g_free(filename);
        return FALSE;
    }
    
    g_free(filename);
    return TRUE;
}



static gboolean execute(const gchar *srv, enum ACTION action)
{
    int retval;
    
    switch (action) {
    case ACTION_START:
        retval = execl("/usr/sbin/invoke-rc.d", "/usr/sbin/invoke-rc.d",
                       srv, "start", NULL);
        break;
    case ACTION_STOP:
        retval = execl("/usr/sbin/invoke-rc.d", "/usr/sbin/invoke-rc.d",
                       srv, "stop", NULL);
        break;
    case ACTION_BOOTON:
        retval = execl("/usr/sbin/update-rc.d", "/usr/sbin/update-rc.d",
                       srv, "defaults", NULL);
        break;
    case ACTION_BOOTOFF:
        retval = execl("/usr/sbin/update-rc.d", "/usr/sbin/update-rc.d",
                       "-f", srv, "remove", NULL);
        break;
    case ACTION_INVALID:
        fprintf(stderr, "ERROR: invalid action in execute\n");
        return FALSE;
        break;
    }

    /* we are here only on error, since we just called exec() */
    
    fprintf(stderr, "ERROR: failed to exec %s (action %d): %s\n",
            srv, action, strerror(errno));
    return FALSE;
}



/* Emacs indentatation information
   Local Variables:
   indent-tabs-mode:nil
   tab-width:4
   c-file-style:"stroustrup"
   c-set-offset:4
   c-basic-offset:4
   End:
*/
