/*
 * This file is part of libgst0.10-dsp
 *
 * Copyright (C) 2006 Nokia Corporation. All rights reserved.
 *
 * Contact: Stefan Kost <stefan.kost@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <unistd.h>
#include <string.h>
#include <gst/gst.h>
#include <gst/audio/gstaudioclock.h>

#include "gstdsppcmsrc.h"

#define DSP_PCM_SID_UPDATE_FRAMESIZE 22

static const gchar *dsp_node_names[2] =
{
  "/dev/dsptask/pcm_rec",
  "/dev/dsptask/g711_enc"
};


static GstStaticPadTemplate dsppcmsrc_src_template =
    GST_STATIC_PAD_TEMPLATE ("src",
       GST_PAD_SRC,
       GST_PAD_ALWAYS,
       GST_STATIC_CAPS ( "audio/x-raw-int, "
           "endianness = (int) { " G_STRINGIFY (G_BYTE_ORDER) " }, "
           "signed = (boolean) TRUE, "
           "width = (int) 16, "
           "depth = (int) 16, "
           "rate = (int) 8000, "
           "channels = (int) 1;"

           "audio/x-alaw, "
           "rate = (int) 8000, "
           "channels = (int) 1; "

           "audio/x-mulaw, "
           "rate = (int) 8000, "
           "channels = (int) 1"
        )
     );

static GstStaticPadTemplate dsppcmsrc_cnsrc_template =
GST_STATIC_PAD_TEMPLATE ("cnsrc%d",
    GST_PAD_SRC,
    GST_PAD_REQUEST,
    GST_STATIC_CAPS ( "audio/CN, "
                      "rate = (int) 8000, "
                      "channels = (int) 1"
                    ) );

// Function prototypes

static void gst_dsppcmsrc_class_init (GstDSPPCMSrcClass * klass);
static void gst_dsppcmsrc_base_init (gpointer g_class);
static void gst_dsppcmsrc_init (GstDSPPCMSrc *dspmp3src, GstDSPPCMSrcClass *g_class);
static void gst_dsppcmsrc_set_property (GObject * object, guint prop_id, const GValue * value, GParamSpec * pspec);
static void gst_dsppcmsrc_get_property (GObject * object, guint prop_id, GValue * value, GParamSpec * pspec);

static GstClock *gst_dsppcmsrc_provide_clock (GstElement * elem);
static GstClockTime gst_dsppcmsrc_get_time (GstClock * clock, GstDSPPCMSrc * src);


static GstCaps *gst_dsppcmsrc_getcaps (GstBaseSrc * src);
static gboolean gst_dsppcmsrc_setcaps (GstBaseSrc * src, GstCaps *caps);
static GstFlowReturn gst_dsppcmsrc_create (GstPushSrc * psrc, GstBuffer ** buf);
static gboolean gst_dsppcmsrc_start (GstBaseSrc * bsrc);
static gboolean gst_dsppcmsrc_stop (GstBaseSrc * bsrc);
static gboolean gst_dsppcmsrc_event (GstBaseSrc *src, GstEvent *event);

static GstStateChangeReturn
gst_dsppcmsrc_change_state (GstElement * element, GstStateChange transition);

static GstPad * gst_dsppcmsrc_request_new_pad (GstElement     *element,
                                               GstPadTemplate *templ,
                                               const gchar    *name);

static void
gst_dsppcmsrc_release_pad (GstElement *element, GstPad *pad);


#ifdef DEBUG
static void _do_init()
{
  DBG_PRINT("PCM DO_INIT\n");
}

GST_BOILERPLATE_FULL (GstDSPPCMSrc, gst_dsppcmsrc, GstPushSrc,
                      GST_TYPE_PUSH_SRC, _do_init);
#else
GST_BOILERPLATE (GstDSPPCMSrc, gst_dsppcmsrc, GstPushSrc,
                 GST_TYPE_PUSH_SRC);
#endif


/**
 * gst_dsppcmsrc_dispose:
 * @object: GObject pointer to element to be deleted
 *
 * Deletes PCM src element instance. Called automatically by
 * GLib framework when element needs to be disposed.
 */
static void
gst_dsppcmsrc_dispose (GObject * object)
{
  GstDSPPCMSrc *dsp = (GstDSPPCMSrc *) object;

  if(dsp->clock) {
    gst_object_unref(dsp->clock);
    dsp->clock = NULL;
  }

  if(dsp->audio) {
    gst_dspaudio_destroy(dsp->audio);
    dsp->audio = NULL;
  }

  if (dsp->lastcn) {
    gst_buffer_unref (dsp->lastcn);
    dsp->lastcn = NULL;
  }

  G_OBJECT_CLASS (parent_class)->dispose (object);
}


/**
 * gst_dsppcmsrc_base_init:
 * @g_class: PCM src class
 *
 * Does the basic initialization of PCM src element class. This is
 * called by GStreamer framework.
 */
static void
gst_dsppcmsrc_base_init (gpointer g_class)
{
  GstElementClass *gstelement_class = GST_ELEMENT_CLASS (g_class);
  static const GstElementDetails details = GST_ELEMENT_DETAILS (
    "DSP PCM Src",
    "Source/Audio",
    "PCM audio src",
    "Makoto Sugano <makoto.sugano@nokia.com>");

  gst_element_class_add_pad_template (gstelement_class,
      gst_static_pad_template_get (&dsppcmsrc_src_template));
  gst_element_class_add_pad_template (gstelement_class,
      gst_static_pad_template_get (&dsppcmsrc_cnsrc_template));
  gst_element_class_set_details (gstelement_class, &details);

}

/**
 * gst_dsppcmsrc_class_init:
 * @klass: PCM Src class object
 *
 * Initializes PCM src element class. This is called by GStreamer
 * framework.
 */
static void
gst_dsppcmsrc_class_init (GstDSPPCMSrcClass * klass)
{
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;
  GstBaseSrcClass *gstbasesrc_class;
  GstPushSrcClass *gstpushsrc_class;

  gobject_class = (GObjectClass *) klass;
  gstelement_class = (GstElementClass *) klass;
  gstbasesrc_class = (GstBaseSrcClass *) klass;
  gstpushsrc_class = (GstPushSrcClass *) klass;

  gobject_class->set_property = gst_dsppcmsrc_set_property;
  gobject_class->get_property = gst_dsppcmsrc_get_property;
  gobject_class->dispose = gst_dsppcmsrc_dispose;

  gstelement_class->provide_clock = GST_DEBUG_FUNCPTR (gst_dsppcmsrc_provide_clock);

   // Use dspaudio utility to install default properties
  gst_dspaudio_install_properties(G_OBJECT_CLASS (klass));

  g_object_class_install_property (gobject_class, DSPPCM_PROP_DTX_MODE,
                                   g_param_spec_enum ("dtx", "DTX mode",
                                       "Discontinuous transmission mode (G.711 only)",
                                       GST_TYPE_DSPAUDIO_DTX_MODE, /* enum type */
                                       GST_DSPAUDIO_DTX_MODE_OFF,  /* default value */
                                       G_PARAM_READWRITE));

  gstelement_class->change_state = GST_DEBUG_FUNCPTR (gst_dsppcmsrc_change_state);
  gstelement_class->request_new_pad = GST_DEBUG_FUNCPTR(gst_dsppcmsrc_request_new_pad);
  gstelement_class->release_pad = GST_DEBUG_FUNCPTR(gst_dsppcmsrc_release_pad);

  gstbasesrc_class->start = GST_DEBUG_FUNCPTR (gst_dsppcmsrc_start);
  gstbasesrc_class->stop = GST_DEBUG_FUNCPTR (gst_dsppcmsrc_stop);
  gstbasesrc_class->get_caps = GST_DEBUG_FUNCPTR (gst_dsppcmsrc_getcaps);
  gstbasesrc_class->set_caps = GST_DEBUG_FUNCPTR (gst_dsppcmsrc_setcaps);
  gstbasesrc_class->event = GST_DEBUG_FUNCPTR (gst_dsppcmsrc_event);

  gstpushsrc_class->create = GST_DEBUG_FUNCPTR (gst_dsppcmsrc_create);
}


/**
 * gst_dsppcmsrc_init:
 * @dsp: DSP PCM src object
 *
 * Initializes the given PCM src element instance and allocates required
 * resources. This also creates an audio stream object. This is
 * called by GStreamer framework.
 */
static void
gst_dsppcmsrc_init (GstDSPPCMSrc * dsppcmsrc, GstDSPPCMSrcClass *g_class)
{
  dsppcmsrc->audio = gst_dspaudio_new();
  gst_base_src_set_live (GST_BASE_SRC (dsppcmsrc), TRUE);
  dsppcmsrc->dtxmode = GST_DSPAUDIO_DTX_MODE_OFF;
  GST_BASE_SRC (dsppcmsrc)->blocksize = 160;    // Default value

  dsppcmsrc->clock = gst_audio_clock_new ("clock",
                                          (GstAudioClockGetTimeFunc) gst_dsppcmsrc_get_time,
                                          dsppcmsrc);
  DBG_PRINT("PCM CLOCK CREATED\n");
}


/**
 *
 *
 */
static GstCaps *
gst_dsppcmsrc_getcaps (GstBaseSrc * src)
{
  DBG_PRINT("gst_dsppcmsrc_getcaps\n");

  GstCaps *caps = gst_caps_copy (
      gst_pad_get_pad_template_caps (GST_BASE_SRC_PAD(src)));

  DBG_PRINT("OUTPUT CAPS: %s\n", gst_caps_to_string(caps));
  return caps;
}


/**
 * gst_dsppcmsrc_round_bsize:
 *
 *
 * Rounds the blocksize to be dividable by 80 (that is 10ms in time)
 * and makes sure that it will be between certain limits
 */
static guint
gst_dsppcmsrc_round_bsize(guint bsize, guint min, guint max)
{
  guint multiplier = bsize / 80;
  guint retval = multiplier * 80;
  if(retval < min) return min;
  if(retval > max) return max;
  return retval;
}


/**
 * gst_dsppcmsink_setcaps:
 * @basesink: GstBaseSrc
 * @caps: GstCaps that were offered by the connecting element
 *
 * Returns: TRUE if caps can be accepted, otherwise FALSE
 */
static gboolean
gst_dsppcmsrc_setcaps(GstBaseSrc *bsrc, GstCaps * caps)
{
  DBG_PRINT("gst_dsppcmsrc_setcaps\n");

  GstDSPPCMSrc *dsp = GST_DSPPCMSRC (bsrc);
  SPEECH_PARAMS_DATA init_data;

  GstStructure *structure = gst_caps_get_structure (caps, 0);
  const gchar *mimetype = gst_structure_get_name (structure);

  if(!strncmp(mimetype, "audio/x-raw-int", 15)) {
    DBG_PRINT("linear\n");
    init_data.audio_fmt = DSP_AFMT_S16_LE;
    init_data.frame_size = gst_dsppcmsrc_round_bsize(bsrc->blocksize, 80, 800);
    dsp->framesize = init_data.frame_size * sizeof(short int);
    dsp->devindex = 0;
    dsp->raw = TRUE;
  }
  else if(!strncmp(mimetype, "audio/x-alaw", 12)) {
    DBG_PRINT("a-law\n");
    if(dsp->dtxmode == GST_DSPAUDIO_DTX_MODE_ON) {
      init_data.audio_fmt = DSP_AFMT_ALAW_DTX;
      init_data.frame_size = gst_dsppcmsrc_round_bsize(bsrc->blocksize, 160, 800);
      dsp->dtx_enabled = TRUE;
      DBG_PRINT("DTX ON\n");
    } else {
      init_data.audio_fmt = DSP_AFMT_ALAW;
      init_data.frame_size = gst_dsppcmsrc_round_bsize(bsrc->blocksize, 80, 800);
      dsp->dtx_enabled = FALSE;
      DBG_PRINT("DTX OFF\n");
    }
    dsp->framesize = init_data.frame_size;
    dsp->devindex = 1;
    dsp->raw = FALSE;
  }
  else if(!strncmp(mimetype, "audio/x-mulaw", 12)) {
    DBG_PRINT("mu-law\n");
    if(dsp->dtxmode == GST_DSPAUDIO_DTX_MODE_ON) {
      init_data.audio_fmt = DSP_AFMT_ULAW_DTX;
      init_data.frame_size = gst_dsppcmsrc_round_bsize(bsrc->blocksize, 160, 800);
      dsp->dtx_enabled = TRUE;
      DBG_PRINT("DTX ON\n");
    } else {
      init_data.audio_fmt = DSP_AFMT_ULAW;
      init_data.frame_size = gst_dsppcmsrc_round_bsize(bsrc->blocksize, 80, 800);
      dsp->dtx_enabled = FALSE;
      DBG_PRINT("DTX OFF\n");
    }
    dsp->framesize = init_data.frame_size;
    dsp->devindex = 1;
    dsp->raw = FALSE;
  }
  else {
    DBG_PRINT("Unknown format\n");
    return FALSE;
  }

  dsp->frametimemillis = init_data.frame_size / 8;

  DBG_PRINT("DSP FRAME SIZE: %d bytes (%d millisecs)\n",
            dsp->framesize, dsp->frametimemillis);

  init_data.dsp_cmd = DSP_CMD_SET_SPEECH_PARAMS;
  init_data.sample_rate = SAMPLE_RATE_8KHZ;
  init_data.ds_stream_ID = 0;
  init_data.stream_priority = dsp->audio->priority;

  if (gst_dspaudio_open(dsp->audio, dsp_node_names[dsp->devindex]) == FALSE) {
    gst_dspaudio_ed_microphone(dsp->audio, FALSE);
    GST_ELEMENT_ERROR (dsp, RESOURCE,
                       OPEN_READ_WRITE, (NULL),
                       ("gst_dspaudio_open"));
    return FALSE;
  }

  // Read the junk out...
  gst_dspaudio_flush(&dsp->audio->codec);

  if(!gst_dspaudio_setparams(dsp->audio, (char *)&init_data, sizeof(SPEECH_PARAMS_DATA))) {
    dsp->audio->mode = DSP_MODE_ERROR;
    return FALSE;
  }
  dsp->audio->mode = DSP_MODE_INITIALIZED;
  gst_dspaudio_play(dsp->audio);
  gst_dspaudio_update_dsp_settings(dsp->audio);

  return TRUE;
}


/**
 * gst_dsppcmsrc_create:
 * src:
 * @buffer:
 *
 */
static GstFlowReturn
gst_dsppcmsrc_create (GstPushSrc * psrc, GstBuffer ** buf)
{
  GstDSPPCMSrc *dsp = GST_DSPPCMSRC(psrc);

  if(dsp->audio->mode == DSP_MODE_ERROR) {
    GST_ELEMENT_ERROR (dsp, RESOURCE, READ, (NULL),
                       ("error cmd: %d status: %d",
                        dsp->audio->error_cmd,
                        dsp->audio->error_status));
    return GST_FLOW_ERROR;
  }

  if (dsp->raw == FALSE) {
    for (;;) {
      *buf = gst_dspaudio_read_frame(dsp->audio, dsp->framesize,
                                     DSP_PCM_SID_UPDATE_FRAMESIZE, TRUE);

      if (!*buf)
        return GST_FLOW_WRONG_STATE;

      GST_BUFFER_DURATION(*buf) = dsp->frametimemillis * GST_MSECOND;
      GST_BUFFER_TIMESTAMP(*buf) = dsp->framecount * dsp->frametimemillis * GST_MSECOND;
      dsp->framecount++;

      DBG_PRINT("G.711 src created a buffer\n");

        /* SID_FRAME_NO_UPDATE: try to resend old CN frame */
      if (GST_BUFFER_SIZE(*buf) == 0) {
        gst_buffer_unref(*buf);
        if (dsp->lastcn && dsp->cnpad) {
          gst_buffer_ref(dsp->lastcn);
          *buf = dsp->lastcn;
        } else {
          continue;
        }
      } else if (GST_BUFFER_FLAG_IS_SET(*buf, GST_BUFFER_FLAG_LAST)) {
        /* is SID_FRAME_UPDATE */
        if (!dsp->cnpad) {
          gst_buffer_unref(*buf);
          continue;
        }
        /* Save for future NO_UPDATE */
        if (dsp->lastcn)
          gst_buffer_unref(dsp->lastcn);

        gst_buffer_ref(*buf);
        dsp->lastcn = *buf;
        gst_buffer_set_caps (dsp->lastcn, GST_PAD_CAPS (dsp->cnpad));
      }

      if (GST_BUFFER_CAPS(*buf)) {
        /* CN Buffer */
        GST_BUFFER_FLAG_UNSET(*buf, GST_BUFFER_FLAG_LAST);
        gst_pad_push(dsp->cnpad, *buf);
        continue;
      } else {
      /* Regular buffer */
        gst_buffer_set_caps (*buf, GST_PAD_CAPS (GST_BASE_SRC_PAD (psrc)));
        return GST_FLOW_OK;
      }
    }
  }
  else {
    // We need to read the frame by ourselves, since raw-PCM mode does not
    // send DTX information
    short int tmp = DSP_CMD_DATA_READ;
    DSPWaitStatus status;
    guint recv_frames = 1;

    do {
        g_mutex_lock (dsp->audio->dsp_mutex);
        status = gst_dspaudio_wait_buffer (dsp->audio);
        if (status == DSP_WAIT_RETRY) {
            g_mutex_unlock (dsp->audio->dsp_mutex);
            continue;
        }
        if (status == DSP_WAIT_INTERRUPT) {
            g_mutex_unlock (dsp->audio->dsp_mutex);
            break;
        }
        else if (status == DSP_WAIT_ERROR) {
            g_mutex_unlock (dsp->audio->dsp_mutex);
            dsp->audio->mode = DSP_MODE_ERROR;
            break;
        }

        recv_frames = dsp->audio->readinfo.frame_size /
                        (dsp->framesize / sizeof (short int));

        *buf = gst_buffer_new_and_alloc (recv_frames * dsp->framesize);
        guint8 *data = GST_BUFFER_DATA (*buf);

        DBG_PRINT ("Received %d frames (framesize = %d short ints)",
                    recv_frames, dsp->audio->readinfo.frame_size);

        memcpy (data, dsp->audio->codec.mmap_buffer,
                dsp->audio->readinfo.frame_size * sizeof (short int));

        if (write (dsp->audio->codec.fd, &tmp, sizeof (short int)) < 0) {
            dsp->audio->mode = DSP_MODE_ERROR;
        }
        g_mutex_unlock (dsp->audio->dsp_mutex);
    }
    while (FALSE);

    if(*buf) {
      GST_BUFFER_DURATION(*buf) = recv_frames * dsp->frametimemillis * GST_MSECOND;
      GST_BUFFER_TIMESTAMP(*buf) = dsp->framecount * dsp->frametimemillis * GST_MSECOND;
      gst_buffer_set_caps (*buf, GST_PAD_CAPS (GST_BASE_SRC_PAD (psrc)));
      dsp->framecount += recv_frames;
      return GST_FLOW_OK;
    }
    return GST_FLOW_WRONG_STATE;
  }
}


/**
 *
 *
 */
static gboolean
gst_dsppcmsrc_start (GstBaseSrc * bsrc)
{
  DBG_PRINT("gst_dsppcmsrc_start\n");
  GstDSPPCMSrc *dsp = GST_DSPPCMSRC(bsrc);

  if (gst_dspaudio_ed_microphone(dsp->audio, TRUE) == FALSE) {
    GST_ELEMENT_ERROR (dsp, RESOURCE,
                       OPEN_WRITE, (NULL),
                       ("gst_dspaudio_ed_microphone"));
    return FALSE;
  }

  if (gst_dspaudio_aep_initialize(dsp->audio) == FALSE) {
    gst_dspaudio_ed_microphone(dsp->audio, FALSE);
    GST_ELEMENT_ERROR (dsp, RESOURCE,
                       OPEN_READ_WRITE, (NULL),
                       ("gst_dspaudio_aep_initialize"));
    return FALSE;
  }

  // This is used for clocking
  dsp->framecount = 0;
  dsp->lastcn = NULL;
  DBG_PRINT("gst_dsppcmsrc_start OK\n");
  return TRUE;
}



/**
 *
 *
 */
static gboolean
gst_dsppcmsrc_stop (GstBaseSrc * bsrc)
{
  DBG_PRINT("gst_dsppcmsrc_stop\n");
  GstDSPPCMSrc *dsp = GST_DSPPCMSRC (bsrc);

  gst_dspaudio_remove_interrupt(dsp->audio);

  g_mutex_lock(dsp->audio->dsp_mutex);

  if ( dsp->audio->all_frames_read ||
       dsp->audio->read_sent )
  {
    DBG_PRINT("Waiting reply for CMD_DATA_READ\n");
    gst_dspaudio_wait_buffer(dsp->audio);
    DBG_PRINT("Got it\n");
  }

  gst_dspaudio_dtmf_tone_stop(dsp->audio);

  if(gst_dspaudio_stop(dsp->audio)) {
    gst_dspaudio_read_cmd(dsp->audio, 500);
  }

  gst_dspaudio_ed_microphone(dsp->audio, FALSE);
  gst_dspaudio_close(dsp->audio);
  gst_dspaudio_aep_close(dsp->audio);
  gst_dspaudio_reset(dsp->audio);
  g_mutex_unlock(dsp->audio->dsp_mutex);

  DBG_PRINT("gst_dsppcmsrc_stop OK\n");
  return TRUE;
}


/**
 * gst_dsppcmsrc_set_property:
 * @object: GObject where the property should be retrieved
 * @prop_id: Unique identifier of the desired property
 * @value: Pointer to a variable where the value should be put
 * @pspec: Parameter specification
 *
 * Stores the property value into GObject
 */
static void
gst_dsppcmsrc_set_property(GObject * object,
                           guint prop_id,
                           const GValue * value,
                           GParamSpec * pspec)
{
  GstDSPPCMSrc *dsp = GST_DSPPCMSRC (object);

  if(prop_id == DSPPCM_PROP_DTX_MODE) {
    dsp->dtxmode = g_value_get_enum(value);
  }
  else if(!gst_dspaudio_set_property(dsp->audio, prop_id, value)) {
    g_warning("trying to set illegal property");
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
  }
}


/**
 * gst_dsppcmsrc_get_property:
 * @object: GObject where the property should be retrieved
 * @prop_id: Unique identifier of the desired property
 * @value: Pointer to a variable where the value should be put
 * @pspec: Parameter specification
 *
 * Retrieves the desired parameter value from object
 */
static void
gst_dsppcmsrc_get_property(GObject * object,
                           guint prop_id,
                           GValue * value,
                           GParamSpec * pspec)
{
  GstDSPPCMSrc *dsp = GST_DSPPCMSRC (object);

  if(prop_id == DSPPCM_PROP_DTX_MODE) {
    g_value_set_enum (value, dsp->dtxmode);
  }
  else if(!gst_dspaudio_get_property(dsp->audio, prop_id, value)) {
    g_warning("trying to get illegal property");
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
  }
}


/*
 * gst_dsppcmsrc_event:
 * @src: GstBaseSrc object
 * @event: GstEvent to be handled
 *
 * Handles incoming events. Checks if the event is special "custom upstream
 * event" and in this case forwards it to dspaudio library for checking if
 * it is DTMF event.
 *
 * Returns: TRUE if the event was handled.
 */
static gboolean
gst_dsppcmsrc_event(GstBaseSrc *src, GstEvent *event)
{
  GstEventType type = event ? GST_EVENT_TYPE (event) : GST_EVENT_UNKNOWN;
  GstDSPPCMSrc *dsp = GST_DSPPCMSRC (src);
  gboolean ret = FALSE;

  if(type == GST_EVENT_CUSTOM_UPSTREAM) {
    ret = gst_dspaudio_check_upstream_event (dsp->audio, event);
  }
  return ret;
}


/**
 *
 *
 */
static GstClock *
gst_dsppcmsrc_provide_clock (GstElement * elem)
{
  DBG_PRINT("gst_dsppcmsrc_provide_clock\n");
  GstDSPPCMSrc *dsp = GST_DSPPCMSRC (elem);
  return GST_CLOCK (gst_object_ref (GST_OBJECT (dsp->clock)));
}


/**
 *
 *
 */
static GstClockTime
gst_dsppcmsrc_get_time (GstClock * clock, GstDSPPCMSrc * src)
{
  GstClockTime result;
  result = src->framecount * src->frametimemillis * GST_MSECOND;
  return result;
}


/**
 *
 *
 */
static GstStateChangeReturn
gst_dsppcmsrc_change_state (GstElement * element, GstStateChange transition)
{
  GstDSPPCMSrc *dsp = GST_DSPPCMSRC (element);

  switch (transition) {
    case GST_STATE_CHANGE_PLAYING_TO_PAUSED:
      DBG_PRINT("PCM SRC PLAYING TO PAUSED\n");
      gst_dspaudio_interrupt_render(dsp->audio);
      break;

    default:
      break;
  }

  return GST_ELEMENT_CLASS (parent_class)->change_state (element, transition);
}


/*
 * gst_dsppcmsrc_request_new_pad:
 *
 */
static GstPad *
gst_dsppcmsrc_request_new_pad (GstElement     *element,
                               GstPadTemplate *templ,
                               const gchar    *name)
{
  GstDSPPCMSrc *dsp = GST_DSPPCMSRC(element);

  DBG_PRINT ("request pad\n");

  if (dsp->cnpad)
    return NULL;

  dsp->cnpad = gst_pad_new_from_template (templ, name);
  gst_pad_set_activatepush_function (dsp->cnpad, gst_dspaudio_activate_push);
  gst_element_add_pad (element, dsp->cnpad);
  return dsp->cnpad;
}


/*
 * gst_dsppcmsrc_release_pad:
 *
 */
static void
gst_dsppcmsrc_release_pad (GstElement *element, GstPad *pad)
{
  GstDSPPCMSrc *dsp = GST_DSPPCMSRC(element);

  if (pad == dsp->cnpad)
    dsp->cnpad = NULL;
}
