/*
 * This file is part of libgst0.10-dsp
 *
 * Copyright (C) 2006 Nokia Corporation. All rights reserved.
 *
 * Contact: Stefan Kost <stefan.kost@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <gst/gst.h>
#include <gst/base/gstbasesink.h>

#include "gstdsppcmsink.h"


static const gchar *pcm_sink_device ="/dev/dsptask/pcm2";


static GstStaticPadTemplate sink_template =
    GST_STATIC_PAD_TEMPLATE ("sink",
        GST_PAD_SINK,
        GST_PAD_ALWAYS,
        GST_STATIC_CAPS ("audio/x-raw-int, "
            "endianness = (int) " G_STRINGIFY (G_BYTE_ORDER) ", "
            "signed = (boolean) { TRUE, FALSE }, "
            "width = (int) { 8, 16 }, "
            "depth = (int) { 8, 16 }, "
            "rate = (int) {  8000, 11025, 12000, "
                          " 16000, 22050, 24000, "
                          " 32000, 44100, 48000, "
                          "  64000, 88200, 96000 }, "
            "channels = (int) [ 1, 2 ]; "

            "audio/x-alaw, "
            "rate = (int) {  8000, 11025, 12000, "
                          " 16000, 22050, 24000, "
                          " 32000, 44100, 48000, "
                          " 64000, 88200, 96000 }, "
            "channels = (int) [ 1, 2 ]; "

            "audio/x-mulaw, "
            "rate = (int) {  8000, 11025, 12000, "
                          " 16000, 22050, 24000, "
                          " 32000, 44100, 48000, "
                          " 64000, 88200, 96000 }, "
            "channels = (int) [ 1, 2 ]"
         )
    );


static guint gst_dsppcmsink_signals[DSPPCM_SIGNAL_LAST] = { 0 };

// Function prototypes
static void gst_dsppcmsink_base_init (gpointer g_class);
static void gst_dsppcmsink_class_init (GstDSPPcmSinkClass * klass);
static void gst_dsppcmsink_init (GstDSPPcmSink * dsp, GstDSPPcmSinkClass *g_class);
static void gst_dsppcmsink_dispose (GObject * object);
static void gst_dsppcmsink_set_property (GObject * object, guint prop_id, const GValue * value, GParamSpec * pspec);
static void gst_dsppcmsink_get_property (GObject * object, guint prop_id, GValue * value, GParamSpec * pspec);

// For GstElement
// static gboolean gst_dsppcmsink_event (GstElement * element, GstEvent * event);
static GstStateChangeReturn gst_dsppcmsink_change_state (GstElement * element, GstStateChange transition);

// For GstBaseSink
static GstCaps *gst_dsppcmsink_getcaps (GstBaseSink *sink);
static gboolean gst_dsppcmsink_setcaps (GstBaseSink *sink, GstCaps *caps);
static GstFlowReturn gst_dsppcmsink_render(GstBaseSink *sink, GstBuffer * buffer);
static gboolean gst_dsppcmsink_event2 (GstBaseSink *sink, GstEvent *event);
static GstClock *gst_dsppcmsink_provide_clock (GstElement * elem);
static GstClockTime gst_dsppcmsink_get_time (GstClock * clock, gpointer sink);

#ifdef DEBUG
static void _do_init()
{
  DBG_PRINT("PCMSINK DO_INIT\n");
}

GST_BOILERPLATE_FULL (GstDSPPcmSink, gst_dsppcmsink, GstBaseSink,
                      GST_TYPE_BASE_SINK, _do_init);
#else
GST_BOILERPLATE (GstDSPPcmSink, gst_dsppcmsink, GstBaseSink,
                 GST_TYPE_BASE_SINK);
#endif


#define GST_TYPE_DSPPCMSINK_AEC_MODE (gst_dsppcmsink_aec_mode_get_type())
static GType
gst_dsppcmsink_aec_mode_get_type (void)
{
  static GType gst_dsppcmsink_aec_mode_type = 0;
  static GEnumValue dsppcmsink_aec_modes[] = {
    {GST_DSPPCMSINK_AEC_OFF, "Don't use AEC", "off"},
    {GST_DSPPCMSINK_AEC_ON, "Use AEC", "on"},
    {0, NULL, NULL},
  };

  if (G_UNLIKELY(!gst_dsppcmsink_aec_mode_type)) {
    gst_dsppcmsink_aec_mode_type = g_enum_register_static ("GstPCMSinkAECMode",
        dsppcmsink_aec_modes);
  }
  return gst_dsppcmsink_aec_mode_type;
}

/**
 * gst_dsppcmsink_dispose:
 * @object: GObject pointer to element to be deleted
 *
 * Deletes PCM sink element instance. Called automatically by
 * GLib framework when element needs to be disposed.
 */
static void
gst_dsppcmsink_dispose (GObject * object)
{
  DBG_PRINT("gst_dsppcmsink_dispose\n");
  GstDSPPcmSink *dsp = (GstDSPPcmSink *) object;

  if(dsp->clock) {
    gst_object_unref(dsp->clock);
    dsp->clock = NULL;
  }

  if(dsp->audio) {
    gst_dspaudio_destroy(dsp->audio);
    dsp->audio = NULL;
  }

  G_OBJECT_CLASS (parent_class)->dispose(object);
  DBG_PRINT("gst_dsppcmsink_dispose OK\n");
}


/**
 * gst_dsppcmsink_base_init:
 * @g_class: PCM sink class
 *
 * Does the basic initialization of PCM sink element class. This is
 * called by GStreamer framework.
 */
static void
gst_dsppcmsink_base_init (gpointer g_class)
{
  GstElementClass *gstelement_class = GST_ELEMENT_CLASS (g_class);
  static const GstElementDetails details = GST_ELEMENT_DETAILS (
    "DSP PCM Sink",
    "Sink/Audio",
    "Raw DSP PCM audio sink",
    "Makoto Sugano <makoto.sugano@nokia.com>");

  gst_element_class_add_pad_template (gstelement_class,
      gst_static_pad_template_get (&sink_template));
  gst_element_class_set_details (gstelement_class, &details);

}


/**
 * gst_dsppcmsink_class_init:
 * @klass: PCM Sink class object
 *
 * Initializes PCM sink element class. This is called by GStreamer
 * framework.
 */
static void
gst_dsppcmsink_class_init (GstDSPPcmSinkClass * klass)
{
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;
  GstBaseSinkClass *gstbasesink_class;

  gobject_class = (GObjectClass *) klass;
  gstelement_class = (GstElementClass *) klass;
  gstbasesink_class = (GstBaseSinkClass *) klass;

  gobject_class->dispose      = GST_DEBUG_FUNCPTR(gst_dsppcmsink_dispose);
  gobject_class->set_property = GST_DEBUG_FUNCPTR(gst_dsppcmsink_set_property);
  gobject_class->get_property = GST_DEBUG_FUNCPTR(gst_dsppcmsink_get_property);

  gstelement_class->provide_clock = GST_DEBUG_FUNCPTR (gst_dsppcmsink_provide_clock);
  gstelement_class->change_state = GST_DEBUG_FUNCPTR(gst_dsppcmsink_change_state);
//   gstelement_class->send_event = GST_DEBUG_FUNCPTR(gst_dsppcmsink_event);

  // Use dspaudio utility to install default properties
  gst_dspaudio_install_properties(G_OBJECT_CLASS (klass));

  g_object_class_install_property (gobject_class, DSPPCMSINK_PROP_AEC_MODE,
                                   g_param_spec_enum ("aec_mode", "AEC Mode",
                                       "Automatic echo cancellation mode.",
                                       GST_TYPE_DSPPCMSINK_AEC_MODE, /* enum type */
                                       GST_DSPPCMSINK_AEC_OFF,       /* default value */
                                       G_PARAM_READWRITE));

  gst_dsppcmsink_signals[DSPPCM_SIGNAL_STREAMID] =
      g_signal_new ("stream_id", G_TYPE_FROM_CLASS (klass), G_SIGNAL_RUN_FIRST,
                    G_STRUCT_OFFSET (GstDSPPcmSinkClass, stream_id), NULL, NULL,
                    g_cclosure_marshal_VOID__INT, G_TYPE_NONE, 1, G_TYPE_INT);

  gstbasesink_class->render = GST_DEBUG_FUNCPTR (gst_dsppcmsink_render);
  gstbasesink_class->set_caps = GST_DEBUG_FUNCPTR (gst_dsppcmsink_setcaps);
  gstbasesink_class->get_caps = GST_DEBUG_FUNCPTR(gst_dsppcmsink_getcaps);
  gstbasesink_class->event = GST_DEBUG_FUNCPTR (gst_dsppcmsink_event2);
}


/**
 * gst_dsppcmsink_init:
 * @dsp: DSP PCM sink object
 *
 * Initializes the given PCM sink element instance and allocates required
 * resources. This also creates an audio stream object. This is
 * called by GStreamer framework.
 */
static void
gst_dsppcmsink_init (GstDSPPcmSink * dsp, GstDSPPcmSinkClass *g_class)
{
  // This is why we need dispose()
  dsp->audio = gst_dspaudio_new();
  gst_base_sink_set_sync (GST_BASE_SINK (dsp), FALSE);

  dsp->clock = gst_audio_clock_new ("clock",
                                    (GstAudioClockGetTimeFunc) gst_dsppcmsink_get_time,
                                    dsp);
}


/**
 * gst_dsppcmsink_format_get:
 * @law: law mode. 0=RAW, 1=u-Law, 2=a-Law
 * @bigend: is the stream big endian?
 * @sign: sign mode. FALSE=unsigned, TRUE=signed
 * @width: Width of the samples. Should be 8 or 16
 * @depth: Depth of samples. Must be same as width
 * @format: Pointer to a variable where format value should be placed
 * @bps: Pointer to a variable where to put bytes-per-sample value
 *
 * Translates various audio-related information into one format definition
 *
 * Returns: TRUE if operation was successful, otherwise FALSE
 */
static gboolean
gst_dsppcmsink_format_get (gint law, gint endianness, gboolean sign, gint width,
    gint depth, guint * format, gint * bps)
{
//   if (width != depth) {
//     DBG_PRINT("WIDTH (%d) != DEPTH (%d)\n", width, depth);
//     return FALSE;
//   }

  *bps = 1;

  if (law == 0) {
    if (width == 16) {
      if (sign == TRUE) {
        if (endianness == G_BIG_ENDIAN) {
          *format = DSP_AFMT_S16_BE;
        } else {
          *format = DSP_AFMT_S16_LE;
        }
      }
      else {
        if (endianness == G_BIG_ENDIAN) {
          *format = DSP_AFMT_U16_BE;
        } else {
          *format = DSP_AFMT_U16_LE;
        }
      }
      *bps = 2;
    }
    else if (width == 8) {
      if (sign == TRUE) {
        *format = DSP_AFMT_S8;
      } else {
        *format = DSP_AFMT_U8;
      }
      *bps = 1;
    }
    else {
      DBG_PRINT("unknown sample width\n");
      return FALSE;
    }
  }
  else if (law == 1) {
    *format = DSP_AFMT_ULAW;
  }
  else if (law == 2) {
    *format = DSP_AFMT_ALAW;
  }
  else {
    DBG_PRINT ("unknown law\n");
    return FALSE;
  }

  return TRUE;
}


/**
 *
 *
 */
static GstCaps *
gst_dsppcmsink_getcaps (GstBaseSink *sink)
{
  return gst_caps_copy (
    gst_pad_get_pad_template_caps (GST_BASE_SINK_PAD(sink)));
}


/**
 *
 *
 */
static gboolean
gst_dsppcmsink_initialize(GstDSPPcmSink *dsp)
{
  // Read the junk out...
  gst_dspaudio_flush(&dsp->audio->codec);

  if(dsp->aec_mode == GST_DSPPCMSINK_AEC_OFF) {

    DBG_PRINT("AEC OFF\n");

    AUDIO_PARAMS_DATA init_data;
    init_data.dsp_cmd = DSP_CMD_SET_PARAMS;
    init_data.audio_fmt = dsp->format;
    init_data.sample_rate = dsp->rate_fmt;
    init_data.number_channels = dsp->channels_fmt;
    init_data.stream_priority = dsp->audio->priority;
    init_data.ds_stream_ID = 0;

    DBG_PRINT("Init data: fmt=%d, rate=%d (%d), channels=%d\n",
              dsp->format, dsp->rate_fmt, dsp->rate,
              dsp->channels);

    if(!gst_dspaudio_setparams(dsp->audio, (char *)&init_data, sizeof(AUDIO_PARAMS_DATA))) {
      DBG_PRINT("SET_PARAMS error\n");
      GST_ELEMENT_ERROR (dsp, RESOURCE, SETTINGS, (NULL),
                         ("gst_dspaudio_setparams"));
      return FALSE;
    }

    // this tells max. amount of bytes we want to put in mmap buffer
    dsp->buf_limit = dsp->bps / 20;        // 50 millisecs per buffer
    dsp->buf_limit = dsp->buf_limit &~ 1;  // make it even
  }
  else {
    DBG_PRINT("AEC ON\n");

    SPEECH_PARAMS_DATA init_data;
    init_data.dsp_cmd = DSP_CMD_SET_SPEECH_PARAMS;
    init_data.audio_fmt = dsp->format;
    init_data.sample_rate = dsp->rate_fmt;
    init_data.stream_priority = dsp->audio->priority;
    init_data.frame_size = 80;
    init_data.ds_stream_ID = 0;

    if(!gst_dspaudio_setparams(dsp->audio, (char *)&init_data, sizeof(SPEECH_PARAMS_DATA))) {
      DBG_PRINT("SET_PARAMS error\n");
      GST_ELEMENT_ERROR (dsp, RESOURCE, SETTINGS, (NULL),
                         ("gst_dspaudio_setparams"));
      return FALSE;
    }
  }


  DBG_PRINT("buf_limit: %d bytes\n", dsp->buf_limit);
  DBG_PRINT("DSP PCM configured\n");

  dsp->audio->mode = DSP_MODE_CONFIGURED;
  // DSP throws an error if volume is not set before PLAY
  dsp->audio->volume_changed = TRUE;
  return TRUE;
}


/**
 *
 *
 */
static gboolean
gst_dsppcmsink_setcaps (GstBaseSink *src, GstCaps *caps)
{
  DBG_PRINT("gst_dsppcmsink_setcaps\n");
  GstDSPPcmSink *dsp = GST_DSPPCMSINK(src);
  gint bps;

  DBG_PRINT("CAPS: %s\n", gst_caps_to_string(caps));
  GstStructure *structure = gst_caps_get_structure (caps, 0);
  const gchar *mimetype = gst_structure_get_name (structure);

  if (!strncmp (mimetype, "audio/x-raw-int", 15)) {
    gst_structure_get_int (structure, "width", &dsp->width);
    gst_structure_get_int (structure, "depth", &dsp->depth);
    gst_structure_get_int (structure, "endianness", &dsp->endianness);
    gst_structure_get_boolean (structure, "signed", &dsp->sign);

    if (dsp->width != dsp->depth)
      return FALSE;

    dsp->buf_limit = 160; // 10 milliseconds
    dsp->law = 0;
  }
  else if(!strncmp (mimetype, "audio/x-mulaw", 13)) {
    dsp->buf_limit = 80; // 10 milliseconds
    dsp->law = 1;
  }
  else if(!strncmp (mimetype, "audio/x-alaw", 12)) {
    dsp->buf_limit = 80; // 10 milliseconds
    dsp->law = 2;
  }
  else {
    g_warning("Unknown mimetype");
    return FALSE;
  }

  if (!gst_dsppcmsink_format_get (dsp->law, dsp->endianness, dsp->sign, dsp->width,
       dsp->depth, &dsp->format, &bps))
  {
    g_warning("Cannot determine audio format");
    return FALSE;
  }

  // These are common for all formats
  gst_structure_get_int (structure, "channels", &dsp->channels);
  gst_structure_get_int (structure, "rate", &dsp->rate);

  dsp->channels_fmt = dsp->channels == 2 ? CHANNELS_2 : CHANNELS_1;

  if(gst_dspaudio_map_samplerate(dsp->rate, &dsp->rate_fmt) == FALSE) {
    g_warning("Not supported sample rate : %d!\n", dsp->rate);
    return FALSE;
  }

  dsp->bps = bps * dsp->channels * dsp->rate;

  if(gst_dsppcmsink_initialize(dsp) == FALSE) {
    g_mutex_unlock(dsp->audio->dsp_mutex);
    GST_ELEMENT_ERROR (dsp, RESOURCE, SETTINGS,
                       (NULL), ("gst_dsppcmsink_initialize"));
    return FALSE;
  }

  return TRUE;
}


/**
 *
 *
 */
static GstFlowReturn
gst_dsppcmsink_render (GstBaseSink *base, GstBuffer * buffer)
{
  DSPWaitStatus status = DSP_WAIT_ERROR;
  GstDSPPcmSink *dsp = GST_DSPPCMSINK (base);
  guint8 *data = (guint8 *) GST_BUFFER_DATA(buffer);
  guint length = GST_BUFFER_SIZE(buffer);
  guint dataIndex = 0, tw, tw1;

  if(dsp->audio->mode == DSP_MODE_CONFIGURED) {
    g_mutex_lock(dsp->audio->dsp_mutex);
    gst_dspaudio_flush(&dsp->audio->codec);
    gst_dspaudio_play(dsp->audio);
    gst_dspaudio_update_dsp_settings(dsp->audio);
    g_mutex_unlock(dsp->audio->dsp_mutex);
  }

  while ( length > 0 ) {

    if(dsp->audio->mode == DSP_MODE_ERROR) {
      DBG_PRINT("ERROR! Leaving...\n");

      GST_ELEMENT_ERROR (dsp, RESOURCE, WRITE,
                         ("dsp error cmd: %d, status: %d",
                          dsp->audio->error_cmd,
                          dsp->audio->error_status), GST_ERROR_SYSTEM);

      return GST_FLOW_ERROR;
    }

    g_mutex_lock(dsp->audio->dsp_mutex);
    status = gst_dspaudio_wait_buffer(dsp->audio);

    if(status == DSP_WAIT_RETRY) {
      g_mutex_unlock(dsp->audio->dsp_mutex);
      DBG_PRINT("PCM RENDER DSP_WAIT_RETRY\n");
      continue;
    }
    else if(status == DSP_WAIT_INTERRUPT) {
      // FIXME: store the buffer
      g_mutex_unlock(dsp->audio->dsp_mutex);
      DBG_PRINT("PCM RENDER DSP_WAIT_INTERRUPT\n");
      break;
    }
    else if(status == DSP_WAIT_OK) {
      dsp->audio->rw_pending = 1;
      tw1 = MIN((dsp->audio->codec.mmap_buffer_size - dsp->outIndex), length);
      tw = MIN(dsp->buf_limit, tw1);
      memcpy(dsp->audio->codec.mmap_buffer + dsp->outIndex, data+dataIndex, tw);
      DBG_PRINT("Copied %d bytes of data\n", tw);
      dsp->outIndex += tw;
      dataIndex += tw;
      length -= tw;
    }
    else {
      DBG_PRINT("WAIT_BUFFER ERROR\n");
      dsp->audio->mode = DSP_MODE_ERROR;
      break;
    }

    if(dsp->outIndex >= dsp->buf_limit) {
      dsp->outbuf.data_size = dsp->outIndex / sizeof(short int);
      if(write(dsp->audio->codec.fd, &dsp->outbuf, sizeof(PCM_DATA_WRITE)) < 0) {
        dsp->audio->mode = DSP_MODE_ERROR;
      }
//       DBG_PRINT("PCM WRITE %d bytes to DSP\n", dsp->outIndex);
      dsp->audio->discont_sent = FALSE;
      dsp->audio->rw_pending = 0;
      dsp->outIndex = 0;
    }

    g_mutex_unlock(dsp->audio->dsp_mutex);
  }
  return GST_FLOW_OK;
}


/**
 * gst_dsppcmsink_set_property:
 * @object: GObject where the property should be retrieved
 * @prop_id: Unique identifier of the desired property
 * @value: Pointer to a variable where the value should be put
 * @pspec: Parameter specification
 *
 * Stores the property value into GObject
 */
static void
gst_dsppcmsink_set_property (GObject * object, guint prop_id, const GValue * value,
    GParamSpec * pspec)
{
  GstDSPPcmSink *dsp = GST_DSPPCMSINK (object);

  if(prop_id == DSPPCMSINK_PROP_AEC_MODE) {
    dsp->aec_mode = g_value_get_enum(value);
  }
  else if(!gst_dspaudio_set_property(dsp->audio, prop_id, value)) {
    g_warning("trying to set illegal property");
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
  }
}


/**
 * gst_dsppcmsink_get_property:
 * @object: GObject where the property should be retrieved
 * @prop_id: Unique identifier of the desired property
 * @value: Pointer to a variable where the value should be put
 * @pspec: Parameter specification
 *
 * Retrieves the desired parameter value from object
 */
static void
gst_dsppcmsink_get_property (GObject * object, guint prop_id, GValue * value,
    GParamSpec * pspec)
{
  GstDSPPcmSink *dsp = GST_DSPPCMSINK (object);

  if(prop_id == DSPPCMSINK_PROP_AEC_MODE) {
    g_value_set_enum (value, dsp->aec_mode);
  }
  else if(!gst_dspaudio_get_property(dsp->audio, prop_id, value)) {
    g_warning("trying to get illegal property");
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
  }
}


/**
 *
 *
 */
static gboolean
gst_dsppcmsink_event2(GstBaseSink *sink, GstEvent *event)
{
  DBG_PRINT("PCM SINK EVENT\n");
  GstDSPPcmSink *dsp = GST_DSPPCMSINK (sink);

  GstEventType type = event ? GST_EVENT_TYPE (event) : GST_EVENT_UNKNOWN;

  if(type == GST_EVENT_EOS) {
    DSPWaitStatus status;

    DBG_PRINT("EOS EVENT\n");
    dsp->audio->codec.stream_id = 0;
    g_signal_emit (G_OBJECT (sink), gst_dsppcmsink_signals[DSPPCM_SIGNAL_STREAMID], 0, 0);

    // Write the rest of the buffer to DSP
    if(dsp->audio->mode == DSP_MODE_PLAYING) {
      short int tmp[2];
      tmp[0] = DSP_CMD_EOF;
      tmp[1] = dsp->outIndex / sizeof(short int);
      DBG_PRINT("WAITING DATA_WRITE in handle_event()\n");
      do {
        g_mutex_lock(dsp->audio->dsp_mutex);
        status = gst_dspaudio_wait_buffer(dsp->audio);
        g_mutex_unlock(dsp->audio->dsp_mutex);
      }
      while (status == DSP_WAIT_RETRY);

      if(status == DSP_WAIT_OK) {
        DBG_PRINT("GOT THE DATA_WRITE\n");
        g_mutex_lock(dsp->audio->dsp_mutex);
        write(dsp->audio->codec.fd, tmp, 2 * sizeof(short int));
        DBG_PRINT("WROTE DSP_CMD_EOF to DSP (%d short ints)\n", tmp[1]);
        g_mutex_unlock(dsp->audio->dsp_mutex);
        dsp->audio->mode = DSP_MODE_EOS;
      }
    }
  }
  if(type == GST_EVENT_NEWSEGMENT)
  {
    DBG_PRINT("NEWSEGMENT EVENT\n");
    g_mutex_lock(dsp->audio->dsp_mutex);

    if(gst_dspaudio_discont(dsp->audio)) {
      DBG_PRINT("DISCONT written... reading reply\n");
      if(gst_dspaudio_read_cmd(dsp->audio, 0) != DSP_WAIT_OK) {
        GST_ELEMENT_ERROR (dsp, RESOURCE, SEEK, (NULL), ("discont"));
      }
      DBG_PRINT("GOT REPLY\n");
      dsp->outIndex = 0;
    }
    g_mutex_unlock(dsp->audio->dsp_mutex);
  }

  return TRUE;
}

/**
 * gst_dsppcmsink_change_state:
 * @element: The GstElement whose state to be changed
 *
 * Changes the state of given element. The desired state information
 * is contained by the element itself, and this method is just
 * a callback, so the plugin knows when the state is changing.
 * This method is called by GStreamer framework
 *
 * Returns: TRUE if operation was successful, otherwise FALSE
 */
static GstStateChangeReturn
gst_dsppcmsink_change_state(GstElement * element, GstStateChange transition)
{
  GstDSPPcmSink *dsp = GST_DSPPCMSINK (element);
  GstStateChangeReturn ret = GST_STATE_CHANGE_SUCCESS;

  DBG_PRINT("PCM2 TRANSITION: %d\n", transition);

  switch (transition) {

    case GST_STATE_CHANGE_NULL_TO_READY:
      DBG_PRINT("PCM NULL TO READY\n");
      if(gst_dspaudio_open(dsp->audio, pcm_sink_device) == FALSE) {
        GST_ELEMENT_ERROR (dsp, RESOURCE, OPEN_WRITE,
                           (NULL), ("gst_dspaudio_open"));
        return GST_STATE_CHANGE_FAILURE;
      }
      g_signal_emit (G_OBJECT (dsp), gst_dsppcmsink_signals[DSPPCM_SIGNAL_STREAMID],
                     0, dsp->audio->codec.stream_id);
      dsp->outbuf.dsp_cmd = DSP_CMD_DATA_WRITE;
      dsp->outIndex = 0;
      break;

    case GST_STATE_CHANGE_READY_TO_PAUSED:
      DBG_PRINT("PCM READY TO PAUSED\n");
      ret = GST_STATE_CHANGE_NO_PREROLL;
      break;

    case GST_STATE_CHANGE_PAUSED_TO_PLAYING:
      DBG_PRINT("PCM PAUSED TO PLAYING\n");
      // Check if the DSP is initialized already
      if(dsp->audio->mode == DSP_MODE_PAUSED) {
        g_mutex_lock(dsp->audio->dsp_mutex);
        gst_dspaudio_play(dsp->audio);
        g_mutex_unlock(dsp->audio->dsp_mutex);
      }
      DBG_PRINT("PCM PAUSED TO PLAYING OK\n");
      break;

    case GST_STATE_CHANGE_PLAYING_TO_PAUSED:
      DBG_PRINT("PCM PLAYING TO PAUSED V1\n");
      gst_dspaudio_interrupt_render(dsp->audio);

    default:
      break;
  }

  gst_dspaudio_clock_change_state(dsp->audio, transition);
  ret = GST_ELEMENT_CLASS (parent_class)->change_state (element, transition);
  DBG_PRINT("PCM PARENT CLASS STATE CHANGE: %d\n", ret);

  switch (transition) {

    case GST_STATE_CHANGE_PLAYING_TO_PAUSED:
      gst_dspaudio_remove_interrupt(dsp->audio);
      g_mutex_lock(dsp->audio->dsp_mutex);
      DBG_PRINT("PCM PLAYING TO PAUSED V2\n");
      if(dsp->audio->mode == DSP_MODE_EOS) {
        gst_dspaudio_wait_eos(dsp->audio);
        DBG_PRINT("GOT EOS RESPONSE\n");
      }
      else {
        if(gst_dspaudio_pause(dsp->audio)) {
          gst_dspaudio_read_cmd(dsp->audio, 0);
          dsp->audio->rw_pending = 0;
        }
      }
      DBG_PRINT("PCM PLAYING TO PAUSED V2 OK\n");
      g_mutex_unlock(dsp->audio->dsp_mutex);
      break;

    case GST_STATE_CHANGE_PAUSED_TO_READY:
      DBG_PRINT("PCM PAUSED TO READY\n");
      g_mutex_lock(dsp->audio->dsp_mutex);
      if(gst_dspaudio_stop(dsp->audio)) {
        gst_dspaudio_read_cmd(dsp->audio, 0);
      }
      g_mutex_unlock(dsp->audio->dsp_mutex);
      DBG_PRINT("PCM PAUSED TO READY OK\n");
      break;

    case GST_STATE_CHANGE_READY_TO_NULL:
      DBG_PRINT("PCM READY To NULL\n");
      gst_dspaudio_close(dsp->audio);
      gst_dspaudio_reset(dsp->audio);
      break;

    default:
      break;
  }

  DBG_PRINT("PCM STATE CHANGE SUCCESSFUL\n");
  return ret;
}


/**
 *
 *
 */
static GstClock *
gst_dsppcmsink_provide_clock (GstElement * elem)
{
  DBG_PRINT("gst_dsppcmsink_provide_clock\n");
  GstDSPPcmSink *dsp = GST_DSPPCMSINK (elem);
  return GST_CLOCK (gst_object_ref (GST_OBJECT (dsp->clock)));
}


/**
 *
 *
 */
static GstClockTime
gst_dsppcmsink_get_time (GstClock * clock, gpointer sink)
{
  DBG_PRINT("gst_dsppcmsink_get_time\n");
  GstDSPPcmSink *dsp = GST_DSPPCMSINK (sink);
  return gst_dspaudio_get_dsp_clock(dsp->audio);
}
