/*
 * This file is part of libgst0.10-dsp
 *
 * Copyright (C) 2006 Nokia Corporation. All rights reserved.
 *
 * Contact: Stefan Kost <stefan.kost@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <unistd.h>
#include <string.h>
#include <gst/gst.h>
#include <gst/audio/gstaudioclock.h>

#include "gstdspg729src.h"

static gchar *devname = "/dev/dsptask/g729_enc";

static GstStaticPadTemplate dspg729src_src_template =
GST_STATIC_PAD_TEMPLATE ("src",
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS ( "audio/g729, "
                      "rate = (int) 8000, "
                      "channels = (int) 1"
                    ) );

// Function prototypes

static void gst_dspg729src_class_init (GstDSPG729SrcClass * klass);
static void gst_dspg729src_base_init (gpointer g_class);
static void gst_dspg729src_init (GstDSPG729Src *dspmp3src, GstDSPG729SrcClass *g_class);
static void gst_dspg729src_set_property (GObject * object, guint prop_id, const GValue * value, GParamSpec * pspec);
static void gst_dspg729src_get_property (GObject * object, guint prop_id, GValue * value, GParamSpec * pspec);

static GstCaps *gst_dspg729src_getcaps (GstBaseSrc * src);
static GstFlowReturn gst_dspg729src_create (GstPushSrc * psrc, GstBuffer ** buf);
static gboolean gst_dspg729src_start (GstBaseSrc * bsrc);
static gboolean gst_dspg729src_stop (GstBaseSrc * bsrc);
static GstStateChangeReturn gst_dspg729src_change_state (GstElement * element, GstStateChange transition);
static gboolean gst_dspg729src_event (GstBaseSrc *src, GstEvent *event);
static GstClock *gst_dspg729src_provide_clock (GstElement * elem);
static GstClockTime gst_dspg729src_get_time (GstClock * clock, GstDSPG729Src * src);
static GstBuffer *gst_dspg729src_voice_frame_handler (short int *mmap_ptr, guint fsize);
static GstBuffer *gst_dspg729src_sid_frame_handler (short int *mmap_ptr, guint fsize);

#define G729_FRAMETIMEMILLIS (10)
#define DSP_G729_SID_UPDATE_FRAMESIZE (8)

#ifdef DEBUG
static void _g729src_do_init()
{
  DBG_PRINT("G729 DO_INIT\n");
}

GST_BOILERPLATE_FULL (GstDSPG729Src, gst_dspg729src, GstPushSrc,
                      GST_TYPE_PUSH_SRC, _g729src_do_init);
#else
GST_BOILERPLATE (GstDSPG729Src, gst_dspg729src, GstPushSrc,
                 GST_TYPE_PUSH_SRC);
#endif


/*
 * gst_dspg729src_dispose:
 * @object: GObject pointer to element to be deleted
 *
 * Deletes G729 src element instance. Called automatically by
 * GLib framework when element needs to be disposed.
 */
static void
gst_dspg729src_dispose (GObject * object)
{
  GstDSPG729Src *dsp = (GstDSPG729Src *) object;

  if(dsp->clock) {
    gst_object_unref(dsp->clock);
    dsp->clock = NULL;
  }

  if(dsp->audio) {
    gst_dspaudio_destroy(dsp->audio);
    dsp->audio = NULL;
  }

  if (dsp->lastcn) {
    gst_buffer_unref (dsp->lastcn);
    dsp->lastcn = NULL;
  }

  G_OBJECT_CLASS (parent_class)->dispose (object);
}


/*
 * gst_dspg729src_base_init:
 * @g_class: G729 src class
 *
 * Does the basic initialization of G729 src element class. This is
 * called by GStreamer framework.
 */
static void
gst_dspg729src_base_init (gpointer g_class)
{
  GstElementClass *gstelement_class = GST_ELEMENT_CLASS (g_class);
  static const GstElementDetails details = GST_ELEMENT_DETAILS (
    "DSP G729 Src",
    "Source/Audio",
    "G.729 audio src",
    "Makoto Sugano <makoto.sugano@nokia.com>");

  gst_element_class_add_pad_template (gstelement_class,
      gst_static_pad_template_get (&dspg729src_src_template));
  gst_element_class_set_details (gstelement_class, &details);
}


/*
 * gst_dspg729src_class_init:
 * @klass: G729 Src class object
 *
 * Initializes G729 src element class. This is called by GStreamer
 * framework.
 */
static void
gst_dspg729src_class_init (GstDSPG729SrcClass * klass)
{
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;
  GstBaseSrcClass *gstbasesrc_class;
  GstPushSrcClass *gstpushsrc_class;

  gobject_class = (GObjectClass *) klass;
  gstelement_class = (GstElementClass *) klass;
  gstbasesrc_class = (GstBaseSrcClass *) klass;
  gstpushsrc_class = (GstPushSrcClass *) klass;

  gobject_class->set_property = GST_DEBUG_FUNCPTR (gst_dspg729src_set_property);
  gobject_class->get_property = GST_DEBUG_FUNCPTR (gst_dspg729src_get_property);
  gobject_class->dispose = GST_DEBUG_FUNCPTR (gst_dspg729src_dispose);

  // Use dspaudio utility to install default properties
  gst_dspaudio_install_properties (G_OBJECT_CLASS (klass));

  g_object_class_install_property (gobject_class, DSPG729_PROP_DTX_MODE,
                                   g_param_spec_enum ("dtx", "DTX mode",
                                       "Discontinuous transmission mode",
                                       GST_TYPE_DSPAUDIO_DTX_MODE,   /* enum type */
                                       GST_DSPAUDIO_DTX_MODE_ON,     /* default value */
                                       G_PARAM_READWRITE));

  gstelement_class->change_state = GST_DEBUG_FUNCPTR (gst_dspg729src_change_state);
  gstelement_class->provide_clock = GST_DEBUG_FUNCPTR (gst_dspg729src_provide_clock);

  gstbasesrc_class->start = GST_DEBUG_FUNCPTR (gst_dspg729src_start);
  gstbasesrc_class->stop = GST_DEBUG_FUNCPTR (gst_dspg729src_stop);
  gstbasesrc_class->get_caps = GST_DEBUG_FUNCPTR (gst_dspg729src_getcaps);
  gstbasesrc_class->event = GST_DEBUG_FUNCPTR (gst_dspg729src_event);
  gstpushsrc_class->create = GST_DEBUG_FUNCPTR (gst_dspg729src_create);
}


/*
 * gst_dspg729src_init:
 * @dsp: DSP G729 src object
 * @g_class: G729 class object
 *
 * Initializes the given G729 src element instance and allocates required
 * resources. This also creates an audio stream object. This is
 * called by GStreamer framework.
 */
static void
gst_dspg729src_init (GstDSPG729Src * dspg729src, GstDSPG729SrcClass *g_class)
{
  dspg729src->audio = gst_dspaudio_new();
  gst_base_src_set_live (GST_BASE_SRC (dspg729src), TRUE);
  dspg729src->dtxmode = GST_DSPAUDIO_DTX_MODE_ON;

  // Override default frame handlers with our own
  dspg729src->audio->handle_voice_frame = gst_dspg729src_voice_frame_handler;
  dspg729src->audio->handle_sid_frame = gst_dspg729src_sid_frame_handler;

  dspg729src->clock = gst_audio_clock_new ("clock",
                                           (GstAudioClockGetTimeFunc) gst_dspg729src_get_time,
                                           dspg729src);
}


/*
 * gst_dspg729src_getcaps:
 * @src: GstBaseSrc
 *
 * Return the output caps of source pad
 */
static GstCaps *
gst_dspg729src_getcaps (GstBaseSrc * src)
{
  GstCaps *caps = gst_caps_copy (
      gst_pad_get_pad_template_caps (GST_BASE_SRC_PAD(src)));

  return caps;
}


/*
 * gst_dspg729src_create:
 * @psrc: GstPushSrc
 * @buffer: pointer where the newly created buffer should be stored
 *
 * Retrieves audio data from DSP and creates a GstBuffer of it.
 *
 * Returns: GstFlowReturn
 */
static GstFlowReturn
gst_dspg729src_create (GstPushSrc * psrc, GstBuffer ** buf)
{
  GstDSPG729Src *dsp = GST_DSPG729SRC(psrc);

  if(dsp->audio->mode == DSP_MODE_ERROR) {
    GST_ELEMENT_ERROR (dsp, RESOURCE, READ, (NULL),
                       ("error cmd: %d status: %d",
                        dsp->audio->error_cmd,
                        dsp->audio->error_status));
    return GST_FLOW_ERROR;
  }

  for (;;) {
    *buf = gst_dspaudio_read_frame(dsp->audio, dsp->framesize,
                                   DSP_G729_SID_UPDATE_FRAMESIZE, TRUE);

    if(*buf) {
      if (GST_BUFFER_SIZE(*buf) == 0) {
        gst_buffer_unref(*buf);
        if (dsp->lastcn && dsp->cn_counter == 0) {
          DBG_PRINT ("Duplicating CN frame\n");
          gst_buffer_ref(dsp->lastcn);
          *buf = dsp->lastcn;
          dsp->cn_counter = 4;
        } else {
          DBG_PRINT ("Dropping CN frame, counter = %d\n", dsp->cn_counter);
          dsp->framecount++;
          dsp->cn_counter--;
          continue;
        }
      } else if (GST_BUFFER_FLAG_IS_SET(*buf, GST_BUFFER_FLAG_LAST)) {
        /* Save for future NO_UPDATE */
        if (dsp->lastcn)
          gst_buffer_unref(dsp->lastcn);
        gst_buffer_ref(*buf);
        dsp->lastcn = *buf;
        dsp->cn_counter = 4;
      }

      GST_BUFFER_DURATION(*buf) = G729_FRAMETIMEMILLIS * GST_MSECOND;
      GST_BUFFER_TIMESTAMP(*buf) = dsp->framecount * G729_FRAMETIMEMILLIS * GST_MSECOND;
      gst_buffer_set_caps (*buf, GST_PAD_CAPS (GST_BASE_SRC_PAD (psrc)));
      dsp->framecount++;
      return GST_FLOW_OK;
    }
    return GST_FLOW_WRONG_STATE;
  }
}


/*
 * gst_dspg729src_start:
 * @bsrc: GstBaseSrc
 *
 * Start the operation. Initializes DSP node
 *
 * Returns: TRUE on success
 */
static gboolean
gst_dspg729src_start (GstBaseSrc * bsrc)
{
  DBG_PRINT("gst_dspg729src_start\n");
  GstDSPG729Src *dsp = GST_DSPG729SRC(bsrc);
  SPEECH_PARAMS_DATA init_data;

  if (gst_dspaudio_ed_microphone(dsp->audio, TRUE) == FALSE) {
    GST_ELEMENT_ERROR (dsp, RESOURCE,
                       OPEN_WRITE, (NULL),
                       ("gst_dspaudio_ed_microphone"));
    return FALSE;
  }

  if (gst_dspaudio_aep_initialize(dsp->audio) == FALSE) {
    gst_dspaudio_ed_microphone(dsp->audio, FALSE);
    GST_ELEMENT_ERROR (dsp, RESOURCE,
                       OPEN_READ_WRITE, (NULL),
                       ("gst_dspaudio_aep_initialize"));
    return FALSE;
  }

  if (gst_dspaudio_open(dsp->audio, devname) == FALSE) {
    gst_dspaudio_ed_microphone(dsp->audio, FALSE);
    GST_ELEMENT_ERROR (dsp, RESOURCE,
                       OPEN_READ_WRITE, (NULL),
                       ("gst_dspaudio_open"));
    return FALSE;
  }

  init_data.dsp_cmd = DSP_CMD_SET_SPEECH_PARAMS;
  if(dsp->dtxmode == GST_DSPAUDIO_DTX_MODE_ON) {
    init_data.audio_fmt = DSP_AFMT_G729_DTX;
    DBG_PRINT("DTX ON\n");
  }
  else {
    init_data.audio_fmt = DSP_AFMT_G729;
    DBG_PRINT("DTX OFF\n");
  }
  init_data.sample_rate = SAMPLE_RATE_8KHZ;
  init_data.ds_stream_ID = 0;
  init_data.stream_priority = dsp->audio->priority;
  init_data.frame_size = 80;    // 10ms
  dsp->framesize = 22;          // bytes

  DBG_PRINT("SRC FRAME SIZE: %d\n", dsp->framesize);

  // Read the junk out...
  gst_dspaudio_flush(&dsp->audio->codec);

  if(!gst_dspaudio_setparams(dsp->audio, (char *) &init_data,
                             sizeof(SPEECH_PARAMS_DATA)))
  {
    dsp->audio->mode = DSP_MODE_ERROR;
    return FALSE;
  }

  gst_dspaudio_play(dsp->audio);
  dsp->framecount = 0;
  dsp->cn_counter = 0;
  dsp->lastcn = NULL;
  DBG_PRINT("gst_dspg729src_start OK\n");
  return TRUE;
}


/*
 * gst_dspg729src_stop:
 * @bsrc: GstBaseSrc
 *
 * Stop the operation.
 *
 * Returns: TRUE on success
 */
static gboolean
gst_dspg729src_stop (GstBaseSrc * bsrc)
{
  DBG_PRINT("gst_dspg729src_stop\n");
  GstDSPG729Src *dsp = GST_DSPG729SRC(bsrc);

  gst_dspaudio_remove_interrupt(dsp->audio);

  g_mutex_lock(dsp->audio->dsp_mutex);

  if ( dsp->audio->all_frames_read ||
       dsp->audio->read_sent )
  {
    DBG_PRINT("Waiting reply for CMD_DATA_READ\n");
    gst_dspaudio_wait_buffer(dsp->audio);
    DBG_PRINT("Got it\n");
  }

  gst_dspaudio_dtmf_tone_stop(dsp->audio);

  if(gst_dspaudio_stop(dsp->audio)) {
    gst_dspaudio_read_cmd(dsp->audio, 500);
  }

  gst_dspaudio_ed_microphone(dsp->audio, FALSE);
  gst_dspaudio_close(dsp->audio);
  gst_dspaudio_aep_close(dsp->audio);
  gst_dspaudio_reset(dsp->audio);
  g_mutex_unlock(dsp->audio->dsp_mutex);

  DBG_PRINT("gst_dspg729src_stop OK\n");
  return TRUE;
}


/*
 * gst_dspg729src_set_property:
 * @object: GObject where the property should be retrieved
 * @prop_id: Unique identifier of the desired property
 * @value: Pointer to a variable where the value should be put
 * @pspec: Parameter specification
 *
 * Stores the property value into GObject
 */

static void
gst_dspg729src_set_property (GObject * object,
                             guint prop_id,
                             const GValue * value,
                             GParamSpec * pspec)
{
  GstDSPG729Src *dsp = GST_DSPG729SRC (object);

  if(prop_id == DSPG729_PROP_DTX_MODE) {
    dsp->dtxmode = g_value_get_enum(value);
  }
  else if(!gst_dspaudio_set_property(dsp->audio, prop_id, value)) {
    g_warning("trying to set illegal property");
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
  }
}


/*
 * gst_dspg729src_get_property:
 * @object: GObject where the property should be retrieved
 * @prop_id: Unique identifier of the desired property
 * @value: Pointer to a variable where the value should be put
 * @pspec: Parameter specification
 *
 * Retrieves the desired parameter value from object
 */
static void
gst_dspg729src_get_property (GObject * object,
                             guint prop_id,
                             GValue * value,
                             GParamSpec * pspec)
{
  GstDSPG729Src *dsp = GST_DSPG729SRC (object);

  if(prop_id == DSPG729_PROP_DTX_MODE) {
    g_value_set_enum (value, dsp->dtxmode);
  }
  else if(!gst_dspaudio_get_property(dsp->audio, prop_id, value)) {
    g_warning("trying to get illegal property");
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
  }
}

/*
 * gst_dspg729src_event:
 * @src: GstBaseSrc object
 * @event: GstEvent to be handled
 *
 * Handles incoming events. Checks if the event is special "custom upstream
 * event" and in this case forwards it to dspaudio library for checking if
 * it is DTMF event.
 *
 * Returns: TRUE if the event was handled.
 */
static gboolean
gst_dspg729src_event (GstBaseSrc *src, GstEvent *event)
{
  GstEventType type = event ? GST_EVENT_TYPE (event) : GST_EVENT_UNKNOWN;
  GstDSPG729Src *dsp = GST_DSPG729SRC (src);
  gboolean ret = FALSE;

  if(type == GST_EVENT_CUSTOM_UPSTREAM) {
    ret = gst_dspaudio_check_upstream_event (dsp->audio, event);
  }
  return ret;
}


/*
 *
 *
 */
static GstClock *
gst_dspg729src_provide_clock (GstElement * elem)
{
  DBG_PRINT("gst_dspg729src_provide_clock\n");
  GstDSPG729Src *dsp = GST_DSPG729SRC (elem);
  return GST_CLOCK_CAST (gst_object_ref (dsp->clock));
}


/*
 *
 *
 */
static GstClockTime
gst_dspg729src_get_time (GstClock * clock, GstDSPG729Src * src)
{
  GstClockTime result;
  result = src->framecount * G729_FRAMETIMEMILLIS * GST_MSECOND;
  return result;
}


/*
 * gst_dspilbcsrc_change_state:
 * @element: GstElement representing G729 element
 *
 * Changes the state of this element.
 *
 * Returns: GstStateChangeReturn
 */
static GstStateChangeReturn
gst_dspg729src_change_state (GstElement * element, GstStateChange transition)
{
  GstDSPG729Src *dsp = GST_DSPG729SRC (element);
  GstStateChangeReturn ret = GST_STATE_CHANGE_SUCCESS;

  switch (transition) {
    case GST_STATE_CHANGE_PLAYING_TO_PAUSED:
      DBG_PRINT("G729 SRC PLAYING TO PAUSED\n");
      gst_dspaudio_interrupt_render(dsp->audio);
      break;

    default:
      break;
  }
  ret = GST_ELEMENT_CLASS (parent_class)->change_state (element, transition);
  return ret;
}


/*
 * gst_dspg729src_voice_parametric_to_bitstream:
 * @in: input frame pointer
 * @out: pointer where to write the re-formatted frame
 *
 * Convert incoming DSP G.729 SID_UPDATE (parametric representation) into
 * "normal" G.729 Comfort Noise frame format (big endian).
 */
static inline void
gst_dspg729src_cn_parametric_to_bitstream (const short int *in, char *out)
{
#if G_BYTE_ORDER == G_LITTLE_ENDIAN
    out[0] = (in[0] << 7) |(in[1] << 2) | (in[2] >> 2);
    out[1] = ((in[2] & 3) << 6) |(in[3] << 1);
#else
    out[1] = (in[0] << 7) |(in[1] << 2) | (in[2] >> 2);
    out[0] = ((in[2] & 3) << 6) |(in[3] << 1);
#endif
}


/*
 * gst_dspg729src_voice_parametric_to_bitstream:
 * @in: input frame pointer
 * @out: pointer where to write the re-formatted frame
 *
 * Convert incoming DSP G.729 frame (parametric representation) into
 * "normal" G.729 voice frame format (big endian).
 */
static inline void
gst_dspg729src_voice_parametric_to_bitstream (const short int *in, char *out)
{
#if G_BYTE_ORDER == G_LITTLE_ENDIAN
    out[0] = in[0];
    out[1] = in[1] >> 2;
    out[2] = ((in[1] & 3) << 6) | (in[2] >> 2);
    out[3] = ((in[2] & 3) << 6) | (in[3] << 5) | (in[4] >> 8);
    out[4] = in[4] & 0xff;
    out[5] = (in[5] << 4) | (in[6] >> 3);
    out[6] = ((in[6] & 0x07) << 5) | in[7];
    out[7] = in[8] >> 5;
    out[8] = ((in[8] & 0x1f) << 3) | (in[9] >> 1);
    out[9] = ((in[9] & 1) << 7) | in[10];
#else
    out[3] = in[0];
    out[2] = in[1] >> 2;
    out[1] = ((in[1] & 3) << 6) | (in[2] >> 2);
    out[0] = ((in[2] & 3) << 6) | (in[3] << 5) | (in[4] >> 8);
    out[7] = in[4] & 0xff;
    out[6] = (in[5] << 4) | (in[6] >> 3);
    out[5] = ((in[6] & 0x07) << 5) | in[7];
    out[4] = in[8] >> 5;
    out[9] = ((in[8] & 0x1f) << 3) | (in[9] >> 1);
    out[8] = ((in[9] & 1) << 7) | in[10];
#endif
}


/*
 * gst_dspg729src_voice_frame_handler:
 * @mmap_ptr: Pointer to audio frame data
 * @fsize: Size of the incoming frame, in BYTES
 *
 * VOICE_FRAME frame handler function for G.729
 * Returns: GstBuffer containing the handled frame
 */
static GstBuffer *
gst_dspg729src_voice_frame_handler (short int *mmap_ptr, guint fsize)
{
    GstBuffer *newbuf = gst_buffer_new_and_alloc(10);
    char *data = GST_BUFFER_DATA (newbuf);
    gst_dspg729src_voice_parametric_to_bitstream (mmap_ptr, data);
    return newbuf;
}


/*
 * gst_dspg729src_sid_frame_handler:
 * @mmap_ptr: Pointer to audio frame data
 * @fsize: Size of the incoming frame, in BYTES
 *
 * SID_UPDATE frame handler function for G.729
 * Returns: GstBuffer containing the handled frame
 */
static GstBuffer *
gst_dspg729src_sid_frame_handler (short int *mmap_ptr, guint fsize)
{
    GstBuffer *newbuf = gst_buffer_new_and_alloc(2);
    char *data = GST_BUFFER_DATA (newbuf);
    gst_dspg729src_cn_parametric_to_bitstream (mmap_ptr, data);
    GST_BUFFER_FLAG_SET(newbuf, GST_BUFFER_FLAG_LAST);
    return newbuf;
}
