/*
 * This file is part of libgst0.10-dsp
 *
 * Copyright (C) 2006 Nokia Corporation. All rights reserved.
 *
 * Contact: Stefan Kost <stefan.kost@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <unistd.h>
#include <string.h>
#include <gst/gst.h>

#include "gstdspg729sink.h"

static const gchar *devname = "/dev/dsptask/g729_dec";

static GstStaticPadTemplate dspg729sink_sink_template =
GST_STATIC_PAD_TEMPLATE ("sink",
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS ( "audio/g729, "
                      "rate = (int) 8000, "
                      "channels = (int) 1"
                    ) );

// Function prototypes

static void gst_dspg729sink_class_init (GstDSPG729SinkClass * klass);
static void gst_dspg729sink_base_init (gpointer g_class);
static void gst_dspg729sink_init (GstDSPG729Sink *dspmp3sink, GstDSPG729SinkClass *g_class);
static void gst_dspg729sink_set_property (GObject * object, guint prop_id, const GValue * value, GParamSpec * pspec);
static void gst_dspg729sink_get_property (GObject * object, guint prop_id, GValue * value, GParamSpec * pspec);
static gboolean gst_dspg729sink_start (GstBaseSink * bsink);
static gboolean gst_dspg729sink_stop (GstBaseSink * bsink);
static GstFlowReturn gst_dspg729sink_render(GstBaseSink *sink, GstBuffer * buffer);
static GstStateChangeReturn gst_dspg729sink_change_state (GstElement * element, GstStateChange transition);


#ifdef DEBUG
static void _g729sink_do_init()
{
  DBG_PRINT("G729 DO_INIT\n");
}

GST_BOILERPLATE_FULL (GstDSPG729Sink, gst_dspg729sink, GstBaseSink,
                      GST_TYPE_BASE_SINK, _g729sink_do_init);
#else
GST_BOILERPLATE (GstDSPG729Sink, gst_dspg729sink, GstBaseSink,
                 GST_TYPE_BASE_SINK);
#endif


/*
 * gst_dspg729sink_dispose:
 * @object: GObject pointer to element to be deleted
 *
 * Deletes G729 sink element instance. Called automatically by
 * GLib framework when element needs to be disposed.
 */
static void
gst_dspg729sink_dispose (GObject * object)
{
  GstDSPG729Sink *dsp = (GstDSPG729Sink *) object;

  if(dsp->audio) {
    gst_dspaudio_destroy(dsp->audio);
    dsp->audio = NULL;
  }

  G_OBJECT_CLASS (parent_class)->dispose (object);
}


/*
 * gst_dspg729sink_base_init:
 * @g_class: G729 sink class
 *
 * Does the basic initialization of G729 sink element class. This is
 * called by GStreamer framework.
 */
static void
gst_dspg729sink_base_init (gpointer g_class)
{
  GstElementClass *gstelement_class = GST_ELEMENT_CLASS (g_class);
  static const GstElementDetails details = GST_ELEMENT_DETAILS (
    "DSP G.729 Sink",
    "Sink/Audio",
    "G729 audio sink",
    "Makoto Sugano <makoto.sugano@nokia.com>");

  gst_element_class_add_pad_template (gstelement_class,
      gst_static_pad_template_get (&dspg729sink_sink_template));
  gst_element_class_set_details (gstelement_class, &details);
}


/*
 * gst_dspg729sink_class_init:
 * @klass: G729 Sink class object
 *
 * Initializes G729 sink element class. This is called by GStreamer
 * framework.
 */
static void
gst_dspg729sink_class_init (GstDSPG729SinkClass * klass)
{
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;
  GstBaseSinkClass *gstbase_sink_class;

  gobject_class = (GObjectClass *) klass;
  gstelement_class = (GstElementClass *) klass;
  gstbase_sink_class = (GstBaseSinkClass *) klass;

  gobject_class->set_property = GST_DEBUG_FUNCPTR (gst_dspg729sink_set_property);
  gobject_class->get_property = GST_DEBUG_FUNCPTR (gst_dspg729sink_get_property);
  gobject_class->dispose = GST_DEBUG_FUNCPTR (gst_dspg729sink_dispose);

   // Use dspaudio utility to install default properties
  gst_dspaudio_install_properties (G_OBJECT_CLASS (klass));

  gstelement_class->change_state = GST_DEBUG_FUNCPTR (gst_dspg729sink_change_state);
  gstbase_sink_class->start = GST_DEBUG_FUNCPTR (gst_dspg729sink_start);
  gstbase_sink_class->stop = GST_DEBUG_FUNCPTR (gst_dspg729sink_stop);
  gstbase_sink_class->render = GST_DEBUG_FUNCPTR (gst_dspg729sink_render);

}


/*
 * gst_dspg729sink_init:
 * @dsp: DSP G729 sink object
 * @g_class: G729 sink class
 *
 * Initializes the given G729 sink element instance and allocates required
 * resources. This also creates an audio stream object. This is
 * called by GStreamer framework.
 */
static void
gst_dspg729sink_init (GstDSPG729Sink * dspg729sink, GstDSPG729SinkClass *g_class)
{
  dspg729sink->audio = gst_dspaudio_new();
  dspg729sink->adapter = gst_adapter_new();
  gst_base_sink_set_sync (GST_BASE_SINK (dspg729sink), FALSE);
}


/*
 *
 *
 */
static inline void
gst_dspg729sink_voice_bitstream_to_parametric (const char *in, short int *out)
{
#if G_BYTE_ORDER == G_LITTLE_ENDIAN
    out[0] = in[0];                                              // L0, L1
    out[1] = ((short int)in[1] << 2) | (in[2] & 0xc0) >> 6;      // L2, L3
    out[2] = ((in[2] & 0x3f) << 2) | (in[3] >> 6);               // P1
    out[3] = (in[3] & 0x20) >> 5;                                // P0
    out[4] = (((short int)(in[3] & 0x1f)) << 8) | in[4];         // C1
    out[5] = (in[5] & 0xf0) >> 4;                                // S1
    out[6] = ((in[5] & 0x0f) << 3) | ((in[6] & 0xe0) >> 5);      // GA1, GB1
    out[7] = in[6] & 0x1f;                                       // P2
    out[8] = (((short int) in[7]) << 5) | ((in[8] & 0xf8) >> 3); // C2
    out[9] = ((in[8] & 7) << 1) | ((in[9] & 0x80) >> 7);         // S2
    out[10] = (in[9] & 0x7F);                                    // GA2, GB2
#else
    out[0] = in[3];                                              // L0, L1
    out[1] = ((short int)in[2] << 2) | (in[1] & 0xc0) >> 6;      // L2, L3
    out[2] = ((in[1] & 0x3f) << 2) | (in[0] >> 6);               // P1
    out[3] = (in[0] & 0x20) >> 5;                                // P0
    out[4] = (((short int)(in[0] & 0x1f)) << 8) | in[7];         // C1
    out[5] = (in[6] & 0xf0) >> 4;                                // S1
    out[6] = ((in[6] & 0x0f) << 3) | ((in[5] & 0xe0) >> 5);      // GA1, GB1
    out[7] = in[5] & 0x1f;                                       // P2
    out[8] = (((short int) in[4]) << 5) | ((in[9] & 0xf8) >> 3); // C2
    out[9] = ((in[9] & 7) << 1) | ((in[8] & 0x80) >> 7);         // S2
    out[10] = (in[8] & 0x7F);                                    // GA2, GB2
#endif
}


/*
 *
 *
 */
static inline void
gst_dspg729sink_cn_bitstream_to_parametric (const char *in, short int *out)
{
#if G_BYTE_ORDER == G_LITTLE_ENDIAN
    out[0] = (in[0] & 0x80) >> 7;
    out[1] = (in[0] & 0x7c) >> 2;
    out[2] = ((in[0] & 3) << 2) | ((in[1] & 0xc0) >> 6);
    out[3] = (in[1] & 0x3e) >> 1;
#else
    out[0] = (in[1] & 0x80) >> 7;
    out[1] = (in[1] & 0x7c) >> 2;
    out[2] = ((in[1] & 3) << 2) | ((in[0] & 0xc0) >> 6);
    out[3] = (in[0] & 0x3e) >> 1;
#endif
}


/*
 * gst_dspg729sink_render:
 * sink: GstBaseSink
 * @buffer: GstBuffer to be consumed
 *
 * Plays the given audio buffer
 */
static GstFlowReturn
gst_dspg729sink_render (GstBaseSink *sink, GstBuffer * buffer)
{
  GstFlowReturn ret = GST_FLOW_OK;
  GstDSPG729Sink *dsp = GST_DSPG729SINK (sink);
  const guint8 *data = GST_BUFFER_DATA(buffer);
  guint avail = GST_BUFFER_SIZE (buffer);
  DSPWaitStatus status;
  guint dataindex = 0;
  gboolean skip = FALSE;

//   DBG_PRINT("G729 RENDER: %d bytes, timestamp = %lld\n",
//     GST_BUFFER_SIZE(buffer), GST_BUFFER_TIMESTAMP(buffer));

  do {
    g_mutex_lock(dsp->audio->dsp_mutex);
    status = gst_dspaudio_wait_buffer(dsp->audio);

    if(status == DSP_WAIT_RETRY) {
      g_mutex_unlock(dsp->audio->dsp_mutex);
      continue;
    }
    if(status == DSP_WAIT_INTERRUPT) {
      g_mutex_unlock(dsp->audio->dsp_mutex);
      break;
    }
    if(status == DSP_WAIT_ERROR) {
      g_mutex_unlock(dsp->audio->dsp_mutex);
      GST_ELEMENT_ERROR (dsp, RESOURCE, WRITE, (NULL),
                         ("loop2: error cmd: %d status: %d",
                          dsp->audio->error_cmd,
                          dsp->audio->error_status));
      ret = GST_FLOW_ERROR;
      break;
    }

    dsp->audio->rw_pending = 1;
    dsp->mmap_ptr[0] = 0;           // BFI (Bad Frame Indication)

    if (avail >= dsp->framesize) {
        dsp->mmap_ptr[1] = VOICE_FRAME; // DTX mode
        dsp->outbuf.data_size = 2 + 11; // BFI + DTX + data
        gst_dspg729sink_voice_bitstream_to_parametric(data+dataindex,
            &(dsp->mmap_ptr[2]));
        dataindex += dsp->framesize;
        avail -= dsp->framesize;
    }
    else if (avail >= 2) {
        if (data[dataindex+0] == dsp->lastcn[0] &&
            data[dataindex+1] == dsp->lastcn[1])
        {
//            dsp->mmap_ptr[1] = SID_FRAME_NO_UPDATE;
//            dsp->outbuf.data_size = 2;
            skip = TRUE;
        }
        else {
/*            dsp->mmap_ptr[1] = SID_FRAME_UPDATE;  // DTX mode
            dsp->outbuf.data_size = 2 + 4;        // BFI + DTX + SID
            gst_dspg729sink_cn_bitstream_to_parametric(data+dataindex,
                &(dsp->mmap_ptr[2]));
*/
            dsp->lastcn[0] = data[dataindex+0];
            dsp->lastcn[1] = data[dataindex+1];
            skip = TRUE;
        }
        dataindex += 2;
        avail -= 2;
    }

    if (skip == FALSE) {
        if (write(dsp->audio->codec.fd, &(dsp->outbuf),
            sizeof (G729_DATA_WRITE)) < 0)
        {
            dsp->audio->mode = DSP_MODE_ERROR;
            GST_ELEMENT_ERROR (dsp, RESOURCE, WRITE, (NULL), ("write = -1"));
            g_mutex_unlock(dsp->audio->dsp_mutex);
            ret = GST_FLOW_ERROR;
            break;
        }
//      DBG_PRINT("G729: Wrote %d bytes to DSP\n",
//                dsp->outbuf.data_size * sizeof(short int));
        dsp->audio->rw_pending = 0;
    }
    g_mutex_unlock(dsp->audio->dsp_mutex);
  }
  while (avail >= 2);

  return ret;
}


/*
 * gst_dspg729sink_start:
 * @bsink: GstBaseSink
 *
 * Open DSP node and start the playback
 *
 * Returns: TRUE on success
 */
static gboolean
gst_dspg729sink_start (GstBaseSink * bsink)
{
  DBG_PRINT("gst_dspg729sink_start\n");
  GstDSPG729Sink *dsp = GST_DSPG729SINK (bsink);
  SPEECH_PARAMS_DATA init_data;

  dsp->outbuf.dsp_cmd = DSP_CMD_DATA_WRITE;
  if (gst_dspaudio_open(dsp->audio, devname) == FALSE) {
    GST_ELEMENT_ERROR (dsp, RESOURCE,
                       OPEN_READ_WRITE, (NULL),
                       ("gst_dspaudio_open"));
    return FALSE;
  }

  init_data.dsp_cmd = DSP_CMD_SET_SPEECH_PARAMS;
  init_data.audio_fmt = DSP_AFMT_G729;
  init_data.sample_rate = SAMPLE_RATE_8KHZ;
  init_data.ds_stream_ID = 0;
  init_data.stream_priority = dsp->audio->priority;
  init_data.frame_size = 80;  // 10 ms
  dsp->framesize = 10;        // bytes

  // Read the junk out...
  gst_dspaudio_flush(&dsp->audio->codec);

  if(!gst_dspaudio_setparams(dsp->audio, (char *)&init_data, sizeof(SPEECH_PARAMS_DATA))) {
    dsp->audio->mode = DSP_MODE_ERROR;
    return FALSE;
  }
  dsp->lastcn[0] = dsp->lastcn[1] = 0;
  dsp->mmap_ptr = (short int *) dsp->audio->codec.mmap_buffer;
  dsp->audio->mode = DSP_MODE_INITIALIZED;
  gst_dspaudio_update_dsp_settings(dsp->audio);
  gst_dspaudio_play(dsp->audio);
  DBG_PRINT("gst_dspg729sink_start OK\n");
  return TRUE;
}


/*
 * gst_dspg729sink_stop:
 * @bsink: GstBaseSink
 *
 * Stop the operation. Uninitialize and close DSP.
 *
 * Returns: TRUE on success
 */
static gboolean
gst_dspg729sink_stop (GstBaseSink * bsink)
{
  DBG_PRINT("gst_dspg729sink_stop\n");
  GstDSPG729Sink *dsp = GST_DSPG729SINK (bsink);

  gst_dspaudio_remove_interrupt(dsp->audio);

  g_mutex_lock(dsp->audio->dsp_mutex);
  if(gst_dspaudio_stop(dsp->audio)) {
    gst_dspaudio_read_cmd(dsp->audio, 0);
  }
  gst_dspaudio_close(dsp->audio);
  gst_dspaudio_reset(dsp->audio);
  gst_adapter_clear(dsp->adapter);
  g_mutex_unlock(dsp->audio->dsp_mutex);

  DBG_PRINT("gst_dspg729sink_stop OK\n");
  return TRUE;
}


/*
 * gst_dspg729sink_set_property:
 * @object: GObject where the property should be retrieved
 * @prop_id: Unique identifier of the desired property
 * @value: Pointer to a variable where the value should be put
 * @pspec: Parameter specification
 *
 * Stores the property value into GObject
 */
static void
gst_dspg729sink_set_property (GObject * object,
                              guint prop_id,
                              const GValue * value,
                              GParamSpec * pspec)
{
  GstDSPG729Sink *dsp = GST_DSPG729SINK (object);

  if(!gst_dspaudio_set_property(dsp->audio, prop_id, value)) {
    g_warning("trying to set illegal property");
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
  }
}


/*
 * gst_dspg729sink_get_property:
 * @object: GObject where the property should be retrieved
 * @prop_id: Unique identifier of the desired property
 * @value: Pointer to a variable where the value should be put
 * @pspec: Parameter specification
 *
 * Retrieves the desired parameter value from object
 */
static void
gst_dspg729sink_get_property (GObject * object,
                              guint prop_id,
                              GValue * value,
                              GParamSpec * pspec)
{
  GstDSPG729Sink *dsp = GST_DSPG729SINK (object);

  if(!gst_dspaudio_get_property(dsp->audio, prop_id, value)) {
    g_warning("trying to get illegal property");
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
  }
}


/*
 * gst_dspg729sink_change_state:
 * @element: GstElement representing G729 element
 *
 * Changes the state of this element.
 *
 * Returns: GstStateChangeReturn
 */
static GstStateChangeReturn
gst_dspg729sink_change_state (GstElement * element, GstStateChange transition)
{
  GstDSPG729Sink *dsp = GST_DSPG729SINK (element);
  GstStateChangeReturn ret = GST_STATE_CHANGE_SUCCESS;

  switch (transition) {
    case GST_STATE_CHANGE_PLAYING_TO_PAUSED:
      GST_DEBUG("PLAYING TO PAUSED");
      gst_dspaudio_interrupt_render(dsp->audio);
      break;

    default:
      break;
  }

  ret = GST_ELEMENT_CLASS (parent_class)->change_state (element, transition);
  return ret;
}


