/*
 * This file is part of libgst0.10-dsp
 *
 * Copyright (C) 2006 Nokia Corporation. All rights reserved.
 *
 * Contact: Stefan Kost <stefan.kost@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#ifndef __GST_DSP_AUDIO_H__
#define __GST_DSP_AUDIO_H__

#include <glib.h>
#include <gst/gst.h>
#include <gst/audio/gstaudioclock.h>
#include <sys/select.h>

#include "omapdsp.h"
#include <dsp/audio_dsptask.h>

G_BEGIN_DECLS

#ifdef DEBUG
#define DBG_PRINT(...) g_print(__VA_ARGS__)
#else
#define DBG_PRINT(...) ((void)(0))
#endif

#ifndef MIN
#define MIN(a,b) ((a)<(b)) ? (a) : (b)
#endif

#define DSP_VOLUME_DEFAULT  0xFFFF
#define DSP_VOLUME_MAX      0xFFFF

#define DSP_PANNING_MIN     -0x4000
#define DSP_PANNING_MAX      0x4000
#define DSP_PANNING_DEFAULT  0

#define DSP_PRIORITY_MIN     0
#define DSP_PRIORITY_MAX     5
#define DSP_PRIORITY_DEFAULT 0


#define SID_FRAME_NO_UPDATE       0           /* SID params no update        */
#define VOICE_FRAME               1           /* data is encoded speech      */
#define SID_FRAME_UPDATE          2           /* SID params to be updated    */

// Internal pipe communication "protocol" commands
#define PIPE_CMD_PROP_UPDATE      1
#define PIPE_CMD_PAUSE            2
#define PIPE_CMD_INTERRUPT        3
#define PIPE_CMD_NEED_LOCK        4

#define AEP_DEVICE_NAME "/dev/dsptask/aep"

#define GST_TYPE_DSPAUDIO_DTX_MODE (gst_dspaudio_dtx_mode_get_type())

typedef enum {
  GST_DSPAUDIO_DTX_MODE_OFF,
  GST_DSPAUDIO_DTX_MODE_ON,
} GstDSPAudioDTXModes;


typedef enum
{
  DSP_WAIT_ERROR,
  DSP_WAIT_OK,
  DSP_WAIT_INTERRUPT,
  DSP_WAIT_RETRY
} DSPWaitStatus;


typedef enum
{
  DSPAUDIO_PROP_0,
  DSPAUDIO_PROP_VOLUME,
  DSPAUDIO_PROP_PANNING,
  DSPAUDIO_PROP_MUTE,
  DSPAUDIO_PROP_STREAMID,
  DSPAUDIO_PROP_POSTPROCESSING,
  DSPAUDIO_PROP_PRIORITY,
  DSPAUDIO_PROP_LAST
} DSPAudioProperties;

typedef struct _GstDSPAudio GstDSPAudio;
typedef struct _DSPNodeInfo DSPNodeInfo;


struct _DSPNodeInfo {
  int            fd;
  guint          stream_id;
  guint          bridge_buffer_size;  // in bytes
  guint          mmap_buffer_size;    // in bytes
  char *         mmap_buffer;
};


struct _GstDSPAudio {
  GMutex         *dsp_mutex;
  DSPNodeInfo    codec;
  DSPNodeInfo    aep;

  /* clock */
  GstClockTime clock_time;
  GstClockTime last_dsp_update;

  /* Property-related */
  guint          volume;
  gboolean       volume_changed;
  gint           panning;
  guint          priority;
  gboolean       panning_changed;
  gboolean       mute;
  gboolean       mute_changed;

  /* DSP communication */
  guint          rw_pending;
  gint           error_status;
  guint          error_cmd;
  gboolean       discont_sent;
  //gboolean       mic_enabled;
  guint          mode;
  gboolean       read_sent;
  gboolean       interrupted;
  gboolean       dtmf_playing;

  // DTX-mode specific
  gboolean        all_frames_read;
  guint           readindex;

  AUDIO_STATUS_INFO info;

  short int      dsp_buffer_free;
  fd_set         read_fdset;
  int            pipe_fds[2];

  struct {
    unsigned short int frame_size;
    unsigned int stream_time_ms;
  } readinfo;

  GCond          *clock_cond;
  GMutex         *clock_mutex;

  /* For resource management */
  gpointer       *dbus_connection;

  /* Audio frame handlers */
  GstBuffer *    (*handle_voice_frame) (short int *, guint);
  GstBuffer *    (*handle_sid_frame) (short int *, guint);
};

// Member functions

GType
gst_dspaudio_dtx_mode_get_type ();

GstDSPAudio*
gst_dspaudio_new();

void
gst_dspaudio_reset (GstDSPAudio *dsp);

void
gst_dspaudio_destroy(GstDSPAudio *dsp);

gboolean
gst_dspaudio_open(GstDSPAudio *dsp, const gchar *devname);

gboolean
gst_dspaudio_close(GstDSPAudio *dsp);

gboolean
gst_dspaudio_setparams(GstDSPAudio *dsp, char *data, guint datasize);

gboolean
gst_dspaudio_play(GstDSPAudio *dsp);

gboolean
gst_dspaudio_pause(GstDSPAudio *dsp);

gboolean
gst_dspaudio_stop(GstDSPAudio *dsp);

gboolean
gst_dspaudio_discont(GstDSPAudio *dsp);

gboolean
gst_dspaudio_map_samplerate(gint rate,
                            gint *dest_rate);

DSPWaitStatus
gst_dspaudio_wait_buffer(GstDSPAudio *dsp);

DSPWaitStatus
gst_dspaudio_read_cmd(GstDSPAudio *dsp, gint timeout_millis);

void
gst_dspaudio_flush(DSPNodeInfo *info);

void
gst_dspaudio_wait_eos(GstDSPAudio *dsp);

void
gst_dspaudio_install_properties(GObjectClass *klass);

gboolean
gst_dspaudio_set_property(GstDSPAudio *dsp,
                          guint prop_id,
                          const GValue *value);

gboolean
gst_dspaudio_get_property(GstDSPAudio *dsp,
                          guint prop_id,
                          GValue *value);

gboolean
gst_dspaudio_query_peer_position(GstPad *sinkpad,
                                 gint64 *value);

gboolean
gst_dspaudio_query_peer_duration(GstPad *sinkpad,
                                 gint64 *value);

gboolean
gst_dspaudio_get_info(GstDSPAudio *dsp);

void
gst_dspaudio_update_dsp_settings(GstDSPAudio *dsp);

gboolean
gst_dspaudio_ed_microphone(GstDSPAudio *dsp, gboolean mic_enable);

void gst_dspaudio_interrupt_render(GstDSPAudio *dsp);
void gst_dspaudio_remove_interrupt(GstDSPAudio *dsp);

gboolean
gst_dspaudio_dtmf_tone_start(GstDSPAudio *dsp, gint tone_id);

gboolean
gst_dspaudio_dtmf_tone_stop(GstDSPAudio *dsp);

gboolean
gst_dspaudio_check_upstream_event(GstDSPAudio *dsp, GstEvent *event);

gboolean
gst_dspaudio_activate_push (GstPad * pad, gboolean active);

// ===================================================
// == AEP ============================================
// ===================================================

gboolean
gst_dspaudio_aep_initialize(GstDSPAudio *dsp);

gboolean
gst_dspaudio_aep_close(GstDSPAudio *dsp);

// ===================================================
// == Clock Helper Function ==========================
// ===================================================

extern GstStateChangeReturn
gst_dspaudio_clock_change_state(GstDSPAudio *dsp, GstStateChange transition);

extern GstClockTime
gst_dspaudio_get_dsp_clock(GstDSPAudio *dsp);

// ===================================================
// == Reading audio from DSP =========================
// ===================================================

GstBuffer *
gst_dspaudio_read_frame(GstDSPAudio *dsp, guint framesize, guint ufs,
                        gboolean need_cn);


G_END_DECLS


#endif
