/*
 * This file is part of libgst0.10-dsp
 *
 * Copyright (C) 2006 Nokia Corporation. All rights reserved.
 *
 * Contact: Stefan Kost <stefan.kost@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */


#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <sys/poll.h>

#include <gst/gst.h>

#include "dspaudio.h"
#include "apps_voltable.h"
#include <dsp/interface_common.h>

#if DSP_API_VERSION != 10
#error Expected DSP API version 10
#endif

#define DBUS_API_SUBJECT_TO_CHANGE
#include <dbus/dbus.h>

#define AUDIO_PM_SERVICE            "com.nokia.osso_audio_pm"
#define AUDIO_PM_PLAYBACK_RESOURCE  "/com/nokia/osso/pm/audio/playback"
#define AUDIO_PM_RECORD_RESOURCE    "/com/nokia/osso/pm/audio/record"
#define RESOURCE_INTERFACE          "com.nokia.osso_resource_manager"
#define RESOURCE_TIMEOUT            200


static gboolean 
gst_dspaudio_open_node(DSPNodeInfo *info, const gchar *devname);
static gboolean
gst_dspaudio_close_node(DSPNodeInfo *info);
static gboolean
gst_dspaudio_check_read(int dev_fd);
static gboolean
gst_dspaudio_set_volume(GstDSPAudio *dsp);
static gboolean
gst_dspaudio_set_panning(GstDSPAudio *dsp);
static gboolean
gst_dspaudio_set_mute(GstDSPAudio *dsp);
static gboolean
gst_dspaudio_reset_node(GstDSPAudio *dsp);
static gboolean
gst_dspaudio_mic_ctl(GstDSPAudio *dsp, gboolean mic_enable);
static void
gst_dspaudio_notify_render(GstDSPAudio *dsp);
static gboolean
gst_dspaudio_clock_interrupt(GstDSPAudio *dsp);
static GstClockTime
gst_dspaudio_get_dsp_time(GstDSPAudio *dsp);
static GstBuffer *
gst_dspaudio_default_voice_frame_handler (short int *mmap_ptr, guint fsize);
static GstBuffer *
gst_dspaudio_default_sid_frame_handler (short int *mmap_ptr, guint fsize);


GType
gst_dspaudio_dtx_mode_get_type (void)
{
  static GType dspaudio_dtx_mode_type = 0;
  static GEnumValue dspaudio_dtx_modes[] = {
    {GST_DSPAUDIO_DTX_MODE_OFF, "Don't use DTX", "off"},
    {GST_DSPAUDIO_DTX_MODE_ON, "Use DTX", "on"},
    {0, NULL, NULL},
  };

  if (G_UNLIKELY(!dspaudio_dtx_mode_type)) {
    dspaudio_dtx_mode_type = g_enum_register_static ("GstDSPAudioDTXMode",
        dspaudio_dtx_modes);
  }
  return dspaudio_dtx_mode_type;
}


/**
 * gst_dspaudio_reset:
 * @dsp: DSP audio object
 *
 * Method for resetting the audio object
 */
void
gst_dspaudio_reset(GstDSPAudio * dsp)
{
  dsp->codec.fd                 = dsp->aep.fd = -1;
  dsp->codec.stream_id          = dsp->aep.stream_id = 0;
  dsp->codec.bridge_buffer_size = dsp->aep.bridge_buffer_size = 0;
  dsp->codec.mmap_buffer_size   = dsp->aep.mmap_buffer_size = 0;
  dsp->codec.mmap_buffer        = dsp->aep.mmap_buffer = NULL;

  dsp->mode = DSP_MODE_UNINITIALIZED;
  dsp->volume_changed = FALSE;
  dsp->mute_changed = FALSE;
  dsp->panning_changed = FALSE;
  dsp->discont_sent = TRUE;
  dsp->rw_pending = 0;
  dsp->all_frames_read = TRUE;
  dsp->readindex = 0;
  //dsp->mic_enabled = FALSE;
  dsp->read_sent = FALSE;
  dsp->interrupted = FALSE;
  dsp->dtmf_playing = FALSE;
  FD_ZERO(&dsp->read_fdset);

  dsp->handle_voice_frame = gst_dspaudio_default_voice_frame_handler;
  dsp->handle_sid_frame = gst_dspaudio_default_sid_frame_handler;
}


/**
 * gst_dspaudio_new:
 *
 * Creates new audio object instance and initializes it
 * with default parameters. After this the device must be
 * separately opened with gst_dspaudio_open() method.
 *
 * Returns: New #GstDSPAudio instance
 */
GstDSPAudio *
gst_dspaudio_new()
{
  GstDSPAudio *dsp = g_new0(GstDSPAudio, 1);
  dsp->volume = DSP_VOLUME_DEFAULT;
  dsp->panning = DSP_PANNING_DEFAULT;
  dsp->priority = DSP_PRIORITY_DEFAULT;
  dsp->mute = FALSE;
  dsp->dsp_mutex = g_mutex_new();
  dsp->clock_mutex = g_mutex_new();
  gst_dspaudio_reset(dsp);

  if(pipe(dsp->pipe_fds) != 0) {
    g_mutex_free(dsp->dsp_mutex);
    g_free(dsp);
    return NULL;
  }

  /* Spec says we should just proceed with playback/record if power mgmt
   * fails, so we don't care about the dbus_bus_get return value here */
  dsp->dbus_connection = (gpointer *) dbus_bus_get(DBUS_BUS_SYSTEM, NULL);

  return dsp;
}


/**
 * gst_dspaudio_destroy:
 * @dsp: DSP audio object
 *
 * Delete given audio object instance
 */
void
gst_dspaudio_destroy(GstDSPAudio * dsp)
{
  if(dsp->dsp_mutex != NULL) {
    g_mutex_free(dsp->dsp_mutex);
    dsp->dsp_mutex = NULL;
  }
  if(dsp->clock_mutex != NULL) {
    g_mutex_unlock(dsp->clock_mutex);
    g_mutex_free(dsp->clock_mutex);
    dsp->clock_mutex = NULL;
  }
  close(dsp->pipe_fds[0]);
  close(dsp->pipe_fds[1]);
//  if(dsp->dbus_connection != NULL) {
//     dbus_connection_close((DBusConnection *)dsp->dbus_connection);
//  }
  g_free(dsp);
}


/*
 * gst_dspaudio_open_node:
 * @dsp: DSP node info object
 * @devname: device name to be opened & initialized
 *
 * Initializes given DSP node.
 *
 * Returns: TRUE if operation was successful, otherwise FALSE
 */
static gboolean
gst_dspaudio_open_node(DSPNodeInfo *info, const gchar *devname)
{
  AUDIO_INIT_STATUS init_status;
  AUDIO_STATUS_INFO statInfo;

  DBG_PRINT("gst_dspaudio_open_node : %s\n", devname);

  if(info == NULL) return FALSE;

  info->fd = open(devname, O_RDWR);
  if(info->fd == -1) {
    DBG_PRINT("Opening %s failed\n", devname);
    return FALSE;
  }

  //if(ioctl(info->fd, OMAP_DSP_TASK_IOCTL_LOCK) < 0) {
  //  DBG_PRINT("Locking of %s failed\n", devname);
  //  return FALSE;
  //}

  // First we try to read all "old stuff"
  gst_dspaudio_flush(info);

  statInfo.dsp_cmd = DSP_CMD_STATE;
  if(write(info->fd, &statInfo, sizeof(short int)) < 0) {
    DBG_PRINT("failed to query %s state\n", devname);
    return FALSE;
  }

  if(read(info->fd, &statInfo, sizeof(AUDIO_STATUS_INFO)) < 0) {
    DBG_PRINT("failed to read CMD_STATE return value\n");
    return FALSE;
  }

  if(statInfo.status == STATE_UNINITIALISED) {

    short int tmp = DSP_CMD_INIT;
    if(write(info->fd, &tmp, sizeof(short int)) < 0) {
      return FALSE;
    }

    if(read(info->fd, &init_status, sizeof(AUDIO_INIT_STATUS)) < 0) {
      DBG_PRINT("Error reading INIT status from %s\n", devname);
      return FALSE;
    }
    else if(init_status.init_status != DSP_OK) {
      DBG_PRINT("INIT_STATUS FAIL: %d\n", init_status.init_status);
      return FALSE;
    }

    info->stream_id = init_status.stream_ID;
    info->bridge_buffer_size = init_status.bridge_buffer_size;
    info->mmap_buffer_size = init_status.mmap_buffer_size * sizeof(short int);
  }
  else {
    info->stream_id = statInfo.stream_ID;
    info->bridge_buffer_size = statInfo.bridge_buffer_size;
    info->mmap_buffer_size = statInfo.mmap_buffer_size * sizeof(short int);
  }

  info->mmap_buffer = (char *)
      mmap( (void *) 0, info->mmap_buffer_size, PROT_READ | PROT_WRITE,
             MAP_SHARED, info->fd, 0);

  if(info->mmap_buffer == NULL) {
    DBG_PRINT("Cannot mmap data buffer");
    return FALSE;
  }

  DBG_PRINT("%s MMAP buffer size: %d bytes\n", devname, info->mmap_buffer_size);
  DBG_PRINT("%s STREAM_ID: %d\n", devname, info->stream_id);
  return TRUE;
}


/*
 * gst_dspaudio_close_node:
 * @dsp: DSP node info object
 *
 * Closes the specific DSP node
 *
 * Returns: TRUE if operation was successful, otherwise FALSE
 */
static gboolean
gst_dspaudio_close_node(DSPNodeInfo *info)
{
  DSP_CMD_STATUS cmd;
  DBG_PRINT("gst_dspaudio_close_node\n");

  if(info == NULL) return FALSE;

  // We may get called even when DSP is not opened yet
  if(info->fd != -1) {

    gst_dspaudio_flush(info);

    cmd.dsp_cmd = DSP_CMD_CLOSE;
    if(write(info->fd, &cmd, sizeof(short int)) > 0) {
      read(info->fd, &cmd, sizeof(DSP_CMD_STATUS));
    }

    if(info->mmap_buffer) {
      munmap(info->mmap_buffer, info->mmap_buffer_size);
      info->mmap_buffer = NULL;
    }

    // we need to give DSP some time to send the last messages
    usleep(200000);

    // flush the message buffer once again to make sure it is empty
    gst_dspaudio_flush(info);

//     ioctl(info->fd, OMAP_DSP_TASK_IOCTL_UNLOCK);
    close(info->fd);
    info->fd = -1;
  }

  DBG_PRINT("NODE CLOSED\n");
  return TRUE;
}


/**
 * gst_dspaudio_flush:
 * @dsp: DSP node info object
 *
 * Reads all unread data from node
 */
void
gst_dspaudio_flush(DSPNodeInfo *info)
{
  short int tmp;

  while(gst_dspaudio_check_read(info->fd)) {
    if(read(info->fd, &tmp, sizeof(short int)) < 0) break;
  }
}


/**
 * gst_dspaudio_open:
 * @dsp: DSP audio object
 * @devname: device name to be opened & initialized
 *
 * Returns: TRUE if operation was successful, otherwise FALSE
 */
gboolean
gst_dspaudio_open(GstDSPAudio *dsp, const gchar *devname)
{
  if(dsp->mode > DSP_MODE_UNINITIALIZED) {
    DBG_PRINT("DSP node '%s' already open\n", devname);
    return TRUE;
  }

  if(gst_dspaudio_open_node(&dsp->codec, devname) == FALSE) {
    dsp->mode = DSP_MODE_ERROR;
    return FALSE;
  }

  short int tmp = DSP_CMD_STOP;
  DSP_CMD_STATUS cmd;

  if(write(dsp->codec.fd, &tmp, sizeof(short int)) < 0) {
    dsp->mode = DSP_MODE_ERROR;
    return FALSE;
  }
  if(read(dsp->codec.fd, &cmd, sizeof(DSP_CMD_STATUS)) < 0) {
    return FALSE;
  }

  DBG_PRINT("MMAP buffer size: %d bytes\n", dsp->codec.mmap_buffer_size);
  DBG_PRINT("STREAM_ID: %d\n", dsp->codec.stream_id);
  dsp->mode = DSP_MODE_INITIALIZED;
  return TRUE;
}


/**
 * gst_dspaudio_close:
 * @dsp: DSP audio object
 *
 * Closes DSP audio
 *
 * Returns: TRUE if operation was successful, otherwise FALSE
 */
gboolean
gst_dspaudio_close(GstDSPAudio *dsp)
{
  gboolean ret = gst_dspaudio_close_node(&dsp->codec);
  dsp->mode = DSP_MODE_UNINITIALIZED;
  return ret;
}


/**
 * gst_dspaudio_setparams:
 * @dsp: DSP audio object
 * @data: pointer to AUDIO_PARAMS_DATA to be set
 *
 * This sets the given parameters to audio node
 *
 * Returns: TRUE if operation was successful, otherwise FALSE
 */
gboolean
gst_dspaudio_setparams(GstDSPAudio *dsp, char *data, guint datasize)
{
  DSP_CMD_STATUS cmd_status;
  gint len;

  if(dsp==NULL) return FALSE;
  if(dsp->codec.fd==-1) return FALSE;
  if(dsp->mode==DSP_MODE_ERROR) return FALSE;

  DBG_PRINT("setparams called\n");

  len = write(dsp->codec.fd, data, datasize);
  if(len == -1) {
    dsp->mode = DSP_MODE_ERROR;
    return FALSE;
  }

  DBG_PRINT("setparams command written\n");

  len = read(dsp->codec.fd, &cmd_status, sizeof(DSP_CMD_STATUS));

  if(len == -1 || cmd_status.status != DSP_OK) {
    dsp->error_cmd = cmd_status.dsp_cmd;
    dsp->error_status = cmd_status.status;
    dsp->mode = DSP_MODE_ERROR;
    g_warning("SET_PARAMS FAILED WITH STATUS %d", cmd_status.status);
    return FALSE;
  }

  dsp->mode = DSP_MODE_CONFIGURED;
  DBG_PRINT("setparams successful\n");
  return TRUE;

}


/*
 * gst_dspaudio_check_read:
 * @dsp: DSP audio object
 *
 * Checks if there is some data available from the DSP
 *
 * Returns: TRUE if data is available, otherwise FALSE
 */
static gboolean
gst_dspaudio_check_read(int dev_fd)
{
  struct pollfd pfd;

  pfd.fd = dev_fd;
  pfd.events = POLLIN;

  if(poll(&pfd,1,0)>0) return TRUE;
  else return FALSE;
}


/**
 * gst_dspaudio_read_cmd:
 * @dsp: DSP audio object
 *
 * This method reads one command reply from DSP. It can also receive
 * DATA_WRITE / DATA_READ command and set the flag accordingly.
 *
 * Returns: DSPWaitStatus
 */
DSPWaitStatus
gst_dspaudio_read_cmd(GstDSPAudio *dsp, gint timeout_millis)
{
  unsigned short int data[3];

  if(dsp==NULL) return DSP_WAIT_ERROR;
  if(dsp->codec.fd==-1) return DSP_WAIT_ERROR;
  if(dsp->mode==DSP_MODE_UNINITIALIZED) return DSP_WAIT_ERROR;
  if(dsp->mode==DSP_MODE_ERROR) return DSP_WAIT_ERROR;

  dsp->error_status = dsp->error_cmd = 0;

  // Read the command and it's status
  while (TRUE) {

    if(dsp->interrupted) {
      DBG_PRINT("READ_CMD INTERRUPT\n");
      return DSP_WAIT_INTERRUPT;
    }

    if(timeout_millis) {
      struct timeval tv;
      fd_set fdset;
      int retval;

      FD_ZERO(&fdset);
      FD_SET(dsp->codec.fd, &fdset);
      tv.tv_sec = timeout_millis / 1000;
      tv.tv_usec = (timeout_millis % 1000) * 1000;

      retval = select(dsp->codec.fd+1, &fdset, NULL, NULL, &tv);
      if(retval < 1) {
        DBG_PRINT("READ_CMD INTERRUPT. NO REPLY\n");
        return DSP_WAIT_INTERRUPT;
      }
    }

    if(read(dsp->codec.fd, data, 2*sizeof(unsigned short int)) < 0) {
      dsp->error_status = -1;
      dsp->error_cmd = DSP_CMD_NONE;
      goto read_cmd_error;
    }
    //DBG_PRINT("READ_CMD GOT REPLY: %d, status: %d\n", data[0], data[1]);

    dsp->error_cmd = data[0];

    if(data[0] == DSP_CMD_STATE) {
      dsp->info.stream_ID = data[1];
      if(read(dsp->codec.fd, &(dsp->info.ds_stream_ID),
         sizeof(AUDIO_STATUS_INFO) - 2 * sizeof(unsigned short int)) < 0)
      {
        dsp->error_status = -1;
        goto read_cmd_error;
      }
      dsp->info.num_frames = ((dsp->info.num_frames & 0x0000ffff) << 16) |
                             ((dsp->info.num_frames & 0xffff0000) >> 16);
      break;
    }

    if(data[1] != DSP_OK) {
     g_warning("READ_CMD got ERROR: cmd = 0x%X, status = 0x%0X",
               data[0], data[1]);

     dsp->error_status = data[1];

     // This error can never be returned with "normal" command
     if(dsp->error_status == DSP_ERROR_SYNC ||
        dsp->error_status == DSP_ERROR_STREAM)
     {
       if(gst_dspaudio_reset_node(dsp) == TRUE) {
          return DSP_WAIT_OK;
        }
     }
     goto read_cmd_error;
    }
   if(data[0] == DSP_CMD_DATA_WRITE) {
      if(read(dsp->codec.fd, &(dsp->dsp_buffer_free),
         sizeof(unsigned short int)) < 0)
      {
        dsp->error_status = -1;
        goto read_cmd_error;
      }
      dsp->rw_pending++;
      continue;
    }
    else if(data[0] == DSP_CMD_DATA_READ) {
      if(read(dsp->codec.fd, &(dsp->readinfo),
         sizeof(READ_STATUS) - 2*sizeof(unsigned short int)) < 0)
      {
        dsp->error_status = -1;
        goto read_cmd_error;
      }
      dsp->read_sent = FALSE;
      dsp->rw_pending++;
      continue;
    }
    // We got an "ordinary" command -> leave
    break;
  }
  return DSP_WAIT_OK;

read_cmd_error:
  dsp->mode = DSP_MODE_ERROR;
  DBG_PRINT("READ_CMD ERROR, cmd: %d, status: %d\n", dsp->error_cmd , dsp->error_status);
  return DSP_WAIT_ERROR;
}


/**
 * gst_dspaudio_wait_buffer:
 * @dsp: DSP audio object
 *
 * This method reads messages from DSP. It reads as many
 * messages as available. It waits especially for DATA_WRITE and
 * DATA_READ messages.
 *
 * Returns: DSPWaitStatus
 */
DSPWaitStatus
gst_dspaudio_wait_buffer(GstDSPAudio *dsp)
{
  DSP_CMD_STATUS incoming;

  if(dsp->mode == DSP_MODE_ERROR) {
    g_warning("Entered wait_buffer in error");
    return DSP_WAIT_ERROR;
  }
  else if(dsp->interrupted) {
    usleep(100*1000);
    return DSP_WAIT_INTERRUPT;
  }

  dsp->error_status = dsp->error_cmd = 0;

  //if(dsp->rw_pending) {
  //  dsp->rw_pending = 0;
  //  return TRUE;
  //}

  while(TRUE) {

    if(dsp->rw_pending) {
      dsp->rw_pending = 0;
      return DSP_WAIT_OK;
    }

    FD_SET(dsp->codec.fd, &dsp->read_fdset);
    FD_SET(dsp->pipe_fds[0], &dsp->read_fdset);
    select((dsp->codec.fd>dsp->pipe_fds[0]?dsp->codec.fd:dsp->pipe_fds[0])+1, &dsp->read_fdset, NULL, NULL, NULL);

    if(FD_ISSET(dsp->pipe_fds[0], &dsp->read_fdset)) {
      char tmp;
      dsp->error_status = dsp->error_cmd = 0;

      read(dsp->pipe_fds[0], &tmp, 1);
      if(tmp == PIPE_CMD_PROP_UPDATE) {
        gst_dspaudio_update_dsp_settings(dsp);
        if(dsp->error_status) {
          DBG_PRINT("UPDATE_ERROR: %d\n", dsp->error_status);
          return DSP_WAIT_ERROR;
        }
        continue;
      }
      else if(tmp == PIPE_CMD_INTERRUPT) {
        DBG_PRINT("PIPE_CMD_INTERRUPT\n");
        return DSP_WAIT_INTERRUPT;
      }
      else if(tmp == PIPE_CMD_NEED_LOCK) {
        //DBG_PRINT("PIPE_CMD_NEED_LOCK\n");
        return DSP_WAIT_RETRY;
      }
    }

    // Read the command
    if(read(dsp->codec.fd, &incoming, sizeof(unsigned short int)) < 0) {
      dsp->error_status = -1;
      dsp->error_cmd = DSP_CMD_NONE;
      goto read_buffer_error;
    }

    dsp->error_cmd = incoming.dsp_cmd;

    if(incoming.dsp_cmd == DSP_CMD_STATE) {
      if(read(dsp->codec.fd, &(dsp->info.stream_ID),
         sizeof(AUDIO_STATUS_INFO) - sizeof(unsigned short int)) < 0)
      {
        dsp->error_status = -1;
        goto read_buffer_error;
      }
    }
    else {

      // Read the command status value
      if(read(dsp->codec.fd, &(incoming.status), sizeof(unsigned short int)) < 0) {
        dsp->error_status = -1;
        goto read_buffer_error;
      }

      if(incoming.status != DSP_OK) {
        dsp->error_status = incoming.status;

        g_warning("WAIT_BUFFER: CMD: %d, ERROR: %d", incoming.dsp_cmd, incoming.status);
        if(incoming.status == DSP_ERROR_SYNC ||
           incoming.status == DSP_ERROR_STREAM)
        {

          if(gst_dspaudio_reset_node(dsp) == TRUE) {
            return DSP_WAIT_OK;
          }
        }
        goto read_buffer_error;
      }

      // Check for "special" commands
      if(incoming.dsp_cmd == DSP_CMD_DATA_WRITE) {
        if(read(dsp->codec.fd, &(dsp->dsp_buffer_free),
           sizeof(unsigned short int)) < 0)
        {
          dsp->error_status = -1;
          goto read_buffer_error;
        }
        dsp->rw_pending++;
      }
      else if(incoming.dsp_cmd == DSP_CMD_DATA_READ) {
        if(read(dsp->codec.fd, &(dsp->readinfo),
           sizeof(READ_STATUS) - 2*sizeof(unsigned short int)) < 0)
        {
          dsp->error_status = -1;
          goto read_buffer_error;
        }
        dsp->read_sent = FALSE;
        dsp->rw_pending++;
      }
    }

    // Read all possible messages from DSP
    if(gst_dspaudio_check_read(dsp->codec.fd)) {
      continue;
    }

    if(dsp->rw_pending) {
      dsp->rw_pending = 0;
      break;
    }
  }
  return TRUE;

read_buffer_error:
  dsp->mode = DSP_MODE_ERROR;
  DBG_PRINT("READ_BUFFER ERROR, cmd: %d, status: %d\n", dsp->error_cmd, dsp->error_status);
  return DSP_WAIT_ERROR;
}


/**
 * gst_dspaudio_play:
 * @dsp: DSP audio object
 *
 * Sends a PLAY command to DSP. After this the DSP will start sending
 * DSP_CMD_DATA_WRITE requests, that can be read with wait_buffer()
 * method.
 *
 * Returns: TRUE if the operation was successful, otherwise FALSE
 */
gboolean
gst_dspaudio_play(GstDSPAudio *dsp)
{
  DSP_CMD_STATUS cmd;
  gint ret;

  if(dsp==NULL) return FALSE;
  if(dsp->codec.fd==-1) return FALSE;
  if(dsp->mode==DSP_MODE_UNINITIALIZED) return FALSE;
  if(dsp->mode==DSP_MODE_ERROR) return FALSE;

  DBG_PRINT("SENDING PLAY To DSP\n");

  cmd.dsp_cmd = DSP_CMD_PLAY;
  ret = write(dsp->codec.fd, &cmd, sizeof(short int));
  if(ret == -1) {
    dsp->mode = DSP_MODE_ERROR;
    return FALSE;
  }

  GST_DEBUG("play successful");
  dsp->mode = DSP_MODE_PLAYING;
  dsp->rw_pending = 0;
  return TRUE;
}


/**
 * gst_dspaudio_pause:
 * @dsp: DSP audio object
 *
 * Sends a PAUSE command to DSP.
 *
 * Returns: TRUE if the operation was successful, otherwise FALSE
 */
gboolean
gst_dspaudio_pause(GstDSPAudio *dsp)
{
  DSP_CMD_STATUS status;
  gint ret;

  if(dsp==NULL) return FALSE;
  if(dsp->codec.fd==-1) return FALSE;
  if(dsp->mode==DSP_MODE_ERROR) return FALSE;
  if(dsp->mode==DSP_MODE_EOS) return FALSE;

  if(dsp->mode != DSP_MODE_PLAYING) {
    GST_DEBUG("not in playing mode -> cannot pause");
    return FALSE;
  }

  status.dsp_cmd = DSP_CMD_PAUSE;
  ret = write(dsp->codec.fd, &status, sizeof(short int));
  if(ret == -1) {
    dsp->mode = DSP_MODE_ERROR;
    return FALSE;
  }

  DBG_PRINT("NODE PAUSED\n");
  dsp->mode = DSP_MODE_PAUSED;
  return TRUE;
}


/**
 * gst_dspaudio_stop:
 * @dsp: DSP audio object
 *
 * Send a STOP command to DSP. This will stop the audio playback and
 * set the audio object mode into DSP_MODE_STOPPED.
 *
 * Returns: TRUE if the operation was successful, otherwise FALSE
 */
gboolean
gst_dspaudio_stop(GstDSPAudio *dsp)
{
  DSP_CMD_STATUS status;
  gint ret;

  if(dsp==NULL) return FALSE;
  if(dsp->codec.fd==-1) return FALSE;
  if(dsp->mode==DSP_MODE_ERROR) return FALSE;
  if(dsp->mode==DSP_MODE_EOS) return FALSE;

  DBG_PRINT("stop called\n");

  // We can stop only from PLAY / PAUSE mode
  if(dsp->mode == DSP_MODE_PLAYING || dsp->mode == DSP_MODE_PAUSED) {
    status.dsp_cmd = DSP_CMD_STOP;

    ret = write(dsp->codec.fd, &status, sizeof(short int));
    if(ret == -1) {
      dsp->mode = DSP_MODE_ERROR;
      return FALSE;
    }

    DBG_PRINT("stop successful\n");
    dsp->mode = DSP_MODE_STOPPED;
    return TRUE;
  }
  else {
    DBG_PRINT("not in playing mode -> cannot stop\n");
    return FALSE;
  }
}


/**
 * gst_dspaudio_wait_eos:
 * @dsp: DSP audio object
 *
 * Waits until EOS is received from DSP
 */
void
gst_dspaudio_wait_eos(GstDSPAudio *dsp)
{
  dsp->mode = DSP_MODE_EOS;
  while (TRUE) {
    DBG_PRINT("WAITING FOR EOS...\n");
    if(gst_dspaudio_get_info(dsp) == FALSE) break;
    if(gst_dspaudio_read_cmd(dsp, 0) != DSP_WAIT_OK) break;
    if(dsp->error_cmd == DSP_CMD_EOF) break;
    if(dsp->interrupted) break;

    DBG_PRINT("GOT CMD: %d\n", dsp->error_cmd);
    usleep(200000);
  }
}


/**
 * gst_dspaudio_discont:
 * @dsp: DSP audio object
 *
 * Informs the DSP about discontinuity of the stream. Must be used
 * when a seek operation is performed on a compressed stream.
 *
 * Returns: TRUE if the operation was successful, otherwise FALSE
 */
gboolean
gst_dspaudio_discont(GstDSPAudio *dsp)
{
  DSP_CMD_STATUS cmd;
  gint ret;

  DBG_PRINT("DSPAUDIO: DISCONT CALLED\n");

  if(dsp==NULL) return FALSE;
  if(dsp->codec.fd==-1) return FALSE;
  if(dsp->discont_sent) return FALSE;
  if(dsp->mode==DSP_MODE_ERROR) return FALSE;

  cmd.dsp_cmd = DSP_CMD_DISCONT;
  ret = write(dsp->codec.fd, &cmd, sizeof(short int));
    if(ret == -1) {
    dsp->mode = DSP_MODE_ERROR;
    return FALSE;
  }
  DBG_PRINT("DSPAUDIO: DISCONT SENT\n");
  dsp->discont_sent = TRUE;
  return TRUE;
}


/*
 * gst_dspaudio_set_volume:
 * @dsp: DSP audio object
 *
 * Sends the volume command to DSP. Volume value is retrieved from
 * "volume" attribute of the audio object. The value can be between
 * 0 - 0xFFFF.
 *
 * Returns: TRUE if the operation was successful, otherwise FALSE
 */
static gboolean
gst_dspaudio_set_volume(GstDSPAudio *dsp)
{
  VOLUME_DATA volume_data;
  gint ret;

  if(dsp==NULL) return FALSE;
  if(dsp->codec.fd==-1) return FALSE;
  if(dsp->mode==DSP_MODE_UNINITIALIZED) return FALSE;
  if(dsp->mode==DSP_MODE_ERROR) return FALSE;

  if(dsp->volume > DSP_VOLUME_MAX) {
    dsp->volume = DSP_VOLUME_MAX;
  }

  volume_data.dsp_cmd = DSP_CMD_SET_VOLUME;

  guint foo = dsp->volume / 0xCCC;  // foo scale: 0x00 - 0x14 --> 21 levels
  volume_data.scale = volumescaletable[foo];
  volume_data.power2 = volumepower2table[foo];

  GST_DEBUG("set_volume called: power2 = %d, scale = 0x%04X",
    volume_data.power2, volume_data.scale );

  ret = write(dsp->codec.fd, &volume_data, sizeof(volume_data));
  if(ret == -1) {
    dsp->mode = DSP_MODE_ERROR;
    return FALSE;
  }
  dsp->volume_changed = FALSE;
  GST_DEBUG("set_volume successful");
  return TRUE;
}


/*
 * gst_dspaudio_set_panning:
 * @dsp: DSP audio object
 *
 * Sends the balance command to DSP. Panning value is retrieved from
 * "panning" attribute of the audio object. This value must be between
 * -0x4000 (left panning) and +0x4000 (right panning)
 *
 * Returns: TRUE if the operation was successful, otherwise FALSE
 */
static gboolean
gst_dspaudio_set_panning(GstDSPAudio *dsp)
{
  PANNING_DATA panning_data;
  gint ret;

  if(dsp==NULL) return FALSE;
  if(dsp->codec.fd==-1) return FALSE;
  if(dsp->mode==DSP_MODE_UNINITIALIZED) return FALSE;
  if(dsp->mode==DSP_MODE_ERROR) return FALSE;

  if(dsp->panning > DSP_PANNING_MAX) {
    dsp->panning = DSP_PANNING_MAX;
  }
  else if(dsp->panning < DSP_PANNING_MIN) {
    dsp->panning = DSP_PANNING_MIN;
  }

  panning_data.dsp_cmd = DSP_CMD_SET_PANNING;
  panning_data.left_gain = DSP_PANNING_MAX - (dsp->panning>0 ? dsp->panning : 0);
  panning_data.right_gain = DSP_PANNING_MAX + (dsp->panning<0 ? dsp->panning : 0);
  panning_data.steps = 1;

  GST_DEBUG("set_panning called: left = 0x%04X, right = 0x%04X",
    panning_data.left_gain, panning_data.right_gain );

  ret = write(dsp->codec.fd, &panning_data, sizeof(panning_data));
  if(ret == -1) {
    dsp->mode = DSP_MODE_ERROR;
    return FALSE;
  }

  dsp->panning_changed = FALSE;
  GST_DEBUG("set_panning successful");
  return TRUE;
}


/*
 * gst_dspaudio_set_mute:
 * @dsp: DSP audio object
 *
 * Send MUTE/UNMUTE command to DSP. The value is retrieved from
 * "mute" attribute of the audio object.
 *
 * Returns: TRUE if the operation was successful, otherwise FALSE
 */
static gboolean
gst_dspaudio_set_mute(GstDSPAudio *dsp)
{
  DSP_CMD_STATUS status;
  gint ret;

  if(dsp==NULL) return FALSE;
  if(dsp->codec.fd==-1) return FALSE;
  if(dsp->mode==DSP_MODE_UNINITIALIZED) return FALSE;
  if(dsp->mode==DSP_MODE_ERROR) return FALSE;

  GST_DEBUG("set (un)mute called, value = %d", dsp->mute);

  status.dsp_cmd = dsp->mute ? DSP_CMD_MUTE : DSP_CMD_UNMUTE;

  ret = write(dsp->codec.fd, &status, sizeof(short int));
  dsp->mute_changed = FALSE;

  if(ret == -1) {
    dsp->mode = DSP_MODE_ERROR;
    return FALSE;
  }

  GST_DEBUG("set_mute successful");
  return TRUE;
}


/*
 * gst_dspaudio_reset_node:
 * @dsp: DSP audio object
 *
 * Resets the audio node by sending STOP and PLAY commands.
 *
 * Returns: TRUE if the operation was successful, otherwise FALSE
 */
static gboolean
gst_dspaudio_reset_node(GstDSPAudio *dsp)
{
  DBG_PRINT("RESET NODE\n");
  if(gst_dspaudio_stop(dsp)) {
    DBG_PRINT("STOP COMMAND SUCCESSFUL\n");
    gboolean ret = gst_dspaudio_play(dsp);
    DBG_PRINT("PLAY COMMAND: %d\n", ret);
    dsp->rw_pending = FALSE;
    return ret;
  }

  return FALSE;
}


/*
 * gst_dspaudio_mic_ctl:
 * @dsp: DSP audio object
 * @mic_enable: If TRUE, microphone input is turned on
 *
 * Control microphone input
 *
 * Returns: TRUE of operation was successful
 */
static gboolean
gst_dspaudio_mic_ctl(GstDSPAudio *dsp, gboolean mic_enable)
{
  DBusConnection *connection;
  DBusMessage *message;
  DBusMessage *reply = NULL;
  gchar *method;

  method = mic_enable ? "request" : "release";

  connection = (DBusConnection *)dsp->dbus_connection;
  if(connection != NULL) {
    message = dbus_message_new_method_call(AUDIO_PM_SERVICE,
                                           AUDIO_PM_RECORD_RESOURCE,
                                           RESOURCE_INTERFACE,
                                           method);
    if(message != NULL) {
      if(!mic_enable) {
        dbus_int32_t handle = 0;
        /* FIXME: append args for request() when defined (currently none,
         * but RM spec still undergoing some changes)
         * Also, this assumes that the resource handle is always 0 (true
         * for PM resources, but could also save the handle from request())
         */
        dbus_message_append_args(message,
                                 DBUS_TYPE_INT32, &handle,
                                 DBUS_TYPE_INVALID);
      }
      reply = dbus_connection_send_with_reply_and_block(connection, message,
                                                        RESOURCE_TIMEOUT,
                                                        NULL);
      /* TODO: check if reply indicates an error (currently doesn't matter
       * except for the debug output, as the return code is ignored anyway)
       * The return parameters might also have to changed a bit still in the
       * RM spec, so we accept anything for now */
      dbus_message_unref(message);
    }
  }
  if(reply == NULL) {
    DBG_PRINT("Mic power management %s failed\n", method);
    return FALSE;
  } else {
    dbus_message_unref(reply);
    return TRUE;
  }
}


/**
 * gst_dspaudio_update_dsp_settings:
 * @dsp DSP audio object
 *
 * Sets the volume, panning and mute values to DSP if they
 * are changed since last time.
 */
void
gst_dspaudio_update_dsp_settings(GstDSPAudio *dsp)
{
  if(dsp->volume_changed) {
    if(gst_dspaudio_set_volume(dsp)) {
      DBG_PRINT("DSP VOLUME SET: %d\n", dsp->volume);
      gst_dspaudio_read_cmd(dsp, 0);
    }
  }
  if(dsp->panning_changed) {
    if(gst_dspaudio_set_panning(dsp)) {
      DBG_PRINT("DSP PANNING SET: %d\n", dsp->panning);
      gst_dspaudio_read_cmd(dsp, 0);
    }
  }
  if(dsp->mute_changed) {
    /*
    // Muting is handled in dsp, and current mic_ctl() doesn't support it
    if(dsp->mic_enabled) {
      DBG_PRINT("MICROPHONE MUTE\n");
      gst_dspaudio_mic_ctl(TRUE, dsp->mute);
      dsp->mute_changed = FALSE;
    }
    else */ if(gst_dspaudio_set_mute(dsp)) {
      DBG_PRINT("DSP MUTE SET: %d\n", dsp->mute);
      gst_dspaudio_read_cmd(dsp, 0);
    }
  }
}


/*
 * gst_dspaudio_notify_render:
 *
 */
static void
gst_dspaudio_notify_render(GstDSPAudio *dsp)
{
  //DBG_PRINT("gst_dspaudio_notify_render: dsp->mode = %d\n", dsp->mode);
  if(dsp->mode == DSP_MODE_PLAYING) {
    char tmp = PIPE_CMD_PROP_UPDATE;
    DBG_PRINT("WRITING TO PIPE\n");
    write(dsp->pipe_fds[1], &tmp, 1);
    DBG_PRINT("WRITING TO PIPE OK\n");
  }
}


/**
 * gst_dspaudio_interrupt_render:
 *
 */
void
gst_dspaudio_interrupt_render(GstDSPAudio *dsp)
{
  //DBG_PRINT("gst_dspaudio_notify_render: dsp->mode = %d\n", dsp->mode);
  if(dsp->mode == DSP_MODE_PLAYING || dsp->mode == DSP_MODE_EOS) {
    char tmp = PIPE_CMD_INTERRUPT;
    DBG_PRINT("WRITING INTERRUPT CMD TO PIPE\n");
    dsp->interrupted = TRUE;
    write(dsp->pipe_fds[1], &tmp, 1);
    DBG_PRINT("WRITING TO PIPE OK\n");
  }
}


/**
 * gst_dspaudio_remove_interrupt:
 *
 */
void
gst_dspaudio_remove_interrupt(GstDSPAudio *dsp)
{
  char tmp;

  dsp->interrupted = FALSE;
  // Flush the "command pipeline"
  while(gst_dspaudio_check_read(dsp->pipe_fds[0])) {
    read(dsp->pipe_fds[0], &tmp,1);
    DBG_PRINT("FLUSHING PIPELINE\n");
  }
}


/**
 * gst_dspaudio_set_property:
 * @dsp: DSP audio object
 * @prop_id: ID value of the property
 * @value: Pointer that holds the value to be set
 *
 * Set a property into certain value
 *
 * Returns: TRUE if property was known, otherwise FALSE
 */
gboolean
gst_dspaudio_set_property (GstDSPAudio *dsp, guint prop_id, const GValue *value)
{
  gboolean ret = TRUE;

  if(dsp==NULL) return FALSE;

  switch (prop_id) {
    case DSPAUDIO_PROP_VOLUME:
      dsp->volume = g_value_get_uint(value);
      DBG_PRINT("SETTING VOLUME PROPERTY: %d\n", dsp->volume);
      if(!dsp->volume_changed) {
        dsp->volume_changed = TRUE;
        gst_dspaudio_notify_render(dsp);
      }
      break;
    case DSPAUDIO_PROP_PANNING:
      dsp->panning = g_value_get_int(value);
      DBG_PRINT("SETTING PANNING PROPERTY: %d\n", dsp->panning);
      if(!dsp->panning_changed) {
        dsp->panning_changed = TRUE;
        gst_dspaudio_notify_render(dsp);
      }
      break;
    case DSPAUDIO_PROP_MUTE:
      dsp->mute = g_value_get_boolean(value);
      DBG_PRINT("SETTING MUTE PROPERTY: %d\n", dsp->mute);
      if(!dsp->mute_changed) {
        DBG_PRINT("MUTE CHANGED NOW\n");
        dsp->mute_changed = TRUE;
        gst_dspaudio_notify_render(dsp);
      }
      break;
    case DSPAUDIO_PROP_STREAMID:
      // This is read-only
      break;
    case DSPAUDIO_PROP_PRIORITY:
      dsp->priority = g_value_get_int(value);
      DBG_PRINT("SETTING PRIORITY PROPERTY: %d\n", dsp->priority);
      break;
    default:
      ret = FALSE;
      break;
  }

  if(dsp->mode != DSP_MODE_PLAYING) {
    g_mutex_lock(dsp->dsp_mutex);
    gst_dspaudio_update_dsp_settings(dsp);
    g_mutex_unlock(dsp->dsp_mutex);
  }
  return ret;
}


/**
 * gst_dspaudio_get_property:
 * @dsp: DSP audio object
 * @prop_id: ID value of the property
 * @value: Pointer where to set the value
 *
 * Retrieve property value from audio plugin
 *
 * Returns: TRUE if property was known, otherwise FALSE
 */
gboolean
gst_dspaudio_get_property (GstDSPAudio *dsp, guint prop_id, GValue *value)
{
  gboolean ret = TRUE;

  if(dsp==NULL) return FALSE;

  switch (prop_id) {
    case DSPAUDIO_PROP_VOLUME:
      g_value_set_uint (value, dsp->volume);
      break;
    case DSPAUDIO_PROP_PANNING:
      g_value_set_int (value, dsp->panning);
      break;
    case DSPAUDIO_PROP_MUTE:
      g_value_set_boolean (value, dsp->mute);
      break;
    case DSPAUDIO_PROP_STREAMID:
      g_value_set_uint (value, dsp->codec.stream_id);
      break;
    case DSPAUDIO_PROP_PRIORITY:
      g_value_set_int (value, dsp->priority);
      break;
    default:
      ret = FALSE;
      break;
  }
  return ret;
}


/**
 * gst_dspaudio_install_properties:
 * @klass: The object where to set the properties
 *
 * Adds default parameters that are common to all audio plugins to object
 */
void
gst_dspaudio_install_properties (GObjectClass *klass)
{
  g_object_class_install_property (klass, DSPAUDIO_PROP_VOLUME,
      g_param_spec_uint ("volume", "Stream volume level",
          "Defines the volume level of the stream", 0, DSP_VOLUME_MAX,
          DSP_VOLUME_DEFAULT, G_PARAM_READWRITE));

  g_object_class_install_property (klass, DSPAUDIO_PROP_PANNING,
      g_param_spec_int ("panning", "Audio balance",
          "Defines the balance between left/right channels",
          DSP_PANNING_MIN, DSP_PANNING_MAX, DSP_PANNING_DEFAULT,
          G_PARAM_READWRITE));

  g_object_class_install_property (klass, DSPAUDIO_PROP_MUTE,
      g_param_spec_boolean ("mute", "Mute on/off",
          "Mutes or unmutes the stream",
          FALSE, G_PARAM_READWRITE));

  g_object_class_install_property (klass, DSPAUDIO_PROP_STREAMID,
      g_param_spec_uint ("stream_id", "Stream id",
          "Unique stream identifier value", 0, G_MAXUSHORT, 0,
          G_PARAM_READABLE));

  g_object_class_install_property (klass, DSPAUDIO_PROP_PRIORITY,
      g_param_spec_int ("priority", "Stream priority",
          "Defines the priority value of the stream",
          DSP_PRIORITY_MIN, DSP_PRIORITY_MAX, DSP_PRIORITY_DEFAULT,
          G_PARAM_READWRITE));

/*  g_object_class_install_property (klass, DSPAUDIO_PROP_PANIC,
      g_param_spec_boolean ("mute", "Mute on/off",
                            "Mutes or unmutes the stream",
                             FALSE, G_PARAM_READWRITE));*/
}


/**
 * gst_dspaudio_map_samplerate:
 * @rate: Source rate to be converted
 * @dest_rate: Pointer where to store DSP-specific rate definition
 *
 * Converts sample rate value into DSP-specific enumeration value
 *
 * Returns: TRUE if conversion was successful, otherwise FALSE
 */
gboolean
gst_dspaudio_map_samplerate(gint rate, gint *dest_rate)
{
  *dest_rate = 0xffff;

  switch(rate) {
    case 96000: *dest_rate = SAMPLE_RATE_96KHZ; break;
    case 88200: *dest_rate = SAMPLE_RATE_88_2KHZ; break;
    case 64000: *dest_rate = SAMPLE_RATE_64KHZ; break;
    case 48000: *dest_rate = SAMPLE_RATE_48KHZ; break;
    case 44100: *dest_rate = SAMPLE_RATE_44_1KHZ; break;
    case 32000: *dest_rate = SAMPLE_RATE_32KHZ; break;
    case 24000: *dest_rate = SAMPLE_RATE_24KHZ; break;
    case 22050: *dest_rate = SAMPLE_RATE_22_05KHZ; break;
    case 16000: *dest_rate = SAMPLE_RATE_16KHZ; break;
    case 12000: *dest_rate = SAMPLE_RATE_12KHZ; break;
    case 11025: *dest_rate = SAMPLE_RATE_11_025KHZ; break;
    case  8000: *dest_rate = SAMPLE_RATE_8KHZ; break;
  }

  return *dest_rate != 0xffff ? TRUE : FALSE;
}


/**
 * gst_dspaudio_query_peer_position:
 * @sinkpad: GstPad where to send the query
 * @value: guint64 pointer where to store the value
 *
 * Utility method for querying position from peer element
 *
 * Returns: TRUE if query succeeded, otherwise FALSE
 */
gboolean
gst_dspaudio_query_peer_position(GstPad *sinkpad, gint64 *value)
{
  GstFormat fmt = GST_FORMAT_TIME;
  gboolean ret;

  ret = gst_pad_query_position(GST_PAD_PEER(sinkpad), &fmt, value);
  if(ret && fmt == GST_FORMAT_TIME) {
    return TRUE;
  }
  return FALSE;
}


/**
 * gst_dspaudio_query_peer_duration:
 * @sinkpad: GstPad where to send the query
 * @value: guint64 pointer where to store the value
 *
 * Utility method for querying duration from peer element
 *
 * Returns: TRUE if query succeeded, otherwise FALSE
 */
gboolean
gst_dspaudio_query_peer_duration(GstPad *sinkpad, gint64 *value)
{
  GstFormat fmt = GST_FORMAT_TIME;
  gboolean ret;

  ret = gst_pad_query_duration(GST_PAD_PEER(sinkpad), &fmt, value);
  if(ret && fmt == GST_FORMAT_TIME) {
    return TRUE;
  }
  return FALSE;
}


/**
 * gst_dspaudio_get_info:
 * @dsp: DSP audio object
 *
 * Send a status query to DSP. The status reply needs to be
 * read with wait_buffer() call.
 *
 * Returns: TRUE, if operation wasa successful. Otherwise FALSE
 */
gboolean
gst_dspaudio_get_info(GstDSPAudio *dsp)
{
  if(dsp==NULL) return FALSE;
  if(dsp->codec.fd==-1) return FALSE;
  if(dsp->mode==DSP_MODE_ERROR) return FALSE;

  unsigned short int cmd = DSP_CMD_STATE;
  gint len;

  GST_DEBUG("gst_dspaudio_get_info");

  len = write(dsp->codec.fd, &cmd, sizeof(short int));
  if(len == -1) {
    dsp->error_cmd = DSP_CMD_STATE;
    dsp->mode = DSP_MODE_ERROR;
    dsp->error_status = -1;
    return FALSE;
  }
  return TRUE;
}


/**
 * gst_dspaudio_aep_initialize:
 * @dsp: DSP audio object
 *
 * Initializes DSP AEP node. This needs to be called before
 * any other method when using DSP.
 *
 * Returns: TRUE if operation was successful, otherwise FALSE
 */
gboolean
gst_dspaudio_aep_initialize(GstDSPAudio *dsp)
{
  if(gst_dspaudio_open_node(&dsp->aep, AEP_DEVICE_NAME) == FALSE) {
    dsp->mode = DSP_MODE_ERROR;
    return FALSE;
  }
  DBG_PRINT("AEP OPENED\n");
  return TRUE;
}


/**
 * gst_dspaudio_aep_close:
 * @dsp: DSP audio object
 *
 * Closes AEP node
 *
 * Returns: TRUE if operation was successful, otherwise FALSE
 */
gboolean
gst_dspaudio_aep_close(GstDSPAudio *dsp)
{
  return gst_dspaudio_close_node(&dsp->aep);
}


/**
 * gst_dspaudio_ed_microphone:
 * @dsp: DSPAudio object
 * @mic_enable: new mic status
 *
 * Enables or disables microphone input
 *
 * Returns: TRUE if operation was successful
 */
gboolean
gst_dspaudio_ed_microphone(GstDSPAudio *dsp, gboolean mic_enable)
{
  /* This used to access the old sysfs interface which could be written
   * to twice without problems. Is it possible that someone might do just
   * that (could result in a double resource allocation, and mic being
   * left in the powered-on state until the plugin exits, if not also
   * disabled twice) */
  gst_dspaudio_mic_ctl(dsp, mic_enable);

  /* Always return true (should continue anyway if resource manager is
   * not responding. This is to let the plugins work if RM is not in use
   * and mic power is on by default. Might be changed in future?) */
  return TRUE;
}


/**
 * gst_dspaudio_default_voice_frame_handler:
 * @mmap_ptr: Pointer to audio frame data
 * @fsize: Size of the incoming frame, in BYTES
 *
 * Default handler for voice frames. It just creates a GstBuffer and copies
 * the audio frame there. Some audio formats (e.g. G.729) need some processing
 * of the frame before it can be passed forward and they can override this
 * handler with their own.
 *
 * Returns: New GstBuffer containing the audio frame
 */
static GstBuffer *
gst_dspaudio_default_voice_frame_handler (short int *mmap_ptr, guint fsize)
{
    GstBuffer *newbuf = gst_buffer_new_and_alloc(fsize);
    memcpy(GST_BUFFER_DATA(newbuf), mmap_ptr, fsize);
    return newbuf;
}


/**
 * gst_dspaudio_default_sid_frame_handler:
 * @mmap_ptr: Pointer to SID_UPDATE frame data
 * @fsize: Size of the incoming frame, in BYTES
 *
 * Default handler for SID_UPDATE frames. Converts the DSP frame representation
 * into G.711 CN frame format. This also sets GST_BUFFER_FLAG_LAST flag in
 * in the buffer to mark it as CN buffer.
 *
 * Returns: New GstBuffer containing the audio frame
 */
static GstBuffer *
gst_dspaudio_default_sid_frame_handler (short int *mmap_ptr, guint fsize)
{
/* Currently the coefficients seem to cause distorted noise, so 
   they are not used. Only the "level" value is sent */

#if 0
    gint output_cn_size = DSP_ILBC_SID_UPDATE_FRAMESIZE / sizeof (short int);
    GstBuffer *newbuf = gst_buffer_new_and_alloc (output_cn_size + 2);
    char *data = GST_BUFFER_DATA (newbuf);
    gint i;

    /* Do transformation into G.711 appendix II A format */
    for (i=0; i < output_cn_size; i++) {
        data[i] = (char) mmap_ptr[i];
    }
#else
    GstBuffer *newbuf = gst_buffer_new_and_alloc (1);
    char *data = GST_BUFFER_DATA (newbuf);
    data[0] = mmap_ptr[0]; /* Copy only the level value */
#endif

    GST_BUFFER_FLAG_SET(newbuf, GST_BUFFER_FLAG_LAST);
    return newbuf;
}


/**
 * gst_dspaudio_read_frame:
 * @dsp: DSPAudio object
 * @framesize: size of the frame to be read, in BYTES
 * @ufs: size of the SID_UPDATE frame, in BYTES
 * @need_cn: TRUE if the plugin wants to generate comfort noise packets
 *
 * Read one audio frame from DSP.
 * 
 * Returns: GstBuffer containing the frame
 */
GstBuffer *
gst_dspaudio_read_frame(GstDSPAudio *dsp, guint framesize, guint ufs,
                        gboolean need_cn)
{
  short int tmp = DSP_CMD_DATA_READ;
  short int *mmap_ptr = (short int *) dsp->codec.mmap_buffer;
  GstBuffer *newbuf = NULL;
  DSPWaitStatus status;

  do {
    if(dsp->all_frames_read) {
      g_mutex_lock(dsp->dsp_mutex);
      status = gst_dspaudio_wait_buffer(dsp);
      g_mutex_unlock(dsp->dsp_mutex);
      if(status == DSP_WAIT_INTERRUPT) {
        // Engine is running the pipeline down. We need to exit now
        DBG_PRINT("READ_DTX_FRAME EXIT\n");
        return NULL;
      }
      else if(status == DSP_WAIT_RETRY) {
        continue;
      }
      else if(status == DSP_WAIT_ERROR) {
        dsp->mode = DSP_MODE_ERROR;
        return NULL;
      }
//       DBG_PRINT("FRAMESIZE: %d\n", dsp->readinfo.frame_size);
      dsp->all_frames_read = FALSE;
      dsp->readindex = 0;
    }

    switch(mmap_ptr[dsp->readindex++]) {
      case VOICE_FRAME:
        //DBG_PRINT("VOICE FRAME\n");
        newbuf = dsp->handle_voice_frame (&(mmap_ptr[dsp->readindex]), framesize);
        dsp->readindex += (framesize / sizeof(short int));
        break;
      case SID_FRAME_UPDATE:
        //DBG_PRINT("SID_FRAME_UPDATE\n");
        if (need_cn) {
            newbuf = dsp->handle_sid_frame (&(mmap_ptr[dsp->readindex]), ufs);
        }
        else {
            newbuf = gst_buffer_new ();
        }
        dsp->readindex += (ufs / sizeof(short int));
        break;
      case SID_FRAME_NO_UPDATE:
        //DBG_PRINT("SID_FRAME_NO_UPDATE\n");
        newbuf = gst_buffer_new ();
        break;
      default:
        DBG_PRINT("Unknown header: %d\n", mmap_ptr[dsp->readindex-1]);
    }

    if(dsp->readindex == dsp->readinfo.frame_size) {
      if(write(dsp->codec.fd, &tmp, sizeof(short int)) < 0) {
        dsp->mode = DSP_MODE_ERROR;
      }
      dsp->all_frames_read = TRUE;
      dsp->read_sent = TRUE;
    }
  }
  while (FALSE);

  return newbuf;
}


/**
 * gst_dspaudio_clock_change_state:
 *
 */
GstStateChangeReturn
gst_dspaudio_clock_change_state(GstDSPAudio *dsp, GstStateChange transition)
{
  switch (transition) {

  case GST_STATE_CHANGE_PLAYING_TO_PAUSED:
    /* Disable interpolating clock time using system clock *
     * until next clock time is read from dsp */
    dsp->last_dsp_update = 0;
    break;
  case GST_STATE_CHANGE_NULL_TO_READY:
    dsp->clock_time = 0;
    dsp->last_dsp_update = 0;
  default:
    break;
  }

  return GST_STATE_CHANGE_SUCCESS;
}


/*
 * gst_dspaudio_clock_interrupt:
 *
 */
static gboolean
gst_dspaudio_clock_interrupt(GstDSPAudio *dsp)
{
  gboolean ret = FALSE;

  if(!dsp->interrupted) {
    char tmp = PIPE_CMD_NEED_LOCK;
    write(dsp->pipe_fds[1], &tmp, 1);
    g_mutex_lock(dsp->dsp_mutex);
    if (gst_dspaudio_get_info(dsp)) {
      if(gst_dspaudio_read_cmd(dsp, 0) == DSP_WAIT_OK) {
        ret = TRUE;
      }
    }
    g_mutex_unlock(dsp->dsp_mutex);
  }
  return ret;
}


/**
 * gst_dspaudio_get_dsp_time:
 *
 */
static GstClockTime
gst_dspaudio_get_dsp_time(GstDSPAudio *dsp)
{
  guint64 time64 = 0;

  // If mutex is not locked, we can freely ask the time from DSP
  if(g_mutex_trylock(dsp->dsp_mutex)) {
    if (gst_dspaudio_get_info(dsp)) {
      if(gst_dspaudio_read_cmd(dsp, 0) != DSP_WAIT_OK) {
        g_mutex_unlock(dsp->dsp_mutex);
        return GST_CLOCK_TIME_NONE;
      }
    }
    g_mutex_unlock(dsp->dsp_mutex);
  }
  else {
    // Mutex is locked. Probably because of the _wait_buffer().
    // In this case we can ask it to interrupt reading and unlock
    // the mutex
    if(!gst_dspaudio_clock_interrupt(dsp)) {
      return GST_CLOCK_TIME_NONE;
    }
  }

  dsp->info.samples_played_low = ( dsp->info.samples_played_low << 16 )|( dsp->info.samples_played_low >> 16 );
  dsp->info.samples_played_high = ( dsp->info.samples_played_high << 16 )|( dsp->info.samples_played_high >> 16 );
  time64 = dsp->info.samples_played_high;
  time64 <<= 32;
  time64 |= dsp->info.samples_played_low;

  dsp->clock_time = time64 * GST_MSECOND;
  //DBG_PRINT("DSP CLOCK TIME: %lld (%lld secs)\n", dsp->clock_time, dsp->clock_time/GST_SECOND);
  return dsp->clock_time;

}


/**
 * gst_dspaudio_get_dsp_clock:
 * @dsp: DSPAudio object
 *
 * Implementation to calculate clock based on DSP Clock
 *
 * Returns: Clock
 */
GstClockTime
gst_dspaudio_get_dsp_clock(GstDSPAudio *dsp)
{
  GstClockTime systime_abs = 0, ret = dsp->clock_time;
  GTimeVal systime;

  // Using dsp clock only after setting dsp to playing state.
  if (dsp->mode == DSP_MODE_PLAYING || dsp->mode == DSP_MODE_EOS) {

    g_get_current_time (&systime);
    systime_abs = GST_TIMEVAL_TO_TIME (systime);

    //DBG_PRINT("DSPAUDIO elapsed time since last update: %lld ms\n",
    //          ((systime_abs - dsp->last_dsp_update)/GST_MSECOND));

    // This function might get called from multiple threads simultaneously
    // so we need to restrict the access to DSP clock reading
    gboolean got_lock = g_mutex_trylock(dsp->clock_mutex);

    if (dsp->mode == DSP_MODE_PLAYING &&
        systime_abs > dsp->last_dsp_update + (50 * GST_MSECOND) &&
        got_lock)
    {
      // Update clock time from dsp
      // If the time query fails, this will return the "old" DSP time.
      // We will not interpolate it by using system time in this case.
      ret = gst_dspaudio_get_dsp_time(dsp);
      if(ret != GST_CLOCK_TIME_NONE) {
        dsp->last_dsp_update = systime_abs;
      }
    }
    else if (dsp->last_dsp_update) {
      // Update clock time using system clock
      ret = dsp->clock_time + systime_abs - dsp->last_dsp_update;
    }
    if(got_lock) g_mutex_unlock(dsp->clock_mutex);
  }
  DBG_PRINT("DSPAUDIO return clock: %"GST_TIME_FORMAT"\n", GST_TIME_ARGS(ret));
  return ret;
}

/**
 * gst_dspaudio_map_dtmf:
 * @tone_char: character to be mapped
 * @tone_id: output parameter for DSP tone ID
 * 
 * Map the given character to DSP tone ID
 * Returns: TRUE if mapping found, otherwise FALSE.
 */
gboolean
gst_dspaudio_map_dtmf(gint tone_id, unsigned short int *dsp_tone_id)
{
  gboolean ret = TRUE;

  switch(tone_id) {
    case 0:  *dsp_tone_id = DTMF_0; break;
    case 1:  *dsp_tone_id = DTMF_1; break;
    case 2:  *dsp_tone_id = DTMF_2; break;
    case 3:  *dsp_tone_id = DTMF_3; break;
    case 4:  *dsp_tone_id = DTMF_4; break;
    case 5:  *dsp_tone_id = DTMF_5; break;
    case 6:  *dsp_tone_id = DTMF_6; break;
    case 7:  *dsp_tone_id = DTMF_7; break;
    case 8:  *dsp_tone_id = DTMF_8; break;
    case 9:  *dsp_tone_id = DTMF_9; break;
    case 10: *dsp_tone_id = DTMF_STAR; break;
    case 11: *dsp_tone_id = DTMF_HASH; break;
    case 12: *dsp_tone_id = DTMF_A; break;
    case 13: *dsp_tone_id = DTMF_B; break;
    case 14: *dsp_tone_id = DTMF_C; break;
    case 15: *dsp_tone_id = DTMF_D; break;
    default: ret = FALSE; break;
  }
  return ret;
}


/**
 * gst_dspaudio_dtmf_start:
 * @dsp: DSP audio object
 * @tone_char: DTMF tone character to be sent
 *
 * Send DTMF_START command to DSP
 *
 * Returns: TRUE if the operation was successful, otherwise FALSE
 */
gboolean
gst_dspaudio_dtmf_tone_start(GstDSPAudio *dsp, gint tone_id)
{
  TONE_STATUS status;

  if(dsp==NULL) return FALSE;
  if(dsp->codec.fd==-1) return FALSE;
  if(dsp->mode==DSP_MODE_ERROR) return FALSE;

  if(dsp->mode != DSP_MODE_PLAYING) {
    GST_DEBUG("not in playing mode\n");
    return FALSE;
  }

  if(dsp->dtmf_playing) {
    // Stop the previous tone before playing new
    DBG_PRINT("turning off old tone\n");
    gst_dspaudio_dtmf_tone_stop(dsp);
  }

  if(gst_dspaudio_map_dtmf(tone_id, &status.tone_id)) {
    DBG_PRINT("starting DTMF tone: %d\n", status.tone_id);
    status.dsp_cmd = DSP_CMD_DTMF_START;
    status.amplitude = 29205; // FIXME: Hardcoded value
    if(write(dsp->codec.fd, &status, sizeof(TONE_STATUS)) < 0) {
        dsp->mode = DSP_MODE_ERROR;
        return FALSE;
    }
    gst_dspaudio_read_cmd(dsp, 0);
    dsp->dtmf_playing = TRUE;
  }
  DBG_PRINT("tone command sent\n");
  return TRUE;
}


/**
 * gst_dspaudio_dtmf_stop:
 * @dsp: DSP audio object
 *
 * Send DTMF_STOP command to DSP
 *
 * Returns: TRUE if the operation was successful, otherwise FALSE
 */

gboolean
gst_dspaudio_dtmf_tone_stop(GstDSPAudio *dsp)
{
  DSP_CMD_STATUS status;

  if(dsp==NULL) return FALSE;
  if(dsp->codec.fd==-1) return FALSE;
  if(dsp->mode==DSP_MODE_ERROR) return FALSE;

  if(dsp->dtmf_playing) {
    DBG_PRINT("stopping DTMF tone\n");
    status.dsp_cmd = DSP_CMD_DTMF_STOP;
    if(write(dsp->codec.fd, &status, sizeof(short int)) < 0) {
        dsp->mode = DSP_MODE_ERROR;
        return FALSE;
    }
    dsp->dtmf_playing = FALSE;
    gst_dspaudio_read_cmd(dsp, 0);
    DBG_PRINT("DTMF tone stopped\n");
  }
  return TRUE;
}


/**
 * gst_dspaudio_check_upstream_event:
 * @dsp: DSP audio object
 * @event: GstEvent to be checked
 *
 * Check if the event is DTMF event
 *
 * Returns: TRUE if the event was handled, otherwise FALSE.
 */

gboolean
gst_dspaudio_check_upstream_event(GstDSPAudio *dsp, GstEvent *event)
{
  const GstStructure *structure = gst_event_get_structure (event);
  if(structure && gst_structure_has_name (structure, "dtmf-event")) {
    gboolean start = FALSE;
    guint method = 0;
    gint tone_id = -1;
    gint event_type;

    DBG_PRINT("got DTMF event\n");
    if (gst_structure_get_int (structure, "type", &event_type) &&
        gst_structure_get_boolean (structure, "start", &start) &&
        event_type == 1)
    {
      if (gst_structure_get_int (structure, "method", &method) &&
          method != 2)
      {
          DBG_PRINT("Wrong DTMF method (not in-band)\n");
          return FALSE;
      }
      g_mutex_lock(dsp->dsp_mutex);
      if(start) {
          if(gst_structure_get_int (structure, "number", &tone_id)) {
              gst_dspaudio_dtmf_tone_start(dsp, tone_id);
          }
      }
      else {
          gst_dspaudio_dtmf_tone_stop(dsp);
      }
      g_mutex_unlock(dsp->dsp_mutex);
      DBG_PRINT("DTMF event handled\n");
      return TRUE;
    }
  }
  return FALSE;
}


/*
 * This function is there so that caps negotiation is done properly
 */
gboolean
gst_dspaudio_activate_push (GstPad * pad, gboolean active)
{
  GstCaps *thiscaps;
  GstCaps *caps = NULL;
  GstCaps *peercaps = NULL;
  gboolean result = FALSE;

  DBG_PRINT ("activate push\n");

  if (!active)
    return TRUE;

  /* first see what is possible on our source pad */
  thiscaps = gst_pad_get_caps (pad);
  DBG_PRINT ("caps of src: %" GST_PTR_FORMAT, thiscaps);
  /* nothing or anything is allowed, we're done */
  if (thiscaps == NULL)
    goto no_nego_needed;

  /* get the peer caps */
  peercaps = gst_pad_peer_get_caps (pad);
  DBG_PRINT ("caps of peer: %" GST_PTR_FORMAT, peercaps);
  if (peercaps) {
    GstCaps *icaps;

    /* get intersection */
    icaps = gst_caps_intersect (thiscaps, peercaps);
    DBG_PRINT ("intersect: %" GST_PTR_FORMAT, icaps);
    gst_caps_unref (thiscaps);
    gst_caps_unref (peercaps);
    if (icaps) {
      /* take first (and best, since they are sorted) possibility */
      caps = gst_caps_copy_nth (icaps, 0);
      gst_caps_unref (icaps);
    }
  } else {
    /* no peer, work with our own caps then */
    caps = thiscaps;
  }
  if (caps) {
    caps = gst_caps_make_writable (caps);
    gst_caps_truncate (caps);

    /* now fixate */
    if (!gst_caps_is_empty (caps)) {
      gst_pad_fixate_caps (pad, caps);
      DBG_PRINT ("fixated to: %" GST_PTR_FORMAT, caps);

      if (gst_caps_is_any (caps)) {
        /* hmm, still anything, so element can do anything and
         * nego is not needed */
        result = TRUE;
      } else if (gst_caps_is_fixed (caps)) {
        /* yay, fixed caps, use those then */
        gst_pad_set_caps (pad, caps);
        result = TRUE;
      }
    }
    gst_caps_unref (caps);
  }
  return result;

no_nego_needed:
  {
    DBG_PRINT ("no negotiation needed");
    if (thiscaps)
      gst_caps_unref (thiscaps);
    return TRUE;
  }
}

