/*
 * This file is part of libgst0.10-dsp
 *
 * Copyright (C) 2006 Nokia Corporation. All rights reserved.
 *
 * Contact: Stefan Kost <stefan.kost@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <string.h>
#include <time.h>
#include <unistd.h>
#include <gst/gst.h>
#include <sys/time.h>

#include "gstdspamrsink.h"

#define MULTIPLE_PACKETS 10

#define AMR_FPS 50
#define AMR_TPF (GST_SECOND/AMR_FPS)

// If we find more than AVG_LEN_CALC_THRESHOLD frames in calc_length() function,
// we will start using an estimation algorithm instead of accurate length
// 5 minutes seems to be good value :)

#define AVG_LEN_CALC_THRESHOLD (5 * 60 * AMR_FPS)

static const short int FrameLength[2][16] =
{
  {13, 14, 16, 18, 20, 21, 27, 32,  6, 1, 1, 1, 1, 1, 1, 1},    // Narrow-band
  {18, 24, 33, 37, 41, 47, 51, 59, 61, 6, 6, 1, 1, 1, 1, 1}     // Wideband
};

static guint gst_dspamrsink_signals[DSPAMR_SIGNAL_LAST] = { 0 };

static const gchar *devname[2] =
{
  "/dev/dsptask/amrnb",
  "/dev/dsptask/amrwb"
};


static GstStaticPadTemplate dspamrsink_sink_template =
GST_STATIC_PAD_TEMPLATE ("sink",
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS ( "audio/x-amr-nb-sh; "
                      "audio/x-amr-wb-sh ") );


// Function prototypes

static void gst_dspamrsink_class_init (GstDSPAMRSinkClass * klass);
static void gst_dspamrsink_base_init (gpointer g_class);
static void gst_dspamrsink_init (GstDSPAMRSink * dspamrsink, GstDSPAMRSinkClass *g_class);

static void gst_dspamrsink_set_property (GObject * object, guint prop_id, const GValue * value, GParamSpec * pspec);
static void gst_dspamrsink_get_property (GObject * object, guint prop_id, GValue * value, GParamSpec * pspec);
static const GstQueryType *gst_dspamrsink_get_query_types (GstElement * element);
static gboolean gst_dspamrsink_query (GstElement * element, GstQuery *query);
static GstClock *gst_dspamrsink_provide_clock (GstElement * elem);
static GstClockTime gst_dspamrsink_get_time (GstClock * clock, gpointer sink);
static gboolean gst_dspamrsink_event (GstElement * element, GstEvent *event);
static gboolean gst_dspamrsink_initialize(GstDSPAMRSink *dsp);
static GstStateChangeReturn gst_dspamrsink_change_state(GstElement * element, GstStateChange transition);
static gboolean gst_dspamrsink_paused_to_playing(GstDSPAMRSink *dsp);

// For basesink
static GstFlowReturn gst_dspamrsink_preroll(GstBaseSink *sink, GstBuffer * buffer);
static GstFlowReturn gst_dspamrsink_render(GstBaseSink *sink, GstBuffer * buffer);
static gboolean gst_dspamrsink_event2 (GstBaseSink *sink, GstEvent *event);
static gboolean gst_dspamrsink_setcaps (GstBaseSink * basesink, GstCaps * caps);
static GstStateChangeReturn gst_dspamrsink_async_play(GstBaseSink *sink);
// static void gst_dspamrsink_calculate_length(GstDSPAMRSink *dsp);


#ifdef DEBUG
static void _do_init()
{
  DBG_PRINT("AMR DO_INIT\n");
}

GST_BOILERPLATE_FULL (GstDSPAMRSink, gst_dspamrsink, GstBaseSink,
                      GST_TYPE_BASE_SINK, _do_init);
#else
GST_BOILERPLATE (GstDSPAMRSink, gst_dspamrsink, GstBaseSink,
                 GST_TYPE_BASE_SINK);
#endif


/*
 * gst_dspamrsink_dispose:
 * @object: GObject pointer to element to be deleted
 *
 * Deletes AMR sink element instance. Called automatically by
 * GLib framework when element needs to be disposed.
 */
static void
gst_dspamrsink_dispose (GObject * object)
{
  GstDSPAMRSink *dsp = (GstDSPAMRSink *) object;

  if(dsp->clock) {
    gst_object_unref(dsp->clock);
    dsp->clock = NULL;
  }

  if(dsp->audio) {
    gst_dspaudio_destroy(dsp->audio);
    dsp->audio = NULL;
  }
  g_object_unref(dsp->adapter);

  G_OBJECT_CLASS (parent_class)->dispose (object);
}


/*
 * gst_dspamrsink_base_init:
 * @g_class: AMR sink class
 *
 * Does the basic initialization of AMR sink element class. This is
 * called by GStreamer framework.
 */
static void
gst_dspamrsink_base_init (gpointer g_class)
{
  GstElementClass *gstelement_class = GST_ELEMENT_CLASS (g_class);
  static const GstElementDetails details = GST_ELEMENT_DETAILS (
    "DSP AMR Sink",
    "Sink/Audio",
    "AMR NB/WB audio sink",
    "Makoto Sugano <makoto.sugano@nokia.com>");

  gst_element_class_add_pad_template (gstelement_class,
      gst_static_pad_template_get (&dspamrsink_sink_template));
  gst_element_class_set_details (gstelement_class, &details);

}


/*
 * gst_dspamrsink_class_init:
 * @klass: AMR Sink class object
 *
 * Initializes AMR sink element class. This is called by GStreamer
 * framework.
 */
static void
gst_dspamrsink_class_init (GstDSPAMRSinkClass * klass)
{
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;
  GstBaseSinkClass *gstbase_sink_class;

  gobject_class = (GObjectClass *) klass;
  gstelement_class = (GstElementClass *) klass;
  gstbase_sink_class = (GstBaseSinkClass *) klass;

  gobject_class->set_property = gst_dspamrsink_set_property;
  gobject_class->get_property = gst_dspamrsink_get_property;
  gobject_class->dispose = gst_dspamrsink_dispose;

  gstelement_class->provide_clock = GST_DEBUG_FUNCPTR (gst_dspamrsink_provide_clock);
  gstelement_class->change_state = gst_dspamrsink_change_state;
  gstelement_class->get_query_types = gst_dspamrsink_get_query_types;
  gstelement_class->query = gst_dspamrsink_query;
  gstelement_class->send_event = gst_dspamrsink_event;

  // Use dspaudio utility to install default properties
  gst_dspaudio_install_properties(G_OBJECT_CLASS (klass));

  gst_dspamrsink_signals[DSPAMR_SIGNAL_STREAMID] =
      g_signal_new ("stream_id", G_TYPE_FROM_CLASS (klass), G_SIGNAL_RUN_FIRST,
                    G_STRUCT_OFFSET (GstDSPAMRSinkClass, stream_id), NULL, NULL,
                    g_cclosure_marshal_VOID__INT, G_TYPE_NONE, 1, G_TYPE_INT);

  gstbase_sink_class->preroll = GST_DEBUG_FUNCPTR (gst_dspamrsink_preroll);
  gstbase_sink_class->render = GST_DEBUG_FUNCPTR (gst_dspamrsink_render);
  gstbase_sink_class->set_caps = GST_DEBUG_FUNCPTR (gst_dspamrsink_setcaps);
  gstbase_sink_class->event = GST_DEBUG_FUNCPTR (gst_dspamrsink_event2);
  gstbase_sink_class->async_play = GST_DEBUG_FUNCPTR(gst_dspamrsink_async_play);
}


/*
 * gst_dspamrsink_init:
 * @dsp: DSP AMR sink object
 *
 * Initializes the given AMR sink element instance and allocates required
 * resources. This also creates an audio stream object. This is
 * called by GStreamer framework.
 */
static void
gst_dspamrsink_init (GstDSPAMRSink * dspamrsink, GstDSPAMRSinkClass *g_class)
{
  GstBaseSink *bsink = GST_BASE_SINK(dspamrsink);

  dspamrsink->sinkpad = bsink->sinkpad;
  dspamrsink->audio = gst_dspaudio_new();
  dspamrsink->adapter = gst_adapter_new();
  gst_base_sink_set_sync (GST_BASE_SINK (dspamrsink), FALSE);

  dspamrsink->clock = gst_audio_clock_new ("clock",
                                           (GstAudioClockGetTimeFunc) gst_dspamrsink_get_time,
                                           dspamrsink);
}


/*
 * gst_dspamrsink_setcaps:
 * @basesink: GstBaseSink
 * @caps: GstCaps that were offered by the connecting element
 *
 * Returns: TRUE if caps can be accepted, otherwise FALSE
 */
static gboolean
gst_dspamrsink_setcaps(GstBaseSink * basesink, GstCaps * caps)
{
  GstDSPAMRSink *dsp = GST_DSPAMRSINK (basesink);
  DBG_PRINT("gst_dspamrsink_setcaps\n");

  GstStructure *structure = gst_caps_get_structure (caps, 0);
  const gchar *mimetype = gst_structure_get_name (structure);

  if(!strncmp(mimetype, "audio/x-amr-nb-sh", 17)) {
    dsp->read_mime_header = FALSE;
    dsp->wide = 0;
  }
  else if(!strncmp(mimetype, "audio/x-amr-wb-sh", 17)) {
    dsp->read_mime_header = FALSE;
    dsp->wide = 1;
  }
  else {
    DBG_PRINT("UNKNOWN CAPS\n");
    return FALSE;
  }

  dsp->bandindex = dsp->wide ? 1 : 0;
  return TRUE;
}


/*
 * gst_dspamrsink_send_discont:
 *
 */
static void
gst_dspamrsink_send_discont(GstDSPAMRSink *dsp)
{
  if(gst_dspaudio_discont(dsp->audio)) {
    DBG_PRINT("DISCONT written... reading reply\n");
    if(gst_dspaudio_read_cmd(dsp->audio, 0) != DSP_WAIT_OK) {
      GST_ELEMENT_ERROR (dsp, RESOURCE, SEEK, (NULL), ("discont"));
    }
    DBG_PRINT("GOT REPLY\n");
  }
}


/*
 * gst_dspamrsink_fill_input_buffer;
 * @dsp: GstDSPAMRSink to be handled
 *
 * This method fills the MMAP buffer with appropriate amount
 * of data according to the header information.
 *
 * Returns: TRUE if buffer was filled, otherwise FALSE
 */
static gboolean
gst_dspamrsink_fill_input_buffer(GstDSPAMRSink *dsp)
{
  guint frame_index, frame_length;
  const guint8 *data;
  char *outbuf;
  guint avail;

  avail = gst_adapter_available(dsp->adapter);

  while (dsp->packet_index < MULTIPLE_PACKETS) {

    // Read frame header
    if(avail < 1) {
      return FALSE;
    }

    data = gst_adapter_peek(dsp->adapter, 1);

    // Check the sync!
    if( (data[0] & 0x83) != 0 ) {
      gst_adapter_flush(dsp->adapter, 1);
//       DBG_PRINT("FIND SYNC curr_frame = %d\n", dsp->current_framecount);
      avail--;
      continue;
    }

    frame_index = (data[0] >> 3) & 0x0F;
    frame_length = FrameLength[dsp->bandindex][frame_index];
    if(avail < frame_length+1) {
      return FALSE;
    }

    data = gst_adapter_peek(dsp->adapter, frame_length+1);

    // Check the next frame header. If a header is not found, this
    // is not a real frame - it just looks like a frame header
    if( (data[frame_length] & 0x83) != 0 ) {
      gst_adapter_flush(dsp->adapter, 1);
//       DBG_PRINT("FIND SYNC curr_frame = %d\n", dsp->current_framecount);
      avail--;
      continue;
    }

    outbuf = dsp->audio->codec.mmap_buffer +
        (dsp->packet_index * (60-dsp->bandindex)) * sizeof(short int);

    // Normal frame must start with 0x0000
    *outbuf++ = 0x00;
    *outbuf++ = 0x00;

    memcpy(outbuf, data, frame_length);
    dsp->current_bytes += frame_length;
    dsp->current_framecount++;
    avail -= frame_length;
    dsp->packet_index++;
    gst_adapter_flush(dsp->adapter, frame_length);
  }
  return TRUE;
}


/*
 * gst_dspamrsink_perform_seek:
 *
 */
static gboolean
gst_dspamrsink_perform_seek(GstDSPAMRSink *dsp, gint64 seek_pos)
{
  guint64 pos_in_frames = seek_pos / AMR_TPF;
  gint64 pos = -1;
  GstFormat fmt = GST_FORMAT_BYTES;

  if(dsp->total_framecount) {
    pos = dsp->total_length_bytes * pos_in_frames / dsp->total_framecount;
  }
  else if (dsp->current_framecount) {
    pos = pos_in_frames * (dsp->current_bytes / dsp->current_framecount);
  }

  if(pos != -1) {
    dsp->current_framecount = pos_in_frames;
    dsp->current_bytes = pos;

    DBG_PRINT("SEEKING POSITION: %lld bytes (%lld secs)\n", pos, seek_pos/GST_SECOND);

    GstEvent *new_event = gst_event_new_seek(1.0, fmt, GST_SEEK_FLAG_FLUSH,
                                              GST_SEEK_TYPE_SET, pos,
                                              GST_SEEK_TYPE_NONE,
                                              GST_CLOCK_TIME_NONE);

    if(gst_pad_push_event (dsp->sinkpad, new_event)) {
      DBG_PRINT("<<<<<<<SEEK SUCCESSFUL>>>>>>>>>\n");
      gst_adapter_clear(dsp->adapter);
      dsp->packet_index  = 0;
      return TRUE;
    }
  }
  return FALSE;
}


/*
 * gst_dspamrsink_render:
 * @sink:
 * @buffer:
 *
 */
static GstFlowReturn
gst_dspamrsink_render(GstBaseSink *sink, GstBuffer * buffer)
{
  DSPWaitStatus ret = DSP_WAIT_OK;
  gint done = 0;

  GstDSPAMRSink *dsp = GST_DSPAMRSINK (sink);

  gst_buffer_ref(buffer);
  gst_adapter_push(dsp->adapter, buffer);
//   DBG_PRINT("AMR : Got buffer of %d bytes, total %d\n",
//             GST_BUFFER_SIZE(buffer), gst_adapter_available(dsp->adapter));

  // Send initial PLAY command if not done yet
  if(dsp->audio->mode < DSP_MODE_PLAYING) {
    g_mutex_lock(dsp->audio->dsp_mutex);
    if( !gst_dspamrsink_paused_to_playing(dsp) ) {
      g_mutex_unlock(dsp->audio->dsp_mutex);
      GST_ELEMENT_ERROR (sink, STREAM, TYPE_NOT_FOUND, (NULL),
                         ("gst_dspamrsink_paused_to_playing"));
      DBG_PRINT("RENDER RETURNING GST_FLOW_ERROR\n");
      return GST_FLOW_ERROR;
    }
    g_mutex_unlock(dsp->audio->dsp_mutex);
  }

  // Consume the incoming buffer
  while(TRUE) {

    g_mutex_lock(dsp->audio->dsp_mutex);

    // Wait for DATA_WRITE command
    ret = gst_dspaudio_wait_buffer(dsp->audio);

    if(ret == DSP_WAIT_RETRY) {
      g_mutex_unlock(dsp->audio->dsp_mutex);
      continue;
    }
    else if(ret == DSP_WAIT_INTERRUPT) {
      g_mutex_unlock(dsp->audio->dsp_mutex);
      break;
    }
    else if(ret == DSP_WAIT_OK) {
      dsp->audio->rw_pending = 1;
      if(gst_dspamrsink_fill_input_buffer(dsp)) {
        dsp->outbuf.data_size = MULTIPLE_PACKETS * (60-dsp->bandindex);
        done = write (dsp->audio->codec.fd, &dsp->outbuf, sizeof(AMR_DATA_WRITE));
        dsp->audio->discont_sent = FALSE;
        dsp->audio->rw_pending = 0;
        dsp->packet_index = 0;
        DBG_PRINT("AMR: Data written to DSP\n");
      }
      else {
        g_mutex_unlock(dsp->audio->dsp_mutex);
//         DBG_PRINT("Not enough data. Only %d bytes\n", gst_adapter_available(dsp->adapter));
        break;
      }
    }

    g_mutex_unlock(dsp->audio->dsp_mutex);

    if(ret == DSP_WAIT_ERROR || done == -1) {
      DBG_PRINT("ERROR OCCURED\n");
      GST_ELEMENT_ERROR (dsp, RESOURCE, WRITE, (NULL),
                        ("error cmd: %d status: %d",
                          dsp->audio->error_cmd,
                          dsp->audio->error_status));
      return GST_FLOW_ERROR;
    }
  }

//   DBG_PRINT("AMR RENDER EXIT\n");
  return GST_FLOW_OK;
}


/*
 * gst_dspamrsink_initialize:
 * @dsp: DSP AMR sink object
 *
 * Initializes the underlying audio stream object and sets up the
 * AMR audio parameters in DSP.
 *
 * Returns: TRUE if operation was successful, otherwise FALSE
 */
static gboolean
gst_dspamrsink_initialize(GstDSPAMRSink *dsp)
{
  AUDIO_PARAMS_DATA init_data;
  unsigned short mode_buf[2];

  init_data.dsp_cmd = DSP_CMD_SET_PARAMS;
  init_data.sample_rate = dsp->wide ? SAMPLE_RATE_16KHZ : SAMPLE_RATE_8KHZ;
  init_data.number_channels = 1;            // It is always mono
  init_data.audio_fmt = DSP_AFMT_S16_LE;
  init_data.stream_priority = dsp->audio->priority;
  init_data.ds_stream_ID = 0;

  if(!gst_dspaudio_setparams(dsp->audio, (char *)&init_data, sizeof(AUDIO_PARAMS_DATA))) {
    return FALSE;
  }

  mode_buf[0] = DSP_CMD_SET_MODE;
  mode_buf[1] = 1;  // MIME mode

  if(write (dsp->audio->codec.fd, mode_buf, 2*sizeof(short int)) < 0) {
    dsp->audio->mode = DSP_MODE_ERROR;
    return FALSE;
  }

  if(gst_dspaudio_read_cmd(dsp->audio, 0) != DSP_WAIT_OK) {
    return FALSE;
  }
  dsp->audio->mode = DSP_MODE_CONFIGURED;
  GST_DEBUG("DSP AMR configured");
  return TRUE;
}


/*
 * gst_dspamrsink_set_property:
 * @object: GObject where the property should be retrieved
 * @prop_id: Unique identifier of the desired property
 * @value: Pointer to a variable where the value should be put
 * @pspec: Parameter specification
 *
 * Stores the property value into GObject
 */
static void
gst_dspamrsink_set_property (GObject * object,
                             guint prop_id,
                             const GValue * value,
                             GParamSpec * pspec)
{
  GstDSPAMRSink *dsp = GST_DSPAMRSINK (object);
  if(!gst_dspaudio_set_property(dsp->audio, prop_id, value)) {
    g_warning("trying to set illegal property");
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
  }
}


/*
 * gst_dspamrsink_get_property:
 * @object: GObject where the property should be retrieved
 * @prop_id: Unique identifier of the desired property
 * @value: Pointer to a variable where the value should be put
 * @pspec: Parameter specification
 *
 * Retrieves the desired parameter value from object
 */
static void
gst_dspamrsink_get_property (GObject * object,
                             guint prop_id,
                             GValue * value,
                             GParamSpec * pspec)
{
  GstDSPAMRSink *dsp = GST_DSPAMRSINK (object);
  if(!gst_dspaudio_get_property(dsp->audio, prop_id, value)) {
    g_warning("trying to get illegal property");
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
  }
}


/*
 * gst_dspamrsink_get_query_types:
 * @element: GstElement that received this query
 *
 * Returns: a list of supported query types
 */
static const GstQueryType *
gst_dspamrsink_get_query_types (GstElement * element)
{
  static const GstQueryType gst_dspamrsink_query_types[] = {
    GST_QUERY_POSITION,
    0
  };

  return gst_dspamrsink_query_types;
}


/*
 * gst_dspamrsink_query:
 * @element: GstElement that received this query
 * @query: GstQuery object
 *
 * Returns: TRUE, iff the query was successful.
 */
static gboolean
gst_dspamrsink_query (GstElement * element, GstQuery *query)
{
  gboolean res = FALSE;
  GstDSPAMRSink *dsp = GST_DSPAMRSINK (element);
  GstQueryType type = GST_QUERY_TYPE (query);
  GstFormat format;
  gint64 value;

  if(type == GST_QUERY_POSITION) {
    gst_query_parse_position(query, &format, NULL);
    if (format != GST_FORMAT_TIME)
      return FALSE;

    // If we cannot query the position from neighbor element
    // (e.g. demuxer), we calculate it by ourselves
    if( ! gst_dspaudio_query_peer_position(dsp->sinkpad, &value) ) {
      value = dsp->current_framecount * AMR_TPF;
    }
    gst_query_set_position (query, GST_FORMAT_TIME, value);
    res = TRUE;
  }
  else if(type == GST_QUERY_DURATION) {
    gst_query_parse_duration(query, &format, NULL);
    if (format != GST_FORMAT_TIME)
      return FALSE;

    if(dsp->total_framecount) {
      value = dsp->total_framecount * AMR_TPF;
    }
    else {
//       value = dsp->avg_framecount * AMR_TPF;
      gdouble rel = (gdouble) dsp->current_bytes / dsp->total_length_bytes;
      value = (guint64) (dsp->current_framecount * AMR_TPF / rel);
      DBG_PRINT("AMR VBR LENGTH: %lld sec\n", value / GST_SECOND);
    }
    gst_query_set_duration (query, GST_FORMAT_TIME, value);
    res = TRUE;
  }
  return res;
}


/*
 * gst_dspamrsink_event:
 * @element: GstElement that received this event
 * @event: The GstEvent structure itself
 *
 * Returns: TRUE, iff the event was successful
 */
static gboolean
gst_dspamrsink_event (GstElement * element, GstEvent * event)
{
  gboolean res = FALSE;

  GstDSPAMRSink *dsp = GST_DSPAMRSINK (element);
  GstEventType type = event
                    ? GST_EVENT_TYPE (event)
                    : GST_EVENT_UNKNOWN;

  DBG_PRINT("AMR: event %p (%d)\n", event, type);

  if(type == GST_EVENT_SEEK) {
    gdouble rate;
    GstFormat format;
    GstSeekFlags flags;
    GstSeekType cur_type, stop_type;
    gint64 cur, stop;

    // Try if previous element can handle the seek event.
    gst_event_ref(event);
    res = gst_pad_send_event (GST_PAD_PEER(dsp->sinkpad), event);
    if ( res ) {
      DBG_PRINT("AMR: Previous element handled the seek event.\n");
    } else {
      gst_event_parse_seek (event, &rate, &format, &flags,
                            &cur_type, &cur, &stop_type, &stop);

      if(format == GST_FORMAT_TIME) {
        gst_dspamrsink_perform_seek(dsp, cur);
        res = TRUE;
      }
    }
  }

  //   res = GST_ELEMENT_CLASS (parent_class)->send_event (element, event);
  gst_event_unref(event);
  DBG_PRINT("EVENT HANDLED\n");
  return res;
}


/*
 * gst_dspamrsink_event2:
 *
 * This method handles events coming from basesink
 */
static gboolean
gst_dspamrsink_event2 (GstBaseSink *sink, GstEvent * event)
{
  GstDSPAMRSink *dsp = GST_DSPAMRSINK (sink);
  gboolean res = FALSE;

  GstEventType type = event ? GST_EVENT_TYPE (event) : GST_EVENT_UNKNOWN;
  if(type == GST_EVENT_EOS) {
    DBG_PRINT("----- EOS EVENT -----\n");
    dsp->audio->codec.stream_id = 0;
    g_signal_emit (G_OBJECT (sink), gst_dspamrsink_signals[DSPAMR_SIGNAL_STREAMID], 0, 0);
    dsp->audio->mode = DSP_MODE_EOS;
    res = TRUE;
  }
  else if(type == GST_EVENT_NEWSEGMENT)
  {
    DBG_PRINT("AMR NEWSEGMENT EVENT\n");
    g_mutex_lock(dsp->audio->dsp_mutex);
    gst_dspamrsink_send_discont(dsp);
    g_mutex_unlock(dsp->audio->dsp_mutex);
    res = TRUE;
  }
  else if(type == GST_EVENT_FLUSH_STOP) {
    DBG_PRINT("AMR FLUSH STOP\n");
    gst_adapter_clear(dsp->adapter);
    dsp->packet_index  = 0;
    res = TRUE;
  }
  return res;
}


/*
 * gst_dspamrsink_check_stream:
 * @dsp: DSP AMR sink object
 *
 * Checks if the incoming stream really is either AMR-NB or
 * AMR-WB format.
 *
 * Returns: TRUE if the content was recognized, otherwise FALSE
 */
static gboolean
gst_dspamrsink_check_stream(GstDSPAMRSink *dsp, const guint8 *data)
{
  gboolean retval = FALSE;

  DBG_PRINT("CHECK_STREAM: %c%c%c%c%c%c%d\n", data[0],data[1],data[2],data[3],data[4],data[5],data[6]);
  if (memcmp(data, "#!AMR-WB\n", 9) == 0) {
    dsp->wide = TRUE;
    dsp->bandindex = 1;
    retval = TRUE;
  }
  else if (memcmp(data, "#!AMR\n", 6) == 0) {
    dsp->wide = FALSE;
    dsp->bandindex = 0;
    retval = TRUE;
  }
  else if(dsp->read_mime_header == FALSE) {
    // Probably we got the band information from Caps
    retval = TRUE;
  }
  DBG_PRINT("gst_dspamrsink_check_stream: %d\n", retval);
  return retval;
}


/*
 * gst_dspamrsink_open:
 *
 */
static gboolean
gst_dspamrsink_open(GstDSPAMRSink *dsp, guint bandindex)
{
  if(!gst_dspaudio_open(dsp->audio, devname[dsp->bandindex])) {
    GST_ELEMENT_ERROR (dsp, RESOURCE,
                       OPEN_READ_WRITE, (NULL),
                       ("gst_dspaudio_open"));
    return FALSE;
  }

  g_signal_emit (G_OBJECT (dsp), gst_dspamrsink_signals[DSPAMR_SIGNAL_STREAMID],
                 0, dsp->audio->codec.stream_id);

  return TRUE;
}


/*
 * gst_dspamrsink_paused_to_playing:
 *
 */
static gboolean
gst_dspamrsink_paused_to_playing(GstDSPAMRSink *dsp)
{
  /*
  if(gst_dspaudio_open(dsp->audio, devname[dsp->bandindex]) == FALSE) {
    GST_ELEMENT_ERROR (dsp, RESOURCE, OPEN_READ_WRITE, (NULL),
                       ("gst_dspaudio_open"));
    return FALSE;
  }

  g_signal_emit (G_OBJECT (dsp), gst_dspamrsink_signals[DSPAMR_SIGNAL_STREAMID],
                 0, dsp->audio->codec.stream_id);

  // AMR-specific initialization part
  if(!gst_dspamrsink_initialize(dsp)) {
    GST_ELEMENT_ERROR (dsp, RESOURCE, SETTINGS, (NULL),
                       ("gst_dspamrsink_initialize"));
    return FALSE;
  }
  */

  gst_dspaudio_play(dsp->audio);

  // Settings might have changed between PAUSE->PLAY transition
  gst_dspaudio_update_dsp_settings(dsp->audio);
  gst_adapter_flush(dsp->adapter, dsp->wide ? 9 : 6);
  return TRUE;
}


/*
 * gst_dspamrsink_query_length:
 * @dsp: GstDSPAMRSink object
 *
 * Retrieves the length of stream, in bytes. Stores the value
 * in dsp->total_length_bytes variable
 */
static void
gst_dspamrsink_query_length(GstDSPAMRSink *dsp)
{
  GstPad *peer;

  peer = gst_pad_get_peer(dsp->sinkpad);
  if(peer) {
    GstFormat fmt = GST_FORMAT_BYTES;
    gboolean ret = gst_pad_query_duration (peer, &fmt, &dsp->total_length_bytes);
    gst_object_unref(peer); // _get_peer() ref'ed it
    if(fmt != GST_FORMAT_BYTES || ret == FALSE) {
      // It gave the length in some other format or query failed
      dsp->total_length_bytes = 0;
    }
  }
  DBG_PRINT("TOTAL FILE LENGTH: %lld bytes\n", dsp->total_length_bytes);
}


/*
 * gst_dspamrsink_preroll:
 *
 */
static GstFlowReturn
gst_dspamrsink_preroll(GstBaseSink *sink, GstBuffer * buffer)
{
  GstDSPAMRSink *dsp = GST_DSPAMRSINK (sink);
  DBG_PRINT("gst_dspamrsink_preroll\n");

  if(dsp->audio->mode < DSP_MODE_CONFIGURED) {
    g_mutex_lock(dsp->audio->dsp_mutex);
    if(!gst_dspamrsink_check_stream(dsp, GST_BUFFER_DATA(buffer))) {
      g_mutex_unlock(dsp->audio->dsp_mutex);
      GST_ELEMENT_ERROR (dsp, STREAM, TYPE_NOT_FOUND, (NULL),
                         ("gst_dspamrsink_preroll"));
      return GST_FLOW_ERROR;
    }
    if(!gst_dspamrsink_open(dsp, dsp->bandindex)) {
      g_mutex_unlock(dsp->audio->dsp_mutex);
      return GST_FLOW_ERROR;
    }
    if(!gst_dspamrsink_initialize(dsp)) {
      g_mutex_unlock(dsp->audio->dsp_mutex);
      GST_ELEMENT_ERROR (dsp, RESOURCE, SETTINGS, (NULL),
                         ("gst_dspamrsink_initialize"));
      return GST_FLOW_ERROR;
    }
    g_mutex_unlock(dsp->audio->dsp_mutex);
    gst_dspamrsink_query_length(dsp);
//     gst_dspamrsink_calculate_length(dsp);
  }
  DBG_PRINT("gst_dspamrsink_preroll OK\n");
  return GST_FLOW_OK;
}


/*
 * gst_dspamrsink_change_state:
 * @element: The GstElement whose state to be changed
 *
 * Returns: GST_STATE_CHANGE_SUCCESS if operation was successful.
 *          Otherwise GST_STATE_CHANGE_FAILURE will be returned.
 */
static GstStateChangeReturn
gst_dspamrsink_change_state(GstElement * element, GstStateChange transition)
{
  GstDSPAMRSink *dspamrsink = GST_DSPAMRSINK (element);
  GstStateChangeReturn ret = GST_STATE_CHANGE_SUCCESS;

  DBG_PRINT("AMR CHANGE_STATE CALLED\n");

  switch (transition) {

    case GST_STATE_CHANGE_NULL_TO_READY:
      DBG_PRINT("AMR GST_STATE_CHANGE_NULL_TO_READY\n");
      dspamrsink->read_mime_header = TRUE;
      dspamrsink->total_length_bytes = 0;
      dspamrsink->current_framecount = 0;
      dspamrsink->total_framecount = 0;
      dspamrsink->avg_framecount = 0;
      dspamrsink->current_bytes = 0;
      dspamrsink->wide = FALSE;
      dspamrsink->bandindex = 0;
      dspamrsink->packet_index = 0;
      dspamrsink->outbuf.dsp_cmd = DSP_CMD_DATA_WRITE;
      break;

    case GST_STATE_CHANGE_READY_TO_PAUSED:
      DBG_PRINT("AMR GST_STATE_CHANGE_READY_TO_PAUSED\n");
      gst_adapter_clear(dspamrsink->adapter);
      break;

    case GST_STATE_CHANGE_PAUSED_TO_PLAYING:
      g_mutex_lock(dspamrsink->audio->dsp_mutex);
      DBG_PRINT("AMR GST_STATE_CHANGE_PAUSED_TO_PLAYING\n");
      // Set the play only if we *really* are at play state
      if(dspamrsink->audio->mode == DSP_MODE_PAUSED) {
        dspamrsink->audio->rw_pending = 0;
        gst_dspaudio_play(dspamrsink->audio);
      }
      DBG_PRINT("AMR GST_STATE_CHANGE_PAUSED_TO_PLAYING OK\n");
      g_mutex_unlock(dspamrsink->audio->dsp_mutex);
      break;

    case GST_STATE_CHANGE_PLAYING_TO_PAUSED:
      DBG_PRINT("AMR GST_STATE_CHANGE_PLAYING_TO_PAUSED\n");
      gst_dspaudio_interrupt_render(dspamrsink->audio);
      break;

    default:
      break;
  }

  gst_dspaudio_clock_change_state(dspamrsink->audio, transition);
  ret = GST_ELEMENT_CLASS (parent_class)->change_state (element, transition);

  switch (transition) {

    case GST_STATE_CHANGE_PLAYING_TO_PAUSED:
      gst_dspaudio_remove_interrupt(dspamrsink->audio);
      g_mutex_lock(dspamrsink->audio->dsp_mutex);
      DBG_PRINT("AMR GST_STATE_CHANGE_PLAYING_TO_PAUSED2\n");
      if(gst_dspaudio_pause(dspamrsink->audio)) {
        gst_dspaudio_read_cmd(dspamrsink->audio, 0);
        dspamrsink->audio->rw_pending = 0;
      }
      DBG_PRINT("AMR GST_STATE_CHANGE_PLAYING_TO_PAUSED2 OK\n");
      g_mutex_unlock(dspamrsink->audio->dsp_mutex);
      break;

    case GST_STATE_CHANGE_PAUSED_TO_READY:
      DBG_PRINT("AMR GST_STATE_CHANGE_PAUSED_TO_READY\n");
      g_mutex_lock(dspamrsink->audio->dsp_mutex);
      if(gst_dspaudio_stop(dspamrsink->audio)) {
        gst_dspaudio_read_cmd(dspamrsink->audio, 0);
//         DBG_PRINT("AMR: GOT STOP RESPONSE\n");
      }
      g_mutex_unlock(dspamrsink->audio->dsp_mutex);
      gst_adapter_clear(dspamrsink->adapter);

      break;

    case GST_STATE_CHANGE_READY_TO_NULL:
      DBG_PRINT("AMR GST_STATE_CHANGE_READY_TO_NULL\n");
      gst_dspaudio_close(dspamrsink->audio);
      gst_dspaudio_reset(dspamrsink->audio);
      break;

    default:
      break;
  }

  return ret;
}


/*
 * gst_dspamrsink_async_play:
 *
 */
static GstStateChangeReturn
gst_dspamrsink_async_play (GstBaseSink *sink)
{
  GstDSPAMRSink *dspamrsink = GST_DSPAMRSINK (sink);
  DBG_PRINT("AMR ASYNC PLAY\n");

  if(dspamrsink->audio->mode == DSP_MODE_PAUSED) {
    g_mutex_lock(dspamrsink->audio->dsp_mutex);
    gst_dspaudio_play(dspamrsink->audio);
    g_mutex_unlock(dspamrsink->audio->dsp_mutex);
    DBG_PRINT("AMR ASYNC PLAY SUCCESS\n");
  }
  return GST_STATE_CHANGE_SUCCESS;
}


/*
 * gst_dspamrsink_provide_clock:
 *
 */
static GstClock *
gst_dspamrsink_provide_clock (GstElement * elem)
{
  DBG_PRINT("gst_dspamrsink_provide_clock\n");
  GstDSPAMRSink *dsp = GST_DSPAMRSINK (elem);
  return GST_CLOCK (gst_object_ref (GST_OBJECT (dsp->clock)));
}


/*
 * gst_dspamrsink_get_time:
 *
 */
static GstClockTime
gst_dspamrsink_get_time (GstClock * clock, gpointer sink)
{
  GstDSPAMRSink *dsp = GST_DSPAMRSINK (sink);
  return gst_dspaudio_get_dsp_clock(dsp->audio);
}


#define AMR_LEN_CALC_BLOCKSIZE (4*1024)

/* For now this can wait until we upgrade the plugin to work
   in both pull and push mode. */

/*
static void
gst_dspamrsink_calculate_length(GstDSPAMRSink *dsp)
{
  guint frame_header, frame_index, frame_length;
  guint64 blockindex, startpos, pos, currlen = 0;
  GstBuffer *buf;
  guint8 *data;

  guint64 blocksize = AMR_LEN_CALC_BLOCKSIZE;
  GstFormat fmt = GST_FORMAT_BYTES;

#ifdef DEBUG
  struct timeval tv1, tv2, tv3;
  gettimeofday(&tv1, NULL);
#endif

// If we can not use pull_range, we can not calculate length
  if(! gst_pad_check_pull_range (dsp->sinkpad) ) {
    DBG_PRINT("AMR: pull_range not supported\n");
    return;
  }

  pos = dsp->wide ? 9 : 6;
  gst_pad_query_position(GST_PAD_PEER(dsp->sinkpad), &fmt, &startpos);
  DBG_PRINT("CURRENT POS: %lld\n", startpos);

  while(TRUE) {

    // Can we read one more byte?
    if( (pos+AMR_LEN_CALC_BLOCKSIZE) >= dsp->total_length_bytes ) {
      blocksize = dsp->total_length_bytes - pos;
    }

    if(!blocksize) break;

    if(gst_pad_pull_range(dsp->sinkpad, pos, blocksize, &buf) != GST_FLOW_OK) {
      return;
    }
    data = GST_BUFFER_DATA(buf);
    blockindex = 0;

    // Loop the data block through
    do {
      frame_header = data[blockindex];
      blockindex++;
      pos++;

      if( (frame_header & 0x83) != 0 ) {
        continue;
      }
      frame_index = (frame_header >> 3) & 0x0F;
      frame_length = FrameLength[dsp->bandindex][frame_index]; //no. of bytes to be read

      pos += frame_length-1;
      blockindex += frame_length-1;
      dsp->total_framecount++;
      currlen += frame_length;
    }
    while(blockindex < blocksize);

    gst_buffer_unref(buf);

    if(dsp->total_framecount > AVG_LEN_CALC_THRESHOLD) {
      dsp->avg_framecount = dsp->total_length_bytes / (currlen / dsp->total_framecount);
      dsp->total_framecount = 0;
      DBG_PRINT("Too many frames\n");
      break;
    }
  }
  DBG_PRINT("AMR: TOTAL FRAMES: %lld, AVG = %lld\n", dsp->total_framecount, dsp->avg_framecount);

  GstEvent *new_event = gst_event_new_seek(1.0, GST_FORMAT_BYTES,
                                           GST_SEEK_FLAG_NONE,
                                           GST_SEEK_TYPE_SET, startpos,
                                           GST_SEEK_TYPE_NONE,
                                           GST_CLOCK_TIME_NONE);

  gst_pad_push_event (dsp->sinkpad, new_event);

#ifdef DEBUG
  gettimeofday(&tv2, NULL);
  timersub(&tv2, &tv1, &tv3);
  printf("AMR length calculation take %d sec %d usec\n", (int) tv3.tv_sec, (int) tv3.tv_usec);
#endif
}
*/


/*
 * plugin_init:
 * @plugin: GstPlugin to be initialized
 *
 * This initializes the plugin when it is loaded by GStreamer
 *
 * Returns: Boolean value, TRUE if initialization was successful
 */
static gboolean
plugin_init (GstPlugin * plugin)
{
  return gst_element_register (plugin, "dspamrsink", GST_RANK_PRIMARY,
      GST_TYPE_DSPAMRSINK);
}

GST_PLUGIN_DEFINE (GST_VERSION_MAJOR,
                   GST_VERSION_MINOR,
                   "dspamr",
                   "DSP AMR NB/WB audio sink",
                   plugin_init,
                   VERSION,
                   "LGPL",
                   "dspaudio",
                   "http://www.nokia.com/");

