/*
 * src/softmac2.h
 *
 * Copyright (C) 2003 Conexant Americas Inc. All Rights Reserved.
 * Copyright (C) 2004, 2005, 2006 Nokia Corporation
 * Author: Samuel Ortiz <samuel.ortiz@nokia.com>
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 */
#ifndef __SOFTMAC2_H__
#define __SOFTMAC2_H__

/**** Default API ***********************************************************/

#ifndef SM_API
#define SM_API
#endif

#ifndef SM_FDECL
#define SM_FDECL
#endif

/**** Upper API *************************************************************/

/* Error codes */
#define SM_ENONE        0
#define SM_EUNSPEC      -1
#define SM_ENOTSUP      -2
#define SM_ECONF        -3
#define SM_EMEM         -4
#define SM_ETIMEOUT     -5
#define SM_EFRAME       -6
#define SM_EOIDUNKN     -7
#define SM_EOVERFLOW    -8
#define SM_ELINK        -9
#define SM_EDSTUNKN     -10
#define SM_EPENDING     -11
#define SM_EMATCH       -12

/* Function callback mask */
#define SM_TRAP         0x0001
#define SM_FRAMETXDONE  0x0002
#define SM_FRAMERX      0x0004
#define SM_IC           0x0008

/* Creation and destoy */
struct s_sm_initdata
{
	uint32_t placeholder[256];   /* content opaque to the driver */
};

struct s_sm_descr
{
	uint16_t header;
	uint16_t mtu;
	uint16_t trailer;
	uint16_t rxoffset;
	uint32_t context_size;
};

#define SM_MODE_CLIENT      0
#define SM_MODE_AP          1
#define SM_MODE_PROMISCUOUS 2

struct s_sm_setup
{
	int8_t mode;
	uint8_t rx_align;           /* deprecated, do not use. */
	uint16_t max_sta;
	uint16_t max_bss;
	uint16_t pad1;
	uint32_t seed;
	uint16_t mtu;
	uint8_t  depth[8];
};

struct s_pda
{
	uint16_t size;
	uint16_t pad1;
};

SM_API int32_t SM_FDECL prism_softmac_parse_bra(const uint8_t* bra, 
						struct s_sm_initdata *data);

SM_API int32_t SM_FDECL prism_softmac_describe(struct s_sm_descr *descr, 
					       struct s_sm_initdata *data); 

SM_API int32_t SM_FDECL prism_softmac_create(uint32_t *context, 
					     struct s_sm_setup *setup, 
					     struct s_pda *pda, 
					     struct s_sm_initdata *data);

SM_API int32_t SM_FDECL prism_softmac_destroy(uint32_t *context);

/* Configuration */
#define SM_CONF_OPGET   0x0
#define SM_CONF_OPSET   0x0001

struct s_sm_conf
{
	uint16_t flags;
	uint16_t length;
	uint32_t oid;
	uint8_t *data;
};

SM_API int32_t SM_FDECL prism_softmac_conf( uint32_t *context, struct s_sm_conf *conf );
SM_API int32_t SM_FDECL prism_softmac_trap( uint32_t *context, struct s_sm_conf *conf );

/* Annex types */
#define SM_ANNEX_UNUSED         0
#define SM_ANNEX_TX             1
#define SM_ANNEX_RX             2

union u_sm_annex
{
	struct s_sm_annex_tx
	{
		uint8_t  type;
		uint8_t  ssnr;
		uint16_t flags;
		
		uint16_t rate;
		uint8_t  antenna;
		uint8_t  preamble;
		
		uint16_t lifetime;
		uint16_t fragment;
		uint8_t  retries;
		int8_t   power;
		uint8_t  pad[2];
	} tx;
	
	struct s_sm_annex_rx
	{
		uint8_t  type;
		uint8_t  ssnr;
		uint16_t flags;
		
		uint16_t rate;
		uint8_t  antenna;
		uint8_t  preamble;
		
		uint32_t clock;
		uint16_t frequency;
		int8_t   rssi;
		uint8_t  quality;
	} rx;
};

/* Promiscuous structtures */
struct sm_promisc_header
{
	uint8_t da_mac[6];
	uint8_t sa_mac[6];
	uint8_t mac_length[2];
	uint8_t llc_hdr[3];
	uint8_t oui[3];
	uint8_t ppe_type[2];
	uint8_t flags;
	uint8_t seqnr;
	uint16_t length;
	uint32_t clock;
	uint16_t reallength;
	uint16_t frequency;
	uint16_t rate;
	int8_t rssi;
	uint8_t properties;
} __attribute__ ((packed));

/* wlan-ng (and hopefully others) AVS header, version one.  Fields in
 * network byte order. */
#define P80211CAPTURE_VERSION 0x80211001
struct avs_80211_1_header {
	uint32_t version;
	uint32_t length;
	uint64_t mactime;
	uint64_t hosttime;
	uint32_t phytype;
	uint32_t channel;
	uint32_t datarate;
	uint32_t antenna;
	uint32_t priority;
	uint32_t ssi_type;
	int32_t ssi_signal;
	int32_t ssi_noise;
	uint32_t preamble;
	uint32_t encoding;
};


/* Data-link */
#define SM_FRAME_PROMISCUOUS    0x01
#define SM_FRAME_WDS            0x02
#define SM_FRAME_VSS            0x04
#define SM_FRAME_TXDROPPED      0x20
#define SM_FRAME_TXFAILED       0x40
#define SM_FRAME_RXFAILED       0x80

struct s_sm_frame
{
	uint32_t handle;
	uint8_t flags;
	uint8_t priority;
	uint16_t length;
	uint8_t *data;
	union u_sm_annex annex;
};

SM_API int32_t SM_FDECL prism_softmac_frame_tx(uint32_t *context, struct s_sm_frame *frame);
SM_API int32_t SM_FDECL prism_softmac_frame_tx_done(uint32_t *context, struct s_sm_frame **frame);
SM_API int32_t SM_FDECL prism_softmac_frame_rx(uint32_t *context, struct s_sm_frame **frame);

/* Service */
SM_API int32_t SM_FDECL prism_softmac_service(uint32_t *context);

/**** Inter-connect API ******************************************************/

/* Message */
#define IC_CHANNEL_CTRL 0
#define IC_CHANNEL_DATA 1

#define IC_FLAG_FREE    0x01

struct s_ic_msg
{
	uint8_t channel;
	uint8_t flags;
	uint16_t length;
	uint32_t address;
	struct s_sm_frame *frame;
	struct s_ic_link
	{
		void *next;
		void *prev;
	} link;
	void *data;
};

/* Outbound */
#define IC_MASK_CTRL    0x01
#define IC_MASK_FRAME   0x02

SM_API int32_t SM_FDECL prism_interconnect_message_query(uint32_t *context, 
							 uint32_t mask, 
							 struct s_ic_msg **msg);

/* Inbound */
SM_API int32_t SM_FDECL prism_interconnect_message_handle(uint32_t *context, struct s_ic_msg *msg);

/* Trigger */
SM_API int32_t SM_FDECL prism_interconnect_trigger(uint32_t *context);

/**** Driver API *************************************************************/

/* Memory management */
#define DRIVER_MEM_NORMAL   0
#define DRIVER_MEM_IC       1

void *prism_driver_alloc(uint32_t *context, uint32_t size, uint8_t type);
void prism_driver_free(uint32_t *context, void *buffer);

/* Frame management */
struct s_sm_frame *prism_driver_frame_alloc(uint32_t *context, uint32_t length);
void prism_driver_frame_free(uint32_t *context, struct s_sm_frame *frame);

/* Timers */
void prism_driver_timer_start(uint32_t *context, uint32_t msec);
void prism_driver_timer_stop(uint32_t *context);
uint32_t prism_driver_time(uint32_t *context);

/* Offloading operations */
#define DRIVER_OFFLOAD_OPERATION_WEP_RC4_ENCRYPT        0
#define DRIVER_OFFLOAD_OPERATION_WEP_RC4_DECRYPT        1
#define DRIVER_OFFLOAD_OPERATION_TKIP_ENCRYPT           2
#define DRIVER_OFFLOAD_OPERATION_TKIP_DECRYPT           3
#define DRIVER_OFFLOAD_OPERATION_MICHAEL_CALC           4
#define DRIVER_OFFLOAD_OPERATION_CCMP_ENCRYPT           5
#define DRIVER_OFFLOAD_OPERATION_CCMP_DECRYPT           6

#define DRIVER_OFFLOAD_FLAG_FAIL                0x01
#define DRIVER_OFFLOAD_FLAG_MISMATCH            0x02
#define DRIVER_OFFLOAD_FLAG_CONTEXTCHANGE       0x04

/*
 *   The structure is used to exchange general offload information
 *   with each offload operation.
 */
struct s_driver_offload
{
	uint32_t handle;
	uint8_t operation;
	uint8_t flags;
	uint16_t length;
	uint8_t *inBuf;
	uint8_t *outBuf;
};

/*
 *   Following structures are used to exchange context-specific
 *   information for the offload operation. This structure is
 *   located right after the generic structure defined before and
 *   starts on an aligned memory location.
 */
struct s_driver_offload_wep
{
	uint16_t key_length;
	uint8_t iv[3];
	uint8_t key[13];
};

struct s_driver_offload_tkip
{
	uint32_t sc_high;
	uint16_t sc_low;
	uint8_t ta[6];
	uint8_t key[16];
};

struct s_driver_offload_michael
{
	uint8_t da[6];
	uint8_t sa[6];
	uint8_t priority;
	uint8_t key[8];
	uint8_t pad[3];
};

struct s_driver_offload_ccmp
{
	uint8_t nonce[13];
	uint8_t aad[32];
	uint8_t key[16];
};

typedef SM_API int32_t SM_FDECL (*cb_driver_offload)
(uint32_t *context, struct s_driver_offload *result);

/*
 * The offload function needs to be defined by the driver according
 * the prototype. the UMAC calls into the driver using this function call. This
 * function takes parameters as described below. This call may be completed
 * synchronously or asynchronously. When using asynchronously, the driver must call the
 * cb_driver_offload() callback function with the results of the offloaded operation at
 * a later time.
 *
 * @returns
 *	SM_ENONE if successfully completed synchronously; SM_EPENDING if completed
 *	asynchronously.
 */
int32_t prism_driver_offload(uint32_t *context,
			     struct s_driver_offload *data, 
			     cb_driver_offload callback);

/* Debugging */
#define SM_DFATAL   0
#define SM_DERROR   1
#define SM_DWARNING 2
#define SM_DINFO    3
#define SM_DDEBUG   4

void prism_driver_printf(uint32_t *context, uint8_t level,
			 uint8_t *format, uint32_t arg0, uint32_t arg1, uint32_t arg2);

void prism_driver_dump(uint32_t *context, uint8_t level, uint8_t *data, uint32_t size);
void prism_driver_assert(uint8_t *error, uint8_t *file, uint32_t line);

/* Profiling */
int32_t prism_driver_prof_register(uint32_t *context, uint8_t *name);
void prism_driver_prof_deregister(uint32_t *context, int32_t id);
void prism_driver_prof_reset(uint32_t *context, int32_t id);
void prism_driver_prof_dump(uint32_t *context);
void prism_driver_prof_enter(uint32_t *context, int32_t id);
uint32_t prism_driver_prof_exit(uint32_t *context, int32_t id);

#endif /* __softmac2_h__ */
