/*
 * src/sm_drv_sysfs_mtum.c
 *
 * Copyright (C) 2005, 2006 Nokia Corporation
 * Author: Samuel Ortiz <samuel.ortiz@nokia.com>
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 */
#include <linux/platform_device.h>

#include "sm_drv.h"
#include "sm_drv_ioctl.h"
#include "sm_drv_sysfs.h"
#include "smoid2.h"

unsigned int driver_type = SM_DRIVER_TYPE_MTUM;

static ssize_t sm_drv_show_dummy(struct device *dev, struct device_attribute *attr, char *buf)
{
	return sprintf(buf, "0\n");
}

/* timeout */
sm_drv_sysfs_ru32_attr(timeout, LMAC_MTUM_TIMEOUT)
sm_drv_sysfs_wu32_attr(timeout, LMAC_MTUM_TIMEOUT)


static ssize_t sm_drv_show_psm(struct device *dev, struct device_attribute *attr, char *buf)
{
	unsigned int mode, value;
	int ret;

	ret = sm_drv_oid_get((struct net_device *)dev_get_drvdata(dev),
			     LMAC_MTUM_MODE,
			     &mode,
			     sizeof(uint32_t));
	
	if (ret < 0)
		return 0;
	
	/* let's check if we're in power save mode*/
	if (mode == TM_PSM)
		value = 2;
	else
		value = 1;
		
	return sprintf(buf, "%d\n", value);
}

static ssize_t sm_drv_store_psm(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	int ret;
	int psm = simple_strtoul(buf, NULL, 0);
	int mode;

	DEBUG(DBG_SYSFS, "Setting value: %d for PSM\n", psm);

	if (psm == 2)
		mode = TM_PSM;
	else
		mode = TM_IDLE;
	


	ret = sm_drv_oid_set((struct net_device *)dev_get_drvdata(dev),
			     LMAC_MTUM_MODE,
			     &mode,
			     sizeof(uint32_t));
	
	if (ret < 0)
		return 0;
		
	return count;
}

/* Reset */
static ssize_t sm_drv_wlan_reset(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	int ret, command = CMD_RESET, mode = TM_IDLE;

	/* Reseting must be done in idle mode */
	ret = sm_drv_oid_set((struct net_device *)dev_get_drvdata(dev),
			     LMAC_MTUM_MODE, &mode, sizeof(uint32_t));
	
	if (ret < 0)
		return 0;
	

	ret = sm_drv_oid_set((struct net_device *)dev_get_drvdata(dev), 
			     LMAC_MTUM_CMD, (void *)&command, sizeof(uint32_t));
	
	if (ret < 0)
		return 0;
	
	return count;
}

static ssize_t sm_drv_show_wlan_reset(struct device *dev, struct device_attribute *attr, char *buf)
{
	return sprintf(buf, "0\n");
}

/* RX/TX */

static ssize_t sm_drv_show_continuous_tx(struct device *dev, struct device_attribute *attr, char *buf)
{
	unsigned int mode, value;
	int ret;

	ret = sm_drv_oid_get((struct net_device *)dev_get_drvdata(dev),
			     LMAC_MTUM_MODE,
			     &mode,
			     sizeof(uint32_t));
	
	if (ret < 0)
		return 0;
	
	/* let's check if we're in continuous TX */
	if (mode == TM_CONT_TX)
		value = 1;
	else
		value = 0;
		
	return sprintf(buf, "%d\n", value);
}

static ssize_t sm_drv_store_continuous_tx(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	int ret;
	int value = simple_strtoul(buf, NULL, 0);
	int mode;

	DEBUG(DBG_SYSFS, "Setting value: %d for (continuous tx)\n", value);

	if (value == 1)
		mode = TM_CONT_TX;
	else
		mode = 0;
	
	ret = sm_drv_oid_set((struct net_device *)dev_get_drvdata(dev),
			     LMAC_MTUM_MODE,
			     &mode,
			     sizeof(uint32_t));
	
	if (ret < 0)
		return 0;
		
	return count;
}


static ssize_t sm_drv_show_per_tx(struct device *dev, struct device_attribute *attr, char *buf)
{
	unsigned int mode, value;
	int ret;

	ret = sm_drv_oid_get((struct net_device *)dev_get_drvdata(dev),
			     LMAC_MTUM_MODE,
			     &mode,
			     sizeof(uint32_t));
	
	if (ret < 0)
		return 0;
	
	/* let's check if we're in repeat TX */
	if (mode == TM_REPEAT_TX)
		value = 1;
	else
		value = 0;
		
	return sprintf(buf, "%d\n", value);
}

static ssize_t sm_drv_store_per_tx(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	int ret;
	int value = simple_strtoul(buf, NULL, 0);
	int mode;

	DEBUG(DBG_SYSFS, "Setting value: %d (per_tx)\n", value);

	if (value == 1)
		mode = TM_REPEAT_TX;
	else
		mode = 0;
	
	ret = sm_drv_oid_set((struct net_device *)dev_get_drvdata(dev),
			     LMAC_MTUM_MODE,
			     &mode,
			     sizeof(uint32_t));
	
	if (ret < 0)
		return 0;
		
	return count;
}

/* TX burst routines */
struct burst_rf_gain {
	uint8_t nr_bursts;
	uint8_t powers[MAX_NR_BURSTS];
	uint8_t pad[3];
};

static struct obj_burstctx burst_tx_params;

static ssize_t sm_drv_store_burst_rf_gain(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	struct burst_rf_gain * rf_gains= (struct burst_rf_gain *)buf;
	int i;
	
	printk("We got %d bursts\n", rf_gains->nr_bursts);
	burst_tx_params.nr_bursts = rf_gains->nr_bursts;
	
	for (i = 0; i < burst_tx_params.nr_bursts; i++) {
		printk("  PCV[%d]: %d", i, rf_gains->powers[i]);
		burst_tx_params.powers[i] = 256 * rf_gains->powers[i];
	}

	return count;
		     
}

static ssize_t sm_drv_store_burst_tx(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	int ret;
	int value = simple_strtoul(buf, NULL, 0);
	int mode;

	if (value == 1) {
		if (burst_tx_params.nr_bursts == 0) {
			printk("Nr burst is 0, not starting\n");
			return count;
		}
		burst_tx_params.mtum_mode = TM_BURST_CONT_TX;
		burst_tx_params.width = 500; /* TP values */
		burst_tx_params.spacing = 100; /* TP values */
		ret = sm_drv_oid_set((struct net_device *)dev_get_drvdata(dev),
				     LMAC_MTUM_MODE_TX_BURST,
				     &burst_tx_params,
				     sizeof(struct obj_burstctx));	
	} else {
		mode = 0;
		ret = sm_drv_oid_set((struct net_device *)dev_get_drvdata(dev),
				     LMAC_MTUM_MODE,
				     &mode,
				     sizeof(uint32_t));	
	}
		
	if (ret < 0)
		return 0;
		
	return count;
}

static ssize_t sm_drv_show_burst_detector(struct device *dev, struct device_attribute *attr, char *buf)
{
	int ret, i;
	struct obj_burstdetector * burst_detector;

	ret = sm_drv_oid_get((struct net_device *)dev_get_drvdata(dev),
			     LMAC_MTUM_BURST_DETECTOR,
			     buf,
			     sizeof(struct obj_burstdetector));
	
	burst_detector = (struct obj_burstdetector *)buf;
	
	for (i = 0; i < burst_tx_params.nr_bursts; i++)
		printk("BURST detector[%d]: %d\n", i, burst_detector->values[i]);
		
	return sizeof(struct obj_burstdetector);
}

sm_drv_sysfs_ru32_attr(tx_pattern, LMAC_MTUM_PATTERN)
sm_drv_sysfs_wu32_attr(tx_pattern, LMAC_MTUM_PATTERN)

static ssize_t sm_drv_store_tx_stop(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	int ret;
	int value = simple_strtoul(buf, NULL, 0);
	int mode;

	if (value == 1) {
		mode = TM_IDLE;
		
		ret = sm_drv_oid_set((struct net_device *)dev_get_drvdata(dev),
				     LMAC_MTUM_MODE,
				     &mode,
				     sizeof(uint32_t));
	
		if (ret < 0)
			return 0;
	}
		
	return count;
}

static ssize_t sm_drv_show_tx_stop(struct device *dev, struct device_attribute *attr, char *buf)
{
	unsigned int mode, value;
	int ret;

	ret = sm_drv_oid_get((struct net_device *)dev_get_drvdata(dev),
			     LMAC_MTUM_MODE,
			     &mode,
			     sizeof(uint32_t));
	
	if (ret < 0)
		return 0;
	
	/* let's check if we're in repeat TX */
	if (mode == TM_CONT_TX 
	    || mode == TM_REPEAT_TX
	    || mode == TM_PROM_RXTX)
		value = 0;
	else
		value = 1;
		
	return sprintf(buf, "%d\n", value);
}


static ssize_t sm_drv_show_per_rx(struct device *dev, struct device_attribute *attr, char *buf)
{
	unsigned int mode, value;
	int ret;

	ret = sm_drv_oid_get((struct net_device *)dev_get_drvdata(dev),
			     LMAC_MTUM_MODE,
			     &mode,
			     sizeof(uint32_t));
	
	if (ret < 0)
		return 0;
	
	/* let's check if we're in repeat TX */
	if (mode == TM_PROM_RX)
		value = 1;
	else
		value = 0;
		
	return sprintf(buf, "%d\n", value);
}

static ssize_t sm_drv_store_per_rx(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	int ret;
	int value = simple_strtoul(buf, NULL, 0);
	int mode;

	DEBUG(DBG_SYSFS, "Setting value: %d (per_rx)\n", value);

	if (value == 1)
		mode = TM_PROM_RX;
	else
		mode = 0;
	
	ret = sm_drv_oid_set((struct net_device *)dev_get_drvdata(dev),
			     LMAC_MTUM_MODE,
			     &mode,
			     sizeof(uint32_t));
	
	if (ret < 0)
		return 0;
		
	return count;
}




static ssize_t sm_drv_store_rx_stop(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	int ret;
	int value = simple_strtoul(buf, NULL, 0);
	int mode;

	DEBUG(DBG_SYSFS, "Setting value: %d (rx_stop)\n", value);
	
	if (value == 1) {
		mode = TM_IDLE;
		
		ret = sm_drv_oid_set((struct net_device *)dev_get_drvdata(dev),
				     LMAC_MTUM_MODE,
				     &mode,
				     sizeof(uint32_t));
	
		if (ret < 0)
			return 0;
	}
		
	return count;
}

static ssize_t sm_drv_show_rx_stop(struct device *dev, struct device_attribute *attr, char *buf)
{
	unsigned int mode, value;
	int ret;

	ret = sm_drv_oid_get((struct net_device *)dev_get_drvdata(dev),
			     LMAC_MTUM_MODE,
			     &mode,
			     sizeof(uint32_t));
	
	if (ret < 0)
		return 0;
	
	/* let's check if we're in repeat TX */
	if (mode == TM_CONT_RX 
	    || mode == TM_PROM_RX
	    || mode == TM_PROM_RXTX)
		value = 0;
	else
		value = 1;
		
	return sprintf(buf, "%d\n", value);
}


/* Preamble, length and interval */
sm_drv_sysfs_ru32_attr(length, LMAC_MTUM_LENGTH)
sm_drv_sysfs_wu32_attr(length, LMAC_MTUM_LENGTH)

sm_drv_sysfs_ru32_attr(preamble, LMAC_MTUM_PREAMBLE)
sm_drv_sysfs_wu32_attr(preamble, LMAC_MTUM_PREAMBLE)

sm_drv_sysfs_ru32_attr(interval, LMAC_MTUM_INTERVAL)
sm_drv_sysfs_wu32_attr(interval, LMAC_MTUM_INTERVAL)


/* Modulation */
sm_drv_sysfs_ru32_attr(modulation, LMAC_MTUM_MODULATION)
sm_drv_sysfs_wu32_attr(modulation, LMAC_MTUM_MODULATION)

/* Power loop */
sm_drv_sysfs_ru32_attr(loop, LMAC_MTUM_POWERLOOP)
sm_drv_sysfs_wu32_attr(loop, LMAC_MTUM_POWERLOOP)


/* Rate and Channels */
static int tss_to_rate(int tss_rate)
{
	switch(tss_rate) {
	case TEST_WLAN_RATE_1M:
		return 10;
	case TEST_WLAN_RATE_2M :
		return 20;
	case TEST_WLAN_RATE_5_5M_CCK:
		return 55;
	case TEST_WLAN_RATE_5_5M_PBCC:
		return 55;
	case TEST_WLAN_RATE_11M_CCK:
		return 110;
	case TEST_WLAN_RATE_11M_PBCC:
		return 110;
	case TEST_WLAN_RATE_22M_PBCC:
		return 220;
	case TEST_WLAN_RATE_6M:
		return 60;
	case TEST_WLAN_RATE_9M:
		return 90;
	case TEST_WLAN_RATE_12M:
		return 120;
	case TEST_WLAN_RATE_18M:
		return 180;
	case TEST_WLAN_RATE_24M:
		return 240;
	case TEST_WLAN_RATE_36M:
		return 360;
	case TEST_WLAN_RATE_48M:
		return 480;
	case TEST_WLAN_RATE_54M:
		return 540;
	default:
		return 10;
	}
		
}

static int rate_to_tss(int rate)
{
	switch(rate) {
	case 10:
		return TEST_WLAN_RATE_1M;
	case 20:
		return TEST_WLAN_RATE_2M;
	case 55:
		return TEST_WLAN_RATE_5_5M_CCK;	
	case 110:
		return TEST_WLAN_RATE_11M_CCK;	
	case 220:
		return TEST_WLAN_RATE_22M_PBCC;
	case 60:
		return TEST_WLAN_RATE_6M;
	case 90:
		return TEST_WLAN_RATE_9M;
	case 120:
		return TEST_WLAN_RATE_12M;
	case 180:
		return TEST_WLAN_RATE_18M;
	case 240:
		return TEST_WLAN_RATE_24M;
	case 360:
		return TEST_WLAN_RATE_36M;
	case 480:
		return TEST_WLAN_RATE_48M;
	case 540:
		return TEST_WLAN_RATE_54M;
	default:
		return 0;
	}
	
}

static ssize_t sm_drv_show_rate(struct device *dev, struct device_attribute *attr, char *buf)
{
	int ret;
	int rate;

	ret = sm_drv_oid_get((struct net_device *)dev_get_drvdata(dev),
			     LMAC_MTUM_RATE, (void *)&rate, sizeof(uint32_t));
	if (ret < 0)
		return ret;
	

	return sprintf(buf, "%x\n", rate_to_tss(rate));
}

static ssize_t sm_drv_store_rate(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	int ret;
	int rate = tss_to_rate(simple_strtoul(buf, NULL, 0));

	DEBUG(DBG_SYSFS, "Setting value: %d (rate)", rate);

	ret = sm_drv_oid_set((struct net_device *)dev_get_drvdata(dev),
			     LMAC_MTUM_RATE, (void *)&rate, sizeof(uint32_t));
	if (ret < 0)
		return ret;

	return count;
}


static ssize_t sm_drv_show_channel(struct device *dev, struct device_attribute *attr, char *buf)
{
	int freq;
	int ret;

	ret = sm_drv_oid_get((struct net_device *)dev_get_drvdata(dev),
			     LMAC_MTUM_CHANNEL, (void *)&freq, sizeof(uint32_t));
	if (ret < 0)
		return ret;
	
	
	return sprintf(buf, "%d\n", channel_of_freq(freq/1000));
}

static ssize_t sm_drv_store_channel(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	int channel = simple_strtoul(buf, NULL, 0);
	int freq = freq_of_channel(channel);
	int ret;


	DEBUG(DBG_SYSFS, "(channel) buffer: %s\n", buf);
	DEBUG(DBG_SYSFS, "Setting value: %d (channel)\n", freq);

	ret = sm_drv_oid_set((struct net_device *)dev_get_drvdata(dev), 
			     LMAC_MTUM_CHANNEL, (void *)&freq, sizeof(uint32_t));
	if (ret < 0)
		return ret;
	
	return count;
}


/* DAC/ADC */

/* 
 * DAC values are in fact power output values
 * This is done to fit with Darium/Phoenix scheme.
 * TSS values range from 0 to 63, power from 0 to 255.
 */

extern uint32_t power_loop_mode;

static ssize_t sm_drv_show_dac_value(struct device *dev, struct device_attribute *attr, char *buf)
{
	uint32_t power_value;
	uint32_t ret;

	ret = sm_drv_oid_get((struct net_device *)dev_get_drvdata(dev), 
			     LMAC_MTUM_POWER, (void *)&power_value, sizeof(uint32_t));
	if (ret < 0)
		return ret;

	printk("Got power value: 0x%x\n", power_value);
	
	power_value &= 0xff;
	
	if (power_value == 0)
		power_value++;

	return sprintf(buf, "%d\n", power_value);
}

static ssize_t sm_drv_store_dac_value(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	int32_t dac_value =  simple_strtol(buf, NULL, 0);
	uint32_t ret;

	DEBUG(DBG_SYSFS, "Setting power: %d \n", dac_value);
	printk("Setting power: %d \n", dac_value);
	ret = sm_drv_oid_set((struct net_device *)dev_get_drvdata(dev), 
			     LMAC_MTUM_POWER, (void *)&dac_value, sizeof(int32_t));
	if (ret < 0)
		return ret;

	return count;

}

/* PA detector value aka ADC value */
static ssize_t sm_drv_show_adc_value(struct device *dev, struct device_attribute *attr, char *buf)
{
	uint32_t detector_value;
	uint32_t ret;

	printk("Calling show_adc_value\n");

	ret = sm_drv_oid_get((struct net_device *)dev_get_drvdata(dev), 
			     LMAC_MTUM_DETECTORVALUE, (void *)&detector_value, sizeof(uint32_t));
	if (ret < 0)
		return ret;

	return sprintf(buf, "%d\n", detector_value);
}

/* PER */
static ssize_t sm_drv_reset_counter(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	int ret, command = CMD_RESET_COUNTERS;

	ret = sm_drv_oid_set((struct net_device *)dev_get_drvdata(dev), 
			     LMAC_MTUM_CMD, (void *)&command, sizeof(uint32_t));
	
	if (ret < 0)
		return 0;
	
	return count;
}

static ssize_t sm_drv_show_reset_counter(struct device *dev, struct device_attribute *attr, char *buf)
{
	return sprintf(buf, "0\n");
}


sm_drv_sysfs_ru32_attr(received_frames, LMAC_MTUM_RXSUCCESS)


static ssize_t sm_drv_show_lost_frames(struct device *dev, struct device_attribute *attr, char *buf)
{
	unsigned int total_frames = 0, temp;
	int ret;

	ret = sm_drv_oid_get((struct net_device *)dev_get_drvdata(dev),
			     LMAC_MTUM_RXFAILED,
			     &temp,
			     sizeof(uint32_t));
	
	if (ret < 0)
		return 0;
	
	total_frames += temp;

	ret = sm_drv_oid_get((struct net_device *)dev_get_drvdata(dev),
			     LMAC_MTUM_RXABORTED,
			     &temp,
			     sizeof(uint32_t));
	
	if (ret < 0)
		return 0;

	total_frames += temp;

	ret = sm_drv_oid_get((struct net_device *)dev_get_drvdata(dev),
			     LMAC_MTUM_RXABORTEDPHY,
			     &temp,
			     sizeof(uint32_t));
	
	if (ret < 0)
		return 0;

	total_frames += temp;

	ret = sm_drv_oid_get((struct net_device *)dev_get_drvdata(dev),
			     LMAC_MTUM_RXFAILEDPRIV,
			     &temp,
			     sizeof(uint32_t));
	
	if (ret < 0)
		return 0;

	total_frames += temp;

	return sprintf(buf, "%d\n", total_frames);
}

sm_drv_sysfs_ru32_attr(FCS_errors, LMAC_MTUM_RXFAILED)
sm_drv_sysfs_ru32_attr(PLCP_errors, LMAC_MTUM_RXABORTEDPHY)

/* IQ calibration/tuning */
static struct obj_iqcal iqcal_params;
static ssize_t sm_drv_iq_cal(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	int i, ret;

	ret = sm_drv_oid_get((struct net_device *)dev_get_drvdata(dev), 
			     LMAC_MTUM_IQCAL, (void *)&iqcal_params, IQCAL_DATA_LEN);

	for (i = 0; i < 6 ; i++)
		     printk("IQ param %d: %d\n", i, iqcal_params.reg_val[i]);

	if (ret < 0 || iqcal_params.reg_val[0])
		return 0;
	
	return count;
}

static ssize_t sm_drv_show_iq_cal(struct device *dev, struct device_attribute *attr, char *buf)
{
	return sprintf(buf, "0\n");
}

static ssize_t sm_drv_store_iq_param(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	return count;
}

#define sm_drv_show_iq_param(param_num)                                        \
static ssize_t sm_drv_show_iq_param_##param_num(struct device *dev,            \
						struct device_attribute *attr, \
						char *buf)                     \
{	                                                                       \
	return sprintf(buf, "%d\n", iqcal_params.reg_val[(param_num)]);        \
}                                                                              \
                                                                               \
static DEVICE_ATTR(iq_param_##param_num, S_IRUGO | S_IWUSR,                    \
		   sm_drv_show_iq_param_##param_num,                           \
		   sm_drv_store_iq_param)

sm_drv_show_iq_param(0);
sm_drv_show_iq_param(1);
sm_drv_show_iq_param(2);
sm_drv_show_iq_param(3);
sm_drv_show_iq_param(4);
sm_drv_show_iq_param(5);

/* Baseband and RF synthetiser settings */
sm_drv_sysfs_ru32_attr(bb_reg_nr, LMAC_MTUM_BBREGNR)
sm_drv_sysfs_wu32_attr(bb_reg_nr, LMAC_MTUM_BBREGNR)
sm_drv_sysfs_ru32_attr(bb_reg_value, LMAC_MTUM_BBREGVALUE)
sm_drv_sysfs_wu32_attr(bb_reg_value, LMAC_MTUM_BBREGVALUE)

sm_drv_sysfs_ru32_attr(rf_reg_nr, LMAC_MTUM_SYNTHNR)
sm_drv_sysfs_wu32_attr(rf_reg_nr, LMAC_MTUM_SYNTHNR)
sm_drv_sysfs_ru32_attr(rf_reg_value, LMAC_MTUM_SYNTH)
sm_drv_sysfs_wu32_attr(rf_reg_value, LMAC_MTUM_SYNTH)

/* RSSI and SQ for flali */

static ssize_t sm_drv_show_signal_quality(struct device *dev, struct device_attribute *attr, char *buf)
{
	uint32_t signal_quality;
	uint32_t ret;
	
	printk("Calling show_adc_value\n");
	
	ret = sm_drv_oid_get((struct net_device *)dev_get_drvdata(dev), 
			     LMAC_MTUM_SQVALUE, (void *)&signal_quality, sizeof(uint32_t));
	if (ret < 0)
		return ret;
	
	return sprintf(buf, "%d\n", signal_quality);
}

static ssize_t sm_drv_show_rssi(struct device *dev, struct device_attribute *attr, char *buf)
{
	uint32_t signal_rssi;
	uint32_t ret;
	
	ret = sm_drv_oid_get((struct net_device *)dev_get_drvdata(dev), 
			     LMAC_MTUM_RAW_RSSI, (void *)&signal_rssi, sizeof(uint32_t));
	if (ret < 0)
		return ret;
	
	return sprintf(buf, "%d\n", signal_rssi);
}

/* 
 * LMAC GPIO stuff.
 * We need that for reading the BT-WLAN
 * coexistence lines.
 */
sm_drv_sysfs_ru32_attr(gpio_value, LMAC_MTUM_GPIOVALUE)
sm_drv_sysfs_wu32_attr(gpio_value, LMAC_MTUM_GPIOVALUE)
sm_drv_sysfs_ru32_attr(gpio_mask, LMAC_MTUM_GPIOMASK)
sm_drv_sysfs_wu32_attr(gpio_mask, LMAC_MTUM_GPIOMASK)
sm_drv_sysfs_ru32_attr(gpio_bank, LMAC_MTUM_GPIOBANK)
sm_drv_sysfs_wu32_attr(gpio_bank, LMAC_MTUM_GPIOBANK)
sm_drv_sysfs_ru32_attr(gpio_direction, LMAC_MTUM_GPIODDR)
sm_drv_sysfs_wu32_attr(gpio_direction, LMAC_MTUM_GPIODDR)

static DEVICE_ATTR(WLAN_reset, S_IRUGO| S_IWUSR, sm_drv_show_wlan_reset, sm_drv_wlan_reset);
static DEVICE_ATTR(timeout, S_IRUGO | S_IWUSR, sm_drv_show_timeout, sm_drv_store_timeout);
static DEVICE_ATTR(psm, S_IRUGO | S_IWUSR, sm_drv_show_psm, sm_drv_store_psm);
static DEVICE_ATTR(rate, S_IRUGO | S_IWUSR, sm_drv_show_rate, sm_drv_store_rate);
static DEVICE_ATTR(channel, S_IRUGO | S_IWUSR, sm_drv_show_channel, sm_drv_store_channel);
static DEVICE_ATTR(packet_preamble, S_IRUGO | S_IWUSR, sm_drv_show_preamble, sm_drv_store_preamble);
static DEVICE_ATTR(packet_interval, S_IRUGO | S_IWUSR, sm_drv_show_interval, sm_drv_store_interval);
static DEVICE_ATTR(packet_length, S_IRUGO | S_IWUSR, sm_drv_show_length, sm_drv_store_length);
static DEVICE_ATTR(modulation, S_IRUGO | S_IWUSR, sm_drv_show_modulation, sm_drv_store_modulation);
static DEVICE_ATTR(loop, S_IRUGO | S_IWUSR, sm_drv_show_loop, sm_drv_store_loop);
static DEVICE_ATTR(continuous_tx, S_IRUGO | S_IWUSR, sm_drv_show_continuous_tx, sm_drv_store_continuous_tx);
static DEVICE_ATTR(PER_tx, S_IRUGO | S_IWUSR, sm_drv_show_per_tx, sm_drv_store_per_tx);
static DEVICE_ATTR(burst_tx, S_IRUGO | S_IWUSR, sm_drv_show_dummy, sm_drv_store_burst_tx);
static DEVICE_ATTR(burst_rf_gain, S_IRUGO | S_IWUSR, sm_drv_show_dummy, sm_drv_store_burst_rf_gain);
static DEVICE_ATTR(burst_detector, S_IRUGO, sm_drv_show_burst_detector, NULL);
static DEVICE_ATTR(tx_pattern, S_IRUGO | S_IWUSR, sm_drv_show_tx_pattern, sm_drv_store_tx_pattern);
static DEVICE_ATTR(tx_stop, S_IRUGO | S_IWUSR, sm_drv_show_tx_stop, sm_drv_store_tx_stop);
static DEVICE_ATTR(PER_rx, S_IRUGO | S_IWUSR, sm_drv_show_per_rx, sm_drv_store_per_rx);
static DEVICE_ATTR(rx_stop, S_IRUGO| S_IWUSR, sm_drv_show_rx_stop, sm_drv_store_rx_stop);
static DEVICE_ATTR(dac_value, S_IRUGO | S_IWUSR, sm_drv_show_dac_value, sm_drv_store_dac_value);
static DEVICE_ATTR(adc_value, S_IRUGO, sm_drv_show_adc_value, NULL);
static DEVICE_ATTR(PER_clear, S_IRUGO | S_IWUSR, sm_drv_show_reset_counter, sm_drv_reset_counter);
static DEVICE_ATTR(PER_received_frames, S_IRUGO, sm_drv_show_received_frames, NULL);
static DEVICE_ATTR(PER_lost_frames, S_IRUGO, sm_drv_show_lost_frames, NULL);
static DEVICE_ATTR(PER_FCS_errors, S_IRUGO, sm_drv_show_FCS_errors, NULL);
static DEVICE_ATTR(PER_PLCP_errors, S_IRUGO, sm_drv_show_PLCP_errors, NULL);
static DEVICE_ATTR(iq_calibration, S_IRUGO | S_IWUSR, sm_drv_show_iq_cal, sm_drv_iq_cal);
static DEVICE_ATTR(bb_reg_nr, S_IRUGO | S_IWUSR, sm_drv_show_bb_reg_nr, sm_drv_store_bb_reg_nr);
static DEVICE_ATTR(bb_reg_value, S_IRUGO | S_IWUSR, sm_drv_show_bb_reg_value, sm_drv_store_bb_reg_value);
static DEVICE_ATTR(rf_reg_nr, S_IRUGO | S_IWUSR, sm_drv_show_rf_reg_nr, sm_drv_store_rf_reg_nr);
static DEVICE_ATTR(rf_reg_value, S_IRUGO | S_IWUSR, sm_drv_show_rf_reg_value, sm_drv_store_rf_reg_value);
static DEVICE_ATTR(signal_quality, S_IRUGO | S_IWUSR, sm_drv_show_signal_quality, NULL);
static DEVICE_ATTR(rssi, S_IRUGO | S_IWUSR, sm_drv_show_rssi, NULL);
static DEVICE_ATTR(gpio_value, S_IRUGO | S_IWUSR, sm_drv_show_gpio_value, sm_drv_store_gpio_value);
static DEVICE_ATTR(gpio_mask, S_IRUGO | S_IWUSR, sm_drv_show_gpio_mask, sm_drv_store_gpio_mask);
static DEVICE_ATTR(gpio_bank, S_IRUGO | S_IWUSR, sm_drv_show_gpio_bank, sm_drv_store_gpio_bank);
static DEVICE_ATTR(gpio_direction, S_IRUGO | S_IWUSR, sm_drv_show_gpio_direction, sm_drv_store_gpio_direction);

extern struct platform_device wlan_omap_device;
int sm_drv_sysfs_umac_create_files(void)
{
	device_create_file(&(wlan_omap_device.dev), &dev_attr_WLAN_reset);

	device_create_file(&(wlan_omap_device.dev), &dev_attr_timeout);
	
	device_create_file(&(wlan_omap_device.dev), &dev_attr_channel);

	device_create_file(&(wlan_omap_device.dev), &dev_attr_rate);

	device_create_file(&(wlan_omap_device.dev), &dev_attr_psm);
	device_create_file(&(wlan_omap_device.dev), &dev_attr_continuous_tx);
	device_create_file(&(wlan_omap_device.dev), &dev_attr_PER_tx);
	device_create_file(&(wlan_omap_device.dev), &dev_attr_tx_pattern);
	device_create_file(&(wlan_omap_device.dev), &dev_attr_tx_stop);

	device_create_file(&(wlan_omap_device.dev), &dev_attr_burst_tx);
	device_create_file(&(wlan_omap_device.dev), &dev_attr_burst_rf_gain);
	device_create_file(&(wlan_omap_device.dev), &dev_attr_burst_detector);

	device_create_file(&(wlan_omap_device.dev), &dev_attr_packet_length);
	device_create_file(&(wlan_omap_device.dev), &dev_attr_packet_preamble);
	device_create_file(&(wlan_omap_device.dev), &dev_attr_packet_interval);

	device_create_file(&(wlan_omap_device.dev), &dev_attr_modulation);

	device_create_file(&(wlan_omap_device.dev), &dev_attr_loop);
	
	device_create_file(&(wlan_omap_device.dev), &dev_attr_PER_rx);
	device_create_file(&(wlan_omap_device.dev), &dev_attr_rx_stop);
	
	device_create_file(&(wlan_omap_device.dev), &dev_attr_dac_value);
	device_create_file(&(wlan_omap_device.dev), &dev_attr_adc_value);
	


	device_create_file(&(wlan_omap_device.dev), &dev_attr_PER_clear);
	device_create_file(&(wlan_omap_device.dev), &dev_attr_PER_received_frames);
	device_create_file(&(wlan_omap_device.dev), &dev_attr_PER_lost_frames);
	device_create_file(&(wlan_omap_device.dev), &dev_attr_PER_FCS_errors);
	device_create_file(&(wlan_omap_device.dev), &dev_attr_PER_PLCP_errors);
	device_create_file(&(wlan_omap_device.dev), &dev_attr_iq_calibration);

	device_create_file(&(wlan_omap_device.dev), &dev_attr_iq_param_0);
	device_create_file(&(wlan_omap_device.dev), &dev_attr_iq_param_1);
	device_create_file(&(wlan_omap_device.dev), &dev_attr_iq_param_2);
	device_create_file(&(wlan_omap_device.dev), &dev_attr_iq_param_3);
	device_create_file(&(wlan_omap_device.dev), &dev_attr_iq_param_4);
	device_create_file(&(wlan_omap_device.dev), &dev_attr_iq_param_5);

	device_create_file(&(wlan_omap_device.dev), &dev_attr_bb_reg_nr);
	device_create_file(&(wlan_omap_device.dev), &dev_attr_bb_reg_value);
	device_create_file(&(wlan_omap_device.dev), &dev_attr_rf_reg_nr);
	device_create_file(&(wlan_omap_device.dev), &dev_attr_rf_reg_value);

	device_create_file(&(wlan_omap_device.dev), &dev_attr_gpio_value);
	device_create_file(&(wlan_omap_device.dev), &dev_attr_gpio_direction);
	device_create_file(&(wlan_omap_device.dev), &dev_attr_gpio_bank);
	device_create_file(&(wlan_omap_device.dev), &dev_attr_gpio_mask);

	device_create_file(&(wlan_omap_device.dev), &dev_attr_psm);
	device_create_file(&(wlan_omap_device.dev), &dev_attr_signal_quality);
	device_create_file(&(wlan_omap_device.dev), &dev_attr_rssi);

	memset(&burst_tx_params, 0, sizeof(struct obj_burstctx));

	return 0;
}
